/*
 *	Routines used to draw a polygon interior.  This is actually a
 *	area fill algorithm where the area is constrained to be a 
 *      polygonal region with less than 256 verticies.
 */

#include <vt_hdrs.h>
#include <graphics.h>
#include <rectangle.h>

#define MAXEDGES    255
#define MAXLINES    1024


static struct edge
{
    struct edge	    *next;
    short	    x;
    short	    dx;
    short	    dy;
    short	    count;
    short	    acc;
    short	    inc;
} edges[MAXEDGES];

static struct edge *buckets[MAXLINES];
static struct edge *active;

static short minx, miny, maxx, maxy;


static MergeLists(edge, list)
register struct edge *edge, **list;
{
    register struct edge *nextedge, *last, *this;

    last = NULL;
    this = *list;
    
    while (edge != NULL)
    {
	nextedge = edge->next;

	while (this != NULL && edge->x > this->x)
	{
	    last = this;
	    this = this->next;
	}

	edge->next = this;
	if (last == NULL)
	    *list = edge;
	else
	    last->next = edge;
	
	last = edge;
	
	edge = nextedge;
    }
}
	
    

static SortList(list)
register struct edge **list;
{
    register struct edge *ie, *je;
    register short temp;

    for (ie = *list;  ie != NULL; ie = ie->next)
        for (je = ie->next;  je != NULL;  je = je->next)
	    if (ie->x > je->x)
	    {
		temp = ie->x;
		ie->x = je->x;
		je->x = temp;

		temp = ie->dx;
		ie->dx = je->dx;
		je->dx = temp;

		temp = ie->dy;
		ie->dy = je->dy;
		je->dy = temp;

		temp = ie->acc;
		ie->acc = je->acc;
		je->acc = temp;

		temp = ie->inc;
		ie->inc = je->inc;
		je->inc = temp;

		temp = ie->count;
		ie->count = je->count;
		je->count = temp;
	    }
}



SetupEdgeTable(n, x, y)
register short n, *x, *y;
{
    int i;
    register short p2, p1, n1, n2;
    short xemin, yemin, xemax, yemax;
    register struct edge *edge;
    extern short cliplx, cliply, clipux, clipuy;

    for (i = 0; i < MAXLINES; i++)
	buckets[i] = NULL;
	
    active = NULL;

    minx = maxx = x[0];
    miny = maxy = y[0];
    
    for (p2 = n-3, p1 = n-2, n1 = n-1, n2 = 0;
	 n2 < n;
	 p2 = p1, p1 = n1, n1 = n2, n2++)
    {
	if (x[n2] < minx)
	    minx = x[n2];
	if (x[n2] > maxx)
	    maxx = x[n2];
	if (y[n2] < miny)
	    miny = y[n2];
	if (y[n2] > maxy)
	    maxy = y[n2];

	if (y[p1] == y[n1])
	    continue;
	
	if (y[p1] < y[n1])
	{
	    xemin = x[p1];
	    yemin = y[p1];
	    xemax = x[n1];
	    yemax = y[n1];
	}
	else
	{
	    xemin = x[n1];
	    yemin = y[n1];
	    xemax = x[p1];
	    yemax = y[p1];
	}

	if (yemin >= clipuy || yemax < cliply)
	    continue;

	edge = &edges[n2];
	
	if (yemin >= cliply)
	{
	    edge->x = xemin;
	    edge->dx = xemax-xemin;
	    edge->dy = yemax-yemin;
	    edge->count = edge->dy+1;
	    edge->acc = edge->dy >> 1;
	    edge->next = NULL;
	}
	else
	{
	    edge->x = xemin+(xemax-xemin)*(cliply-yemin)/(yemax-yemin);
	    edge->dx = xemax-xemin;
	    edge->dy = yemax-yemin;
	    edge->count = yemax-cliply+1;
	    edge->acc = edge->dy >> 1;
	    edge->next = NULL;
	    yemin = cliply;
	}

	if (edge->dx < 0)
	{
	    edge->dx = -edge->dx;
	    edge->inc = -1;
	}
	else
	    edge->inc = 1;

	if ((y[p1] < y[n1] && y[n1] <= y[n2]) ||
	    (y[n1] < y[p1] && y[p1] <= y[p2]))
	    edge->count--;

	MergeLists(edge, &buckets[yemin]);
    }
}


DrawPolygonInterior(n, x, y, color, pat, xphase, yphase)
int n;
short *x, *y;
COLORT color;
PATTERN pat;
int xphase, yphase;
{
    register int i;
    register struct edge *ledge, *redge;
    struct rectangle bounds;
    extern short cliplx, cliply, clipux, clipuy;
    register struct edge *ep;

    SetupEdgeTable(n, x, y);
    
    bounds.x = (minx >= cliplx) ? minx : cliplx;
    bounds.y = (miny >= cliply) ? miny : cliply;
    bounds.w = ((maxx < clipux) ? maxx+1 : clipux) - bounds.x;
    bounds.h = ((maxy < clipuy) ? maxy+1 : clipuy) - bounds.y;
    
    for (i = bounds.y; i < bounds.y+bounds.h; i++)
    {
	MergeLists(buckets[i], &active);
	
	for (ledge = active; ledge != NULL; ledge = redge->next)
	{
	    redge = ledge->next;
 	    Scanline(ledge->x, redge->x, i, color, pat, xphase, yphase);
	}
	
	ledge = NULL;
	for (redge = active; redge != NULL; redge = redge->next)
	{
	    if (--redge->count > 0)
	    {
	        redge->acc += redge->dx;
		while (redge->acc > redge->dy)
		{
		    redge->x += redge->inc;
		    redge->acc -= redge->dy;
		}
		ledge = redge;
	    }
	    else
	    {
		if (ledge == NULL)
		    active = redge->next;
		else
		    ledge->next = redge->next;
	    }
	}

        SortList(&active);
    }
}
