/*
 * Copyright (c) 1982 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)ufs_dsort.c	6.3 (Berkeley) 6/8/85
 */

/*
 * Seek sort for disks.  We depend on the driver
 * which calls us using b_resid as the current cylinder number.
 *
 * The argument dp structure holds a b_actf activity chain pointer
 * on which we keep a single queue, containing the non-ASYNC requests
 * followed by the ASYNC requests.  The queue is sorted starting
 * at the current cylinder and going in one direction to the farthest
 * request, then in the other direction to the opposite end of the drive.
 * We thus implement a two-way scan, going one direction then turning
 * around and going back the other way.
 */

#include "param.h"
#include "systm.h"
#include "buf.h"

#define	b_cylin	b_resid

disksort(dp, bp)
	register struct buf *dp, *bp;
{
	register struct buf *ap, *tp;
	register long bc, tc, cyldiff, async;

	if ((ap = dp->b_actf) == NULL) {
		/*
		 * Queue is empty.  New item becomes only item.
		 */
		dp->b_actf = bp;
		dp->b_actl = bp;
		bp->av_forw = NULL;
		return;
	}

	if (bp->b_flags & B_ASYNC) {
		async = 0;
		while (tp = ap->av_forw) {
			if (tp->b_flags & B_ASYNC)
				break;
			else
				ap = tp;
		}
	} else
		async = B_ASYNC;

	while (tp = ap->av_forw) {
		if (tp->b_flags & async)
			break;
		/*
		 * Skip over initial entries that all lie
		 *  on the same cylinder, so we can determine
		 *  whether queue is going up then down, or down then up.
		 */
		if ((cyldiff = tp->b_cylin - ap->b_cylin) == 0) {
			ap = tp;
			continue;
		}
		bc = bp->b_cylin;

		if (cyldiff > 0) {
			/*
			 * Queue goes up, then down.
			 */
			if (bc > ap->b_cylin) {
				/*
				 * New entry belongs on first half
				 *  of up-down queue.
				 */
				while (tp = ap->av_forw) {
					if (tp->b_flags & async)
						break;
					tc = tp->b_cylin;
					if (tc > bc  ||  tc < ap->b_cylin)
						break;
					ap = tp;
				}
			} else {
				/*
				 * New entry belongs on second half
				 *  of up-down queue.
				 */
				while (tp = ap->av_forw) {
					if (tp->b_flags & async)
						break;
					if (tp->b_cylin < bc)
						break;
					ap = tp;
				}
			}
		} else {
			/*
			 * Queue goes down, then up.
			 */
			if (bc < ap->b_cylin) {
				/*
				 * New entry belongs on first half
				 *  of down-up queue.
				 */
				while (tp = ap->av_forw) {
					if (tp->b_flags & async)
						break;
					tc = tp->b_cylin;
					if (tc < bc  ||  tc > ap->b_cylin)
						break;
					ap = tp;
				}
			} else {
				/*
				 * New entry belongs on second half
				 *  of down-up queue.
				 */
				while (tp = ap->av_forw) {
					if (tp->b_flags & async)
						break;
					if (tp->b_cylin > bc)
						break;
					ap = tp;
				}
			}
		}
		break;
	}
	/*
	 * Now ap points at item in queue that bp should come after,
	 *  and tp points at what will be the item to follow bp.
	 *  If bp is to become the last item, then tp will be NULL,
	 *  which is just fine.
	 */
	bp->av_forw = tp;
	ap->av_forw = bp;
	if(tp == NULL)
		dp->b_actl = bp;
}
