#include "param.h"

#include "user.h"
#include "kernel.h"
#include "file.h"
#include "dir.h"
#include "pathname.h"
#include "stat.h"
#include "vnode.h"
#include "vfs.h"
#include "proc.h"
#include "uio.h"
#include "buf.h"
#include "ioctl.h"
#include "acct.h"
#include "vmparam.h"
#include "vmmac.h"

#include "../h/mbuf.h"
#include "../h/protosw.h"
#include "../h/socket.h"
#include "../h/socketvar.h"
#include "../h/unpcb.h"
#include "../h/un.h"

#include "../ufs/inode.h"
#include "../machine/board.h"
#include "../wipc/wipc.h"

#ifdef	SYSV
#include "../sysv/sys/stat.h"
#include "../sysv/sys/statfs.h"
#include "../sysv/sys/fcntl.h"
#include "../sysv/sys/errno.h"
#ifdef RFS
#include "../sysv/sys/sysmacros.h"
#endif RFS
#endif	SYSV

/*
 * TRFS in a nutshell:
 *
 * System call handlers as defined in sys/init_sysent.c are revectored into
 * handlers prefixed with "Cl" for client.  This revectoring is accomplished 
 * by the #defines in trfs/syscalls.h.  The "Cl" routines, after setting up a 
 * long jump trampoline, call the normal unix system call routine.  For local
 * operations the unix handler will return, never using the trampoline.  If, 
 * however, during pathname expansion for the system call a trfs pathname 
 * is encountered the trampoline is used to get back into the "Cl" code and 
 * originate a request for the system call to be preformed by a cousin on
 * another machine.  This request is encoded by the routines in trfs/remote.c 
 * with the "Sd" prefix and sent out over the wire.  On the server/cousin side 
 * these requests are eventualy recieved by the routines in trfs/remote.c 
 * prefixed with "Sv".  The "Sv" routines encode the results after having 
 * called the cousin service routine and reply back back over the wire to the 
 * client.  The cousin service routines use the "Co" prefix and are (for the 
 * most part) defined in this file.  A typical "Co" service routine will
 * call the same "Cl" routine that started it all, except this time the call
 * occurs on the server machine.
 */

extern unsigned long RmidToRpid();

int ClRw(), ClIoctl(), ClSelect(), ClClose();	/* for remote operations */
struct fileops tremoteops = { ClRw, ClIoctl, ClSelect, ClClose };

int vno_rw(), vno_ioctl(), vno_select();	/* for imported devices */
struct fileops tdeviceops = { vno_rw, vno_ioctl, vno_select, ClClose };

/* 
 * On the client side, many of the operations preformed by a TRFS revectored 
 * system call handler fall into one of two catagories: They are simple 
 * operations on files names, or they are operations on file descriptors.
 * To reduce code length the generic processing for this type of system call
 * is defined in the macros below.
 */
#define	TRFS_PNFREE()							\
	if (u.u_tpnp) {							\
		pn_dfree(u.u_tpnp); u.u_tpnp = (struct pathname *)0;	\
	}
#define	Cl_FNAM(UNIX_OP, TRFS_Sd)					\
	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {			\
		UNIX_OP; u.u_tok = 0; return;				\
	}								\
	if (u.u_error == 0  &&  u.u_tclient)				\
		u.u_error = ENOENT;					\
	if (u.u_error == 0)						\
		u.u_error = TRFS_Sd;					\
	TRFS_PNFREE();
#define	Cl_FDES(UNIX_OP, TRFS_Sd)					\
	register struct file *fp;					\
	unsigned long rpid;						\
									\
	GETF(fp, uap->fdes);						\
	if (fp->f_type != DTYPE_TRFS) {					\
		UNIX_OP; return;					\
	}								\
	if ((rpid = RmidToRpid(fp->f_rmid)) == 0) {			\
		u.u_error = EBADF; return;				\
	}								\
	u.u_error = TRFS_Sd;

/*
 * On the cousin side, most handlers initialize system call arguments and
 * reference reference the standard or revectored handler.  To reduce code 
 * length the generic processing of this type is defined in the macros below.
 */ 
#define	Co_OP1(OP, A1)							\
	uap->A1=A1; OP; return u.u_error
#define	Co_OP2(OP, A1, A2)						\
	uap->A1=A1;uap->A2=A2; OP; return u.u_error
#define	Co_OP3(OP, A1, A2, A3)						\
	uap->A1=A1;uap->A2=A2;uap->A3=A3; OP; return u.u_error
#define	Co_OP4(OP, A1, A2, A3, A4)					\
	uap->A1=A1;uap->A2=A2;uap->A3=A3;uap->A4=A4; OP; return u.u_error

/* ------------------------- system calls ---------------------------- */

ClFork()
{
	struct dmap cdmap, csmap;

	cdmap = zdmap;
	csmap = zdmap;
	if (swpexpand(u.u_dsize, u.u_ssize, &cdmap, &csmap) == 0) {
		u.u_r.r_val2 = 0;
		return;
	}
	u.u_cdmap = &cdmap;
	u.u_csmap = &csmap;
	ClFork1(0);
 	u.u_cdmap = (struct dmap *)0;
 	u.u_csmap = (struct dmap *)0;
}

ClVfork()
{
	ClFork1(1);
}

ClFork1(isvfork)
{
	unsigned long newrpid[sizeof u.u_trpid / sizeof u.u_trpid[0]];
	register unsigned long rpid, *p = u.u_trpid, *q = newrpid, *r;
	register int n;
	int err, fatal = 0;
	char msg[80];

	bzero(newrpid, sizeof newrpid);
	r = &u.u_trpid[sizeof u.u_trpid/sizeof u.u_trpid[0]];
	while (p < r  && (rpid = *p++)) {
		if (CousinNeeded(MID(rpid))) {
			if ((n = (int)SdFork(rpid)) > 0)
				*q++ = n;
			else if (n < 0) {
				n = (-n);
				printf("TRFS: cousin fork error %d\n", n);
				switch (n) {
				  case EAGAIN:		/* no more processes */
				  case ENOMEM:		/* not enough core */
					break;
				  default:
					/* kill self */
					fatal = 1;
				}
				u.u_error = n;
				break;
			}
		}
	}
	if (u.u_error == 0)
		fork1(isvfork);
	if (u.u_error) {
		err = u.u_error;
		u.u_error = 0;
		q = newrpid;
		while (q < p  &&  (n = *q++))
			SdExit(n, 0);
		u.u_error = err;
		if (fatal) {
			sprintf(msg, 
			    "pid %d (%s) killed due to loss of cousin %d @%s",
			    u.u_procp->p_pid, u.u_comm, 
			    PID(rpid), RpidToName(rpid));
			printf("TRFS: %s\n", msg);
			uprintf("Sorry, %s\n", msg);
			psignal(u.u_procp, SIGKILL);
		}
	} else if (u.u_r.r_val2)
		bcopy(newrpid, u.u_trpid, sizeof u.u_trpid);
}

CoFork()
{
	struct dmap cdmap, csmap;

	cdmap = zdmap;
	csmap = zdmap;
	u.u_cdmap = &cdmap;
	u.u_csmap = &csmap;
	fork1(0);
 	u.u_cdmap = (struct dmap *)0;
 	u.u_csmap = (struct dmap *)0;
	if (u.u_error)
		return -u.u_error;
	if (u.u_r.r_val2) {			/* child */
		setctx(u.u_procp->p_context = CTX_SYS);
		RunAway();
		sprintf(u.u_comm, "%s %d", u.u_comm, u.u_r.r_val1);
		return 0;
	}
	return u.u_r.r_val1;
}

ClRexit(uap)
	struct a {
		int	rval;
	} *uap;
{
	ClExit((uap->rval & 0377) << 8);
}

ClExit(rval)
{
	register unsigned long rpid;
	register unsigned long *p, *q;

	q = &u.u_trpid[sizeof u.u_trpid/sizeof *q];
	for (p = u.u_trpid;  p < q  &&  (rpid = *p);  p += 1) {
		SdExit(rpid, rval);
		*p = 0;
	}

	/* could free packets here, maybe? */
	exit(rval);			/* no return */
}

ClOpen(uap)
	register struct a {
		char *fname;
		int fmode;
		int cmode;
	} *uap;
{
	ClCopen(uap->fname, uap->fmode - FOPEN, uap->cmode);
}

ClCreat(uap)
	register struct a {
		char *fname;
		int cmode;
	} *uap;
{
	ClCopen(uap->fname, FWRITE|FCREAT|FTRUNC, uap->cmode);
}

ClCopen(fname, mode, cmode)
	char	*fname;
{
	unsigned long dev, inum, imode, rdev, rpid;
	register struct file *fp;
	int rfdes, rfd, i;

	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		u.u_error = copen(fname, mode, cmode);
		u.u_tok = 0;
		return;
	}
	if (u.u_error == 0  &&  u.u_tclient)
		u.u_error = ENOENT;	/* really shouldn't happen */

	/* pick up the slot allocated in copen */
	if (u.u_error || (fp = getf(i = u.u_r.r_val1)) == 0) {
		TRFS_PNFREE();
		return;
	}
	fp->f_type = DTYPE_TRFS;
	fp->f_ops = &tdeviceops;
	fp->f_data = (caddr_t)0;
	fp->f_rmid = 0;
	rfdes = SdOpen((rpid = u.u_tcurrpid), u.u_tpnp->pn_path, mode, 
			cmode & ~u.u_cmask, &dev, &inum, &imode, &rdev);
	TRFS_PNFREE();
	if (rfdes >= 0) {
		fp->f_flag = (mode & FMASK) | FSELTRUE;
		fp->f_rmid = MID(rpid);
		fp->f_rfdes = rfdes;
		if (!ImportedDevice(imode, rdev)) {
			fp->f_ops = &tremoteops;
			if ((imode & IFMT) == IFCHR)
				fp->f_flag |= FNONIDEM;
			return;
		}

		/* import device from server, set up local v/inode */
		fp->f_data = (caddr_t)trfs_dev_set(dev, inum, imode, rdev);
		if ((setjmp(&u.u_qsave) == 0) && ((u.u_error = 
		    trfs_dev_open((struct vnode *)fp->f_data, mode)) == 0) )
			return;
		if (u.u_error == 0)
			u.u_error = EINTR;
	} else
		u.u_error = -rfdes;
	u.u_ofile[i] = NULL;
	closef(fp, 0);		/* not realy open on client!! */
}

CoCopen(fname, mode, cmode, devp, inump, modep, rdevp)
	char	*fname;
	unsigned long *devp, *inump, *modep, *rdevp;
{
	register struct inode *ip;
	struct inode *geti();

	ClCopen(fname, mode, cmode);
	if (u.u_error)
		return -u.u_error;
	if (ip = geti(u.u_r.r_val1)) {
		if (devp)
			*devp = ip->i_dev;
		if (inump)
			*inump = ip->i_number;
		if (modep)
			*modep = ip->i_mode;
		if (rdevp)
			*rdevp = ip->i_rdev;
	}
	return u.u_r.r_val1;
}

ClStat(uap)
	struct a {
		char	*fname;
		struct	stat *ub;
	} *uap;
{
	ClStat1(uap, FOLLOW_LINK);
}

ClLstat(uap)
	struct a {
		char	*fname;
		struct	stat *ub;
	} *uap;
{
	ClStat1(uap, NO_FOLLOW);
}

ClStat1(uap, follow)
	register struct a {
		char	*fname;
		struct	stat *ub;
	} *uap;
{
	if (!useracc(uap->ub, sizeof (struct stat), B_WRITE))
		u.u_error = EFAULT;
	else {
	    Cl_FNAM(u.u_error = stat1(uap, follow), 
		SdStat(u.u_tcurrpid, u.u_tpnp->pn_path, follow, -1, 
			uap->ub, sizeof(struct stat), UIO_USERSPACE));
	    if (diskless && (MID(u.u_tcurrpid) == MID(servermid)))
		u_bzero(&uap->ub->st_spare4[0], sizeof(uap->ub->st_spare4[0]));
	}
}

CoStat(fname, follow, ub)
	char	*fname;
	struct stat *ub;
{
	register struct a {
		char	*fname;
		struct	stat *ub;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(ClStat1(uap, follow), fname, ub);
}

ClFstat(uap)
	register struct a {
		int		fdes;
		struct stat	*ub;
	} *uap;
{
	if (!useracc(uap->ub, sizeof (struct stat), B_WRITE))
		u.u_error = EFAULT;
	else {
	    Cl_FDES(fstat(uap), 
		SdStat(rpid, (char *)0, 0, fp->f_rfdes, uap->ub, 
			sizeof(struct stat), UIO_USERSPACE));
	    if (diskless && (MID(rpid) == MID(servermid)))
		u_bzero(&uap->ub->st_spare4[0], sizeof(uap->ub->st_spare4[0]));
	}
}

ClFlen(fp)		/* NOT a system call... for lseek */
	register struct file *fp;
{
	unsigned long rpid;
	struct stat stbuf;

	stbuf.st_size = 0;
	if ((rpid = RmidToRpid(fp->f_rmid)) == 0) {
		u.u_error = EBADF;
		return 0;
	}
	u.u_error = SdStat(rpid, (char *)0, 0, fp->f_rfdes,
			&stbuf, sizeof(struct stat), UIO_SYSSPACE);
	return stbuf.st_size;
}


CoFstat(fdes, ub)
	struct stat *ub;
{
	register struct a {
		int		fdes;
		struct stat	*ub;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(fstat(uap), fdes, ub);
}

ClAccess(uap)
	register struct a {
		char	*fname;
		int	fmode;
	} *uap;
{
	Cl_FNAM(access(uap), 
		SdAccess(u.u_tcurrpid, u.u_tpnp->pn_path, uap->fmode));
}

CoAccess(fname, fmode, ruid, rgid)
	char	*fname;
	uid_t	ruid;
	gid_t	rgid;
{
	register struct a {
		char	*fname;
		int	fmode;
	} *uap = (struct a *)u.u_ap;
	uid_t	svuid;
	gid_t	svgid;

	uap->fname = fname;
	uap->fmode = fmode;
	svuid = u.u_ruid;
	svgid = u.u_rgid;
	u.u_ruid = ruid;
	u.u_rgid = rgid;
	ClAccess(uap);
	u.u_ruid = svuid;
	u.u_rgid = svgid;
	return u.u_error;
}

ClReadlink(uap)
	register struct a {
		char	*name;
		char	*buf;
		int	count;
	} *uap;
{
	int count = uap->count;

	if (!useracc(uap->buf, count, B_WRITE)) {
		u.u_error = EFAULT;
		return;
	}
	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		readlink(uap);
		u.u_tok = 0;
		return;
	}
	if (u.u_error == 0  &&  u.u_tclient)
		u.u_error = ENOENT;	/* really shouldn't happen */
	if (u.u_error == 0)
		if ((u.u_error = SdReadlink(u.u_tcurrpid, u.u_tpnp->pn_path,
				uap->buf, &count, UIO_USERSPACE)) == 0)
			u.u_r.r_val1 = count;
	TRFS_PNFREE();
}

CoReadlink(name, buf, count)
	char *name, *buf;
{
	register struct a {
		char	*name;
		char	*buf;
		int	count;
	} *uap = (struct a *)u.u_ap;

	uap->name = name;
	uap->buf = buf;
	uap->count = count;
	ClReadlink(uap);
	if (u.u_error)
		return -u.u_error;
	return u.u_r.r_val1;
}

ClUnlink(uap)
	register struct a {
		char	*pnamep;
	} *uap;
{
	Cl_FNAM(unlink(uap), 
		SdUnlink(u.u_tcurrpid, u.u_tpnp->pn_path));
}

CoUnlink(pnamep)
	char *pnamep;
{
	register struct a {
		char	*pnamep;
	} *uap = (struct a *)u.u_ap;

	Co_OP1(ClUnlink(uap), pnamep);
}

ClChmod(uap)
	register struct a {
		char	*fname;
		int	fmode;
	} *uap;
{
	Cl_FNAM(chmod(uap), 
		SdChmod(u.u_tcurrpid, u.u_tpnp->pn_path, -1, uap->fmode));
}

CoChmod(fname, fmode)
	char *fname;
{
	register struct a {
		char	*fname;
		int	fmode;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(ClChmod(uap), fname, fmode);
}

ClFchmod(uap)
	register struct a {
		int	fdes;
		int	fmode;
	} *uap;
{
	Cl_FDES(fchmod(uap), 
		SdChmod(rpid, (char *)0, fp->f_rfdes, uap->fmode));
}

CoFchmod(fdes, fmode)
{
	register struct a {
		int	fdes;
		int	fmode;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(fchmod(uap), fdes, fmode);
}

ClChown(uap)
	register struct a {
		char	*fname;
		int	uid;
		int	gid;
	} *uap;
{
	Cl_FNAM(chown(uap), 
		SdChown(u.u_tcurrpid, u.u_tpnp->pn_path, -1, (uid_t)uap->uid, 
			(gid_t)uap->gid));
}

CoChown(fname, uid, gid)
	char	*fname;
	uid_t	uid;
	gid_t	gid;
{
	register struct a {
		char	*fname;
		int	uid;
		int	gid;
	} *uap = (struct a *)u.u_ap;

	Co_OP3(ClChown(uap), fname, uid, gid);
}

ClFchown(uap)
	register struct a {
		int	fdes;
		int	uid;
		int	gid;
	} *uap;
{
	Cl_FDES(fchown(uap), 
		SdChown(rpid, (char *)0, fp->f_rfdes, (uid_t)uap->uid, 
			(gid_t)uap->gid));
}

CoFchown(fdes, uid, gid)
	uid_t	uid;
	gid_t	gid;
{
	register struct a {
		int	fdes;
		int	uid;
		int	gid;
	} *uap = (struct a *)u.u_ap;

	Co_OP3(fchown(uap), fdes, uid, gid);
}

ClTruncate(uap)
	register struct a {
		char	*fname;
		int	length;
	} *uap;
{
	Cl_FNAM(truncate(uap), 
		SdTruncate(u.u_tcurrpid, u.u_tpnp->pn_path, -1, uap->length));
}

CoTruncate(fname, length)
	char *fname;
	unsigned long length;
{
	register struct a {
		char	*fname;
		int	length;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(ClTruncate(uap), fname, length);
}

ClFtruncate(uap)
	register struct a {
		int	fdes;
		int	length;
	} *uap;
{
	Cl_FDES(ftruncate(uap), 
		SdTruncate(rpid, (char *)0, fp->f_rfdes, uap->length));
}

CoFtruncate(fdes, length)
	unsigned long length;
{
	register struct a {
		int	fdes;
		int	length;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(ftruncate(uap), fdes, length);
}

ClTimes(uap)
	register struct a {
		char	*fname;
		struct	timeval *tptr;
	} *uap;
{
	register int err;
	struct timeval tv[2];

	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
#ifdef	SYSV
		if (u.u_procp->p_universe == UNIVERSE_SYSV)
			utime(uap);
		else
#endif	SYSV
		utimes(uap);
		u.u_tok = 0;
		return;
	}
	if ((err = u.u_error) == 0  &&  u.u_tclient)
		err = ENOENT;	/* really shouldn't happen */
	if (err == 0) {
#ifdef	SYSV
	    if (u.u_procp->p_universe == UNIVERSE_SYSV
					&& uap->tptr == (struct timeval *) 0)
		err = SdTimes(u.u_tcurrpid, u.u_tpnp->pn_path, (char *)0, 0);
	    else
#endif	SYSV
	    if (err = copyin(uap->tptr, tv, sizeof(tv)) == 0)
		err = SdTimes(u.u_tcurrpid, u.u_tpnp->pn_path, tv, sizeof tv);
	}
	u.u_error = err;
	TRFS_PNFREE();
}

CoTimes(fname, tptr)
	char *fname;
	struct timeval *tptr;
{
	register struct a {
		char	*fname;
		struct	timeval *tptr;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(ClTimes(uap), fname, tptr);
}

ClRename(uap)
	register struct a {
		char	*from;
		char	*to;
	} *uap;
{
	register struct pathname *pnp = (struct pathname *)0;
	register unsigned long rpid, here = 0;
	struct pathname pn;

	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		if (u.u_error = pn_get(uap->from, UIOSEG_USER, &pn))
			return;
		lookuppn(&pn, FOLLOW_LINK, 
				(struct vnode **)0, (struct vnode **)0);
		pn_free(&pn);
		u.u_tok = 0;
		here += 1;
	} else {
		rpid = u.u_tcurrpid;
		pnp = u.u_tpnp;
		u.u_tpnp = (struct pathname *)0;
	}
	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		if (u.u_error = pn_get(uap->to, UIOSEG_USER, &pn)) {
			if (pnp)
				pn_dfree(pnp);
			return;
		}
		lookuppn(&pn, FOLLOW_LINK, 
				(struct vnode **)0, (struct vnode **)0);
		pn_free(&pn);
		u.u_tok = 0;
		here += 1;
	}
	switch (here) {
	    case 2:
		if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
			rename(uap);
			u.u_tok = 0;
		} else
			panic("ClRename");
		break;
	    case 0:
		if (u.u_tclient)
			u.u_error = ENOENT;	/* really shouldn't happen */
		else if (rpid != u.u_tcurrpid)
			u.u_error = EXDEV;
		else
			u.u_error = SdRename(rpid, pnp->pn_path,
							u.u_tpnp->pn_path);
		break;
	    default:
		u.u_error = EXDEV;
		break;
	}
	if (pnp)
		pn_dfree(pnp);
	TRFS_PNFREE();
}

CoRename(from, to)
	char *from, *to;
{
	register struct a {
		char	*from;
		char	*to;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(ClRename(uap), from, to);
}

/*
 * Don't ignore the return codes from lookuppn.  Very strange things will
 * happen if pn is pointing to a RFS node.
 */
ClLink(uap)
	register struct a {
		char	*from;
		char	*to;
	} *uap;
{
	register struct pathname *pnp = (struct pathname *)0;
	register unsigned long rpid, here = 0;
	struct pathname pn;
	struct vnode *fvp, *tvp;	/* SYSV: for RFS */
	int error;

	fvp = tvp = (struct vnode *)0;
#if	defined(SYSV) && defined(RFS)
	u.u_pdir = NULL;
#endif
	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		if (u.u_error = pn_get(uap->from, UIOSEG_USER, &pn))
			return;
		if (u.u_error = lookuppn(&pn, FOLLOW_LINK,
					 (struct vnode **)0, &fvp))
		  	goto out;
		pn_free(&pn);
		u.u_tok = 0;
		here += 1;
	} else {
		rpid = u.u_tcurrpid;
		pnp = u.u_tpnp;
		u.u_tpnp = (struct pathname *)0;
	}
#if defined(SYSV) && defined(RFS)
	u.u_pdir = fvp;
	if (server())
		goto out;
#endif
	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		if (u.u_error = pn_get(uap->to, UIOSEG_USER, &pn))
		  	goto out;
		if (u.u_error = lookuppn(&pn, FOLLOW_LINK, &tvp,
					 (struct vnode **)0))
		  	goto out;
		pn_free(&pn);
		u.u_tok = 0;
		here += 1;
	}
	if (fvp)
		VN_RELE(fvp);
	if (tvp)
		VN_RELE(tvp);
	switch (here) {
	  case 2:
		if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
			link(uap);
			u.u_tok = 0;
		} else
			panic("ClLink");
		break;
	  case 0:
		if (u.u_tclient)
			u.u_error = ENOENT;	/* really shouldn't happen */
		else if (rpid != u.u_tcurrpid)
			u.u_error = EXDEV;
		else
			u.u_error = SdLink(rpid, pnp->pn_path,
						u.u_tpnp->pn_path, 0);
		break;
	  default:
		u.u_error = EXDEV;
		break;
	}
	if (pnp)
		pn_dfree(pnp);
	TRFS_PNFREE();
	return;
out:
	if (fvp)
	  	VN_RELE(fvp);
	if (pnp)
	  	pn_dfree(pnp);
	pn_free(&pn);
}

ClSymlink(uap)
	register struct a {
		char	*target;
		char	*linkname;
	} *uap;
{
	struct pathname tpn;

	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		symlink(uap);
		u.u_tok = 0;
		return;
	}
	if (u.u_tclient)
		u.u_error = ENOENT;	/* really shouldn't happen */
	else if ((u.u_error = pn_get(uap->target, UIOSEG_USER, &tpn)) == 0) {
		u.u_error = SdLink(u.u_tcurrpid, tpn.pn_path,
						u.u_tpnp->pn_path, 1);
		pn_free(&tpn);
	}
	TRFS_PNFREE();
}

CoLink(target, linkname, sym)
	char *target, *linkname;
{
	register struct a {
		char	*target;
		char	*linkname;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(sym ? ClSymlink(uap) : ClLink(uap), target, linkname);
}

ClMkdir(uap)
	register struct a {
		char	*dirnamep;
		int	dmode;
	} *uap;
{
	Cl_FNAM(mkdir(uap), 
		SdMkdir(u.u_tcurrpid, u.u_tpnp->pn_path, 
			uap->dmode & ~u.u_cmask));
}

CoMkdir(dirnamep, dmode)
	char *dirnamep;
{
	register struct a {
		char	*dirnamep;
		int	dmode;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(ClMkdir(uap), dirnamep, dmode);
}

ClRmdir(uap)
	register struct a {
		char	*dirnamep;
	} *uap;
{
	Cl_FNAM(rmdir(uap), 
		SdRmdir(u.u_tcurrpid, u.u_tpnp->pn_path));
}

CoRmdir(dirnamep)
	char *dirnamep;
{
	register struct a {
		char	*dirnamep;
	} *uap = (struct a *)u.u_ap;

	Co_OP1(ClRmdir(uap), dirnamep);
}

ClMknod(uap)
	register struct a {
		char		*pnamep;
		int		fmode;
		int		dev;
	} *uap;
{
	Cl_FNAM(mknod(uap), 
		SdMknod(u.u_tcurrpid, u.u_tpnp->pn_path,
			uap->fmode & ~u.u_cmask, uap->dev));
}

CoMknod(fname, fmode, dev)
	char *fname;
{
	register struct a {
		char	*fname;
		int	fmode;
		int	dev;
	} *uap = (struct a *)u.u_ap;

	Co_OP3(ClMknod(uap), fname, fmode, dev);
}

ClFsync(uap)
	register struct a {
		int	fdes;
	} *uap;
{
	Cl_FDES(fsync(uap), 
		SdFsync(rpid, fp->f_rfdes));
}

CoFsync(fdes)
{
	register struct a {
		int	fdes;
	} *uap = (struct a *)u.u_ap;

	Co_OP1(fsync(uap), fdes);
}

ClChdir(uap)
	register struct a {
		char *dirnamep;
	} *uap;
{
	ClChdirec(uap, 0);
}

ClChroot(uap)
	register struct a {
		char *dirnamep;
	} *uap;
{
	ClChdirec(uap, 1);
}

ClChdirec(uap, isroot)
	register struct a {
		char *dirnamep;
	} *uap;
{
	register unsigned long rpid, oldrpid, *midp;

	midp = isroot ? &u.u_trrdirmid : &u.u_trcdirmid;
	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		if (isroot)
			chroot(uap);
		else
			chdir(uap);
		u.u_tok = 0;
		if (u.u_error == 0  &&  *midp) {
			if (rpid = RmidToRpid(*midp))
				SdChdir(rpid, "/", isroot, 0);
			*midp = 0;
		}
		return;
	}
	if (u.u_tclient)
		u.u_error = ENOENT;	/* really shouldn't happen */
	else
		u.u_error = SdChdir((rpid = u.u_tcurrpid),
					u.u_tpnp->pn_path, isroot, 1);
	if (u.u_error == 0) {
		if (*midp && (oldrpid = RmidToRpid(*midp)) && oldrpid != rpid)
			SdChdir(oldrpid, "/", isroot, 0);
		*midp = MID(rpid);
	}
	TRFS_PNFREE();
}

CoChdirec(fname, isroot, onoff)
	char *fname;
{
	register struct a {
		char	*fname;
	} *uap = (struct a *)u.u_ap;

	uap->fname = fname;
	if (isroot)
		ClChroot(uap);
	else
		ClChdir(uap);
	if (u.u_error == 0) {
		if (onoff)
			u.u_tdir |= isroot ? TRDIR : TCDIR;
		else
			u.u_tdir &= isroot ? ~TRDIR : ~TCDIR;
	}
	return u.u_error;
}

ClGetdir(uap)
	register struct a {
		int	fdes;
		char	*buf;
		u_int	count;
		long	*basep;
	} *uap;
{
	if (!useracc(uap->buf, uap->count, B_WRITE) ||
	    !useracc(uap->basep, sizeof(long), B_WRITE))
		u.u_error = EFAULT;
	else {
	    Cl_FDES(getdirentries(uap), 
		SdGetdir(rpid, fp->f_rfdes, uap->buf, uap->count, uap->basep,
			sizeof(*uap->basep), &fp->f_offset, &u.u_r.r_val1));
	}
}

CoGetdir(fdes, buf, count, basep, offset, rval)
	char	*buf;
	unsigned count;
	long	*basep, *offset, *rval;
{
	register struct a {
		int	fdes;
		char	*buf;
		u_int	count;
		long	*basep;
	} *uap = (struct a *)u.u_ap;
	register struct file *fp;

	if ((fp = getf(fdes)) == 0)
		return u.u_error;
	fp->f_offset = *offset;
	Co_OP4(ClGetdir(uap); *rval = u.u_r.r_val1; *offset=fp->f_offset, 
		fdes, buf, count, basep);
}

ClStatfs(uap)
	struct a {
		char	*fname;
		struct	statfs *ub;
	} *uap;
{
	if (!useracc(uap->ub, sizeof (struct statfs), B_WRITE))
		u.u_error = EFAULT;
	else {
	    Cl_FNAM(statfs(uap), 
		SdStatfs(u.u_tcurrpid, u.u_tpnp->pn_path, -1, 
			uap->ub, sizeof(struct statfs), UIO_USERSPACE));
	}
}

CoStatfs(fname, ub)
	char *fname;
	struct statfs *ub;
{
	register struct a {
		char	*fname;
		struct	statfs *ub;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(ClStatfs(uap), fname, ub);
}


ClFstatfs(uap)
	struct a {
		int		fdes;
		struct	statfs	*ub;
	} *uap;
{
	if (!useracc(uap->ub, sizeof (struct statfs), B_WRITE))
		u.u_error = EFAULT;
	else {
	    Cl_FDES(fstatfs(uap), 
		SdStatfs(rpid, (char *)0, fp->f_rfdes, 
			uap->ub, sizeof(struct statfs), UIO_USERSPACE));
	}
}

CoFstatfs(fdes, ub)
	struct statfs *ub;
{
	register struct a {
		int		fdes;
		struct statfs	*ub;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(ClFstatfs(uap), fdes, ub);
}

ClFlock(uap)
	struct a {
		int	fdes;
		int	how;
	} *uap;
{
	Cl_FDES(flock(uap),
		SdFlock(rpid, fp->f_rfdes, uap->how));
}

CoFlock(fdes, how)
{
	struct a {
		int	fdes;
		int	how;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(ClFlock(uap), fdes, how);
}

ClLockf(uap)
	register struct a {
		int		fdes;
		int		flag;
		unsigned long	size;
	} *uap;
{
	Cl_FDES(lockf(uap),
		SdLockf(rpid, fp->f_rfdes, uap->flag, uap->size, fp->f_offset));
}

CoLockf(fdes, flag, size, offset)
off_t offset;
{
	register struct a {
		int		fdes;
		int		flag;
		unsigned long	size;
	} *uap = (struct a *)u.u_ap;
	register struct file *fp;

	if ((fp = getf(fdes)) == 0)
		return u.u_error;
	fp->f_offset = offset;
	Co_OP3(ClLockf(uap), fdes, flag, size);
}

ClFcntl(uap)
	register struct a {
		int		fdes;
		int		cmd;
		int		arg;
	} *uap;
{
	register char *buf = (char *) 0;
	register int bufsize = 0;
	register int read = 0;
	register int write = 0;

	switch (uap->cmd) {
		case F_GETLK:
			write++;
		case F_SETLK:
		case F_SETLKW:
			read++;
			bufsize = sizeof(struct flock);
			buf = (char *)uap->arg;
			break;

#ifdef	FOO
/* KLUDGE!!!!!
 * taking this out breaks fcntl's setting APPEND across trfs, keeping it in 
 * breaks X on a diskless.
 */
		case F_GETFL:
		case F_SETFL:
			break;
#endif	FOO

		default:
			fcntl(uap);
			return;

	}
	{
	Cl_FDES(fcntl(uap),
		SdFcntl(rpid, fp->f_rfdes, uap->cmd, uap->arg, fp->f_offset,
				buf, bufsize, read, write));
	}
}

CoFcntl(fdes, cmd, arg, offset)
off_t offset;
{
	register struct a {
		int		fdes;
		int		cmd;
		int		arg;
	} *uap = (struct a *)u.u_ap;
	register struct file *fp;

	if ((fp = getf(fdes)) == 0)
		return u.u_error;
	fp->f_offset = offset;
	Co_OP3(ClFcntl(uap), fdes, cmd, arg);
}

ClUnpBind(unp, nam)
	struct unpcb *unp;
	struct mbuf *nam;
{
	unsigned long rpid;
	int rfd, err;

	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		err = unp_bind(unp, nam);
		u.u_tok = 0;
		return err;
	}
	if (u.u_tclient)
		err = ENOENT;
	else
		err = SdUnpBind((rpid = u.u_tcurrpid), u.u_tpnp->pn_path,
			unp->unp_socket, &rfd);
	if (err == 0) {
		unp->unp_rmid = MID(rpid);
		unp->unp_rfd = rfd;
		unp->unp_vnode = (struct vnode *)(-1);	/* for listen() */
	}
	TRFS_PNFREE();
	return err;
}

CoUnpBind(fname, so, rmid, rfdp)
	char *fname;
	struct socket *so;
	unsigned long rmid;
	int *rfdp;
{
	struct vnode *vp;
	struct vattr vattr;
	register struct file *fp;
	int err;

	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		vattr_null(&vattr);
		vattr.va_type = VSOCK;
		vattr.va_mode = 0777;
		err = vn_create(fname, UIOSEG_USER, &vattr, EXCL, 0, &vp);
		u.u_tok = 0;
		if (err) {
			if (err == EEXIST)
				return (EADDRINUSE);
			return (err);
		}
		if ((fp = falloc()) == NULL) {
			VN_RELE(vp);
			return EADDRINUSE;	/* it is now! */
		}
		fp->f_flag = FREAD|FWRITE;
		fp->f_type = DTYPE_VNODE;
		fp->f_ops = &tremoteops;
		fp->f_data = (caddr_t)vp;
		vp->v_socket = so;
		vp->v_sockrmid = rmid;
		*rfdp = u.u_r.r_val1;
		return 0;
	}
	TRFS_PNFREE();
	return ENOENT;
}

ClUnpDetach(unp)
struct unpcb *unp;
{
	unsigned long rpid;

	if (unp->unp_rmid) {
		if (rpid = RmidToRpid(unp->unp_rmid))
			SdUnpDetach(rpid, unp->unp_rfd);
		unp->unp_rmid = 0;
		unp->unp_rfd = 0;
		unp->unp_vnode = (struct vnode *)0;
	}
	unp_detach(unp);
}

CoUnpDetach(fd)
int fd;
{
	struct a {
		int	fd;
	} *uap = (struct a *)u.u_ap;

	Co_OP1(close(uap), fd);
}

ClUnpConnect(so, nam)
	struct socket *so;
	struct mbuf *nam;
{
	struct socket *so2;
	int err;

	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		err = unp_connect(so, nam);
		u.u_tok = 0;
		return err;
	}
	if (u.u_tclient)
		err = ENOENT;
	else
		err = SdUnpConnect(u.u_tcurrpid, u.u_tpnp->pn_path, &so2);
	TRFS_PNFREE();
	if (err)
		return err;
	if (so->so_type != so2->so_type)
		return EPROTOTYPE;
	if (so->so_proto->pr_flags & PR_CONNREQUIRED &&
	    ((so2->so_options&SO_ACCEPTCONN) == 0 ||
	     (so2 = sonewconn(so2)) == 0))
		return ECONNREFUSED;
	return unp_connect2(so, so2);
}

CoUnpConnect(fname, sop, rmid)
	char *fname;
	struct socket **sop;
	unsigned long rmid;
{
	struct pathname pn;
	struct vnode *vp;
	int err = 0;

	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		err = pn_get(fname, UIOSEG_USER, &pn);
		if (err == 0) {
			err = lookuppn(&pn, FOLLOW_LINK, (struct vnode **)0, &vp);
			pn_free(&pn);
		}
		u.u_tok = 0;
		if (err || vp == (struct vnode *)0)
			return err;
		if (vp->v_type != VSOCK)
			err = ENOTSOCK;
		else if (vp->v_sockrmid != rmid)
			err = EXDEV;
		else
			*sop = vp->v_socket;
		VN_RELE(vp);
		return err;
	}
	TRFS_PNFREE();
	return ENOENT;
}

ClCore()
{
	register int fd, err;
	register unsigned long rpid;

	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		err = core();
		u.u_tok = 0;
		return err;
	}
	rpid = u.u_tcurrpid;
	if (u.u_error == 0 && u.u_tclient)
		err = ENOENT;
	if (err == 0) {
#ifndef	NOMMAP
		for (fd = 0; fd < NOFILE; fd++) 	/* unmap mmap's */
			if (u.u_ofile[fd] && (u.u_pofile[fd] & UF_MAPPED))
				munmapfd(fd);
#endif	NOMMAP
		if ((err = SdCore(rpid, &u, ctob(UPAGES), 0, 1)) == 0  &&
		    (err = SdCore(rpid, ctob(dptov(u.u_procp, 0)),
				ctob(u.u_dsize), ctob(UPAGES), 0)) == 0  &&
		    (err = SdCore(rpid, ctob(sptov(u.u_procp, u.u_ssize - 1)),
				ctob(u.u_ssize), ctob(UPAGES)+ctob(u.u_dsize),
								0)) == 0)
			u.u_acflag |= ACORE;
	}
	TRFS_PNFREE();
	return err;
}

CoCore(base, len, offset, seg)
caddr_t base;
{
	struct vnode *vp = 0;
	struct vattr vattr;

	if (base == 0) {
		vattr_null(&vattr);
		vattr.va_type = VREG;
		vattr.va_mode = 0644;
		u.u_error = vn_create("core", UIOSEG_KERNEL, &vattr, 
		    NONEXCL, VWRITE, &vp);
		if (u.u_error == 0 && vattr.va_nlink != 1)
			u.u_error = EFAULT;
	} else
		u.u_error = lookupname("core", UIOSEG_KERNEL, FOLLOW_LINK,
			(struct vnode **)0, &vp);
	if (u.u_error == 0)
		u.u_error = vn_rdwr(UIO_WRITE, vp, base, len, offset, seg, 
			IO_UNIT, (int *)0);
	if (vp)
		VN_RELE(vp);
	return u.u_error;
}

CoGetlink(name, buf, lenp, followp, cindexp, clenp)
	char *name, *buf;
	int *lenp, *followp, *cindexp, *clenp;
{
	register char *p, *cp, *ep, c;
	struct pathname path, linkpath;
	struct pathname *pnp = &path, *lpnp = &linkpath;
	char component[MAXNAMLEN+1];
	char *mname = 0;
	int index;
	struct vnode *vp = 0;
	struct iovec aiov;
	struct uio auio;

	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		pn_alloc(pnp);
		pn_alloc(lpnp);
		copyin(name, pnp->pn_path, *lenp);
		pnp->pn_pathlen = *lenp;
		mname[0] = 0;
		if (pn_peekchar(pnp) == '/') {
			pn_skipslash(pnp);
			if (pn_peekchar(pnp) == '@') {
				while (pn_peekchar(pnp) == '@')
					pn_getchar(pnp);
				pn_getcomponent(pnp, component);
				for (p = component; *p && *p != '@'; p++)
					;
				if (*p == '@') {
					*p++ = 0;
					mname = p;
				} else
					mname = component;
			} else
				pn_ungetchar(pnp);
		}
		if (diskless) {
			lpnp->pn_path[0] = '/';
			lpnp->pn_path[1] = '@';
			bcopy(servername, &lpnp->pn_path[2], 
				servernamelen);
			lpnp->pn_path[servernamelen+2] = '@';
			bcopy(hostname, &lpnp->pn_path[servernamelen+3], 
				hostnamelen);
			*lenp = servernamelen + hostnamelen + 3;
			lpnp->pn_path[*lenp] = 0;
			*followp = FOLLOW_LINK;
			*cindexp = 0;
			*clenp = pnp->pn_path - pnp->pn_buf;
			copyout(lpnp->pn_path, buf, *lenp);
			u.u_tok = 0;
			pn_free(pnp);
			pn_free(lpnp);
			return (0);
		}
		p = pnp->pn_path;
		ep = &pnp->pn_path[pnp->pn_pathlen];
		*lenp = 0;
		while (p < ep) {
			for (cp = p; (cp < ep) && (*cp != '/'); cp++)
				;
			c = *cp;
			*cp = 0;
			if (lookupname(pnp->pn_path, UIOSEG_KERNEL, 
				    NO_FOLLOW, (struct vnode *)0, &vp))
			    break;
			*cp = c;
			if (vp->v_type == VLNK) {
				aiov.iov_base = lpnp->pn_path;
				aiov.iov_len = MAXPATHLEN;
				auio.uio_iov = &aiov;
				auio.uio_iovcnt = 1;
				auio.uio_offset = 0;
				auio.uio_seg = UIOSEG_KERNEL;
				auio.uio_resid = MAXPATHLEN;
				if (VOP_READLINK(vp, &auio, u.u_cred))
					break;
				lpnp->pn_pathlen = MAXPATHLEN - auio.uio_resid;
				VN_RELE(vp);
				pn_macroexp(lpnp, mname, 2);
				if (pn_peekchar(lpnp) == '/')
					p = pnp->pn_buf;
				*lenp = lpnp->pn_pathlen;
				lpnp->pn_path[*lenp] = 0;
				*cindexp = p - pnp->pn_buf;
				*clenp = cp - p;
				copyout(lpnp->pn_path, buf, lpnp->pn_pathlen);
				u.u_tok = 0;
				pn_free(pnp);
				pn_free(lpnp);
				return (0);
			} else
				p = cp + 1;
			VN_RELE(vp);
			vp = 0;
		}
		if (vp)
			VN_RELE(vp);
		pn_free(pnp);
		pn_free(lpnp);
		u.u_tok = 0;
		return (0);
	} else {
		TRFS_PNFREE();
		return ENOENT;		/* really shouldn't happen */
	}
}

/* ------------------------- fileop calls ---------------------------- */

ClClose(fp)
	struct file *fp;
{
	unsigned long rpid = fp->f_rmid;
	int rfdes = fp->f_rfdes;

	fp->f_rmid = 0;
	if (fp->f_data)
		vno_close(fp);
	if (u.u_error  ||  (u.u_procp->p_flag & SWEXIT))
		return 0;			/* cousins already exitted */
	if (rpid = RmidToRpid(rpid))
		u.u_error = SdClose(rpid, rfdes);
	else
		u.u_error = EIO;
	return u.u_error;
}

CoClose(fdes)
{
	register struct a {
		int	fdes;
	} *uap = (struct a *)u.u_ap;

	Co_OP1(close(uap), fdes);
}

ClRw(fp, rw, uio)
	register struct file *fp;
	enum uio_rw rw;
	register struct uio *uio;
{
	unsigned long rpid;
	int count = uio->uio_resid;
	int err;

	if (rpid = RmidToRpid(fp->f_rmid)) {
		if (!useracc(uio->uio_iov->iov_base, count, 
					rw==B_READ?B_WRITE:B_READ))
			return EFAULT;
		u.u_tuio = uio;		/* for use with readv, writev */
		if ((err = SdRw(rpid, fp->f_rfdes, rw, uio->uio_iov->iov_base,
		    &uio->uio_offset, &uio->uio_resid, uio->uio_segflg,
		    u.u_rlimit[RLIMIT_FSIZE].rlim_cur,
		    fp->f_flag & FNONIDEM)) == EFBIG)
#ifdef	SYSV
			if (u.u_procp->p_universe == UNIVERSE_BSD)
#endif	SYSV
			   psignal(u.u_procp, SIGXFSZ);
#ifdef	SYSV
		if (u.u_procp->p_universe == UNIVERSE_SYSV && 
					uio->uio_offset < 0) {	/* KLUDGE */
			uio->uio_offset = -uio->uio_offset;
			fp->f_offset = uio->uio_offset - 
					(count - uio->uio_resid);
		}
#endif	SYSV
		return err;
	}
	return EIO;
}

CoRw(fdes, rw, base, offsetp, residp, segflg, maxfsize)
	enum uio_rw rw;
	char *base;
	unsigned long *offsetp, *residp;
{
	register struct a {
		int		fdes;
		char		*cbuf;
		unsigned long	count;
	} *uap = (struct a *)u.u_ap;
	register struct file *fp;
	register unsigned long count;
	struct uio auio;
	struct iovec aiov;
	int err;

	u.u_rlimit[RLIMIT_FSIZE].rlim_cur = maxfsize;
  restart:
	count = *residp;
	auio.uio_iov = &aiov;
	auio.uio_iovcnt = 1;
	auio.uio_segflg = segflg;
	auio.uio_offset = *offsetp;
	aiov.iov_base = uap->cbuf = base;
	auio.uio_resid = aiov.iov_len = uap->count = count;
	if (fp = getf(uap->fdes = fdes)) {
		if ((fp->f_flag & (rw == UIO_READ ? FREAD : FWRITE)) == 0)
			return EBADF;
		if (auio.uio_resid < 0)
			return EINVAL;
#ifdef	notyet
		if ((u.u_procp->p_flag&SOUSIG) == 0  &&  setjmp(&u.u_qsave))
			if (auio.uio_resid == count)
					goto restart;	/* what about errors? */
#endif	notyet
		{
			extern int vno_rw();

			if (fp->f_ops->fo_rw != vno_rw)
				panic("TRFS: CoRw");
		}
#ifdef	SYSV
		if ((u.u_procp->p_universe == UNIVERSE_SYSV) 
		    && (fp->f_type == DTYPE_VNODE) 
		    && (((struct vnode *)fp->f_data)->v_type == VDIR)) {
			auio.uio_offset_sysv = fp->f_offset_sysv;
			auio.uio_offset_buf = fp->f_offset_buf;
			/* 1 signifies compatability */
			err = VOP_S5GETDENTS((struct vnode *)fp->f_data,
							&auio, fp->f_cred, 1);
			fp->f_offset_sysv = auio.uio_offset_sysv;
			fp->f_offset_buf = auio.uio_offset_buf;
			fp->f_offset = auio.uio_offset;
			*residp = auio.uio_resid;
			*offsetp = -auio.uio_offset;
			return err;
		}
#endif	SYSV
		err = (*fp->f_ops->fo_rw)(fp, rw, &auio);
		fp->f_offset = *offsetp + count - (*residp = auio.uio_resid);
		return err;
	}
	return u.u_error;
}

ClIoctl(fp, com, data)
	struct file *fp;
	char *data;
{
	unsigned long rpid;

	if (rpid = RmidToRpid(fp->f_rmid)) {
#ifdef	SYSV
		if (u.u_procp->p_universe == UNIVERSE_SYSV)
			return SdSysv_Ioctl(rpid, fp->f_rfdes, com, data);
		else
#endif	SYSV
		return SdIoctl(rpid, fp->f_rfdes, com, data, IOCPARM_MASK+1);
	}
	return EIO;
}

CoIoctl(fdes, cmd, cmarg)
	caddr_t cmarg;
{
	register struct a {
		int	fdes;
		int	cmd;
		caddr_t	cmarg;
	} *uap = (struct a *)u.u_ap;

#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV) {
		Co_OP3(sysv_ioctl(uap), fdes, cmd, cmarg);
	} else {
		Co_OP3(ioctl(uap), fdes, cmd, cmarg);
	}
#else	SYSV
	Co_OP3(ioctl(uap), fdes, cmd, cmarg);
#endif	SYSV
}

ClSelect(fp, flag)
	struct file *fp;
{
	unsigned long rpid;

	if (rpid = RmidToRpid(fp->f_rmid))
		return SdSelect(rpid, fp->f_rfdes, flag);
	return 0;
}

CoSelect(fdes, flag)
{
	register struct file *fp;

	if ((fp = getf(fdes)) == 0)
		return 0;
	else
		return (*fp->f_ops->fo_select)(fp, flag);
}

/* ------------------------- swap calls ---------------------------- */

/* ClSwap: client side of swap is generated in ../is68kdev/rd.c */

CoSwap(rpid, rw, addr, len, blkno)
	unsigned long rpid, len, blkno;
	enum uio_rw rw;
	char *addr;
{
	struct vnode *vp, *GetSwapVp();
	int err;

	if ((vp = GetSwapVp(rpid)) == (struct vnode *)0) {
		if (err = SwapInit(rpid))
			return err;
		if ((vp = GetSwapVp(rpid)) == (struct vnode *)0)
			return EIO;
	}
	return vn_rdwr(rw, vp, addr, len, dbtob(blkno), UIOSEG_USER, 
		IO_UNIT, (int *)0);
}

#ifdef	SYSV
ClSysv_Open(uap)
	register struct a {
		char	*fname;
		int	mode;
		int	crtmode;
	} *uap;
{
	register unsigned i;

	/*
	 * Some jugglery with mode bits as they are not the same
	 * in System V and 4.3 Bsd
	 */
	uap->mode &= ~O_SYNC;	/* Not yet implemented */
	i = uap->mode & SYSV_BITS;
	uap->mode &= ~SYSV_BITS;
	uap->mode |= (i << 1);
	ClCopen(uap->fname, uap->mode - FOPEN, uap->crtmode);
}

ClSysv_Chroot(uap)
	register struct a {
		char *dirnamep;
	} *uap;
{
	if (!suser())
		return;
	ClChdirec(uap, 1);
}

ClSysv_Stat(uap)
	register struct a {
		char	*fname;
		struct sysv_stat *sb;
	} *uap;
{

	if (!useracc(uap->sb, sizeof (struct sysv_stat), B_WRITE))
		u.u_error = EFAULT;
	else {
	    Cl_FNAM(sysv_stat(uap), 
		SdSysv_Stat(u.u_tcurrpid, u.u_tpnp->pn_path, 0, -1,
			uap->sb, sizeof(struct sysv_stat), UIO_USERSPACE));
	}
}

CoSysv_Stat(fname, sb)
	char	*fname;
	struct sysv_stat *sb;
{
	register struct a {
		char	*fname;
		struct	sysv_stat *sb;
	} *uap = (struct a *)u.u_ap;

      Co_OP2(ClSysv_Stat(uap), fname, sb);
}

ClSysv_Fstat(uap)
	register struct a {
		int	fdes;
		struct sysv_stat *sb;
	} *uap;
{
	if (!useracc(uap->sb, sizeof (struct sysv_stat), B_WRITE))
		u.u_error = EFAULT;
	else {
	    Cl_FDES(sysv_fstat(uap), 
		SdSysv_Stat(rpid, (char *)0, 0, fp->f_rfdes,
			uap->sb, sizeof(struct sysv_stat), UIO_USERSPACE));
	}
}

CoSysv_Fstat(fdes, sb)
	struct sysv_stat *sb;
{
	register struct a {
		int	fdes;
		struct	sysv_stat *sb;
	} *uap = (struct a *)u.u_ap;

	Co_OP2(sysv_fstat(uap), fdes, sb);
}

ClSysv_Statfs(uap)
	register struct a {
		char	*fname;
		struct	sysv_statfs *sb;
		int	len;
		int	fstyp;
	} *uap;
{
	if (!useracc(uap->sb, sizeof (struct sysv_statfs), B_WRITE))
		u.u_error = EFAULT;
	else {
	    Cl_FNAM(sysv_statfs(uap), 
		SdSysv_Statfs(u.u_tcurrpid, u.u_tpnp->pn_path, -1,
			uap->len, uap->fstyp,
			uap->sb, sizeof(struct sysv_statfs), UIO_USERSPACE));
	}
}

CoSysv_Statfs(fname, sb, len, fstyp)
	char *fname;
	struct sysv_statfs *sb;
{
	register struct a {
		char	*fname;
		struct	sysv_statfs *sb;
		int	len;
		int	fstyp;
	} *uap = (struct a *)u.u_ap;

	Co_OP4(ClSysv_Statfs(uap), fname, sb, len, fstyp);
}

ClSysv_Fstatfs(uap)
	register struct a {
		int	fdes;
		struct	sysv_statfs *sb;
		int	len;
		int	fstyp;
	} *uap;
{
	if (!useracc(uap->sb, sizeof (struct sysv_statfs), B_WRITE))
		u.u_error = EFAULT;
	else {
	    Cl_FDES(sysv_statfs(uap), 
		SdSysv_Statfs(rpid, (char *)0, 0, fp->f_rfdes,
			uap->len, uap->fstyp,
			uap->sb, sizeof(struct sysv_stat), UIO_USERSPACE));
	}
}

CoSysv_Fstatfs(fdes, sb, len, fstyp)
	struct sysv_statfs *sb;
{
	register struct a {
		int	fdes;
		struct	sysv_statfs *sb;
		int	len;
		int	fstyp;
	} *uap = (struct a *)u.u_ap;

	Co_OP4(ClSysv_Fstatfs(uap), fdes, sb, len, fstyp);
}

ClGetdents(uap)
	register struct a {
		int fdes;
		char *buf;
		int nbytes;
	} *uap;
{
	if (!useracc(uap->buf, uap->nbytes, B_WRITE))
		u.u_error = EFAULT;
	else {
	    Cl_FDES(getdents(uap), 
		SdGetdents(rpid, fp->f_rfdes, uap->buf, uap->nbytes,
			&fp->f_offset_sysv, &u.u_r.r_val1));
	}
}

CoGetdents(fdes, buf, nbytes, offset, rval)
	char	*buf;
	long	*offset, *rval;
{
	register struct a {
		int	fdes;
		char	*buf;
		int	nbytes;
	} *uap = (struct a *)u.u_ap;
	register struct file *fp;

	if ((fp = getf(fdes)) == 0)
		return u.u_error;
	sysv_to_bsd ((struct vnode *)fp->f_data, *offset, fp);
	Co_OP3(ClGetdents(uap); *rval = u.u_r.r_val1;
	       *offset = fp->f_offset_sysv, fdes, buf, nbytes);
}

extern char sysv_mapfcntl[];

ClSysv_Fcntl(uap)
	register struct a {
		int		fdes;
		int		cmd;
		int		arg;
	} *uap;
{
	uap->cmd = sysv_mapfcntl[uap->cmd];

	ClFcntl(uap);
	if (u.u_error == EDEADLK)
		u.u_error = EDEADLK_SV;
	else if (u.u_error == ENOLCK)
		u.u_error = ENOLCK_SV;
}

sysGetRpid(uap)
	register struct a {
		int		fdes;
	} *uap;
{
	register struct file *fp;

	GETF(fp, uap->fdes);
	if (diskless && (fp->f_rmid == MID(servermid)))
	  u.u_r.r_val1 = 0;
	else
	  u.u_r.r_val1 = RmidToRpid(fp->f_rmid);
}

ClSysv_Lseek(uap)
	register struct a {
		int	fdes;
		off_t	off;
		int	sbase;
	} *uap;
{
	Cl_FDES(sysv_lseek(uap), 
		u.u_error = SdSysv_Lseek(rpid, fp->f_rfdes, uap->off,
			uap->sbase, &u.u_r.r_off, &fp->f_offset));
}

CoSysv_Lseek(fdes, off, sbase, voffsetp, foffsetp)
	off_t	off;
	off_t	*voffsetp, *foffsetp;
{
	register struct a {
		int	fdes;
		off_t	off;
		int	sbase;
	} *uap = (struct a *)u.u_ap;
	register struct file *fp;

	if ((fp = getf(fdes)) == 0)
		return u.u_error;
	fp->f_offset = *foffsetp;
	Co_OP3(ClSysv_Lseek(uap); *voffsetp = u.u_r.r_off;
				*foffsetp = fp->f_offset, fdes, off, sbase);
}
#endif	SYSV
