#include "param.h"
#include "user.h"
#include "kernel.h"
#include "proc.h"
#include "vnode.h"
#include "../machine/board.h"
#include "../wipc/wipc.h"
#include "../trfs/trfs.h"

/*
 * Called from kernel initialization code. Starts up a new system process 
 * which loops forever, creating cousins as needed.
 */
TrfsServer()
{
	register unsigned long rpid, newrpid;
	register struct wmsg *mp;
	register struct proc *p;
	struct wmsg wmsg;
	extern int server_pid;

	if (newproc(0) == 0)
		return;
	u.u_cred = crcopy(u.u_cred);
	p = u.u_procp;
	p->p_flag |= SLOAD|SSYS;
	p->p_pgrp = p->p_pid;
	setctx(p->p_context = CTX_SYS);
	strcpy(u.u_comm, "TRFS server");
	server_pid = p->p_pid;
	TrfsWatchman();
	EnableRecQueue();
	EnableRepQueue();
	mp = &wmsg;
	bzero(mp, sizeof *mp);
	while (1) {
		while (setjmp(&u.u_qsave))
			ServerSig(p);
		u.u_eosys = JUSTRETURN;
		u.u_error = 0;
		u.u_tsrcxs = u.u_tdstxs = XS_USER;
		u.u_tclient = rpid = xReceive(mp, 0);
		switch(mp->h.rcode) {
		  case RQ_CONNECT:
			/* mp->h.rcode = 0;	/* don't clear rcode! */
			GotConnect(mp, rpid);
			break;
		  case RQ_DISCONNECT:
			/* mp->h.rcode = 0;	/* don't clear rcode! */
			GotDisconnect(mp, rpid);
			break;
		  case RQ_ALIVE:
			mp->h.rcode = 0;
			SvIsAlive(mp, rpid);
			break;
		  case RQ_SWAP:
			mp->h.rcode = 0;
			SvSwap(mp, rpid);
			break;
		  case RQ_GET_IN_ADDR:
			mp->h.rcode = 0;
			SvINAddr(mp, rpid);
			break;
		  default:
			if (newrpid = NewCousin(mp, rpid))
				xForward(rpid, newrpid);
			else {
				printf("TRFS: request %d: cant find cousin: @%s %d\n", 
					mp->h.rcode, RpidToName(rpid), PID(rpid));
				mp->h.rcode = ESRCH;
			}
			break;
		}
		xReply(mp, rpid);
	}
}

ServerSig(p)
	register struct proc *p;
{
	if (p->p_cursig  ||  ISSIG(p)) {
		switch (p->p_cursig) {
		    default:
/*			printf("TRFS: server signal %d\n", p->p_cursig); /**/
			break;
		}
		p->p_cursig = 0;
		p->p_pri = p->p_usrpri;
	}
}

WatchmanTimeout(p)
	register struct proc *p;
{
	psignal(p, SIGALRM);
	timeout(WatchmanTimeout, p, 60*hz);
}

TrfsWatchman()
{
	register struct proc *p;

	if (newproc(0) == 0)
		return;			/* parent */
	u.u_cred = crcopy(u.u_cred);
	p = u.u_procp;
	p->p_flag |= SLOAD|SSYS;
	p->p_pgrp = p->p_pid;
	setctx(p->p_context = CTX_SYS);
	strcpy(u.u_comm, "TRFS watchman");
	u.u_signal[SIGALRM] = SIG_CATCH;
	timeout(WatchmanTimeout, p, 60*hz);
	while (1) {
		while (setjmp(&u.u_qsave))
			if (p->p_cursig  ||  ISSIG(p)) {
				switch (p->p_cursig) {
				    case SIGALRM:
					MachineSynch(hostname);
					break;

				    default:
					break;
				}
				p->p_cursig = 0;
				p->p_pri = p->p_usrpri;
			}
		sleep((caddr_t)&u, PSLEP);
	}
}

NewCousin(mp, rpid)
	struct wmsg *mp;
	unsigned long rpid;
{
	int pid;

	if ((mp->h.rflags & RF_CRCOUSIN)  &&
	    ((pid = FindCousin(rpid))  ||  (pid = MakeNewCousin(rpid))))
		return SELF_MID | (unsigned long)pid;
	return 0;
}

MakeNewCousin(rpid)
	unsigned long rpid;
{
	fork1(0);
	if (u.u_r.r_val2) {			/* child */
		/*
		 * What about the SSYS bit?  Are we sure we won't get a context
		 *  fault if we're swapped before we get here?  Will the new
		 *  cousin subsequently be swappable?
		 */
		u.u_cred = crcopy(u.u_cred);
		setctx(u.u_procp->p_context = CTX_SYS);
		u.u_procp->p_flag &= ~SSYS;
		SetCousinRpid(u.u_procp->p_pid, rpid);
		u.u_procp->p_pgrp = SetPgrp(rpid, u.u_procp->p_pid);
		u.u_tclient = rpid;
		sprintf(u.u_comm, "@%s %d", RpidToName(rpid), PID(rpid));
		RunAway();
		Cousin();
		exit(0);
	}
	if (u.u_error)
		return 0;
	SetCousinRpid(u.u_r.r_val1, rpid);
	return u.u_r.r_val1;
}

Cousin()
{
	register int alive = 1;
	register struct wmsg *mp;
	register unsigned long rpid;
	unsigned char rcode;
	struct wmsg wmsg;

	u.u_ap = u.u_arg;
	mp = &wmsg;
	while (alive) {
		if (setjmp(&u.u_qsave)) {
			if (CousinSig(u.u_procp))
				alive = 0;
			continue;
		}
		u.u_eosys = JUSTRETURN;
		u.u_error = 0;
		u.u_tsrcxs = u.u_tdstxs = XS_USER;
		rpid = xReceive(mp, 0);
#ifdef	SYSV
		if (mp->h.rflags & RF_UNIVERSE_SYSV)
			u.u_procp->p_universe = UNIVERSE_SYSV;
		else
			u.u_procp->p_universe = UNIVERSE_BSD;
#endif	SYSV
		if (rpid != u.u_tclient) {
			/* check for intruder? */
			u.u_tclient = rpid;
			sprintf(u.u_comm, "@-%s %d", 
				RpidToName(rpid), PID(rpid));
		}
		rcode = mp->h.rcode;
		mp->h.rcode = 0;
#ifdef	SYSV
		if (u.u_texec && rcode != RQ_LOAD
				&& rcode != RQ_PLOAD && rcode != RQ_GETHEAD) {
#else	SYSV
		if (u.u_texec && rcode != RQ_LOAD) {
#endif	SYSV
			VN_RELE(u.u_texec);
			u.u_texec = NULL;
		}
		switch(rcode) {
		  case RQ_EXIT:
			alive = 0;
			SvExit(mp, rpid);
			break;
		  case RQ_FORK:
			if (SvFork(mp, rpid) == 0) {
				mp->h.rflags = 0;	/* clear RF_NONIDEM */
				continue;		/* child: no reply */
			}
			break;
		  case RQ_CHEXEC:
			SvChexec(mp, rpid);
			break;
		  case RQ_LOAD:
			SvLoad(mp, rpid);
			break;
		  case RQ_OPEN:
			SvOpen(mp, rpid);
			break;
		  case RQ_CLOSE:
			SvClose(mp, rpid);
			break;
		  case RQ_RW:
			SvRw(mp, rpid);
			break;
		  case RQ_STAT:
			SvStat(mp, rpid);
			break;
		  case RQ_ACCESS:
			SvAccess(mp, rpid);
			break;
		  case RQ_RDLINK:
			SvReadlink(mp, rpid);
			break;
		  case RQ_UNLINK:
			SvUnlink(mp, rpid);
			break;
		  case RQ_CHMOD:
			SvChmod(mp, rpid);
			break;
		  case RQ_CHOWN:
			SvChown(mp, rpid);
			break;
		  case RQ_TRUNC:
			SvTruncate(mp, rpid);
			break;
		  case RQ_TIMES:
			SvTimes(mp, rpid);
			break;
		  case RQ_RENAME:
			SvRename(mp, rpid);
			break;
		  case RQ_LINK:
			SvLink(mp, rpid);
			break;
		  case RQ_MKDIR:
			SvMkdir(mp, rpid);
			break;
		  case RQ_RMDIR:
			SvRmdir(mp, rpid);
			break;
		  case RQ_MKNOD:
			SvMknod(mp, rpid);
			break;
		  case RQ_IOCTL:
			SvIoctl(mp, rpid);
			break;
		  case RQ_SELECT:
			SvSelect(mp, rpid);
			break;
		  case RQ_FSYNC:
			SvFsync(mp, rpid);
			break;
		  case RQ_CHDIR:
			SvChdir(mp, rpid);
			break;
		  case RQ_FLOCK:
			SvFlock(mp, rpid);
			break;
		  case RQ_LOCKF:
			SvLockf(mp, rpid);
			break;
		  case RQ_GETDIR:
			SvGetdir(mp, rpid);
			break;
		  case RQ_FCNTL:
			SvFcntl(mp, rpid);
			break;
		  case RQ_STATFS:
			SvStatfs(mp, rpid);
			break;
		  case RQ_UNP_BIND:
			SvUnpBind(mp, rpid);
			break;
		  case RQ_UNP_CONNECT:
			SvUnpConnect(mp, rpid);
			break;
		  case RQ_UNP_DETACH:
			SvUnpDetach(mp, rpid);
			break;
#ifdef	SYSV
		  case RQ_SYSV_STATFS:
			SvSysv_Statfs(mp, rpid);
			break;
		  case RQ_GETDENTS:
			SvGetdents(mp, rpid);
			break;
		  case RQ_SYSV_IOCTL:
			SvSysv_Ioctl(mp, rpid);
			break;
		  case RQ_SYSV_LSEEK:
			SvSysv_Lseek(mp, rpid);
			break;
		  case RQ_PLOAD:
			SvPload(mp, rpid);
			break;
		  case RQ_GETHEAD:
			SvGethead(mp, rpid);
			break;
		  case RQ_SYSV_STAT:
			SvSysv_Stat(mp, rpid);
			break;
#endif	SYSV
		  case RQ_GETLINK:
			SvGetlink(mp, rpid);
			break;
		  default:
			printf("TRFS: unknown request %d?\n", rcode);
			mp->h.rcode = EIO;
			break;
		}
		xReply(mp, rpid);
	}
}

CousinSig(p)
	register struct proc *p;
{
	if (p->p_cursig  ||  ISSIG(p)) {
		switch (p->p_cursig) {
		  default:
/*			printf("TRFS: cousin signal %d\n", p->p_cursig); /**/
			break;
		}
		p->p_cursig = 0;
		p->p_pri = p->p_usrpri;
	}
	return 1;	/* die */
}

TrfsInit()
{
	DisableRecQueue();
	DisableRepQueue();
	wloop_attach();
}

TrfsShutdown()
{
	WipcShutdown();
}

Tcopyin(src, dest, count)
	char *src, *dest;
{
	return xCopyFrom(u.u_tclient, src, dest, count, u.u_tsrcxs, XS_KERN);
}

Tcopyout(src, dest, count)
	char *src, *dest;
{
	return xCopyTo(u.u_tclient, src, dest, count, XS_KERN, u.u_tdstxs);
}

Tfubyte(addr)
	char *addr;
{
	unsigned char c;

	if (Tcopyin(addr, &c, sizeof c))
		return -1;
	return c;
}

Tfuword(addr)
	char *addr;
{
	long w;

	if (Tcopyin(addr, &w, sizeof w))
		return -1;
	return w;
}

Tsubyte(addr, c)
	char *addr;
	char c;
{
	if (Tcopyout(&c, addr, sizeof c))
		return -1;
	return 0;
}

Tsuword(addr, w)
	char *addr;
	long w;
{
	if (Tcopyout(&w, addr, sizeof w))
		return -1;
	return 0;
}

Tcopyinstr(src, dest, maxlen, len)
	char *src, *dest;
	int *len;
{
	register int i, v = 1;

	for (i = 0 ; i < maxlen && v; i++) {
		if ((v = Tfubyte(src++)) == -1)
			return EFAULT;
		*dest++ = v;
	}
	*len = i;
	if (i == maxlen)
		return ENOENT;
	return 0;
}

Tcopyoutstr(src, dest, maxlen, len)
	char *src, *dest;
	int *len;
{
	register int i, v = 1;

	for (i = 0 ; i < maxlen && v; i++) {
		v = *src++;
		if (Tsubyte(dest++, v) == -1)
			return EFAULT;
	}
	*len = i;
	if (i == maxlen)
		return ENOENT;
	return 0;
}

Tuseracc(addr, count, mode)
	char *addr;
{
	return 1;
}
