#ifdef	SYSV
#include "../h/types.h"
#include "../machine/pte.h"
#include "param.h"
#include "systm.h"
#include "map.h"
#include "user.h"
#include "proc.h"
#include "text.h"
#include "buf.h"
#include "seg.h"
#include "vm.h"
#include "cmap.h"
#include "../sysv/sys/region.h"
#include "../sysv/sys/ipc.h"
#include "../sysv/sys/shm.h"
#include "../sysv/sys/sysinfo.h"

#ifdef	is68k
#include "../machine/board.h"
#endif	is68k

reg_t	rfree, nullregion, *region;
preg_t	prfree, nullpregion, *pregion;
int	nregion, npregion, pregpp;
struct shlbinfo shlbinfo;

/*
 * Initilize region related global parameters.
 * Since ntext is adjusted for the shared regions,
 * this routine must be invoked before ntext is used.
 */
reg_varinit()
{
	pregpp = 1 + shmseg + shlbmax;
	npregion = pregpp * nprocsh;
	nregion = shlbmax * nprocsh;
	shlbinfo.shlbs = shlbmax;
	shminfo.shmmax = shmmax;
	shminfo.shmmin = shmmin;
	shminfo.shmmni = shmmni;
	shminfo.shmseg = shmseg;
	shminfo.shmall = shmall;
	ntext += (shlbmax + shmmni);
}

void
reg_init()
{
	register reg_t		*rp;

	rfree.r_forw = &rfree;
	rfree.r_back = &rfree;

	for (rp = region; rp < &region[nregion] ; rp++) {
		rp->r_back = rfree.r_back;
		rp->r_forw = &rfree;
		rfree.r_back->r_forw = rp;
		rfree.r_back = rp;
	}
}

reg_t *
reg_alloc()
{
	register reg_t *rp;

	if ((rp = rfree.r_forw) == &rfree) {
		u.u_error = EAGAIN;
		return(NULL);
	}
	/*
	 * Remove from free list
	 */
	rp->r_back->r_forw = rp->r_forw;
	rp->r_forw->r_back = rp->r_back;
	*rp = nullregion;
	return(rp);
}

void
reg_free(rp)
register reg_t *rp;
{
	if (rp != NULL) {
		rp->r_back = rfree.r_back;
		rp->r_forw = &rfree;
		rfree.r_back->r_forw = rp;
		rfree.r_back = rp;
	}
}

void
preg_init()
{
	register preg_t		*prp;

	prfree.p_forw = &prfree;
	prfree.p_back = &prfree;

	for (prp = pregion; prp < &pregion[npregion] ; prp += pregpp) {
		prp->p_back = prfree.p_back;
		prp->p_forw = &prfree;
		prfree.p_back->p_forw = prp;
		prfree.p_back = prp;
	}
}

preg_t *
preg_alloc()
{
	register preg_t *prp;
	register int i;

	if ((prp = prfree.p_forw) == &prfree) {
		u.u_error = EAGAIN;
		return(NULL);
	}
	/*
	 * Remove from free list
	 */
	prp->p_back->p_forw = prp->p_forw;
	prp->p_forw->p_back = prp->p_back;
	for (i = 0 ; i < pregpp ; i++)
		prp[i] = nullpregion;
	return(prp);
}

void
preg_free(prp)
register preg_t *prp;
{
	if (prp != NULL) {
		prp->p_back = prfree.p_back;
		prp->p_forw = &prfree;
		prfree.p_back->p_forw = prp;
		prfree.p_back = prp;
	}
}

/*
 * Free regions structures.
 */
preg_freeregs(p)
register struct proc *p;
{
	register preg_t *prp = p->p_region;

	if (prp != NULL) {
		while (prp->p_type != PT_UNUSED) {
			if (prp->p_type == PT_LIBDAT)
				reg_free(prp->p_pp);
			prp++;
		}
	}
}


/*
 * Locate process region for a given virtual page.
 */
preg_t *
preg_vtopreg(p, vp)
register struct proc *p;
register u_int vp;
{
	register preg_t *prp;
	register caddr_t hi, lo, vaddr = ptob(vp);

	if (p->p_region) {
		for (prp = p->p_region; prp->p_type != PT_UNUSED ; prp++) {
			if (prp->p_pp == NULL) 
				continue;
			lo = prp->p_regva;
			if (prp->p_type == PT_LIBDAT)
				hi = lo +(unsigned long)ptob(prp->p_pp->r_pgsz);
			else
				hi = lo +(unsigned long)ptob(prp->p_ps->x_size);
			if ((unsigned long)vaddr >= (unsigned long)lo
			  && (unsigned long)vaddr < (unsigned long)hi)
				return(prp);
		}
	}
	return(NULL);
}

/*
 * Locate process region pte for a given virtual page.
 */
struct pte *
preg_vtopte(p, vp)
register struct proc *p;
register u_int vp;
{
	register preg_t *prp;
	register reg_t *rp;

	if (prp = preg_vtopreg(p, vp))
		return(&prp->p_pte[vp - btop(prp->p_regva)]);
	else
		return(NULL);
}


/*
 * Locate process region for a given pte
 */
preg_t *
preg_ptetopreg(p, pte)
register struct proc *p;
register struct pte *pte;
{
	register preg_t *prp;
	register struct pte *lo, *hi;

	if (p->p_region) {
		for (prp = p->p_region; prp->p_type != PT_UNUSED; prp++) {
			if (prp->p_pp == NULL)
				continue;
			lo = prp->p_pte;
			if (prp->p_type == PT_LIBDAT)
				hi = lo + prp->p_pp->r_pgsz;
			else
				hi = lo + prp->p_ps->x_size;
			if (pte >= lo && pte < hi)
				return(prp);
		}
	}
	return(NULL);
}


/*
 * Locate virtual page number associated with a pte.
 */
preg_ptetov(p, pte)
register struct proc *p;
register struct pte *pte;
{
	register preg_t *prp;
	register reg_t *rp;

	if (prp = preg_ptetopreg(p, pte))
		return(pte - &prp->p_pte[0] + btop(prp->p_regva));
	else
		return(NULL);
}

/*
 * Attach a region to a process' address space
 * The pregion slot for the region is initialized.
 * Page table is also allocated.
 */
preg_t *
preg_attachreg(up, rp, vaddr, type, prot, size)
struct user *up;
register reg_t *rp;	/* pointer to region to be attached */
caddr_t	vaddr;		/* virtual address to attach at */
int	type;		/* Type to make the pregion. */
int	prot;		/* permissions for segment table entries. */
int	size;		/* size of the region */
{
	register preg_t *prp1, *first;
	register preg_t *prp;
	register struct proc *p = up->u_procp;
	register long a;
	register int i;
	preg_t  tmp;


	/*	Check attach address.
	 *	It must be segment aligned.
	 */

	 if (((u_int) vaddr >= USRSTACK) || ((int)vaddr & SOFFMASK)) {
		u.u_error = EINVAL;
		return(NULL);
	 }

	/*	Allocate a pregion.  We should always find a
	 *	free pregion because of the way the system
	 *	is configured.
	 */

	prp = p->p_region;
	while (prp->p_type != PT_UNUSED)
		prp++;
	
	if (prp == &p->p_region[pregpp]) {
		u.u_error = EMFILE;
		return(NULL);
	}

	/*	init pregion
	 */

	prp1 = &tmp;
	*prp1 = nullpregion;
	prp1->p_pp = rp;
	prp1->p_regva = vaddr;
	prp1->p_type = type;
	prp1->p_flags = 0;
	prp1->p_procp = p;
	if (prot)
		prp1->p_flags |= PF_RDONLY;

	/*	Check that region does not go beyond end of virtual
	 *	address space.
	 */
	 if (preg_chkinsert(prp1,size)) {
		u.u_error = EINVAL;
		return(NULL);
	}

	/* Insert tmp in the pregion list
	*/

	first = p->p_region;
	for (prp1=prp; prp1!=first; --prp1)
		if ((prp1-1)->p_regva > vaddr) {
			preg_xmove(prp1-1, prp1);
			*prp1 = *(prp1-1);
		}
		else break;
	*prp1 = tmp;
	preg_expand(size, 0, 1, btop(prp1->p_regva));
	prp1->p_pte = &p->p_p0br[(int)btoc(prp1->p_regva)];
	return(prp1);
}

/*	Detach a region from a process' address space.
 *	And free the page table.
**/

void
preg_delete(prp, size)
register preg_t *prp;
register int size;
{
	register int i;
	register long a;

	preg_expand(-size, 0, 1, (int)(btop(prp->p_regva)) + size);

	/* Insert tmp in the pregion list
	/*	Clear the proc region we just detached.
	 */
	for (i = prp - u.u_procp->p_region; i < pregpp-1; i++, prp++) {
		preg_xmove(prp+1, prp);
		*prp = *(prp+1);
	}
	*prp = nullpregion;
}

/*
 * Check that growth of stack or mmap
 * will collide with the regions.
 * returns 0 if legal, -1 if illegal.
 */
preg_chkgrow(start, size)
register caddr_t start;	/* starting page number */
register int	size;		/* New size in pages. */
{
	register caddr_t end, vaddr;
	register preg_t *prp;

	if ((prp = u.u_procp->p_region) == NULL)
		return(0);
	start = (caddr_t)ctob((int)start);
	size = ctob(size);
	end = (size>0)? start+size-1:start;
	for (;prp->p_pp; ++prp) {
		vaddr = (caddr_t)prp->p_regva;
		if (end<vaddr)
			break;
		if (start == vaddr || (start< vaddr && end>=vaddr))
			return(-1);
		if (start>vaddr) {
			if (prp->p_type == PT_LIBDAT)
				size = ctob(prp->p_pp->r_pgsz);
			else
				size = ctob(prp->p_ps->x_size);
			if (size != 0 && start<vaddr+size)
				return(-1);
		}
	}
	return(0);
}


/*
 * Check that insertion of a pregion is legal
 * returns 0 if legal, -1 if illegal.
 */

preg_chkinsert(prp, psize)
register preg_t	*prp;
register int	psize;	/* New size in pages. */
{
	register preg_t *prp1;
	register caddr_t start, end, vaddr;
	register int size;
	register struct proc *p;

	p = u.u_procp;
	prp1 = p->p_region;
	start = (caddr_t) prp->p_regva;
	size = ctob(psize);
	end = (size>0)? start+size-1:start;

	/* Check for overlap with text and data segments */
	vaddr = (caddr_t)((int)ctob(u.u_procp->p_tstart) & ~SOFFMASK);
	if (vaddr < start) {
		if ((caddr_t)(ctob(dptov(p, u.u_dsize))) >= start)
			return(-1);
	} else {
		if (end >= vaddr)
			return(-1);
	}

	/* Check for overlap with stack segment */
	vaddr = (caddr_t)((USRSTACK - (int)ctob(u.u_ssize)) & ~SOFFMASK);
	size = ctob(u.u_ssize);
	if (vaddr < start) {
		if ((vaddr + size) >= start)
			return(-1);
	} else {
		if (end >= vaddr)
			return(-1);
	}

	for (;prp1->p_pp; ++prp1) {
		if (prp1==prp) continue;
		vaddr = (caddr_t)prp1->p_regva;
		if (end<vaddr)
			break;
		if (start == vaddr || (start< vaddr && end>=vaddr))
			return(-1);
		if (start>vaddr) {
			if (prp1->p_type == PT_LIBDAT)
				size = ctob(prp1->p_pp->r_pgsz);
			else
				size = ctob(prp1->p_ps->x_size);
			if (size != 0 && start<vaddr+size)
				return(-1);
		}
	}
	return(0);
}
#endif	SYSV
