extern int end;
/*
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)kern_exec.c	7.1 (Berkeley) 6/5/86
 */

#include "../h/types.h"

#include "../machine/reg.h"
#include "../machine/pte.h"
#include "../machine/psl.h"

#include "param.h"
#include "systm.h"
#include "map.h"
#include "dir.h"
#include "user.h"
#include "kernel.h"
#include "proc.h"
#include "buf.h"
#include "socketvar.h"
#include "vnode.h"
#include "pathname.h"
#include "seg.h"
#include "vm.h"
#include "text.h"
#include "file.h"
#include "uio.h"
#include "acct.h"
#include "exec.h"
#include "vfs.h"

#ifdef TRFS
#include "../wipc/wipc.h"
#endif TRFS

#ifdef vax
#include "../vax/mtpr.h"
#endif vax

#ifdef	SYSV
#include "../sysv/sys/errno.h"
#include "../sysv/sys/filehdr.h"
#endif	SYSV

/*
 * exec system call, with and without environments.
 */
struct execa {
	char	*fname;
	char	**argp;
	char	**envp;
};

execv(uap)
	register struct execa *uap;
{
	uap->envp = NULL;
	execve(uap);
}

execve(uap)
	register struct execa *uap;
{
	register nc;
	register char *cp;
	register struct buf *bp = 0;
	int na = 0, ne = 0, cc = 0;
	int ucp, ap;
	unsigned len;
	int indir = 0;
	uid_t uid = u.u_uid;
	gid_t gid = u.u_gid;
	char *sharg, *execnamep;
	struct vnode *vp = NULL;
	struct vattr vattr;
	struct pathname pn;
	struct pathname *pnp = &pn;
	register struct proc *p = u.u_procp;
	swblk_t bno = 0;
	char cfname[MAXCOMLEN + 1];
#define	SHSIZE	32
	char cfarg[SHSIZE];
	union exdata {
		char	ex_shell[SHSIZE];	/* #! and name of interpreter */
		struct	exec ex_exec;
	} exdata;
	int resid, error;
	unsigned long rpid = 0;			/* ISI: TRFS: */

#ifdef	SYSV
	struct sysv_exdata sv_exdata;
	char universe = UNIVERSE_BSD, ouniverse;
	unsigned shlb_scnsz, shlb_datsz;
	int shlb_buf = 0;
	struct sysv_exdata *shlb_dat, *shlb_exp;
#endif	SYSV

	if (u.u_error = pn_get(uap->fname, UIOSEG_USER, pnp))
		return;
	bcopy((caddr_t)pnp->pn_buf, (caddr_t)cfname, MAXCOMLEN);
	cfname[MAXCOMLEN] = '\0';


#ifdef	TRFS
	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
#endif	TRFS
		u.u_error = lookuppn(pnp, FOLLOW_LINK, (struct vnode **)0, &vp);
		u.u_tok = 0;			/* ISI: TRFS: */
		if (u.u_error) {
			pn_free(pnp);
			return;
		}
		if (u.u_error = VOP_GETATTR(vp, &vattr, u.u_cred))
			goto bad;
		if ((vp->v_vfsp->vfs_flag & VFS_NOSUID) == 0) {
			if (vattr.va_mode & VSUID)
				uid = vattr.va_uid;
			if (vattr.va_mode & VSGID)
				gid = vattr.va_gid;
		} else if ((vattr.va_mode & VSUID) || (vattr.va_mode & VSGID))
			uprintf("%s: Setuid execution not allowed\n", cfname);

again:		if (u.u_error = VOP_ACCESS(vp, VEXEC, u.u_cred))
			goto bad;
		if ((p->p_flag&STRC)
		    && (u.u_error = VOP_ACCESS(vp, VREAD, u.u_cred)))
			goto bad;
		if (vp->v_type != VREG ||
		   (vattr.va_mode & (VEXEC|(VEXEC>>3)|(VEXEC>>6))) == 0) {
			u.u_error = EACCES;
			goto bad;
		}

		/*
		 * Read in a.out header.
		 *	OMAGIC = plain executable
		 *	NMAGIC = RO text
		 *	ZMAGIC = demand paged RO text
		 *	0443   = shared library
		 * Also an ASCII line beginning with #! is the file name of 
		 * a ``shell'' and arguments may be prepended to the argument 
		 * list if given here.
		 *
		 * SHELL NAMES ARE LIMITED IN LENGTH.
		 *
		 * ONLY ONE ARGUMENT MAY BE PASSED TO THE SHELL FROM
		 * THE ASCII LINE.
		 */
		exdata.ex_shell[0] = 0;		/* for zero length files */
		if (u.u_error = vn_rdwr(UIO_READ, vp, (caddr_t)&exdata, 
		    sizeof (exdata), 0, UIOSEG_KERNEL, IO_UNIT, &resid))
			goto bad;
#ifndef lint
		if (resid > sizeof (exdata) - sizeof (exdata.ex_exec) &&
		    exdata.ex_shell[0] != '#') {
			u.u_error = ENOEXEC;
			goto bad;
		}
#endif
#ifdef	TRFS
	} else if (u.u_error || (u.u_error = ClExec(&rpid, u.u_tpnp->pn_path, 
	    u.u_tpnp->pn_pathlen, &exdata,
#ifdef	SYSV
	    &sv_exdata,
#endif	SYSV
	    indir?(uid_t *)0:&uid, indir?(gid_t *)0:&gid)) )
		goto bad;
#endif	TRFS

#ifdef	SYSV
chkmag:	sv_exdata.a_mag = exdata.ex_exec.a_magic;
	u.u_filemagic = exdata.ex_exec.a_machtype;
#endif	SYSV
	u.u_magic = exdata.ex_exec.a_magic;
	switch (exdata.ex_exec.a_magic) {
	    case OMAGIC:
		exdata.ex_exec.a_data += exdata.ex_exec.a_text;
		exdata.ex_exec.a_text = 0;

#ifdef	SYSV
		sv_exdata.a_datorg = sv_exdata.a_txtorg;
		sv_exdata.a_doffset = sv_exdata.a_toffset;
#endif	SYSV

		break;

	    case ZMAGIC:
	    case NMAGIC:
		if (exdata.ex_exec.a_text == 0) {
			u.u_error = ENOEXEC;
			goto bad;
		}
		break;

#ifdef	SYSV
	    case 0443:
		u.u_error = ELIBEXEC;
		break;
#endif	SYSV

	    default:
#ifdef	SYSV
		if (exdata.ex_exec.a_machtype == COFFMAGIC ||
			exdata.ex_exec.a_machtype == NCRMAGIC) {
#ifdef	TRFS
			if (vp == NULL) {
				u.u_error = ENOEXEC;
				goto bad;
			}
#endif	TRFS
			if (sysv_gethead(vp, &exdata, &sv_exdata))
				goto bad;
			universe = UNIVERSE_SYSV ;
			goto chkmag ;
		}
#endif	SYSV
		if (exdata.ex_shell[0] != '#' ||
		    exdata.ex_shell[1] != '!' ||
		    indir) {
			u.u_error = ENOEXEC;
			goto bad;
		}
		cp = &exdata.ex_shell[2];		/* skip "#!" */
		while (cp < &exdata.ex_shell[SHSIZE]) {
			if (*cp == '\t')
				*cp = ' ';
			else if (*cp == '\n') {
				*cp = '\0';
				break;
			}
			cp++;
		}
		if (*cp != '\0') {
			u.u_error = ENOEXEC;
			goto bad;
		}
		cp = &exdata.ex_shell[2];
		while (*cp == ' ')
			cp++;
		execnamep = cp;
		while (*cp && *cp != ' ')
			cp++;
		cfarg[0] = '\0';
		if (*cp) {
			*cp++ = '\0';
			while (*cp == ' ')
				cp++;
			if (*cp)
				bcopy((caddr_t)cp, (caddr_t)cfarg, SHSIZE);
		}
		indir = 1;
		if (vp)				/* ISI: TRFS */
			VN_RELE(vp);
		vp = NULL;
		if (u.u_error = pn_set(&pn, execnamep))
			goto bad;
#ifdef	TRFS
		/*
		 * u.u_tpnp and pnp share the same buffer, break this by saying
		 * that u.u_tpnp has no buffer, and free u.u_tpnp.
		 */
		if (u.u_tpnp) {
			u.u_tpnp->pn_buf = 0;
			pn_dfree(u.u_tpnp);
			u.u_tpnp = (struct pathname *)0;
		}
		u.u_tok = 2;
		/* TRFS references will return to setjmp above!! */
		u.u_error = lookuppn(pnp, FOLLOW_LINK, (struct vnode **)0, &vp);
		u.u_tok = 0;
#else	TRFS
		u.u_error = lookuppn(pnp, FOLLOW_LINK, (struct vnode **)0, &vp);
#endif	TRFS
		if (u.u_error)
			goto bad;
		if (u.u_error = VOP_GETATTR(vp, &vattr, u.u_cred))
			goto bad;
		bcopy((caddr_t)execnamep, (caddr_t)cfname, MAXCOMLEN);
		cfname[MAXCOMLEN] = '\0';
		rpid = 0;			/* ISI: TRFS: */
		goto again;
	}

#ifdef	SYSV
#ifdef	TRFS
	if (vp == NULL && (exdata.ex_exec.a_machtype == COFFMAGIC ||
			   exdata.ex_exec.a_machtype == NCRMAGIC))
		universe = UNIVERSE_SYSV;
#endif	TRFS
	ouniverse = p->p_universe; /* save for signal stuff */
	p->p_universe = universe ;
	if ((universe == UNIVERSE_SYSV) && sv_exdata.a_nshlibs) {
		register int n;
		shlb_scnsz = (sv_exdata.a_lsize + NBPW) & (~(NBPW -1 ));
		shlb_datsz = sv_exdata.a_nshlibs * sizeof(struct sysv_exdata);
		shlb_buf = (int)kmem_alloc(shlb_scnsz + shlb_datsz);
		shlb_dat = (struct sysv_exdata *)(shlb_buf + shlb_scnsz);
		shlb_exp = shlb_dat;
		n = sv_exdata.a_nshlibs;
		while (n--)
			shlb_exp++->vp = NULL;
#ifdef	TRFS
		if (vp == NULL) {
			if (trfs_getshlibs(rpid, uap->fname, shlb_buf,
					shlb_dat, &sv_exdata))
				goto bad;
		} else
#endif	TRFS
		if (sysv_getshlibs(shlb_buf, shlb_dat, &sv_exdata))
			goto bad;
	}
#endif	SYSV

	/*
	 * Collect arguments on "file" in swap space.
	 */
	na = 0;
	ne = 0;
	nc = 0;
	cc = 0;
	bno = rmalloc(argmap, (long)ctod(clrnd((int)btoc(NCARGS))));
#ifdef	is68k
	if (rootvp == argdev_vp)
		bno += MINIROOTSIZE;
#endif	is68k
	if (bno == 0) {
		swkill(p, "exece: no arg swap space");
		goto bad;
	}
	if (bno % CLSIZE)
		panic("execa rmalloc");
	/*
	 * Copy arguments into file in argdev area.
	 */
	if (uap->argp) for (;;) {
		ap = NULL;
		sharg = NULL;
		if (indir && na == 0) {
			sharg = cfname;
			ap = (int)sharg;
			uap->argp++;		/* ignore argv[0] */
		} else if (indir && (na == 1 && cfarg[0])) {
			sharg = cfarg;
			ap = (int)sharg;
		} else if (indir && (na == 1 || na == 2 && cfarg[0]))
			ap = (int)uap->fname;
		else if (uap->argp) {
			ap = fuword((caddr_t)uap->argp);
			uap->argp++;
		}
		if (ap == NULL && uap->envp) {
			uap->argp = NULL;
			if ((ap = fuword((caddr_t)uap->envp)) != NULL)
				uap->envp++, ne++;
		}
		if (ap == NULL)
			break;
		na++;
		if (ap == -1) {
			u.u_error = EFAULT;
			break;
		}
		do {
			if (cc <= 0) {
				/*
				 * We depend on NCARGS being a multiple of
				 * CLSIZE*NBPG.  This way we need only check
				 * overflow before each buffer allocation.
				 */
				if (nc >= NCARGS-1) {
					error = E2BIG;
					break;
				}
				if (bp)
					bdwrite(bp);
				cc = CLSIZE*NBPG;
				bp = getblk(argdev_vp, bno + ctod(nc/NBPG), cc);
				cp = bp->b_un.b_addr;
			}
			if (sharg) {
				error = copystr(sharg, cp, (unsigned)cc, &len);
				sharg += len;
			} else {
				error = copyinstr((caddr_t)ap, cp, (unsigned)cc,
				    &len);
				ap += len;
			}
			cp += len;
			nc += len;
			cc -= len;
		} while (error == ENOENT);
		if (error) {
			u.u_error = error;
			if (bp)
				brelse(bp);
			bp = 0;
			goto badarg;
		}
	}
	if (bp)
		bdwrite(bp);
	bp = 0;
	nc = (nc + NBPW-1) & ~(NBPW-1);

#ifdef	TRFS
	if (vp == NULL)
		ClGetxfile(rpid, &exdata.ex_exec,
#ifdef	SYSV
			&sv_exdata,
#endif	SYSV
			nc + (na+4)*NBPW, uid, gid, &pn);
	else
#endif	TRFS
		getxfile(vp, &exdata.ex_exec,
#ifdef	SYSV
			&sv_exdata,
#endif	SYSV
			nc + (na+4)*NBPW, uid, gid, &pn);

#ifdef	SYSV
	if (universe == UNIVERSE_SYSV && sv_exdata.a_nshlibs && !u.u_error)
#ifdef	TRFS
		if (vp == NULL)
			trfs_loadshlibs(rpid, shlb_buf, shlb_dat, &sv_exdata);
		else
#endif	TRFS
			sysv_loadshlibs(shlb_buf, shlb_dat, &sv_exdata);
#endif	SYSV

	if (u.u_error) {
badarg:
		for (cc = 0; cc < nc; cc += CLSIZE*NBPG) {
			bp = baddr(argdev_vp, bno + ctod(cc/NBPG), CLSIZE*NBPG);
			if (bp) {
				bp->b_flags |= B_AGE;		/* throw away */
				bp->b_flags &= ~B_DELWRI;	/* cancel io */
				brelse(bp);
				bp = 0;
			}
		}
		goto bad;
	}
	/* can we release the vnode here, as done for inodes?? CTH */

	/*
	 * Copy back arglist.
	 */
	ucp = USRSTACK - nc - NBPW;
	ap = ucp - na*NBPW - 3*NBPW;
	u.u_ar0[SP] = ap;
	(void) suword((caddr_t)ap, na-ne);
	nc = 0;
	cc = 0;
	for (;;) {
		ap += NBPW;
		if (na == ne) {
			(void) suword((caddr_t)ap, 0);
			ap += NBPW;
		}
		if (--na < 0)
			break;
		(void) suword((caddr_t)ap, ucp);
		do {
			if (cc <= 0) {
				if (bp)
					brelse(bp);
				cc = CLSIZE*NBPG;
				bp = bread(argdev_vp, bno + ctod(nc / NBPG), cc);
				bp->b_flags |= B_AGE;		/* throw away */
				bp->b_flags &= ~B_DELWRI;	/* cancel io */
				cp = bp->b_un.b_addr;
			}
			error = copyoutstr(cp, (caddr_t)ucp, (unsigned)cc, 
			    &len);
			ucp += len;
			cp += len;
			nc += len;
			cc -= len;
		} while (error == ENOENT);
		if (error == EFAULT)
			panic("exec: EFAULT");
	}
	(void) suword((caddr_t)ap, 0);

	/*
	 * Reset caught signals.  Held signals
	 * remain held through p_sigmask.
	 */
	while (p->p_sigcatch) {
		nc = ffs((unsigned)p->p_sigcatch);
		p->p_sigcatch &= ~sigmask(nc);
		u.u_signal[nc] = SIG_DFL;
	}
	/*
	 * Reset stack state to the user stack.
	 * Clear set of signals caught on the signal stack.
	 */
	u.u_onstack = 0;
	u.u_sigsp = 0;
	u.u_sigonstack = 0;
	u.u_sigintr = 0;
#ifdef	SYSV
	/*
	 * For System V processes clear the deferred signals
	 */
	if (universe == UNIVERSE_SYSV)
		p->p_chold = 0;
	/*
	 * If the execve is switching the universe all settings beyond
	 * SIGTERM should not be allowed to pass as they will not be
	 * meaningful
	 */
	 if (universe != ouniverse) {
		register int i;

		p->p_sig &= SIG_ACROSS;
		p->p_sigcatch &= SIG_ACROSS;	/* can't be caught/masked */
		p->p_sigmask &= SIG_ACROSS;	/* or ignored */
		p->p_sigignore &= SIG_ACROSS;	/* across universes */
		for(i = SIGTERM+1; i <= NSIG; i++)
			u.u_signal[i] = SIG_DFL;
	}
#endif	SYSV

	for (nc = u.u_lastfile; nc >= 0; --nc) {
		if (u.u_pofile[nc] & UF_EXCLOSE) {
			register struct file *fp;

			fp = u.u_ofile[nc];
			u.u_ofile[nc] = NULL;
			closef(fp, 1);
			u.u_pofile[nc] = 0;
		}
		u.u_pofile[nc] &= ~UF_MAPPED;
	}
	while (u.u_lastfile >= 0 && u.u_ofile[u.u_lastfile] == NULL)
		u.u_lastfile--;
#ifdef	SYSV
	if (p->p_universe == UNIVERSE_SYSV
			&& u.u_rlimit[RLIMIT_FSIZE].rlim_max == RLIM_INFINITY) {
		u.u_rlimit[RLIMIT_FSIZE].rlim_max = 
		u.u_rlimit[RLIMIT_FSIZE].rlim_cur =
		u.u_rlimit[RLIMIT_CORE].rlim_max =
		u.u_rlimit[RLIMIT_CORE].rlim_cur = (CDLIMIT << 9);
	}
#endif	SYSV
	setregs(exdata.ex_exec.a_entry);

	/*
	 * Remember file name for accounting.
	 */
	u.u_acflag &= ~AFORK;
	bcopy((caddr_t)cfname, (caddr_t)u.u_comm, MAXCOMLEN);
bad:
#ifdef	TRFS
	if (u.u_tpnp) {
		pn_dfree(u.u_tpnp);
		u.u_tpnp = (struct pathname *)0;
	} else
#endif	TRFS
		pn_free(pnp);
	if (bp)
		brelse(bp);
#ifdef	is68k
	if (bno) {
	    	if (rootvp == argdev_vp)
			bno -= MINIROOTSIZE;
		rmfree(argmap, (long)ctod(clrnd((int)btoc(NCARGS))), (long)bno);
	}
#else	is68k
	if (bno)
		rmfree(argmap, (long)ctod(clrnd((int)btoc(NCARGS))), (long)bno);
#endif	is68k
	if (vp)
		VN_RELE(vp);

#ifdef	SYSV
	if ((universe == UNIVERSE_SYSV) && sv_exdata.a_nshlibs) {
		shlb_exp = shlb_dat;
		while (sv_exdata.a_nshlibs--)
			if (shlb_exp->vp)
				VN_RELE(shlb_exp++->vp);
		if (shlb_buf)
			kmem_free(shlb_buf, shlb_scnsz + shlb_datsz);
	}
#endif	SYSV
}

int execwarn = 1;

/*
 * Read in and set up memory for executed file.
 */
getxfile(vp, ep,
#ifdef	SYSV
	sv_ep,
#endif	SYSV
	nargc, uid, gid, pnp)
	register struct vnode *vp;
	register struct exec *ep;
#ifdef	SYSV
	register struct sysv_exdata *sv_ep;
#endif	SYSV
	int nargc, uid, gid;
	struct pathname *pnp;
{
	size_t ts, ds, ids, uds, ss;
	int pagi;
	register struct proc *p = u.u_procp;
	struct dmap *tdmap = 0;		/* temporaries */
#ifdef	SYSV
	register unsigned long datasize;
#endif	SYSV

	/*
	 * Check to make sure nobody is modifying the text right now
	 */
	if ((vp->v_flag & VTEXTMOD) != 0) {
		u.u_error  = ETXTBSY;
		goto bad;
	}
	if ((ep->a_text != 0) && ((vp->v_flag & VTEXT) == 0) &&
	    (vp->v_count != 1)) {
		register struct file *fp;

		for (fp = file; fp < fileNFILE; fp++) {
			if (fp->f_type == DTYPE_VNODE &&
			    fp->f_count > 0 &&
			    (struct vnode *)fp->f_data == vp &&
			    (fp->f_flag&FWRITE)) {
				u.u_error = ETXTBSY;
				goto bad;
			}
		}
	}

	/*
	 * See if the image will run on this machine/kernel
	 */
	if (u.u_error = chkaout(ep->a_machtype))
		goto bad;

	/*
	 * Compute text and data sizes and make sure not too large.
	 * NB - Check data and bss separately as they may overflow 
	 * when summed together.
	 */
	ids = clrnd(btoc(ep->a_data));
	uds = clrnd(btoc(ep->a_bss));
	ts = clrnd(btoc(ep->a_text));
#ifdef	SYSV
  	if (p->p_universe == UNIVERSE_SYSV) {
		if (ep->a_magic != OMAGIC
		    && stoc(ctos(btoc(ep->a_text + sv_ep->a_txtorg)))
					> btop(sv_ep->a_datorg)) {
			ep->a_magic = OMAGIC;
			datasize = ep->a_data + sv_ep->a_datorg
				- (long)(ptob(btop(sv_ep->a_txtorg)));
			ids = clrnd(btoc(datasize));
			ts = 0;
			ds = clrnd(btoc(datasize + ep->a_bss));
			ep->a_text = 0;
		} else {
			ts = clrnd(btoc(ep->a_text + sv_ep->a_txtorg
				- (long)(ptob(btop(sv_ep->a_txtorg)))));
			ds = clrnd(btoc(ep->a_data + ep->a_bss + sv_ep->a_datorg
				- (long)(ptob(btop(sv_ep->a_datorg)))));
			ids = clrnd(btoc(ep->a_data + sv_ep->a_datorg
				- (long)(ptob(btop(sv_ep->a_datorg)))));
			datasize = 0;
		}
	} else
#endif	SYSV
		ds = clrnd(btoc(ep->a_data + ep->a_bss));
	ss = clrnd(SSIZE + btoc(nargc));
	if (chksize((unsigned)ts, (unsigned)ids, (unsigned)uds, (unsigned)ss))
		goto bad;

	/*
	 * Determine whether to demand load the image
	 */
	if ((ep->a_magic == ZMAGIC) && ((ts + ids) > MIN(freemem, pgthresh)))
		if (vp->v_vfsp->vfs_bsize < CLBYTES) {
			if (execwarn) {
				uprintf("Warning: file system block size ");
				uprintf("too small, '%s' not demand paged\n",
				    pnp->pn_buf);
			}
			pagi = 0;
#ifdef	M68025
		} else if (ep->a_text & 0x1000 || 
		    (ep->a_data+((ep->a_bss+0xFFF)&~0x1000)) & 0x1000) {
			/*
			 * a.out's that were padded to 4k pages and are odd 
			 * aligned cannot be demand paged!
			 */
			if (execwarn) {
				uprintf("Warning: image not padded to ");
				uprintf("page boundary, '%s' not demand paged\n",
				pnp->pn_buf);
			}
			pagi = 0;
#endif	M68025
#if defined(SYSV) && defined(RFS)
		} else if (vp->v_flag & VRFSNODE) {
			pagi = 0;
#endif
		} else
			pagi = SPAGV;
	else
		pagi = 0;

	/*
	 * Make sure enough space to start process.
	 */
	tdmap = (struct dmap *)kmem_alloc(2 * sizeof (struct dmap));
 	tdmap[0] = zdmap;
 	tdmap[1] = zdmap;
 	if (swpexpand(ds, ss, &tdmap[0], &tdmap[1]) == NULL)
		goto bad;

	/*
	 * At this point, committed to the new image!
	 * Release virtual memory resources of old process, and
	 * initialize the virtual memory of the new process.
	 * If we resulted from vfork(), instead wakeup our
	 * parent who will set SVFDONE when he has taken back
	 * our resources.
	 */
	if ((p->p_flag & SVFORK) == 0)
		vrelvm();
	else {
		p->p_flag &= ~SVFORK;
		p->p_flag |= SKEEP;
		wakeup((caddr_t)p);
		while ((p->p_flag & SVFDONE) == 0)
			sleep((caddr_t)p, PZERO - 1);
		p->p_flag &= ~(SVFDONE|SKEEP);
	}
	p->p_flag &= ~(SPAGV|SSEQL|SUANOM|SOUSIG);
	p->p_flag |= pagi  | SKEEP;
	u.u_dmap = tdmap[0];
	u.u_smap = tdmap[1];
#ifdef	SYSV
	if (p->p_universe == UNIVERSE_SYSV)
		p->p_tstart = btop(sv_ep->a_txtorg);
	else
#endif	SYSV
		p->p_tstart = 0;
	vgetvm(ts, ds, ss);

	if (pagi == 0)
#ifdef	SYSV
  		if (p->p_universe == UNIVERSE_SYSV) {
		    if (datasize) {
			    u.u_error = vn_rdwr(UIO_READ, vp, 
				(char *)sv_ep->a_txtorg,
				(int)sv_ep->a_tsize,
				(int)sv_ep->a_toffset,
				UIOSEG_USER, IO_UNIT, (int *)0);
		    }
		    if (u.u_error == 0)
			    u.u_error = vn_rdwr(UIO_READ, vp, 
				(char *)sv_ep->a_datorg,
				(int)ep->a_data,
				(int)sv_ep->a_doffset,
				UIOSEG_USER, IO_UNIT, (int *)0);
		} else
#endif	SYSV
		    u.u_error = vn_rdwr(UIO_READ, vp, 
			(char *)ctob(dptov(p, 0)), (int)ep->a_data, 
			(int)(ep->a_text + ((ep->a_magic==ZMAGIC) ? 0 :
			sizeof (struct exec))), UIOSEG_USER, IO_UNIT, (int *)0);
	if (u.u_error == 0)
#ifdef	SYSV
		xalloc(vp, ep, sv_ep, pagi);
#else	SYSV
		xalloc(vp, ep, pagi);
#endif	SYSV
	if (pagi && p->p_textp)
#ifdef	SYSV
  		if (p->p_universe == UNIVERSE_SYSV) {
			vinifod((struct fpte *)dptopte(p, 0), PG_FTEXT, 
			    p->p_textp->x_vptr,
			    (long)(clrnd(btop(sv_ep->a_doffset))),
			    clrnd(btoc(ep->a_data + sv_ep->a_datorg
				  - (long)(ptob(btop(sv_ep->a_datorg))))));

			/* Bss overlaps with data, zero the bss  */
			/* portion of the page , since the       */
			/* file may not be zero padded for bss.  */
			if (sv_ep->a_bssorg & CLOFSET) 
				u_bzero(sv_ep->a_bssorg, 
				   CLBYTES-(sv_ep->a_bssorg & CLOFSET));

		} else
#endif	SYSV
		vinifod((struct fpte *)dptopte(p, 0), PG_FTEXT, 
		    p->p_textp->x_vptr, (long)(ts/CLSIZE), 
		    (int)btoc(ep->a_data));

	p->p_flag &= ~SKEEP;
#ifdef vax
	/* THIS SHOULD BE DONE AT A LOWER LEVEL, IF AT ALL */
	mtpr(TBIA, 0);
#endif vax

	if (u.u_error)
		swkill(p, "getxfile: I/O error mapping pages");
	/*
	 * set SUID/SGID protections, if no tracing
	 */
	if ((p->p_flag&STRC)==0) {
		if (uid != u.u_uid || gid != u.u_gid)
			u.u_cred = crcopy(u.u_cred);
		u.u_uid = uid;
		p->p_uid = uid;
		u.u_gid = gid;
#ifdef	SYSV
		p->p_suid = uid;
		p->p_sgid = gid;
#endif	SYSV
	} else
		psignal(p, SIGTRAP);
	u.u_tsize = ts;
	u.u_dsize = ds;
	u.u_ssize = ss;
	u.u_prof.pr_scale = 0;
bad:
	if (tdmap)
 		kmem_free((caddr_t)tdmap, 2 * sizeof (struct dmap));
}
