/*
 *	(c) 1986, Kinetics, Inc.
 *	
 *	@(#)ddp_usrreq.c	1.1 (Kinetics) 9/10/86
 */
#include "param.h"
#include "dir.h"
#include "user.h"
#include "kernel.h"
#include "mbuf.h"
#include "protosw.h"
#include "socket.h"
#include "socketvar.h"
#include "errno.h"
#include "ioctl.h"

#include "../net/if.h"
#include "../netinet/in.h"
#include "../netinet/if_ether.h"

#include "../netddp/atalk.h"
#include "../netddp/katalk.h"

struct	sockaddr_at ddp_at = { AF_APPLETALK };

struct atcb	ddpcb;
int		ddpqmaxlen = IFQ_MAXLEN;
int		ddpdebug = 0;
struct ifnet	*ifp_Abridge;		/* interface that received last rtmp */

#define	MAXATIFS	6		/* maximum number of Ether AT nodes */
struct atif	atiflist[MAXATIFS+1];

ddp_init()
{
	ddpcb.at_next = ddpcb.at_prev = &ddpcb;
	ddpintq.ifq_maxlen = ddpqmaxlen;
}

ddpintr()
{
	struct ddp		ddp_header;
	struct lap		lap_header;
	register struct ddp	*ddp = &ddp_header;
	register struct lap	*lp = &lap_header;
	register struct atcb	*atp;
	register struct mbuf	*m, *n;
	register struct ifnet	*ifp;
	struct sockaddr_at	daddr;
	struct sockaddr_at	saddr;
	struct a_addr		*ap;
	int			len, l, ddpsize, s;

	DDPSTAT(ddpintr);
	if (ddpdebug) printf("[ddpintr]\n");

	/* Get next datagram off input queue */
again:	s = splimp();
	IF_DEQUEUEIF(&ddpintq, m, ifp);
	splx(s);
	if (m == 0)
		return;

	/* get lap header */
	DDPSTAT(ddppckts);
	if ((m->m_len < SIZEOFLAP) && ((m = m_pullup(m, SIZEOFLAP)) == 0)) {
		DDPSTAT(hdrops);
		goto drop;
	}
	bcopy(mtod(m, caddr_t), (caddr_t)lp, SIZEOFLAP);
	m->m_off += SIZEOFLAP;
	m->m_len -= SIZEOFLAP;

	/* determine size of ddp */
	if (lp->type == LT_SHORTDDP)
		ddpsize = SIZEOFDDPSHORT;
	else
		ddpsize = SIZEOFDDP;

	/* get ddp header */
	if ((m->m_len < ddpsize) && ((m = m_pullup(m, ddpsize)) == 0)) {
		DDPSTAT(hdrops);
		goto drop;
	}
	bcopy(mtod(m, caddr_t), (caddr_t)ddp, ddpsize);

	/* convert short ddp headers to long format */
	if ((lp->type == LT_SHORTDDP) && (ddpsintr(m, lp, ddp, ifp) == 0))
		goto again;
	m->m_off += ddpsize;
	m->m_len -= ddpsize;

	/* get host order */
	ddp->d_cksum	= ntohs(ddp->d_cksum);
	ddp->d_dnet	= ntohs(ddp->d_dnet);
	ddp->d_snet	= ntohs(ddp->d_snet);

	/* reject ddp's not intended for this net/node */
	ap = ddp_getaddr(ifp);
	if (ddpdebug) printf("nettest:%d dest,%d here\n",ddp->d_dnet,ap->at_Net);
	if (ddp->d_dnet && (ddp->d_dnet != ap->at_Net)) {
		DDPSTAT(wrongnet);
		goto drop;
	}
	if (!atforme(ddp)) {
		if (ddpdebug) ddpdump("nfm", ddp);
		DDPSTAT(forward);
		goto drop;
	}

	/* Locate pcb for datagram.  */
	daddr.at_net	= ddp->d_dnet;
	daddr.at_node	= ddp->d_dnode;
	saddr.at_net	= ddp->d_snet;
	saddr.at_node	= ddp->d_snode;
	if (ddpdebug) printf("[lookup(net %x, node %x, sock %x)]",
			daddr.at_net, daddr.at_node, ddp->d_dsno);
	if ((atp = at_pcblookup(&ddpcb, daddr.at_addr, saddr.at_addr,
	    ddp->d_dsno&0xff, ATPLOOKUP_WILDCARD)) == NULL) {
		if (ddpdebug) printf("[-nope-]");
		DDPSTAT(noddpcb);
		goto drop;
	}

	/*
	 * Make mbuf data length reflect DDP length. If not enough data to 
	 * reflect DDP length drop packet. If too much data, drop data at
	 * end of mbuf chain.
	 */
	if ((len = atgetlength(ddp) - ddpsize) > DDPMAXSIZE) {
		DDPSTAT(badlen);
		goto drop;
	}
	n = m;
	l = 0;
	while (n) {
		l += n->m_len;
		if (n->m_next == 0)
			break;
		n = n->m_next;
	}
	if (len != l) {
		if (len > l) {
			DDPSTAT(badlen);
			goto drop;
		}
		l = l - len;
		if (l < n->m_len)
			n->m_len -= l;
		else
			m_adj(m, -l);
	}

	/* Construct sockaddr format source address. */
	ddp_at.at_sno		= ddp->d_ssno;
	ddp_at.at_net		= ddp->d_snet;
	ddp_at.at_node		= ddp->d_snode;
	ddp_at.at_ptype		= ddp->d_type;
	ddp_at.at_abridge	= ap->at_Abridge;
	if (ddpdebug) printf("[call sbappendaddr]");

	/* Stuff source address  and datagram in user buffer. */
	if (sbappendaddr(&atp->at_socket->so_rcv, (struct sockaddr *)&ddp_at,
	    m, (struct mbuf *)0) == 0) {
		DDPSTAT(sockerr);
		goto drop;
	}
	sorwakeup(atp->at_socket);
	goto again;

drop:	if (ddpdebug) printf("bad\n");
	DDPSTAT(bad);
	m_freem(m);
	goto again;
}

/*
 * Make a short ddp into a long ddp. The only consistant way to tell anyone
 * the network from which the dgram came in from is to build a complete
 * applebus internet address -- eg, build a long ddp.
 */
ddpsintr(m, lp, ddp, ifp)
	register struct mbuf	*m;
	register struct lap	*lp;
	register struct ddp	*ddp;
	register struct ifnet	*ifp;
{
	register struct ddpshort *ddps = mtod(m, struct ddpshort *);
	struct a_addr *ap = ddp_getaddr(ifp);
	u_short net;

	DDPSTAT(shortddps);

	/* do simple rtmp */
	if (ddps->D_type == ddpTypeRTMP) {
		DDPSTAT(rtmppckts);
		if ((m->m_len < (SIZEOFDDPSHORT+sizeof(short))) && 
		    ((m = m_pullup(m, SIZEOFDDPSHORT+sizeof(short))) == 0)) {
			DDPSTAT(hdrops);
			goto drop;
		}
		ddps = mtod(m, struct ddpshort *);
		bcopy(((caddr_t)ddps)+SIZEOFDDPSHORT, (caddr_t)&net,
		  sizeof(short));
		if (ddpdebug) printf("net=%x,ntohs(net)=%x", net, ntohs(net));
		net = ntohs(net);

		/* set net and bridge numbers */
		ap->at_Net	= net;
		ap->at_Abridge	= lp->src;
		ifp_Abridge	= ifp;
		if (ddpdebug) printf("rtmp:(%x,%x,%x)\n",
				ap->at_Net, ap->at_Node, ap->at_Abridge);
		goto drop;
	}
	ddp->d_cksum	= 0;
	ddp->d_dsno	= ddps->D_dsno;
	ddp->d_ssno	= ddps->D_ssno;

#ifdef	notdef
	/*
	 * If it came from loopback, say it was actually from the local net.
	 * Makes atforme() simpler.
	 */
	if ((net = ifpgetnet(ifp)) == ifpgetnet(&loif))
		net = 0;
	bcopy(&net, &ddp->d_dnet, sizeof(short));
	bcopy(&net, &ddp->d_snet, sizeof(short));
#endif	notdef
	ddp->d_dnet	= ddp->d_snet = htons(ap->at_Net);
	ddp->d_snode	= lp->src;
	ddp->d_dnode	= lp->dst;
	ddp->d_type	= ddps->D_type;
	return 1;
drop:	m_freem(m);
	return 0;
}

#ifdef	notdef
at_forward(ddp, m)
	register struct ddp *ddp;
	struct mbuf *m;
{
	register struct ifnet	*ifp;
	struct sockaddr_at	dst;
	register struct lap	*lp;
	register struct mbuf	*n;
	register int		error = ENETUNREACH;

	if (atgethops(ddp) > ATMAXHOPS)
		goto drop;
	atbumphops(ddp);		/* bump hop count */

	dst.at_family	= AF_APPLETALK;
	dst.at_net	= ddp->d_dnet;
	dst.at_node	= ddp->d_dnode;
	if ((ifp = atroute(&dst)) == 0)
		goto drop;

	/* back to net order */
	ddp->d_dnet	= htons(ddp->d_dnet);
	ddp->d_snet	= htons(ddp->d_snet);

	/* get an mbuf for ddp */
	MGET(n, M_DONTWAIT, MT_HEADER);
	if (n == 0) {
		error = ENOBUFS;
		goto drop;
	}
	n->m_next = m;
	m = n;
	m->m_off = MMINOFF;
	m->m_len = SIZEOFDDP;
	bcopy((caddr_t)ddp, mtod(m, caddr_t), SIZEOFDDP);

	/* form lap header */
	MGET(n, M_DONTWAIT, MT_HEADER);
	if (n == 0) {
		error = ENOBUFS;
		goto drop;
	}
	n->m_next = m;
	m = n;
	m->m_off = MMINOFF;
	m->m_len = SIZEOFLAP;
	lp = mtod(n, struct lap *);
	lp->type = LT_DDP;
	return (*ifp->if_output)(ifp, m, &dst);

drop:	m_freem(m);
	return error;
}
#endif	notdef

ddp_output(atp, m)
	register struct atcb	*atp;
	register struct mbuf	*m;
{
	register struct ifnet	*ifp;
	register struct mbuf	*n;
	struct ddp		*ddp;
	struct ddpshort		*ddps;
	register struct lap	*lp;
	struct a_addr		*ap;
	struct sockaddr_at	dst;
	register int		len;
	int			error, size;

	if (ddpdebug) printf("[ddpoutput]\n");
	dst = atp->at_faddr;
	if ((ifp = atroute(&dst)) == 0) {
		error = ENETUNREACH;
		goto drop;
	}

	/* Calculate data length and get a mbuf for DDP header. */
	for (n = m, len = 0; n; n = n->m_next)
		len += n->m_len;
	if (len > DDPMAXSIZE) {
		error = EINVAL;
		goto drop;
	}

	/* set up ddp header */
	MGET(n, M_DONTWAIT, MT_HEADER);
	if (n == 0) {
		error = ENOBUFS;
		goto drop;
	}
	n->m_next = m;
	m = n;
	m->m_off = MMINOFF;
	if (ddpdebug) printf("[sending to {%d/%d/%d}]\n",
		    dst.at_net, dst.at_node&0xff,dst.at_sno&0xff);
	ap = ddp_getaddr(ifp);
	if (dst.at_net == 0 || dst.at_net == ap->at_Net) {
		size = SIZEOFDDPSHORT;
		ddps = mtod(m, struct ddpshort *);
		atsetlength(ddps, len + size);
		ddps->D_ssno	= atp->at_lport;
		ddps->D_dsno	= atp->at_fport;
		ddps->D_type	= atp->atcb_ptype;
	} else {
		size = SIZEOFDDP;
		ddp = mtod(m, struct ddp *);
		atsetlength(ddp, len + size);
		ddp->d_cksum	= 0;
		ddp->d_dnet	= htons(atp->at_fnet);
		ddp->d_dnode	= atp->at_fnode;
		ddp->d_snet	= htons(ap->at_Net);
		ddp->d_snode	= atp->at_lnode;
		ddp->d_ssno	= atp->at_lport;
		ddp->d_dsno	= atp->at_fport;
		ddp->d_type	= atp->atcb_ptype;
	}
	m->m_len = size;

	/* set up lap header */
	MGET(n, M_DONTWAIT, MT_HEADER);
	if (n == 0) {
		error = ENOBUFS;
		goto drop;
	}
	n->m_next = m;
	m = n;
	m->m_off = MMINOFF;
	m->m_len = SIZEOFLAP;
	lp = mtod(m, struct lap *);
	if (size == SIZEOFDDPSHORT) {
		lp->type = LT_SHORTDDP;
		lp->dst = dst.at_node;
	} else
		lp->type = LT_DDP;
	return (*ifp->if_output)(ifp, m, &dst);

drop:	m_freem(m);
	return error;
}

int	ddpudebug = 0;
int	ddp_sendspace = 2048;
int	ddp_recvspace = 2048;

/*ARGSUSED*/
ddp_usrreq(so, req, m, nam, rights)
	struct socket *so;
	int req;
	struct mbuf *m, *nam, *rights;
{
	struct atcb *atp = sotoatcb(so);
	struct sockaddr_at save_sat, *sat;
	int error = 0;

	if (atp == NULL && req != PRU_ATTACH) {
		error = EINVAL;
		goto drop;
	}

	switch (req) {

	case PRU_ATTACH:
		if (atp != NULL) {
			error = EINVAL;
			break;
		}
		error = at_pcballoc(so, &ddpcb);
		if (error)
			break;
		error = soreserve(so, ddp_sendspace, ddp_recvspace);
		break;

	case PRU_DETACH:
		at_detach(atp);
		break;

	case PRU_BIND:
		error = at_pcbbind(so, nam);
		break;

	case PRU_CONNECT:
		if ((atp->atcb_flags & (AT_FADDR|AT_IGNADDR)) == (AT_FADDR)) {
			error = EISCONN;
			break;
		}
		at_connaddr(atp, nam);
		soisconnected(so);
		break;

	case PRU_DISCONNECT:
		if ((atp->atcb_flags & AT_FADDR) == 0) {
			error = ENOTCONN;
			break;
		}
		at_disconnect(atp);
		soisdisconnected(so);
		break;

	case PRU_SHUTDOWN:
		socantsendmore(so);
		break;

	case PRU_SEND:
		if (nam) {
 			bcopy(&atp->at_laddr, &save_sat, sizeof(save_sat));
			if (atp->atcb_flags & AT_FADDR) {
				error = EISCONN;
				break;
			}
/* SPL ?? */
			at_connaddr(atp, nam);
		} else if ((atp->atcb_flags & AT_FADDR) == 0) {
			error = ENOTCONN;
			break;
		}
		error = ddp_output(atp, m);
		m = NULL;
		if (nam) {
			at_disconnect(atp);
 			bcopy(&save_sat, &atp->at_laddr, sizeof(save_sat));
/* SPLX ?? */
		}
		break;

	case PRU_ABORT:
		at_disconnect(atp);
		sofree(so);
		soisdisconnected(so);
		break;

	case PRU_SOCKADDR:
		at_setsockaddr(atp, nam);
		break;

	case PRU_CONTROL:
		if ((int)m == IOC_VOID|SIOIGNADDR) {
			atp->atcb_flags |= AT_IGNADDR;
		} else
			error = EINVAL;
		m = NULL;
		break;

	case PRU_SENSE:
		m = NULL;
		/* fall thru... */

	case PRU_LISTEN:
	case PRU_ACCEPT:
	case PRU_SENDOOB:
	case PRU_CONNECT2:
	case PRU_PEERADDR:
	case PRU_FASTTIMO:
	case PRU_SLOWTIMO:
	case PRU_PROTORCV:
	case PRU_PROTOSEND:
		error =  EOPNOTSUPP;
		break;

	case PRU_RCVD:
	case PRU_RCVOOB:
		return (EOPNOTSUPP);	/* do not free mbuf's */

	default:
		panic("ddp_usrreq");
	}
drop:	m_freem(m);
	return (error);
}

/*
 * Can't/don't use standard routing routines (rtalloc()) since they are all
 * heavily oriented toward the Internet world.  We maintain a table of ifp's
 * that have received AppleTalk packets and do simple routing based on info
 * in this table (atiflist).
 *
 * Note that this is not an rtmp router yet, just a dumb search of the if's
 * for a matching net.
 *
 * Also note that if the destination net is 0 (the local net), send to the
 * if that received the last rtmp.
 */
struct ifnet *
atroute(sat)
	struct sockaddr_at *sat;
{
	register struct ifnet *ifp;
	register u_short net = sat->at_net;
	register struct atif *aifp;

	if (net == 0)
		return(ifp_Abridge);

	/*
	 * start at 2 since this is first dynamic node entry. Chooses first 
	 * if that matches.
	 */
	for (aifp = &atiflist[2]; aifp < &atiflist[MAXATIFS]; aifp++)
		if (aifp->aa.at_Net == net)
			return(aifp->ifp);

	/* when all else fails maybe the bridge can help... */
	return(ifp_Abridge);
}

/*
 *     ***** Routines to manage the appletalk protocol control blocks. *****
 */
at_pcballoc(so, head)
	struct socket *so;
	struct atcb *head;
{
	struct mbuf *m;
	register struct atcb *atp;

	m = m_getclr(M_DONTWAIT, MT_PCB);
	if (m == NULL)
		return (ENOBUFS);
	atp = mtod(m, struct atcb *);
	atp->at_socket = so;
	insque(atp, head);
	so->so_pcb = (caddr_t)atp;
	return (0);
}

/* Allocate local host information. */
at_pcbbind(so, nam)
	struct socket *so;
	struct mbuf *nam;
{
	register struct atcb *atp;
	register struct atcb *head = &ddpcb;
	register struct sockaddr_at *sat;
	u_char lport = 0;
	struct sockaddr_at saddr;

	if (ifnet == 0)
		return (EADDRNOTAVAIL);
	saddr.at_net = saddr.at_node = 0;
	if (nam) {
		sat = mtod(nam, struct sockaddr_at *);
		if (sat->at_family != AF_APPLETALK)
			return (EAFNOSUPPORT);
		lport = sat->at_sno&0xff;
		if (ddpdebug) printf("[rqst sno %d]\n", lport);
		/*
		 * Dissallow on this port if:
		 * 	1.  There is already a wildcard client or
		 *	2.  There is already an exact-match client
		 */
		if (lport) {
			if (at_pcblookup(head, sat->at_addr, saddr.at_addr,
			  lport, ATPLOOKUP_WILDCARD))
				    return (EADDRINUSE);
		}
	}
	atp = sotoatcb(so);
	if (sat) {
		atp->at_laddr = *sat;
		atp->atcb_ptype = sat->at_ptype;
	}
	if (lport == 0) {
		/*
		 * system assigns local port.  watchout for too many
		 * associations
		 */
		int ntimes = 0;
		static first = 1;

		do {
			if (first) {
				first = 0;
				head->at_lport = ATPORT_RESERVED -1;
			}
			if (ntimes++ > ATPORT_RESERVED)
				goto bad2;
			if ((head->at_lport+1) > ((ATPORT_RESERVED*2)-1))
				head->at_lport = ATPORT_RESERVED;
			else
				head->at_lport++;
			lport = head->at_lport;
		} while (at_pcblookup(head, atp->at_laddr.at_addr,
			saddr.at_addr, lport, ATPLOOKUP_WILDCARD));
	}
	atp->at_lport = lport;
	if (ddpdebug) printf("[assign lport %d]\n", atp->at_lport);
	return (0);
bad2:	sbrelease(&so->so_snd);
bad:	m_free(dtom(atp));
	return (ENOBUFS);
}

/* Associate a peer's address with a ddp connection block. */
at_connaddr(atp, nam)
	register struct atcb *atp;
	register struct mbuf *nam;
{
	struct a_addr *ap;
	struct sockaddr_at *sat;

	sat = mtod(nam, struct sockaddr_at *);
	atp->at_faddr = *sat;
	atp->atcb_flags |= AT_FADDR;
	if ((atp->at_laddr.at_net == 0) && (atp->at_laddr.at_node == 0)) {
		struct sockaddr_at d;
		struct ifnet *ifp;

		d.at_family = AF_APPLETALK;
		d.at_net = sat->at_net;
		d.at_node = sat->at_node;
		ifp = atroute(&d);
		if (ifp == 0)
			return ENETUNREACH;
		ap = ddp_getaddr(ifp);
		atp->at_laddr.at_net = ap->at_Net;
		atp->at_laddr.at_node = ap->at_Node;
		atp->at_laddr.at_abridge = ap->at_Abridge;
	}
}

/* Disconnect and possibly release resources. */
at_disconnect(atp)
	struct atcb *atp;
{
	if ((atp->atcb_flags & AT_IGNADDR) == 0)
		atp->at_fnet = atp->at_fnode = atp->at_fport = 0;
	atp->atcb_flags &= ~AT_FADDR;
	if (atp->at_socket->so_state & SS_NOFDREF)
		at_detach(atp);
}

/* Detach the raw connection block and discard socket resources. */
at_detach(atp)
	register struct atcb *atp;
{
	struct socket *so = atp->at_socket;

	if (ddpdebug) printf("[at_detach]\n");
	so->so_pcb = 0;
	sofree(so);
	remque(atp);
	m_free(dtom(atp));
	if (ddpdebug) printf("[leave at_detach]\n");
}

at_setsockaddr(atp, nam)
	register struct atcb *atp;
	struct mbuf *nam;
{
	register struct sockaddr_at *sat = mtod(nam, struct sockaddr_at *);
	
	nam->m_len = sizeof (*sat);
	sat = mtod(nam, struct sockaddr_at *);
	bzero((caddr_t)sat, sizeof (*sat));
	*sat = atp->at_laddr;
}

/*
 * called with a wildcard -- says match this if you can.  Always return the
 * first matched atcb.
 *
 * TODO -- don't allow reception by a ddp client of a ddp with a protocol not
 * matching the protocol specified -- that's how the Mac works...  Hmmmm.
 */ 
struct atcb *
at_pcblookup(head, laddr, faddr, lport, flags)
	register struct atcb *head;
	struct a_addr laddr, faddr;
	register u_char lport;
	int flags;
{
	register struct atcb *atp, *match = 0;
	register int matchwild = 2, wildcard;

	for (atp = head->at_next; atp != head; atp = atp->at_next) {
		if (ddpdebug)
			printf("checking %d/%d/%d...\n",
				atp->at_laddr.at_net, atp->at_laddr.at_node,
				atp->at_laddr.at_sno);

		if (atp->at_lport != lport)
			continue;
		wildcard = 0;
		/*
		 * a wildcard address is a (net, node) pair both 0.  Below,
		 * first, if the local address is not zero client wants only
		 * dgrams from a specific address.  Else, he will accept
		 * from anyone.  Second, if the user has established a con-
		 * nection, check to see that the foreign address matches.
		 */
		if ((atp->at_laddr.at_net != 0)||(atp->at_laddr.at_node != 0)) {
			/* client wants specific address. check for match */
			if ((atp->at_laddr.at_net != laddr.at_Net)
			  || (atp->at_laddr.at_node != laddr.at_Node)) {
				continue;
			}
			if ((atp->at_faddr.at_net != 0) ||
			    (atp->at_faddr.at_node != 0)) {
			    /* check for specific foreign address */
				if ((atp->at_faddr.at_net != faddr.at_Net) || 
				    (atp->at_faddr.at_node != faddr.at_Node)) {
					continue;
				} else {
					return(atp);
				}
			} else {
				wildcard++;
			}
		} else 
			wildcard++;
		if (wildcard && (flags & ATPLOOKUP_WILDCARD) == 0)
			continue;		/* want exact match */
		if (wildcard < matchwild) {
			/*
			 * Found at least one match.   (Use it unless find
			 * exact match.)
			 */
			match = atp;	
			matchwild = wildcard;
			if (matchwild == 0)
				break;
		}
	}
	return (match);
}

/* Is this AppleTalk segment for me? */
atforme(dp)
	register struct ddp *dp;
{
	register struct atif *aifp;

	if (ddpdebug)
		printf("atforme:[%x,%x]", dp->d_dnet,dp->d_dnode);
	if (dp->d_dnet) {
		for (aifp = &atiflist[1]; aifp < &atiflist[MAXATIFS]; aifp++) {
			if (dp->d_dnet == aifp->aa.at_Net) {
				if ((dp->d_dnode == aifp->aa.at_Node) ||
					dp->d_dnode == 0xff)
					return 1;
			}
		}
	} else
		return 1;
	return 0;
}

/* Given an interface, return AppleTalk address block */
struct a_addr *
ddp_getaddr(ifp)
	register struct ifnet *ifp;
{
	register struct atif *aifp;

	for (aifp = &atiflist[1]; aifp < &atiflist[MAXATIFS]; aifp++)
		if (aifp->ifp == ifp)
			break;
	/* if we don't find an if, we'll point to a zero'd sentinal */
	return(&(aifp->aa));
}

/* getatnode() - pick AppleTalk node number dynamically. */
#define AT_NTRIES	3
#define	AT_MAXNODES	254
int atnode;
int atprobe = 0;		/* probe in use */

getatnode(ac)
	register struct arpcom *ac;
{
	int node;

	while (atprobe)
		sleep(&atprobe, PZERO-1);
	atprobe = 1;
	node = atnodeprobe(ac);
	atprobe = 0;
	wakeup(&atprobe);
	return node;
}

atnodetimeout()
{
	wakeup(&atnode);
}

atnodeprobe(ac)
	register struct arpcom *ac;
{
	char edest[6];
	int node = 2;
/* ac->ac_enaddr[sizeof(ac->ac_enaddr)-1]; seed this with other than 2 */
	static char current[4];
	int times = 0;
	int usetrailers;

	while (1) {
	    current[3] = node;
	    if (aarpresolve(ac, NULL, current, edest, &usetrailers)) {
		    arpdelete(current);
		    if (node++ > AT_MAXNODES)
			    return 0;
		    times = 0;
	    } else {
		if (++times >= AT_NTRIES) {	/* assume good node by now */
		    ddp_setifp(&ac->ac_if, node);
		    printf("Ethernet address (%e) assigned Appletalk node %d\n",
			    ac->ac_enaddr, node);
		    arpdelete(current);
		    bzero((caddr_t)&ddpstat, sizeof (struct ddpstat));
		    return node;
		} else {
		    timeout(atnodetimeout, (caddr_t) 0, (30*hz)/times);
		    sleep(&atnode, PZERO-1);
		}
	    }
	}
}

ddpdump(){}
