/*
 * This program is designed to load other diagnostics.  The motivations are
 *  1) users do not need to know where a diagnostic is on the tape.
 *  2) we do not need to document the position of diagnostics on tape.
 *  3) we can have things on the tape that the user does not know about.
 *  4) ability to add new diagnostics/machine support with minimal effort.
 *  5) all diagnostics can be placed in one file on the tape.
 */
#include "sadie.h"
#include "saio.h"
#include "sais68k.h"
#include <a.out.h>

extern char	*start;
extern int	autoboot;

/*
 * This program supports loading a diagnostic from a "diagnostic image".
 * The diagnostic image contains the executable of all diagnostics.  This 
 * image is made by the program "makeimage" in this directory.  The program
 * "makeimage" also patches the directory array below with the block offset 
 * to a given diagnostic in the diagnostic image. The directory array contains 
 * NMACHTYPES slots for each diagnostic, even if the diagnostic is not 
 * supported on all machines.  A slot is patched to -1 for machines on which
 * the diagnostic is not supported.  The directory array is indexed based on 
 * the test to be run, and the type of machine.  The machine portion of the 
 * index must be in the same order as the ON_ defines below.
 */
int directory[NMACHTYPES * NDIAGNOSTICS]  = {SADIE_MAGIC, };

#define	ON_V20		0x00000001
#define	ON_V25		0x00000002
#define	ON_Q10		0x00000004
#define	ON_V10		0x00000008

/* Groupings of above definitions */
#define	ON_ALL		ON_Q10 | ON_V10 | ON_V20 | ON_V25
#define	ON_QBUS		ON_Q10
#define	ON_VBUS		ON_V10 | ON_V20 | ON_V25
#define	ON_010		ON_Q10 | ON_V10
#define	ON_020		ON_V20 | ON_V25

/* Define the type of machine that we are running on */
#ifdef	QBUS
#ifdef	M68020
#define	ON		ON_Q20
#else	M68020
#define	ON		ON_Q10
#endif	M68020
#else	QBUS
#ifdef	M68020
#ifdef	M68025
#define	ON		ON_V25
#else	M68025
#define	ON		ON_V20
#endif	M68025
#else	M68020
#define	ON		ON_V10
#endif	M68020
#endif	QBUS

/*
 * The following structure contains the names of all diagnostics, and the
 * machines on which they can run.  Do not delete lines which say "DO NOT 
 * DELETE", they are used to determine the arguments to "makeimage". Some 
 * of the diagnostics are too complex or are undocumented, these are marked 
 * with the ON_EXPERT bit, and are only available if you have entered 
 * "EXPERT" mode.
 */
#define	ON_EXPERT	0x80000000

struct diagnostics {
	char	*d_diag;
	int	d_mach;
} diag[] = {
/* BEGIN: DO NOT DELETE THIS LINE! */
	{ "copy",	ON_ALL },
	{ "bad144",	ON_ALL },
	{ "diskformat",	ON_ALL },
	{ "mmu",	ON_ALL },
	{ "mem",	ON_ALL },
	{ "dma",	ON_ALL },
	{ "memtest",	ON_ALL | ON_EXPERT },
	{ "boot",	ON_ALL | ON_EXPERT },
	{ "dmax",	ON_ALL | ON_EXPERT },
	{ "cat",	ON_ALL | ON_EXPERT },
	{ "od",		ON_ALL | ON_EXPERT },
	{ "dpreform",	ON_ALL | ON_EXPERT },
	{ "diskread",	ON_ALL | ON_EXPERT },
	{ "diskwrite",	ON_ALL | ON_EXPERT },
	{ "disktest",	ON_ALL | ON_EXPERT },
	{ "dcopy",	ON_ALL | ON_EXPERT },
	{ "cpu",	ON_ALL | ON_EXPERT },
	{ "cmp",	ON_ALL | ON_EXPERT },

	{ "clock",	ON_VBUS | ON_EXPERT },
	{ "icp",	ON_VBUS | ON_EXPERT },
	{ "interrupt",	ON_VBUS | ON_EXPERT },
	{ "gip",	ON_VBUS | ON_EXPERT },	/* KLUDGE: EXTERNAL */

	{ "fault",	ON_V20 | ON_EXPERT },
	{ "prmmu",	ON_V20 | ON_EXPERT },
	{ "chktxt",	ON_V20 | ON_EXPERT },
/* END: DO NOT DELETE THIS LINE! */
	{ 0,0 },
		/* TODO: */
	{ "nw",		ON_VBUS },		/* KLUDGE: EXTERNAL */
};

/*
 * The following structure has the types of tape drives that sadie tapes
 * can be run from.  If a machine has more that one of these drives, the
 * first one will be tried first.
 */
char *tape_controllers[] = { 
	"ts", 
#ifdef	QBUS
	"tm", 
#endif	QBUS
	0
};

/*
 * Some optimizations were made in the standalone tape drivers to avoid 
 * excess tape motions to determine the the blocksize.  The diagnostic
 * image must be writen to tape with a block size of SADIE_BSIZE.
 */
int	sadie_blocksize = SADIE_BSIZE;

char	bootstring[30];
char	buf[30];
int	expert = 0;

main()
{
	struct diagnostics *dp = diag;
	register int testnumber;
	register int testletter, maxtestletter;
	register int testfile;
	register int test;
	register int on;
	char **tape;
	int i, j;
	int saddr;

	*sadie_magic = 0;
	printf("\n||| %i Standalone Diagnostics Executive |||\n");

	if (directory[0] == SADIE_MAGIC) {
		printf("sadie directory not patched by makeimage!\n");
		exit(0);
	}
menu:	printf("\n");
	for (dp = diag, maxtestletter = 'a'; dp->d_diag; dp++)
		if ((dp->d_mach & ON) && 
		    (expert || ((dp->d_mach & ON_EXPERT) == 0))) {
			printf("   %c) %s", maxtestletter++, dp->d_diag);
			if ((maxtestletter-'a') % 4) {
				j = 14 - strlen(dp->d_diag);
				for (i = 0 ; i < j; i++)
					printf(" ");
			} else
				printf("\n");
		}
	maxtestletter--;
	printf("\n");
loop:	printf("\nEnter letter of desired test [a-%c] : ", maxtestletter);
	stripwhite(gets(buf));
	if ((test = buf[0]) == 0)
		goto loop;
	if (test >= 'A' && test <= 'Z')
		test = 'a' + (test - 'A');

	/* change the sense of "EXPERT" mode */
	if (test == '@') {
		expert ^= ON_EXPERT;
		goto menu;
	} else if (test > maxtestletter)
		goto loop;

	for (dp = diag, testletter = 'a', testnumber = 0; dp->d_diag; 
	    dp++, testnumber++) {
		if (test == testletter)
			break;
		if ((dp->d_mach & ON) && 
		    (expert || ((dp->d_mach & ON_EXPERT) == 0)))
			testletter++;
	}
	if (dp->d_diag == 0 || test != testletter) 
		goto loop;
	on = ON;
	testfile = testnumber * NMACHTYPES;
	while ((on & 1) == 0) {
		testfile++;
		on >>= 1;
	}
	if (directory[testfile] < 0) {
		printf("internal error: %s not found", dp->d_diag);
		goto loop;
	}
	for (tape = tape_controllers; *tape; tape++) {
		sprintf(bootstring, "%s(0,%d)", *tape, NMACHTYPES);
#ifdef	DEBUG
		printf("    loading %s: offset %d: ",
			bootstring, directory[testfile]);
#endif	DEBUG
		autoboot = 1;
		if (saddr = load(bootstring, directory[testfile])) {
			*sadie_magic = SADIE_MAGIC;
			booty(saddr);
			*sadie_magic = 0;
		}
		autoboot = 0;
	}
	printf("could not load\n");
	goto loop;
}

/* from boot.c */
load(file, offset)
	register char *file;
{
	union {				/* KLUDGE: for sm block transfer */
		struct exec exec;
		char dummy[520];
	} X;
#define	x	X.exec
	register int i, io;
	register char *addr;

	if ((io = open(file, 0)) < 0)
		goto ret0;

	lseek(io, offset*SADIE_BSIZE, 0);
	i = read(io, (char *)&x, sizeof x);
	if (i != sizeof x || x.a_magic != 0407) {
		printf("file not bootable\n");
		goto ret0;
	}
	printf("%d", x.a_text);
	x.a_entry &= 0x0FFFFFFF;
	addr = (char *)x.a_entry;
	if (addr+(x.a_text+x.a_data+x.a_bss) >= start)
		goto nomem;
	if (read(io, addr, x.a_text) != x.a_text)
		goto shread;
	addr += x.a_text;
	printf("+%d", x.a_data);
	if (read(io, addr, x.a_data) != x.a_data)
		goto shread;
	addr += x.a_data;
	printf("+%d", x.a_bss);
	bzero(addr, x.a_bss);
	close(io);
	printf(" start 0x%x", x.a_entry);
	autoboot = 0;
	printf("\n");
	return(x.a_entry);

shread:	printf("short read\n");
	goto ret0;

nomem:	printf("too big\n");
ret0:	close(io);
	return(0);
}
