/*
 * QBUS|VMEBUS: TS11 tape driver
 */
#include "ts.h"
#if NTS > 0 || Nts > 0
#include "../machine/pte.h"

#include "param.h"
#include "systm.h"
#include "buf.h"
#include "dir.h"
#include "conf.h"
#include "user.h"
#include "file.h"
#include "map.h"
#include "vm.h"
#include "ioctl.h"
#include "mtio.h"
#include "cmap.h"
#include "uio.h"
#include "tty.h"
#include "kernel.h"

#include "../machine/board.h"
#include "../is68kdev/qbvar.h"
#include "../is68kdev/tsreg.h"

u_short	*tsstd[] = { 
#ifdef	QBUS
		(u_short *)0x3FF550, 
		(u_short *)0x3FF520,
#else	QBUS
		(u_short *)&vme_stdio[0xFFF550],
		(u_short *)&vme_stdio[0xFFF520],
		(u_short *)&vme_stdio[0xFFF540],
#endif	QBUS
		0 };

/*
 * There is a ctsbuf per tape controller. It is used as the token to pass to 
 * the internal routines to execute tape ioctls. In particular, when the tape 
 * is rewinding on close we release the user process but any further attempts 
 * to use the tape drive before the rewind completes will hang waiting for 
 * ctsbuf.
 */
struct	buf	ctsbuf[NTS];

int		tswaitloc[NTS];
#define	MAXTIME 0x300		

/*
 * Raw tape operations use a rtsbuf.  The driver notices when a rtsbuf is being
 * used and allows the user program to continue after errors and read records
 * not of the standard length (BSIZE).
 */
struct buf rtsbuf[NTS];

/*
 * Driver unibus interface routines and variables.
 */
int	tsprobe(), tsslave(), tsattach(), tsintr(), tstimer(), tsminphys();
struct	qb_ctlr		*tscinfo[NTS];
struct	qb_device	*tsdinfo[NTS];
struct	buf		tsutab[NTS];

#ifndef	QBUS
extern u_short		*battery_clock;
static short		ts_battery_clock;
#endif	QBUS

struct	qb_driver 	TSdriver =
	{ tsprobe, tsslave, tsattach, tsstd, "ts", tsdinfo, "TS", tscinfo };

/* bits in minor device */
#define	TSUNIT(dev)	(minor(dev)&03)
#define	T_NOREWIND	0x04
#define T_DENSITY	0x08
#define	T_QIC11		0x10
#define	T_NOSWAP	0x20
#define T_FMT		0x40
#define T_FIXEDREC	0x80

#define	INF	(daddr_t)1000000L

/*
 * Software state per tape transport.
 * Also contains hardware state in message packets.
 *
 * 1. A tape drive is a unique-open device; we refuse opens when it is already.
 * 2. We keep track of the current position on a block tape and seek
 *    before operations by forward/back spacing if necessary.
 * 3. We remember if the last operation was a write on a tape, so if a tape
 *    is open read write and the last thing done is a write we can
 *    write a standard end of tape mark (two eofs).
 * 4. We remember the status registers after the last command, using
 *    then internally and returning them to the SENSE ioctl.
 */
struct	ts_softc {
	char	sc_openf;	/* lock against multiple opens */
	char	sc_lastiow;	/* last op was a write */
	short	sc_resid;	/* copy of last bc */
	daddr_t	sc_blkno;	/* block number, for block device tape */
	daddr_t	sc_nxrec;	/* position of end of tape, if known */
	short	sc_dev_char;	/* device charactistic mask */
	struct	tty *sc_ttyp;	/* record user's tty for errors */
	struct	ts_cmd *sc_cmd;	/* the command packet */
	int	sc_cmd_vdma;	/* vdma address of the command packet */
	struct	ts_sts sc_sts;	/* status packet, for returned status */
	int	sc_sts_vdma;	/* vdma address for status packet */
	struct	ts_char sc_char; /* characteristics packet */
	int	sc_char_vdma;	/* vdma address for characteristics */
#ifdef	QBUS
# define PGMADDR(x)	(x)->tsdb = sc->sc_cmd_vdma | \
				(hiword(sc->sc_cmd_vdma) & 0x03)
#else	QBUS
# define PGMADDR(x)	{ (x)->tssr = ((sc->sc_cmd_vdma >> 16) & 0xFF);\
			(x)->tsdb = sc->sc_cmd_vdma; }
#endif	QBUS
} ts_softc[NTS];

int tsctlr_qic2 = 0;		/* bit per controller */

/*
 * States for qm->qm_tab.b_active, the per controller state flag.
 * This is used to sequence control in the driver.
 */
#define	SSEEK	1		/* seeking */
#define	SIO	2		/* doing seq i/o */
#define	SCOM	3		/* sending control command */
#define	SREW	4		/* sending a drive rewind */

/*
 * Determine if there is a controller for a ts at address reg.  Our goal is to 
 * make the device interrupt.
 */
tsprobe(tsaddr,tsunit)
register struct tsdevice *tsaddr;
{
	register struct ts_softc *sc = &ts_softc[tsunit];

	tsaddr->tssr = TSSR_RESET;
	DELAY(200);
	if ((tsaddr->tssr & TS_NBA) == 0)
		return(0);

	sc->sc_dev_char = 0;
	tsaddr->tssr = TSSR_RESET;	/* subsystem initialize */
	tswait(tsaddr);
	if (sc->sc_cmd == 0) {
	    sc->sc_cmd = (struct ts_cmd *)
		(0x400 + (sizeof(struct ts_cmd) * tsunit)); /* KLUDGE */
	    sc->sc_cmd_vdma = svqballoc(SYSV_BASE+0x400 + 
		(sizeof(struct ts_cmd) * tsunit), sizeof(struct ts_cmd));
	    sc->sc_sts_vdma = svqballoc(&sc->sc_sts, sizeof(struct ts_sts));
	    sc->sc_char_vdma = svqballoc(&sc->sc_char, sizeof(struct ts_char));
	}

	if (tssetchar(tsaddr, tsunit, 1))
		return(0);

#ifndef	QBUS
	if (!badaddr(((u_short *)tsaddr)+2, 2) && battery_clock == 0  && 
	   ((*(((u_short *)tsaddr)+2))&0x1) == 0 ) {
		battery_clock = ((u_short *)tsaddr)+2;
		ts_battery_clock = 1;
	} else
		ts_battery_clock = 0;
#endif	QBUS
	clevmax = clev_biomax;
	clev_bio = MAX(clev, clev_bio);
	return (sizeof (struct tsdevice));
}
/*
 * TS11 only supports one drive per controller; check for qi_slave == 0.
 */
tsslave(qi, reg)
struct qb_device *qi;
caddr_t reg;
{
	if (qi->qi_slave)		/* non-zero slave not allowed */
		return(0);
	return (1);
}

/*
 * Record attachment of the unit to the controller.
 */
tsattach(qi)
struct qb_device *qi;
{
#ifdef	QBUS
	if (qi->qi_flags == 0) {	/* 0 -> QIC2 */
		tsctlr_qic2 |= 1 << qi->qi_ctlr;
		printf("	1/4\" QIC2 tape drive");
	} else
		printf("	1/2\" tape drive");
#else	QBUS
	struct tsdevice *tsaddr = (struct tsdevice *)qi->qi_mi->qm_addr;

	if (badaddr(((u_short *)tsaddr)+3, 2)) {
		tsctlr_qic2 |= 1 << qi->qi_ctlr;
		printf("	1/4\" QIC2 tape drive");
	} else
		printf("	1/2\" tape drive");
	if (ts_battery_clock)
		printf(" with battery clock");
#endif	QBUS
	return (1);
}

/*
 * Open the device.  Tapes are unique open devices, so we refuse if it is 
 * already open. We also check that a tape is available, and don't block waiting
 * here; if you want to wait for a tape you should timeout in user code.
 */
tsopen(dev, flag)
dev_t dev;
int flag;
{
	register int tsunit;
	register struct qb_device *qi;
	register struct ts_softc *sc;
	int s;

	tsunit = TSUNIT(dev);
	if (tsunit>=NTS || (qi = tsdinfo[tsunit]) == 0 || qi->ui_alive == 0)
		return (ENXIO);
	if ((sc = &ts_softc[tsunit])->sc_openf)
		return (EBUSY);
	/*
	 * Lock the file as soon as possible. The subsequent tssetchar calls
	 * can take up to a couple of seconds apiece, in which time some
	 * other process has ample time to get it also. This is not a
	 * problem in earlier versions, because the drive characteristics. 
	 * Note!! All premature RETURNS from this routine need to unlock
	 *	  this device. 
	 */
	sc->sc_openf = 1;

	/* get drive characteristics from minor device number */
	sc->sc_dev_char = 0;
	if (minor(dev) & T_FIXEDREC)
		sc->sc_dev_char |= TS_RAW;
	if (minor(dev) & T_QIC11)
		sc->sc_dev_char |= TS_Q11;
	if (minor(dev) & T_FMT)
		sc->sc_dev_char |= TS_FMT;

	if (tsinit(tsunit)) {
		sc->sc_openf = 0;	/* unlock file when error */	
		return (ENXIO);
	}
	tscommand(dev, TS_SENSE, 1);
	if ((sc->sc_sts.s_xs0&TS_ONL) == 0) {
		tprintf(sc->sc_ttyp, "ts%d: not online\n", tsunit);
		sc->sc_openf = 0;	/* unlock file when error */	
		return (EIO);
	}
	if ((flag&(FREAD|FWRITE)) == FWRITE && (sc->sc_sts.s_xs0&TS_WLK)) {
		tprintf(sc->sc_ttyp, "ts%d: no write ring\n", tsunit);
		sc->sc_openf = 0;	/* unlock file when error */	
		return (EIO);
	}
	sc->sc_blkno = (daddr_t)0;
	sc->sc_nxrec = INF;
	sc->sc_lastiow = 0;
	sc->sc_ttyp = u.u_ttyp;
	return (0);
}

/*
 * Close tape device. If tape was open for writing or last operation was a 
 * write, then write two EOF's and backspace over the last one. Unless this 
 * is a non-rewinding special file, rewind the tape. Make the tape available 
 * to others.
 */
tsclose(dev, flag)
register dev_t dev;
register flag;
{
	register struct ts_softc *sc = &ts_softc[TSUNIT(dev)];

	if (flag == FWRITE || (flag&FWRITE) && sc->sc_lastiow) {
		tscommand(dev, TS_WEOF, 1);
		tscommand(dev, TS_WEOF, 1);
		tscommand(dev, TS_SREV, 1);
	}
	/*
	 * 0 count means don't hang waiting for rewind complete rather ctsbuf 
	 * stays busy until the operation completes preventing further opens 
	 * from completing by preventing a TS_SENSE from completing.
	 */
	if ((minor(dev)&T_NOREWIND) == 0) {
		/* BUG in CDC streaming 1/2 inch needs a delay here */
		sleep((caddr_t)&lbolt, PRIBIO);
		sleep((caddr_t)&lbolt, PRIBIO);
		tscommand(dev, TS_REW, 0);
	}
	sc->sc_openf = 0;
}

/*
 * Initialize the TS11.  Set up device characteristics.
 */
tsinit(tsunit)
register int tsunit;
{
	register struct ts_softc *sc = &ts_softc[tsunit];
	register struct qb_ctlr *qm = tscinfo[tsunit];
	register struct tsdevice *tsaddr = (struct tsdevice *)qm->qm_addr;
	int i;

	/* Initialize the TS11 controller. Set the characteristics. */
	if (tsaddr->tssr & (TS_NBA|TS_OFL))
		tsaddr->tssr = TSSR_RESET;	/* subsystem initialize */
	if (tssleep(tsaddr))
		return(1);
	if (tssetchar(tsaddr, tsunit, 0))
		return(1);
	return(0);
}

tssetchar(tsaddr, tsunit, ie)
struct tsdevice *tsaddr;
{
	register struct ts_softc *sc = &ts_softc[tsunit];

	if (hiword(sc->sc_cmd_vdma)&~0x03)
		panic("ts: command packet not in 18 bit land\n"); 
	if (sc->sc_cmd_vdma & 0x03)
		panic("ts: command packet not long aligned\n"); 

	sc->sc_char.char_ladr = loword(sc->sc_sts_vdma);
	sc->sc_char.char_hadr = hiword(sc->sc_sts_vdma);
	sc->sc_char.char_size = sizeof(struct ts_sts);
	sc->sc_char.char_mode = TS_ESS | sc->sc_dev_char;
	sc->sc_char.ext_char_mode = TS_WRB;

	sc->sc_cmd->c_cmd = (ie ? TS_IE : 0) | TS_ACK | TS_CVC | TS_SETCHR;
	sc->sc_cmd->c_loba = loword(sc->sc_char_vdma);
	sc->sc_cmd->c_hiba = hiword(sc->sc_char_vdma);
	sc->sc_cmd->c_size = sizeof(struct ts_char);

	PGMADDR(tsaddr);
	tswait(tsaddr);		/* hope that EOT errs dont make it this far */ 

	if (sc->sc_sts.s_xs0&TS_ILC) {
		printf("ts%d: Configuration Failed.\n", tsunit);
		return(1);
	}
	return (tsaddr->tssr & TS_NBA);		/* 0 => all ok */
}

/*
 * Execute a command on the tape drive a specified number of times.
 */
tscommand(dev, com, count)
dev_t dev;
int com, count;
{
	register struct buf *bp;
	register int s;

	bp = &ctsbuf[TSUNIT(dev)];
	s = spl6();
	while (bp->b_flags&B_BUSY) {
		/*
		 * This special check is because B_BUSY never gets cleared in 
		 * the non-waiting rewind case.
		 */
		if (bp->b_repcnt == 0 && (bp->b_flags&B_DONE))
			break;
		bp->b_flags |= B_WANTED;
		sleep((caddr_t)bp, PRIBIO);
	}
	bp->b_flags = B_BUSY|B_READ;
	splx(s);
	bp->b_dev = dev;
	bp->b_repcnt = count;
	bp->b_command = com;
	bp->b_blkno = 0;
	tsstrategy(bp);
	/*
	 * In case of rewind from close, don't wait. This is the only case 
	 * where count can be 0.
	 */
	if (count == 0)
		return;
	iowait(bp);
	if (bp->b_flags&B_WANTED)
		wakeup((caddr_t)bp);
	bp->b_flags &= B_ERROR;
}

/*
 * Queue a tape operation.
 */
tsstrategy(bp)
register struct buf *bp;
{
	int tsunit = TSUNIT(bp->b_dev);
	register struct qb_ctlr *qm;
	register struct buf *dp;
	register int s;

	/*
	 * Put transfer at end of controller queue
	 */
	bp->av_forw = NULL;
	qm = tsdinfo[tsunit]->qi_mi;
	s = spl6();
	dp = &tsutab[tsunit];
	if (dp->b_actf == NULL)
		dp->b_actf = bp;
	else
		dp->b_actl->av_forw = bp;
	dp->b_actl = bp;
	qm->qm_tab.b_actf = qm->qm_tab.b_actl = dp;
	/*
	 * If the controller is not busy, get it going.
	 */
	if (qm->qm_tab.b_active == 0)
		tsstart(qm);
	splx(s);
}

/*
 * Start activity on a ts controller.
 */
tsstart(qm)
register struct qb_ctlr *qm;
{
	register struct buf *bp;
	register struct tsdevice *tsaddr = (struct tsdevice *)qm->qm_addr;
	register struct ts_softc *sc;
	register struct ts_cmd *tc;
	register struct qb_device *qi;
	int tsunit, cmd, ioaddr;
	daddr_t blkno;

	/*
	 * Start the controller if there is something for it to do.
	 */
loop:
	if ((bp = qm->qm_tab.b_actf->b_actf) == NULL)
		return;
	tsunit = TSUNIT(bp->b_dev);
	qi = tsdinfo[tsunit];
	sc = &ts_softc[tsunit];
	tc = sc->sc_cmd;
	/*
	 * Default is that last command was NOT a write command; if we do a 
	 * write command we will notice this in tsintr().
	 */
	sc->sc_lastiow = 0;
	if (sc->sc_openf < 0 || (tsaddr->tssr&TS_OFL)) {
		/*
		 * Have had a hard error on a non-raw tape or the tape unit is 
		 * now unavailable (e.g. taken off line).
		 */
		bp->b_flags |= B_ERROR;
		goto next;
	}
	if (bp == &ctsbuf[TSUNIT(bp->b_dev)]) {
		/*
		 * Execute control operation with the specified count.
		 */
		qm->qm_tab.b_active = (bp->b_command == TS_REW) ? SREW : SCOM;
		tc->c_repcnt = bp->b_repcnt;
		goto dobpcmd;
	}
	/*
	 * The following checks handle boundary cases for operation
	 * on non-raw tapes.  On raw tapes the initialization of
	 * sc->sc_nxrec by tsminphys causes them to be skipped normally
	 * (except in the case of retries).
	 */
	if (bdbtofsb(bp->b_blkno) > sc->sc_nxrec) {
		/*
		 * Can't read past known end-of-file.
		 */
		bp->b_flags |= B_ERROR;
		bp->b_error = ENXIO;
		goto next;
	}
	if (bdbtofsb(bp->b_blkno) == sc->sc_nxrec && bp->b_flags&B_READ) {
		/*
		 * Reading at end of file returns 0 bytes.
		 */
		bp->b_resid = bp->b_bcount;
		clrbuf(bp);
		goto next;
	}
	if ((bp->b_flags&B_READ) == 0)
		/*
		 * Writing sets EOF
		 */
		sc->sc_nxrec = bdbtofsb(bp->b_blkno) + 1;
	/*
	 * If the data transfer command is in the correct place,
	 * set up all the registers except the csr, and give
	 * control over to the UNIBUS adapter routines, to
	 * wait for resources to start the i/o.
	 */
	if ((blkno = sc->sc_blkno) == bdbtofsb(bp->b_blkno)) {
		if ((bp->b_flags&B_READ) == 0)
			cmd = TS_WCOM;
		else
			cmd = TS_RCOM;
		if (qm->qm_tab.b_errcnt)
			cmd |= TS_RETRY;
		qm->qm_tab.b_active = SIO;
		tc->c_cmd = TS_ACK | TS_CVC | TS_IE | cmd;
		if (bp->b_dev & T_NOSWAP)
			tc->c_cmd |= TS_SWB;
		tc->c_size = bp->b_bcount + (bp->b_bcount & 1);
		ioaddr = qbaddr(bp);
		tc->c_loba = loword(ioaddr);
		tc->c_hiba = hiword(ioaddr);
		PGMADDR(tsaddr);
		timeout(tstimer, (caddr_t)bp->b_dev, 5*(60*hz));/* 5 mins */
		return;
	}
	/*
	 * Tape positioned incorrectly;
	 * set to seek forwards or backwards to the correct spot.
	 * This happens for raw tapes only on error retries.
	 */
	qm->qm_tab.b_active = SSEEK;
	if (blkno < bdbtofsb(bp->b_blkno)) {
		bp->b_command = TS_SFORW;
		tc->c_repcnt = bdbtofsb(bp->b_blkno) - blkno;
	} else {
		bp->b_command = TS_SREV;
		tc->c_repcnt = blkno - bdbtofsb(bp->b_blkno);
	}
dobpcmd:
	/*
	 * Do the command in bp.
	 */
	tc->c_cmd = TS_ACK | TS_CVC | TS_IE | bp->b_command;
	PGMADDR(tsaddr);
	timeout(tstimer, (caddr_t)bp->b_dev, 15*(60*hz));	/* 15 mins */
	return;

next:
	/*
	 * Done with this operation due to error or the fact that it doesn't 
	 * do anything. Dequeue the transfer 
	 * and continue processing this slave.
	 */
	qm->qm_tab.b_errcnt = 0;
	qm->qm_tab.b_actf->b_actf = bp->av_forw;
	iodone(bp);
	goto loop;
}

/*
 * Ts interrupt routine.
 */
tsintr(ctlr)
int ctlr;
{
	register struct buf *bp;
	register struct qb_ctlr *qm = tscinfo[ctlr];
	register struct tsdevice *tsaddr;
	register struct ts_softc *sc;
	int tsunit;
	register state;

	if ((bp = qm->qm_tab.b_actf->b_actf) == NULL) {
		printf("ts%d: unexpected interrupt\n", ctlr);
		return;
	}
	untimeout(tstimer, (caddr_t)bp->b_dev); 
	tsunit = TSUNIT(bp->b_dev);
	tsaddr = (struct tsdevice *)qm->qm_addr;
	/*
	 * If last command was a rewind, and tape is still rewinding, wait for 
	 * the rewind complete interrupt.
	 *
	 * SHOULD NEVER GET AN INTERRUPT IN THIS STATE.
	 */
	if (qm->qm_tab.b_active == SREW) {
		qm->qm_tab.b_active = SCOM;
		if ((tsaddr->tssr&TS_SSR) == 0)
			return;
	}
	/*
	 * An operation completed... record status
	 */
	sc = &ts_softc[tsunit];
	if ((bp->b_flags & B_READ) == 0)
		sc->sc_lastiow = 1;
	state = qm->qm_tab.b_active;
	qm->qm_tab.b_active = 0;
	/*
	 * Check for errors.
	 */
	if (tsaddr->tssr&TS_SC) {
		switch (tsaddr->tssr & TS_TC) {
		case TS_UNREC:		/* unrecoverable */
		case TS_FATAL:		/* fatal error */
		case TS_ATTN:		/* attention (shouldn't happen) */
		case TS_RECNM:		/* recoverable, no motion */
			break;

		case TS_SUCC:		/* success termination */
			printf("ts%d: success\n", TSUNIT(minor(bp->b_dev)));
			goto ignoreerr;

		case TS_ALERT:		/* tape status alert */
			/*
			 * If we hit the end of the tape file, update position.
			 */
			if (sc->sc_sts.s_xs0 & (TS_TMK|TS_EOT)) {
				tsseteof(bp);		/* set blkno, nxrec */
				state = SCOM;		/* force completion */
				/*
				 * Stuff bc so it will be unstuffed correctly
				 * later to get resid.
				sc->sc_sts.s_rbpcr = bp->b_bcount;
				 */
				goto opdone;
			}
			/*
			 * If reading raw tape and the record was too long
			 * or too short, then we don't consider this an error.
			 */
			if ((rtsbuf[tsunit].b_flags&B_PHYS) &&
			    (bp->b_flags&B_READ) &&
			    sc->sc_sts.s_xs0&(TS_RLS|TS_RLL))
				goto ignoreerr;
		case TS_RECOV:		/* recoverable, tape moved */
			/*
			 * If this was an i/o operation retry up to 8 times.
			 */
			if (state==SIO)
#ifdef	vax
				if (++qm->qm_tab.b_errcnt < 7)
#endif	vax
#ifdef	is68k
				/* no retries for QIC2 controllers! */
				if (((tsctlr_qic2&(1<<ctlr)) == 0) && 
				    ++qm->qm_tab.b_errcnt < 7)
#endif	is68k
					goto opcont;
				else
					sc->sc_blkno++;
			else if (sc->sc_openf>0 && bp != &rtsbuf[tsunit])
				/*
				 * Non-i/o errors on non-raw tape: close.
				 */
				sc->sc_openf = -1;
			break;

		case TS_REJECT:		/* function reject */
			if (state == SIO && sc->sc_sts.s_xs0 & TS_WLE)
				tprintf(sc->sc_ttyp, "ts%d: write locked\n",
				    TSUNIT(bp->b_dev));
			if ((sc->sc_sts.s_xs0 & TS_ONL) == 0)
				tprintf(sc->sc_ttyp, "ts%d: offline\n",
				    TSUNIT(bp->b_dev));
			break;
		}
		/*
		 * Couldn't recover error
		 */
		tprintf(sc->sc_ttyp, "ts%d: hard error bn%d xs0=%b", 
		    TSUNIT(bp->b_dev), bp->b_blkno, sc->sc_sts.s_xs0, TSXS0_BITS);
		if (sc->sc_sts.s_xs1)
			tprintf(sc->sc_ttyp, 
				" xs1=%b", sc->sc_sts.s_xs1, TSXS1_BITS);
		if (sc->sc_sts.s_xs2)
			tprintf(sc->sc_ttyp, 
				" xs2=%b", sc->sc_sts.s_xs2, TSXS2_BITS);
		if (sc->sc_sts.s_xs3)
			tprintf(sc->sc_ttyp, 
				" xs3=%b", sc->sc_sts.s_xs3, TSXS3_BITS);
		tprintf(sc->sc_ttyp, "\n");
		bp->b_flags |= B_ERROR;
		goto opdone;
	}
	/*
	 * Advance tape control FSM.
	 */
ignoreerr:
	switch (state) {

	case SIO:
		/*
		 * Read/write increments tape block number
		 */
		sc->sc_blkno++;
		goto opdone;

	case SCOM:
		/*
		 * For forward/backward space record update current position.
		 */
		if (bp == &ctsbuf[TSUNIT(bp->b_dev)])
			switch (bp->b_command) {
			case TS_SFORW:
				sc->sc_blkno += bp->b_repcnt; break;

			case TS_SREV:
				sc->sc_blkno -= bp->b_repcnt; break;
			}
		goto opdone;

	case SSEEK:
		sc->sc_blkno = bdbtofsb(bp->b_blkno);
		goto opcont;

	default:
		panic("tsintr");
	}
opdone:
	if (sc->sc_sts.s_xs0 & TS_EOT) {
/*		printf("ts%d: unexpected physical EOT\n",tsunit); /**/
		bp->b_flags |= B_ERROR;
		bp->b_error = ENOSPC;
		if (tsctlr_qic2&(1<<ctlr)) {
			tsreset(tsunit);
			return;
		}
	}
	/*
	 * Reset error count and remove from device queue.
	 */
	qm->qm_tab.b_errcnt = 0;
	qm->qm_tab.b_actf->b_actf = bp->av_forw;
	sc->sc_resid = bp->b_resid = sc->sc_sts.s_rbpcr;
	iodone(bp);
opcont:
	tsstart(qm);
}

tsseteof(bp)
register struct buf *bp;
{
	register int tsunit = TSUNIT(bp->b_dev);
	register struct ts_softc *sc = &ts_softc[tsunit];

	if (bp == &ctsbuf[TSUNIT(bp->b_dev)]) {
		if (sc->sc_blkno > bdbtofsb(bp->b_blkno)) {
			/* reversing */
			sc->sc_nxrec = bdbtofsb(bp->b_blkno) - sc->sc_sts.s_rbpcr;
			sc->sc_blkno = sc->sc_nxrec;
		} else {
			/* spacing forward */
			sc->sc_blkno = bdbtofsb(bp->b_blkno) + sc->sc_sts.s_rbpcr;
			sc->sc_nxrec = sc->sc_blkno - 1;
		}
		return;
	} 
	/* eof on read */
	sc->sc_nxrec = bdbtofsb(bp->b_blkno);
}

tsread(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	register int tsunit = TSUNIT(dev);
	register struct qb_device *qi;
	int errno;

	if (tsunit>=NTS || (qi = tsdinfo[tsunit]) == 0 || qi->ui_alive == 0)
		return (ENXIO);
	return (physio(tsstrategy, &rtsbuf[TSUNIT(dev)], dev, B_READ, 
			tsminphys, uio));
}

tswrite(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	register int tsunit = TSUNIT(dev);
	register struct qb_device *qi;
	int errno;

	if (tsunit>=NTS || (qi = tsdinfo[tsunit]) == 0 || qi->ui_alive == 0)
		return (ENXIO);
	return (physio(tsstrategy, &rtsbuf[TSUNIT(dev)], dev, B_WRITE, 
			tsminphys, uio));
}

tsreset(ctlr)
int	ctlr;
{
	register struct qb_ctlr *qm;
	register struct buf *bp;
	register struct tsdevice *tsaddr;
	int s;

	if ((qm = tscinfo[ctlr]) == 0 || qm->qm_alive == 0 )
		return(0); 

	s = spl6();
	if (ts_softc[ctlr].sc_openf) {
		if ((bp = qm->qm_tab.b_actf->b_actf) != NULL)
			untimeout(tstimer,(caddr_t)bp->b_dev);
		ts_softc[ctlr].sc_openf = -1;
	}
	qm->qm_tab.b_active = 0;  
	tsaddr = (struct tsdevice *)qm->qm_addr;
	tsaddr->tssr = TSSR_RESET;
	tsstart(qm);
	splx(s);
}

tsioctl(dev, cmd, data, flag)
caddr_t data;
dev_t dev;
{
	int tsunit = TSUNIT(dev);
	register struct ts_softc *sc = &ts_softc[tsunit];
	register struct buf *bp = &ctsbuf[TSUNIT(dev)];
	register callcount;
	int fcount;
	struct mtop *mtop;
	struct mtget *mtget;
	/* we depend of the values and order of the MT codes here */
	static tsops[] =
	 {TS_WEOF,TS_SFORWF,TS_SREVF,TS_SFORW,TS_SREV,TS_REW,TS_OFFL,TS_SENSE,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, TS_RET};

	switch (cmd) {

	case MTIOCTOP:	/* tape operation */
		mtop = (struct mtop *)data;
		switch (mtop->mt_op) {

		case MTWEOF:
			callcount = mtop->mt_count;
			fcount = 1;
			break;

		case MTFSF: case MTBSF:
		case MTFSR: case MTBSR:
			callcount = 1;
			fcount = mtop->mt_count;
			break;

		case MTREW:
			/* BUG in CDC streaming 1/2 inch needs a delay here */
			sleep((caddr_t)&lbolt, PRIBIO);
			sleep((caddr_t)&lbolt, PRIBIO);

		case MTOFFL: case MTNOP: case MTRET:
			callcount = 1;
			fcount = 1;
			break;

		case MTRST:
			tsreset(tsunit);	/* mt reset uses this ioctl */
			if (tsinit(tsunit))	/* reset characteristics */	
				return(ENXIO);
		 	return(0);	

		default:
			return (ENXIO);
		}
		if (callcount <= 0 || fcount <= 0)
			return (EINVAL);
		while (--callcount >= 0) {
			tscommand(dev, tsops[mtop->mt_op], fcount);
			if ((mtop->mt_op == MTFSR || mtop->mt_op == MTBSR) &&
			    bp->b_resid)
				return (EIO);
			if ((bp->b_flags&B_ERROR) || sc->sc_sts.s_xs0&TS_BOT)
				break;
		}
		return (geterror(bp));

	case MTIOCGET:
		mtget = (struct mtget *)data;
		mtget->mt_dsreg = 0;
		mtget->mt_erreg = sc->sc_sts.s_xs0;
		mtget->mt_resid = sc->sc_resid;
		mtget->mt_type = MT_ISTS;
		break;

	default:
		return (ENXIO);
	}
	return (0);
}

#define	DBSIZE	20
extern int dumpsize;

tsdump(dev)
dev_t dev;
{
	register int tsunit = TSUNIT(dev);
	register struct ts_softc *sc = &ts_softc[tsunit];
	register struct qb_device *qi = tsdinfo[tsunit];
	register struct tsdevice *tsaddr;
	int blk, num;
	int start;

	if (tsunit >= Nts || qi == 0 || qi->qi_alive == 0)
		return ENXIO;
	num = ctod(dumpsize);		/* memory size in disk blocks */
	start = 0x0;			/* start dumping at physical 0 */
	tsaddr = (struct tsdevice *)qi->qi_mi->qm_addr;
	tsaddr->tssr = TSSR_RESET;
	DELAY(100);
	if ((tsaddr->tssr & TS_NBA) == 0)
		return EFAULT;
	if (tsinit(tsunit))
		return EFAULT;
	if (tsdumpcom(dev, start, TS_SENSE, 0, 1))
		return EFAULT;
	if (sc->sc_sts.s_xs0&TS_ONL == 0) {
		printf("ts%d: not online\n", tsunit);
		return EFAULT;
	}
	if (sc->sc_sts.s_xs0&TS_WLK) {
		printf("ts%d: no write ring\n", tsunit);
		return EFAULT;
	}
	while (num > 0) {
		blk = num > DBSIZE ? DBSIZE : num;
		if (tsdumpcom(dev, start, TS_WCOM, dtob(blk), 1))
			return EIO;
		start += dtob(blk);
		num -= blk;
	}
	if (tsdumpcom(dev, 0, TS_WEOF, 0, 1))
		return EIO;
	if (tsdumpcom(dev, 0, TS_WEOF, 0, 1))
		return EIO;
	if (tsdumpcom(dev, 0, TS_REW, 0, 1))
		return EIO;
	tsaddr->tssr = TSSR_RESET;
	if (tswait(tsaddr,tsunit))
		return EIO;
	return 0;
}

tsdumpcom(dev, ioaddr, cmd, size, count)
dev_t dev;
caddr_t ioaddr;
int cmd;
int size;
int count;
{
	register int tsunit = TSUNIT(dev);
	register struct tsdevice *tsaddr = 
			(struct tsdevice *)tsdinfo[tsunit]->qi_mi->qm_addr;
	register struct ts_softc *sc = &ts_softc[tsunit];
	register struct ts_cmd *tc = sc->sc_cmd;

dumpretry:
	tc->c_cmd = TS_ACK|TS_CVC|cmd;
	if (dev & T_NOSWAP)
		tc->c_cmd |= TS_SWB;
	tc->c_repcnt = count;
	tc->c_loba = loword(ioaddr);
	tc->c_hiba = hiword(ioaddr);
	tc->c_size = size + (size & 1);
	PGMADDR(tsaddr);
	if (tswait(tsaddr,tsunit))
		return EIO;
	if (tsaddr->tssr&TS_SC) {
		switch (tsaddr->tssr & TS_TC) {
		case TS_UNREC:		/* unrecoverable */
		case TS_FATAL:		/* fatal error */
		case TS_ATTN:		/* attention (shouldn't happen) */
		case TS_REJECT:		/* function reject */
			printf("ts%d: unrecoverable error: ",tsunit);
			return EIO;

		case TS_RECNM:		/* recoverable, no motion */
			goto dumpretry;

		case TS_RECOV:		/* recoverable, tape moved */
			printf("ts%d: recoverable error, but tough: ",tsunit);
			return EIO;

		case TS_SUCC:		/* success termination */
			return 0;

		case TS_ALERT:		/* tape status alert */
			if (sc->sc_sts.s_xs0 & TS_EOT) 
				printf("ts%d: end of tape: ",tsunit);
			else if (sc->sc_sts.s_xs0 & TS_TMK)
				return 0;
			else
				printf("ts%d: unknown error: ",tsunit);
			return EIO;
		}
		printf("ts%d: hard error xs0=%b", TSUNIT(dev),
					sc->sc_sts.s_xs0, TSXS0_BITS);
		if (sc->sc_sts.s_xs1)
			printf(" xs1=%b", sc->sc_sts.s_xs1, TSXS1_BITS);
		if (sc->sc_sts.s_xs2)
			printf(" xs2=%b", sc->sc_sts.s_xs2, TSXS2_BITS);
		if (sc->sc_sts.s_xs3)
			printf(" xs3=%b", sc->sc_sts.s_xs3, TSXS3_BITS);
		if ((sc->sc_sts.s_len >= (sizeof(struct ts_sts)-4)) && 
					(sc->sc_sts.s_xs4))
		printf(" xs4=%b", sc->sc_sts.s_xs4, TSXS4_BITS);
		printf("\n");
		printf("tssr= %b \n", tsaddr->tssr, TSSR_BITS);
		return EIO;
	}
	return 0;
}

tstimer(dev)
int dev;
{
	printf("ts%d: lost interrupt\n", TSUNIT(dev));
	tsreset(TSUNIT(dev));
}

/* Do not tie up the kernel with a tswait when doing a conventional open. */
tssleep(tsaddr,tsunit)
register struct tsdevice *tsaddr;
int tsunit;
{
	register int i;
	register u_short addr;

	for ( i = 0 ; i < MAXTIME ; i++) {
		DELAY(200);
		if ((tsaddr->tssr & TS_SSR) != 0)
			return (0);
		timeout(wakeup, (caddr_t)&tswaitloc[tsunit], 2*hz); /* 2 sec */
		sleep((caddr_t)&(tswaitloc[tsunit]), PRIBIO);
	}
	return (1);
}

tswait(tsaddr)
register struct tsdevice *tsaddr;
{
	register u_short s;

	do {
		DELAY(1000);
		s = tsaddr->tssr;
	} while ((s & TS_SSR) == 0);
}

tseof(tsaddr)
struct tsdevice *tsaddr;
{

	tswait(tsaddr);
#ifdef notyet
	tsaddr->tscs = TS_WEOF | TM_GO;
#endif
}

tsminphys(bp, uio)
	struct buf *bp;
	register struct uio *uio;
{
	int tsunit = TSUNIT(bp->b_dev);
	register struct ts_softc *sc = &ts_softc[tsunit];
	int a;

	bp->b_bcount = MIN(bp->b_bcount, (63*1024));

	a = bdbtofsb(uio->uio_offset >> 9);
	sc->sc_blkno = a;
	sc->sc_nxrec = a + 1;
}
#endif
