/*
 * QBUS: RX01/2 floppy disk device driver
 */
/*#define RXDEBUG /* */

#include "rx.h"
#if	NRX > 0
/*
 * TODO:
 *	- clean up the code for multisector transfers using a 'transfer in 
 *	  progress' flag
 *	- Test Deleted Data read/write 
 *	- Test error handling/reporting and 'volume valid' stuff
 *
 * 	Note: If the drive subsystem is	powered off at boot time, the controller
 *	won't interrupt!
 */

#include "../machine/pte.h"

#include "param.h"
#include "buf.h"
#include "systm.h"
#include "conf.h"
#include "errno.h"
#include "time.h"
#include "kernel.h"
#include "uio.h"
#include "file.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/rxreg.h"

#define b_cylin	b_resid

/* 
 * per-controller data
 */
struct	rx_ctlr {
	int	rxc_state;	/* controller state */
#define	RXS_READ	1	/* read started	*/
#define	RXS_EMPTY	2	/* empty started */
#define	RXS_FILL	3	/* fill started	*/
#define	RXS_WRITE	4	/* write started */
#define	RXS_FORMAT	5	/* format started */
#define	RXS_RDSTAT	6	/* status read started */
#define	RXS_RDERR	7	/* error read started */
#define	RXS_IDLE	8	/* device is idle */
	u_short	rxc_rxcs;	/* extended error status */
	u_short	rxc_rxdb;
	u_short	rxc_rxxt[4];
	int	rxc_tocnt;	/* for watchdog routine */
#define	RX_MAXTIMEOUT	30	/* # seconds to wait before giving up */
} rx_ctlr[NRX];

/* 
 * per-drive buffers
 */
struct buf	rrxbuf[Nrx];	/* buffers for raw I/O */
struct buf	erxbuf[Nrx];	/* buffers for reading error status */
struct buf	rxutab[Nrx];	/* per drive buffers */

/*
 * per-drive data
 */
struct rx_softc {
	int	sc_flags;	/* drive status flags */
#define	RXF_DIRECT	0x01	/* if set: use direct sector mapping */
#define	RXF_TRKONE	0x02	/* if set: start mapping on track 1 */
#define	RXF_DSIDE	0x04	/* use double sided */
#define	RXF_DEVTYPE	0x07	/* mapping flags */
#define	RXF_LOCK	0x10	/* exclusive use */
#define	RXF_DDMK	0x20	/* deleted-data mark detected */
#define	RXF_USEWDDS	0x40	/* write deleted-data sector */
#define	RXF_FORMAT	0x80	/* format in progress */
#define	RXF_BAD		0x100	/* drive bad, cannot be used */
#define	RXF_SDEN	0x000	/* diskette is double density */
#define	RXF_DDEN	0x200	/* diskette is double density */
	int	sc_csbits;	/* constant bits for CS register */
	int	sc_open;	/* count number of opens */
	int	sc_offset;	/* raw mode kludge to avoid restricting */
				/* single sector transfers to start on */
				/* DEV_BSIZE boundaries */
	/*
	 * The rest of this structure is used to store temporaries while 
	 * simulating multi sector transfers
	 */
	caddr_t	sc_uaddr;	/* Qbus base address */
	long	sc_bcnt;	/* total transfer count */
	long	sc_resid;	/* no. of bytes left to transfer */
} rx_softc[Nrx];

extern struct rxerr {
	short	rxcs;
	short	rxdb;
	short	rxxt[4];	/* error code dump from controller */
} rxerr[Nrx];
/* End of per-drive data */

struct	qb_device *rxdinfo[Nrx];
struct	qb_ctlr *rxcinfo[NRX];

struct buf *savebp;

int rxprobe(), rxslave(), rxattach(), rxintr(), rxwatch(), rxphys();
int rxminphys();
u_short *rxstd[] = { (u_short *)0x3FFE78, (u_short *)0x3FFE68, 0 };
struct qb_driver RXdriver =
  { rxprobe, rxslave, rxattach, rxstd, "rx", rxdinfo, "RX", rxcinfo };

int	rxwstart;
#define	RXUNIT(dev)		(minor(dev)>>3)
#define	MASKREG(reg)		(reg&0xffff)
#define WAITTREQ(rxaddr)	while ((rxaddr->rxcs&RX_TREQ) == 0);
#define WAITDONE(rxaddr)	while ((rxaddr->rxcs&RX_DONE) == 0);

/*
 * constants related to floppy data capacity
 */
#define	RXSECS	2002				/* # sectors on a floppy */
#define	DDSTATE	(sc->sc_flags&RXF_DDEN)		/* non zero -> double density */
#define	DSSTATE	(sc->sc_flags&RXF_DSIDE)	/* non zero -> double sided */
#define	NBPS	(DDSTATE ? 256 : 128)		/* # bytes per sector */
#define	RXSIZE	(NBPS*RXSECS*(DSSTATE ? 2 : 1))	/* # bytes per disk */
#define	SECMASK	(DDSTATE ? 0xff : 0x7f)		/* shifted-out bits of offset */

#define	B_CTRL		0x80000000		/* control (format) request */
#define B_RDSTAT	0x40000000		/* read drive status (open) */

rxprobe (rxaddr)
	register struct rxdevice *rxaddr;
{
	extern int cvec;

	rxaddr->rxcs = RX_INTR;
	DELAY(200);
	rxaddr->rxcs = 0;
	clevmax = clev_biomax;
	clev_bio = MAX(clev, clev_bio);
	return (sizeof (*rxaddr));
}

rxslave(qi, rxaddr)
	struct qb_device *qi;
	register struct rxdevice *rxaddr;
{
	qi->qi_dk = 1;
	return (qi->qi_slave == 0 || qi->qi_slave == 1);
}

rxattach(qi)
	struct qb_device *qi;
{
	return(1);
}

rxopen(dev, flag)
	dev_t dev;
{ 
	register int unit = RXUNIT(dev);
	register struct rx_softc *sc;
	register struct qb_device *qi;
	struct rx_ctlr *rxc;

	if (unit >= Nrx || (qi = rxdinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	sc = &rx_softc[unit];
	if (sc->sc_open == 0 && sc->sc_csbits == 0) {
		struct buf *bp = &erxbuf[unit];
		/*
		 * lock the device while an open is in progress
		 */
		sc->sc_flags = (minor(dev) & RXF_DEVTYPE) | RXF_LOCK;
		sc->sc_csbits = RX_INTR;
		sc->sc_csbits |= qi->qi_slave == 0 ? RX_DRV0 : RX_DRV1;

		bp->b_dev = dev;
		bp->b_flags = B_RDSTAT | B_BUSY;
		bp->b_error = 0;
		bp->b_blkno = 0;
		sc->sc_offset = 0;
		sc->sc_resid  = 0;
		/*
		 * read device status to determine if a floppy is present in the
		 * drive and what density it is, and how many sides it has.
		 */
		rxstrategy(bp);
		iowait(bp);
		if (bp->b_flags & B_ERROR) {
			sc->sc_csbits = 0;
			return (bp->b_error);
		}
		if (rxwstart++ == 0) {
			rxc = &rx_ctlr[qi->qi_mi->qm_ctlr];
			rxc->rxc_tocnt = 0;
			timeout(rxwatch, (caddr_t)0, hz);  /* start watchdog */
		}
#ifdef	RXDEBUG
		printf("rxopen: csbits=%b\n", sc->sc_csbits, RXCS_BITS);
#endif	RXDEBUG
#ifdef	RXDEBUG
#endif	RXDEBUG
		printf("DISKETTE: %s sided %s density\n",
			sc->sc_flags & RXF_DSIDE ? "double" : "single",
			sc->sc_flags & RXF_DDEN  ? "double" : "single");
		sc->sc_flags &= ~RXF_LOCK;
	} else	{
		if (sc->sc_flags & RXF_LOCK)
			return(EBUSY);
	}
	sc->sc_open++;
	return (0);
}

/*ARGSUSED1*/
rxclose(dev, flag)
	dev_t dev;
{
	register struct rx_softc *sc = &rx_softc[RXUNIT(dev)];

	--sc->sc_open;
#ifdef	RXDEBUG
	printf("rxclose: dev=0x%x, sc_open=%d\n", dev, sc->sc_open);
#endif	RXDEBUG
}

rxstrategy(bp)
	register struct buf *bp;
{
	struct qb_device *qi;
	register struct buf *dp;
	struct rx_softc *sc;
	int s, unit = RXUNIT(bp->b_dev);

	if (unit >= Nrx)
		goto bad;
	qi = rxdinfo[unit];
	if (qi == 0 || qi->qi_alive == 0) 
		goto bad;
	sc = &rx_softc[unit];
	if (bp->b_blkno < 0 || dbtob(bp->b_blkno) > RXSIZE )
		goto bad;
	if (sc->sc_flags & RXF_BAD) {
		bp->b_error = EIO;
		goto dbad;
	}
	s = splx(qi->qi_mi->qm_psl);
#ifdef	RXDEBUG
	printf("rxstrat: bp=0x%x, fl=0x%x, un=%d, bl=%d, cnt=%d\n", 
		bp, bp->b_flags, unit, bp->b_blkno, bp->b_bcount);
#endif	RXDEBUG
	bp->b_cylin = bp->b_blkno;	/* don't care to calculate trackno */
	dp = &rxutab[unit];
	disksort(dp, bp);
	if (dp->b_active == 0) {
		rxustart(qi);
		bp = &qi->qi_mi->qm_tab;
		if (bp->b_actf && bp->b_active == 0)
			rxstart(qi->qi_mi);
	}
	splx(s);
	return;

bad:
	bp->b_error = ENXIO;
dbad:
	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
}

/*
 * Unit start routine.  Put this unit on the ready queue for the controller
 */
rxustart(qi)
	register struct qb_device *qi;
{
	struct buf *dp = &rxutab[qi->qi_unit];
	struct qb_ctlr *qm = qi->qi_mi;
	
	dp->b_forw = NULL;
	if (qm->qm_tab.b_actf == NULL)
		qm->qm_tab.b_actf = dp;
	else
		qm->qm_tab.b_actl->b_forw = dp;
	qm->qm_tab.b_actl = dp;
	dp->b_active++;
}

/*
 * Sector mapping routine. Two independent sets of choices are available:
 *
 * (a) The first logical sector may either be on track 1 or track 0.
 * (b) The sectors on a track may either be taken in 2-for-1 interleaved fashion
 *     or directly.
 *
 * This gives a total of four possible mapping schemes. Physical tracks on the 
 * RX02 are numbered 0-76.  Physical sectors on each track are numbered 1-26.
 * When interleaving is used, sectors on the first logical track are taken in 
 * the order 1, 3, 5, ..., 25, 2, 4, 6, ..., 26.  A skew of six sectors per 
 * track is also used (to allow time for the heads to move); hence, the sectors
 * on the second logical track are taken in the order 7, 9, 11, ..., 25, 1, 3, 
 * 5, 8, 10, 12, ..., 26, 2, 4, 6; the third logical track starts with sector 
 * 13; and so on.
 *   When the mapping starts with track 1, track 0 is the last logical track, 
 * and this track is always handled directly (without interleaving), even when 
 * the rest of the disk is interleaved.  (This is still compatible with DEC 
 * RT-11, which does not use track 0 at all.)
 */
rxmap(bp, psector, ptrack, phead)
	struct buf *bp;
	int *psector, *ptrack, *phead;
{
	register int ls, ph, pt, ps, ptoff, nspc;
	struct rx_softc *sc = &rx_softc[RXUNIT(bp->b_dev)];

	ls = (dbtob(bp->b_blkno) + (sc->sc_offset - sc->sc_resid)) / NBPS;
	nspc = DSSTATE ? 52 : 26;	
	ph = (ls%nspc >= 26);
	pt = ls / nspc;
	ps = ls % 26;
	/*
	 * The "physical track offset" (ptoff) takes the starting physical 
	 * track (0 or 1) and the desired interleaving into account.  
	 * If lt+ptoff >= 77, then interleaving is not performed.
	 */
	ptoff = 0;
	if (sc->sc_flags & RXF_DIRECT)
		ptoff = 77;
	if (sc->sc_flags & RXF_TRKONE)
		ptoff++;
	if (pt + ptoff < 77)
		ps = ((ps << 1) + (ps >= 13) + ph) % 26;
	*phead = ph;
	*ptrack = (pt + ptoff) % 77;
	*psector = 1 + (ps + 6*pt) % 26;
#ifdef	RXDEBUG
	printf("rxmap:(%d:%d %d %d)\n",ls+1,*phead,*ptrack,*psector);
#endif	RXDEBUG
}

/*
 * Controller start routine. Start a new transfer or continue a multisector 
 * transfer. If this is a new transfer (dp->b_active == 1) save the start 
 * address of the data buffer and the total byte count in the soft control 
 * structure. These are restored into the buffer structure when the transfer 
 * has been completed, before calling 'iodone'.
 */
rxstart(qm)
	register struct qb_ctlr *qm;
{
	register struct rxdevice *rxaddr;
	register struct rx_ctlr *rxc;
	register struct rx_softc *sc;
	struct buf *dp, *bp;
	int unit, sector, track, head;

	if (qm->qm_tab.b_active)
		return;
loop:
	if ((dp = qm->qm_tab.b_actf) == NULL)
		return;
	if ((bp = dp->b_actf) == NULL) {
		qm->qm_tab.b_actf = dp->b_forw;
		goto loop;
	}
	qm->qm_tab.b_active++;
	unit = RXUNIT(bp->b_dev);
	sc = &rx_softc[unit];
	if (sc->sc_flags & RXF_BAD) {
		rxpurge(qm);
		return;
	}
	if (dp->b_active == 1) {
		sc->sc_resid = bp->b_bcount;
		sc->sc_uaddr = bp->b_un.b_addr;
		sc->sc_bcnt = bp->b_bcount;
		sc->sc_offset += sc->sc_bcnt;
		dp->b_active++;
	}
	rxaddr = (struct rxdevice *)qm->qm_addr;
	rxc = &rx_ctlr[qm->qm_ctlr];
	bp->b_bcount = sc->sc_resid;
	if (bp->b_bcount > NBPS)
		bp->b_bcount = NBPS;
	rxc->rxc_tocnt = 0;
	if (rxaddr->rxcs == RX_RX02) {
		/*
		 * 'Volume valid'? (check if the drive has been powered down)
		 */
		rxaddr->rxcs = RX_INIT;
		WAITDONE(rxaddr);
	}
	if (bp->b_flags & B_CTRL) {			/* format */
		rxc->rxc_state = RXS_FORMAT;
		rxaddr->rxcs = RX_FORMAT | sc->sc_csbits;
		WAITTREQ(rxaddr);
		rxaddr->rxdb = 'I';
		return;
	}
	if (bp->b_flags & B_RDSTAT) {			/* read drive status */
		rxc->rxc_state = RXS_RDSTAT;
		rxaddr->rxcs = RX_RDSTAT | sc->sc_csbits;
		return;
	}

	if (bp->b_flags & B_READ) {
		rxmap(bp, &sector, &track, &head);	/* read */
#ifdef	RXDEBUG
		printf("read tr=%d, sc=%d, hd=%d\n", track, sector, head);
#endif	RXDEBUG
		rxc->rxc_state = RXS_READ;
		rxaddr->rxcs = RX_READ | sc->sc_csbits | (head << 9);
		WAITTREQ(rxaddr);
		rxaddr->rxdb = (u_short)sector;
		WAITTREQ(rxaddr);
		rxaddr->rxdb = (u_short)track;
	} else {
		rxc->rxc_state = RXS_FILL;		/* write */
		(void) rxgo(rxdinfo[unit], RX_FILL);
	}
}

rxgo(qi, cmd)
	struct qb_device *qi;
{
	struct qb_ctlr *qm = qi->qi_mi;
	register struct rxdevice *rxaddr = (struct rxdevice *)qm->qm_addr;
	struct buf *bp = qm->qm_tab.b_actf->b_actf;
	struct rx_softc *sc = &rx_softc[RXUNIT(bp->b_dev)];
	struct rx_ctlr *rxc = &rx_ctlr[qm->qm_ctlr];
	int ioaddr;

	ioaddr = qbaddr(bp);
	rxaddr->rxcs = cmd | ((ioaddr & 0x30000) >> 4) | sc->sc_csbits;
	if (rxc->rxc_state != RXS_RDERR) {
		WAITTREQ(rxaddr);
		rxaddr->rxdb = (u_short) bp->b_bcount >> 1;
	}
	WAITTREQ(rxaddr);
	rxaddr->rxdb = loword(ioaddr);
}

rxintr(ctlr)
	int ctlr;
{
	int unit, sector, track, head;
	struct qb_ctlr *qm = rxcinfo[ctlr];
	register struct rxdevice *rxaddr;
	register struct buf *bp, *dp;
	register struct rx_softc *sc;
	struct qb_device *qi;
	struct rxerr *er;
	struct rx_ctlr *rxc;

	if (!qm->qm_tab.b_active)
		return;
	dp = qm->qm_tab.b_actf;
	if (!dp->b_active)
		return;
	bp = dp->b_actf;
	unit = RXUNIT(bp->b_dev);
	sc = &rx_softc[unit];
	qi = rxdinfo[unit];
	rxaddr = (struct rxdevice *)qm->qm_addr;
	rxc = &rx_ctlr[qm->qm_ctlr];
	rxc->rxc_tocnt = 0;
	er = &rxerr[unit];
#ifdef	RXDEBUG
	printf("rxint: dev=%x, st=%d, cs=0x%x, db=0x%x\n", 
		bp->b_dev, rxc->rxc_state, rxaddr->rxcs, rxaddr->rxdb);
#endif	RXDEBUG
	if ((rxaddr->rxcs & RX_ERR) &&
	    (rxc->rxc_state != RXS_RDSTAT) && (rxc->rxc_state != RXS_RDERR))
		goto error;
	switch (rxc->rxc_state) {

	/*
	 * Incomplete commands.  Perform next step and return.  Note that 
	 * b_active is set on entrance and, therefore, also on exit.
	 */
	case RXS_READ:
		if (rxaddr->rxdb & RXES_DDMARK)
			sc->sc_flags |= RXF_DDMK;
		else
			sc->sc_flags &= ~RXF_DDMK;
		rxc->rxc_state = RXS_EMPTY;
		(void) rxgo(qi, RX_EMPTY);
		return;

	case RXS_FILL:
		rxc->rxc_state = RXS_WRITE;
		rxmap(bp, &sector, &track, &head);
#ifdef	RXDEBUG
		printf("write tr=%d, sc=%d, hd=%d\n", track, sector, head);
#endif	RXDEBUG
		if (sc->sc_flags & RXF_USEWDDS) {
			rxaddr->rxcs = RX_WDDS | sc->sc_csbits | (head << 9);
			sc->sc_flags &= ~RXF_USEWDDS;
		} else
			rxaddr->rxcs = RX_WRITE | sc->sc_csbits | (head << 9);
		WAITTREQ(rxaddr);
		rxaddr->rxdb = sector;
		WAITTREQ(rxaddr);
		rxaddr->rxdb = track;
		return;

	/*
	 * Possibly completed command.
	 */
	case RXS_RDSTAT:
		if (bp->b_flags & B_RDSTAT) {
			if ((rxaddr->rxdb&RXES_READY) == 0) {
				bp->b_flags |= B_ERROR;
				bp->b_error = ENODEV;
			} else {
				sc->sc_csbits |= rxaddr->rxdb&RXES_DBLDEN ?
					RX_DDEN : RX_SDEN;
				sc->sc_flags |= rxaddr->rxdb&RXES_DBLDEN ?
					RXF_DDEN : RXF_SDEN;
			}
			goto rdone;
		}
		if (rxaddr->rxdb&RXES_READY)
			goto rderr;
		bp->b_error = ENODEV;
		bp->b_flags |= B_ERROR;
		goto done;

	/*
	 * Command completed.
	 */
	case RXS_EMPTY:
	case RXS_WRITE:	
		goto done;

	case RXS_FORMAT:
		goto rdone;

	case RXS_RDERR:
		*er = *((struct rxerr *)0x400);		/* KLUDGE */
		bp = savebp;
		rxmap(bp, &sector, &track, &head);
		printf("rx%d: hard error, tr %d sc %d hd %d\n",
			unit, track, sector, head);
		printf("     cs=%b, db=%b; ", MASKREG(er->rxcs), 
			RXCS_BITS, MASKREG(er->rxdb), RXES_BITS);
		printf("err %x, %x, %x, %x\n", MASKREG(er->rxxt[0]),
			MASKREG(er->rxxt[1]), MASKREG(er->rxxt[2]), 
			MASKREG(er->rxxt[3]));
		goto done;

	default:
		printf("rx%d: state %d (reset)\n", unit, rxc->rxc_state);
		rxreset();
		return;
	}
error:
	/*
	 * In case of an error:
	 *  (a) Give up now if a format (ioctl) was in progress, if a density 
	 *	error was detected, or if the drive went offline
	 *  (b) Retry up to nine times if a CRC (data) error was detected, then
	 *	give up if the error persists.
	 *  (c) In all other cases, reinitialize the drive and try the operation
	 *	once more before giving up.
	 */
	if (rxc->rxc_state == RXS_FORMAT || (rxaddr->rxdb&RXES_DENERR))
		goto giveup;
	if (rxaddr->rxdb & RXES_CRCERR) {
		if (++qm->qm_tab.b_errcnt >= 10)
			goto giveup;
		goto retry;
	}
	qm->qm_tab.b_errcnt += 9;
	if (qm->qm_tab.b_errcnt >= 10)
		goto giveup;
	/* no way to get an interrupt for "init done", so just wait */
	rxaddr->rxcs = RX_INIT;
	WAITDONE(rxaddr);
	/* if someone opened the drive: give up */
	if ((rxaddr->rxdb&RXES_READY) == 0)
		goto giveup;
retry:
	qm->qm_tab.b_active = 0;
	rxstart(qm);
	return;

giveup:
	/*
	 * Hard I/O error -- ALL errors are considered fatal and will abort the
	 * transfer and purge the i/o request queue
	 */
	sc->sc_flags |= RXF_BAD;
	sc->sc_resid = 0;	/* make sure the transfer is terminated */
	rxc->rxc_state = RXS_RDSTAT;
	rxaddr->rxcs = RX_RDSTAT | sc->sc_csbits;
	return;

rderr:
	/*
	 * A hard error (other than not ready) has occurred. Read the extended 
	 * error status information. Before doing this, save the current CS 
	 * and DB register values, because the read error status operation may 
	 * modify them. Insert buffer with request at the head of the queue.
	 */
	bp->b_error = EIO;
	bp->b_flags |= B_ERROR;
	savebp = bp;
	er->rxcs = rxaddr->rxcs;
	er->rxdb = rxaddr->rxdb;
	bp = &erxbuf[unit];
	bp->b_un.b_addr = (caddr_t)0x400;	/* KLUDGE */
	bp->b_bcount = sizeof (er->rxxt);
	bp->b_flags &= ~(B_DIRTY|B_UAREA|B_PHYS|B_PAGET);
	if (dp->b_actf == NULL)
		dp->b_actl = bp;
	bp->b_forw = dp->b_actf;
	dp->b_actf = bp;
	rxc->rxc_state = RXS_RDERR;
	(void) rxgo(qi, RX_RDERR);
	return;

done:
rdone:
	qm->qm_tab.b_active = 0;
	qm->qm_tab.b_errcnt = 0;
	if ((sc->sc_resid -= NBPS) > 0) {
		bp->b_un.b_addr += NBPS;
		rxstart(qm);
		return;
	}
	bp->b_un.b_addr = sc->sc_uaddr;
	bp->b_resid = 0;
	bp->b_bcount = sc->sc_bcnt;
	dp->b_actf = bp->av_forw;
	iodone(bp);
	sc->sc_offset = 0;
	rxc->rxc_state = RXS_IDLE;
	qm->qm_tab.b_actf = dp->b_forw;
	dp->b_active = 0;
	dp->b_errcnt = 0;
	/*
	 * If this unit has more work to do, start it up right away
	 */
	if (dp->b_actf)
		rxustart(qi);

	rxstart(qm);
}

/*ARGSUSED*/

rxwatch()
{
	register struct qb_device *qi;
	register struct qb_ctlr *qm;
	register struct rx_softc *sc;
	struct rx_ctlr *rxc;
	int i, dopen = 0;

	for (i=0; i<Nrx; i++) {
		qi = rxdinfo[i];
		if (qi == 0 || qi->qi_alive == 0)
			continue;
		sc = &rx_softc[i];
		if ((sc->sc_open == 0) && (rxutab[i].b_active == 0)) {
			sc->sc_csbits = 0;
			continue;
		}
		dopen++;
		qm = qi->qi_mi;
		rxc = &rx_ctlr[qm->qm_ctlr];
		if (++rxc->rxc_tocnt >= RX_MAXTIMEOUT) {
			rxc->rxc_tocnt = 0;
			if (qm->qm_tab.b_active) {	
				printf("rx%d: timeout\n", i);/* for debugging */
				rxintr(qm->qm_ctlr);
			}
		}
	}
	if (dopen)
		timeout(rxwatch, (caddr_t)0, hz);
	else
		rxwstart = 0;
}

rxreset()
{
	register struct qb_ctlr *qm;
	register struct rxdevice *rxaddr;
	register int ctlr;

	for (ctlr = 0; ctlr < NRX; ctlr++) {
		if ((qm = rxcinfo[ctlr]) == 0 || qm->qm_alive == 0)
			continue;
		rx_ctlr[ctlr].rxc_state = RXS_IDLE;
		rxaddr = (struct rxdevice *)qm->qm_addr;
		rxaddr->rxcs = RX_INIT;
		WAITDONE(rxaddr);
		rxstart(qm);
	}
}

rxread(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	int unit = RXUNIT(dev);
	struct rx_softc *sc = &rx_softc[unit];

	if (uio->uio_offset + uio->uio_resid > RXSIZE)
		return (ENXIO);
	if (uio->uio_offset < 0 || (uio->uio_offset & SECMASK) != 0)
		return (ENXIO);
	sc->sc_offset = uio->uio_offset % DEV_BSIZE;
	return (physio(rxstrategy, &rrxbuf[unit], dev, B_READ, rxminphys, uio));
}

rxwrite(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	int unit = RXUNIT(dev);
	struct rx_softc *sc = &rx_softc[unit];

	if (uio->uio_offset + uio->uio_resid > RXSIZE)
		return (ENXIO);
	if (uio->uio_offset < 0 || (uio->uio_offset & SECMASK) != 0)
		return (ENXIO);
	sc->sc_offset = uio->uio_offset % DEV_BSIZE;
	return(physio(rxstrategy, &rrxbuf[unit], dev, B_WRITE, rxminphys, uio));
}

/*
 * Control routine. Processes four kinds of requests:
 *  (1) Set density (i.e., format the diskette) according to that specified
 *	data parameter
 *  (2) Arrange for the next sector to be written with a deleted data mark.
 *  (3) Report whether the last sector read had a deleted-data mark
 *  (4) Report the density of the diskette in the indicated drive (since the 
 *	density it automatically determined by the driver, this is the only way
 *	to let an application program know the density)
 *
 * Requests relating to deleted-data marks can be handled right here. A 
 * "set density" (format) request, however, must additionally be processed 
 * through "rxstart", just like a read or write request.
 */

rxioctl(dev, cmd, data, flag)
	dev_t dev;
	caddr_t data;
{   
	int unit = RXUNIT(dev);
	struct rx_softc *sc = &rx_softc[unit];
	int i;

	switch (cmd) {

	case RXIOC_FORMAT:
		if ((flag&FWRITE) == 0)
			return (EBADF);
		if (sc->sc_open > 1)
			return (EBUSY);
		sc->sc_csbits |= ((*(int *)data) & RX_DDEN);
		sc->sc_flags |= sc->sc_csbits&RX_DDEN ? RXF_DDEN : RXF_SDEN;
		return (rxformat(dev));

	case RXIOC_WDDS:
		sc->sc_flags |= RXF_USEWDDS;
		return (0);

	case RXIOC_RDDSMK:
		*(int *)data = sc->sc_flags & RXF_DDMK;
		return (0);

	case RXIOC_GDENS:
		*(int *)data = sc->sc_csbits & RX_DDEN;
		return (0);
	}
	return (ENXIO);
}

/*
 * Initiate a format command.
 */
rxformat(dev)
	dev_t dev;
{
	int unit = RXUNIT(dev);
	struct buf *bp;
	struct rx_softc *sc = &rx_softc[unit];
	int s, error = 0;

	bp = &rrxbuf[unit];
	bp->b_flags = B_BUSY | B_CTRL;
	sc->sc_flags |= RXF_FORMAT | RXF_LOCK;
	bp->b_dev = dev;
	bp->b_error = 0;
	bp->b_blkno = 0;
	rxstrategy(bp);
	iowait(bp);
	if (bp->b_flags & B_ERROR)
		error = bp->b_error;
	bp->b_flags &= ~B_BUSY;
	sc->sc_flags &= ~RXF_LOCK;
	return (error);
}

/*
 * A permanent hard error condition has occured, purge the buffer queue
 */
rxpurge(qm)
	register struct qb_ctlr *qm;
{
	register struct buf *bp, *dp;

	dp = qm->qm_tab.b_actf;
	while (dp->b_actf) {
		dp->b_errcnt++;
		bp = dp->b_actf;
		bp->b_error = EIO;
		bp->b_flags |= B_ERROR;
		iodone(bp);
		dp->b_actf = bp->av_forw;
	}
}

rxminphys(bp)
	struct buf *bp;
{
	bp->b_bcount = MIN(bp->b_bcount, (63*1024));
}
#endif	NRX > 0
