/*
 * QBUS: RL01/2 disk driver
 */

#include "rl.h"
#if	NRL > 0
#include "../machine/pte.h"

#include "param.h"
#include "systm.h"
#include "dk.h"
#include "dkbad.h"
#include "buf.h"
#include "conf.h"
#include "user.h"
#include "map.h"
#include "vm.h"
#include "cmap.h"
#include "uio.h"
#include "kernel.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/rlreg.h"

/* 
 * Pending Controller items and statistics
 */
struct	rl_softc {
	int	rl_ndrive;	/* Number of drives on controller */
	int	rl_wticks;	/* Monitor time for function */
	int	rl_18bit;	/* non zero if controler is 18 bit */
	int	rl_opened;	/* non zero if controler opened */
} rl_softc[NRL];

/* 
 * State of controller from last transfer. Since only one transfer can be done 
 * at a time per controller, only allocate one for each controller.
 */
struct	rl_stat {
	short	rl_cyl[4];	/* Current cylinder for each drive */
	short	rl_dt[4];	/* disk drive type (index into rlstr) */
	short	rl_dn;		/* drive number currently transferring */
	short	rl_cylnhd;	/* current cylinder and head of transfer */
	u_short	rl_bleft;	/* bytes left to transfer */
	u_short	rl_bpart;	/* bytes transferred */
} rl_stat[NRL];

/* THIS SHOULD BE READ OFF THE PACK, PER DRIVE */
/* Last cylinder not used. Saved for Bad Sector File */
struct	size {
	daddr_t	nblocks;
	int	cyloff;
};
struct	size rl01_sizes[8] = {
	10200,		0,		/* A=cyl   0 thru 254 */
	    0,		0,		/* B= Not Defined     */
	10240,		0,		/* C=cyl   0 thru 255 */
	    0,		0,		/* D= Not Defined     */
	    0,          0,		/* F= Not Defined     */
	10200,	        0,		/* G=cyl   0 thru 254 */
	    0,          0,		/* H= Not Defined     */
};
struct	size rl02_sizes[8] = {
	15884,		0,		/* A=cyl   0 thru 397 */
	 4520,		398,		/* B=cyl 398 thru 510 */
	20480,		0,		/* C=cyl   0 thru 511 */
	 4520,		398,		/* D=cyl 398 thru 510 */
	    0,          0,		/* F= Not Defined     */
	20440,	        0,		/* G=cyl   0 thru 510 */
	    0,          0,		/* H= Not Defined     */
};
/* END OF STUFF WHICH SHOULD BE READ IN PER DISK */

int	rlprobe(),	rlslave(),	rlattach(),	rlintr();
int	rlminphys();
struct	qb_ctlr		*rlcinfo[NRL];
struct	qb_device	*rldinfo[Nrl];
struct	qb_device	*rlip[NRL][4];

u_short	*rlstd[] = { (u_short *)0x3FF900, (u_short *)0x3FF910, 
		     (u_short *)0x3FF920, 0 };

/* RL01/2 driver structure */
struct	qb_driver RLdriver =
    { rlprobe, rlslave, rlattach, rlstd, "rl", rldinfo, "RL", rlcinfo };

/* RL01/2 drive structure */
struct	rlstruct {
	short	nbpt;		/* Number of 512 byte blocks/track */
	short	ntrak;		/* Number of tracks/cylinder */
	short	nbpc;		/* Number of 512 byte blocks/cylinder */
	short	ncyl;		/* Number of cylinders */
	short	btrak;		/* Number of bytes/track */
	struct	size *sizes;
} rlstr[] = {
#define RL01	0
	20,	2,	20*2,	256,	20*512,	rl01_sizes, /* rl01/DEC */
#define RL02	1
	20,	2,	20*2,	512,	20*512,	rl02_sizes /* rl02/DEC */
};

struct	buf	rrlbuf[Nrl];

#define	b_cylin b_resid		/* Last seek as CYL<<1 | HD */
#define rlwait(rladdr)		{while((rladdr->rlcs & RL_CRDY) == 0);}
#define	rlunit(bp)	(minor((bp)->b_dev) >> 3)
#define	rlblock(bp)	((bp)->b_blkno)

int	rlwstart, rlwatch();		/* Have started guardian */

rlprobe(rladdr)
	register struct rldevice *rladdr;
{
	rladdr->rlda.getstat = RL_RESET;
	rladdr->rlcs = RL_IE | RL_GETSTAT;
	DELAY(100000);
	rladdr->rlda.getstat = RL_GSTAT;
	rladdr->rlcs = RL_GETSTAT;
	clevmax = clev_biomax;
	clev_bio = MAX(clev, clev_bio);
	return (sizeof (struct rldevice));
}

rlslave(qi, rladdr)
	struct qb_device *qi;
	register struct rldevice *rladdr;
{
	short ctr = 0;

	/*
	 * DEC reports that:
	 * For some unknown reason the RL01/2 (seems to be only drive 1)
	 * does not return a valid drive status the first time that a
	 * GET STATUS request is issued for the drive, in fact it can
	 * take up to three or more GET STATUS requests to obtain the
	 * correct status.
	 * In order to overcome this, the driver has been modified to
	 * issue a GET STATUS request and validate the drive status
	 * returned.  If a valid status is not returned after eight
	 * attempts, then an error message is printed.
	 */
	do {
		rladdr->rlda.getstat = RL_RESET;
		rladdr->rlcs = (qi->qi_slave <<8) | RL_GETSTAT; /* Get status*/
		rlwait(rladdr);
	} while ((rladdr->rlmp.getstat&RLMP_STATUS) != RLMP_STATOK && ++ctr<8);
	if ((rladdr->rlcs & RL_DE) || (ctr >= 8))
		return (0);
	return (1);
}

rlattach(qi)
	register struct qb_device *qi;
{
	register struct rldevice *rladdr;
	struct rlstruct *str;

	if (rlwstart == 0) {
		timeout(rlwatch, (caddr_t)0, hz);
		rlwstart++;
	}
	rlip[qi->qi_ctlr][qi->qi_slave] = qi;
	rladdr = (struct rldevice *)qi->qi_mi->qm_addr;
	if (badaddr(&rladdr->rlbae, 2))
		rl_softc[qi->qi_ctlr].rl_18bit++;
	rl_softc[qi->qi_ctlr].rl_ndrive++;
	/* get drive type */
	/* reset controller */
	rladdr->rlda.getstat = RL_RESET;	/* SHOULD BE REPEATED? */
	rladdr->rlcs = (qi->qi_slave << 8) | RL_GETSTAT; /* Reset DE bit */
	rlwait(rladdr);
	printf("	");
	if (rl_softc[qi->qi_ctlr].rl_18bit)
		printf("18bit ");
	if ((rladdr->rlmp.getstat & RLMP_DT) == RL01 ) {
		rl_stat[qi->qi_ctlr].rl_dt[qi->qi_unit] = RL01;
		printf("RL01 ");
	} else {
		rl_stat[qi->qi_ctlr].rl_dt[qi->qi_unit] = RL02;
		printf("RL02 ");
	}
	/* determine disk posistion */
	rladdr->rlcs = (qi->qi_slave << 8) | RL_RHDR;
	rlwait(rladdr);
	/* save disk drive posistion */
	rl_stat[qi->qi_ctlr].rl_cyl[qi->qi_slave] =
	     (rladdr->rlmp.readhdr & 0177700) >> 6;
	rl_stat[qi->qi_ctlr].rl_dn = -1;
	str = &rlstr[rl_stat[qi->qi_ctlr].rl_dt[qi->qi_unit]];
	printpart(str->sizes,str->nbpc);
}
 
rlopen(dev)
	dev_t dev;
{
	register int unit = minor(dev) >> 3;
	register struct qb_device *qi;

	if (unit >= Nrl || (qi = rldinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	if (!rl_softc[qi->qi_ctlr].rl_opened++ && 
		rl_softc[qi->qi_ctlr].rl_18bit)
			bdevsw[major(dev)].d_flags |= B_18BIT;
	return (0);
}

rlstrategy(bp)
	register struct buf *bp;
{
	register struct qb_device *qi;
	register int unit;
	register struct buf *dp;
	int partition = minor(bp->b_dev) & 07, s;
	struct rlstruct *str;
	long bn, sz;

	unit = rlunit(bp);
	if (unit >= Nrl)
		goto bad;
	qi = rldinfo[unit];
	if (qi == 0 || qi->qi_alive == 0)
		goto bad;
	sz = (bp->b_bcount+511) >> 9;
	str = &rlstr[rl_stat[qi->qi_ctlr].rl_dt[qi->qi_unit]];
	if (bp->b_blkno < 0 ||
	    (bn = rlblock(bp))+sz > str->sizes[partition].nblocks)
		goto bad;
	/* bn is in 512 byte block size */
	bp->b_cylin = bn/str->nbpc + str->sizes[partition].cyloff;
	s = splx(qi->qi_mi->qm_psl);
	dp = &qi->qi_mi->qm_tab;
	disksort(dp, bp);
	if (dp->b_active == 0)
		rlstart(qi->qi_mi);
	splx(s);
	return;

bad:	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
}

/*
 * Start up a transfer on a drive.
 */
rlstart(qm)
	register struct qb_ctlr *qm;
{
	register struct buf *bp;
	register struct qb_device *qi;
	register struct rldevice *rladdr;
	register struct rl_stat *st = &rl_stat[qm->qm_ctlr];
	struct rlstruct *str;
	daddr_t bn;
	int ioaddr;
	short sn, cyl, cmd, hd, diff;

	if ((bp = qm->qm_tab.b_actf) == NULL) {
		st->rl_dn = -1;
		st->rl_cylnhd = st->rl_bpart = st->rl_bleft = 0;
		return;
	}
	/*
	 * Mark controller busy, and determine destination.
	 */
	qm->qm_tab.b_active++;
	qi = rldinfo[rlunit(bp)];		/* Controller */
	str = &rlstr[st->rl_dt[qi->qi_unit]];
	bn = rlblock(bp);			/* 512 byte Block number */
	rladdr = (struct rldevice *)qm->qm_addr;
	hd = (bn / str->nbpt) & 1;		/* Get head required */
	diff = (st->rl_cyl[qi->qi_slave] >> 1) - bp->b_cylin;
	if ( diff != 0 || (st->rl_cyl[qi->qi_slave]&1) != hd) {
		st->rl_cyl[qi->qi_slave] = (bp->b_cylin<<1)|hd;
		if (diff < 0)
			rladdr->rlda.seek = -diff << 7 | RLDA_HGH | hd << 4;
		else
			rladdr->rlda.seek = diff << 7 | RLDA_LOW | hd << 4;
		rladdr->rlcs = (qi->qi_slave << 8) | RL_SEEK;
		rlwait(rladdr);
	}
	cyl = st->rl_cyl[qi->qi_slave];
	sn = (bn % str->nbpt) << 1;		/* Sector number */
	rladdr->rlda.rw = cyl<<6 | sn;
	/* save away current transfers drive status */
	st->rl_dn = qi->qi_slave;
	st->rl_cylnhd = cyl;
	st->rl_bleft = bp->b_bcount;
	st->rl_bpart = str->btrak - (sn * NRLBPSC);
	/*
	 * RL02 must seek between cylinders and between tracks, determine 
	 * maximum data transfer at this time.
	 */
	if (st->rl_bleft < st->rl_bpart)
		st->rl_bpart = st->rl_bleft;
	rladdr->rlmp.rw = -(st->rl_bpart >> 1);
	if (bp->b_flags & B_READ)
		cmd = RL_IE | RL_READ | (qi->qi_slave << 8);
	else
		cmd = RL_IE | RL_WRITE | (qi->qi_slave << 8);
	ioaddr = qbaddr(bp);
	rladdr->rlba = loword(ioaddr);
	if (!rl_softc[qi->qi_ctlr].rl_18bit)
		rladdr->rlbae = hiword(ioaddr);
	rladdr->rlcs = cmd|(((int)ioaddr>>12)&RL_BAE);
	if (qi->qi_dk >= 0) {
		dk_busy |= 1 << qi->qi_dk;
		dk_xfer[qi->qi_dk]++;
		dk_wds[qi->qi_dk] += bp->b_bcount >> 6;
	}
}

/*
 * Handle a disk interrupt.
 */
rlintr(ctlr)
	register ctlr;
{
	register struct buf *bp;
	register struct qb_ctlr *qm = rlcinfo[ctlr];
	register struct qb_device *qi;
	register struct rldevice *rladdr = (struct rldevice *)qm->qm_addr;
	register unit;
	struct rl_softc *rl = &rl_softc[qm->qm_ctlr];
	struct rl_stat *st = &rl_stat[qm->qm_ctlr];
	int status;

	rl->rl_wticks = 0;
	bp = qm->qm_tab.b_actf;
	qi = rldinfo[rlunit(bp)];
	dk_busy &= ~(1 << qi->qi_dk);

	/*
	 * Check for and process errors on
	 * either the drive or the controller.
	 */
	if (rladdr->rlcs & RL_ERR) {
		u_short err;

		rlwait(rladdr);
		err = rladdr->rlcs;
		/* get staus and reset controller */
		rladdr->rlda.getstat = RL_GSTAT;
		rladdr->rlcs = (qi->qi_slave << 8) | RL_GETSTAT;
		rlwait(rladdr);
		status = rladdr->rlmp.getstat;
		/* reset drive */
		rladdr->rlda.getstat = RL_RESET;
		rladdr->rlcs = (qi->qi_slave <<8) | RL_GETSTAT; /* Get status*/
		rlwait(rladdr);
		if ((status & RLMP_WL) == RLMP_WL) {
			/*
			 * Give up on write protected devices immediately.
			 */
			printf("rl%d: write protected\n", rlunit(bp));
			bp->b_flags |= B_ERROR;
		} else if (++qm->qm_tab.b_errcnt > 10) {
			/*
			 * After 10 retries give up.
			 */
			harderr(bp, "rl");
			printf("cs=%b mp=%b\n", err, RLCS_BITS,
			    status, RLER_BITS);
			bp->b_flags |= B_ERROR;
		} else
			qm->qm_tab.b_active = 0;	 /* force retry */
		/* determine disk position */
		rladdr->rlcs = (qi->qi_slave << 8) | RL_RHDR;
		rlwait(rladdr);
		/* save disk drive position */
		st->rl_cyl[qi->qi_slave] =
		    (rladdr->rlmp.readhdr & 0177700) >> 6;
	}
	/*
	 * If still ``active'', then don't need any more retries.
	 */
	if (qm->qm_tab.b_active) {
		/* RL02 check if more data from previous request */
		if ((bp->b_flags & B_ERROR) == 0 &&
		     (int)(st->rl_bleft -= st->rl_bpart) > 0) {
			int ioaddr, cmd, diff, head;
			caddr_t saveaddr;

			/* seek to next head/track */
			/* increment head and/or cylinder */
			st->rl_cylnhd++;
			diff = (st->rl_cyl[qi->qi_slave] >> 1) -
				(st->rl_cylnhd >> 1);
			st->rl_cyl[qi->qi_slave] = st->rl_cylnhd;
			head = st->rl_cylnhd & 1;
			rlwait(rladdr);
			if (diff < 0)
			    rladdr->rlda.seek = -diff<<7 | RLDA_HGH | head<<4;
			else
			    rladdr->rlda.seek = diff<<7 | RLDA_LOW | head<<4;
			rladdr->rlcs = (qi->qi_slave << 8) | RL_SEEK;
			saveaddr = bp->b_un.b_addr;
			bp->b_un.b_addr += bp->b_bcount - st->rl_bleft;
			ioaddr = qbaddr(bp);
			bp->b_un.b_addr = saveaddr;
			qm->qm_tab.b_active++;
			rlwait(rladdr);
			rladdr->rlda.rw = st->rl_cylnhd << 6;
			if (st->rl_bleft < 
			    (st->rl_bpart=rlstr[st->rl_dt[qi->qi_unit]].btrak))
				st->rl_bpart = st->rl_bleft;
			rladdr->rlmp.rw = -(st->rl_bpart >> 1);
			cmd = (bp->b_flags & B_READ ? RL_READ : RL_WRITE) |
			    RL_IE | (qi->qi_slave << 8);
			cmd |= ((ioaddr >> 12) & RL_BAE);
			rladdr->rlba = ioaddr;
			if (!rl_softc[qi->qi_ctlr].rl_18bit)
				rladdr->rlbae = hiword(ioaddr);
			rladdr->rlcs = cmd;
			return;
		}
		qm->qm_tab.b_active = qm->qm_tab.b_errcnt = 0;
		bp->b_resid = st->rl_bleft;
		qm->qm_tab.b_actf = bp->av_forw;
		st->rl_bpart = st->rl_bleft = 0;
		iodone(bp);
	}
	/* reset state info */
	st->rl_dn = -1;
	st->rl_cylnhd = st->rl_bpart = st->rl_bleft = 0;
	if (qm->qm_tab.b_actf && qm->qm_tab.b_active == 0)
		rlstart(qm);
}

rlread(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	register int unit = minor(dev) >> 3;

	if (unit >= Nrl)
		return (ENXIO);
	return (physio(rlstrategy, &rrlbuf[unit], dev, B_READ, rlminphys, uio));
}

rlwrite(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	register int unit = minor(dev) >> 3;

	if (unit >= Nrl)
		return (ENXIO);
	return (physio(rlstrategy, &rrlbuf[unit], dev, B_WRITE, rlminphys, uio));
}

/*
 * Reset driver after lost interupt. 
 * Cancel software state of all pending transfers and restart all units and the
 * controller.
 */
rlreset()
{
	register struct qb_ctlr *qm;
	register struct qb_device *qi;
	register struct rldevice *rladdr;
	register struct rl_stat *st;
	register int ctlr, unit;

	for (ctlr = 0; ctlr < NRL; ctlr++) {
		if ((qm = rlcinfo[ctlr]) == 0 || qm->qm_alive == 0)
			continue;
		printf(" RL%d", ctlr);
		rladdr = (struct rldevice *)qm->qm_addr;
		st = &rl_stat[ctlr];
		qm->qm_tab.b_active = 0;
		/* reset controller */
		st->rl_dn = -1;
		st->rl_cylnhd = 0;
		st->rl_bleft = 0;
		st->rl_bpart = 0;
		rlwait(rladdr);
		for (unit = 0; unit < 4; unit++) {
			rladdr->rlcs = (unit << 8) | RL_GETSTAT;
			rlwait(rladdr);
			/* Determine disk posistion */
			rladdr->rlcs = (unit << 8) | RL_RHDR;
			rlwait(rladdr);
			/* save disk drive posistion */
			st->rl_cyl[unit] =
				(rladdr->rlmp.readhdr & 0177700) >> 6;
		}
		rlstart(qm);
	}
}

/*
 * Wake up every second and if an interrupt is pending but nothing has happened
 * increment a counter. If nothing happens for 20 seconds, reset the controler
 * and begin anew.
 */
rlwatch()
{
	register struct qb_ctlr *qm;
	register ctlr, unit;
	register struct rl_softc *rl;

	timeout(rlwatch, (caddr_t)0, hz);
	for (ctlr = 0; ctlr < NRL; ctlr++) {
		qm = rlcinfo[ctlr];
		if (qm == 0 || qm->qm_alive == 0 || qm->qm_tab.b_active == 0)
			continue;
		rl = &rl_softc[ctlr];
		rl->rl_wticks++;
		if (rl->rl_wticks >= 20) {
			rl->rl_wticks = 0;
			printf("RL%d: lost interrupt\n", ctlr);
			rlreset();
		}
	}
}

/*ARGSUSED*/
rldump(dev)
	dev_t dev;
{
	return(ENODEV);
}

rlsize(dev)
	dev_t dev;
{
	register int unit = minor(dev) >> 3;
	register struct qb_device *qi;

	if (unit >= Nrl || (qi = rldinfo[unit]) == 0 || qi->qi_alive == 0)
		return (-1);
	return (rlstr[rl_stat[qi->qi_ctlr].rl_dt[qi->qi_unit]].sizes[minor(dev)
		 & 07].nblocks);
}

rlminphys(bp)
	struct buf *bp;
{
	bp->b_bcount = MIN(bp->b_bcount, (63*1024));
}
#endif	NRL > 0
