/*
 * QBUS: RK611/RK0[67] disk driver
 */
#include "rk.h"
#if	NRK > 0
#ifdef RKDEBUG
int	rkdebug = RKDEBUG;
#endif
#ifdef RKBDEBUG
int	rkbdebug = RKBDEBUG;
#endif
/*
 * This driver mimics up.c; see it for an explanation of common code.
 *
 * TODO:
 *	Learn why we lose an interrupt sometime when spinning drives down
 */
#include "../machine/pte.h"

#include "param.h"
#include "systm.h"
#include "buf.h"
#include "conf.h"
#include "user.h"
#include "map.h"
#include "vm.h"
#include "dk.h"
#include "cmap.h"
#include "dkbad.h"
#include "uio.h"
#include "kernel.h"
#include "syslog.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/rkreg.h"

struct	rk_softc {
	int	sc_softas;
	int	sc_ndrive;
	int	sc_wticks;
	int	sc_recal;
	int	sc_18bit;
} rk_softc[NRK];

/* THIS SHOULD BE READ OFF THE PACK, PER DRIVE */
struct size {
	daddr_t	nblocks;
	int	cyloff;
} rk7_sizes[8] ={
	15884,	0,		/* A=cyl 0 thru 240 */
	10032,	241,		/* B=cyl 241 thru 392 */
	53790,	0,		/* C=cyl 0 thru 814 */
	0,	0,
	0,	0,
	0,	0,
	27786,	393,		/* G=cyl 393 thru 813 */
	0,	0,
}, rk6_sizes[8] ={
	15884,	0,		/* A=cyl 0 thru 240 */
	11154,	241,		/* B=cyl 241 thru 409 */
	27126,	0,		/* C=cyl 0 thru 410 */
	0,	0,
	0,	0,
	0,	0,
	0,	0,
	0,	0,
};
/* END OF STUFF WHICH SHOULD BE READ IN PER DISK */

#define	MAX_rk_RK	4

int	rkprobe(), rkslave(), rkattach(), rkdgo(), rkintr();
int	rkstrat(), rkminphys();
struct	qb_ctlr *rkcinfo[NRK];
struct	qb_device *rkdinfo[Nrk];
struct	qb_device *rkip[NRK][MAX_rk_RK];

u_short	*rkstd[] = { (u_short *)0x3FFF20, (u_short *)0x3FFDC0, 0 };

struct	qb_driver RKdriver =
 { rkprobe, rkslave, rkattach, rkstd, "rk", rkdinfo, "RK", rkcinfo };

struct	buf rkutab[Nrk];
short	rkcyl[Nrk];
struct	dkbad rkbad[Nrk];
struct	buf brkbuf[Nrk];

struct	rkst {
	short	nspt;
	short	ntpc;
	short	nspc;
	short	ncpd;
	struct	size *sizes;
} rkst[] = {
	NRKSECT, NRKTRK, NRKSECT*NRKTRK,	NRK7CYL,	rk7_sizes,
	NRKSECT, NRKTRK, NRKSECT*NRKTRK,	NRK6CYL,	rk6_sizes,
};
short	rktypes[] = { RK_CDT, 0 };

u_char 	rk_offset[16] = { 
	RKAS_P400,RKAS_M400,RKAS_P400,RKAS_M400,RKAS_P800,RKAS_M800,RKAS_P800,
    	RKAS_M800,RKAS_P1200,RKAS_M1200,RKAS_P1200,RKAS_M1200,0,0,0,0
};

struct	buf rrkbuf[Nrk];

#define	b_cylin	b_resid
#define	rkunit(bp)	(minor((bp)->b_dev) >> 3)
#define	rkblock(bp)	((bp)->b_blkno)

#define SC02	300		/* EMMULEX SC02 delay (worst with TS/QIC2) */

int	rkwstart, rkwatch();

rkprobe(rkaddr)
register struct rkdevice *rkaddr;
{
	/* 
	 * The primary base of the RK is the same as the optional base
	 * of some HP's.  The rkcyl register is read write, the corrisponding
	 * register on an HP (hpla) is read only.
	 */
	rkaddr->rkcyl = 1;
	DELAY(100);
	if (rkaddr->rkcyl != 1)
		return (0);
	rkaddr->rkcs1 = RK_CDT|RK_IE|RK_CRDY;
	DELAY(100);
	rkaddr->rkcs1 = RK_CDT;
	rkaddr->rkcs2 = RKCS2_SCLR;
	rkwait(rkaddr);
	clevmax = clev_biomax;
	clev_bio = MAX(clev, clev_bio);
	return (sizeof (struct rkdevice));
}

rkslave(qi, rkaddr)
struct qb_device *qi;
register struct rkdevice *rkaddr;
{
	qi->qi_type = 0;			/* assume RK07 */
	rkaddr->rkcs1 = RK_CCLR;
	DELAY(SC02);
	rkaddr->rkcs2 = qi->qi_slave;
	rkaddr->rkcs1 = RK_CDT|RK_DCLR|RK_GO;
	rkwait(rkaddr);
	if (rkaddr->rkcs2&RKCS2_NED || (rkaddr->rkds&RKDS_SVAL) == 0) {
		rkaddr->rkcs1 = RK_CCLR;
		DELAY(SC02);
		return (0);
	}
	if (rkaddr->rkcs1&RK_CERR && rkaddr->rker&RKER_DTYE) {
		qi->qi_type = 1;		/* its an RK06 */
		rkaddr->rkcs1 = RK_CCLR;
		DELAY(SC02);
	}
	return (1);
}

rkattach(qi)
register struct qb_device *qi;
{
	struct rkst *st = &rkst[qi->qi_type];

	if (rkwstart == 0) {
		timeout(rkwatch, (caddr_t)0, hz);
		rkwstart++;
	}
	rkip[qi->qi_ctlr][qi->qi_slave] = qi;
	rk_softc[qi->qi_ctlr].sc_ndrive++;
	rkcyl[qi->qi_unit] = -1;
	qi->qi_flags = 0;
	printf("	");
	if (badaddr(&((struct rkdevice *)qi->qi_mi->qm_addr)->rkbae, 2)) {
		rk_softc[qi->qi_ctlr].sc_18bit++;
		printf("18bit ");
	}
	printf("	%mM RK0%d (%d x %d x %d) ", 
		qi->qi_type ? 6 : 7, 
		st->nspt * st->ntpc * st->ncpd * 512,
		st->nspt, st->ntpc, st->ncpd);
	printpart(st->sizes,st->nspt*st->ntpc);

	/* Initialize iostat values */
	if (qi->qi_dk >= 0)
		dk_bps[qi->qi_dk] = (3600/60)*512*st->nspt;
}
 
rkopen(dev)
dev_t dev;
{
	register int unit = minor(dev) >> 3;
	register struct qb_device *qi;

	if (unit >= Nrk || (qi = rkdinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	if (rk_softc[qi->qi_ctlr].sc_18bit)
			bdevsw[major(dev)].d_flags |= B_18BIT;
	return (0);
}

rkstrategy(bp)
register struct buf *bp;
{
	register struct qb_device *qi;
	register struct rkst *st;
	register int unit;
	int partition = minor(bp->b_dev) & 07;
	long bn, sz;

	sz = (bp->b_bcount+511) >> DEV_BSHIFT;
	unit = rkunit(bp);
	if (unit >= Nrk)
		goto bad;
	qi = rkdinfo[unit];
	if (qi == 0 || qi->qi_alive == 0)
		goto bad;
	st = &rkst[qi->qi_type];
	if (bp->b_blkno < 0 ||
	    (bn = rkblock(bp))+sz > st->sizes[partition].nblocks)
		goto bad;
	badstrat(qi, bp, &rkbad[qi->qi_unit], rkstrat, 
		st->nspt, st->ntpc, st->ncpd,
		st->sizes[partition].cyloff);
	return;
bad:
	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
}

rkstrat(qi, bp)
register struct qb_device *qi;
register struct buf *bp;
{
	register struct buf *dp;
	int partition = minor(bp->b_dev) & 07;
	register struct rkst *st = &rkst[qi->qi_type];
	int s;

	bp->b_cylin = rkblock(bp)/st->nspc + st->sizes[partition].cyloff;
	s = splx(qi->qi_mi->qm_psl);
	dp = &rkutab[qi->qi_unit];
	disksort(dp, bp);
	if (dp->b_active == 0) {
		rkustart(qi);
		bp = &qi->qi_mi->qm_tab;
		if (bp->b_actf && bp->b_active == 0)
			rkstart(qi->qi_mi);
	}
	splx(s);
}

rkustart(qi)
register struct qb_device *qi;
{
	register struct buf *bp, *dp;
	register struct qb_ctlr *qm;
	register struct rkdevice *rkaddr;
	int didie = 0;

	if (qi == 0)
		goto out;
	dk_busy &= ~(1<<qi->qi_dk);
	dp = &rkutab[qi->qi_unit];
	qm = qi->qi_mi;
	if (qm->qm_tab.b_active) {
		rk_softc[qm->qm_ctlr].sc_softas |= 1<<qi->qi_slave;
		goto out;
	}
	if ((bp = dp->b_actf) == NULL)
		goto out;
	rkaddr = (struct rkdevice *)qm->qm_addr;
	rkaddr->rkcs1 = RK_CERR;
	DELAY(SC02);
	rkaddr->rkcs2 = qi->qi_slave;
	rkaddr->rkcs1 = rktypes[qi->qi_type]|RK_DCLR|RK_GO;
	rkwait(rkaddr);
	if ((rkaddr->rkds & RKDS_VV) == 0 || qi->qi_flags == 0) {
		/* SHOULD WARN SYSTEM THAT THIS HAPPENED */
		struct rkst *st = &rkst[qi->qi_type];
		struct buf *bbp = &brkbuf[qi->qi_unit];

		rkaddr->rkcs1 = rktypes[qi->qi_type]|RK_PACK|RK_GO;
		qi->qi_flags = 1;
#ifdef	orig144
		bbp->b_flags = B_READ|B_BUSY;
		bbp->b_dev = bp->b_dev;
		bbp->b_bcount = 512;
		bbp->b_un.b_addr = (caddr_t)&rkbad[qi->qi_unit];
		bbp->b_blkno = st->ncpd*st->nspc - st->nspt;
		bbp->b_cylin = st->ncpd - 1;
		dp->b_actf = bbp;
		bbp->av_forw = bp;
		bp = bbp;
		rkwait(rkaddr);
#else	orig144
		rkbad[qi->qi_unit].bt_csn = 0;
#endif	orig144
	}
	if (dp->b_active)
		goto done;
	dp->b_active = 1;
	if ((rkaddr->rkds & RKDS_DREADY) != RKDS_DREADY)
		goto done;
	if (rk_softc[qm->qm_ctlr].sc_ndrive == 1)
		goto done;
	if (bp->b_cylin == rkcyl[qi->qi_unit])
		goto done;
	rkaddr->rkcyl = rkcyl[qi->qi_unit] = bp->b_cylin;
	rkaddr->rkcs1 = rktypes[qi->qi_type]|RK_IE|RK_SEEK|RK_GO;
	didie = 1;
	if (qi->qi_dk >= 0) {
		dk_busy |= 1<<qi->qi_dk;
		dk_seek[qi->qi_dk]++;
	}
	goto out;
done:
	if (dp->b_active != 2) {
		dp->b_forw = NULL;
		if (qm->qm_tab.b_actf == NULL)
			qm->qm_tab.b_actf = dp;
		else
			qm->qm_tab.b_actl->b_forw = dp;
		qm->qm_tab.b_actl = dp;
		dp->b_active = 2;
	}
out:
	return (didie);
}

rkstart(qm)
register struct qb_ctlr *qm;
{
	register struct buf *bp, *dp;
	register struct qb_device *qi;
	register struct rkdevice *rkaddr;
	struct rkst *st;
	daddr_t bn;
	int sn, tn, cmd;
	int ioaddr;

loop:
	if ((dp = qm->qm_tab.b_actf) == NULL)
		return (0);
	if ((bp = dp->b_actf) == NULL) {
		qm->qm_tab.b_actf = dp->b_forw;
		goto loop;
	}
	qm->qm_tab.b_active++;
	qi = rkdinfo[rkunit(bp)];
	bn = rkblock(bp);
	st = &rkst[qi->qi_type];
	sn = bn%st->nspc;
	tn = sn/st->nspt;
	sn %= st->nspt;
	rkaddr = (struct rkdevice *)qi->qi_mi->qm_addr;
retry:
	rkaddr->rkcs1 = RK_CCLR;
	DELAY(SC02);
	rkaddr->rkcs2 = qi->qi_slave;
	rkaddr->rkcs1 = rktypes[qi->qi_type]|RK_DCLR|RK_GO;
	rkwait(rkaddr);
	if ((rkaddr->rkds&RKDS_SVAL) == 0)
		goto nosval;
	if (rkaddr->rkds&RKDS_PIP)
		goto retry;
	if ((rkaddr->rkds&RKDS_DREADY) != RKDS_DREADY) {
		printf("rk%d: not ready", rkunit(bp));
		if ((rkaddr->rkds&RKDS_DREADY) != RKDS_DREADY) {
			printf("\n");
			rkaddr->rkcs1 = RK_CCLR;
			DELAY(SC02);
			rkaddr->rkcs2 = qi->qi_slave;
			rkaddr->rkcs1 = rktypes[qi->qi_type]|RK_DCLR|RK_GO;
			rkwait(rkaddr);
			qm->qm_tab.b_active = 0;
			qm->qm_tab.b_errcnt = 0;
			dp->b_actf = bp->av_forw;
			dp->b_active = 0;
			bp->b_flags |= B_ERROR;
			iodone(bp);
			goto loop;
		}
		printf(" (came back!)\n");
	}
nosval:
	rkaddr->rkcyl = bp->b_cylin;
	rkcyl[qi->qi_unit] = bp->b_cylin;
	rkaddr->rkda = (tn << 8) + sn;
	if (bp->b_flags & B_READ)
		cmd = rktypes[qi->qi_type]|RK_IE|RK_READ|RK_GO;
	else
		cmd = rktypes[qi->qi_type]|RK_IE|RK_WRITE|RK_GO;
	ioaddr = qbaddr(bp);
	rkaddr->rkba = loword(ioaddr);
	if (!rk_softc[qi->qi_ctlr].sc_18bit)
		rkaddr->rkbae = RKBAE_22 | hiword(ioaddr);
	rkaddr->rkwc = -bp->b_bcount / sizeof (short);
	qm->qm_tab.b_active = 2;
	rkaddr->rkcs1 = cmd | ((ioaddr>>8) & RK_BA);
	if (qi->qi_dk >= 0) {
		dk_busy |= 1 << qi->qi_dk;
		dk_xfer[qi->qi_dk]++;
		dk_wds[qi->qi_dk] += bp->b_bcount >> 6;
	}
	return (1);
}

rkintr(ctlr)
int ctlr;
{
	register struct qb_ctlr *qm = rkcinfo[ctlr];
	register struct qb_device *qi;
	register struct rkdevice *rkaddr = (struct rkdevice *)qm->qm_addr;
	register struct buf *bp, *dp;
	struct rk_softc *sc = &rk_softc[qm->qm_ctlr];
	int as = (rkaddr->rkatt >> 8) | sc->sc_softas;
	int unit, needie = 1;

	sc->sc_wticks = 0;
	sc->sc_softas = 0;
	if (qm->qm_tab.b_active == 2 || sc->sc_recal) {
		qm->qm_tab.b_active = 1;
		dp = qm->qm_tab.b_actf;
		bp = dp->b_actf;
		qi = rkdinfo[rkunit(bp)];
		dk_busy &= ~(1 << qi->qi_dk);
		rkaddr->rkcs2 = qi->qi_slave;
#ifdef	orig144
		if (bp->b_flags&B_BAD)
			if (rkecc(qi, CONT))
				return;
#endif	orig144
		if (rkaddr->rkcs1 & RK_CERR) {
			int recal;
			u_short ds = rkaddr->rkds;
			u_short cs2 = rkaddr->rkcs2;
			u_short er = rkaddr->rker;
#ifdef RKDEBUG
			if (rkdebug) {
				printf("cs2=%b ds=%b er=%b\n", cs2, RKCS2_BITS,
					 ds, RKDS_BITS, er, RKER_BITS);
			}
#endif
			if (er & RKER_WLE) {
				printf("rk%d: write locked\n", rkunit(bp));
				bp->b_flags |= B_ERROR;
			} else if (++qm->qm_tab.b_errcnt > 28 ||
			    ds&RKDS_HARD || er&RKER_HARD || cs2&RKCS2_HARD) {
hard:
				harderr(bp, "rk");
				printf("cs2=%b ds=%b er=%b\n",
				    cs2, RKCS2_BITS, ds, 
				    RKDS_BITS, er, RKER_BITS);
				bp->b_flags |= B_ERROR;
				sc->sc_recal = 0;
			} else if (er & RKER_BSE) {
#ifdef	orig144
				if (rkecc(qi, BSE))
					return;
				else
#endif	orig144
					goto hard;
			} else {
				if ((er & (RKER_DCK|RKER_ECH)) == RKER_DCK) {
					if (rkecc(qi, ECC))
						return;
				} else
					qm->qm_tab.b_active = 0;
			}
			if (cs2&RKCS2_MDS) {
				rkaddr->rkcs2 = RKCS2_SCLR;
				DELAY(SC02);
				goto retry;
			}
			recal = 0;
			if (ds&RKDS_DROT || er&(RKER_OPI|RKER_SKI|RKER_UNS) ||
			    (qm->qm_tab.b_errcnt&07) == 4)
				recal = 1;
			rkaddr->rkcs1 = RK_CCLR;
			DELAY(SC02);
			rkaddr->rkcs2 = qi->qi_slave;
			rkaddr->rkcs1 = rktypes[qi->qi_type]|RK_DCLR|RK_GO;
			rkwait(rkaddr);
			if (recal && qm->qm_tab.b_active == 0) {
				rkaddr->rkcs1 = rktypes[qi->qi_type]|RK_IE|RK_RECAL|RK_GO;
				rkcyl[qi->qi_unit] = -1;
				sc->sc_recal = 0;
				goto nextrecal;
			}
		}
retry:
		switch (sc->sc_recal) {

		case 1:
			rkaddr->rkcyl = bp->b_cylin;
			rkcyl[qi->qi_unit] = bp->b_cylin;
			rkaddr->rkcs1 = rktypes[qi->qi_type]|RK_IE|RK_SEEK|RK_GO;
			goto nextrecal;
		case 2:
			if (qm->qm_tab.b_errcnt < 16 ||
			    (bp->b_flags&B_READ) == 0)
				goto donerecal;
			rkaddr->rkatt = rk_offset[qm->qm_tab.b_errcnt & 017];
			rkaddr->rkcs1 = rktypes[qi->qi_type]|RK_IE|RK_OFFSET|RK_GO;
			/* fall into ... */
		nextrecal:
			sc->sc_recal++;
			rkwait(rkaddr);
			qm->qm_tab.b_active = 1;
			return;
		donerecal:
		case 3:
			sc->sc_recal = 0;
			qm->qm_tab.b_active = 0;
			break;
		}
		if (qm->qm_tab.b_active) {
			qm->qm_tab.b_active = 0;
			qm->qm_tab.b_errcnt = 0;
			qm->qm_tab.b_actf = dp->b_forw;
			dp->b_active = 0;
			dp->b_errcnt = 0;
			dp->b_actf = bp->av_forw;
			bp->b_resid = -rkaddr->rkwc * sizeof(short);
			iodone(bp);
			if (dp->b_actf)
				if (rkustart(qi)) {
					needie = 0;
					DELAY(SC02);
				}
		}
		as &= ~(1<<qi->qi_slave);
	}
	for (unit = 0; as; as >>= 1, unit++)
		if (as & 1) {
			qi = rkip[ctlr][unit];
			if (qi) {
				if (rkustart(qi)) {
					needie = 0;
					DELAY(SC02);
				}
			} else {
				rkaddr->rkcs1 = RK_CCLR;
				DELAY(SC02);
				rkaddr->rkcs2 = unit;
				rkaddr->rkcs1 = RK_CDT|RK_DCLR|RK_GO;
				rkwait(rkaddr);
				if (rkaddr->rkcs1&RK_CERR && 
					rkaddr->rker&RKER_DTYE)
						rkaddr->rkcs1 = RK_DCLR|RK_GO;
				rkwait(rkaddr);
				rkaddr->rkcs1 = RK_CCLR;
				DELAY(SC02);
				rkaddr->rkcs2 = unit;
			}
		}
		
	if (qm->qm_tab.b_actf && qm->qm_tab.b_active == 0)
		if (rkstart(qm)) {
			needie = 0;
			DELAY(SC02);
		}
	if (needie)
		rkaddr->rkcs1 = RK_IE;
}

rkwait(addr)
register struct rkdevice *addr;
{
	DELAY(SC02);
	while ((addr->rkcs1 & RK_CRDY) == 0)
		DELAY(30);
}

rkread(dev, uio)
dev_t dev;
struct uio *uio;
{
	register int unit = minor(dev) >> 3;

	if (unit >= Nrk)
		return (ENXIO);
	return (physio(rkstrategy, &rrkbuf[unit], dev, B_READ, rkminphys, uio));
}

rkwrite(dev, uio)
dev_t dev;
struct uio *uio;
{
	register int unit = minor(dev) >> 3;

	if (unit >= Nrk)
		return (ENXIO);
	return (physio(rkstrategy, &rrkbuf[unit], dev, B_WRITE, rkminphys, uio));
}

rkecc(qi, flag)
register struct qb_device *qi;
{
	register struct buf *bp = rkutab[qi->qi_unit].b_actf;
	register struct qb_ctlr *qm = qi->qi_mi;
	register struct rkdevice *rkaddr = (struct rkdevice *)qm->qm_addr;
	register struct rkst *st;
	int npf, cmd, ioaddr;
	int bn, cn, tn, sn;
	int bit, mask, lval;

	/* define number of sectors transfered */
	if (flag == CONT)
		npf = bp->b_error;
	else
		npf = ((rkaddr->rkwc*sizeof(short))+bp->b_bcount) >> DEV_BSHIFT;
	bn = rkblock(bp);
	st = &rkst[qi->qi_type];
	cn = bp->b_cylin;
	sn = bn%st->nspc + npf;
	tn = sn/st->nspt;
	sn %= st->nspt;
	cn += tn/st->ntpc;
	tn %= st->ntpc;
	switch (flag) {
	    case ECC:
		log(LOG_WARNING, "rk%d%c: soft ecc sn%d\n", rkunit(bp),
		    'a'+(minor(bp->b_dev)&07), bp->b_blkno + npf + 1);
		ioaddr = rkaddr->rkec1 - 1;	/* -1 makes 0 origin */
		mask = rkaddr->rkec2 << (ioaddr&0xF);
		mask = loword(mask)<<16 | hiword(mask);
		ioaddr = qbaddr(bp) + ((npf-1)<<DEV_BSHIFT) + ((ioaddr&~0xF)>>3);
		putmeml(ioaddr, getmeml(ioaddr)^mask);
		break;

#ifdef	orig144
	    case BSE:
#ifdef RKBDEBUG
		if (rkbdebug)
	printf("rkecc, BSE: bn %d cn %d tn %d sn %d\n", bn, cn, tn, sn);
#endif
		if ((bn = isbad(&rkbad[qi->qi_unit], cn, tn, sn)) < 0)
			return(0);
		bp->b_flags |= B_BAD;
		bp->b_error = npf + 1;
		bn = st->ncpd*st->nspc - st->nspt - 1 - bn;
		cn = bn/st->nspc;
		sn = bn%st->nspc;
		tn = sn/st->nspt;
		sn %= st->nspt;
#ifdef RKBDEBUG
		if (rkbdebug)
	printf("revector to cn %d tn %d sn %d\n", cn, tn, sn);
#endif
		rkaddr->rkwc = -(512 / sizeof (short));
		break;

	    case CONT:
#ifdef RKBDEBUG
		if (rkbdebug)
	printf("rkecc, CONT: bn %d cn %d tn %d sn %d\n", bn,cn,tn,sn);
#endif
		bp->b_flags &= ~B_BAD;
		rkaddr->rkwc = -((bp->b_bcount - (int)ptob(npf)) / sizeof (short));
		break;
#endif	orig144
	}
	if (rkaddr->rkwc == 0) {
 		qm->qm_tab.b_active = 0;
		return (0);
	}
	rkaddr->rkcs1 = RK_CCLR;
	DELAY(SC02);
	rkaddr->rkcs2 = qi->qi_slave;
	rkaddr->rkcs1 = rktypes[qi->qi_type]|RK_DCLR|RK_GO;
	rkwait(rkaddr);
	rkaddr->rkcyl = cn;
	rkaddr->rkda = (tn << 8) | sn;
	ioaddr = qbaddr(bp) + (npf << DEV_BSHIFT);
	rkaddr->rkba = loword(ioaddr);
	if (!rk_softc[qi->qi_ctlr].sc_18bit)
		rkaddr->rkbae = RKBAE_22 | hiword(ioaddr);
	cmd = (bp->b_flags&B_READ ? RK_READ : RK_WRITE)|RK_IE|RK_GO;
	cmd |= (hiword(ioaddr) >> 8) & 0x300;
	cmd |= rktypes[qi->qi_type];
	rkaddr->rkcs1 = cmd;
	qm->qm_tab.b_active = 2;	/* continuing */
	qm->qm_tab.b_errcnt = 0;	/* error has been corrected */
	return (1);
}

rkreset()
{
	register struct qb_ctlr *qm;
	register struct qb_device *qi;
	register ctlr, unit;

	for (ctlr = 0; ctlr < NRK; ctlr++) {
		if ((qm = rkcinfo[ctlr]) == 0 || qm->qm_alive == 0)
			continue;
		printf(" RK%d", ctlr);
		qm->qm_tab.b_active = 0;
		qm->qm_tab.b_actf = qm->qm_tab.b_actl = 0;
		rk_softc[qm->qm_ctlr].sc_recal = 0;
		rk_softc[qm->qm_ctlr].sc_wticks = 0;
		for (unit = 0; unit < Nrk; unit++) {
			if ((qi = rkdinfo[unit]) == 0)
				continue;
			if (qi->qi_alive == 0 || qi->qi_mi != qm)
				continue;
			rkutab[unit].b_active = 0;
			rkustart(qi);
		}
		rkstart(qm);
	}
}

rkwatch()
{
	register struct qb_ctlr *qm;
	register ctlr, unit;
	register struct rk_softc *sc;

	timeout(rkwatch, (caddr_t)0, hz);
	for (ctlr = 0; ctlr < NRK; ctlr++) {
		qm = rkcinfo[ctlr];
		if (qm == 0 || qm->qm_alive == 0)
			continue;
		sc = &rk_softc[ctlr];
		if (qm->qm_tab.b_active == 0) {
			for (unit = 0; unit < Nrk; unit++)
				if (rkutab[unit].b_active &&
				    rkdinfo[unit]->qi_mi == qm)
					goto active;
			sc->sc_wticks = 0;
			continue;
		}
active:
		sc->sc_wticks++;
		if (sc->sc_wticks >= 20) {
			sc->sc_wticks = 0;
			printf("RK%d: lost interrupt\n", ctlr);
			rkreset(ctlr);
		}
	}
}

#define	DBSIZE	20

rkdump(dev)
	dev_t dev;
{
	return(ENODEV);
}
 
rksize(dev)
	dev_t dev;
{
	int unit = minor(dev) >> 3;
	struct qb_device *qi;
	struct rkst *st;

	if (unit >= Nrk || (qi = rkdinfo[unit]) == 0 || qi->qi_alive == 0)
		return (-1);
	st = &rkst[qi->qi_type];
	return (st->sizes[minor(dev) & 07].nblocks);
}

rkminphys(bp)
	struct buf *bp;
{
	bp->b_bcount = MIN(bp->b_bcount, (63*1024));
}
#endif	NRK > 0
