
#define	byte	unsigned char		/* 8 bit quantity */
#define	word	unsigned short		/* 16 bit quantity */
#define	dword	unsigned int		/* 32 bit quantity */

/* UNIX minor device number
 *
 *	The minor device number is broken down as follows:
 *
 *	   7     6     5     4     3     2     1     0
 *	+-----+-----+-----+-----+-----+-----+-----+-----+
 *      |                 | Unit no.  |    Partition    |
 *	+-----+-----+-----+-----+-----+-----+-----+-----+
 *
 *	Unit number	Specifies disk and volume (0-3)
 *	Partition	Specifies UNIX logical disk partition (0-7)
 *
 *	Unit numbers are assigned by the controller as follows:
 *
 *		Number		Unit
 *
 *		  0		SMD drive 0, main volume (fixed)
 *		  1		SMD drive 0, second volume (removable)
 *		  2		SMD drive 1, main volume (fixed)
 *		  3		SMD drive 1, second volume (removable)
 *
 *	Note:  Unit numbers issued to the controller range from 1 to 4
 *	       respectively.  This allows commands not associated with
 *	       a specific drive and/or volume to use a unit number of 0.
 */


/* Hardware Ports
 *
 *	The Rimfire 3200 is seen as a 512 byte area in the 16 bit
 *	address space.  There are four registers in this space,
 *	each on an 8 byte boundary.  The registers are defined below.
 */
#define	SWAB(x)	((x) & 0xffff)		/* no need to swap bytes */

/* Hardware ports */
typedef struct {
#ifdef BOGUS
	word	resv;		/* msb address */
#endif BOGUS
	word	addrbuf;	/* Address Buffer Port (see below) */
	word	space1[3];
	word	attention;	/* Channel Attention Port */
	word	space2[3];
	word	status;		/* Board Status Port */
	word	space3[3];
	word	reset;		/* Controller Reset Port */
} RF32REG;

/* Address Buffer Port
 * 	Requires three writes in the order provided below:
 *
 *	15				0
 *	+---------------+---------------+
 *	|    Control    | AM bits for PB|
 *	+---------------+---------------+
 *	|     PB address:  16 MSB       |
 *	+-------------------------------+
 *	|     PB address:  16 LSB       |
 *	+-------------------------------+
 */

/* Control Field bit masks */
#define	CTRLSET	0x80		/* apply these controls this command */
#define	CTRLWID	0x04		/* width of data transfer, 0=16, 1=32 bit */
#define	CTRLWSW	0x02		/* word swap control, 0=no swap, 1=swap */
#define	CTRLBSW	0x01		/* byte swap control, 0=no swap, 1=swap */

/* Status Port bit masks */
#define	STATRST	0xfe		/* status of a board reset */
#define	STATBSY	0x01		/* status of the Address Buffer Port */
#define	STATRDY	0x02		/* board is ready */
#define	SRAMBAD	0x04		/* static RAM had an error */
#define	DRAMBAD	0x08		/* dynamic RAM had an error */
#define	RESETDONE STATRDY	/* reset has completed */
#define	STATCTYPE 0xFF00	/* Mask to grab controller type. */
#define	STATRF3200 0x0000	/* A Rimfire 3200 Controller */
#define	STATRF3400 0x0100	/* A Rimfire 3400 Controller */

/* Standard Parameter Block
 *
 *	+-----------------------------------------------+
 *	|                  Identifier                   |
 *	+-----------+-----------+-----------+-----------+
 *	| Addr Mod  |   Unit    |  Control  |  Command  |
 *	+-----------+-----------+-----------+-----------+
 *	|                 Disk Address                  |
 *	+-----------------------------------------------+
 *	|              VME Memory Address               |
 *	+-----------------------+-----------------------+
 *	|       Reserved        |      Block Count      |
 *	+-----------------------+-----------------------+
 */
typedef struct {
	dword	id;		/* unique command identifier */
	byte	addrmod;	/* address modifier used to access VME memory */
	byte	unit;		/* unit for the command to apply */
	byte	control;	/* I/O control group */
	byte	command;	/* command to execute */
	dword	diskaddr;	/* absolute starting sector number */
	dword	vmeaddr;	/* VME address to read from or write to */
	word	reserved;	/* always zero */
	word	blkcount;	/* number of logical blocks to process */
} PARMBLK;

/* Command control bit masks */
#define	RF_IOCG		0x0f	/* I/O control group */

/* Status Block
 *
 *	+-----------------------+-----------+-----------+
 *	|                  Identifier                   |
 *	+-----------------------+-----------+-----------+
 *	|   Extra Information   |   Error   |  Status   |
 *	+-----------------------+-----------+-----------+
 *	|               Extra Information               |
 *	+-----------------------------------------------+
 */
typedef struct {
	dword	id;		/* command identifier generating status */
	word	errextra;	/* extra error information */
	byte	errcode;	/* error, retry, or status which occured */
	byte	status;		/* type of status block and results */
	dword	errdata;	/* error code specific data */
} STATBLK;

/* Status Block Status Field bit masks */
#define	ST_CC	0x80		/* Command complete, last status block */
#define	ST_ERR	0x40		/* Error, check error code */
#define	ST_RTY	0x20		/* Retry required */
#define	ST_COR	0x10		/* Correction required */
#define ST_BITS	"\20\10CC\7ERR\6RETRY\5CORR"

/* Extended Parameter Block
 *
 *	+-----------------------------------------------+
 *	|                                               |
 *	+--                                           --+
 *	|                                               |
 *	+--         Standard Parameter Block          --+
 *	|                                               |
 *	+--                                           --+
 *	|                                               |
 *	+--                                           --+
 *	|                                               |
 *	+-----------------------+-----------------------+
 *	|    Error Interrupt    |   No Error Interrupt  |
 *	+-----------------------+-----------------------+
 *	|               Reserved, MUST be 0             |
 *	+-----------------------------------------------+
 *	|                                               |
 *	+--                                           --+
 *	|                 Status Block                  |
 *	+--                                           --+
 *	|                                               |
 *	+-----------------------+-----------------------+
 */
typedef struct {
	PARMBLK pb;		/* standard parameter block */
	word	errintr;	/* error interrupt */
	word	intr;		/* no error interrupt */
	dword	resv;		/* reserved, must be 0 (zero) */
	STATBLK	sb;		/* status block */
} EXTPB;


/* Command list format
 *
 *	+-----------------------------------------------+
 *	|       Parameter block IN pointer              |
 *	+-----------------------------------------------+
 *	|       Parameter block OUT pointer             |
 *	+-----------------------------------------------+
 *	|         Status block IN pointer               |
 *	+-----------------------------------------------+
 *	|         Status block OUT pointer              |
 *	+-----------------------------------------------+
 *	|        Parameter block area size              |
 *	+-----------------------------------------------+
 *	|         Status block area size                |
 *	+-----------------------------------------------+
 *	|                 Reserved                      |
 *	+-----------------------------------------------+
 *	|                 Reserved                      |
 *	+-----------------------------------------------+
 *	|                                               |
 *	|                                               |
 *	|           Parameter block area                |
 *	|                                               |
 *	|                                               |
 *	+-----------------------------------------------+
 *	|                                               |
 *	|                                               |
 *	|             Status block area                 |
 *	|                                               |
 *	|                                               |
 *	+-----------------------------------------------+
 */
#define	RF32NPB	0x40		/* Max. number of parameter blocks */
#define	RF32NSB	0x80		/* Max. number of status blocks */

/* Masks for wrappping around IN/OUT pointers - do NOT define these if
 * the corresponding NxB value is not a power of 2.
 */
#define NPBMASK	0x3f		/* mask for valid parameter block index */
#define	NSBMASK	0x7f		/* mask for valid status block index */

typedef struct {
	dword	pbin;		/* parameter block in pointer */
	dword	pbout;		/* parameter block out pointer */
	dword	sbin;		/* status block in pointer */
	dword	sbout;		/* status block out pointer */
	dword	pbsize;		/* paramter block area size */
	dword	sbsize;		/* status block area size */
	dword	resv1;
	dword	resv2;
	PARMBLK	pblist[RF32NPB];	/* parameter blocks */
	STATBLK	sblist[RF32NSB];	/* status blocks */
} CMDLIST;

/* RF 3200 command definitions */
#define	C_SETUPCL	1	/* setup command list */
#define	C_STOPCL	2	/* stop command list */

#define	C_IDENTIFY	5	/* identify */
#define	C_STATS		6	/* board statistics */
#define	C_OPTION	7	/* general options */
#define	C_SETIOCG	8	/* init I/O control group */
#define	C_DIAGNOSTIC	9	/* board self-test command */

#define	C_SETCONFIG	16	/* configure disk */
#define	C_SETSMDPARAM	17	/* define SMD parameters */
#define	C_GETCONFIG	18	/* return drive configuration */
#define	C_INTERROGATE	19	/* 3200 Only - interrogate disk */
#define	C_GETESDICONF	19	/* 3400 Only - Return ESDI config/status */
#define	C_TAGSTATUS	20	/* issue tag / return status */

#define	C_READ		24	/* read */
#define	C_WRITE		25	/* write */
#define	C_SREAD		26	/* scatter read */
#define	C_GWRITE	27	/* gather write */

#define	C_FORMAT	32	/* format */
#define	C_VERIFY	33	/* verify */
#define	C_SLIPSECTOR	34	/* slip sector */
#define	C_MAPSECTOR	35	/* map sector */
#define	C_MAPTRACK	36	/* map track */
#define	C_READDEFECT	37	/* read defect data */

#define	C_IGNOREID	40	/* ignore id read data */
#define	C_READLONG	41	/* read long */
#define	C_WRITELONG	42	/* write long */
#define	C_READID	43	/* read a single id */
#define	C_READTID	44	/* read a track of id's */
#define	C_REZERO	45	/* rezero */
#define	C_SEEK		46	/* perform a seek */

#define	C_WRITEMAP	56	/* write defect map table */
#define	C_READMAP	57	/* read defect map table */

/************************************************************************
 *	Command #1 - Setup Command List
 ************************************************************************/

/* Alternate Parameter Block - Setup command list */
typedef struct {
	dword	id;		/* unique command identifier */
	byte	addrmod;	/* address modifier used to access VME memory */
	byte	unit;		/* unit (always 0) */
	byte	iocg;		/* I/O control group */
	byte	command;	/* command to execute */
	word	errintr;	/* error interrupt */
	word	doneintr;	/* done interrupt */
	dword	start;		/* command list start address */
	byte	resv[3];	/* reserved */
	byte	listno;		/* number of the command list (1-7) */
} SETUPPB;

/************************************************************************
 *	Command #2 - Stop Command List
 ************************************************************************/

/* Alternate Parameter Block - Stop command list */
typedef struct {
	dword	id;		/* unique command identifier */
	byte	resv1;		/* reserved */
	byte	unit;		/* unit (always 0) */
	byte	iocg;		/* I/O control group */
	byte	command;	/* command to execute */
	dword	resv2;		/* reserved */
	byte	resv3[3];	/* reserved */
	byte	listno;		/* number of the command list (1-7) */
	dword	resv4;		/* reserved */
} STOPPB;

/************************************************************************
 *	Command #5 - Identify
 ************************************************************************/

/* Alternate Status Block for Identify command */
typedef struct {
	dword	id;		/* identifier */
	byte	fwrev;		/* firmware revision level */
	byte	engrev;		/* Engineering revision level */
	byte	error;		/* error code */
	byte	status;		/* status */
	byte	flags;		/* configuration of board (see below) */
	byte	day;		/* day the firmware was generated */
	byte	month;		/* month the firmware was generated */
	byte	year;		/* year the firmware was generated */
} RETID;

/* Flags byte in Identify Status Block */
#define IDNDRIV	0x01		/* set 	 = 4-drive 3200
				 * reset = 2-drive 3200 */
/************************************************************************
 *	Command #6 - Board Statistics
 ************************************************************************/

/* Alternate Parameter Block - Board Statistics */
typedef struct {
	dword	id;		/* unique command identifier */
	byte	addrmod;	/* address modifier used to access VME memory */
	byte	unit;		/* unit for the command to apply */
	byte	control;	/* I/O control group */
	byte	command;	/* command to execute */
	dword	spare;		/* unused */
	dword	vmeaddr;	/* where to put statistics data */
	word	resv;		/* reserved */
	byte	resv1;		/* reserved */
	byte	clear;		/* clear statistics if non-zero */
} STATPB;

/* Statistics Block - returned from Board statistics command */
typedef struct {
	dword	commands;	/* total number of commands */
	dword	reads;		/* number of disk reads */
	dword	diskrd;		/* sectors read from disk */
	dword	cacherd;	/* sectors read from cache */
	dword	writes;		/* number of disk writes */
	dword	diskwr;		/* sectors written to disk */
	word	d1_seek;	/* smd 1: number of seek errors */
	word	d1_smdint;	/* smd 1: number of smd interface errors */
	word	d1_ecccor;	/* smd 1: number of ecc corrections */
	word	d1_nonecc;	/* smd 1: number of non-ecc data errors */
	word	d1_altseek;	/* smd 1: number of alternate seeks */
	word	d1_corfail;	/* smd 1: number of ecc correction failures */
	word	d2_seek;	/* smd 2: number of seek errors */
	word	d2_smdint;	/* smd 2: number of smd interface errors */
	word	d2_ecccor;	/* smd 2: number of ecc corrections */
	word	d2_nonecc;	/* smd 2: number of non-ecc data errors */
	word	d2_altseek;	/* smd 2: number of alternate seeks */
	word	d2_corfail;	/* smd 2: number of ecc correction failures */
	word	d3_seek;	/* smd 3: number of seek errors */
	word	d3_smdint;	/* smd 3: number of smd interface errors */
	word	d3_ecccor;	/* smd 3: number of ecc corrections */
	word	d3_nonecc;	/* smd 3: number of non-ecc data errors */
	word	d3_altseek;	/* smd 3: number of alternate seeks */
	word	d3_corfail;	/* smd 3: number of ecc correction failures */
	word	d4_seek;	/* smd 4: number of seek errors */
	word	d4_smdint;	/* smd 4: number of smd interface errors */
	word	d4_ecccor;	/* smd 4: number of ecc corrections */
	word	d4_nonecc;	/* smd 4: number of non-ecc data errors */
	word	d4_altseek;	/* smd 4: number of alternate seeks */
	word	d4_corfail;	/* smd 4: number of ecc correction failures */
} BOARDST;

/* Tuning Statistics Block - Used by DKIOCGTUNE & DKIOCSTUNE Ioctl Command */
typedef	struct {
	u_short	sort_limit;
	u_char	ready_limit,
		combining;
	u_short	max_clist,
		max_cboard,
		max_pbxferlen,
		max_sbxferlen;
	u_long	board_empty;
	u_long	tot_req_sorted,
		tot_req_combined,
		hit_sort_limit,
		cnt_pbxfer,
		tot_pbxfer_size,
		nopb_ca,
		cnt_sbxfer,
		tot_sbxfer_size,
		norm_mem_xfer_cnt,
		corequest_mem_xfer_cnt;
	u_short	cache_exp_count,
		blk_alloc_fail,
		xfer_restart,
		cache_alloc_fail,
		disk_idle,
		xfer_idle,
		mtr_idle,
		disk_collision,
		max_pbxfersize,
		mtr_collision,
		max_sbxfersize,
		resv1,
		drv0_li,
		drv0_fdl,
		drv0_cg,
		drv0_sb,
		drv1_li,
		drv1_fdl,
		drv1_cg,
		drv1_sb,
		drv2_li,
		drv2_fdl,
		drv2_cg,
		drv2_sb,
		drv3_li,
		drv3_fdl,
		drv3_cg,
		drv3_sb;
} TUNEST;

/************************************************************************
 *	Command #7 - General Options
 ************************************************************************/

/* Alternate Parameter Block - General Options command */
typedef struct {
	dword	id;		/* identifier */
	byte	throttle;	/* bus throttle */
	byte	unit;		/* always zero */
	byte	control;	/* IO control group */
	byte	command;	/* command */
	word	sbintr;		/* status block interrupts (see below) */
	word	sbenable;	/* status block enables (see below) */
	dword	resv[2];
} GOPTPB;

/* Status Block Enable and Interrupt Words */
#define	GOPTWDL	0x01		/* Return Write Data Loaded on writes */
#define	GOPTDTE	0x02		/* Return Retries and Corrections */
#define	GOPTIDE	0x04		/* Return ID/interface related errors */

/************************************************************************
 *	Command #8 - Initialize I/O Control Group
 ************************************************************************/

/* Alternate Parameter Block - Initialize I/O Control Group */
typedef struct {
	dword	id;		/* command identifier */
	byte	resv1;		/* reserved */
	byte	unit;		/* unit (always 0) */
	byte	control;	/* I/O control group */
	byte	command;	/* command to execute */
	word	cache;		/* cache control (see below) */
	byte	readahead;	/* read ahead */
	byte	recovery;	/* recovery */
	word	resv2;		/* reserved */
	byte	dretry;		/* maximum data retries */
	byte	ndretry;	/* maximum non-data retries */
	dword	resv3;		/* reserved */
} IOCGPB;

/* Cache Control word
 
      15  14  13  12  11  10  9   8   7   6   5   4   3   2   1   0
    -----------------------------------------------------------------
    | 0 | 0 | 0 | 0 | 0 | 0 |RCC|RCT| 0 | 0 |SWT|SRD|CWT|RAP|CRD|SEA|
    -----------------------------------------------------------------   
*/

#define	COPTSEA	0x01		/* search cache */
#define	COPTCRD	0x02		/* save read data in cache */
#define	COPTRAP	0x04		/* read ahead priority */
#define	COPTCWT	0x08		/* save write data in cache */
#define	COPTSRD	0x10		/* sort reads */
#define	COPTSWT	0x20		/* sort writes */
#define COPTXHD 0x100           /* read ahead across head */   
#define COPTXCY 0x200           /* read ahead across cylinder */
#define CACHE_BITS	"\20\12XCY\11XHD\6SWT\5SRD\4CWT\3RAP\2CRD\1SEA"

/* Recovery 
 
      7   6   5   4   3   2   1   0
    ---------------------------------
    | 0 | 0 | 0 | 0 | 0 |EXT|COR|IGE|
    ---------------------------------
*/

#define	REC_IGE	0x01		/* ignore ECC */
#define	REC_COR	0x02		/* disable error correction */
#define	REC_EXT	0x04		/* apply track/data separators */
#define RECOV_BITS	"\20\3EXT\2COR\1IGE"

/************************************************************************
 *	Command #16 - Configure Disk
 ************************************************************************/

/* Alternate Parameter Block - Configure disk */
typedef struct {
	dword	id;		/* identifier */
	byte	resv1;		/* reserved */
	byte	unit;		/* unit for this command */
	byte	control;	/* I/O control group */
	byte	command;	/* command to execute */
	word	bytsec;		/* bytes per sector */
	word	cyldsk;		/* cylinders per disk */
	byte	nspares;	/* number of spare sectors per track */
	byte	sectrk;		/* sectors per track */
	byte	headcyl;	/* heads per cylinder */
	byte	basehead;	/* first physical head number this volume */
	byte	resv2[3];	/* reserved */
	byte	flags;		/* characteristics of drive (see below) */
} CONFPB;

/* Flags field in Configure Disk command */
#define	CONFSSP	0x01		/* short sector present */
#define	CONFEAD	0x02		/* extended addressing, 0=use 11th cyl,
				 * 1=use SMD-E addressing */
#define	MASKCFF		(3)	/* all valid configuration flags	    */

/************************************************************************
 *	Command #17 - Define SMD Parameters
 ************************************************************************/

/* Alternate Parameter Block - Define SMD parameters */
typedef struct {
	dword	id;		/* identifier */
	byte	resv1;		/* reserved */
	byte	unit;		/* unit for this command */
	byte	control;	/* I/O control group */
	byte	command;	/* command to execute */
	byte	interleave;	/* interleave factor to use */
	byte	headskew;	/* head skew to use */
	byte	cylskew;	/* cylinder skew to use */
	byte	hgskew;		/* 3400 Only - Head group skew */
	word	recovery;	/* data recovery (see below) */
	byte	idpre;		/* 3200 Only - ID preamble length */
	byte	datapre;	/* 3200 Only - data preamble length */
	dword	resv3;		/* reserved */
} DEFSMDPB;

/* Data Recovery for Define SMD parameters command */
#define	DSMDBUS	0x03ff		/* Data bus value */
#define	DSMDTAG	0xe000		/* tag number (1 to 6) */

/************************************************************************
 *	Command #18 - Return Drive Configuration
 ************************************************************************/

/* Structure for Return Configuration of an SMD drive */
typedef struct {
	word	bytsec;		/* bytes per sector */
	word	cyldsk;		/* cylinders per disk */
	byte	nspares;	/* number of spare sectors per track */
	byte	sectrk;		/* sectors per track */
	byte	headcyl;	/* heads per cylinder */
	byte	basehead;	/* head offset for this volume */
	byte	zero1[3];	/* zeros */
	byte	flags;		/* as in configure */
	dword	secdsk;		/* number of logical sectors per disk */
	byte	interleave;	/* interleave factor */
	byte	headskew;	/* head skew factor */
	byte	cylskew;	/* cylinder skew factor */
	byte	hgskew;		/* 3400 Only - Head group skew */
	word	recovery;	/* data recovery */
	byte	idpre;		/* 3200 Only - id preamble length */
	byte	datapre;	/* 3200 Only - data preamble length */
	dword	zero3;		/* zero */
	dword	zero4;		/* zero */
} RETCONF;
/************************************************************************
*	Command # 44 - read track of id's
************************************************************************/


typedef struct  {
	word    cyl;
	byte    sec;
        byte	head;
	byte	alt;
	byte	flag;
}TRAKID;
/*********** Structure defect map is not currently used.    ****************/

typedef struct {
	word terr;
	word ter;
	word;
	word;
	word;
	word;
	word;
	word;
	word;
	word;
	word;
	word;
	word;
	}DEFECTMAP;

/************************************************************************
*      Command # 46 - Seek to a specific head/cyclinder
*************************************************************************/
typedef struct {
	word cyl;
	byte head;
	byte sec;
}LOOK;

/************************************************************************
 *	Command #19 - Return ESDI Drive Configuration
 ************************************************************************/

/* Structure for Return Configuration of an ESDI drive */
typedef struct {
	word	genconf;	/* general configuration */
	word 	fixcyl; 	/* number of fixed cylinders */
	word 	remcyl;		/* number of removable cylinders */
	byte	remhead;	/* number of removable heads */
	byte	fixhead;	/* number of fixed heads */
	word	minbytrk;	/* minumun unformatted bytes per track */
	word	bytesec;	/* unformatted bytes per sector */
	byte	resv0;		/* reserved */
	byte	sectrk;		/* sectors per track */
	byte	isgbyte;	/* isg bytes after index */
	byte 	minisg;		/* minimum isg bytes */
	byte	spare;		/* spare */
	byte	plosync;	/* bytes in plo sync */
	word	nwords;		/* number of words of vendor-unique status */
	word	resv[5];	/* reserved */
	byte	vendid;		/* vendor id */
	byte	vendmodel;	/* vendor model */
	word	statbits;	/* general drive status bits */
	word	resv1;		/* reserved */
} ESDIPARAMS;


/*
 * Definitions for General Configuration Word
 */
#define GC_TAPE		0x8000	/* Tape drive				*/
#define GC_TOLGAP	0x4000	/* Format speed tolerance gap required	*/
#define GC_TRKOFF	0x2000	/* Track offset option available	*/
#define GC_DATAOFF	0x1000	/* Data strobe offset option available	*/
#define GC_SPEEDTOL	0x0800	/* Rotational speed tolerance > 0.5%	*/
#define GC_XFER10	0x0400	/* Transfer rate > 10MHz		*/
#define GC_XFER510	0x0200	/* Transfer rate > 5MHz, <= 10MHz	*/
#define GC_XFER5	0x0100	/* Transfer rate <= 5MHz		*/
#define GC_REMOV	0x0080	/* Removable cartridge drive		*/
#define GC_FIXED	0x0040	/* Fixed drive				*/
#define GC_MOTORCTL	0x0020	/* Spindle motor control option implmn. */
#define GC_HDSWLONG	0x0010	/* Head switch time > 15 uS		*/
#define GC_RLL		0x0008	/* RLL encoded (not MFM)		*/
#define GC_CTLSOFT	0x0004	/* Controller soft-sectored (ADR mark)	*/
#define GC_DRVHARD	0x0002	/* Drive hard-sectored (Sector pulse)	*/
#define GC_CTLHARD	0x0001	/* Controller hard-sectored (Byte clock)*/

/*
 * Definitions for General drive status bits above
 */
/*			0x8000		Not used			*/
#define DS_NOREMOV	0x4000	/* Removable media not present		*/
#define DS_WPREM	0x2000	/* Write protected, removable media	*/
#define DS_WPFIX	0x1000	/* Write protected, fixed media		*/
/*			0x0800		Not used			*/
/*			0x0400		Not used			*/
#define DS_SPINDWN	0x0200	/* Spindle motor stopped		*/
#define DS_POR		0x0100	/* Power on reset conditions exist	*/
#define DS_CMDPARITY	0x0080	/* Command data parity fault		*/
#define DS_INTFLT	0x0040	/* Interface fault			*/
#define DS_INVCMD	0x0020	/* Invalid or unimplemented cmd fault	*/
#define DS_SEEKFLT	0x0010	/* Seek fault				*/
#define DS_TOFLT	0x0008	/* Write gate with track offset fault	*/
#define DS_STATAVAIL	0x0004	/* Vendor-unique status available	*/
#define DS_WRTFLT	0x0002	/* Write fault				*/
#define DS_MEDIACHG	0x0001	/* Removable media changed		*/


/************************************************************************
 *	Command #20 - Issue Tag / Return Status
 ************************************************************************/

/* Alternate Parameter Block for Issue Tag / Return Status command */
typedef struct {
	dword	id;		/* identifier */
	byte	resv1;		/* reserved */
	byte	unit;		/* unit */
	byte	control;	/* IO Control Group */
	byte	command;	/* command number */
	word	resv2;		/* reserved */
	word	data;		/* data to send */
	word	resv3;		/* reserved */
	word	tag;		/* tag number */
	dword	resv4;		/* reserved */
} TAGPB;

/* Issue Tag Status Block */
typedef struct {
	dword	id;		/* identifier */
	word	zero1;		/* zero */
	byte	error;		/* error code */
	byte	status;		/* status */
	byte	zero2[3];	/* zeros */
	byte	data;		/* drive data */
} TAGSB;

/************************************************************************
 *	Command #26 and #27 - Scatter Read and Gather Write
 ************************************************************************/

/* Structure for the scatter/gather descriptor entry */
typedef struct {
	dword	sdptr;		/* source/destination pointer */
	dword	bytes;		/* byte count */
} SGDESCRIPTOR;

#define sglistsize	blkcount	/* number of sg pairs in list */

/************************************************************************
 *	Command #32 and #33 - Format and Verify
 ************************************************************************/

/* Alternate Parameter Block - format and verify */
typedef struct {
	dword	id;		/* unique command identifier */
	byte	addrmod;	/* address modifier used to access VME memory */
	byte	unit;		/* unit for the command to apply */
	byte	control;	/* I/O control group */
	byte	command;	/* command to execute */
	dword	diskaddr;	/* absolute starting sector number */
	dword	vmeaddr;	/* VME address to read from or write to */
	dword	blkcount;	/* number of logical blocks to process */
} FMTVERPB;

/************************************************************************
 *	Command #34, #35, and #36 - Slip Sector, Map Track, and Map Sector
 ************************************************************************/

/* Alternate Parameter Block - map and slip */
typedef struct {
	dword	id;		/* unique command identifier */
	byte	resv1;		/* reserved */
	byte	unit;		/* unit for the command to apply */
	byte	control;	/* I/O control group */
	byte	command;	/* command to execute */
	dword	baditem;	/* bad sector or track (first sector) */
	dword	alternate;	/* alternate track or sector */
	word	resv2;		/* reserved */
	word	recover;	/* recover data in error? */
} MAPPB;
/************************************************************************
 *	Command #37 - Read Defect Data
 ************************************************************************/

/* Generic disk defect structure */
typedef struct {
	byte	zero1;		/* zero */
	byte	x19;		/* Constant (0x19) */
	word	cylinder;	/* cylinder number */
	byte	head;		/* head number */
	byte	zero2;		/* zero */
	word	pos1;		/* position 1 */
	word	len1;		/* length 1 */
	word	pos2;		/* position 2 */
	word	len2;		/* length 2 */
	word	pos3;		/* position 3 */
	word	len3;		/* length 3 */
	word	pos4;		/* position 4 */
	word	len4;		/* length 4 */
	byte	zero3;		/* zero */
	byte	xf0;		/* Constant (0xf0) */
} DEFECTDB;

/* maximum size for writing */
#define	MAX_RF_BSIZE	1024*63  /* maximum block size is 48k */
 
/* number of spares for sector slipping */
/* Define this if there is the element dkl_apc in your disk label structure. */
#define NSPARES		/* Label contains # of spare sectors per cylinder */
			/* On SunOS 3.2, this IS in disk label. */
/* This is the number of spares PER CYLINDER! */

/* Alternate Parameter Block - Initialize I/O Control Group */
struct iocgrp {
	word    group;		/* I/O control group */
	word	cache;		/* cache control (see below) */
	byte	readahead;	/* read ahead */
	byte	recovery;	/* recovery */
	byte	dretry;		/* maximum data retries */
	byte	ndretry;	/* maximum non-data retries */
	word	resv;		/* reserved */
};

#ifdef	RFSTATS
/* For gathering statistics: */
typedef struct {
	struct timeval	iorq_time;
	u_long	iorq_bn;
	u_char	iorq_rq;
	u_char	iorq_ns;
	u_char	iorq_type;
} IORQ_STAT;
/* Types: */
#define	IORQ_START	0x1
#define	IORQ_FINISH	0x2

typedef struct {
	long	which;
	IORQ_STAT iorq;
} IORQS;

typedef struct {
	long	niorq;
	IORQ_STAT	iorqs[512 * 20];
} IORQ;
#endif

struct rf_geom {
	unsigned short	dk_cksum;
	unsigned short	dk_magic;
	unsigned short	dk_nbps;
	unsigned short	dk_ncyl;
	unsigned short	dk_ntpc;
	unsigned short	dk_nspt;
	unsigned short	dk_intrlv;
	unsigned short	dk_gap1;
	unsigned short	dk_gap2;
	unsigned short	dk_nspares;
	unsigned short	dk_hdskew;
	unsigned short	dk_cylskew;
	unsigned short	dk_hgskew;
	unsigned short	dk_cache;
	unsigned short	dk_readahead;
	unsigned short	dk_recovery;
	unsigned short	dk_dretry;
	unsigned short	dk_ndretry;
};
#define	GEOMSIZE	sizeof(struct rf_geom)/sizeof(unsigned short)

#define	ADRM_STD_S_P		0x3E	/*    Standard Supervisory Program */
#define	ADRM_STD_S_D		0x3D	/*    Standard Supervisory Data */
#define	ADRM_STD_N_P		0x3A	/*    Standard Normal Program */
#define	ADRM_STD_N_D		0x39	/*    Standard Normal Data */
#define	ADRM_SHT_S_IO		0x2D	/*    Short Supervisory IO */
#define	ADRM_SHT_N_IO		0x29	/*    Short Normal IO */
#define	ADRM_EXT_S_P		0x0E	/*    Extended Supervisory Program */
#define	ADRM_EXT_S_D		0x0D	/*    Extended Supervisory Data */
#define	ADRM_EXT_N_P		0x0A	/*    Extended Normal Program */

#define	ADRM_EXT_N_D		0x09	/*    Extended Normal Data */
#define	VME_ADD_MOD	ADRM_STD_N_D
#define THROTTLE	8	/* 8 transfers (32 bytes) - size of the bus fifo */

#define	RF_32BIT	CTRLWID
#define	RF_16BIT	0

#ifdef	M68020
#ifdef	M68025
#define	RF_XFER		(RF_32BIT)
#else	M68025
#define	RF_XFER		(((*BSR) & BSR_32BITMEM) ? RF_16BIT : RF_32BIT)
#endif	M68025
#else	M68020
#define	RF_XFER		(RF_16BIT)
#endif	M68020

#define	CNTRL		(CTRLSET | RF_XFER)	/* OR in CTRLWID for 32 bit xfer*/

#define	RF_MAGIC 0xc37a

#define	UTOCN(x)	((x) >> 2)
#define	UTODR(x)	((x) & 3)
#define	DRTORFU(x)	(((x) << 1) + 1)

/* Maximum # of IO Control Group used - This means we use groups 1-3 */
#define	MAXIOCG 3

#define IOCG_NORMAL	1
#define IOCG_BB		2
#define IOCG_DEFAULT	3		/* should be same as iocgrp #0 */

#define CACHE_1	(COPTSEA | COPTCRD | COPTCWT | COPTXHD)
#define CACHE_2	CACHE_1
#define CACHE_3	(COPTSEA | COPTCRD | COPTSRD | COPTXHD)

#define RECOV_1	0
#define RECOV_2	(REC_IGE | REC_COR)
#define RECOV_3	0

