#define	DEBUG
#define	SGLCMD

/*
 * Unix driver for the Caprico Rimfire EDSI disk controller
 */

/* 
 *	Nrf	- Defines how many disk drives are hooked up to RF3200's
 *	NRF	- Defines how many RF 3200's are plugged into the system
 */
#include "rf.h"

#if NRF > 0

#include "../machine/pte.h"
#include "../machine/board.h"

#include "param.h"
#include "systm.h"
#include "dk.h"
#include "dkbad.h"
#include "buf.h"
#include "conf.h"
#include "user.h"
#include "map.h"
#include "vm.h"
#include "cmap.h"
#include "uio.h"
#include "kernel.h"
#include "syslog.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/rfreg.h"

/* For those routines in the driver which are visible to the outside world,
 * define what they return.
 */
int rfprobe(), rfattach(), rfslave(), rfopen(), rfintr();
int rfioctl(), rfread(),   rfwrite(), rfdump();

void 			rfstrategy(), rfstrat();
int			rfminphys();

#define	RFOFFSET	0x10
#define	RFADDR0	  	((RF32REG *)&vme_short[0xEE00+RFOFFSET])
#define	RFADDR1	  	((RF32REG *)&vme_short[0xE600+RFOFFSET])

u_short *RFstd[] = { 
		RFADDR0,
		RFADDR1,
		0 };


struct	RF_stat {
	short		rf_ndrive;	/* Number of drives on controller */
	short		rf_wticks;	/* Monitor time for function */
	u_short		rf_cmd;		/* command to issue */
	u_int		rf_bn;		/* block number */
	u_int		rf_addr;	/* physical address of transfer */
	int		rf_nblks;	/* number of blocks to transfer */
	int		rf_drive;	/* drive/volume to issue */
	int		rf_vector;	/* interrupt vector */
#ifdef	SGLCMD
	u_int		rf_status;	/* status block returned */
#else	SGLCMD
	int		rf_req;		/* pending requests */
	CMDLIST		rf_cmdlist;	/* command list */
#endif	SGLCMD
	int		rf_bsybit;	/* bsybit */
	int		rf_ctype;	/* controller type */
} RF_stat[NRF];

static struct qb_ctlr		 *rfcinfo[NRF];	/* per controller */
static struct qb_device		 *rfdinfo[Nrf];	/* per drive */

struct qb_driver RFdriver = {
	rfprobe,		/* the probe routine */
	rfslave,		/* the slave routine */
	rfattach,		/* the attach routine */
	RFstd,			/* controller addresses */
	"rf",			/* the device name */
	rfdinfo,		/* info per drive */
	"RF",			/* the controller name */
	rfcinfo,		/* info per controller */
};

struct iocgrp iocg[MAXIOCG] = {
	{ 1, CACHE_1, 24, RECOV_1,  3, 3, 0},		/* 'normal' i/o */
	{ 2, CACHE_2, 24, RECOV_2,  3, 3, 0},		/* bad blocking i/o */
	{ 3, CACHE_3, 24, RECOV_3, 11, 3, 0},		/* default? */
	};


#ifdef	RFSTATS
/* Statistic gathering information: */
static int		iorqsi = 0;
static IORQ		iorq;
static int		niorqs = sizeof (iorq.iorqs) / (sizeof iorq.iorqs[0]);
extern struct timeval	time;
#endif

struct buf		rrfbuf[Nrf];
struct dkbad		rfbad[Nrf];

struct	qb_diskst	rfst[Nrf];
EXTPB			rf_expb;

#define	b_bn		b_resid	
#define	rfunit(bp)	(minor((bp)->b_dev) >> 3)
#define	rfblock(bp)	((bp)->b_blkno)

static int		 vector;

int			rfwstart, rfwatch();	/* Have started guardian */

#define RF_MAXCNT	MAX_RF_BSIZE
#define	RF_TIMEOUT	hz*5
#define	RF_RETRYCNT	10

char			rf_geombuf[512];		/* disk geom */


/* To keep track of all the parameter blocks in the RF3200, the ID field
 * of the standard parameter block is actually a pointer to the associated
 * UNIX buffer.
 * Thus, when we receive a status block from the controller, we can
 * use its identifier to determine which UNIX buffer to free.
 * We also need to keep the return value of mbsetup (mbinfo) so that
 * we can release the entries in the kernel memory map via mbrelse.
 * Since we don't use the available list pointers in the buf structure
 * we will abuse them by saving mbinfo as a pointer to struct buf.
 */

#define	RFINTLEV	3		/* interrupt priority level */

rfprobe(rfaddr)
register RF32REG	*rfaddr;
{

	extern int	cvec;

	/* Reset controller and check if it exists */
	rfaddr = (char *)rfaddr - RFOFFSET;
	*(char *)&rfaddr->reset = 1;
	if (rfwait(rfaddr) != 0)
		return 0;

	cvec = vector = freevec();
	clev = RFINTLEV;
	clevmax = clev_biomax;
	clev_bio = MAX(clev, clev_bio);
	return(sizeof(RF32REG));
}

rfslave(qi, rfaddr)
register struct qb_device	*qi;
register RF32REG		*rfaddr;
{
	register struct rf_geom		*geom;

	rfaddr = (char *)rfaddr - RFOFFSET;
	if (rfinit(qi, rfaddr))
		return 0;
	geom = (struct rf_geom *)rf_geombuf;
	if (rfgeom(qi, rfaddr, geom, C_READ, 1))
		return 0;
	return 1;
}

rfattach(qi)
register struct qb_device	*qi;
{
	register struct qb_diskst	*st;
	register struct rf_geom		*geom;

	/*
	 * Start watchdog for lost interupts from drive
	 */
	if (rfwstart == 0) {
		timeout(rfwatch, (caddr_t)0, RF_TIMEOUT);
		rfwstart++;
	}

	RF_stat[qi->qi_ctlr].rf_ndrive++;
	st = &rfst[qi->qi_unit];

	geom = (struct rf_geom *)rf_geombuf;
	if (rfconfig(qi, geom)) {
		printf("	TIMEOUT ");
		return;
	}

	ST_ntpc = geom->dk_ntpc;	/* # of tracks (heads) */
	ST_ncpd = geom->dk_ncyl-1;	/* # of cylinders */
	ST_nspt = geom->dk_nspt;	/* # of sectors per track */
	printf("	%mM (%d x %d x %d) ",
	    ST_nspt * ST_ntpc * ST_ncpd * 512,
 	    ST_nspt, ST_ntpc, ST_ncpd);
	diskpart(ST_size, ST_nspt, ST_ntpc, ST_ncpd);
	ST_nspc = ST_nspt * ST_ntpc;

	/* Initialize iostat values */
	if (qi->qi_dk >= 0)
		dk_bps[qi->qi_dk] = (3600/60)*512*ST_nspt;
}

rfinit(qi, rfaddr)
register struct qb_device	*qi;
register RF32REG		*rfaddr;
{
	int			foo = 0;
	register EXTPB		*xpb;
	EXTPB			expb;
	register CONFPB		*cpb;
	register GOPTPB		*gopt;
	register int		ctlr = qi->qi_ctlr;

	if (RF_stat[ctlr].rf_ndrive)
		return 0;		/* already initialized */
	RF_stat[ctlr].rf_bsybit = 0;
	RF_stat[ctlr].rf_ctype = rfaddr->status & STATCTYPE;
	RF_stat[ctlr].rf_vector = vector;

	xpb = &expb;

	bzero(xpb, sizeof(expb));
	gopt = (GOPTPB *)&xpb->pb;
	gopt->id = 0;
	gopt->command = C_OPTION;
	gopt->control = 0;
	gopt->unit = 0;
	gopt->throttle = THROTTLE;
	gopt->sbenable = GOPTWDL | GOPTDTE | GOPTIDE;
	gopt->sbintr = GOPTWDL;
	gopt->resv[0] = 0;
	gopt->resv[1] = 0;
	if (rfsgl(xpb, ctlr, rfaddr, 1)) {
#ifdef	DEBUG
		printf("rf: cannot set option, err:%x, stat:%b\n",
				xpb->sb.errcode, xpb->sb.status, ST_BITS);
#endif	DEBUG
		return(-1);
	}


	bzero(xpb, sizeof(expb));
	cpb = (CONFPB *)&xpb->pb;
	cpb->id = 0;
	cpb->command = C_SETCONFIG;
	cpb->control = 0;
	cpb->unit = DRTORFU(qi->qi_slave);
	cpb->resv1 = 0;
	cpb->bytsec = 512;
	cpb->basehead = 0;
	if (RF_stat[ctlr].rf_ctype == STATRF3200) {
		cpb->sectrk = 1;
		cpb->cyldsk = 1;
		cpb->headcyl = 1;
		cpb->flags = CONFSSP;
	} else {
		cpb->sectrk = 1;
		cpb->cyldsk = 1;
		cpb->headcyl = 1;
		cpb->flags = 0;	/* STATRF3400 */
	}

	cpb->nspares = 0;
	cpb->resv2[0] = 0;
	cpb->resv2[1] = 0;
	cpb->resv2[2] = 0;

	if (rfsgl(xpb, ctlr, rfaddr, 1)) {
#ifdef	DEBUG
		printf("rf: cannot initialize, err:%x, stat:%b\n",
				xpb->sb.errcode, xpb->sb.status, ST_BITS);
#endif	DEBUG
		return (-1);
	}

	return(0);
}

unsigned short
rfchksum(geom)
	struct rf_geom		*geom;
{
	int			i;
	unsigned short		*wp;
	unsigned short		cksum;

	cksum = 0;
	for (i = 0, wp = (unsigned short *)&(geom->dk_magic);
	    				i < GEOMSIZE - 1; i++)
		cksum ^= *wp++;
	return cksum;
}

rfgeom(qi, rfaddr, geom, func, silent)
struct qb_device	*qi;
register RF32REG	*rfaddr;
struct rf_geom		*geom;
int			func;
int			silent;
{
	register EXTPB		*xpb;
	EXTPB			expb;
	register PARMBLK	*pb;

	/* set up a read command for the first sector */
	xpb = &expb;

	bzero(xpb, sizeof(expb));
	pb = &xpb->pb;
	pb->id = 0;
	pb->control = 0;
	pb->unit = DRTORFU(qi->qi_slave);
	pb->addrmod = VME_ADD_MOD;
	pb->diskaddr = 0;
	pb->vmeaddr = svtop(geom);
	pb->blkcount = 1;
	pb->reserved = 0;
	pb->command = func;

	if (func == C_WRITE) {
		geom->dk_magic = RF_MAGIC;
		geom->dk_cksum = rfchksum(geom);
		if (rfsgl(xpb, qi->qi_ctlr, rfaddr, 1)) {
#ifdef	DEBUG
			printf("rf: can't write geometry, err:%x, stat:%b\n",
				xpb->sb.errcode, xpb->sb.status, ST_BITS);
#endif	DEBUG
			return (-1);
		}
	} else {
		if (rfsgl(xpb, qi->qi_ctlr, rfaddr, 1)) {
#ifdef	DEBUG
			if (silent == 0)
			   printf("rf: can't read geometry, err:%x, stat:%b\n",
				xpb->sb.errcode, xpb->sb.status, ST_BITS);
#endif	DEBUG
			return (-1);
		}
		if (geom->dk_magic != RF_MAGIC) {
#ifdef	DEBUG
			printf("rf: bad geometry magic\n");
#endif	DEBUG
			return (-1);
		}
		if (rfchksum(geom) != geom->dk_cksum) {
#ifdef	DEBUG
			printf("rf: invalid geometry checksum\n");
#endif	DEBUG
			return (-1);
		}
	}
	return (0);
}

rfconfig(qi, geom)
struct qb_device	*qi;
struct rf_geom		*geom;
{
	register EXTPB		*xpb;
	static EXTPB		expb;
	register CONFPB		*cpb;
	register DEFSMDPB	*smdpb;
	register IOCGPB 	*iocgpb;
	register int		i;
	register RF32REG	*rfaddr =
			(RF32REG *)((char *) qi->qi_mi->qm_addr - RFOFFSET);

	xpb = &expb;

	bzero(xpb, sizeof(expb));
	cpb = (CONFPB *)&xpb->pb;
	cpb->id = 0;
	cpb->command = C_SETCONFIG;
	cpb->control = 0;
	cpb->unit = DRTORFU(qi->qi_slave);
	cpb->resv1 = 0;
	cpb->cyldsk = geom->dk_ncyl;
	cpb->bytsec = geom->dk_nbps;
	cpb->basehead = 0;
	cpb->headcyl = geom->dk_ntpc;
	cpb->sectrk = geom->dk_nspt;
	cpb->nspares = geom->dk_nspares;
	cpb->flags = 0;
	cpb->resv2[0] = 0;
	cpb->resv2[1] = 0;
	cpb->resv2[2] = 0;

	if (rfsgl(xpb, qi->qi_ctlr, rfaddr, 1)) {
#ifdef	DEBUG
		printf("rf: cannot configure, err:%x, stat:%b\n",
				xpb->sb.errcode, xpb->sb.status, ST_BITS);
#endif	DEBUG
		return (-1);
	}

	bzero(xpb, sizeof(expb));
	smdpb = (DEFSMDPB *)&xpb->pb;
	smdpb->id = 0;
	smdpb->resv1 = 0;
	smdpb->unit = DRTORFU(qi->qi_slave);
	smdpb->control = 0;
	smdpb->command = C_SETSMDPARAM;
	smdpb->interleave = 1;
	smdpb->headskew = geom->dk_hdskew;
	smdpb->cylskew = geom->dk_cylskew;
	smdpb->hgskew = 0;
	smdpb->recovery = 0;
	if (RF_stat[qi->qi_ctlr].rf_ctype == STATRF3200) {
		smdpb->idpre = geom->dk_gap1;
		smdpb->datapre = geom->dk_gap2;
	} else {
		smdpb->idpre = 0;
		smdpb->datapre = 0;
	}
	smdpb->resv3 = 0;

	if (rfsgl(xpb, qi->qi_ctlr, rfaddr, 1)) {
#ifdef	DEBUG
		printf("rf: define params failed, err:%x, stat:%b\n",
				xpb->sb.errcode, xpb->sb.status, ST_BITS);
#endif	DEBUG
		return(-1);
	}

	iocg[IOCG_NORMAL].cache = geom->dk_cache;
	iocg[IOCG_NORMAL].readahead = geom->dk_readahead;
	iocg[IOCG_NORMAL].recovery = geom->dk_recovery;
	iocg[IOCG_NORMAL].dretry = geom->dk_dretry;
	iocg[IOCG_NORMAL].ndretry = geom->dk_ndretry;

	iocg[IOCG_BB] = iocg[IOCG_NORMAL];	/* bb should be as in normal */
	iocg[IOCG_BB].recovery = RECOV_2;	/* except for recovery */

	for (i = 1; i <= MAXIOCG; i++) {
		bzero(xpb, sizeof(expb));

		iocgpb = (IOCGPB *)&xpb->pb;
		iocgpb->id = 0;
		iocgpb->command = C_SETIOCG;
		iocgpb->control = i;
		iocgpb->unit = 0;
		iocgpb->resv1 = 0;
		iocgpb->recovery = iocg[i-1].recovery;
		iocgpb->readahead = iocg[i-1].readahead;
		iocgpb->cache = iocg[i-1].cache;
		iocgpb->ndretry = iocg[i-1].ndretry;
		iocgpb->dretry = iocg[i-1].dretry;
		iocgpb->resv2 = 0;
		iocgpb->resv3 = 0;

		if (rfsgl(xpb, qi->qi_ctlr, rfaddr, 1)) {
#ifdef	DEBUG
			printf("rf: cannot set iocg, err:%x, stat:%b\n",
				xpb->sb.errcode, xpb->sb.status, ST_BITS);
#endif	DEBUG
		}
	}
	return(0);
}

rfopen(dev)
dev_t dev;
{
	register int		 	unit = minor(dev) >> 3;
	register struct qb_device 	*qi;

	if (unit >= Nrf || (qi = rfdinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	return (0);
}

void
rfstrategy(bp)
register struct buf *bp;
{
	register struct qb_device	 *qi;
	register struct qb_diskst	 *st;
	register struct buf		 *bbp;
	register int			 unit;
	int				 partition = minor(bp->b_dev) & 07;
	long bn, sz;

	unit = rfunit(bp);
	if (unit >= Nrf)
		goto bad;
	qi = rfdinfo[unit];
	if (qi == 0 || qi->qi_alive == 0)
		goto bad;
	st = &rfst[qi->qi_unit];
	sz = (bp->b_bcount+511) >> 9;
	if (bp->b_blkno < 0 ||
	    (bn = rfblock(bp))+sz > ST_size[partition].nblocks)
		goto bad;
	/* 
	 * The controller can only do sector transfers. On a read we must
	 * do the transfer into a kernel buffer and then out to memory so
	 * we do not overwrite memory beyond the requested location.
	 */
	if ((bp->b_bcount & 511) && (bp->b_flags&B_READ)) {
		bbp = (struct buf *)getqeblk((bp->b_bcount+511) & ~511, 
			BQUALIFY(bp->b_dev));
		bbp->b_dev = bp->b_dev;
		bbp->b_blkno = bp->b_blkno;
		bbp->b_flags |= (bp->b_flags & (B_READ|B_WRITE));
		badstrat(qi, bbp, &rfbad[qi->qi_unit], rfstrat, ST_nspt, 
		    ST_ntpc, ST_ncpd, ST_size[partition].cyloff);
		biowait(bbp);
		pcopyout(bbp->b_un.b_addr, qbaddr(bp), bp->b_bcount);
		bp->b_flags |= (bbp->b_flags & B_ERROR);
		bp->b_resid = bbp->b_resid;
		bbp->b_dev = NODEV;
		bfree(bbp);
		brelse(bbp);
		iodone(bp);
		return;
	}
	badstrat(qi, bp, &rfbad[qi->qi_unit], rfstrat, ST_nspt, 
	    ST_ntpc, ST_ncpd, ST_size[partition].cyloff);
	return;

bad:	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
}

void
rfstrat(qi, bp)
register struct qb_device	 *qi;
register struct buf		 *bp;
{
	register struct qb_diskst	 *st = &rfst[qi->qi_unit];
	int				 partition = minor(bp->b_dev) & 07;
	int				 s;
	register struct buf		 *dp;

	bp->b_bn = rfblock(bp) + (ST_size[partition].cyloff * ST_nspc);
	s = splx(qi->qi_mi->qm_psl);
	dp = &qi->qi_mi->qm_tab;
	disksort(dp, bp);
	if (dp->b_active == 0)
		rfstartup(qi->qi_mi);
	splx(s);
}

/*
 * Start up a transfer on a drive.
 */
rfstartup(qm)
register struct qb_ctlr		 *qm;
{
	register struct buf		 *bp;
	register struct qb_device	 *qi;
	register struct RF_stat		 *stat;

	if ((bp = qm->qm_tab.b_actf) == NULL)
		return;

	/* Mark controller busy, and determine destination.  */
	qm->qm_tab.b_active++;
	qi = rfdinfo[rfunit(bp)];		/* Controller */
	stat = &RF_stat[qm->qm_ctlr];

	/* save away current data transfer drive info */
	stat->rf_bn = bp->b_bn + rfst[qi->qi_unit].st_nspc;
	stat->rf_addr = qbaddr(bp);
	stat->rf_nblks = (bp->b_bcount+511) >> 9;
	stat->rf_drive = qi->qi_slave & 0x3;
	stat->rf_cmd = (bp->b_flags & B_READ) ? C_READ : C_WRITE;
	rfio(qm->qm_ctlr, (RF32REG *)((char *)qm->qm_addr - RFOFFSET), stat,1);

	if (qi->qi_dk >= 0) {
		dk_busy |= 1 << qi->qi_dk;
		dk_xfer[qi->qi_dk]++;
		dk_wds[qi->qi_dk] += bp->b_bcount >> 6;
	}
}

rfio(ctlr, rfaddr, stat, int_enable)
int			ctlr;
register RF32REG	*rfaddr;
register struct RF_stat	*stat;
int			int_enable;
{
	register EXTPB		*xpb;
	register PARMBLK	*pb;

	xpb = &rf_expb;

	bzero(xpb, sizeof(EXTPB));
	pb = &xpb->pb;
	pb->id = 0;
	pb->unit = DRTORFU(stat->rf_drive);
	pb->addrmod = VME_ADD_MOD;
	pb->diskaddr = stat->rf_bn;
	pb->vmeaddr = stat->rf_addr;
	pb->blkcount = stat->rf_nblks;
	pb->command = stat->rf_cmd;
	stat->rf_status = (u_int)&(xpb->sb);
	if (int_enable) {
		xpb->intr = (RFINTLEV << 8) | RF_stat[ctlr].rf_vector;
		(void) rfsgl(xpb, ctlr, rfaddr, 0);
	} else
		(void) rfsgl(xpb, ctlr, rfaddr, 1);
}

/*
 * Handle a disk interrupt.
 */
rfintr(ctlr)
register int	ctlr;
{
	register struct qb_ctlr		*qm = rfcinfo[ctlr];
	register struct buf		*bp = qm->qm_tab.b_actf;
	register struct qb_device	*qi = rfdinfo[rfunit(bp)];
	register struct RF_stat		*stat = &RF_stat[ctlr];
	register STATBLK		*sb = (STATBLK *)stat->rf_status;

	stat->rf_wticks = 0;
	dk_busy &= ~(1 << qi->qi_dk);
	/*
	 * Check for and process errors on either the drive or the controller.
	 */
	if (sb->status & ST_ERR) {
/* printf("err:%x, stat:%b\n", xpb->sb.errcode, xpb->sb.status, ST_BITS); /**/
		/*
		 * After 10 retries give up.
		 */
		if (++qm->qm_tab.b_errcnt > RF_RETRYCNT) {
			harderr(bp, "rf");
			printf("err:%x, stat:%b\n",
				sb->errcode, sb->status, ST_BITS);
			bp->b_flags |= B_ERROR;
		} else
			qm->qm_tab.b_active = 0;	/* force retry */
	}
	/*
	 * If still ``active'', then don't need any more retries.
	 */
	if (qm->qm_tab.b_active) {
		dk_busy &= ~(1 << qi->qi_dk);
		qm->qm_tab.b_active = qm->qm_tab.b_errcnt = 0;
		bp->b_resid = 0;
		qm->qm_tab.b_actf = bp->av_forw;
		iodone(bp);
	}
	wakeup((caddr_t)&stat->rf_status);
	/*
	 * If controller not transferring and have request, start controller.
	 */
	if (qm->qm_tab.b_actf && qm->qm_tab.b_active == 0)
		rfstartup(qm);
}

rfsgl(xpb, ctlr, rfaddr, wait)
register EXTPB		*xpb;
int			ctlr;
register RF32REG 	*rfaddr;
int			wait;
{
	dword	pb;
	int	timer;

	xpb->pb.control = IOCG_NORMAL;
	if (rfbusy(ctlr, rfaddr, wait))
		return -1;

	pb = svtop(xpb);

	/* Write the control and address modifier to the address buffer port */
	rfaddr->addrbuf = SWAB(CNTRL << 8 | VME_ADD_MOD);
	rfaddr->addrbuf = SWAB((pb >> 16) & 0xffff);
	rfaddr->addrbuf = SWAB(pb & 0xffff);

	rfaddr->attention = 0;

	if (!wait)
		return(0);
	timer = 0;
	while ((xpb->sb.status & ST_CC) == 0) {
		if (++timer == 50000) {
			printf("rf: timeout\n");
			return (-1);
		} else
			DELAY(400);
	}
	if (xpb->sb.status & ST_ERR)
		return(-1);
	return(0);
}

rfread(dev, uio)
dev_t		dev;
struct uio	*uio;
{
	register int		unit = minor(dev) >> 3;

	if (unit >= Nrf)
		return (ENXIO);
	return (physio(rfstrategy, &rrfbuf[unit], dev, B_READ, rfminphys, uio));
}

rfwrite(dev, uio)
dev_t		dev;
struct uio	*uio;
{
	register int		unit = minor(dev) >> 3;

	if (unit >= Nrf)
		return (ENXIO);
	return (physio(rfstrategy, &rrfbuf[unit], dev, B_WRITE, rfminphys, uio));
}

/*
 * Reset driver after lost interupt. Cancel software state of all pending 
 * transfers and restart all units and the controller.
 */
rfreset()
{
	register struct qb_ctlr		*qm;
	register int			ctlr;
	register RF32REG		*rfaddr;
			

	for (ctlr = 0; ctlr < NRF; ctlr++) {
		qm = rfcinfo[ctlr];
		if (qm == 0 || qm->qm_alive == 0 || qm->qm_tab.b_active == 0)
			continue;
		printf(" RF%d", ctlr);
		qm->qm_tab.b_active = 0;
		rfstartup(qm);
	}
}

rfsizeof(dev)
dev_t	dev;
{
	register int			unit = minor(dev) >> 3;
	register struct qb_device	*qi;
	register struct qb_diskst	*st;

	if (unit >= Nrf || (qi = rfdinfo[unit]) == 0 || qi->qi_alive == 0)
		return (-1);
	st = &rfst[qi->qi_unit];
	return (ST_size[minor(dev)&0x7].nblocks);
}

rfwait(rfaddr)
RF32REG 	*rfaddr;
{ 	
	register int		l = 0;

	DELAY(400);
	while ((rfaddr->status & SWAB(STATRST)) != SWAB(RESETDONE)) {
		if (++l == 50000) {
			printf("rf: timeout\n");
			return (-1);
		} else
			DELAY(400);
	}
	return (0);
}

rfbusy(ctlr, rfaddr, wait)
RF32REG 	*rfaddr;
int		ctlr;	
{
	register int 		l = 0;
	int			s;

	if (wait) {
		DELAY(400);
		while ((rfaddr->status & SWAB(STATBSY)) !=
				SWAB(RF_stat[ctlr].rf_bsybit)) {
			if (++l == 50000) {
				printf("rf: timeout\n");
				return (-1);
			} else
				DELAY(400);
		}
		RF_stat[ctlr].rf_bsybit ^= 1;
		return (0);
	}

	if ((rfaddr->status & SWAB(STATBSY)) !=
				SWAB(RF_stat[ctlr].rf_bsybit))
		panic("rfbusy: controller busy");
	RF_stat[ctlr].rf_bsybit ^= 1;
	return 0;
}

rfsplx(ctlr)
int	ctlr;
{
	return (splx(rfcinfo[ctlr]->qm_psl));
}


/*
 * Wake up every second and if an interrupt is pending but nothing has happened
 * increment a counter. If nothing happens for 20 seconds, reset the controller
 * and begin anew.
 */
rfwatch()
{
	register struct qb_ctlr		*qm;
	register			ctlr;
	register struct RF_stat		*stat;

	timeout(rfwatch, (caddr_t)0, RF_TIMEOUT);
	for (ctlr = 0; ctlr < NRF; ctlr++) {
		qm = rfcinfo[ctlr];
		if (qm == 0 || qm->qm_alive == 0 || qm->qm_tab.b_active == 0)
			continue;
		stat = &RF_stat[ctlr];
		stat->rf_wticks++;
		if (stat->rf_wticks >= 20) {
			stat->rf_wticks = 0;
			printf("RF%d: lost interrupt\n", ctlr);
			rfreset();
		}
	}
}

rfminphys(bp)
struct buf	*bp;
{
	bp->b_bcount = MIN(bp->b_bcount, RF_MAXCNT);
}

int rfdumpstrat();
extern struct buf dumpbuf;
extern int dumpsize;

rfdump(dev)
	dev_t dev;
{
	register int unit = minor(dev) >> 3;
	register int partition = minor(dev) & 07;
	register struct buf *bp = &dumpbuf;
	register struct qb_device *qi;
	register struct qb_diskst *st;
	int num, start, bcount;

	if (unit >= Nrf || (qi = rfdinfo[unit]) == 0 || qi->qi_alive == 0)
		return ENXIO;
	num = ctod(dumpsize);		/* memory size in disk blocks */
	start = 0x0;			/* start dumping at physical 0 */
	st = &rfst[unit];
	if (dumplo < 0 || dumplo+num > ST_size[partition].nblocks)
		return EINVAL;
	while (num > 0) {
		bcount = (dbtob(num) > RF_MAXCNT) ? RF_MAXCNT : dbtob(num);
		bp->b_dev = dev;
		bp->b_blkno = dumplo + btodb(start);
		bp->b_bcount = bcount;
		bp->b_un.b_addr = (caddr_t) start;
		bp->b_flags = B_WRITE; 
		badstrat(qi, bp, &rfbad[qi->qi_unit], rfdumpstrat,
		    ST_nspt, ST_ntpc, ST_ncpd, ST_size[partition].cyloff);
		if (bp->b_flags & B_ERROR || bp->b_resid)
			return EIO;
		start += bcount;
		num -= btodb(bcount);
	}
	return 0;
}

rfdumpstrat(qi, bp)
	register struct qb_device *qi;
	register struct buf *bp;
{
	register struct RF_stat		*stat = &RF_stat[qi->qi_ctlr];
	register struct qb_diskst	*st = &rfst[qi->qi_unit];
	register int			drive = qi->qi_slave & 0x3;
	register int			errcount;
	int				i;
	register RF32REG		*rfaddr =
			(RF32REG *)((char *) qi->qi_mi->qm_addr - RFOFFSET);
	register STATBLK		*sb = (STATBLK *)stat->rf_status;

	errcount = 0;
	while (1) {
		stat->rf_bn = rfblock(bp) + 
		    (ST_size[minor(bp->b_dev)&07].cyloff*ST_nspc) +
			ST_nspc;
		stat->rf_addr = (u_int) bp->b_un.b_addr;
		stat->rf_nblks = (bp->b_bcount+511) >> 9;
		stat->rf_drive = drive;
		stat->rf_cmd = (bp->b_flags & B_READ) ? C_READ : C_WRITE;
		rfio(qi->qi_ctlr, rfaddr, stat, 0);
		if (sb->status & ST_ERR) {
			printf("rf%d: sn%d HARD %s err: %x, stat:%b\n",
				drive, i - ST_nspc, "WRITE", 
				sb->errcode, sb->status, ST_BITS);
			errcount++;
			if (++errcount == RF_RETRYCNT) {
				bp->b_flags |= B_ERROR;
				bp->b_resid = bp->b_bcount;
				return EIO;
			}
		} else {
			bp->b_resid = 0;
			break;
		}
	}
	bp->b_flags |= B_DONE;
	return 0;
}

#endif
