/*
 * QBUS: IEE488 driver
 */
#include "ib.h"

#if	NIB > 0

#include "param.h"
#include "systm.h"
#include "ioctl.h"
#include "tty.h"
#include "buf.h"
#include "conf.h"
#include "user.h"
#include "map.h"
#include "vm.h"
#include "cmap.h"
#include "uio.h"
#include "kernel.h"

#include "../is68kdev/qbvar.h"

struct buf      ibbuf;

int	ibprobe(), ibslave(), ibattach(), ibintr(), ibstrat(), ibminphys();
struct	qb_ctlr		*ibcinfo[NIB];
struct	qb_device	*ibdinfo[Nib];

/* 
 * IB controller addresses 
 */
u_short *IBstd[] = { (u_short *)0x3FEFC8,
		     (u_short *)0x3FEFD0,
		     (u_short *)0x3FEFD8,
		     (u_short *)0x3FEFE0, 0 };  /* apparently, 0 terminates */

int ibaddrs[] = {	0x3FEFC8,
			0x3FEFD0,
			0x3FEFD8,
			0x3FEFE0 };

/*
 * IB/ib controller/device driver structure
 */
struct	qb_driver IBdriver =
	{ ibprobe, ibslave, ibattach, IBstd, "ib", ibdinfo, "IB", ibcinfo };

/*
	GPIB driver software for National Instruments GPIB11V-2 
	controller.  (part of Unix kernal)
 
 	Copyright 1980 National Instruments
 	REV C:  01/01/82
 
 	16mar82- Changed HZ to Hz; defined integ,psword.  These changes
 	were necessary to make the software compatible with
 	Version 7 Xenix. 

 	05may82- Fixed bug in ibstrategy.  Changed union reference from
 	bp->b_addr to bp->b_un.b_addr. (error in Nat'l software)
 	Proper data is now sent out on the bus. 

	13aug82- Fixed timeout problem (was crashing system on timeout).
 	Had to make 'ibtimer()' an external function. 

	26jul83- Modified to work on System-III Unix and Integrated
	Solutions 68000 processor board.  Byte swapping was the big
	problem.
*/


#define IB      ((struct ib *) ibaddrs[unit]) /* Q-bus address */
#define Ib	ib[unit]
#define SIGSRQ  SIGINT

/* discard the next line if physio supports odd base and count */
int iboddc, ibodda;

char *base;

#define PPENAB  0       /* Leave PPENAB undefined to eliminate PP code */
#define TRI     0       /* 4 for Three-state highspeed timing; 0 for Normal timing */
#define SAC     0       /* Leave SAC undefined if the GPIB11V-2 is not the System Controller */
#define EXT     1       /* 2 for Extended; 1 for Normal GPIB addressing */
#define MSA     0140    /* Msa&037 for Extended; 0140 for Normal GPIB addressing */
#define MA      025     /* GPIB address */
#define ONLYC   0       /* ONLYC may be defined (as anything) if there is but a */
/* single Controller on the GPIBus.  ONLYC implies SAC. */

#define Exclude sps=spl5()
#define Unexclude splx(sps)

#define INACTIVE   0            /* Software controller states */
#define IDLE       1
#define INCHARGE   2
#define STANDBY    3

#define GO      1               /* Control/status register bits */
#define OUT     2
#define SEL     4
#define XCC     010
#define IE      0100
#define LMR     0200
#define CIC     0400
#define ATN     01000
#define EOI     02000
#define OCSW    02000
#define TCS     04000
#define DAV     04000
#define SRQ_IE  010000
#define SRQ     010000
#define INT     020000
#define DMA_ENAB 020000
#define NEXX     040000
#define REN     040000
#define IFC     0100000

#define DIR     0               /* Internal register addresses */
#define DOR     0
#define ISR1    1
#define IMR1    1
#define ISR2    2
#define IMR2    2
#define SPS     3
#define SPM     3
#define ADS     4
#define ADM     4
#define CPT     5
#define AUX     5
#define AD0     6
#define ADR     6
#define AD1     7
#define EOS     7

#define ERR_IE      4           /* Internal register bits */
#define END_IE    020
#define CPT_IE   0200
#define DMAI_ENAB 020
#define DMAO_ENAB 040
#define SRQS     0100
#define RSV      0100
#define TA          2
#define LA          4
#define LON      0100
#define TON      0200
#define DL        040
#define DT       0100
#define ARS      0200

#define _CLKR     040           /* Hidden register addresses & offsets */
#define _PPR     0140
#define _AUXA    0200
#define _AUXB    0240
#define CLKR        0
#define PPR         1
#define AUXA        2
#define AUXB        3

#define U         020           /* Hidden register bits */
#define BIN       020
#define S         010
#define REOS        4
#define HLDE        2
#define HLDA        1
#define CPT_ENAB    1
#define PACS        1           /* Software status bits */
#define MON         2

#define IST     011             /* Special interface functions */
#define NIST      1
#define VSC     017
#define NVSC      7
#define SEOI      6
#define FH        3
#define IR        2
#define PON       0

#define OK         1    /* Error codes */
#define ENONE     -1    /* No command byte available (READCOMMAND) */
#define ECACFLT   -2    /* ATN not unasserted after IFC sent (bus problem) */
#define ENOTCAC   -3    /* Not Active Controller for operation requiring CAC (software problem) */
#define ENOTSAC   -4    /* Not System Controller for operation requiring SAC (software problem) */
#define EIFCLR    -5    /* IFC caused operation to abort (bus problem) */
#define ETIMO     -6    /* Operation did not complete within allotted time (bus problem) */
#define ENOFUN    -7    /* Non-existent function code (software problem) */
#define ETCTIMO   -8    /* Take control not completed within allotted time (bus problem) */
#define ENOIBDEV  -9    /* No Listeners addressed or no devices connected (bus problem) */
#define EIDMACNT -10    /* Internal DMA completed without bcr going to 0 (hardware problem) */
#define ENOPP    -11    /* PP operation attempted on three-state GPIB (software problem) */
#define EITIMO   -12    /* Internal DMA did not complete within allotted time (hardware problem) */
#define EINEXM   -13    /* Internal DMA aborted due to non-existent memory (software/hardware problem) */
#define ENEXMEM  -14    /* GPIB DMA aborted due to non-existent memory (software/hardware problem) */
#define ECNTRS   -15    /* Bar and bcr are inconsistent following GPIB DMA (hardware problem) */

#define RQC_STB  (RSV | 1)      /* Service request status bytes */
#define RQT_STB  (RSV | 2)
#define RQL_STB  (RSV | 4)

#define RQC     1               /* Asynchronous op codes */
#define CAC     2
#define TAC     3
#define LAC     4
#define CWT     5

#define TCT     011             /* GPIB multiline messages */
#define PPC     5
#define PPU     025
#define SCG     0140

#define IN      0
#define ITIMO   25              /* Internal loopcount timeout */
#define GTIMO   2		/* Default GPIB timeout in seconds */
#define TCTIMO  100
#define HUMSEC	100		/* 100 usec loop count */
#define MONHWAT 32

#define RD      (DMA_ENAB|TCS|IN|GO)
#define WT      (DMA_ENAB|TCS|OUT|GO)
#define RDIR    (DMA_ENAB|IN|SEL|GO)
#define WTIR    (DMA_ENAB|OUT|SEL|GO)

#define Try(f)  if((x=(f))<0) return x


struct  ib {
	short   bcr, bar, csr;  /* Q-bus device registers               */
	char    xba, ccf;
	char    internal[8];    /* Internal registers                   */
	char    hidden[4];      /* Hidden registers                     */

	char    cstate;
	char    istr, op;
	int     ans;
	int     timo;           /* Watchdog timer- secs. */
	struct proc *owner;     /* GPIB owning process                  */
	int     rdbcr, rdbar, rdcsr;
	char    rdinternal[8];
#ifndef ONLYC
	struct clist monq;
#endif
	short	arg[3];
} 
ib[NIB];
#define	ibdevice	ib

struct Ureg { 
	short   ureg[4]; 
};

static int command(), transfer(), clear(), remote(), local(), ppoll();
static int passctrl(), setstat(), monitor(), readcmd(), setparam(), testsrq();

int (*ibfn[])() = {
	command, transfer, clear, remote, local, ppoll,
	passctrl, setstat, monitor, readcmd, setparam, testsrq };

#define NFNS    ((sizeof ibfn)/(sizeof ibfn[0]))

static char uswbuf[512], xswbuf[8];

/************************************************************************/
/*	start of the driver code...	*/
/************************************************************************/

/*
 * Determine if there is a controller for an ib at address reg.  Our goal is to 
 * make the device interrupt.
 */
ibprobe(ibaddr)
register struct ibdevice *ibaddr;
{
	ibaddr->csr = LMR;
	ibaddr->bcr = -1;
	ibaddr->bar = 0;

	ibaddr->csr = DMA_ENAB | SEL | OUT | IE | GO;
	DELAY(10000);

	ibaddr->csr = LMR;
	return 8;
}

ibslave(qi, ibaddr)
struct qb_device *qi;
struct ibdevice *ibaddr;
{
	if (qi->qi_slave != 0)
		return 0;
	return (1);
}

ibattach(qi)
register struct qb_device *qi;
{
}

ibopen(dev,rw){
	register int sps, unit;
	int err_stat = 0;
	
	unit = minor(dev);
	Exclude;
	if(Ib.owner){ 
		err_stat = ENXIO;
		goto out; 
	}
	Ib.owner= u.u_procp;
	if ((Ib.ans= init(unit)) < 0) {
		Ib.owner= 0;
		err_stat = EIO;
	}
out:    
	Unexclude; 
	return err_stat;
}

/************************************************************************/
ibclose(dev){
	register int sps, unit;

	unit = minor(dev);
	Exclude;
	Ib.cstate= INACTIVE;
	Ib.csr= 0;
	IB->csr= LMR;
	Ib.owner= 0;
	Unexclude; 
	return 0;
}

/************************************************************************/
ibioctl(dev, cmd, av, mode)
dev_t dev;
int cmd;
short av[];
int mode;
{
	register int sps, x, unit;
	int err_stat = 0;

	unit = minor(dev);
	Exclude;
	if(cmd==TIOCGETP){
	av[0] = Ib.arg[0]= Ib.ans;
		av[1] = Ib.arg[1]= IB->csr;
		av[2] = Ib.arg[2]= IB->bcr;
	}
	else {  
		Ib.arg[0] = av[0];
		Ib.arg[1] = av[1];
		Ib.arg[2] = av[2];
		if((x=Ib.arg[0])<0 || x>=NFNS){
			Ib.ans= ENOFUN; 
			err_stat = EIO; 
		}
		else if((Ib.ans= (*ibfn[x])(unit))<0) 
			err_stat = EIO;
	}
	Unexclude; 
	return err_stat;
}

/************************************************************************/
static command(unit)
{
	Ib.op= CWT;
	return OK; 
}

/************************************************************************/
static transfer(unit)
{
	return gts(unit,EXT); 
}

/************************************************************************/
static clear(unit)
{
	register int x;
	Try(unhold(unit));
#ifdef SAC
	IB->csr= Ib.csr | IFC;
	wait100us();
	if(IB->csr&ATN) return ECACFLT;
	Ib.cstate= INCHARGE;
	IB->csr= (Ib.csr|= ATN | SRQ_IE | CIC);
	return lun(unit,TON);
#else
	return ENOTSAC;
#endif
}

/************************************************************************/
static remote(unit)
{
#ifdef SAC
	IB->csr= (Ib.csr|= REN);
	return OK;
#else
	return ENOTSAC;
#endif
}

/************************************************************************/
static local(unit)
{
#ifdef SAC
	IB->csr= (Ib.csr&= ~REN);
	return OK;
#else
	return ENOTSAC;
#endif
}

/************************************************************************/
static ppoll(unit)
{
	register int x;
#ifdef PPENAB
	Try(tcs(unit));
	Try(lun(unit,LON));
	IB->csr= (Ib.csr|= EOI);
	Try(xfer(unit, RDIR, &Ib.rdinternal[CPT], 1, CPT));
	IB->csr= (Ib.csr&= ~EOI);
	return (Ib.rdinternal[CPT]&0377|0400);
#else
	return ENOPP;
#endif
}

/************************************************************************/
static passctrl(unit)
{
	register int x;
#ifdef ONLYC
	return ENOIBDEV;
#else
	if(Ib.cstate!=INCHARGE) return ENOTCAC;
	Try(lun(unit,EXT));
	IB->csr= (Ib.csr&= ~(SRQ_IE | ATN | CIC));
	Ib.cstate= IDLE;
	return OK;
#endif
}

/************************************************************************/
static setstat(unit)
{
#ifndef ONLYC
#ifdef PPENAB
	if((Ib.istr&S) && Ib.arg[1]==0){
		Ib.istr&= ~S; 
		return ldaux(unit,NIST); 
	}
	if((Ib.istr&S)==0 && Ib.arg[1]){
		Ib.istr|= S; 
		return ldaux(unit,IST); 
	}
#endif
#endif
	return OK; 
}

/************************************************************************/
static monitor(unit)
{
#ifndef ONLYC
	if(Ib.arg[1]) Ib.istr|= MON;
	else Ib.istr&= ~MON;
#endif
	return OK; 
}

/************************************************************************/
static readcmd(unit)
{
	register x;
#ifndef ONLYC
	if((x=getc(&Ib.monq))>=0) return (x|0400);
#endif
	return ENONE; 
}

/************************************************************************/
static setparam(unit)
{
	Ib.timo= Ib.arg[1];
	Ib.internal[0]= Ib.arg[2];
	Ib.internal[EOS]= Ib.arg[2]>>8;
	return OK; 
}

/************************************************************************/
static testsrq(unit)
{
	if(IB->csr&SRQ) return OK;
	if(Ib.csr&CIC) IB->csr= (Ib.csr|= SRQ_IE);
	return ENONE; 
}

/************************************************************************/
static init(unit)
{
	register char *cp;
	register int x;

	Ib.csr= 0; 
	IB->csr= LMR;
	Ib.ccf= 0;
	Ib.xba= 0;
	cp= &Ib.hidden[0];
	*cp++= _CLKR | 5;
	*cp++= _PPR | U;
	*cp++= _AUXA;
	*cp++= _AUXB | TRI | CPT_ENAB;

	cp= &Ib.internal[0];
	*cp++= 0;
	*cp++= CPT_IE | ERR_IE;
	*cp++= 0;
	*cp++= 0;
	*cp++= EXT;
	*cp++= Ib.hidden[CLKR];
	*cp++= ARS | MSA;
	*cp++= 0;

	Ib.istr= 0;
	Ib.op= 0;
	Ib.ans= 0;
	Ib.timo= GTIMO;
	Try(irload(unit));
	Ib.cstate= IDLE;
	IB->csr= Ib.csr= IE;
#ifndef ONLYC
	while(getc(&Ib.monq)>=0) ;
#endif
	return 0; 
}

/************************************************************************/
static ibstop(unit)
{
	register int x;

	IB->csr= (Ib.csr&= ~(DMA_ENAB|GO));
	Ib.op= 0;
	iodone(&ibbuf);
#ifndef ONLYC
	Try(rqs(0));
#endif
	return ETIMO; 
}

/************************************************************************/
static irload(unit)
{
	register int x;
	Try(xfer(unit,WTIR,&Ib.internal[ISR1],7,ISR1));
	Ib.internal[AUX]= Ib.hidden[AUXA];
	Ib.internal[ADR]= MA;
	Try(xfer(unit,WTIR,&Ib.internal[AUX],2,AUX));
	Try(xfer(unit,WTIR,&Ib.hidden[AUXB],1,AUX));
	x= Ib.internal[ADM];
	Ib.internal[ADM]= 0;
	return lun(unit,x); 
}

/************************************************************************/
static lun(unit,newadm) char newadm;
{        
	/* Note: rsv is cleared and not restored*/
	register int x;

	if(Ib.internal[ADM]==newadm) return OK;
	Ib.internal[ADM]= newadm;
	Ib.internal[AUX]= PON;
	Try(xfer(unit,WTIR,&Ib.internal[ADM],2,ADM));
	Try(xfer(unit,WTIR,&Ib.hidden[PPR],1,AUX));
	return (Ib.istr&S)? ldaux(unit,IST): OK; 
}

/************************************************************************/
ibwrite(dev, uio)
dev_t	dev;
struct uio *uio;
{
	register int sps, unit;
	int ibstrategy();
	int err_stat;

	unit = minor(dev);
	Exclude;
	if((Ib.ans=wsetup(unit,uio))>=0)
		err_stat = physio(ibstrategy, &ibbuf, dev, B_WRITE, ibminphys, uio);
	Ib.op= 0;
	if(Ib.ans>=0) uio->uio_resid= -ibbuf.b_resid;	/* ??? */
	else err_stat = EIO;
	Unexclude; 
	return err_stat;
}

/************************************************************************/
static wsetup(unit,uio)
struct uio *uio;
{
	register x;
	Ib.ccf= 0;
	Ib.csr&= ~XCC;
	if(Ib.op==CWT)
		if((x=tcs(unit))<0){
#ifndef ONLYC
			if(uio->uio_resid==0 || x!=ENOTCAC) return x;
			Try(rqs(unit,RQC_STB));
			Ib.op= RQC;
#else
			return x;
#endif
		}
		else {  
			Try(lun(unit,TON));
			Ib.op= CAC; 
		}
	else {  
		if((x=gts(unit,TON))<0){
#ifndef ONLYC
			if(uio->uio_resid==0 || x!=ENOTCAC) return x;
			Try(rdads(unit));
			if((Ib.rdinternal[ADS]&TA)==0)
				Try(rqs(unit,RQT_STB));
#else
			return x;
#endif
		}
		if(Ib.internal[0]&2){
			Ib.ccf= SEOI; 
			Ib.csr|= XCC; 
		}
		Ib.op= TAC; 
	}
	/* discard the next 4 lines if physio supports odd base and count */
	iboddc= uio->uio_iov->iov_len&1;
	uio->uio_iov->iov_len+= iboddc;
	ibodda= (int)uio->uio_iov->iov_base & 1;
	base = uio->uio_iov->iov_base=(caddr_t)((int)(uio->uio_iov->iov_base)& ~1);
	return OK; 
}

/************************************************************************/
ibread(dev, uio)
dev_t	dev;
struct uio *uio;
{
	register int	sps, unit;
	int err_stat;
	register char	*p;
	register int	i,n;
	int ibstrategy();

	unit = minor(dev);
	Exclude;
	if((Ib.ans=rsetup(unit,uio))>=0) {
		/*
		 * The following swap assumes that this code will not in fact
		 *  ever get executed at the interrupt level.
		 *
		 *
		 *  4.2 mod: need to grab return address before calling 
		 *  physio, since physio destroys it.
		 */
		p= (char *)uio->uio_iov->iov_base;
		err_stat = physio(ibstrategy, &ibbuf, dev, B_READ, ibminphys, uio);

		/* physio terminates, but count & data are wrong? */
		n= ibbuf.b_bcount-iboddc;
		for (i=0;i<n;i++) subyte(p++, uswbuf[(i&1)?(i&~1):(i|1)]);
	}
	Ib.op= 0;
	if(Ib.ans>=0) uio->uio_resid= -ibbuf.b_resid;	/* ??? */
	else err_stat = EIO;
	Unexclude; 
	return err_stat;
}

/************************************************************************/
static rsetup(unit,uio)
struct uio *uio;
int unit;
{
	register int x;
	Ib.ccf= 0;
	Ib.csr&= ~XCC;
	if((x=gts(unit,LON))<0){
#ifndef ONLYC
		if(uio->uio_resid==0 || x!=ENOTCAC) return x;
		Try(rdads(unit));
		if((Ib.rdinternal[ADS]&LA)==0)
			Try(rqs(unit,RQL_STB));
#else
		return x;
#endif
	}
	Ib.hidden[AUXA]|= (Ib.internal[0]&(2<<4)? REOS:0) | (Ib.internal[0]&(1<<4)? BIN:0);
	if(Ib.internal[0]&(4<<4)) Ib.internal[IMR1]&= ~END_IE;
	else {  
		Ib.internal[IMR1]|= END_IE;
		if(Ib.cstate==STANDBY) Ib.hidden[AUXA]|= HLDE; 
	}
	Ib.internal[AUX]= Ib.hidden[AUXA];
	Try(xfer(unit,WTIR, &Ib.internal[IMR1], 7, IMR1));
	if(Ib.cstate==STANDBY){
		Ib.ccf= Ib.hidden[AUXA]= Ib.hidden[AUXA]&~HLDE|HLDA;
		Ib.csr|= XCC; 
	}
	Ib.op= LAC;
	/* discard the next 4 lines if physio supports odd base and count */
	iboddc= uio->uio_iov->iov_len&1;
	uio->uio_iov->iov_len+= iboddc;
	ibodda= (int)uio->uio_iov->iov_base & 1;
	base = uio->uio_iov->iov_base=(caddr_t)((int)(uio->uio_iov->iov_base)& ~1);
	return OK; 
}

/************************************************************************/
ibstrategy(bp) register struct buf *bp; 
{
	register int x, rw, unit;
	paddr_t addr;

	/* if physio supports odd base and count replace the next line with:  
	addr = paddr(bp);  */

	unit = minor(bp->b_dev);
	addr = bp->b_un.b_addr + ibodda;
	rw= bp->b_flags&B_READ? RD:(Ib.op==RQC? OUT:WT);
	/* if physio supports odd base and count replace the next line with:
	if((x=xfer(unit, rw, bp->b_un.b_addr, bp->b_bcount, 0))<0)  */

	if((x=xfer(unit, rw, loword(addr), bp->b_bcount-iboddc, 0))<0)
		{
		Ib.ans= x;
		iodone(bp);
		}       
}

/************************************************************************/
#ifndef ONLYC
static rdads(unit)
{
	return xfer(unit, RDIR, &Ib.rdinternal[ADS], 1,ADS); 
}

/************************************************************************/
static rqs(unit,r) char r;
{
	if(Ib.internal[SPM]==r) return OK;
	Ib.internal[SPM]= r;
	return xfer(unit ,WTIR, &Ib.internal[SPM], 1,SPM); 
}

/********************************************:***************************/
static valid(unit)
{
	return ldaux(unit,VSC); 
}
#endif

/************************************************************************/
static gts(unit,newadm) char newadm;
{
	register int x;
	Try(unhold(unit));
	if(Ib.cstate==STANDBY)
		return (Ib.internal[ADM]==newadm)? OK : ENOIBDEV;
	if(Ib.cstate==IDLE) return ENOTCAC;
	Try(lun(unit,newadm));
	IB->csr= (Ib.csr&= ~ATN);
	Ib.cstate= STANDBY;
	return OK; 
}

/************************************************************************/
static tcs(unit)
{
	if(Ib.cstate==INCHARGE) return OK;
	if(Ib.cstate==IDLE) return ENOTCAC;
	IB->csr= (Ib.csr|= ATN);
	Ib.cstate= INCHARGE;
	return unhold(unit); 
}

/************************************************************************/
static unhold(unit)
{
	register int x;
	if(Ib.hidden[AUXA]&(HLDE|HLDA)){
		Ib.hidden[AUXA]= _AUXA;
		Try(ldaux(unit,FH));
		return xfer(unit, WTIR, &Ib.hidden[AUXA], 1, AUX); 
	}
	return OK; 
}

/************************************************************************/
xfer(unit,rw,bp,n,fr)
register int unit;
{       /* fr is internal reg addr  */
	register int i, x;
	int ibtimer();
	register char *p;
	paddr_t addr;
	long svtop();

	if(n<=0) return OK;
	if(rw&SEL){
		/* read/write internal registers */
		/*
			Copy the data pointed to by bp into xswbuf,
			swapping each pair of bytes as we go.
		*/
		p = (char *)bp;
		if (rw == WTIR)
			for(i=0; i<n; i++)
				xswbuf[(i&1)?(i&~1):(i|1)] = *p++;
		addr = (paddr_t)svtop((caddr_t)&xswbuf[0]);
		IB->bcr= (-n<<8) | fr & 7;
		IB->bar= loword(addr);
		((struct Ureg *)IB)->ureg[3]= hiword(addr) << 8; /* xba */
		IB->csr= Ib.csr & (REN|SRQ_IE|EOI|ATN|CIC) | rw;
		for(i=ITIMO; !((x=IB->csr)&DONE); )
			if(--i<=0) return EITIMO;
		if(x&NEXX) return EINEXM;
		if(IB->bcr&0177400) return EIDMACNT;
		IB->csr= Ib.csr & (REN|SRQ_IE|ATN|CIC|IE);
		return n; 
	}
	Ib.internal[IMR2]= rw&OUT? DMAO_ENAB:DMAI_ENAB;
	Try(xfer(unit,WTIR,&Ib.internal[IMR2],1,IMR2));
	/*
		The following 3 lines assume that this code will not in fact
		ever get executed at the interrupt level.
	*/
	p = (char *)base;
	if (rw==WT)
		for (i=0;i<n;i++) uswbuf[(i&1)?(i&~1):(i|1)] = fubyte(p++);
	addr = (paddr_t)svtop((caddr_t)&uswbuf[0]);
	IB->bcr= Ib.bcr= -n;
	IB->bar= Ib.bar= loword(addr);
	Ib.xba= hiword(addr);
	((struct Ureg *)IB)->ureg[3]= ((struct Ureg *)&Ib)->ureg[3];
	IB->csr= Ib.csr= Ib.csr & (REN|SRQ_IE|TCS|ATN|CIC|XCC) | IE | rw;
	Ib.ans= 0;
	if(Ib.timo) {
		timeout(ibtimer,unit,Ib.timo*hz);
	}
	return OK; 
}

/************************************************************************/
static ldaux(unit,a)
{
	Ib.internal[AUX]= a;
	return xfer(unit, WTIR, &Ib.internal[AUX], 1, AUX); 
}

/************************************************************************/
/*
	ibtimer- This routine is called by the Unix clock interrupt
		routine when its address is part of the 'callout'
		structure and a timeout has occurred.
		The Unix system call 'timeout' stores addresses of
		routines like ibtimer in the callout structure, and
		tells Unix how soon (in secs.) the routine is to
		be called.  
*/
ibtimer(unit){
	int sps;
	Exclude;
	Ib.ans= ibstop(unit);
	Unexclude; 
}

/************************************************************************/
ibintr(unit)
{
	register int x, i;

	Ib.rdbcr= IB->bcr;
	Ib.rdbar= IB->bar;
	Ib.rdcsr= IB->csr;
#ifndef SAC
	if(IB->csr<0){
		IB->csr= (Ib.csr&= ~(DMA_ENAB|SRQ_IE|TCS|ATN|CIC|GO));
		if(Ib.cstate!=INACTIVE){
			Ib.cstate= IDLE;
			Ib.hidden[AUXA]= _AUXA;
			Ib.internal[AUX]= FH;
			Ib.internal[ADM]= EXT;
			Ib.internal[IMR2]= 0;
			Ib.internal[SPM]= 0;
			if((Ib.ans=irload(unit))>=0) Ib.ans= EIFCLR;
			Ib.op= 0;
			wakeup(&Ib); 
		}
		return; 
	}
#endif
	if((IB->csr&SRQ) && (Ib.csr&SRQ_IE)){
		IB->csr= (Ib.csr&= ~SRQ_IE) & ~GO;
		if(Ib.owner) psignal(Ib.owner,SIGSRQ); 
	}
	if(IB->csr&INT){
		IB->csr= Ib.csr & ~(DMA_ENAB|GO);
		if((x=xfer(unit, RDIR, &Ib.rdinternal[ISR1], 1, ISR1))<0) goto quit;
		if((x=xfer(unit, RDIR, &Ib.rdinternal[ISR1+1],1,ISR1+1))<0) goto quit;
		if((x=xfer(unit, RDIR, &Ib.rdinternal[ISR1+2],1,ISR1+2))<0) goto quit;
		if((x=xfer(unit, RDIR, &Ib.rdinternal[ISR1+3],1,ISR1+3))<0) goto quit;
		if((x=xfer(unit, RDIR, &Ib.rdinternal[ISR1+4],1,ISR1+4))<0) goto quit;
		if(Ib.rdinternal[ISR1]&ERR_IE) Ib.ans= ENOIBDEV;
#ifndef ONLYC
		else if(Ib.rdinternal[IMR1]&CPT_IE){
			if((Ib.istr&MON) && Ib.monq.c_cc<MONHWAT)
				putc(Ib.rdinternal[CPT], &Ib.monq);
#ifdef PPENAB
			if((Ib.istr&PACS) && Ib.rdinternal[CPT]>=SCG){
				Ib.istr&= ~PACS;
				Ib.hidden[PPR]= Ib.rdinternal[CPT];
				goto ldlpe; 
			}
#endif
			switch(Ib.rdinternal[CPT]){
			case TCT:
				if((Ib.rdinternal[ADS]&TA)==0) break;
				if((x=valid(unit))<0) goto quit;
				for(i=TCTIMO; IB->csr&ATN; i--)
					if(i==0){ 
						x= ETCTIMO; 
						goto quit; 
					}
				IB->csr= (Ib.csr|= SRQ_IE|ATN|CIC) & ~GO;
				Ib.cstate= INCHARGE;
				break;
#ifdef PPENAB
			case PPC:
				if(Ib.rdinternal[ADS]&LA) Ib.istr|= PACS;
				break;
			case PPU:
				Ib.hidden[PPR]= _PPR | U;
ldlpe:  
				if((IB->csr&OCSW)==0) break;
				if((x=xfer(unit,WTIR,&Ib.hidden[PPR],1,AUX))<0)
					goto quit;
				break;
#endif
			}
			if((x=valid(unit))<0) goto quit;
		}
#endif
		if(Ib.internal[IMR1]&END_IE){
			Ib.internal[IMR1]&= ~END_IE;
			if((x=xfer(unit,WTIR,&Ib.internal[IMR1],1,IMR1))<0) goto quit;
		}       
	}
	if((IB->csr&DONE) && (Ib.csr&GO)){
		Ib.csr&= ~GO;
		if(Ib.timo) untimeout(ibtimer,unit);   /* Is really unit??? */
		if(Ib.rdcsr&NEXX) Ib.ans= ENEXMEM;
#ifndef ONLYC
		if((x=rqs(unit,0))<0) goto quit;
#endif
		x= Ib.rdbcr - Ib.bcr;
		if(Ib.rdbar - Ib.bar != x) x= ECNTRS;
		ibbuf.b_resid= Ib.rdbcr;
		if(Ib.ans==0)
quit:                   
			Ib.ans= x;
		Ib.op= 0;
		iodone(&ibbuf); 
	}
#ifndef ONLYC
	else if)Ib.op==RQC && (IB->csr&CIC)){
		if((x=lun(unit,TON))<0) goto quit;
		Ib.op= CAC;
		IB->csr= Ib.csr|= WT;
		return; 
	}
#endif
}

/************************************************************************/
static wait100us()
{
	register int i;
	for(i=HUMSEC; i-->0; ) ; 
}

/************************************************************************/
ibminphys(bp)
struct buf *bp;
{
	bp->b_bcount = MIN(bp->b_bcount, 512);
}
#endif	NIB > 0
