/*
 * VMEBUS: Integrated Solutions GIP graphics controler
 * 	   Integrated Solutions Monochrome WorkStation (MWS) card
 */

#include "gp.h"

#if	NGP > 0
#include "../machine/pte.h"

#include "param.h"
#include "systm.h"
#include "user.h"
#include "kernel.h"
#include "proc.h"
#include "dk.h"
#include "dkbad.h"
#include "buf.h"
#include "conf.h"
#include "map.h"
#include "vm.h"
#include "cmap.h"
#include "uio.h"
#include "../vt/vt_hdrs.h"
#include "../vt/vt_output.h"
#include "../vt/vt_kernel.h"
#include "../vt/vt_key.h"

#include "../machine/board.h"
#include "../machine/psl.h"
#include "../is68kdev/qbvar.h"
#include "../is68kdev/gpreg.h"

/*
 * possible graphics processor address tuples
 */
struct gp_addr_tuple {
	ushort		*dm;		/* display memory */
	struct gpdevice	*addr;		/* gip address */
} gp_addrs[] = {
	{(ushort *)&vme_std[0xE00000],(struct gpdevice *)&vme_stdio[0xFFC000]},
	{(ushort *)&vme_std[0xD00000],(struct gpdevice *)&vme_stdio[0xFF8000]},
	{(ushort *)&vme_std[0xC00000],(struct gpdevice *)&vme_stdio[0xFF4000]},
	{(ushort *)&vme_std[0xB00000],(struct gpdevice *)&vme_stdio[0xFF0000]},
	{(ushort *)&vme_std[0xA00000],(struct gpdevice *)&vme_stdio[0xFEC000]},
	{(ushort *)&vme_std[0x900000],(struct gpdevice *)&vme_stdio[0xFE8000]},
	{(ushort *)&vme_std[0x800000],(struct gpdevice *)&vme_stdio[0xFE4000]},
	{(ushort *)&vme_std[0x700000],(struct gpdevice *)&vme_stdio[0xFE0000]},
	{(ushort *)0, (struct gpdevice *)0}	/* null terminated */
};

int	gpprobe(),	gpslave(),	gpattach(),	gpintr();
struct	qb_ctlr		*gpcinfo[NGP];
struct	qb_device	*gpdinfo[Ngp];

u_short *GPstd[NGP+1];			/* gpdevice addresses */
u_char	*GPstd_dm[NGP + 1];		/* display memory addresses */
struct	qb_driver GPdriver =
	{ gpprobe, gpslave, gpattach, GPstd, "gp", gpdinfo, "GP", gpcinfo };

extern int		rrtimeslice;	/* nth of sec round robin time slice */
extern			flushvtecho();
extern int		gp_probe;	/* gp being probed */
extern unsigned short	vidbuf[];	/* virtual address for display memory */
extern char		flushvtok;

unsigned char	gptype = GPHASGIP|GPCANINT; /* default configuration */
					/* remove GPHASGIP, treat GIP as MWS */
char		*vtpool;
short		GPflevel = 1;		/* amount of FIFO currently used */

struct proc	*GPowner = 0;		/* process which controls GIP */
short		GPwanted = 0;		/* someone wants GIP ownership */
char		GPu_comm[MAXCOMLEN + 1];/* name of GIP owner */
int		GPdaemonpid;		/* pid of gpdaemon */
int		GPvector;		/* interrupt vector for MWS */

#define	gppanic	panic		/**/

/*
 * This routine is called early in kernel start to set up the graphics system:
 *
 *	o	determine existance of gip. 
 *	o	define gpaddr to point to the gip device registers.
 *	o	map the display memory into kernel address space.
 *	o	initialize the gip.
 */
GP_initialize(top_of_mem, fa)
ushort *top_of_mem;
{
	register int i = vbnum;
	register u_char *gpdm;
	int	s;
	short	x;

	if (
#ifdef	M68010
	    (gp_addrs[i].dm > (ushort *)svtop(&vme_short[0xFFFF])) &&
#endif	M68010
	    (gp_addrs[i].dm >= top_of_mem)) {
	    /* see if there is display memory there */ 
	    mapin(vidbufmap, btop(vidbuf), btop(gp_addrs[i].dm), 1,
		    PG_V|PG_FOD|PG_KW|PG_VMESTD);
	    if (!badaddr(vidbuf, 2)) {
		GPstd_dm[0] = gpdm = (u_char *)gp_addrs[i].dm;

		/* determine if GIP or MWS */
		if ((gptype&GPHASGIP) && !badaddr(gp_addrs[i].addr, 2) &&
		    /* 
		     * KLUDGE: Distinguish GIP from Excelan board.
		     * Some early GIP boards supported byte accesses, 
		     * so look to see if the upper byte is FF -- if so 
		     * assume it is an Excelan board.		CTH
		     */
		    (badaddr((int)gp_addrs[i].addr + 1, 1)  ||
		    ((gp_addrs[i].addr->gp_fifo & 0xFF00) != 0xFF00))) {

		    /* color or monochrome GIP? (wraps around at .5meg) */
		    x = vidbuf[0];
		    vidbuf[0] = 0xCACA;
		    mapin(vidbufmap, btop(vidbuf), btop(gpdm + (512*1024)),
			    1, PG_V|PG_FOD|PG_KW|PG_VMESTD);
		    vidbuf[0] = 0;
		    mapin(vidbufmap, btop(vidbuf), btop(gpdm), 
			    1, PG_V|PG_FOD|PG_KW|PG_VMESTD);
		    if ((unsigned int)vidbuf[0] == 0xCACA)
			    gptype |= GPCOLOR;
		    vidbuf[0] = x;

		    if (vbnum != 0)		/* interrupting ? */
			    gptype &= ~GPCANINT;
		    else {
			    s = spl0();		/* KLUDGE: catch strays */
			    DELAY(0x8000);
			    splx(s);
		    }
		} else {			/* MWS */
    mws:	    GPflevel = 0;
		    gptype = 0;
		}

		if (gptype & GPHASGIP) {
		    gpaddr = gp_addrs[i].addr;
		    gpwcs = (ushort *)gpaddr + (GPWCS / sizeof (short));
		    load_wcs();
		    mapin(vidbufmap, btop(vidbuf), btop(gpdm),
			(((gptype & GPCOLOR) ? 1024 : 512) * 1024) / NBPG, 
			PG_V|PG_FOD|PG_KW|PG_VMESTD);
		} else {
		    mapin(vidbufmap, btop(vidbuf), btop(gpdm),
			(256 * 1024) / NBPG, PG_V|PG_FOD|PG_KW|PG_VMESTD);
		    gpaddr = (struct gpdevice *)(((char *)vidbuf)+GPCSR_256_OFF);
		}
		GPstd[0] = ((ushort *)gpaddr);

		/*
		 * If we want interrupts, see if there is a hole in the bus.
		 * If we find a hole (stray interrupt) then reset the gip
		 * to negate the INTON instruction, and dont use interrupts!
		 */
		if (gptype & GPCANINT) {
		    if (gptype & GPCOLOR)
			DELAY(0x10000);
		    gp_probe = 1;
		    gpaddr->gp_fifo = GP_INTON;
		    s = spl0();			/* catch gip */
		    DELAY(0x8000);
		    splx(s);
		    gpaddr->gp_fifo = GP_INTOF;
		}

		vtpool = (char *)(SYSV_BASE | (int)ptob(fa));
		mapin(&Sysmap[fa], btop(vtpool), fa, btoc(VTSIZE), 
			PG_V|PG_KW/*|PG_CACHE*/);
		if ((gptype&GPHASGIP) == 0 && 
		    (gpaddr->gp_csr & GPCSR_R_MASK) == GPCSR_R_1024x1440)
			InitializeVT(1024, 1440, vidbuf, 256*1024);
		else
			InitializeVT(1280, 1024, vidbuf, 256*1024);
		FontInit();
		gptype |= GPEXISTS;			/* last thing! */
		vt_cninit();
#ifdef	M68020
		rrtimeslice *= 2;
#endif	M68020
		if ((gptype & GPCANINT) && gp_probe == 2)
			cprintf("***  NOTICE:  polled graphics processor: hole in bus ***\n");
		gp_probe = 0;
		return (btoc(VTSIZE));
	    }
	}
	printf("***  NOTICE:  Configured graphics processor not found  ***\n");
	return (0);
}

/*
 * if we have graphics, 
 *	fork off a kernel process to handle typeahead
 */
GP_fork()
{
	register int s;

	if (gptype & GPEXISTS) {
	    if (newproc(0)) {
		spl0();
		u.u_procp->p_flag |= SLOAD|SSYS;
		u.u_procp->p_pgrp = u.u_procp->p_pid;
#ifdef	M68020
		setctx(u.u_procp->p_context = CTX_SYS);
#endif	M68020
		strcpy(u.u_comm, "graphics daemon");
		if (gptype & GPCANINT)
			gptype |= GPUSEINT;
		GPdaemonpid = u.u_procp->p_pid;
		mark_daemon();
		gpget();
		while(1) {
			s = spltty();
			while (flushvtok == 0) {
			    gpgive();
			    sleep(&flushvtok, GP_FLUSHVTPRI);
			    gpget();
			}
			flushvtok = 0;
			splx(s);
			flushvtecho();
		}
	    }
	}
}

gpprobe(gp_addr)
struct gpdevice *gp_addr;
{
	register int i;

	if (gp_addr != gpaddr)
		return(0);
	if (gptype & GPHASGIP) {			/* GIP */
	    if (gptype & GPCANINT) {
		gpaddr->gp_fifo = GP_INTON;
		DELAY(0x8000);
		gpaddr->gp_fifo = GP_INTOF;
	    } else
		cvec = 0;
	} else {				/* MWS */
		GPvector = freevec();
		gpaddr->gp_csr = (GPvector & GPCSR_W_IVMASK) | 
				GPCSR_W_RETINT | GPCSR_W_PLANE;
		for (i = 5000; i && clev == -1; i--)
			DELAY(100);
		gpaddr->gp_csr = GPCSR_W_PLANE;
	}
	return(sizeof(struct gpdevice));
}

gpslave(qi, gp_addr)
register struct qb_device	*qi;
register struct gpdevice	*gp_addr;
{
	return(1);
}

gpattach(qi)
register struct qb_device	*qi;
{
	short	n, ncol, i;

	if ((gptype&GPHASGIP) == 0 && 
	    (gpaddr->gp_csr & GPCSR_R_MASK) == GPCSR_R_1024x1440)
		printf("\t(1024x1440)");
	else
		printf("\t(1280x1024)");
	printf(" %s %s at 0x%x\n", (gptype & GPCOLOR) ? "color" : "mono",
		(gptype & GPHASGIP) ? "GIP" : "memory", GPstd_dm[qi->qi_unit]);
	printf("\t\tterminal emulators:\t");
	ncol = 8*5;
	for(i=0; i<nterms; i++){
		n = strlen(termsw[i].name)+1;
		if((80-ncol) < (n+1)){
			printf("\n\t\t\t\t\t");
			ncol = 8*5;
		}
		ncol += n;
		printf("%s%c ", termsw[i].name, (i==nterms-1)?'.':',');
	}
}

gpintr(ctlr)
int ctlr;
{
	if (!(gptype & GPHASGIP))
		return;
	gpaddr->gp_fifo = GP_INTOF;
	if (gp_probe) {
		gptype &= ~GPCANINT;		/* dont use interrupts */
		if (gpaddr->gp_reset) 		/* reset interrupt request */
			;
		DELAY(0x8000);
		if (gptype & GPCOLOR)
			DELAY(0x10000);
		gp_probe = 2;
	} else {
		if (!(gptype & GPCANINT))
			printf("gp: stray interrupt\n");
		wakeup(&gpaddr);
	}
}

gpget()
{
	register int s;
	register struct proc *p = u.u_procp;

	if (GPdaemonpid == 0) {		/* wait till initialized! */
		p->p_flag |= SOWNGIP;
		return;
	}
	if (p->p_flag&SOWNGIP) 
		if (gptype & GPEXCLUSIVE)
			return;
		else
			gppanic("gp: double get");

	s = spltty();
	while (GPowner) {
		GPwanted++;
		if (p->p_pid == GPdaemonpid)
		    sleep(&GPowner, GP_FLUSHVTPRI);
		else
		    sleep(&GPowner, GP_WANTPRI);
		GPwanted--;
	}
	p->p_flag |= SOWNGIP;
	GPowner = p;
	bcopy(u.u_comm, GPu_comm, sizeof(GPu_comm));
	splx(s);
}

gpgive()
{
	register struct proc *p = u.u_procp;

	if (GPdaemonpid == 0) {		/* wait till initialized! */
		p->p_flag &= ~SOWNGIP;
		return;
	}
	if (p->p_flag&SOWNGIP) {
		if ((gptype & GPEXCLUSIVE) == 0) {
			if (GPwanted)
				wakeup(&GPowner);
			GPowner = NULL;
			p->p_flag &= ~SOWNGIP;
		}
	} else
		gppanic("gp: non-owner give");
	return;
}

gpillegal()
{
	cprintf("gp: illegal access\n");
}

gpwho()
{
	if (GPowner)
		printf("gp: owned by pid %d(%s),", GPowner->p_pid, GPu_comm);
	else
		printf("gp: not owned, last owner was %s, ", GPu_comm);
	if (GPwanted)
		printf("wanted by %d processes\n", GPwanted);
	else
		printf("not wanted\n");
}
#else	NGP > 0
unsigned char	gptype;
GP_initialize(top_of_mem) {gpaddr = 0; return(0)};
GP_fork() {};
#endif	NGP > 0
