/*
 * QBUS: Integrated Solutions 'Extended RL' (EL controler, el disk) driver
 */
#include "el.h"

#if	NEL > 0
#include "../machine/pte.h"

#include "param.h"
#include "systm.h"
#include "dk.h"
#include "dkbad.h"
#include "buf.h"
#include "conf.h"
#include "user.h"
#include "map.h"
#include "vm.h"
#include "cmap.h"
#include "uio.h"
#include "kernel.h"
#include "syslog.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/elreg.h"

#define Nel_EL		4	/* max number of el drives on EL controler */
#define EL_MAXCNT	(63*1024)

/* 
 * State of controller from last transfer. Since only one transfer can be done 
 * at a time per controller, only allocate one for each controller.
 */
struct	EL_stat {
	short		el_ndrive;	/* Number of drives on controller */
	short		el_wticks;	/* Monitor time for function */
	u_short		el_tn;		/* logical track number */
	u_short		el_sn;		/* sector number */
	u_short		el_cmd;		/* command to issue */
	int		el_addr;	/* physical address of transfer */
	u_short		el_bleft;	/* bytes left to transfer */
	u_short		el_bpart;	/* bytes transferred */
} 			EL_stat[NEL];

int	elprobe(),	elslave(),	elattach(),	elintr();
int	elstrat(),	elminphys();
struct	qb_ctlr		*elcinfo[NEL];
struct	qb_device	*eldinfo[Nel];

/* 
 * EL controler addresses 
 */
u_short *ELstd[] = { (u_short *)0x3FF900, (u_short *)0x3FF910, 0 };

/*
 * EL/el controler/device driver structure
 */
struct	qb_driver ELdriver =
	{ elprobe, elslave, elattach, ELstd, "el", eldinfo, "EL", elcinfo };

/* 
 * el drive structure: each drive can have a different geometry.
 */
struct qb_diskst	elst[Nel];

struct	buf	relbuf[Nel];
struct	dkbad	elbad[Nel];

#define	b_track		b_resid	
#define	elunit(bp)	(minor((bp)->b_dev) >> 3)
#define	elblock(bp)	((bp)->b_blkno)

int	elwstart, elwatch();		/* Have started guardian */

/* 
 * Check that controller exists
 */
elprobe(eladdr)
register struct eldevice	*eladdr;
{
	/* 
	 * Try to determine if what we have is an EL or an RL. This is done by 
	 * setting number of heads to 0, issuing a GETSTAT on the EL for el0. 
	 * If the number of heads is still 0, then it must be an RL controler.
	 * NOTE: The code assumes that if you have an EL then you also have an 
	 * el0 (i.e. if you have a controler, you have a drive 0). 
	 */
	eladdr->elba.getstat = 0;
	eladdr->elcs = EL_GETSTAT;
	if (elwait(eladdr))
		return (0);
	if (eladdr->elba.getstat == 0)
		return (0);

	/*
	 * Issue the same command, but this time it is to determine the interupt
	 * vector used by the controller. We should hear back in 200 micro secs.
	 */
	eladdr->elcs = EL_IE | EL_GETSTAT;
	DELAY(200);
	clevmax = clev_biomax;
	clev_bio = MAX(clev, clev_bio);
	return (sizeof (struct eldevice));
}

/*
 * Check that a given drive exists
 */
elslave(qi, eladdr)
register struct qb_device	*qi;
register struct eldevice	*eladdr;
{
	eladdr->elcs = (qi->qi_slave << 8) | EL_GETSTAT;
	if (elwait(eladdr))
		return (0);
	if (eladdr->elcs & EL_DE)
		return (0);
	return (1);
}

/*
 * Logicaly attach a unit to a drive/controler, and set up unit info
 */
elattach(qi)
register struct qb_device	*qi;
{
	register struct eldevice *eladdr;
	register struct qb_diskst *st;

	/*
	 * Start watchdog for lost interupts from drive
	 */
	if (elwstart == 0) {
		timeout(elwatch, (caddr_t)0, hz);
		elwstart++;
	}

	EL_stat[qi->qi_ctlr].el_ndrive++;
	eladdr = (struct eldevice *)qi->qi_mi->qm_addr;
	st = &elst[qi->qi_unit];

	/* 
	 * get geometry of drive
	 */
	eladdr->elcs = (qi->qi_slave << 8) | EL_GETSTAT;
	if (elwait(eladdr))
		printf("el: phantom drive???");
	ST_nspt = eladdr->elmp.getstat;
	ST_ntpc = eladdr->elba.getstat;
	ST_ncpd = eladdr->elda.getstat/ST_ntpc;
	ST_nspc = ST_nspt * ST_ntpc;
	printf("	%mM (%d x %d x %d) ",
	    ST_nspt * ST_ntpc * ST_ncpd * 512, ST_nspt, ST_ntpc, ST_ncpd);
	diskpart(ST_size, ST_nspt, ST_ntpc, ST_ncpd);

	/* Initialize iostat values */
	if (qi->qi_dk >= 0)
		dk_bps[qi->qi_dk] = (3600/60)*512*ST_nspt;
}

elopen(dev)
	dev_t dev;
{
	register int unit = minor(dev) >> 3;
	register struct qb_device *ui;

	if (unit >= Nel || (ui = eldinfo[unit]) == 0 || ui->qi_alive == 0)
		return (ENXIO);
	return (0);
}

elstrategy(bp)
	register struct buf *bp;
{
	register struct qb_device *qi;
	register struct qb_diskst *st;
	register int unit;
	int partition = minor(bp->b_dev) & 07, s;
	long bn, sz;

	unit = elunit(bp);
	if (unit >= Nel)
		goto bad;
	qi = eldinfo[unit];
	if (qi == 0 || qi->qi_alive == 0)
		goto bad;
	st = &elst[qi->qi_unit];
	sz = (bp->b_bcount+511) >> 9;			/* number of sectors */
	if (bp->b_blkno < 0 ||
	    (bn = elblock(bp))+sz > ST_size[partition].nblocks)
		goto bad;
	badstrat(qi, bp, &elbad[qi->qi_unit], elstrat,
	    ST_nspt, ST_ntpc, ST_ncpd, ST_size[partition].cyloff);
	return;
bad:
	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
}

elstrat(qi, bp)
	register struct qb_device *qi;
	register struct buf *bp;
{
	register struct qb_diskst *st = &elst[qi->qi_unit];
	int partition = minor(bp->b_dev) & 07, s;
	register struct buf *dp;

	bp->b_track = elblock(bp)/ST_nspt + ST_size[partition].cyloff*ST_ntpc;
	s = splx(qi->qi_mi->qm_psl);
	dp = &qi->qi_mi->qm_tab;
	disksort(dp, bp);
	if (dp->b_active == 0)
		elstart(qi->qi_mi);
	splx(s);
}

/*
 * Start up a transfer on a drive.
 */
elstart(qm)
	register struct qb_ctlr *qm;
{
	register struct buf *bp;
	register struct qb_device *qi;
	register struct eldevice *eladdr;
	register struct EL_stat *stat;
	register struct qb_diskst *st;
	daddr_t bn;

	if ((bp = qm->qm_tab.b_actf) == NULL)
		return;
	/*
	 * Mark controller busy, and determine destination.
	 */
	qm->qm_tab.b_active++;
	qi = eldinfo[elunit(bp)];		/* Controller */
	stat = &EL_stat[qm->qm_ctlr];
	st = &elst[qi->qi_unit];
	bn = elblock(bp);			/* 512 byte Block number */
	eladdr = (struct eldevice *)qm->qm_addr;
	/* 
	 * save away current data transfer drive info
	 */
	stat->el_tn = bp->b_track;
	stat->el_sn = bn % ST_nspt;
	stat->el_addr = qbaddr(bp);
	stat->el_bleft = bp->b_bcount;
	if (bp->b_flags & B_READ)
		stat->el_cmd = EL_IE | EL_READ | (qi->qi_slave << 8);
	else
		stat->el_cmd = EL_IE | EL_WRITE | (qi->qi_slave << 8);
	elio(eladdr, stat);
	if (qi->qi_dk >= 0) {
		dk_busy |= 1 << qi->qi_dk;
		dk_xfer[qi->qi_dk]++;
		dk_wds[qi->qi_dk] += bp->b_bcount >> 6;
	}
}

elio(eladdr, stat)
register struct eldevice *eladdr;
register struct EL_stat	*stat;
{
	stat->el_bpart = (stat->el_bleft>EL_MAXCNT)? EL_MAXCNT: stat->el_bleft;
	eladdr->elda.rw = stat->el_tn;
	eladdr->elba.rw = stat->el_addr & 0xFFFF;
	eladdr->elbae = (stat->el_sn << 6) | ((stat->el_addr>>16)&0x3F);
	eladdr->elmp.rw = -(stat->el_bpart >> 1);
	eladdr->elcs = stat->el_cmd;
}

/*
 * Handle a disk interrupt.
 */
elintr(ctlr)
	register ctlr;
{
	register struct buf *bp;
	register struct qb_ctlr *qm = elcinfo[ctlr];
	register struct qb_device *qi;
	register struct eldevice *eladdr = (struct eldevice *)qm->qm_addr;
	register unit;
	struct EL_stat *stat = &EL_stat[ctlr];
	register struct qb_diskst *st;

	stat->el_wticks = 0;
	bp = qm->qm_tab.b_actf;
	qi = eldinfo[elunit(bp)];
	st = &elst[qi->qi_unit];
	dk_busy &= ~(1 << qi->qi_dk);
	/*
	 * Check for and process errors on either the drive or the controller.
	 */
	if (eladdr->elcs & EL_ERR)
		/*
		 * After 10 retries give up.
		 */
		if (++qm->qm_tab.b_errcnt > 10) {
			harderr(bp, "el");
			printf("cs=%b\n", eladdr->elcs, ELCS_BITS);
			bp->b_flags |= B_ERROR;
		} else
			qm->qm_tab.b_active = 0;	 /* force retry */
	if (eladdr->elcs & (EL_SOFT | EL_ECC))
		log(LOG_WARNING, "el%d%c: sn %d, recovered %b\n", elunit(bp),
	    		'a'+(minor(bp->b_dev)&07), bp->b_blkno,
			eladdr->elcs, ELCS_BITS);
	/*
	 * If still ``active'', then don't need any more retries.
	 */
	if (qm->qm_tab.b_active) {
		/* 
		 * check if more data from previous request
		 */
		if ((bp->b_flags & B_ERROR) == 0 &&
		     (stat->el_bleft -= stat->el_bpart) > 0) {
			stat->el_addr += stat->el_bpart;
			stat->el_bpart /= 512;
			stat->el_tn += stat->el_bpart / ST_nspt;
			stat->el_sn = (stat->el_sn+stat->el_bpart)%ST_nspt;
			elio(eladdr, stat);
			return;
		}
		dk_busy &= ~(1 << qi->qi_dk);
		qm->qm_tab.b_active = qm->qm_tab.b_errcnt = 0;
		bp->b_resid = stat->el_bleft;
		qm->qm_tab.b_actf = bp->av_forw;
		iodone(bp);
	}
	/*
	 * If controller not transferring and have request, start controller.
	 */
	if (qm->qm_tab.b_actf && qm->qm_tab.b_active == 0)
		elstart(qm);
}

elread(dev, uio)
dev_t	dev;
struct uio *uio;
{
	register int unit = minor(dev) >> 3;

	if (unit >= Nel)
		return (ENXIO);
	return (physio(elstrategy, &relbuf[unit], dev, B_READ, elminphys, uio));
}

elwrite(dev, uio)
dev_t dev;
struct uio *uio;
{
	register int unit = minor(dev) >> 3;

	if (unit >= Nel)
		return (ENXIO);
	return (physio(elstrategy, &relbuf[unit], dev, B_WRITE, elminphys, uio));
}

#include "ioctl.h"

elioctl(dev, cmd, data)
dev_t             dev;
int               cmd;
struct qb_diskst *data;
{
	register int unit = minor(dev) >> 3;

	if (unit > Nel)
		return(ENXIO);
	if (cmd != DIOCGGEO)
		return(EIO);
	*data = elst[unit];
	return(0);
}

/*
 * Reset driver after lost interupt. Cancel software state of all pending 
 * transfers and restart all units and the controller.
 */
elreset()
{
	register struct qb_ctlr *qm;
	register struct qb_device *qi;
	register struct eldevice *eladdr;
	register struct EL_stat *stat;
	register int ctlr, unit;

	for (ctlr = 0; ctlr < NEL; ctlr++) {
		if ((qm = elcinfo[ctlr]) == 0 || qm->qm_alive == 0)
			continue;
		printf(" EL%d", ctlr);
		eladdr = (struct eldevice *)qm->qm_addr;
		stat = &EL_stat[ctlr];
		qm->qm_tab.b_active = 0;
		stat->el_bleft = 0;
		stat->el_bpart = 0;
		elwait(eladdr);
		for (unit = 0; unit < Nel_EL; unit++) {
			eladdr->elcs = (unit << 8) | EL_GETSTAT;
			elwait(eladdr);
			if ((qi = eldinfo[unit]) == 0)
				continue;
			if (qi->qi_alive == 0 || qi->qi_mi != qm)
				continue;
		}
		elstart(qm);
	}
}

/*
 * Wake up every second and if an interrupt is pending but nothing has happened
 * increment a counter. If nothing happens for 20 seconds, reset the controler
 * and begin anew.
 */
elwatch()
{
	register struct qb_ctlr *qm;
	register ctlr, unit;
	register struct EL_stat *stat;

	timeout(elwatch, (caddr_t)0, hz);
	for (ctlr = 0; ctlr < NEL; ctlr++) {
		qm = elcinfo[ctlr];
		if (qm == 0 || qm->qm_alive == 0 || qm->qm_tab.b_active == 0)
			continue;
		stat = &EL_stat[ctlr];
		stat->el_wticks++;
		if (stat->el_wticks >= 20) {
			stat->el_wticks = 0;
			printf("EL%d: lost interrupt\n", ctlr);
			elreset();
		}
	}
}

int eldumpstrat();
extern struct buf dumpbuf;
extern int dumpsize;

eldump(dev)
	dev_t dev;
{
	register int unit = minor(dev) >> 3;
	register int partition = minor(dev) & 07;
	register struct buf *bp = &dumpbuf;
	register struct qb_device *qi;
	register struct qb_diskst *st;
	int num, start, bcount;

	if (unit >= Nel || (qi = eldinfo[unit]) == 0 || qi->qi_alive == 0)
		return ENXIO;
	num = ctod(dumpsize);		/* memory size in disk blocks */
	start = 0x0;			/* start dumping at physical 0 */

	/*
	 * KLUDGE: the QBUS-68010 does not allow you to read location 0 
	 * so start the dump 512!!!
	 */
	num--;
	start += dbtob(1);

	st = &elst[unit];
	if (dumplo < 0 || dumplo+num > ST_size[partition].nblocks)
		return EINVAL;
	while (num > 0) {
		bcount = (dbtob(num) > EL_MAXCNT) ? EL_MAXCNT : dbtob(num);
		bp->b_dev = dev;
		bp->b_blkno = dumplo + btodb(start);
		bp->b_bcount = bcount;
		bp->b_un.b_addr = (caddr_t) start;
		bp->b_flags = B_WRITE; 
		badstrat(qi, bp, &elbad[qi->qi_unit], eldumpstrat, 
		    ST_nspt, ST_ntpc, ST_ncpd, ST_size[partition].cyloff);
		if (bp->b_flags & B_ERROR || bp->b_resid)
			return EIO;
		start += bcount;
		num -= btodb(bcount);
	}
	return 0;
}

eldumpstrat(qi, bp)
	register struct qb_device *qi;
	register struct buf *bp;
{
	register struct EL_stat	*stat = &EL_stat[qi->qi_ctlr];
	register struct eldevice *eladdr=(struct eldevice *)qi->qi_mi->qm_addr;
	register struct qb_diskst *st = &elst[qi->qi_unit];
	register int errcount;

	errcount = 0;
	while (1) {
		stat->el_tn = elblock(bp)/ST_nspt + 
		    ST_size[minor(bp->b_dev)&07].cyloff*ST_ntpc;
		stat->el_sn = elblock(bp) % ST_nspt;
		stat->el_addr = (u_int) bp->b_un.b_addr;
		stat->el_bleft = bp->b_bcount;
		stat->el_cmd = ((bp->b_flags & B_READ)? EL_READ : EL_WRITE) |
			 (qi->qi_slave << 8);
		elio(eladdr, stat);
		if (elwait(eladdr)) {
			printf("el%d%c: HUNG sn%d ", minor(bp->b_dev) >> 3, 
				'a'+(minor(bp->b_dev)&07), bp->b_blkno);
			bp->b_flags |= B_ERROR;
			bp->b_resid = bp->b_bcount;
			return EIO;
		} else if (eladdr->elcs & EL_ERR) {
			if (++errcount > 10) {
				harderr(bp, "el");
				printf("cs=%b:", eladdr->elcs, ELCS_BITS);
				bp->b_flags |= B_ERROR;
				bp->b_resid = bp->b_bcount;
				return EIO;
			}
		} else if (eladdr->elcs & (EL_SOFT | EL_ECC)) {
			printf("el%d%c: sn %d, recovered %b: ", elunit(bp),
	    			'a'+(minor(bp->b_dev)&07), bp->b_blkno,
				eladdr->elcs, ELCS_BITS);
		} else {
			bp->b_resid = 0;
			break;
		}
	}
	bp->b_flags |= B_DONE;
	return 0;
}

elsize(dev)
	dev_t dev;
{
	register int unit = minor(dev) >> 3;
	register struct qb_device *qi;
	register struct qb_diskst *st;

	if (unit >= Nel || (qi = eldinfo[unit]) == 0 || qi->qi_alive == 0)
		return (-1);
	st = &elst[qi->qi_unit];
	return (ST_size[minor(dev)&0x7].nblocks);
}

static 
elwait(eladdr)
struct eldevice *eladdr;
{ 	
	long l = 0;

	while ((eladdr->elcs & EL_CRDY) == 0) {
		if (++l == 800)
			return (-1);
		DELAY(1000);
	}
	return (0);
}

elminphys(bp)
	struct buf *bp;
{
	bp->b_bcount = MIN(bp->b_bcount, EL_MAXCNT);
}
#endif	NEL > 0
