/*
 * QBUS: DZ-11/DZ-32 Driver
 */
#include "dz.h"
#include "bk.h"

#if	NDZ > 0
#include "../machine/pte.h"

#include "param.h"
#include "systm.h"
#include "ioctl.h"
#include "tty.h"
#include "user.h"
#include "proc.h"
#include "map.h"
#include "buf.h"
#include "vm.h"
#include "conf.h"
#include "bk.h"
#include "file.h"
#include "uio.h"
#include "kernel.h"
#include "syslog.h"

#include "../is68kdev/pdma.h"
#include "../is68kdev/qbvar.h"
#include "../is68kdev/dzreg.h"

#ifdef	SYSV
#include "../sysv/sys/termio.h"
#endif	SYSV


#define	TTY_NODELAY

/*
 * Driver information for auto-configuration stuff.
 */
int	dzprobe(), dzslave(), dzattach(), dzrint();
struct	qb_device *dzinfo[NDZ];
u_short	*dzstd[] = { (u_short *)0x3FE008, (u_short *)0x3FE010,
		     (u_short *)0x3FE018, (u_short *)0x3FE020,
		     (u_short *)0x3FE028, (u_short *)0x3FE030,
		     (u_short *)0x3FE038, (u_short *)0x3FE040,
		     0 };
struct	qb_driver DZdriver =
	{ dzprobe, dzslave, dzattach, dzstd, "dz", dzinfo, "DZ" };

#define	NDZLINE 	(NDZ*8)
#define	FASTTIMER	(hz/30)		/* rate to drain silos, when in use */

int	dzstart(), dzxint(), dzdma(), ttrstrt();
struct	tty dz_tty[NDZLINE];
int	ndz_tty = 0;
int	dzact;
int	dzsilos;			/* mask of dz's with silo in use */
int	dzchars[NDZ];			/* recent input count */
int	dzrate[NDZ];			/* smoothed input count */
int	dztimerintvl;			/* time interval for dztimer */
int	dzhighrate = 100;		/* silo on if dzchars > dzhighrate */
int	dzlowrate = 75;			/* silo off if dzrate < dzlowrate */

#define dzwait(x)	while (((x)->dzlcs & DZ_ACK) == 0)

/*
 * Software copy of dzbrk since it isn't readable
 */
char	dz_brk[NDZ];
char	dzsoftCAR[NDZ];
char	dz_lnen[NDZ];	/* saved line enable bits for DZ32 */
short	dz_lpr[NDZLINE];

/*
 * The dz11 doesn't interrupt on carrier transitions, so
 * we have to use a timer to watch it.
 */
char	dz_timer;		/* timer started? */

/*
 * Pdma structures for fast output code
 */
struct	pdma dzpdma[NDZLINE];

char	dz_speeds[] =
	{ 0,020,021,022,023,024,0,025,026,027,030,032,034,036,037,0 };
 
#ifndef PORTSELECTOR
#define	ISPEED	B9600
#define	IFLAGS	(EVENP|ODDP|ECHO)
#else
#define	ISPEED	B4800
#define	IFLAGS	(EVENP|ODDP)
#endif

dzprobe(reg)
	caddr_t reg;
{
	extern int cvec;
	register struct dzdevice *dzaddr = (struct dzdevice *)reg;

#ifdef lint
	br = 0; cvec = br; br = cvec;
	dzrint(0); dzxint((struct tty *)0);
#endif
	DELAY(1000);
	dzaddr->dzcsr = DZ_NU;	/* dont confuse a dz with a dh */
	DELAY(1000);
	if (dzaddr->dzcsr & DZ_NU)
		return (0);
	dzaddr->dzcsr = DZ_CLR;			/* reset everything */
	DELAY(1000);
	if (dzaddr->dzcsr & DZ_CLR)
		return (sizeof (struct dzdevice));
	dzaddr->dzcsr = DZ_TIE|DZ_MSE|DZ_32;
	DELAY(1000);
	if (dzaddr->dzcsr & DZ_32)
		dzaddr->dzlnen = 1;
	else
		dzaddr->dztcr = 0xff;		/* enable any line */
	DELAY(100000);
	dzaddr->dzcsr = DZ_CLR|DZ_32;		/* reset everything */
	DELAY(10000);				/* DILOG-CQ800 bug */
	dzaddr->dztcr = 0x00;			/* DILOG-CQ800 bug */
	ndz_tty += 8;				/* not really true, DZ32 */
	if (cvec && cvec != 0x100)
		cvec -= 4;
	clevmax = clev_ttymax;
	clev_tty = MAX(clev, clev_tty);
	return (sizeof (struct dzdevice));
}

dzslave(reg)
	caddr_t reg;
{
	return (1);
}

dzattach(qi)
	register struct qb_device *qi;
{
	register struct pdma *pdp = &dzpdma[qi->qi_unit*8];
	register struct tty *tp = &dz_tty[qi->qi_unit*8];
	register int cntr;
	extern dzscan();

	for (cntr = 0; cntr < 8; cntr++) {
		pdp->p_addr = (int)qi->qi_mi->qm_addr;
		pdp->p_arg = (int)tp;
		pdp->p_fcn = dzxint;
		pdp++, tp++;
	}
	dzsoftCAR[qi->qi_unit] = qi->qi_flags;
	if (dz_timer == 0) {
		dz_timer++;
		timeout(dzscan, (caddr_t)0, hz);
		dztimerintvl = FASTTIMER;
	}
}

#ifdef	SYSV
int sv_dzproc();
#endif	SYSV

/*ARGSUSED*/
dzopen(dev, flag)
	dev_t dev;
{
	register struct tty *tp;
	register int unit;
	register int rval;
 
	unit = minor(dev);
	if (unit >= ndz_tty || dzpdma[unit].p_addr == 0)
		return (ENXIO);
	tp = &dz_tty[unit];
	tp->t_addr = (caddr_t)&dzpdma[unit];
	tp->t_oproc = dzstart;
	tp->t_state |= TS_WOPEN;
#ifdef	SYSV
	map_state(tp, TOSV);
	tp->svt_proc = sv_dzproc;
#endif	SYSV
	if ((tp->t_state & TS_ISOPEN) == 0) {
		ttychars(tp);
		if (tp->t_ispeed == 0) {
			tp->t_ospeed = tp->t_ispeed = ISPEED;
			tp->t_flags = IFLAGS;
		}
#ifdef	SYSV
		sv_ttinit(tp);
#endif	SYSV
		dzparam(unit);
	} else if ((tp->t_state&TS_XCLUDE) && 
	    ((u.u_uid!=0) || (u.u_procp->p_flag&SLOGIN)))
		return (EBUSY);

	(void) dzmctl(dev, DZ_ON, DMSET);
	(void) spl5();
#ifdef TTY_NODELAY
	if ((flag & FNDELAY) == 0)
#endif
	while ((tp->t_state & TS_CARR_ON) == 0) {
		tp->t_state |= TS_WOPEN;
#ifdef	SYSV
		map_state(tp, TOSV);
#endif	SYSV
		sleep((caddr_t)&tp->t_rawq, TTIPRI);
		if ((tp->t_state&TS_XCLUDE) && (u.u_procp->p_flag&SLOGIN)) {
			(void) spl0();
			return (EBUSY);
		}
	}
	(void) spl0();
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_open)(dev, tp, flag));
	else
#endif	SYSV
		return ((*linesw[tp->t_line].l_open)(dev, tp));
}
 
dzclose(dev, flag)
	dev_t dev;
{
	register struct tty *tp;
	register int unit;
	register struct dzdevice *dzaddr;
	int dz;
 
	unit = minor(dev);
	dz = unit >> 3;
	tp = &dz_tty[unit];
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		(*sv_linesw[tp->svt_line].l_close)(tp);
	else
#endif	SYSV
		(*linesw[tp->t_line].l_close)(tp);
	dzaddr = (struct dzdevice *)dzpdma[unit].p_addr;
	if (dzaddr->dzcsr&DZ_32)
		(void) dzmctl(dev, DZ_BRK, DMBIC);
	else
		dzaddr->dzbrk = (dz_brk[dz] &= ~(1 << (unit&07)));
	if ((tp->t_state&(TS_HUPCLS|TS_WOPEN)) || (tp->t_state&TS_ISOPEN) == 0)
		(void) dzmctl(dev, DZ_OFF, DMSET);
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		sv_ttyclose(tp);
	else
#endif	SYSV
		ttyclose(tp);
}
 
dzread(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	register struct tty *tp;
 
	tp = &dz_tty[minor(dev)];
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_read)(tp, uio));
	else
#endif	SYSV
		return ((*linesw[tp->t_line].l_read)(tp, uio));
}
 
dzwrite(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	register struct tty *tp;
 
	tp = &dz_tty[minor(dev)];
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV)
		return ((*sv_linesw[tp->svt_line].l_write)(tp, uio));
	else
#endif	SYSV
		return ((*linesw[tp->t_line].l_write)(tp, uio));
}
 
dzrint(dz)
	int dz;
{
	register struct tty *tp;
	register int c;
	register struct dzdevice *dzaddr;
	register struct tty *tp0;
	register int unit;
	int overrun = 0;
 
	if ((dzact & (1<<dz)) == 0)
		return;
	unit = dz * 8;
	dzaddr = (struct dzdevice *)dzpdma[unit].p_addr;
	tp0 = &dz_tty[unit];
	dzaddr->dzcsr &= ~(DZ_RIE|DZ_MIE);	/* the manual says this song */
	dzaddr->dzcsr |= DZ_RIE|DZ_MIE;		/*   and dance is necessary */
	while (dzaddr->dzcsr & DZ_MSC) {	/* DZ32 modem change interrupt */
		c = dzaddr->dzmtsr;
		tp = tp0 + (c&7);
		if (tp >= &dz_tty[ndz_tty])
			break;
		dzaddr->dzlcs = c&7;	/* get status of modem lines */
		dzwait(dzaddr);		/* wait for them */
		if (c & DZ_CD)		/* carrier status change? */
		if (dzaddr->dzlcs & DZ_CD)	/* carrier up? */
#ifdef	SYSV
		    if (tp->t_universe == UNIVERSE_SYSV)
			(void)(*sv_linesw[tp->svt_line].l_mdmint)(tp, 1);
		    else
#endif	SYSV
			(void)(*linesw[tp->t_line].l_modem)(tp, 1);
		else 
#ifdef	SYSV
		    if (tp->t_universe == UNIVERSE_SYSV) {
			if ((*sv_linesw[tp->svt_line].l_mdmint)(tp, 0) == 0)
				dzaddr->dzlcs = DZ_ACK|(c&7);
		    } else 
#endif	SYSV
			if ((*linesw[tp->t_line].l_modem)(tp, 0) == 0)
				dzaddr->dzlcs = DZ_ACK|(c&7);
	}
	while ((c = dzaddr->dzrbuf) < 0) {	/* char present */
		dzchars[dz]++;
		tp = tp0 + ((c>>8)&07);
		if (tp >= &dz_tty[ndz_tty])
			continue;
		if ((tp->t_state & TS_ISOPEN) == 0) {
			wakeup((caddr_t)&tp->t_rawq);
#ifdef PORTSELECTOR
			if ((tp->t_state&TS_WOPEN) == 0)
#endif
			continue;
		}
#ifdef	SYSV
		if (tp->t_universe == UNIVERSE_SYSV) {
			if (c & DZ_PE) {
				if (tp->svt_iflag & SV_IGNPAR)
					continue;
			    	if (tp->svt_iflag & SV_PARMRK)
					;		/* ??? */
			}
			if (c & DZ_DO && overrun == 0) {
				log(LOG_WARNING, "dz%d,%d: silo overflow\n",
					dz, (c>>8)&7);
				overrun = 1;
			}
			if (c & DZ_FE) {
			    	if (tp->svt_iflag & SV_IGNBRK)
					continue;
			    	if (tp->svt_iflag & SV_BRKINT)
					(*sv_linesw[tp->svt_line].l_input)
						(c, tp, L_BREAK);
			    	c = 0;
			}
			if (tp->svt_iflag & SV_ISTRIP)
				c &= 0177;
			else
				c &= 0377;
		} else {
#endif	SYSV
		if (c&DZ_FE) {
			if (tp->t_flags & RAW)
				c = 0;
			else
				c = tp->t_intrc;
		}
		if (c&DZ_DO && overrun == 0) {
			log(LOG_WARNING, "dz%d,%d: silo overflow\n",
				dz, (c>>8)&7);
			overrun = 1;
		}
		if (c&DZ_PE)	
			if (((tp->t_flags & (EVENP|ODDP)) == EVENP)
			  || ((tp->t_flags & (EVENP|ODDP)) == ODDP))
				continue;
#ifdef	SYSV
		}
		if (tp->t_universe == UNIVERSE_SYSV)
			(*sv_linesw[tp->svt_line].l_input)(c, tp, 0);
		else
#endif	SYSV
#if NBK > 0  ||  Nbk > 0
		if (tp->t_line == NETLDISC) {
			c &= 0177;
			BKINPUT(c, tp);
		} else
#endif
			(*linesw[tp->t_line].l_rint)(c, tp);
	}
}
 
/*ARGSUSED*/
dzioctl(dev, cmd, data, flag)
	dev_t dev;
	caddr_t data;
{
	register struct tty *tp;
	register int unit = minor(dev);
	register int dz = unit >> 3;
	register struct dzdevice *dzaddr;
	int error;
 
	tp = &dz_tty[unit];
#ifdef	SYSV
	if (u.u_procp->p_universe == UNIVERSE_SYSV) {
		if ((error = sv_ttiocom(tp, cmd, data)) > 0)
			return error;
		if (cmd == TCSETAF || cmd == TCSETA || cmd == TCSETAW)
			dzparam(unit);
		if (cmd == TCSBRK)
			;		/* ??? */
		return 0;
	}
#endif	SYSV
	error = (*linesw[tp->t_line].l_ioctl)(tp, cmd, data, flag);
	if (error >= 0)
		return (error);
	error = ttioctl(tp, cmd, data, flag);
	if (error >= 0) {
		if (cmd == TIOCSETP || cmd == TIOCSETN ||
		  cmd == TIOCLSET || cmd == TIOCLBIS || cmd == TIOCLBIC)
			dzparam(unit);
		return (error);
	}
	switch (cmd) {

	case TIOCSBRK:
		dzaddr = (struct dzdevice *)((struct pdma *)(tp->t_addr))->p_addr;
		if (dzaddr->dzcsr&DZ_32)
			(void) dzmctl(dev, DZ_BRK, DMBIS);
		else
			dzaddr->dzbrk = (dz_brk[dz] |= 1 << (unit&07));
		break;

	case TIOCCBRK:
		dzaddr = (struct dzdevice *)((struct pdma *)(tp->t_addr))->p_addr;
		if (dzaddr->dzcsr&DZ_32)
			(void) dzmctl(dev, DZ_BRK, DMBIC);
		else
			dzaddr->dzbrk = (dz_brk[dz] &= ~(1 << (unit&07)));
		break;

	case TIOCSDTR:
		(void) dzmctl(dev, DZ_DTR|DZ_RTS, DMBIS);
		break;

	case TIOCCDTR:
		(void) dzmctl(dev, DZ_DTR|DZ_RTS, DMBIC);
		break;

	case TIOCMSET:
		(void) dzmctl(dev, dmtodz(*(int *)data), DMSET);
		break;

	case TIOCMBIS:
		(void) dzmctl(dev, dmtodz(*(int *)data), DMBIS);
		break;

	case TIOCMBIC:
		(void) dzmctl(dev, dmtodz(*(int *)data), DMBIC);
		break;

	case TIOCMGET:
		*(int *)data = dztodm(dzmctl(dev, 0, DMGET));
		break;

	default:
		return (ENOTTY);
	}
	return (0);
}

dmtodz(bits)
	register int bits;
{
	register int b;

	b = (bits >> 1) & 0370;
	if (bits & DML_ST) b |= DZ_ST;
	if (bits & DML_RTS) b |= DZ_RTS;
	if (bits & DML_DTR) b |= DZ_DTR;
	if (bits & DML_LE) b |= DZ_LE;
	return(b);
}

dztodm(bits)
	register int bits;
{
	register int b;

	b = (bits << 1) & 0360;
	if (bits & DZ_DSR) b |= DML_DSR;
	if (bits & DZ_DTR) b |= DML_DTR;
	if (bits & DZ_ST) b |= DML_ST;
	if (bits & DZ_RTS) b |= DML_RTS;
	return(b);
}
 
dzparam(unit)
	register int unit;
{
	register struct tty *tp;
	register struct dzdevice *dzaddr;
	register short lpr;
 
	tp = &dz_tty[unit];
	dzaddr = (struct dzdevice *)dzpdma[unit].p_addr;
	if (dzsilos & (1 << (unit >> 3)))
		dzaddr->dzcsr = DZ_IEN | DZ_SAE;
	else
		dzaddr->dzcsr = DZ_IEN;
	dzact |= (1 << (unit >> 3));
	if (tp->t_ispeed == 0) {
		(void) dzmctl(unit, DZ_OFF, DMSET);	/* hang up line */
		return;
	}
	lpr = (dz_speeds[tp->t_ispeed]<<8) | (unit & 07);
#ifdef	SYSV
	if (tp->t_universe == UNIVERSE_SYSV) {
		if ((tp->svt_cflag&SV_CSIZE) == SV_CS8)
			lpr |= BITS8;
		else
			lpr |= BITS7;
		if (tp->svt_cflag&SV_CSTOPB)
			lpr |= TWOSB;
		if (tp->svt_cflag&SV_PARENB) {
			lpr |= PENABLE;
			if (tp->svt_cflag&SV_PARODD)
				lpr |= OPAR;
		}
	} else {
#endif	SYSV
	if (tp->t_flags & (RAW|LITOUT|PASS8))
		lpr |= BITS8;
	else
		lpr |= (BITS7|PENABLE);
	if ((tp->t_flags & EVENP) == 0)
		lpr |= OPAR;
	if ((tp->t_ospeed) == B110 || tp->t_state & TS_TWOSB)
		lpr |= TWOSB;
#ifdef	SYSV
	}
#endif	SYSV
	if (dz_lpr[unit] == lpr)
		return;
	dz_lpr[unit] = lpr;
	if (tp->t_ospeed <= B1200)
		DELAY(100);
	dzaddr->dzlpr = lpr;
}

#ifdef	SYSV
sv_dzproc(tp, cmd)
register struct tty *tp;
register cmd;
{
	sv_proc(tp, cmd, dzstart, dzparam);
}
#endif	SYSV
 
dzxint(tp)
	register struct tty *tp;
{
	register struct pdma *dp;
	register s, dz, unit;
 
	s = spl5();		/* block pdma interrupts */
	dp = (struct pdma *)tp->t_addr;
	tp->t_state &= ~TS_BUSY;
#ifdef	SYSV
	map_state(tp, TOSV);
#endif	SYSV
	if (tp->t_state & TS_FLUSH) {
		tp->t_state &= ~TS_FLUSH;
#ifdef	SYSV
		map_state(tp, TOSV);
#endif	SYSV
	} else {
		ndflush(&tp->t_outq, dp->p_mem-tp->t_outq.c_cf);
		dp->p_end = dp->p_mem = tp->t_outq.c_cf;
	}
#ifdef	SYSV
	if (tp->t_universe == UNIVERSE_SYSV) {
		if ((*sv_linesw[tp->svt_line].l_output)(tp))
			dzstart(tp);
	} else
#endif	SYSV
		if (tp->t_line)
			(*linesw[tp->t_line].l_start)(tp);
		else
			dzstart(tp);
	dz = minor(tp->t_dev) >> 3;
	unit = minor(tp->t_dev) & 7;
	if (tp->t_outq.c_cc == 0 || (tp->t_state&TS_BUSY)==0)
		if (((struct dzdevice *)dp->p_addr)->dzcsr & DZ_32)
			((struct dzdevice *)dp->p_addr)->dzlnen = (dz_lnen[dz] &= ~(1<<unit));
		else
			((struct dzdevice *)dp->p_addr)->dztcr &= ~(1<<unit);
	splx(s);
}

dzstart(tp)
	register struct tty *tp;
{
	register struct pdma *dp;
	register struct dzdevice *dzaddr;
	register int cc;
	int s, dz, unit;
 
	dp = (struct pdma *)tp->t_addr;
	dzaddr = (struct dzdevice *)dp->p_addr;
	s = spl5();
	if (tp->t_state & (TS_TIMEOUT|TS_BUSY|TS_TTSTOP))
		goto out;
	if (tp->t_outq.c_cc <= TTLOWAT(tp)) {
		if (tp->t_state&TS_ASLEEP) {
			tp->t_state &= ~TS_ASLEEP;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
			wakeup((caddr_t)&tp->t_outq);
		}
		if (tp->t_wsel) {
			selwakeup(tp->t_wsel, tp->t_state & TS_WCOLL);
			tp->t_wsel = 0;
			tp->t_state &= ~TS_WCOLL;
		}
	}
	if (tp->t_outq.c_cc == 0)
		goto out;
	if (tp->t_flags & (RAW|LITOUT))
		cc = ndqb(&tp->t_outq, 0);
	else {
		cc = ndqb(&tp->t_outq, 0200);
		if (cc == 0) {
			cc = getc(&tp->t_outq);
			timeout(ttrstrt, (caddr_t)tp, (cc&0x7f) + 6);
			tp->t_state |= TS_TIMEOUT;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
			goto out;
		}
	}
	tp->t_state |= TS_BUSY;
#ifdef	SYSV
	map_state(tp, TOSV);
#endif	SYSV
	dp->p_end = dp->p_mem = tp->t_outq.c_cf;
	dp->p_end += cc;
	dz = minor(tp->t_dev) >> 3;
	unit = minor(tp->t_dev) & 7;
	if (dzaddr->dzcsr & DZ_32)
		dzaddr->dzlnen = (dz_lnen[dz] |= (1<<unit));
	else
		dzaddr->dztcr |= (1<<unit);
out:
	splx(s);
}
 
/*
 * Stop output on a line.
 */
/*ARGSUSED*/
dzstop(tp, flag)
	register struct tty *tp;
{
	register struct pdma *dp;
	register int s;

	dp = (struct pdma *)tp->t_addr;
	s = spl5();
	if (tp->t_state & TS_BUSY) {
		dp->p_end = dp->p_mem;
		if ((tp->t_state&TS_TTSTOP)==0) {
			tp->t_state |= TS_FLUSH;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
		}
	}
	splx(s);
}
 
dzmctl(dev, bits, how)
	dev_t dev;
	int bits, how;
{
	register struct dzdevice *dzaddr;
	register int unit, mbits;
	int b, s;

	unit = minor(dev);
	b = 1<<(unit&7);
	dzaddr = (struct dzdevice *)dzpdma[unit].p_addr;
	s = spl5();
	if (dzaddr->dzcsr & DZ_32) {
		dzwait(dzaddr)
		DELAY(100);		/* IS 100 TOO MUCH? */
		dzaddr->dzlcs = unit&7;
		DELAY(100);
		dzwait(dzaddr)
		DELAY(100);
		mbits = dzaddr->dzlcs;
		mbits &= 0177770;
	} else {
		mbits = (dzaddr->dzdtr & b) ? DZ_DTR : 0;
		mbits |= (dzaddr->dzmsr & b) ? DZ_CD : 0;
		mbits |= (dzaddr->dztbuf & b) ? DZ_RI : 0;
	}
	switch (how) {
	case DMSET:
		mbits = bits;
		break;

	case DMBIS:
		mbits |= bits;
		break;

	case DMBIC:
		mbits &= ~bits;
		break;

	case DMGET:
		(void) splx(s);
		return(mbits);
	}
	if (dzaddr->dzcsr & DZ_32) {
		mbits |= DZ_ACK|(unit&7);
		dzaddr->dzlcs = mbits;
	} else {
		if (mbits & DZ_DTR)
			dzaddr->dzdtr |= b;
		else
			dzaddr->dzdtr &= ~b;
	}
	(void) splx(s);
	return(mbits);
}
 
int dztransitions, dzfasttimers;		/*DEBUG*/
dzscan()
{
	register i;
	register struct dzdevice *dzaddr;
	register bit;
	register struct tty *tp;
	register car;
	int olddzsilos = dzsilos;
	int dztimer();
 
	for (i = 0; i < ndz_tty ; i++) {
		dzaddr = (struct dzdevice *)dzpdma[i].p_addr;
		if (dzaddr == 0)
			continue;
		tp = &dz_tty[i];
		bit = 1<<(i&07);
		car = 0;
		if (dzsoftCAR[i>>3]&bit)
			car = 1;
		else if (dzaddr->dzcsr & DZ_32) {
			dzaddr->dzlcs = i&07;
			dzwait(dzaddr);
			car = dzaddr->dzlcs & DZ_CD;
		} else
			car = dzaddr->dzmsr&bit;
		if (car) {
			/* carrier present */
			if ((tp->t_state & TS_CARR_ON) == 0)
#ifdef	SYSV
			    if (tp->t_universe == UNIVERSE_SYSV)
				(void)(*sv_linesw[tp->svt_line].l_mdmint)(tp,1);
			    else
#endif	SYSV
				(void)(*linesw[tp->t_line].l_modem)(tp, 1);
		} else if (tp->t_state&TS_CARR_ON) {
#ifdef	SYSV
		    if (tp->t_universe == UNIVERSE_SYSV) {
			if ((*sv_linesw[tp->svt_line].l_mdmint)(tp, 0) == 0)
				dzaddr->dzdtr &= ~bit;
		    } else
#endif	SYSV
		    if ((*linesw[tp->t_line].l_modem)(tp, 0) == 0)
			dzaddr->dzdtr &= ~bit;
		}
	}
	for (i = 0; i < NDZ; i++) {
		dzaddr = (struct dzdevice *)dzpdma[i << 3].p_addr;
		ave(dzrate[i], dzchars[i], 8);
		if (dzchars[i] > dzhighrate && ((dzsilos & (1 << i)) == 0)) {
			int s = spl5();
			dzaddr->dzcsr = DZ_IEN | DZ_SAE;
			dzsilos |= (1 << i);
			dzrint(i);	/* drain silos after changing SAE! */
			splx(s);
			dztransitions++;		/*DEBUG*/
		} else if ((dzsilos & (1 << i)) && (dzrate[i] < dzlowrate)) {
			int s = spl5();
			dzaddr->dzcsr = DZ_IEN;
			dzsilos &= ~(1 << i);
			dzrint(i);	/* drain silos after changing SAE! */
			splx(s);
		}
		dzchars[i] = 0;
	}
	if (dzsilos && !olddzsilos)
		timeout(dztimer, (caddr_t)0, dztimerintvl);
	timeout(dzscan, (caddr_t)0, hz);
}

dztimer()
{
	register int dz;
	register int s;

	if (dzsilos == 0)
		return;
	s = spl5();
	dzfasttimers++;		/*DEBUG*/
	for (dz = 0; dz < NDZ; dz++)
		if (dzsilos & (1 << dz))
		    dzrint(dz);
	splx(s);
	timeout(dztimer, (caddr_t) 0, dztimerintvl);
}

/*
 * Reset state of driver if UBA reset was necessary.
 * Reset parameters and restart transmission on open lines.
 */
dzreset()
{
	register int unit;
	register struct tty *tp;
	register struct qb_device *ui;

	for (unit = 0; unit < NDZLINE; unit++) {
		ui = dzinfo[unit >> 3];
		if (ui == 0 || ui->ui_alive == 0)
			continue;
		if (unit%8 == 0)
			printf(" dz%d", unit >> 3);
		tp = &dz_tty[unit];
		if (tp->t_state & (TS_ISOPEN|TS_WOPEN)) {
			dzparam(unit);
			(void) dzmctl(unit, DZ_ON, DMSET);
			tp->t_state &= ~TS_BUSY;
#ifdef	SYSV
			map_state(tp, TOSV);
#endif	SYSV
			dzstart(tp);
		}
	}
}
#endif	NDZ > 0
