/*
 * Copyright (c) 1982 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 *
 *	@(#)trap.c	6.8 (Berkeley) 2/23/86
 */

#define TRAPTRACE	0	/* enable and initial value of TRAPTRACE */
#define UTRACE  	0	/* enable and initial value of UTRACE */
/*#define NOFAULT 	1	/* enable NOFAULT TRACE */
#define SYSCALLTRACE	0	/* enable and initial value of SYSCALLTRACE */
/*#define SYSCALL2TRACE	0	/* system call to trace, 0 -> all */
/*#define SCTRACENMI 		/* use NMI to toggle syscall trace */
/*#define TRAPTRACENMI 		/* use NMI to toggle trap trace */

#define	TRAP_TYPE		/* used in "trap.h" include file */

#include "../machine/psl.h"
#include "../machine/reg.h"
#include "../machine/pte.h"
#include "../machine/vmparam.h"

#include "param.h"
#include "systm.h"
#include "dir.h"
#include "user.h"
#ifdef	vax
#include "assym.s"
#endif	vax
#include "proc.h"
#include "seg.h"
#include "vmmac.h"
#include "../machine/trap.h"
#include "../machine/scb.h"
#include "../machine/board.h"
#include "acct.h"
#include "kernel.h"
#ifdef	SYSV
#include "../sysv/sys/systm.h"
#endif	SYSV

#ifdef	SYSCALLTRACE
#include "../sys/syscalls.c"
int	syscalltrace = SYSCALLTRACE;
int	syscallpidtrace = 0;		/* pid of process to trace */
int	syscallpidgetrace = 0;		/* trace all processes with pid >= */
#ifndef	SYSCALL2TRACE
#define	SYSCALL2TRACE	0
#endif	SYSCALL2TRACE
int	syscalltotrace = SYSCALL2TRACE;
#endif	SYSCALLTRACE
#ifdef	TRAPTRACE
int	traptrace = TRAPTRACE;
#endif	TRAPTRACE
#ifdef	UTRACE
int	utrace = UTRACE;
#endif	UTRACE

struct	sysent	sysent[];
int		nsysent;
int		gp_probe = 0;		/* GWS */
static int	haltreboot = 0;
extern int	cvec;			/* autoconf interrupt vector */
extern int	clev;			/* autoconf interrupt level */
extern u_short	*caddr;
extern caddr_t	nofault;		/* pc for fault recovery */

struct stkframe {				/* *** stack frame format *** */
	int		xsf_sp;		/* user stack pointer */
	int		xsf_regs[4];	/* saved temp registers */
	short		xsf_code;	/* trap code (trap dependent) */
	short		xsf_type;	/* trap type (see trap.h) */
	union xsf	xsf;
};
#	define	sp	xsf_sp
#	define	type	xsf_type
#	define	code	xsf_code
#	define	pc	xsf.xf0.xsf_pc
#	define	psl	xsf.xf0.xsf_psl
#	define	vec	xsf.xf0.xsf_vec
#ifdef	M68020
#	define	ssw	xsf.xfa.xsf_ssw_20
#	define	faddr	xsf.xfa.xsf_faddr_20
#	define	ibaddr	xsf.xfb.xsf_ibaddr
#else	M68020
#	define	ssw	xsf.xf8.xsf_ssw_10
#	define	faddr	xsf.xf8.xsf_faddr_10
#endif	M68020

halt_tick()
{
	haltreboot = 0;
}

/*
 * The arguments and local variables within trap and syscall have been declared
 * in such a way as to cause d2-d7 & a2-a5 to be saved.  Furthermore, there
 * can be no local non-register variables within these routines, nor can the
 * compiler allocate any space for intermediate temporaries within the frames.
 * If this is violated, trapcheck will print a message as the kernel is booted.
 */

/*
 * Called from the trap handler when a processor trap occurs.
 */
/*ARGSUSED*/
trap(exsf)
struct stkframe exsf;
{
	register int		*locr0 = ((int *)&exsf.psl)-PS;
	register struct proc 	*p;
	register caddr_t	nf;
	register int		i, f, x, a;
	register long		syst_tv_sec, syst_tv_usec;
	register unsigned int	utype = exsf.type;

	syst_tv_sec = u.u_ru.ru_stime.tv_sec;
	syst_tv_usec = u.u_ru.ru_stime.tv_usec;

	if (USERMODE(locr0[PS])) {
		utype |= USER;
		u.u_ar0 = locr0;
	}
#ifdef	TRAPTRACE
	if (traptrace && utype != T_ASTFLT+USER /* && exsf.type != T_PAGEFLT */)
		tprint(&exsf, utype);
#endif	TRAPTRACE

	switch (utype) {
	    case T_BUSERR:
	    case T_PROTFLT:
	    case T_ADDERR:
		if (nofault) {
	    nofault_recovery:
			/* change pc and inhibit rerun of bus cycle */
			exsf.pc = (int)nofault;
			nofault = 0;
		    	exsf.vec |= ESF_SOFT;
#ifdef	M68020
		    	exsf.ssw &= ESF_SSW_20_SOFTMSK;
#else	M68020
			exsf.ssw |= ESF_SSW_10_RR;
#endif	M68020
#ifdef	NOFAULT
			printf("NOFAULT: new pc=%a, new ssw=%x\n",
				nofault, exsf.ssw);
#endif	NOFAULT
			return;
		}
		/* fall thru */

	    default:
		tprint(&exsf, utype);
		if (exsf.type < NTRAP_TYPES)
			panic(trap_type[exsf.type]);
		panic("?? trap");

	    case T_BUSERR+USER:
	    case T_PROTFLT+USER:
	    case T_ADDERR+USER:
	    case T_FMTERR+USER:
		i = SIGBUS;
		break;

#ifdef	M68020
	    /*
	     * 68020 Bus error handler code:
	     */
	    case T_CNTXFLT+USER:
	    case T_CNTXFLT:
		ctx_alloc();
		return;

	    case T_SEGVIO:
	    case T_SEGVIO+USER:
	    case T_PAGEFLT:
	    case T_PAGEFLT+USER:
		/* actual fault address is not always faddr! */
		x = exsf.ssw;
		f = exsf.vec&ESF_FMTMSK;
		if (x & ESF_SSW_20_DF)
			a = exsf.faddr;
		else if (x & ESF_SSW_20_FB) {
			if (f == ESF_FMTA)
				a = exsf.pc + 4;
			else
				a = exsf.ibaddr;
		} else if (x & ESF_SSW_20_FC) {
			if (f == ESF_FMTA)
				a = exsf.pc + 2;
			else
				a = exsf.ibaddr - 2;
		} else {
			a = exsf.faddr;
			tprint(&exsf, utype);
			panic("segpageflt classification");
		}
		if (((a&MEMR_MASK) == USRV_BASE) && (utype == exsf.type))
			a = a&NONT_MASK;
		if (a & MEMR_MASK) {
			if (nofault)
				goto nofault_recovery;
			if ((utype&USER) == 0) {
				tprint(&exsf, utype);
				panic("system fault");
			}
			i = SIGSEGV;
			break;
		}
		if (exsf.type == T_SEGVIO) {
/* This aspect of bad clasification of bus errors is not understood. CTH */
#define	SPURSEG
#ifdef	SPURSEG
			x = 0;	
#endif	SPURSEG
	    segvio:	if (grow((unsigned)locr0[SP]) || grow(a)) {
				if (utype&USER)
					goto out;
				return;
			}
#ifdef	SPURSEG
			if (x == 0)
				goto pagflt;
#endif	SPURSEG
			if (nofault)
				goto nofault_recovery;
			i = SIGSEGV;
			break;
		} else {
#ifdef	SPURSEG
			x = 1;
	    pagflt:
#endif	SPURSEG
			i = u.u_error;
			nf = nofault;
			nofault = 0;
			if ( pagein(a, 0) == 0) {	/* can't page in */
				nofault = nf;
				u.u_error = i;
#ifdef	SPURSEG
				x = 1;
#endif	SPURSEG
				goto segvio;
			}
#ifdef	SPURSEG
			if (x == 0)
/*				printf("Spur segvio: adr=%a,%x ssw=%x esr=%b ctx=%x\n", 
				    exsf.faddr, svtop(exsf.faddr), exsf.ssw, 
				    *ESR, ESR_BERBITS, ctx) /**/;
#endif	SPURSEG
			nofault = nf;
			u.u_error = i;
			if (utype&USER)
				goto out;
			return;
		}
#else	M68020
	    /*
	     * 68010 Bus error handler code:
	     */
#ifdef	QBUS
#define	CADDR2	((short *)(0x800000 + (int)&u - (2 * NBPG)))
#else	QBUS
#define	CADDR2	((short *)(PAGREGBASE + \
			(((int)&u - (2 * NBPG)) >> (PGSHIFT-PAGREGSHIFT))))
#endif	QBUS
	    /*
	     * If the user SP is above the stack segment, grow the stack 
	     * automatically.  If the stack did not grow, try to page in the
	     * violation, and if that fails then call it a segvio.  Valid 
	     * non-mapped pages will be caught as a SEGVIO if a previous page 
	     * in the segment was faulted in and, in doing that, a page bank 
	     * had to be freed (segfree). All the other pages in that bank are 
	     * marked invalid by newptes even though they may be valid. We try 
	     * to fault those pages in here. 				CTH
	     */
	    case T_SEGVIO+USER:
	    case T_SEGVIO:
		a = exsf.faddr;
		if (a+1 >= USRSTACK)		/* out of range */
			goto segvio;
		if (btop(a) < (u.u_pcb.pcb_p0lr&~AST_CLR) || 
		    btop(a) >= (u.u_pcb.pcb_p1lr&~PME_CLR))
			goto pagflt;
    		if (grow((unsigned)locr0[SP]) || grow(a)) {
			if (utype&USER)
				goto out;
			return;
		}

	    segvio:
		if (nofault)
			goto nofault_recovery;
		i = SIGSEGV;
		if (utype&USER)
			break;
		return;

	    case T_PAGEFLT:		/* allow page faults in kernel mode */
	    case T_SEGFLT:		/* allow seg faults in kernel mode */
	    case T_PAGEFLT+USER:	/* page fault */
	    case T_SEGFLT+USER:		/* segment fault */
		a = exsf.faddr;
	    pagflt:
		i = u.u_error;
		nf = nofault;
		nofault = 0;
		x = *CADDR2;
		if (!pagein(a, 0)) {
			a = (int)CADDR2;
			*(short *)a = x;
			nofault = nf;
			u.u_error = i;
			goto segvio;		/* can't page in */
		}
		a = (int)CADDR2;
		*(short *)a = x;
		nofault = nf;
		u.u_error = i;
		if (utype&USER)
			goto out;
		return;
#endif	M68020

	    case T_PRIVINFLT+USER:
	    case T_RESADFLT+USER:
	    case T_RESOPFLT+USER:
	    case T_COPROC+USER:
		u.u_code = exsf.type;
		i = SIGILL;
		break;

	    case T_ASTFLT+USER:
		astoff();
		if ((u.u_procp->p_flag & SOWEUPC) && u.u_prof.pr_scale) {
			addupc(exsf.pc, &u.u_prof, 1);
			u.u_procp->p_flag &= ~SOWEUPC;
		}
		goto out;

	    case T_FPE+USER:
		u.u_code = fpevec(exsf.vec >> 2);
		i = SIGFPE;
		break;

	    case T_ZERODIV+USER:
		u.u_code = FPE_INTDIV_TRAP;
		i = SIGFPE;
		break;

	    case T_CHK+USER:
		u.u_code = FPE_SUBRNG_TRAP;
		i = SIGFPE;
		break;

	    case T_TRAPV+USER:
		u.u_code = FPE_INTOVF_TRAP;
		i = SIGFPE;
		break;

	    case T_L1010EM+USER:		/* line emulation traps */
	    case T_L1111EM+USER:
		u.u_code = fuword(exsf.pc - 2) & 0xFFFF;
		i = SIGEMT;
		break;

	    case T_TRCTRAP:			/* trace trap in system mode */
		locr0[PS] &= ~PSL_T;
		return;

	    case T_BPTFLT+USER:			/* breakpoint trap */
		locr0[PC] -= 2;
	    case T_TRCTRAP+USER:		/* trace trap */
		locr0[PS] &= ~PSL_T;
		u.u_code = 0;
		i = SIGTRAP;
		break;

	    case T_PARITY:
	    case T_PARITY+USER:
		tprint(&exsf, utype);
#ifdef	M68025
		trap15();
#endif	M68025
		return;

	    case T_NMI:				/* nonmaskable interrupt */
	    case T_NMI+USER:
#ifdef	SCTRACENMI
		syscalltrace ^= 1;
#endif	SCTRACENMI
#ifdef	TRAPTRACENMI
		traptrace ^= 1;
#endif	TRAPTRACENMI
		x = spl7();
#ifdef	M68020
		i = *ESR;
		if ((i&ESR_PARITY_OK) == 0)
			printf ("Parity Error ");
#ifdef	M68025
		if ((i&ESR_UPDATE) == 0)
			printf ("Cache ");
#else	M68025
		if ((i&ESR_AC_OK) == 0)
			printf ("Halt or Power Fail ");
#endif	M68025
#else	M68020
#ifdef	QBUS
		if (exsf.code & BSR_POWERFAIL)
			printf ("Power Fail ");
		if (exsf.code & BSR_BUSPARITY)
			printf ("QBus Parity Error ");
		if (exsf.code & BSR_LOCALPARITY)
			printf ("HSBus Parity Error ");
		if (exsf.code & BSR_HALT) 
			printf ("Halt ");
#else	QBUS
		printf ("Parity Error or Power Fail ");
#endif	QBUS
#endif	M68020
		tprint(&exsf, utype);
		if (haltreboot == 0)
			timeout(halt_tick, (caddr_t)0, hz);
		if (++haltreboot >= 3)
			panic("Three NMI's within one second.");
		splx(x);
		return;

	    case T_TRAP+USER:
		u.u_code = exsf.code;
		i = SIGEMT;
		break;

	    case T_STRAY:
	    case T_STRAY+USER:
		i = (exsf.vec >> 2) & 0xFF;
#ifdef	GWS
		/* 
		 * catch stray interrupts from gip and disable interrupt
		 * driven gip code if iack timeout (hole in bus).
		 */
		if (gp_probe) {
		    if (i == 0x18)
			gpintr(0);
		    return;
		}
#endif	GWS
		/* 
		 * If we are doing 'autoconf' then cvec will be NSCBVEC, and 
		 * should be modified to be the vector at which a controler
		 * interrupted at.  We modify the saved psl so that we return to
		 * the drivers probe routine at high priority.
		 */
		if (cvec == NSCBVEC) {
		    cvec = i;
		    clev = (spl() & PSL_HIGH) >> 8;
		    locr0[PS] |= PSL_HIGH;
		} else
		    printf("Stray interrupt/trap: vector 0x%x level %d\n", 
			i, ((spl() & PSL_HIGH) >> 8)
#ifdef	QBUS
 							+ 3
#endif	QBUS
								);
		return;
	}
#ifdef	UTRACE
	if (utrace && (i == SIGBUS || i == SIGSEGV))
		tprint(&exsf, utype);
#endif	UTRACE
	psignal(u.u_procp, i);

out:	p = u.u_procp;
	if (p->p_cursig || ISSIG(p))
		psig();
	p->p_pri = p->p_usrpri;
	if (runrun) {
		/*
		 * Since we are u.u_procp, clock will normally just change
		 * our priority without moving us from one queue to another
		 * (since the running process is not on a queue.)
		 * If that happened after we setrq ourselves but before we
		 * swtch()'ed, we might not be on the queue indicated by
		 * our priority.
		 */
		(void) spl6();
		setrq(p);
		u.u_ru.ru_nivcsw++;
		swtch();
	}
	if (u.u_prof.pr_scale) {
		register struct timeval *tv = &u.u_ru.ru_stime;

		i = (tv->tv_sec - syst_tv_sec) * 1000;  /* dont generate tmps */
		a = (tv->tv_usec - syst_tv_usec) / 1000;
		i = i + a;
		a = tick / 1000;
		i = i / a;
		if (i)
			addupc(locr0[PC], &u.u_prof, i);
	}
	curpri = p->p_pri;
	asm("	.globl	_trap_end;	_trap_end:");
}

tprint(exsfp, utype)
	struct stkframe *exsfp;
{
	register int f, a, s;

	s = spl7();
	if ((exsfp->type < NTRAP_TYPES) && (exsfp->type >= 0))
		printf("%s %s:",utype&USER ? "USR" : "SYS", 
			trap_type[exsfp->type]);
	else
		printf("TRAP %x:",utype);
	printf(" %s: ", u.u_comm);
	if ((exsfp->pc&MEMR_MASK) == SYSV_BASE)
		printf("pc=%a", exsfp->pc);
	else
		printf("pc=%x", exsfp->pc);
	printf(" psl=%b vec=%x\n", exsfp->psl, PSL_BITS, (exsfp->vec>>2)&0xFF);
	f = exsfp->vec&ESF_FMTMSK;
	if (f == ESF_FMT8 || f == ESF_FMTA || f == ESF_FMTB) {
#ifdef	M68020
		if (exsfp->ssw & ESF_SSW_20_DF)
	    		a = exsfp->faddr;
		else if (exsfp->ssw & ESF_SSW_20_FB) {
	    		if (f == ESF_FMTA)
				a = exsfp->pc + 4;
	    		else
				a = exsfp->ibaddr;
		} else if (exsfp->ssw & ESF_SSW_20_FC) {
	    		if (f == ESF_FMTA)
				a = exsfp->pc + 2;
	    		else
				a = exsfp->ibaddr - 2;
		} else
		    a = exsfp->faddr;
		if (((a&MEMR_MASK) == USRV_BASE) || utype&USER)
			printf("  adr=%x", a);
		else
			printf("  adr=%a,%x", a, svtop(a));
		printf(" ssw=%b esr=%b ctx=%x\n", 
			exsfp->ssw, ESF_SSW_20_BITS, *ESR, ESR_BERBITS, ctx);
#else	M68020
    		a = exsfp->faddr;
		printf("  adr=%a,%x", a, svtop(a));
		printf(" ssw=%b ctx=%x\n", 
			exsfp->ssw, ESF_SSW_10_BITS, (*CXR & 0xFF));
#endif	M68020
	}
	splx(s);
}

/*
 * Called from the trap handler when a system call occurs
 */
/*ARGSUSED*/
syscall(exsf)
struct stkframe exsf;
{
	register int		*locr0 = ((int *)&exsf.psl)-PS;
	register struct timeval *tv = &u.u_ru.ru_stime;
	register struct sysent	*callp;
	register struct proc	*p;
	register int		opc, i, t, x;
	register long		syst_tv_sec, syst_tv_usec;

	u.u_procp->p_systemcall = exsf.code;
	syst_tv_sec = u.u_ru.ru_stime.tv_sec;
	syst_tv_usec = u.u_ru.ru_stime.tv_usec;
	if (!USERMODE(locr0[PS])) {
		panic("syscall");
		opc = i = t = x = 0;
	}

#ifdef	SYSCALLTRACE
	if (syscalltrace && (syscalltotrace==0||syscalltotrace==exsf.code) &&
	    (syscallpidtrace==0||u.u_procp->p_pid==syscallpidtrace) && 
	    (syscallpidgetrace==0||u.u_procp->p_pid>=syscallpidgetrace))
		printf("PID%d:%s:%x: ",u.u_procp->p_pid,u.u_comm,exsf.pc);
#endif	SYSCALLTRACE
	u.u_ar0 = locr0;

#ifdef	SYSV
	if(u.u_procp->p_universe != UNIVERSE_BSD && exsf.code >= 0 
					&& !syscomm(exsf.code)) {
		u.u_code = 1;
		psignal(u.u_procp, SIGEMT);
		goto done;
	}
#endif	SYSV
	if (exsf.code < 0) {
		sigcleanup(exsf.code);
		goto done;
	}
	u.u_error = 0;
	opc = exsf.pc - 2;
	callp = (exsf.code >= nsysent || exsf.code < 0) ? &sysent[63] : &sysent[exsf.code];
	u.u_r.r_val1 = 0;
	u.u_r.r_val2 = locr0[R1];
	if (setjmp(&u.u_qsave)) {
		if (u.u_error == 0 && u.u_eosys != RESTARTSYS)
			u.u_error = EINTR;
	} else {
		u.u_eosys = NORMALRETURN;
#ifdef	SYSCALLTRACE
		if (syscalltrace && (syscalltotrace==0||syscalltotrace==exsf.code) &&
	    	   (syscallpidtrace==0||u.u_procp->p_pid==syscallpidtrace) && 
	    	   (syscallpidgetrace==0||u.u_procp->p_pid>=syscallpidgetrace)){
			if (exsf.code >= nsysent || exsf.code < 0)
				printf("%x(", exsf.code);
			else
				printf("%s(", syscallnames[exsf.code]);
			for (i=0 ; i < callp->sy_narg;) {
				printf("%x", u.u_arg[i]);
				i++;
				if (i < callp->sy_narg)
					printf(",");
			}
			printf(")");
		}
#endif	SYSCALLTRACE
		(*(callp->sy_call))(u.u_ap);
	}
	if (u.u_eosys == NORMALRETURN) {
		if (u.u_error) {
			locr0[R0] = u.u_error;
			locr0[PS] |= PSL_C;	/* carry bit */
		} else {
			locr0[R0] = u.u_r.r_val1;
			locr0[R1] = u.u_r.r_val2;
			locr0[PS] &= ~PSL_C;
		}
	} else if (u.u_eosys == RESTARTSYS)
		exsf.pc = opc;
	/* else if (u.u_eosys == JUSTRETURN) */
		/* nothing to do */
done:
	p = u.u_procp;
	if (p->p_cursig || ISSIG(p))
		psig();
	p->p_pri = p->p_usrpri;
	if (runrun) {
		/*
		 * Since we are u.u_procp, clock will normally just change
		 * our priority without moving us from one queue to another
		 * (since the running process is not on a queue.)
		 * If that happened after we setrq ourselves but before we
		 * swtch()'ed, we might not be on the queue indicated by
		 * our priority.
		 */
		(void) spl6();
		setrq(p);
		u.u_ru.ru_nivcsw++;
		swtch();
	}
	if (u.u_prof.pr_scale) {
		/* dont generate tmps */
		t = (tv->tv_sec - syst_tv_sec) * 1000;
		i = (tv->tv_usec - syst_tv_usec) / 1000;
		x = tick / 1000;
		t += i;
		t /= x;
		if (t)
			addupc(locr0[PC], &u.u_prof, t);
	}
	curpri = p->p_pri;
#ifdef	SYSCALLTRACE
	if (syscalltrace && (syscalltotrace==0||syscalltotrace==exsf.code) &&
	    (syscallpidtrace==0||u.u_procp->p_pid==syscallpidtrace) && 
	    (syscallpidgetrace==0||u.u_procp->p_pid>=syscallpidgetrace))
		printf(" :%d:%x:PID%d\n",u.u_error,exsf.pc,u.u_procp->p_pid);
#endif	SYSCALLTRACE
	asm("	.globl	_syscall_end; _syscall_end:");
}

#ifdef	SYSV
struct	sysv_sysent	sysv_sysent[];
extern	int	nsysv_sysent;

#ifdef	DEBUGGER
int show_error = 0;
#endif	DEBUGGER

/*
 * Called from the trap handler when a System V system call occurs
 */
/*ARGSUSED*/
sysv_syscall(exsf)
struct stkframe exsf;
{
	register int		*locr0 = ((int *)&exsf.psl)-PS;
	register struct timeval *tv = &u.u_ru.ru_stime;
	register struct sysv_sysent	*callp;
	register struct proc	*p;
	register int		i, t, x;
	register long		syst_tv_sec, syst_tv_usec;
#ifdef	RFS
	extern 	int bootstate;
#endif	RFS

	syst_tv_sec = u.u_ru.ru_stime.tv_sec;
	syst_tv_usec = u.u_ru.ru_stime.tv_usec;
	/*	sysinfo.syscall++;
		when this structure becomes available! */
	if (!USERMODE(locr0[PS]))
		panic("syscall");

#ifdef	SYSCALLTRACE
	if (syscalltrace && (syscalltotrace==0||syscalltotrace==exsf.code) &&
	    (syscallpidtrace==0||u.u_procp->p_pid==syscallpidtrace) && 
	    (syscallpidgetrace==0||u.u_procp->p_pid>=syscallpidgetrace))
		printf("SVPID%d:%s:%x: ",u.u_procp->p_pid,u.u_comm,exsf.pc);
#endif	SYSCALLTRACE
	u.u_ar0 = locr0;

	if (u.u_procp->p_universe != UNIVERSE_SYSV && !sysv_syscomm(exsf.code)) {
		u.u_code = 0;
		psignal(u.u_procp, SIGEMT);
		goto done;
	}
	u.u_procp->p_systemcall = exsf.code;
		/* System V keeps the code in u area! */
	u.u_error = 0;
	callp = (exsf.code >= nsysv_sysent-1 || exsf.code < 0) ? 
		&sysv_sysent[nsysv_sysent-1] : &sysv_sysent[exsf.code];
				/* Args were copied in locore.s */
	u.u_r.r_val1 = 0;
	u.u_r.r_val2 = locr0[R1];
	u.u_ap = u.u_arg;

	u.u_eosys = JUSTRETURN;
	i = u.u_sigintr;
	u.u_sigintr = -1;/* All signals can interrupt System V call */
#ifdef	RFS
	if ((bootstate || !callp->sy_setjmp) && setjmp(&u.u_qsave)) {
		if (!(u.u_rflags & U_RSYS) && !u.u_error)
			u.u_error = EINTR;
	} else
		(*callp->sy_call)(u.u_ap);

	u.u_rflags &= ~(U_RSYS | U_DOTDOT | U_LBIN);
#else	RFS
	if ((!callp->sy_setjmp) && setjmp(&u.u_qsave)) {
			u.u_error = EINTR;
	} else
		(*callp->sy_call)(u.u_ap);
#endif	RFS

	/*
	 * System V does not restart system calls if they are interrupted.
	 * So the RESTARTSYS even if set is not honored!!
	 */
#ifdef	notdef
	if (u.u_eosys == SIMULATERTI)
		dorti();
	else
#endif
	if (u.u_error) {
#ifdef	DEBUGGER
		if (show_error)
			printf("syscall %x: err %x: PID %d: %s\n", exsf.code,
				u.u_error, u.u_procp->p_pid, u.u_comm);
#endif	DEBUGGER
		locr0[R0] = u.u_error;
		locr0[PS] |= PSL_C;		/* carry bit */
	} else {
		locr0[PS] &= ~PSL_C;
		locr0[R0] = u.u_r.r_val1;
		locr0[R1] = u.u_r.r_val2;
	}
done:
	p = u.u_procp;
	if (p->p_cursig || ISSIG(p))
		psig();
	p->p_pri = p->p_usrpri;
	if (runrun) {
		/*
		 * Since we are u.u_procp, clock will normally just change
		 * our priority without moving us from one queue to another
		 * (since the running process is not on a queue.)
		 * If that happened after we setrq ourselves but before we
		 * swtch()'ed, we might not be on the queue indicated by
		 * our priority.
		 */
		(void) spl6();
		setrq(p);
		u.u_ru.ru_nivcsw++;
		swtch();
	}
	if (u.u_prof.pr_scale) {
		/* dont generate tmps */
		t = (tv->tv_sec - syst_tv_sec) * 1000;
		i = (tv->tv_usec - syst_tv_usec) / 1000;
		x = tick / 1000;
		t += i;
		t /= x;
		if (t)
			addupc(locr0[PC], &u.u_prof, t);
	}
	curpri = p->p_pri;
#ifdef	SYSCALLTRACE
	if (syscalltrace && (syscalltotrace==0||syscalltotrace==exsf.code) &&
	    (syscallpidtrace==0||u.u_procp->p_pid==syscallpidtrace) && 
	    (syscallpidgetrace==0||u.u_procp->p_pid>=syscallpidgetrace))
		printf(" :%d:%x:SVPID%d\n",u.u_error,exsf.pc,u.u_procp->p_pid);
#endif	SYSCALLTRACE
	asm("	.globl	_sysv_syscall_end; _sysv_syscall_end:");
}
#endif	SYSV

/*
 * nonexistent system call-- signal process (may want to handle it)
 * flag error if process won't see signal immediately
 * Q: should we do that all the time ??
 */
nosys()
{
	if (u.u_signal[SIGSYS] == SIG_IGN || u.u_signal[SIGSYS] == SIG_HOLD)
		u.u_error = EINVAL;
	psignal(u.u_procp, SIGSYS);
}

/*
 * remap floating point exceptions to a linear sequence
 */
int ftrap_vecs[] =  {
	NULL,
	NULL,
	NULL,
	FPE_TRAPCC_VEC,
	FPE_FLINE_VEC,
	FPE_COPROC_VEC,
	FPE_FBSUN_VEC,
	FPE_INEXCT_VEC,
	FPE_FLTDIV_VEC,
	FPE_FLTUND_VEC,
	FPE_FOPERR_VEC,
	FPE_FLTOVF_VEC,
	FPE_FLTNAN_VEC
};

fpevec(val)
	int val;
{
	register int i;

	for (i = 0; i < sizeof(ftrap_vecs)/sizeof(int); i++) 
		if (val == ftrap_vecs[i])
			return i+1;
	printf("unknown fpe trap vector: %x\n", val);
	return FPE_FOPERR_TRAP;
}
