#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/proc.h"
#include "../h/cmap.h"
#include "../h/mount.h"
#include "../h/text.h"
#include "../h/vm.h"
#include "../machine/pte.h"
#include "../machine/mmu.h"
#include "../machine/vmeboard.h"

/* address of segment register given its slot and context */
#define sreg(sl, c)	((u_char *)(SEGREGBASE + \
			(sl << SEGREGSHIFT) + (c << SEGCTXSHIFT) )  )

/* address of page register given its segment and page address */
#define preg(s, v) 	((u_short *)(PAGREGBASE + \
			((s & (NPAGBANKS-1)) << PAGSEGSHIFT) + \
			((v & (NPAGPERSEG-1)) << PAGREGSHIFT) )  )

#define vptosreg(v,c)		sreg(ctos(v), c)
#define vtosreg(v,c)		vptosreg(btoc(v), c)
#define stopreg(s,v)		preg(s, v)
#define vptopreg(v,c)		stopreg(*vptosreg(v,c), v)
#define vtopreg(v,c)		vptopreg(btoc(v), c)
#define IS_VALIDSEG(sl,c)	((*sreg(sl, c) & SEGPROTMASK)!=SEGPROT_NO)
#define IS_INVALIDSEG(sl,c)	((*sreg(sl, c) & SEGPROTMASK)==SEGPROT_NO)
#define VALIDATE_SEG(sl,c,p,s)	(*sreg(sl, c) = p | s)
#define INVALIDATE_SEG(sl,c)	VALIDATE(sl, c, SEGPROT_NO, 0)

extern u_short *TLBmap;
#define pcbaccess(p)	*TLBmap = p->p_addr->pg_pfnum

struct vm_context vm_context[NCTX];
struct vm_segment vm_segment[NUSERSEG];
static char	  vm_prot[16] = {		/* convert vax prot bits */
			SEGPROT_NO, SEGPROT_NO, SEGPROT_ALL, SEGPROT_REO, 
			SEGPROT_ALL, SEGPROT_NO, SEGPROT_NO, SEGPROT_NO, 
			SEGPROT_NO, SEGPROT_NO, SEGPROT_NO, SEGPROT_NO, 
			SEGPROT_NO, SEGPROT_NO, SEGPROT_NO, SEGPROT_REO
};

u_short	cur_ctx;	/* software copy of context register */
u_short ctx_clk;	/* clock used for context allocation algorithms */
u_short seg_clk;	/* clock used for segment allocation algorithms */

extern struct pcb TLBpcb;

/*
 * Allocate a context slot for the current process. This routine is called from
 * resume() after the u area has been setup.  We scan the context table first 
 * looking for a slot we already occupy and secondly for the least recently 
 * used slot. We then setup the slot and point the context register at it.
 * If it was stolen from another process, we have to update the status bits in 
 * the old process' page table.
 */
ctx_alloc()
{
	register u_short i, lru, oldest;
	register struct vm_context *vc, *lruc;
	register struct proc *p = u.u_procp;

	if (p->p_flag & SSYS)
		return;
	lru = 1;
	vc = &vm_context[1];
	oldest = vc->vc_lastaccess;
	for (i=1; i<NCTX; i++, vc++)
		if (vc->vc_occupant == p) {
			lru = i;
			break;
		} else if (vc->vc_lastaccess < oldest) {
			lru = i;
			oldest = vc->vc_lastaccess;
		}
	vc = &vm_context[lru];
	if (vc->vc_occupant != p && oldest != 0 )
		tbiap(vc->vc_occupant);
	vc->vc_occupant = p;
	if (++ctx_clk == 0)
		ctx_clk = 1;
	vc->vc_lastaccess = ctx_clk;
	*CXR = cur_ctx = lru;
}

/*
 * Allocate a hardware segment. This routine is called from the page fault 
 * handler after it has been determined that the fault was in-bounds but had 
 * no hardware segment available to map it. We scan the segment table to find 
 * the segment that has been allocated the longest. We then steal the segment 
 * from the current owner and flush the status bits into the old owners RAM 
 * page table. If the allocation time on the segment is zero, then the segment 
 * is free for the taking and no bookkeeping is neccessary on the old owners 
 * page table.
 */
seg_alloc(vaddr)
u_int vaddr;			/* fault address */
{
	register u_short i, lru, oldest;
	register struct vm_segment *vs;

	if (++ctx_clk == 0) 
		ctx_clk = 1;
	vm_context[cur_ctx].vc_lastaccess = ctx_clk;
	lru = 0;
	oldest = -1;
	vs = &vm_segment[0];
	for (i = 0; i < NUSERSEG; i++, vs++)
		if (vs->vs_lastalloc == 0) {
			lru = i;
			oldest = 0;
			break;
		} else if (vs->vs_lastalloc < oldest) { 
			lru = i;
			oldest = vs->vs_lastalloc;
		}
	vs = &vm_segment[lru];
	if (oldest != 0) {
		pcbaccess(vm_context[vs->vs_pcontext].vc_occupant);
		tbiseg(vs->vs_psegslot, lru, vs->vs_pcontext);
	}
	vs->vs_pcontext = cur_ctx;
	vs->vs_psegslot = ctos(btoc(vaddr));
	tbvseg(vs->vs_psegslot, lru, cur_ctx);
}

/*
 * Free the segment at segslot in the current context. Flush the status bits 
 * into the owners page table and clear the hardware bits. Point all the pages 
 * to the no access page. Then point the segment to the no access segment.
 * It is assume that the owners pcb is mapped to TLBpcb.
 */
tbiseg(segslot, seg, ctx)
register u_short segslot, seg;
{
	register u_int v, i;
	register struct pte *pte, *endpte;
	register u_short pr;

	if (IS_INVALIDSEG(segslot, ctx))
		panic("dup tbiseg");
	v = stoc(segslot);
	if (v < TLBpcb.pcb_p0lr) {
		pte = TLBpcb.pcb_p0br + v;	/* appears in p0 space */
		endpte = TLBpcb.pcb_p0br + TLBpcb.pcb_p0lr;
		for (i=0; i<NPAGPERSEG; i++) {
			if (pte >= endpte)
				break;
			pr = *vptopreg(v, ctx);
			if ( pr & (1 << PAGMODBIT) )
				pte->pg_m = 1;
			if ( pr & (1 << PAGREFBIT) )
				pte->pg_r = 1;
			pte++; v++;
		}
	} else if ((v += NPAGPERSEG - 1) >= TLBpcb.pcb_p1lr) {
		pte = TLBpcb.pcb_p1br + v;	/* appears in p1 space */
		endpte = TLBpcb.pcb_p1br + TLBpcb.pcb_p1lr;
		for (i=0; i<NPAGPERSEG; i++) {
			if (pte < endpte)
				break;
			pr = *vptopreg(v, ctx);
			if ( pr & (1 << PAGMODBIT) )
				pte->pg_m = 1;
			if ( pr & (1 << PAGREFBIT) )
				pte->pg_r = 1;
			pte--; v--;
		}
	} else
		panic("tbiseg");
	INVALIDATE_SEG(segslot, ctx);
	vm_segment[seg].vs_lastalloc = 0;
}


/*
 * Validate the segment at segslot in the current context. Map all of the
 * page frames and protections of this section of the pte into the hardware.
 */
tbvseg(segslot, seg, ctx)
register u_short segslot, seg;
{
	register u_int v, i;
	register struct pte *pte, *endpte;
	register u_char *sr;
	register u_short *pr;

	if (IS_VALIDSEG(segslot, ctx))
		panic("dup tbvseg");
	if (++seg_clk == 0) 
		seg_clk = 1;
	vm_segment[seg].vs_lastalloc = seg_clk;
	v = stoc(segslot);
	if (v < u.u_pcb.pcb_p0lr) {
		pte = u.u_pcb.pcb_p0br + v;
		endpte = u.u_pcb.pcb_p0br + u.u_pcb.pcb_p0lr;
		VALIDATE_SEG(segslot, ctx, vm_prot[pte->pg_prot], seg);
		for (i=0; i<NPAGPERSEG; i++, pte++, v++) {
			pr = vptopreg(v, ctx);
			if (pte >= endpte)
				*pr = (1 << PAGINVBIT);
			else
				tbvs(pte, pr);
		}
	} else if ((v += NPAGPERSEG - 1) >= u.u_pcb.pcb_p1lr) {
		pte = u.u_pcb.pcb_p1br + v;
		endpte = u.u_pcb.pcb_p1br + u.u_pcb.pcb_p1lr;
		VALIDATE_SEG(segslot, ctx, vm_prot[pte->pg_prot], seg);
		for (i=0; i<NPAGPERSEG; i++, pte--, v--) {
			pr = vptopreg(v, ctx);
			if (pte < endpte)
				*pr = (1 << PAGINVBIT);
			else
				tbvs(pte, pr);
		}
	} else
		panic("tbvseg");
}

/*
 * Validate the hardware map for the given pte.
 */
tbvs(pte, pr)
register struct pte *pte;
register u_short *pr;
{
	if (pte->pg_v)
		*pr = pte->pg_pfnum;
	else
		*pr = (1 << PAGINVBIT) | (1 << PAGOUTBIT);
}

/*
 * Validate user virtual address for the given process. Function name means 
 * Translation Buffer Ivalidate Single for Process. This name comes from the vax
 * model; the actual operation is to validate, but that's not important now.
 * This function is called after a page has been re-mapped and the status bits 
 * are tossed.
 */
tbisp(p, pte, v)
register struct proc *p;
register struct pte *pte;
register u_int v;
{
	register u_short *pr;
	register u_char sr;
	register int i;
	register struct vm_context *vc = &vm_context[0];

	for (i=1; i<NCTX; i++, vc++)
		if (vc->vc_occupant == p) {
			sr = *vptosreg(v, i);
			if ((sr & SEGPROTMASK) != SEGPROT_NO)
				tbvs(pte, stopreg(sr, v));
			break;
		}
}

/*
 * Translation Buffer Invalidate All for Process. This routine could use another
 * argument to indicate whither or not to salvage the status table bits. For 
 * exit() and exec(), the status bits are not useful, for swapout() the modified
 * bit is mandatory. Will have to add the saving of the reference bit later.
 */
tbiap(p)
register struct proc *p;
{
	register u_short i, j;
	register struct vm_segment *vs;
	register struct vm_context *vc = &vm_context[0];

	for (i=1; i<NCTX; i++, vc++)
		if (vc->vc_occupant == p) {
			pcbaccess(p);
			for (j=0, vs = &vm_segment[0]; j<NUSERSEG; j++, vs++)
			    if (vs->vs_pcontext == i && vs->vs_lastalloc != 0)
				tbiseg(vs->vs_psegslot, j, i);
			break;
		}
}

/*
 * Pull the status bits through the TLB and into the RAM pte. This routine must
 * be called before examining the dirty and referenced bits; such as in the 
 * macro dirtycl().
 */
getstatusbits(p, pte, v)
register struct proc *p;
register struct pte *pte;
register u_int v;
{
	register u_char sr;
	register u_short pr;
	register u_short i;
	register struct vm_context *vc = &vm_context[0];

	for (i=1; i<NCTX; i++, vc++)
		if (vc->vc_occupant == p) {
			sr = *vptosreg(v, i);
			if ((sr & SEGPROTMASK) != SEGPROT_NO) {
				pr = *stopreg(sr, v);
				if ( pr & (1 << PAGMODBIT) )
					pte->pg_m = 1;
				if ( pr & (1 << PAGREFBIT) )
					pte->pg_r = 1;
			}
			break;
		}
}

cleardirtybit(p, pte, v)
register struct proc *p;
register struct pte *pte;
register u_int v;
{
	register u_char sr;
	register u_short *pr;
	register u_short i;
	register struct vm_context *vc = &vm_context[0];

	for (i=1; i<NCTX; i++, vc++)
		if (vc->vc_occupant == p) {
			sr = *vptosreg(v, i);
			if ((sr & SEGPROTMASK) != SEGPROT_NO)
				*stopreg(sr, v) &= ~(1 << PAGMODBIT);
			break;
		}
}

clearrefbit(p, pte, v)
register struct proc *p;
register struct pte *pte;
register u_int v;
{
	register u_char sr;
	register u_short i;
	register struct vm_context *vc = &vm_context[0];

	for (i=1; i<NCTX; i++, vc++)
		if (vc->vc_occupant == p) {
			sr = *vptosreg(v, i);
			if ((sr & SEGPROTMASK) != SEGPROT_NO)
				*stopreg(sr, v) &= ~(1 << PAGREFBIT);
			break;
		}
}

clearstatusbits(vaddr)
register u_int vaddr;
{
	register u_short *pr;

	pr = vtopreg(vaddr, cur_ctx);
	*pr &= ~((1 << PAGREFBIT) | (1 << PAGMODBIT));
}


segvalid(vaddr)
register u_int vaddr;
{
	register u_char *sr;

	sr = vtosreg(vaddr, cur_ctx);
	return(!(*sr&SEG_INVALID));
}
