/* windLib.c - internal VxWorks kernel library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01d,19nov88,jcf  validated task id before operations undertaken.
		   changed lstFirst, lstPrevious, lstNext to macros.
01c,23aug88,gae  documentation.
01b,12aug88,jcf	 cleanup.
01a,12mar88,jcf	 written.
*/

/*
DESCRIPTION
This module contains the internal routines of the VxWorks kernel.
These routines assume that they have exclusive access
to the kernel data structures.

INTERNAL
Exclusion is achieved by the higher level routines by setting the
global variable kernelState.  When kernelState is TRUE, the function
of interest is not invoked directly, but rather the work is enqueued to be
done by the task (or interrupt service routine) which set kernelState to
TRUE.  This approach is best described as "the last person out of the room
must turn out the light".

SEE ALSO: windALib (1)
*/

#include "vxWorks.h"
#include "wind.h"
#include "taskLib.h"

/* Macro's are used within wind to avoid costly breakpoint overhead for
 * these common routines.
 */
#define LST_NEXT(pNode)		(((NODE *)(pNode))->next)
#define LST_PREVIOUS(pNode)	(((NODE *)(pNode))->previous)
#define LST_FIRST(pList)	(((LIST *)(pList))->node.next)

/* imports */
IMPORT ULONG vxTicks;			/* current time in ticks */
IMPORT ULONG taskIdCurrent;		/* current executing task */
IMPORT FUNCPTR taskSwitchTable [];	/* task switch routine to be run */
IMPORT LIST activeList;			/* active list of TCBs*/
IMPORT LIST readyList;			/* ready list of TCBs */
IMPORT LIST delayList;			/* delayed list of TCBs */
IMPORT LIST workList;			/* list of jobs */
IMPORT LIST freeJobList;		/* free list of jobs */


/* globals */

BOOL kernelState;	/* whether we are in a critical section or not */
BOOL contextSwitching;	/* whether we are context switching or not */
int vxLastTask;		/* to remember the task run before context switch */

/* forward declarations */

LOCAL WIND_TCB *windPriListGet ();


/*******************************************************************************
* 
* windSpawn - create a task and leave it in the suspend state
*
* This routine simply inserts the specified tcb into the active list.
* A task is created in the suspended state.
*
* RETURNS: OK
*
* NOMANUAL
*/  

STATUS windSpawn (pTcb)
    FAST WIND_TCB *pTcb;	/* address of new task's tcb */

    {
    windTcbListPut (&activeList, pTcb);	/* put in active list */

    return (OK);
    }
/*******************************************************************************
*
* windSuspend - suspend a task
*
* Suspension is an additive state.  When a task is on the ready list it
* is removed and changed to suspended.  Otherwise, the status is updated
* to include suspended.
*
* RETURNS: OK
*
* NOMANUAL
*/ 

STATUS windSuspend (pTcb)
    FAST WIND_TCB *pTcb;	/* address of task's tcb, 0 = current task */

    {
    if (taskIdVerify ((int) pTcb) != OK)
	return (ERROR);				/* task not found */

    if (pTcb->status == WIND_READY)
	lstDelete (&readyList, &pTcb->priNode);  /* take off ready list */
    pTcb->status |= WIND_SUSPEND;		   /* update status */

    return (OK);
    }
/*******************************************************************************
*
* windResume - resume a task
*
* Resume the specified task.
*
* RETURNS: OK
*
* NOMANUAL
*/ 

STATUS windResume (pTcb)
    FAST WIND_TCB *pTcb;	/* address of task's tcb, 0 = current task */

    {
    if (taskIdVerify ((int) pTcb) != OK)
	return (ERROR);				/* task not found */

    if (pTcb->status == WIND_SUSPEND)	/* if just suspended */
	windPriListPut (&readyList, pTcb);

    pTcb->status &= ~WIND_SUSPEND;	/* its on some other list too! */
    return (OK);
    }
/*******************************************************************************
*
* windPrioritySet - set the priority of a task
*
* Set the priority of a task.  The list that task is on (if any) is reordered.
*
* RETURNS: OK
*
* NOMANUAL
*/  

STATUS windPrioritySet (pTcb, priority)
    FAST WIND_TCB *pTcb;	/* address of task's tcb, 0 = current task */
    int priority;		/* new priority */

    {
    if (taskIdVerify ((int) pTcb) != OK)
	return (ERROR);				/* task not found */

    pTcb->priority = priority;			/* set the priority */

    if (pTcb->status == WIND_READY)		/* if ready */
	{
	lstDelete (&readyList, &pTcb->priNode);
	windPriListPut (&readyList, pTcb);
	}
    else if (pTcb->status & WIND_PEND)		/* if pended */
	{
	lstDelete (&pTcb->semId->semList, &pTcb->priNode);
	windPriListPut (&pTcb->semId->semList, pTcb);
	}

    return (OK);
    }
/*******************************************************************************
*
* windDelete - delete a task
*
* Delete task and reorganize delay list if task was delayed.
*
* RETURNS: OK
*
* NOMANUAL
*/  

STATUS windDelete (pTcb)
    FAST WIND_TCB *pTcb;	/* address of task's tcb */

    {
    NODE *pNode;

    if (taskIdVerify ((int) pTcb) != OK)
	return (ERROR);				/* task not found */

    if (pTcb->status == WIND_READY)		/* was on ready list */
	lstDelete (&readyList, &pTcb->priNode);
    else if (pTcb->status & WIND_PEND)		/* was on sem list */
	lstDelete (&pTcb->semId->semList, &pTcb->priNode);
    else if (pTcb->status & WIND_DELAY)		/* was on delay list */
	{
	/* if it was on the delay list, add dead guys ticks to next task */
	pNode = LST_NEXT (&pTcb->priNode);	/* get next delayed task */

	if (pNode != NULL)
	    {
	    ((WIND_TCB *)
	     ((int)pNode - OFFSET (WIND_TCB, priNode)))->delay += pTcb->delay;
	    }

	lstDelete (&delayList, &pTcb->priNode);
	}

    pTcb->status = WIND_DEAD;	        	/* mark as dead */
    pTcb->valid = 0;				/* invalidate */
    lstDelete (&activeList, &pTcb->tcbNode);	/* deactivate */

    return (OK);
    }
/*******************************************************************************
*
* windSemGive - give a semaphore
*
* Give a semaphore and unblock highest priority task pending on this semaphore.
*
* NOMANUAL
*/  

VOID windSemGive (semId)
    FAST SEM_ID semId;		/* semaphore to give */

    {
    FAST WIND_TCB *pTcb = (WIND_TCB *) ((int)lstGet (&semId->semList) -
					OFFSET (WIND_TCB, priNode));
    if (pTcb != NULL)
	{
	if (pTcb->status == WIND_PEND)		/* pend only */
	    windPriListPut (&readyList, pTcb);

	pTcb->status &= ~WIND_PEND;		/* on some other list */
	}
    else
	semId->count = 1;			/* make available */
    }
/*******************************************************************************
*
* windSemTake - take a semaphore
* 
* Take a semaphore.  If semaphore is not available, then wait for it.
*
* NOMANUAL
*/  

VOID windSemTake (semId)
    FAST SEM_ID semId;		/* semaphore to take */

    {
    if (semId->count == 0)
	{
	/* if task was ready, delete from ready list */
	if (((WIND_TCB *) taskIdCurrent)->status == WIND_READY)
	    lstDelete (&readyList, &(((WIND_TCB *) taskIdCurrent)->priNode));

	((WIND_TCB *) taskIdCurrent)->status |= WIND_PEND;  /* update status */
	((WIND_TCB *) taskIdCurrent)->semId = semId;	    /* keep semId */

	/* insert task on waiting list in order of priority */
	windPriListPut (&semId->semList, (WIND_TCB *) taskIdCurrent);
	}
    else
	semId->count = 0;			/* make unavailable */
    }
/*******************************************************************************
*
* windTickAnnounce - acknowledge the passing of time
*
* Process delay list.  Make tasks at the end of their delay  active.
* Perform round robin scheduling if selected.
*
* NOTE:
* Round-robin scheduling not implemented.
*
* NOMANUAL
*/  

VOID windTickAnnounce ()

    {
    FAST WIND_TCB *pTcb;
    FAST NODE *pNode = LST_FIRST (&delayList);

    vxTicks++;		/* advance time */

    if (pNode != NULL)
	{
	/* only decrement first one in delay list */

	((WIND_TCB *) ((int)pNode - OFFSET (WIND_TCB, priNode)))->delay--;
	}

    while (pNode != NULL)
	{
	pTcb = (WIND_TCB *) ((int)pNode - OFFSET (WIND_TCB, priNode));

	if (pTcb->delay != 0)
	    break;				/* still delaying... bye */

	pTcb->status &= ~WIND_DELAY;	    	/* undelay it */
	pNode = LST_NEXT (pNode);		/* save pNode of next task */
						/* 'cause it might wakeup too */
	lstDelete (&delayList, &pTcb->priNode);	/* get off delay list */

	if (pTcb->status == WIND_READY)	    	/* if ready */
	    windPriListPut (&readyList, pTcb);  /* insert in ready list */
	}

    /* take care of timeouts for semaphores */
	/* NOT IMPLEMENTED YET */

    /* If RRS then deal with it */
	/* NOT IMPLEMENTED YET */
    }
/*******************************************************************************
*
* windDelay - put running task asleep for specified ticks
*
* Insert task in delay list at correct spot dictated by the specified duration.
*
* RETURNS: OK
*
* NOMANUAL
*/  

STATUS windDelay (ticks)
    int ticks;

    {
    FAST WIND_TCB *pTcb = windPriListGet (&readyList);	/* get off ready list */
    FAST NODE *pNode = LST_FIRST (&delayList);

    pTcb->status |= WIND_DELAY;				/* delayed */

    while (pNode != NULL)				/* empty list? */
        {
	if (ticks < ((WIND_TCB *)
		     ((int)pNode - OFFSET (WIND_TCB, priNode)))->delay)
	    {
	    /* We've reached the place in the delay list to insert this task. */

	    lstInsert (&delayList, LST_PREVIOUS (pNode), &pTcb->priNode);

	    /* Now decrement the delay field of the guy behind us. */

	    ((WIND_TCB *)
	     ((int)pNode - OFFSET (WIND_TCB, priNode)))->delay -= ticks;

	    pTcb->delay = ticks;	/* enter delay field */
	    return (OK);		/* We're done. */
	    }
	ticks -= ((WIND_TCB *)((int)pNode - OFFSET (WIND_TCB, priNode)))->delay;
	pNode = LST_NEXT (pNode);
	}

    /* We fall through while loop if delay is longest in the list */

    pTcb->delay = ticks;			/* enter delay field */
    lstAdd (&delayList, &pTcb->priNode);	/* add to end of delay list */

    return (OK);
    }

/*******************************************************************************
*
* windTcbListPut - put a tcb on the FIFO list specified
*
* The node in the tcb used for this routine is pTcb->tcbNode
* The only list which uses this field is the activeList.
*/  

LOCAL VOID windTcbListPut (pList, pTcb)
    LIST *pList;
    WIND_TCB *pTcb;

    {
    lstAdd (pList, &pTcb->tcbNode);
    }
/*******************************************************************************
*
* windPriListGet - get the highest priority tcb from the list specified
*
* The node in the tcb used for this routine is pTcb->priNode.
* The only lists which use this field are the readyList and the blockList.
*
* RETURNS: pTcb or NULL if empty list
*/  

LOCAL WIND_TCB *windPriListGet (pList)
    LIST *pList;

    {
    FAST int first = (int) lstGet (pList);

    if (first == NULL)
	return (NULL);

    return ((WIND_TCB *) (first - OFFSET (WIND_TCB, priNode)));
    }
/*******************************************************************************
*
* windPriListPut - put the highest priority tcb on the list specified
*
* The node in the tcb used for this routine is pTcb->priNode.
* The only lists which use this field are the readyList and the blockList.
*/  

LOCAL VOID windPriListPut (pList, pTcb)
    LIST *pList;
    WIND_TCB *pTcb;

    {
    NODE *pNode = LST_FIRST (pList);

    while (pNode != NULL)
        {
    	if (((WIND_TCB *) ((int)pNode - OFFSET (WIND_TCB, priNode)))->priority
	    >= (pTcb->priority))
	    {
	    pNode = LST_PREVIOUS (pNode);
	    lstInsert (pList, pNode, &pTcb->priNode);
	    return;
	    }
	pNode = LST_NEXT (pNode);
	}

    lstAdd (pList, &pTcb->priNode);
    }
/*******************************************************************************
*
* windWorkAdd - add functions to the wind work queue
*
* When kernel facilities are called from interrupt level they are queued in
* a work list.  The work list is emptied by reschedule (2).
*
* RETURNS: OK
*
* SEE ALSO: reschedule (2)
*
* NOMANUAL
* VARARGS
*/  

STATUS windWorkAdd (func, arg1, arg2)
    FUNCPTR func;	/* function to invoke */
    int arg1;		/* parameter one to function */
    int arg2;		/* parameter two to function */

    {
    int level = intLock ();	/* LOCK INTERRUPTS */
    FAST JOB *pJob = (JOB *) lstGet (&freeJobList);

    intUnlock (level);		/* UNLOCK INTERRUPTS */

    pJob->function = func;		/* fill in function */

    pJob->arg1 = arg1;			/* fill in arguments */
    pJob->arg2 = arg2;

    level = intLock ();		/* LOCK INTERRUPTS */
    lstAdd (&workList, &pJob->node);	/* add fuction to queue. */
    intUnlock (level);		/* UNLOCK INTERRUPTS */

    return (OK);
    }
/*******************************************************************************
*
* idle - idle
*
* This task is created at lowest priority.  We drop to here if
* nothing else to do.
*
* NOMANUAL
*/

VOID idle ()

    {
    FOREVER;
    }
/*******************************************************************************
*
* windDummyCreate - create a dummy task
*
* This routine sets up an initial "fake" TCB for the kernel.
*
* RETURNS: dummy task id
*
* NOMANUAL
*/

int windDummyCreate ()

    {
    static WIND_TCB dummyTcb;	/* context for kernelInit */
    FAST TCBX *pTcbX;		/* pointer to tcb extension */
    int ix;			/* index */

    dummyTcb.priority	= 255;			/* set priority */
    dummyTcb.name	= "dummy";		/* set name */
    dummyTcb.valid	= ((((int) &dummyTcb) & 0xffff0000) | 
		           ((~(int) &dummyTcb) & 0x0000ffff));
    dummyTcb.status	= WIND_READY;		/* set status */
    dummyTcb.mode	= 0;			/* set mode */
    dummyTcb.delay	= 0;			/* delay left in ticks */
    dummyTcb.tslice	= 0;			/* slice left in ticks */
    dummyTcb.foroff	= 0;			/* format/offset for 68020 */
    dummyTcb.sr		= 0x3000;		/* SR with M + S */

    pTcbX = &dummyTcb.tcbx;

    pTcbX->entry	= (FUNCPTR) 0x0;
    pTcbX->errorStatus	= OK;
    pTcbX->topOfStack	= (char *) 0x0;
    pTcbX->botOfStack	= (char *) 0x0;
    pTcbX->initialSP	= (char *) 0x0;
    pTcbX->options	= VX_SUPERVISOR_MODE;
    pTcbX->name 	= "dummy";
    pTcbX->taskVar 	= NULL;			/* task variables */
    pTcbX->fpContext	= NULL;			/* floating point context */
    pTcbX->taskId	= (int) &dummyTcb;	/* the taskId for this task */
    pTcbX->taskTicks	= 0;			/* total number of ticks */
    pTcbX->taskIncTicks	= 0;			/* number of ticks in slice */

    for (ix = 0; ix < 3; ix++)
	pTcbX->taskStd [ix] = ix;

    /* put in active list */
    windTcbListPut ((LIST *) &activeList, &dummyTcb);

    /* put in ready list */
    windPriListPut ((LIST *) &readyList, &dummyTcb);

    taskIdCurrent = (int) &dummyTcb;

    return (taskIdCurrent);
    }
