/* taskLib.c - task management interface for the VxWorks kernel */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01f,19nov88,jcf  taskDelay can not be called from interrupt level.
                   task ids of zero are translated to taskCurrentId here now.
01e,23sep88,gae  documentation touchup.
01d,07sep88,gae  documentation.
01c,23aug88,gae  documentation.
01b,20jul88,jcf  clean up.
01a,10mar88,jcf  written.
*/

/*
DESCRIPTION
This library provides a generic interface to the VxWorks kernel
task management routines.  Task management includes:

    - create, delete, suspend, resume, restart, prioritize tasks,
    - lock and unlock scheduling,
    - delay a task from running,
    - get and set tasks' registers,
    - get and set tasks' options,
    - find TCBs (task control blocks),
    - find TCBXs (task control block extensions),
    - find TDs (kernel independent task descriptors),
    - translate task ids to task names, and vice versa.

INCLUDE FILE: taskLib.h

SEE ALSO:
taskHookLib (1), taskVarLib (1), semLib (1), kernelLib (1), "Architecture"
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "taskLib.h"
#include "memLib.h"
#include "strLib.h"
#include "wind.h"

IMPORT VOID vxTaskEntry ();
IMPORT FUNCPTR taskCreateTable [];
IMPORT FUNCPTR taskDeleteTable [];
IMPORT BOOL kernelState;
IMPORT int contextSwitching;
IMPORT LIST activeList;

/* globals */

/* taskIdCurrent is modified on each context switch to be the task id of 
 * the running task; it is referred to by taskId to get the taskId
 * of the running task
 */

ULONG taskIdCurrent;

/* locals */

LOCAL int nameForNameless = 1;	/* name for the nameless tasks spawned */
LOCAL FUNCPTR taskBpHook;	/* rtn to install/remove bkpts for task */

/* forward declarations */

WIND_TCB *taskTcb ();
TCBX *taskTcbX ();
STATUS windSpawn ();
STATUS windDelete ();
STATUS windResume ();
STATUS windSuspend ();
STATUS windDelay ();
STATUS windPrioritySet ();


/*******************************************************************************
*
* taskBpHookSet - set breakpoint hook for dbgLib
*
* This routine allows dbgLib to install its break-point install/remove routine
* used by taskOptionsSet. It should only be called by dbgInit (2).
*
* NOMANUAL
*/

VOID taskBpHookSet (bpHook)
    FUNCPTR bpHook;

    {
    taskBpHook = bpHook;
    }
/*******************************************************************************
*
* taskSpawn - spawn a task
*
* This routine creates a new task with the given priority, id, and options.
* If the task name is specified as NULL, the name will be an ASCII
* number which increments as tasks are spawned
*
* Bits in the options argument may be set to run with the following modes.
* .CS
*	VX_SUPERVISOR_MODE	-  execute in supervisor mode
*	VX_UNBREAKABLE		-  don't allow break point debugging
*	VX_DEALLOC_STACK	-  deallocate the stack on deletion
*	VX_FP_TASK		-  execute with coprocessor support
*	VX_STDIO		-  execute with standard I/O support
* .CE
* See definitions in taskLib.h.
*
* The entry address is the address of the main routine of the task.
* This routine will be called with the specified arguments (up to 10).  
* A stack of the specified size will be allocated from the memory pool.  
* Stack size should be even.  Every byte of this stack is initialized to 0xEE, 
* as a debugging aid.  See checkStack (2) for stack size checking aids.
* 
* See taskCreate (1) for creating tasks with preallocated stacks.
*
* BUGS:
* All tasks are spawned in supervisor mode, regardless of 
* the specified option.  In effect, VX_SUPERVISOR_MODE is or'd into the
* user specified options.  This is done because other VxWorks facilities
* do not make a distinction of privileged instructions.
*
* RETURNS: task id, or ERROR if out of memory or unable to create task
*
* SEE ALSO: taskCreate (2), taskActivate (2)
*
* VARARGS5
*/

int taskSpawn (name, priority, options, stacksize, entryAddress, arg1,
	       arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10)
    char *name;			/* task name of new task     */
    int priority;		/* priority of new task      */
    int options;		/* options word for new task */
    int stacksize;		/* size (in bytes) of stack  */
    FUNCPTR entryAddress;	/* entry point of task       */
    int arg1;			/* arguments passed to task  */
    int arg2;
    int arg3;
    int arg4;
    int arg5;
    int arg6;
    int arg7;
    int arg8;
    int arg9;
    int arg10;

    {
    char *topOfStack;
    char *bottomOfStack;
    char newName [12];
    char *mallocName; /* malloced name */
    int tid;

    /* name the nameless */

    if ((int) name == NULL)	/* name the nameless */
	{
	sprintf (newName, "%x", nameForNameless++);
	name = newName;
	}

    /* allocate memory for stack, tcb extension and name. */
    /* Fill it with 0xee for debugging, and checkStack (2) */

    topOfStack = malloc ((unsigned) (stacksize + sizeof (WIND_TCB)
				              + strlen (name) + 1));

    if (topOfStack == NULL)
	return (ERROR);		/* egads! we're out of memory */

    bottomOfStack = topOfStack + stacksize;

    mallocName = (char *)(((int) bottomOfStack) + sizeof (WIND_TCB));

    strcpy (mallocName, name);
    
    bfill (topOfStack, (int) stacksize, 0xee);

    tid = taskCreate (mallocName, priority, options | VX_DEALLOC_STACK,
		      bottomOfStack, stacksize, (WIND_TCB *) bottomOfStack,
		      entryAddress, arg1, arg2, arg3, arg4,
		      arg5, arg6, arg7, arg8, arg9, arg10);

    if (tid == ERROR)
	free (topOfStack);
    else
	{
	if (taskActivate (tid) == ERROR)
	    return (ERROR);
	}

    return (tid);
    }
/*******************************************************************************
*
* taskCreate - create a task with stack at specified address
*
* This routine works like taskSpawn (2),
* but instead of allocating the stack and TCB extension automatically,
* it uses the stack and TCB extension passed as arguments.
* The stack will grow towards lower memory locations
* starting from the specified bottom of stack.
*
* Bits in the options argument may be set to run with the following modes.
* .CS
*	VX_SUPERVISOR_MODE	-  execute in supervisor mode
*	VX_UNBREAKABLE		-  don't allow debugging
*	VX_DEALLOC_STACK	-  deallocate the stack on deletion
*	VX_FP_TASK		-  execute with coprocessor support
*	VX_STDIO		-  execute with standard I/O support
* .CE
* See definitions in taskLib.h.
*
* The name of a task should be malloc'd along with the stack and TCB extension.
* taskSpawn (2) will automatically name an unnamed task but taskCreate will not.
* Normally, tasks should be started by taskSpawn (2), not by taskCreate.
*
* BUGS:
* All tasks are spawned in supervisor mode, regardless of 
* the specified option.  In effect, VX_SUPERVISOR_MODE is or'd into the
* user specified options.  This is done because other VxWorks facilities
* do not yet make the distinction of privileged instructions.
*
* RETURNS: task id or ERROR if unable to create task
*
* SEE ALSO: taskActivate (2), taskSpawn (2)
*
* VARARGS7
*/

int taskCreate (name, priority, options, botOfStack, stksize, pTcb,
		entryAdrs, arg1)
    char *name;			/* name of new task           */ 
    int priority;		/* priority of new task       */
    int options;		/* task option word           */
    FAST char *botOfStack;	/* bottom of new task's stack */
    int stksize;		/* size (bytes) of stack      */
    FAST WIND_TCB *pTcb;	/* address of new task's TCB  */
    FUNCPTR entryAdrs;		/* initial pc of new task     */
    int arg1;			/* 1st of up to 10 arguments  */

    {
    FAST int *argptr;
    FAST int *sp;
    FAST int ix;
    FAST TCBX *pTcbX;		/* address of new task's tcbX */
    STATUS status;

    if ((int) name == NULL)
	return (ERROR);

    options |= VX_SUPERVISOR_MODE;	/* TEMP! set SUP mode for all tasks */

    /* push args on the stack */

    argptr = &arg1 + MAX_TASK_ARGS;
    sp = (int *) botOfStack;

    for (ix = 0; ix < MAX_TASK_ARGS; ix++)
	*--sp = *--argptr;

    /* initialize tcb */

    pTcb->ssp      = (char *) sp;		/* initial stack pointer */
    pTcb->sr       = 0x3000;			/* set status register */
    pTcb->pc       = (INSTR *) vxTaskEntry;	/* set entry point */
    pTcb->priority = priority;			/* set priority */
    pTcb->name     = name;			/* set name */
    pTcb->valid    = (((int)pTcb & 0xffff0000) | (~(int)pTcb & 0x0000ffff));
    pTcb->status   = WIND_SUSPEND;		/* set status */
    pTcb->mode     = 0;				/* set mode */
    pTcb->delay    = 0;			/* remainder of delay in ticks */
    pTcb->tslice   = 0;			/* remainder of slice in ticks */
    pTcb->foroff   = 0;			/* format/offset frame (68020 only) */
    pTcb->d0 = 0;			/* initialize register set */
    pTcb->d1 = 0;
    pTcb->d2 = 0;
    pTcb->d3 = 0;
    pTcb->d4 = 0;
    pTcb->d5 = 0;
    pTcb->d6 = 0;
    pTcb->d7 = 0;
    pTcb->a0 = 0;
    pTcb->a1 = 0;
    pTcb->a2 = 0;
    pTcb->a3 = 0;
    pTcb->a4 = 0;
    pTcb->a5 = 0;
    pTcb->a6 = 0;

    /* initialize historical tcb extension */

    pTcbX = (TCBX *) &pTcb->tcbx;

    bzero ((char *) pTcbX, sizeof (TCBX));	/* zero the extension */

    pTcbX->entry       = entryAdrs;			/* entry address */
    pTcbX->errorStatus = OK;				/* errorStatus */
    pTcbX->topOfStack  = botOfStack - stksize;		/* top of stack */
    pTcbX->botOfStack  = botOfStack;			/* bottom of stack */
    pTcbX->initialSP   = (char *) sp;			/* initial stack ptr */
    pTcbX->options     = options;			/* options */
    pTcbX->taskVar     = NULL;		/* task variables */
    pTcbX->fpContext   = NULL;		/* floating point context */
    pTcbX->name        = name;		/* name of this task */
    pTcbX->taskId      = (int) pTcb;	/* the taskId for this task */

    for (ix = 0; ix < 3; ix++)
	pTcbX->taskStd [ix] = ix;

    if (intCount () > 0)
	{
	logMsg ("taskCreate: Not callable from interrupt level.\n");
	return (ERROR);
	}
    else
	{
	kernelState = TRUE;			/* KERNEL_ENT */
	status = windSpawn (pTcb);
	windExit ();				/* KERNEL_EXIT */
	return ((status == ERROR) ? ERROR : (int) pTcb);
	}
    }
/*******************************************************************************
*
* taskActivate - activate an already created task
*
* A task is created in the suspended state and must be activated before it
* joins the ready list.  taskCreate and taskActivate are the lower level
* routines to taskSpawn (2).  A task may only be activated once.
*
* RETURNS: OK or ERROR if invalid task id or unable to activate task
*
* SEE ALSO: taskCreate(2)
*/

STATUS taskActivate (tid)
    int tid;			/* task id of new task */

    {
    FAST int ix;
    FAST TCBX *pTcbX = taskTcbX (tid);

    if (pTcbX == NULL)		/* task non-existent */
	return (ERROR);

    /* run the create hooks */

    for (ix = 0; ix < VX_MAX_TASK_CREATE_RTNS; ++ix)	/* run create hooks */
	{
	if (taskCreateTable[ix] != NULL)
	    (*taskCreateTable[ix]) (pTcbX);
	}

    return (taskResume (tid));	/* resume the task */
    }
/*******************************************************************************
*
* taskDelete - delete a task
*
* This routine suspends the task, calls any routines specified by
* taskDeleteHookAdd (2), deletes the task, and reclaims its stack.
*
* If a task attempts to delete itself, the deletion will be performed in
* the context of the exception task.
*
* SEE ALSO: excLib (1), taskHookLib (1), taskDeleteHookAdd (2)
*/

STATUS taskDelete (tid)
    int tid;			/* task id of task to delete */

    {
    STATUS status;
    FAST int ix;
    TCBX *pTcbX = taskTcbX (tid);

    if (pTcbX == NULL)			/* task non-existent */
	return (ERROR);

    /* requests to kill self get sent to the exception task */

    if ((tid == 0) || (tid == taskIdSelf ()))
	{
	/* if a task with id 0 tries to kill itself, we can't reclaim its
	 * stack or run delete hooks because there is no handle for the
	 * task for the exception task to use. 
	 */

	if (taskIdSelf () == 0)
	    logMsg ("taskDelete: Unexpected task id of zero.\n");
	else
	    excJobAdd (taskDelete, taskIdSelf ());

	taskSuspend (0);	/* wait for exception task to kill us */
	}


    /* the rest of this routine will only be executed by a task other than
     * the one being deleted */

    taskSuspend (tid);

    /* run the delete hooks */

    for (ix = 0; ix < VX_MAX_TASK_DELETE_RTNS; ++ix)
	{
	if (taskDeleteTable[ix] != NULL)
	    (*taskDeleteTable[ix]) (pTcbX);
	}

    if (kernelState)			/* we interrupted the kernel */
	{
        windWorkAdd (windDelete, tid);
	return (OK);
	}
    else
	{
	kernelState = TRUE;			/* KERNEL_ENT */
	status = windDelete ((WIND_TCB *) tid);
	windExit ();				/* KERNEL_EXIT */

	/* is the deallocate stack bit set in the task option word? */

	if (pTcbX->options & VX_DEALLOC_STACK)
	    {
	    /* topOfStack points to the stack and includes the TCBX */
	    free (pTcbX->topOfStack);
	    }
	return (status);
	}
    }
/*******************************************************************************
*
* exit - exit a task
*
* A task may call this routine to exit (cease to exist as a task).
* A task can also exit simply by returning from the main routine.
*
* SEE ALSO: taskDelete (2)
*
* VARARGS0
* ARGSUSED
*/

VOID exit (code)
    int code;	/* just because standard C libray exit takes an argument */

    {
    (void) taskDelete (0);
    }
/*******************************************************************************
*
* taskSuspend - suspend a task
*
* The specified routine will be suspended.  A task id of zero results in
* the suspension of the calling routine.
*
* RETURNS: OK or ERROR if unable to suspend task
*/

STATUS taskSuspend (tid)
    int tid;			/* task id of task to suspend */

    {
    STATUS status;

    if (tid == 0)		/* task id of zero is calling task */
	tid = taskIdCurrent;


    if (kernelState)
	{
	windWorkAdd (windSuspend, tid);
	return (OK);
	}
    else
	{
	kernelState = TRUE;		/* KERNEL ENTER */
	status = windSuspend ((WIND_TCB *) tid);
	windExit ();			/* KERNEL EXIT */
	return (status);
	}
    }
/*******************************************************************************
*
* taskResume - resume a task
*
* The specified task will be resumed.
*
* RETURNS: OK or ERROR if invalid task id or unable to resume task
*/

STATUS taskResume (tid)
    int tid;			/* task id of task to resume */

    {
    STATUS status;

    if (tid == 0)		/* task id of zero is calling task */
	tid = taskIdCurrent;

    if (kernelState)
	{
	windWorkAdd (windResume, tid);
	return (OK);
	}
    else
	{
	kernelState = TRUE;		/* KERNEL ENT */
	status = windResume ((WIND_TCB *) tid);
	windExit ();			/* KERNEL EXIT */
	return (status);
	}
    }
/*******************************************************************************
*
* taskRestart - restart a task
*
* This routine "restarts" a task.  The task is first deleted.
* Then it is recreated, with the same ID, priority, options, 
* original entry point, stack size, and (up to 10) parameters it had
* when it was deleted.
*
* NOTE
* If the task has modified any of its startup parameters, the
* restarted task will start with the changed values.
*
* RETURNS:
*  OK, or
*  ERROR if invalid task id, or couldn't be re-started
*/

int taskRestart (tid)
    int tid;			/* task id of task to restart */

    {
    int priority;
    int options;
    FUNCPTR entry;	/* entry point of task */
    char *botOfStack;
    int stacksize;
    int args [MAX_TASK_ARGS];
    TCBX *pTcbx;
    char *name;
    WIND_TCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)
	return (ERROR);		/* specified task not found */

    pTcbx = &pTcb->tcbx;

    name	= pTcbx->name;
    priority	= pTcb->priority;
    options	= pTcbx->options;
    entry	= pTcbx->entry;
    stacksize	= pTcbx->botOfStack - pTcbx->topOfStack;
    botOfStack	= pTcbx->botOfStack;
    bcopy ((char *) pTcbx->initialSP, (char *) args, sizeof (args));

    pTcbx->options &= ~VX_DEALLOC_STACK;	/* don't free memory */
    (void)taskDelete ((int) pTcb);		/* delete the task */

    /* XXX should check return value */
    taskActivate (taskCreate (name, priority, options, botOfStack,
			      stacksize, pTcb, entry, args [0], args [1],
			      args [2], args [3], args [4], args [5],
			      args [6], args [7], args [8], args [9]));
    return (tid);
    }
/*******************************************************************************
*
* taskOptionsSet - change task options
*
* Change the execution options of a task.
*
* Bits in the options argument may be set to run with the following modes.
* .CS
*	VX_SUPERVISOR_MODE	-  execute in supervisor mode
*	VX_UNBREAKABLE		-  don't allow debugging
*	VX_DEALLOC_STACK	-  deallocate the stack on deletion
*	VX_FP_TASK		-  execute with coprocessor support
*	VX_STDIO		-  execute with standard I/O support
* .CE
* See definitions in taskLib.h.
*
* RETURNS: OK or ERROR if invalid task id
*
* SEE ALSO: taskOptionsSet (2)
*/

STATUS taskOptionsSet (tid, mask, newOptions)
    int tid;			/* task id                         */
    int mask;			/* mask, 1 = ok to change this bit */
    int newOptions;		/* new options                     */

    {
    FAST TCBX *pTcbX = taskTcbX (tid);

    if (pTcbX == NULL)
	return (ERROR);

    pTcbX->options = (pTcbX->options & ~mask) | newOptions;

    /* if we are setting/resetting unbreakable option for current task,
     * then call breakpoint callout */

    if ((mask & VX_UNBREAKABLE) &&
	((tid == 0) || (tid == taskIdCurrent)) &&
	(taskBpHook != NULL))
	{
	(* taskBpHook) ((newOptions & VX_UNBREAKABLE) == 0);
	}

    return (OK);
    }
/*******************************************************************************
*
* taskOptionsGet - examine task options
*
* This routine gets the specified tasks current execution options.
*
* Bits in the options argument may be set to run with the following modes.
* .CS
*	VX_SUPERVISOR_MODE	-  execute in supervisor mode
*	VX_UNBREAKABLE		-  don't allow debugging
*	VX_DEALLOC_STACK	-  deallocate the stack on deletion
*	VX_FP_TASK		-  execute with coprocessor support
*	VX_STDIO		-  execute with standard I/O support
* .CE
* See definitions in taskLib.h.
*
* RETURNS: OK or ERROR if invalid task id
*
* SEE ALSO: taskOptionsSet (2)
*/

STATUS taskOptionsGet (tid, pOptions)
    int tid;			/* task id        */
    int *pOptions;		/* task's options */

    {
    FAST TCBX *pTcbX = taskTcbX (tid);

    if (pTcbX == NULL)
	return (ERROR);

    *pOptions = pTcbX->options;
    return (OK);
    }
/*******************************************************************************
*
* taskPrioritySet - change priority of a task
*
* This routine changes the specified task's priority to the specified priority.
*
* RETURNS: OK or ERROR if invalid task id
*
* SEE ALSO: taskPriorityGet (2)
*/

STATUS taskPrioritySet (tid, newPriority)
    int tid;			/* task id      */
    int newPriority;		/* new priority */

    {
    STATUS status;

    if (tid == 0)		/* task id of zero is calling task */
	tid = taskIdCurrent;

    if (kernelState)
	{
	windWorkAdd (windPrioritySet, tid, newPriority);
	return (OK);
	}
    else
	{
	kernelState = TRUE;				/* KERNEL_ENT */
	status = windPrioritySet ((WIND_TCB *) tid, newPriority);
	windExit ();					/* KERNEL_EXIT */
	return (status);
	}
    }
/*******************************************************************************
*
* taskPriorityGet - examine priority of a task
*
* This routine is used to determine the current priority of a specified task.
*
* RETURNS:
*  OK and pPriority gets task priority, or
*  ERROR if invalid task id
*
* SEE ALSO: taskPrioritySet (2)
*/

STATUS taskPriorityGet (tid, pPriority)
    int tid;			/* task id of new task */
    int *pPriority;		/* old priority        */

    {
    WIND_TCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)
	return (ERROR);

    *pPriority = pTcb->priority;

    return (OK);
    }
/*******************************************************************************
*
* taskLock - disable task rescheduling
*
* This routine disables task context switching.
* The task that calls this routine
* must never preempt itself while task switching is disabled.
* For instance, semaphores should not be taken.
*
* This routine does not lock out interrupts.
* To do so one should call intLock (2).
* taskLock (2) is preferable to intLock (2) because locking out
* interrupts adds interrupt latency to the system.
*
* RETURNS: OK
*
* SEE ALSO: taskUnlock(2), intLock(2)
*/

STATUS taskLock ()

    {
    contextSwitching = FALSE;		/* atomically reset contextSwitching */
    return (OK);
    }
/*******************************************************************************
*
* taskUnlock - enable task rescheduling
*
* This routine is used to resume task context switching after a
* taskLock(2) has disabled it.
* Any tasks which were eligible to preempt the current task will now execute.
* 
* RETURNS: OK
*
* SEE ALSO: taskLock(2)
*/

STATUS taskUnlock ()

    {
    contextSwitching = TRUE;		/* atomically set contextSwitching */
    kernelState = TRUE;			/* KERNEL_ENT */
    windExit ();			/* KERNEL_EXIT */
    return (OK);
    }
/*******************************************************************************
*
* taskDelay - delay a task from executing
*
* This routine caused the calling task to relinquish the CPU for the duration
* (in ticks) specified.  This is commonly referred to as manual rescheduling but
* it is also useful when waiting for some external condition that does not have
* an interrupt associated with it.
*
* RETURNS: OK or ERROR if invalid task id (called from interrupt level)
*/

STATUS taskDelay (ticks)
    int ticks;		/* number of ticks to suspend task */

    {
    STATUS status;

    if (ticks == 0)		/* delay for no time == NOP */
	return (OK);

    if (intCount () > 0)
	{
	logMsg ("taskDelay: Not callable from interrupt level.\n");
	return (ERROR);
	}
    else
	{
	kernelState = TRUE;				/* KERNEL_ENT */
	status = windDelay (ticks);
	windExit ();					/* KERNEL_EXIT */
	return (status);
	}
    }
/*******************************************************************************
*
* taskTcb - get the task control block for a task id
*
* This routine is used to get a pointer to the task control block associated
* with a task id.  Although all of the state information is contained in the
* task control block, user are discouraged from changing them directly.  To
* change the registers, for instance, a better interface is provided via
* taskRegsSet (2) and taskRegsGet (2).
*
* RETURNS: pointer to WIND TCB, or NULL if invalid task id
*/

WIND_TCB *taskTcb (tid)
    int tid;			/* task id */

    {
    if (tid == 0)
	tid = taskIdSelf ();

    if (taskIdVerify (tid) != OK)
	{
	errnoSet (S_taskLib_TASK_ID_ERROR);
	return (NULL);
	}
    else
	return ((WIND_TCB *) tid);
    }
/*******************************************************************************
*
* taskTcbX - get the task control block extension associated with a task ID
*
* This routine returns a pointer to the task control block extension.
* The contents of the TCBX is defined in taskLib.h.
* Users should not directly modify the contents of the TCB extension.
*
* RETURNS: pointer to TCB extension, or NULL if invalid task id
*/

TCBX *taskTcbX (tid)
    int tid;			/* task id */

    {
    if (tid == 0)
	tid = taskIdSelf ();

    if (taskIdVerify (tid) != OK)
	return (NULL);
    else
	return ((TCBX *) &(((WIND_TCB *) tid)->tcbx));
    }
/*******************************************************************************
*
* taskTcbToTd - fill in the task descriptor structure from WIND TCB
*
* This routine assumes we have a legal pTcb.  It is a local routine that
* does all the work for the higher level call taskInfoGet (2).
*/

LOCAL VOID taskTcbToTD (pTcb, pTaskDesc)
    WIND_TCB *pTcb;
    TASK_DESC *pTaskDesc;

    {
    pTaskDesc->td_priority	= pTcb->priority;	/* task priority */
    pTaskDesc->td_id		= (int) pTcb;		/* task id */
    pTaskDesc->td_status	= pTcb->status;		/* task status*/
    pTaskDesc->td_sp		= pTcb->ssp;		/* stack pointer save */
    pTaskDesc->td_usp		= pTcb->ssp;		/* user stack pointer */
    pTaskDesc->td_delay		= pTcb->delay;		/* delay/timeout ticks*/

    pTaskDesc->td_spbottom	= pTcb->tcbx.botOfStack;/* bottom of stack ptr*/
    pTaskDesc->td_name		= pTcb->tcbx.name;	/* name of task */
    pTaskDesc->entry		= pTcb->tcbx.entry;	/* entry of task */
    pTaskDesc->errorStatus      = pTcb->tcbx.errorStatus;/* most recent error */
    pTaskDesc->topOfStack	= pTcb->tcbx.topOfStack;/* top of stack */
    pTaskDesc->initialSP	= pTcb->tcbx.initialSP; /* initial stack ptr. */
    pTaskDesc->options		= pTcb->tcbx.options;	/* task option bits */
    }
/*******************************************************************************
*
* taskInfoGet - get pointer to task's task descriptor
*
* This routine finds the TASK_DESC (task descriptor) of the specified task.
*
* NOTE
* Examination of TCBs should be restricted to debugging aids.
*
* RETURNS:
*  OK or ERROR if invalid task id
*/

STATUS taskInfoGet (tid, pTaskDesc)
    int tid;			/* task id                          */
    TASK_DESC *pTaskDesc;	/* where to return task description */

    {
    WIND_TCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)
	return (ERROR);

    taskTcbToTD (pTcb, pTaskDesc);
    return (OK);
    }
/*******************************************************************************
*
* taskName - get name associated with a task id
*
* This routine returns the name of a task.
*
* RETURNS: pointer to task name, or NULL if invalid task id
*/

char *taskName (tid)
    int tid;		/* task id */

    {
    TCBX *pTcbX = taskTcbX (tid);

    if (pTcbX == NULL)
	return ((char *) NULL);

    return (pTcbX->name);
    }
/*******************************************************************************
*
* taskNameToId - lookup task id associated with a name
*
* This routine returns the id of a task that matches the name specified.
* This is an inefficient manner to reference a task and use should be
* restricted to user interfaces such as the shell.
*
* RETURNS: task id or ERROR if task by the name `name' not found
*/

int taskNameToId (name)
    char *name;		/* task name to look up */

    {
    WIND_TCB *pTcb;
    NODE *pNode = lstFirst (&activeList);

    while (pNode != NULL)	/* search for named task */
	{
	pTcb = (WIND_TCB *) ((int)pNode - OFFSET (WIND_TCB, tcbNode));

	if (strcmp (pTcb->tcbx.name, name) == 0)
	    return ((int) pTcb);

	pNode = lstNext (pNode);
	}

    errnoSet (S_taskLib_NAME_NOT_FOUND);
    return (ERROR);				/* found no match */
    }
/*******************************************************************************
*
* taskIsSuspended - check if a task is suspended
* 
* This routine tests the status field of the specified task to determine
* if it is suspended.
*
* RETURNS: TRUE if task is suspended, otherwise FALSE
*/

BOOL taskIsSuspended (tid)
    int tid;	/* task id */

    {
    FAST WIND_TCB *pTcb = taskTcb (tid);

    if (pTcb != NULL && pTcb->status & WIND_SUSPEND)
	return (TRUE);

    return (FALSE);
    }
/*******************************************************************************
*
* taskIsReady - check if a task is ready to run
*
* This routine tests the status field of the specified task to determine
* if it is ready.
*
* RETURNS: TRUE if task is ready, otherwise FALSE
*/

BOOL taskIsReady (tid)
    int tid;	/* task id */

    {
    FAST WIND_TCB *pTcb = taskTcb (tid);

    if (pTcb != NULL && pTcb->status == WIND_READY)
	return (TRUE);

    return (FALSE);
    }
/*******************************************************************************
*
* taskStatusString - return the task's status as a string
*
* This routine converts the WIND task status word in the TCB
* and returns the appropriate string.
*
* EXAMPLE
* .CS
*  -> taskStatusString (taskNameToId ("shell"), xx=malloc(10));
*  new symbol "xx" added to symbol table.
*  value = 0 = 0x0
*  -> printf ("shell status = <%s>\n", xx)
*  shell status = <READY>
*  value = 2 = 0x2
*  ->
* .CE
*
* RETURNS:
*  OK or ERROR if invalid task id, `pString' gets status string
*/

STATUS taskStatusString (tid, pString)
    int tid;		/* task to get string for            */
    char *pString;	/* where to return string (10 bytes) */

    {
    WIND_TCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)
	return (ERROR);

    if (pTcb->status == WIND_SUSPEND)
	{
	strcpy (pString, "SUSPEND");
	return (OK);
	}

    switch (pTcb->status & ~WIND_SUSPEND)
	{
	case WIND_READY:	strcpy (pString, "READY");  break;
	case WIND_DELAY:	strcpy (pString, "DELAY");  break;
	case WIND_EVENT:	strcpy (pString, "EVENT");  break;
	case WIND_PEND:		strcpy (pString, "PEND");   break;
	case WIND_DEAD:		strcpy (pString, "DEAD");   break;

	default:			/* unanticipated combination */
	    sprintf (pString, "0x%02x", pTcb->status);
	    return (ERROR);
	}

    if (pTcb->status & WIND_SUSPEND)
	strcat (pString, "+S");		/* task also explicitly suspended */

    return (OK);
    }
/*******************************************************************************
*
* taskIdDefault - default task id
*
* This routine maintains a global default task id.  This id is used by 
* libraries that want to allow a task id argument to take on a default
* value if the user did not explicitly supply one.
* If the task id supplied to this routine is not zero (i.e. the user
* did specify a taskId) then the default id is set to that value,
* and that value is returned.  If the task id supplied to this routine
* is zero (i.e. the user did not specify a task id) then the default id
* is not changed and its value is returned.  Thus the value returned
* is always the last task id the user specified.
*
* RETURNS: most recently set non-zero task id
*
* SEE ALSO: dbgLib (1), "Debugging"
*/

int taskIdDefault (tid)
    int tid;	/* user supplied task id, if 0 then just return default */

    {
    static int defaultTaskId;	/* current default task id */

    if (tid != 0)
	defaultTaskId = tid;		/* update default */

    return (defaultTaskId);
    }
/*******************************************************************************
*
* taskIdSelf - get task id of running task
*
* Get task id of calling task. The task id will be invalid if called
* at interrupt level.  This routine simply returns global variable
* `taskIdCurrent' which is only valid during a task context.
*
* RETURNS: task id of calling task
*/

int taskIdSelf ()

    {
    return (taskIdCurrent);
    }
/*******************************************************************************
*
* taskIdVerify - verify the existence of a task
*
* This verifies the existence of the specified task by matching the task id
* with a encrypted version located within the task control block.
*
* RETURNS: OK or ERROR if invalid task id
*/

STATUS taskIdVerify (tid)
    int tid;	/* task id */

    {
    if ((tid & 0x1) || ((WIND_TCB *)tid)->valid !=
    	((tid & 0xffff0000) | (~tid & 0x0000ffff)))
	{
	errnoSet (S_taskLib_TASK_ID_ERROR);
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* taskIdListGet - get list of active task IDs
*
* This routine provides the calling task with a list of all of the active
* tasks.  The list consists of task ids, the last task id is a terminating 0.
* The list is unordered.
*
* WARNING:
* Kernel rescheduling is disabled with taskLock (2) while
* tasks are looked up.  There is no guarantee that all the
* tasks are valid and/or no new tasks have been created by
* the time this routine returns.
*
* RETURNS: number of tasks in list
*/

int taskIdListGet (idList, maxTasks)
    FAST int idList[];		/* array of task ids to be filled in */
    FAST int maxTasks;		/* max tasks idList can accommodate  */

    {
    NODE *pNode = lstFirst (&activeList);
    FAST int *pId  = idList;

    taskLock ();

    while ((pNode != NULL) && (--maxTasks >= 0))
	{
	*(pId++) = (int) ((int)pNode - OFFSET (WIND_TCB, tcbNode));
	pNode = lstNext (pNode);
	}

    taskUnlock ();

    return (pId - idList);	/* return count of active tasks */
    }
/*******************************************************************************
*
* taskRegsGet - get a task's registers from TCB
*
* This routine gathers information about a task which is kept in the TCB.
* It returns, in the locations whose pointers are
* passed as parameters, the task's address and data registers, and its
* SP, SR and PC.  The address and data registers are returned in
* separate arrays, each containing 8 registers.
*
* NOTE
* This routine only works well if the specified task's TCB is not that
* of the calling task.  Results are unpredictable if a task tries to
* discover its own registers.
*
* SEE ALSO: taskRegsSet(2)
*/

STATUS taskRegsGet (tid, dregs, aregs, sp, sr, pc)
    int tid;		/* task id                            */
    int dregs[];	/* buffer for data registers (0-7)    */
    int aregs[];	/* buffer for address registers (0-6) */
    char **sp;		/* buffer for stack pointer           */
    USHORT *sr;		/* buffer for status register         */
    INSTR **pc;		/* buffer for program counter         */

    {
    FAST WIND_TCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)		/* task non-existent */
	return (ERROR);

    dregs[0] = pTcb->d0;
    dregs[1] = pTcb->d1;
    dregs[2] = pTcb->d2;
    dregs[3] = pTcb->d3;
    dregs[4] = pTcb->d4;
    dregs[5] = pTcb->d5;
    dregs[6] = pTcb->d6;
    dregs[7] = pTcb->d7;
    
    aregs[0] = pTcb->a0;
    aregs[1] = pTcb->a1;
    aregs[2] = pTcb->a2;
    aregs[3] = pTcb->a3;
    aregs[4] = pTcb->a4;
    aregs[5] = pTcb->a5;
    aregs[6] = pTcb->a6;

    *sp = pTcb->ssp;
    *sr = pTcb->sr;
    *pc = pTcb->pc;

    return (OK);
    }
/*******************************************************************************
*
* taskRegsSet - set a task's registers 
*
* This routine loads the given values into the given tasks TCB.
* The registers are contained in two arrays, which contain d0-d7
* and a0-a6, and three more fields are used for the SR, PC, SP. 
*
* RETURNS: OK or ERROR if invalid task id
*
* SEE ALSO: taskRegsGet(2)
*/

STATUS taskRegsSet (tid, pDregs, pAregs, sp, sr, pc)
    int tid;		/* task id                 */
    int pDregs[];	/* data registers (0-7)    */
    int pAregs[];	/* address registers (0-6) */
    char *sp;		/* stack pointer           */
    USHORT sr;		/* status register         */
    INSTR *pc;		/* program counter         */

    {
    FAST WIND_TCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)		/* task non-existent */
	return (ERROR);

    pTcb->d0 = pDregs[0];
    pTcb->d1 = pDregs[1];
    pTcb->d2 = pDregs[2];
    pTcb->d3 = pDregs[3];
    pTcb->d4 = pDregs[4];
    pTcb->d5 = pDregs[5];
    pTcb->d6 = pDregs[6];
    pTcb->d7 = pDregs[7];

    pTcb->a0 = pAregs[0];
    pTcb->a1 = pAregs[1];
    pTcb->a2 = pAregs[2];
    pTcb->a3 = pAregs[3];
    pTcb->a4 = pAregs[4];
    pTcb->a5 = pAregs[5];
    pTcb->a6 = pAregs[6];
    pTcb->ssp = sp;

    pTcb->sr = sr;

    pTcb->pc = pc;

    return (OK);
    }
/*******************************************************************************
*
* taskRegsShow - print contents of a task's registers
*
* This routine prints to standard out the contents of a task's registers.
*
* EXAMPLE
* .CS
*  -> taskRegsShow (taskNameToId ("shell"))  
*
*  D0 =       0  D4 =       0  A0 =       0  A4 =       0
*  D1 =       0  D5 =       0  A1 =       0  A5 =  3f4a08  SR =   3004
*  D2 =       0  D6 =       0  A2 =       0  A6 =  3d6458  PC =  39ae2
*  D3 =       0  D7 =       a  A3 =       0  A7 =  3d6458
*  value = 1 = 0x1
* .CE
*/

VOID taskRegsShow (tid)
    int tid;		/* task id */

    {
    int dregs[8];		/* task's data registers */
    int aregs[7];		/* task's address registers */
    char *sp;			/* task's stack pointer */
    USHORT sr;			/* task's status register */
    INSTR *pc;			/* task's pc */

    if (taskRegsGet (tid, dregs, aregs, &sp, &sr, &pc) == ERROR)
	{
	printf ("taskRegsShow: invalid task id %#x\n");
	return;
	}

    printf ("\n");
    printf ("D0 = %8x   D4 = %8x   A0 = %8x   A4 = %8x\n",
	    dregs[0], dregs[4], aregs[0], aregs[4]);
    printf ("D1 = %8x   D5 = %8x   A1 = %8x   A5 = %8x   SR = %8x\n",
	    dregs[1], dregs[5], aregs[1], aregs[5], sr & 0xffff);
    printf ("D2 = %8x   D6 = %8x   A2 = %8x   A6 = %8x   PC = %8x\n",
	    dregs[2], dregs[6], aregs[2], aregs[6], pc);
    printf ("D3 = %8x   D7 = %8x   A3 = %8x   A7 = %8x\n",
	    dregs[3], dregs[7], aregs[3], sp);
    }
/*******************************************************************************
*
* taskSRSet - set task status register
*
* This routine sets the status register of a task not running (i.e. the TCB
* must NOT be that of the calling task).  It is used by the debugging
* facilities to set the trace bit in the SR of a task being single-stepped.
*
* RETURNS: OK or ERROR if invalid task id
*/

STATUS taskSRSet (tid, sr)
    int tid;	 	/* task id */
    USHORT sr;		/* new SR */

    {
    FAST WIND_TCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)		/* task non-existent */
	return (ERROR);

    pTcb->sr = sr;

    return (OK);
    }
