/* semLib.c - VxWorks/VRTX semaphore library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
02g,08sep88,jcf  got semClear right... this time for sure.
02f,07sep88,gae  fudged 02e.
02e,01sep88,jcf  reworked semClear so it tells you when you got the semaphore.
02d,24aug88,gae  documentation.
02b,28may88,dnw  lint.
02a,04may88,jcf  added semClear (), semCreate (), and semDelete ().
01h,22apr88,gae  changed VRTX functions to use vrtx prefix.
01g,19jan88,jcf  version 4.
01h,03nov87,ecs  documentation.
01g,23mar87,jlf  documentation.
01f,21dec86,dnw  changed to not get include files from default directories.
01e,13dec86,dnw  changed semGive to not post if semaphore is already given.
01d,16may85,jlf  documentation fixes.
01c,10sep84,jlf  added copyright, comments.  removed GLOBAL.
01b,15jun84,dnw  added LINTLIBRARY for lint.
01a,15nov83,ecs  ripped off from drvLib.c
*/

/*
DESCRIPTION
This library provides a generic interface to the VRTX kernel semaphores.
It translates calls from VxWorks into VRTX calls contained in vrtxALib (1).
Semaphore mangement includes:

    - create a semaphore,
    - take a semaphore,
    - give a semaphore,
    - delete a semaphore,
    - empty a semaphore.

SEMAPHORES
This implementation provides a binary semaphore facility.  Semaphores can
be used to control access to shared devices or data structures, and
to synchronize multiple tasks, or task level and interrupt level processes.

A semaphore is just a memory location that is accessible to all the processes
that need it.  The semaphore must be created (by semCreate) before it
is ever used.  Thereafter, every time the semaphore is given (semGive) it
becomes full, and every time it is taken (semTake), it becomes empty.
A semaphore can be emptied by semClear.  This operation corresponds to taking
the semaphore only if it is available.  A semaphore is deleted by semDelete.

TASK STATE CHANGES
When a task takes a semaphore, that task will become pended if the semaphore
is currently empty.  Any number of tasks may be simultaneously pended on
the same semaphore.  If the semaphore is full when the semTake is done, the
calling task continues running, and the semaphore is made empty.  Only
one task may have a semaphore at a time.

When a task gives a semaphore, that semaphore becomes full.  The kernel
immediately does a scan, to see if any other tasks are currently pended on
that semaphore (because they had done a semTake).  If there are one or more
such tasks, the highest priority one is made ready to run.  If that task
is higher priority than the task doing the semGive, that task will now get
the CPU, and the giving task will be preempted.  If no other tasks, or only
lower priority tasks, are pended on the semaphore, the giving task will
keep running.

INTERRUPT LEVEL USE
Semaphores may be given, but not taken, from interrupt level.  An interrupt
level process that gives a semaphore will not be preempted, no matter who
becomes ready, but the task that had been running when the interrupt
occurred may be preempted when the interrupt level code is done.

PROTOCOL HINTS
Although any number of tasks may give, get, and initialize the same semaphore,
you must be careful about when some of these actions are taken.

There is no danger in any number of processes doing semTakes whenever they
like.  SemGives and semInits, however, should be controlled.  If semGive's
are done only by one process, there is no problem.  If they are done by
multiple processes, only a process that already has the semaphore should
give it.  Otherwise, multiple copies of the semaphore could be running
around, and mutual-exclusion will no longer be in effect.  SemInit's should
only be performed if some process (perhaps the same one that does the semInit)
is going to do a semGive without doing a semTake first.

DEFICIENCIES
There is no mechanism to give back semaphores automatically
when tasks are suspended or deleted.  If a task ceases execution unexpectedly
(a bus error, deleted from the shell, etc.) any semaphores that it currently
has taken will never be given again, unless there is a watchdog or something
in operation that can do it.  Since the I/O system uses semaphores, this
can cause a device to get hung up if a task doing I/O to that device has
blown up.

SEE ALSO: "Architecture", taskLib (1), kernelLib (1), vrtxALib (1)
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "semLib.h"
#include "memLib.h"


/*******************************************************************************
*
* semCreate - initialize semaphore
*
* Create a semaphore.  The semaphore is initially empty.  After
* initialization, it must be given before it can be taken.
*
* RETURNS: SEM_ID or NULL if unable to allocate memory
*
* SEE ALSO: semInit (2)
*/

SEM_ID semCreate ()

    {
    SEM_ID semId = (SEM_ID) malloc (sizeof (SEMAPHORE));

    if (semId != NULL)
	semId->count = 0;		/* initially taken */

    return (semId);
    }
/*******************************************************************************
*
* semGive - give semaphore
*
* Gives the semaphore.  If a higher priority task has already taken
* the semaphore (so that it is now pended waiting for it), that task
* will now become ready to run, and preempt the task that does the semGive.
* If the semaphore is already full (it has been given but not taken) this
* call is essentially a no-op.
*
* SEE ALSO: vrtxPost (2)
*/

VOID semGive (semId)
    SEM_ID semId;	/* semaphore id to give */

    {
    if (semId->count == 0)	/* if semaphore is not already given */
	vrtxPost (&semId->count, 1);
    }
/*******************************************************************************
*
* semTake - take semaphore
*
* Takes the semaphore.  If the semaphore is empty (it has not been given
* since the last semTake or semInit), this task will become pended until
* the semaphore becomes available by some other process doing a semGive
* of it.  If the semaphore is already available, this call will empty
* the semaphore, so no other process can take it till this task gives it
* back, and this task will continue running.
*
* WARNING
* This routine may not be used from interrupt level.
*
* SEE ALSO: vrtxPend (2)
*/

VOID semTake (semId)
    SEM_ID semId;	/* semaphore id to take */

    {
    int dummy;

    vrtxPend (&semId->count, 0, &dummy);
    }
/*******************************************************************************
*
* semDelete - delete semaphore
*
* Frees the memory allocated for the semaphore.  A deleted semaphore can
* not be resurrected.  If the semaphore was declared as a global variable
* and not created via semCreate then this should NOT be called.
*
* WARNING
* This routine may not be used from interrupt level.
*/  

VOID semDelete (semId)
    SEM_ID semId;	/* semaphore id to delete */

    {
    free ((char *) semId);
    }
/*******************************************************************************
*
* semClear - take semaphore if semaphore is available
* 
* If the semaphore is available, it is taken, otherwise no action is taken.
* This routine never preempts the caller.  If semaphore was not available
* this routine returns ERROR.
*
* RETURNS: OK or ERROR if semaphore was unavailable
*/  

STATUS semClear (semId)
    SEM_ID semId;	/* semaphore to empty */

    {
    int dummy;

    return (vrtxAccept (&semId->count, &dummy));
    }
/*******************************************************************************
*
* semInit - initialize a declared semaphore
*
* In some instances a semaphore cannot be created with semCreate (2)
* but is global variable.  This routine must be called to initialize it.
*
* SEE ALSO: semCreate (2)
*/  

VOID semInit (semId)
    SEM_ID semId;	/* semaphore id to initialize */

    {
    semId->count = 0;	/* initially taken */
    }
