/* taskLib.c - task management interface for pSOS kernel */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01m,17apr89,jcf  lint.
01l,31mar89,llk  made taskIdVerify() check for PSOS_INACTIVE.
01k,21nov88,jcf  added variables contextSwitching and kernelState for dbgLib.
01j,23sep88,gae  documentation touchup.
01i,07sep88,gae  documentation.
01h,25aug88,gae  documentation.
01g,07jul88,jcf  lint.
01f,24jun88,dnw  name tweaks.
01e,05jun88,dnw  added exit(), taskRegsShow() and taskBpCalloutSet().
		 made taskOptionsSet() deal with installing removing breakpts.
		 made taskIdVerify() set S_taskLib_TASK_ID_ERROR,
		   and removed setting that error in 12 higher level routines.
		 changed vxCurrentTask to taskIdCurrent.
		 removed taskSRGet().
		 changed taskSpawn to task NULL task name instead of NONE.
		 changed taskSpawn/taskCreate args.
01d,30may88,dnw  changed to v4 names.
01c,29may88,jcf  add filbuf to taskCreate so tcbx is cleared.
01b,14apr88,gae  added taskStd initialization to taskCreate().
		 Lint.  Documentation.
01a,19jan88,jcf  written.
*/

/*
DESCRIPTION
This library provides a generic interface to the pSOS kernel
task management.  Task management includes:

    - create, delete, suspend, resume, restart, prioritize tasks,
    - lock and unlock scheduling,
    - delay a task from running,
    - get and set tasks' registers,
    - get and set tasks' options,
    - find TCBs (task control blocks),
    - find TCBXs (task control block extensions),
    - find TDs (kernel independent task descriptors),
    - translate task ids to task names, and vice versa.

SEE ALSO:
taskHookLib (1), taskVarLib (1), semLib (1), kernelLib (1), "Architecture"
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "taskLib.h"
#include "memLib.h"
#include "strLib.h"
#include "psos.h"


IMPORT VOID vxTaskEntry ();
IMPORT FUNCPTR taskCreateTable [];
IMPORT FUNCPTR taskDeleteTable [];
IMPORT PSOS_CONFIG_TABLE usrCftbl;


/* globals */

/* taskIdCurrent is modified on each context switch to be the task id of 
 * the running task;  it is referred to by taskIdSelf() to get the taskId
 * of the running task */

ULONG taskIdCurrent;		/* task id of current task */
BOOL contextSwitching = TRUE;	/* whether preemption is enabled */
BOOL kernelState = FALSE;	/* wind global variable that is expected in */
				/*   kernel independent libraries */


/* locals */

LOCAL int nameForNameless = 1;	/* unique name to give the nameless */
LOCAL FUNCPTR taskBpHook;	/* rtn to install/remove bkpts for task */


/* forward declarations */

PSOS_PCB *taskTcb ();


/*******************************************************************************
*
* taskBpHookSet - set breakpoint hook for dbgLib
*
* This routine allows dbgLib to install its break-point install/remove routine
* used by taskOptionsSet.
* It should only be called by dbgInit(2).
*
* NOMANUAL
*/

VOID taskBpHookSet (bpHook)
    FUNCPTR bpHook;

    {
    taskBpHook = bpHook;
    }
/*******************************************************************************
*
* taskSpawn - spawn a task
*
* This routine creates and activates a new task with the specified parameters.
* If the task name is specified as NULL, the name will be an ASCII
* hexadecimal number of the task id.
*
* Bits in the options argument may be set to run with the following modes.
* .CS
*       VX_SUPERVISOR_MODE      -  execute in supervisor mode
*       VX_UNBREAKABLE          -  don't allow break point debugging
*       VX_DEALLOC_STACK        -  deallocate the stack on deletion
*       VX_FP_TASK              -  execute with coprocessor support
*       VX_STDIO                -  execute with standard I/O support
* .CE
* See definitions in taskLib.h.
*
* The specified entry address is the address of the main routine of the task.
* This routine will be called with the specified arguments (up to 10).  
* A stack of the specified size will be allocated from the memory pool.  
* Stack size should be even.  Every byte of this stack is initialized to 0xEE, 
* as a debugging aid.  See checkStack(2) for stack size checking aids.
* 
* See taskCreate (2) for creating tasks with preallocated stacks.
*
* BUGS:
* All tasks are spawned in supervisor mode, regardless of 
* the specified options.  In effect, VX_SUPERVISOR_MODE is or'd into the
* user specified options. This is done because other VxWorks facilities
* do not make a distinction of privileged instructions.
*
* RETURNS: task id, or ERROR if out of memory or unable to create task
*
* VARARGS5
*/

int taskSpawn (name, priority, options, stacksize, entryAddress, arg1,
	       arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10)
    char *name;			/* task name of new task     */
    int priority;		/* priority of new task      */
    int options;		/* options word for new task */
    int stacksize;		/* size (bytes) of stack     */
    FUNCPTR entryAddress;	/* entry point of task       */
    int arg1;			/* arguments passed to task  */
    int arg2;
    int arg3;
    int arg4;
    int arg5;
    int arg6;
    int arg7;
    int arg8;
    int arg9;
    int arg10;

    {
    char *topOfStack;
    char *bottomOfStack;
    char newName [12];
    char *mallocName;
    int tid;

    /* name the nameless */

    /* nameless task gets unique number */
    if (name == NULL)
	{
	sprintf (newName, "%x", nameForNameless++);
	name = newName;
	}

    /* allocate memory for stack, tcb extension and name, and fill it in for
     * debugging */

    topOfStack = malloc ((unsigned) (stacksize + sizeof (TCBX) + 
				     strlen (name) + 1));

    if (topOfStack == NULL)
	return (ERROR);

    bottomOfStack = topOfStack + stacksize;

    mallocName = (char *)(((int) bottomOfStack) + sizeof (TCBX));

    strcpy (mallocName, name);
    
    bfill (topOfStack, stacksize, 0xee);

    tid = taskCreate (mallocName, priority, options | VX_DEALLOC_STACK,
		      bottomOfStack, stacksize, (TCBX *) bottomOfStack,
		      entryAddress, arg1, arg2, arg3, arg4,
		      arg5, arg6, arg7, arg8, arg9, arg10);

    if (tid == ERROR || taskActivate (tid) == ERROR)
	{
	free (topOfStack);
	return (ERROR);
	}

    return (tid);
    }
/*******************************************************************************
*
* taskCreate - create a task with stack at specified address
*
* This routine works like taskSpawn (2),
* but instead of allocating the stack and TCB extension automatically,
* it uses the stack and TCB extension passed as arguments.
* The stack will grow towards lower memory locations
* starting from the specified bottom of stack.
*
* Bits in the options argument may be set to run with the following modes.
* .CS
*       VX_SUPERVISOR_MODE      -  execute in supervisor mode
*       VX_UNBREAKABLE          -  don't allow debugging
*       VX_DEALLOC_STACK        -  deallocate the stack on deletion
*       VX_FP_TASK              -  execute with coprocessor support
*       VX_STDIO                -  execute with standard I/O support
* .CE
* See definitions in taskLib.h.
*
* Normally, tasks should be started by taskSpawn (2), not by taskCreate.
*
* BUGS:
* All tasks are spawned in supervisor mode, regardless of 
* the specified options.  In effect, VX_SUPERVISOR_MODE is or'd into the
* user specified options.  This is done because other VxWorks facilities
* do not yet make the distinction of privileged instructions.
*
* RETURNS: task id, or ERROR if unable to create task
*
* VARARGS7
*/

int taskCreate (name, priority, options, botOfStack, stksize, pTcbX,
		entryAdrs, arg1)
    char *name;			/* name of new task           */ 
    int priority;		/* priority of new task       */
    int options;		/* task option word           */
    FAST char *botOfStack;	/* bottom of new task's stack */
    int stksize;		/* size (bytes) of stack      */
    FAST TCBX *pTcbX;		/* address of new task's TCBX */
    FUNCPTR entryAdrs;		/* initial pc of new task     */
    int arg1;			/* 1st of up to 10 arguments  */

    {
    FAST int *argptr;
    FAST int *sp;
    FAST int ix;
    int	pid;
    STATUS status;

    priority = psosPriorityToVx (priority);

    options |= VX_SUPERVISOR_MODE;	/* TEMP! set SUP mode for all tasks */

    /* push args on the stack */

    argptr = &arg1 + MAX_TASK_ARGS;
    sp = (int *) botOfStack;

    for (ix = 0; ix < MAX_TASK_ARGS; ix++)
	*--sp = *--argptr;

    /* initialize tcb extension */

    bzero ((char *) pTcbX, sizeof (TCBX));	/* zero the extension */

    pTcbX->entry       = entryAdrs;
    pTcbX->errorStatus = OK;
    pTcbX->botOfStack  = botOfStack;
    pTcbX->topOfStack  = botOfStack - stksize;
    pTcbX->initialSP   = (char *) sp;
    pTcbX->options     = options;
    pTcbX->taskVar     = NULL;
    pTcbX->fpContext   = NULL;
    pTcbX->name        = name;

    for (ix = 0; ix < 3; ix++)
	pTcbX->taskStd [ix] = ix;

    status = psosSpawnP (name, 0, priority, STK_SIZES, 0, &pid);

    if (status == OK)
	{
	pTcbX->taskId = pid;			/* set id in tcbx */

	((PSOS_PCB *) pid)->pcb_extend = pTcbX;	/* set tcbx ptr in pcb */

	/* set usp stack ptr in pcb (sp), vxTaskEntry will switch stacks
	 * to supervisor for us */

	((PSOS_PCB *) pid)->pcb_usp = (char *) sp;
	}

    return ((status == ERROR) ? ERROR : pid);
    }
/*******************************************************************************
*
* taskActivate - activate an already created task
*
* A task is created in the suspended state and must be activated before it
* joins the ready list.  taskCreate and taskActivate are the lower level
* routines to taskSpawn (2).  A task may only be activated once.
*
* RETURNS: OK or ERROR if invalid task id or unable to activate task
*/

STATUS taskActivate (tid)
    int tid;			/* task id of new task */

    {
    TCBX *pTcbX = taskTcbX (tid);
    int i;

    if (pTcbX == NULL)		/* task non-existant */
	return (ERROR);

    for (i = 0; taskCreateTable [i] != 0;
	    *(FUNCPTR) taskCreateTable [i++] (pTcbX))
	;

    return (psosActivateP (tid, vxTaskEntry));	/* enter vxTaskEntry first */
    }
/*******************************************************************************
*
* taskDelete - delete a task
*
* This routine suspends the task, calls any routines specified by
* taskDeleteHookAdd (2), deletes the task, and reclaims its stack.
*
* If a task attempts to delete itself, the deletion will be performed in
* the context of the exception task.
*
* RETURNS: OK or ERROR if invalid task id or unable to delete task
*
* SEE ALSO: excLib (1), taskDeleteHookAdd (2)
*
* VARARGS0
*/

STATUS taskDelete (tid)
    int tid;			/* task id of task to delete */

    {
    STATUS status;
    FAST int ix;
    TCBX *pTcbX = taskTcbX (tid);

    if (pTcbX == NULL)			/* task non-existent */
	return (ERROR);

    /* requests to kill self get sent to the exception task */

    if ((tid == 0) || (tid == taskIdCurrent))
	{
	/* if a task with id 0 tries to kill itself, we can't reclaim its
	 * stack or run delete hooks because there is no handle for the
	 * task for the exception task to use. 
	 */
	if (taskIdCurrent == 0)
	    psosDeleteP (0);
	else
	    excJobAdd (taskDelete, taskIdCurrent);

	psosSuspendP (0);		/* wait for exception task to kill us */
	}

    /* the rest of this routine will only be executed by a task other than
     * the one being deleted */

    psosSuspendP (tid);

    /* run the delete hooks */

    for (ix = 0; ix < VX_MAX_TASK_DELETE_RTNS; ++ix)
	{
	if (taskDeleteTable [ix] != NULL)
	    (*taskDeleteTable [ix]) (pTcbX);
	}

    status = psosDeleteP (tid);

    /* is the deallocate stack bit set in the task option word? */

    if (pTcbX->options & VX_DEALLOC_STACK)
	{
	/* topOfStack points to the stack and includes the TCBX;
	 * get excTsk to free it.
	 */
        free (pTcbX->topOfStack);
	}

    return (status);
    }
/*******************************************************************************
*
* exit - exit a task
*
* A task may call this routine to exit (cease to exist as a task).
* A task can also exit simply by returning from the main routine.
*
* SEE ALSO: taskDelete (2)
*
* VARARGS0
* ARGSUSED
*/

VOID exit (code)
    int code;   /* just because standard C libray exit takes an argument */

    {
    (void) taskDelete (0);
    }
/*******************************************************************************
*
* taskSuspend - suspend a task
*
* The specified routine will be suspended.  A task id of zero results in
* the suspension of the calling routine.
*
* RETURNS: OK or ERROR if unable to suspend task
*/

STATUS taskSuspend (tid)
    int tid;			/* task id of task to suspend */

    {
    return (psosSuspendP (tid));
    }
/*******************************************************************************
*
* taskResume - resume a task
*
* The specified task will be resumed.
*
* RETURNS: OK or ERROR if invalid task id or unable to resume task
*/

STATUS taskResume (tid)
    int tid;			/* task id of task to resume */

    {
    return (psosResumeP (tid));
    }
/*******************************************************************************
*
* taskRestart - restart a task
*
* This routine "restarts" a task.  The task is first deleted.
* Then it is recreated, with the same id, priority, options, 
* original entry point, stack size, and (up to 10) parameters it had
* when it was deleted.
*
* NOTE
* If the task has modified any of its startup parameters, the
* restarted task will start with the changed values.
*
* RETURNS:
*  OK, or
*  ERROR if invalid task id, or couldn't be re-starged
*/

int taskRestart (tid)
    int tid;			/* task id of task to restart */

    {
    int priority;
    int options;
    FUNCPTR entry;		/* entry point of task */
    int stacksize;
    int args [MAX_TASK_ARGS];
    char *name;
    PSOS_PCB *pTcb = taskTcb (tid);
    TCBX *pTcbX;

    if (pTcb == NULL)
	return (ERROR);		/* specified task not found */

    /* save info from tcb that is about to be deleted */

    name	= (char *) pTcb->pcb_name;
    pTcbX	= pTcb->pcb_extend;
    priority	= psosPriorityToVx ((int)pTcb->pcb_prior);
    options	= pTcbX->options;
    entry	= pTcbX->entry;
    stacksize	= pTcbX->botOfStack - pTcbX->topOfStack;
    bcopy ((char *) pTcbX->initialSP, (char *) args, sizeof (args));

    (void)taskDelete (tid);

    return (taskSpawn (name, priority, options, stacksize, entry,
		       args [0], args [1], args [2], args [3], args [4],
		       args [5], args [6], args [7], args [8], args [9]));
    }
/*******************************************************************************
*
* taskOptionsSet - change task options
*
* Change the execution options of a task.
*
* Bits in the options argument may be set to run with the following modes.
* .CS
*       VX_SUPERVISOR_MODE      -  execute in supervisor mode
*       VX_UNBREAKABLE          -  don't allow debugging
*       VX_DEALLOC_STACK        -  deallocate the stack on deletion
*       VX_FP_TASK              -  execute with coprocessor support
*       VX_STDIO                -  execute with standard I/O support
* .CE
* See definitions in taskLib.h.
*
* RETURNS: OK, or ERROR if invalid task id
*
* SEE ALSO: taskOptionsSet (2)
*/

STATUS taskOptionsSet (tid, mask, value)
    int tid;		/* task id                         */
    int mask;		/* mask, 1 = ok to change this bit */
    int value;		/* new options                     */

    {
    FAST TCBX *pTcbX = taskTcbX (tid);

    if (pTcbX == NULL)
	return (ERROR);

    pTcbX->options = (pTcbX->options & ~mask) | value;

    /* if we are setting/resetting unbreakable option for current task,
     * then call breakpoint hook */

    if ((mask & VX_UNBREAKABLE) &&
	((tid == 0) || (tid == taskIdCurrent)) &&
	(taskBpHook != NULL))
	{
	(* taskBpHook) ((value & VX_UNBREAKABLE) == 0);
	}

    return (OK);
    }
/*******************************************************************************
*
* taskOptionsGet - examine task options
*
* This routine gets the specified tasks current execution options.
*
* Bits in the options argument may be set to run with the following modes.
* .CS
*       VX_SUPERVISOR_MODE      -  execute in supervisor mode
*       VX_UNBREAKABLE          -  don't allow debugging
*       VX_DEALLOC_STACK        -  deallocate the stack on deletion
*       VX_FP_TASK              -  execute with coprocessor support
*       VX_STDIO                -  execute with standard I/O support
* .CE
* See definitions in taskLib.h.
*
* RETURNS: OK or ERROR if invalid task id
*
* SEE ALSO: taskOptionsSet (2)
*/

STATUS taskOptionsGet (tid, pOptions)
    int tid;		/* task id        */
    int *pOptions;	/* task's options */

    {
    FAST TCBX *pTcbX = taskTcbX (tid);

    if (pTcbX == NULL)
	return (ERROR);

    *pOptions = pTcbX->options;
    return (OK);
    }
/*******************************************************************************
*
* taskPrioritySet - change priority of a task
*
* This routine changes the specified task's priority to the specified priority.
*
* RETURNS: OK or ERROR if invalid task id or unable to change priority
*
* SEE ALSO: taskPriorityGet (2)
*/

STATUS taskPrioritySet (tid, newPriority)
    int tid;			/* task id      */
    int newPriority;		/* new priority */

    {
    int oldPriority;		/* where to return old process priority */
    int increment;
    PSOS_PCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)
	return (ERROR);
    
    increment = psosPriorityToVx ((int)pTcb->pcb_prior) - newPriority;

    return (psosPriorityP (tid, increment, &oldPriority));
    }
/*******************************************************************************
*
* taskPriorityGet - examine priority of a task
*
* This routine is used to determine the current priority of a specified task.
*
* RETURNS:
*  OK and pPriority gets task priority, or
*  ERROR if invalid task id
*
* SEE ALSO: taskPrioritySet (2)
*/

STATUS taskPriorityGet (tid, pPriority)
    int tid;		/* task id                  */
    int *pPriority;	/* where to return priority */

    {
    if (psosPriorityP (tid, 0, pPriority) != OK)
	return (ERROR);

    *pPriority = psosPriorityToVx (*pPriority);
    return (OK);
    }
/*******************************************************************************
*
* taskLock - disable kernel rescheduling
*
* This routine disables task context switching.
* The task that calls this routine
* must never preempt itself while task switching is disabled.
* For instance, semaphores should not be taken.
*
* This routine does not lock out interrupts.
* To do so one should call intLock (2).
* taskLock (2) is preferable to intLock (2) because locking out interrupts
* adds interrupt latency to the system.
*
* RETURNS: OK or ERROR
*
* SEE ALSO: taskUnlock(2), intLock(2)
*/

STATUS taskLock ()

    {
    int oldMode;
    int status = psosModeP (0x80, 0x80, &oldMode);

    contextSwitching = FALSE;		/* breakpoints are now ignored */
    return (status);
    }
/*******************************************************************************
*
* taskUnlock - enable kernel rescheduling
*
* This routine is used to resume task context switching after a
* taskLock (2) has disabled it.
* Any tasks which were eligible to preempt the current task will now execute.
*
* RETURNS: OK
*
* SEE ALSO: taskLock (2)
*/

STATUS taskUnlock ()

    {
    int oldMode;

    contextSwitching = TRUE;		/* breakpoint handling resumed */

    return (psosModeP (0x00, 0x80, &oldMode));
    }
/*******************************************************************************
*
* taskDelay - delay a task from executing
*
* This routine caused the calling task to relinquish the CPU for the duration
* (in ticks) specified.  This is commonly refered to as manual rescheduling but
* it is also useful when waiting for some external condition that does not have
* an interrupt associated with it.
*
* RETURNS: OK or ERROR if invalid task id
*/

STATUS taskDelay (ticks)
    int ticks;		/* number of ticks for each task */

    {
    return (psosPauseP (ticks));
    }
/*******************************************************************************
*
* taskTcb - get the task control block for a task id
*
* This routine simply calls taskIdVerify (2) to check task id, then
* returns tid as a TCB.
*
* RETURNS: pointer to pSOS TCB, or NULL if invalid task id
*/

PSOS_PCB *taskTcb (tid)
    int tid;			/* task id */

    {
    if (tid == 0)
	tid = taskIdCurrent;
    else
	{
	if (taskIdVerify (tid) != OK)
	    return (NULL);
	}

    return ((PSOS_PCB *) tid);
    }
/*******************************************************************************
*
* taskTcbX - get the task control block extension for a task id
*
* This routine returns a pointer to the task control block extension.
* The contents of the TCBX is defined in taskLib.h.
* Users should not directly modify the contents of the TCB extension.
*
* RETURNS: pointer to TCB extension, or NULL if invalid task id
*/

TCBX *taskTcbX (tid)
    int tid;			/* task id */

    {
    if (tid == 0)
	tid = taskIdCurrent;
    else if (taskIdVerify (tid) != OK)
	{
	errnoSet (S_taskLib_TASK_ID_ERROR);
	return (NULL);
	}

    return (((PSOS_PCB *)tid)->pcb_extend);
    }
/*******************************************************************************
*
* taskTcbToTd - take a pTcb and fill in the task descriptor structure
*
* This routine assumes we have a legal pTcb.  It is a local routine that
* does all the work for the higher level call taskInfoGet (2).
*/

LOCAL VOID taskTcbToTD (pTcb, pTaskDesc)
    PSOS_PCB *pTcb;		/* pointer to pSOS task control block */
    TASK_DESC *pTaskDesc;	/* task descriptor to be filled in    */

    {
    FAST TCBX *pTcbX;

    pTaskDesc->td_name		= (char *)pTcb->pcb_name; /* name of task */
    pTaskDesc->td_priority	= psosPriorityToVx ((int)pTcb->pcb_prior);
   							/* task priority */
    pTaskDesc->td_id		= (int)pTcb;		/* task id */
    pTaskDesc->td_status	= pTcb->pcb_status;	/* task status*/
    pTaskDesc->td_sp		= pTcb->pcb_ssp;	/* stack pointer save */
    pTaskDesc->td_usp		= pTcb->pcb_usp;	/* user stack pointer */
    pTaskDesc->td_delay		= pTcb->pcb_spare2;	/* delay/timeout ticks*/

    pTcbX = pTcb->pcb_extend;

    if (pTcbX == NULL)
	{
	pTaskDesc->td_spbottom	= 0;			/* bottom of stack ptr*/
	pTaskDesc->entry	= 0;			/* entry of task */
	pTaskDesc->errorStatus	= 0;			/* most recent error */
	pTaskDesc->topOfStack	= 0;			/* top of stack */
	pTaskDesc->initialSP	= 0;			/* initial stack ptr. */
	pTaskDesc->options	= 0;			/* task option bits */
	}
    else
	{
	pTaskDesc->td_spbottom	= pTcbX->botOfStack;	/* bottom of stack ptr*/
	pTaskDesc->entry	= pTcbX->entry;		/* entry of task */
	pTaskDesc->errorStatus	= pTcbX->errorStatus;	/* most recent error */
	pTaskDesc->topOfStack	= pTcbX->topOfStack;	/* top of stack */
	pTaskDesc->initialSP	= pTcbX->initialSP;	/* initial stack ptr. */
	pTaskDesc->options	= pTcbX->options;	/* task option bits */
	}
    }
/*******************************************************************************
*
* taskInfoGet - get pointer to task's task descriptor
*
* This routine finds the TASK_DESC (task descriptor) of the specified task.
*
* NOTE
* Examination of TCBs should be restricted to debugging aids.
*
* RETURNS: OK or ERROR if invalid task id
*
* SEE ALSO: pSOS documentation
*/

STATUS taskInfoGet (tid, pTaskDesc)
    int tid;			/* task id                        */
    TASK_DESC *pTaskDesc;	/* where to store task descriptor */

    {
    PSOS_PCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)		/* task non-existant */
	return (ERROR);

    taskTcbToTD (pTcb, pTaskDesc);
    return (OK);
    }
/*******************************************************************************
*
* taskName - get name associated with a task id
*
* This routine returns the name of a task.
*
* RETURNS: pointer to task name, or NULL if invalid task id
*/

char *taskName (tid)
    int tid;		/* task id */

    {
    if (tid == 0)
	tid = taskIdCurrent;
    else if (taskIdVerify (tid) != OK)
	return (NULL);

    return ((char *)(((PSOS_PCB *) tid)->pcb_name));
    }
/*******************************************************************************
*
* taskNameToId - lookup task id associated with a name
*
* This routine returns the task id.
*
* RETURNS: task id or ERROR if task by name `name' not found
*/

int taskNameToId (name)
    char *name;		/* name of task to look up */

    {
    PSOS_PCB *pTcb = ((PDAT *) (usrCftbl.ramStart1))->pActiveHead;
					/* ptr to the running pcb */ 

    while ((int)pTcb != 0)			/* find named task */
	{
	if (strcmp (pTcb->pcb_extend->name, name) == 0)
	    return ((int) pTcb);

        pTcb = pTcb->pcb_thread;		/* handle on next task */
	}

    errnoSet (S_taskLib_NAME_NOT_FOUND);
    return (ERROR);				/* found no match */
    }
/*******************************************************************************
*
* taskIsSuspended - check if a task is suspended
*
* This routine tests the status field of the specified task to determine
* if it is suspended.
*
* RETURNS: TRUE if task is suspended, otherwise FALSE
*/

BOOL taskIsSuspended (tid)
    int tid;	/* task id */

    {
    FAST PSOS_PCB *pTcb = taskTcb (tid);

    if (pTcb != NULL && pTcb->pcb_status & PSOS_SUSPEND)
	return (TRUE);

    return (FALSE);
    }
/*******************************************************************************
*
* taskIsReady - check if a task is ready to run
* This routine tests the status field of the specified task to determine
* if it is ready.
*
* RETURNS: TRUE if task is ready, otherwise FALSE
*/

BOOL taskIsReady (tid)
    int tid;	/* task id */

    {
    FAST PSOS_PCB *pTcb = taskTcb (tid);

    if (pTcb != NULL && pTcb->pcb_status == PSOS_READY)
	return (TRUE);

    return (FALSE);
    }
/*******************************************************************************
*
* taskStatusString - return the task's status as a string
*
* This routine converts the pSOS task status word in the TCB
* and returns the appropriate string.
*
* EXAMPLE
* .CS
*  -> taskStatusString (taskNameToId ("shell"), xx=malloc(10));
*  new symbol "xx" added to symbol table.
*  value = 0 = 0x0
*  -> printf ("shell status = <%s>\n", xx)
*  shell status = <READY>
*  value = 2 = 0x2
*  ->
* .CE
*
* RETURNS:
*  OK, or ERROR if invalid task id, `pString' gets status string
*/

STATUS taskStatusString (tid, pString)
    int tid;		/* task to get string for           */
    char *pString;	/* where to return string (10 bytes)*/

    {
    FAST PSOS_PCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)
	return (ERROR);

    if (pTcb->pcb_status == PSOS_SUSPEND)
	{
	strcpy (pString, "SUSPEND");
	return (OK);
	}

    switch (pTcb->pcb_status & 0xf8)
	{
	case PSOS_READY:	strcpy (pString, "READY");  break;
	case PSOS_DELAY:	strcpy (pString, "DELAY");  break;
	case PSOS_EVENT:	strcpy (pString, "EVENT");  break;
	case PSOS_EXCHANGE:	strcpy (pString, "EXCHG");  break;
	case PSOS_MEM:		strcpy (pString, "MEM");    break;
	case PSOS_DEAD:		strcpy (pString, "INACT");  break;

	default:			/* unanticipated combination */
	    sprintf (pString, "0x%02x", pTcb->pcb_status);
	    return (OK);
	}

    if (pTcb->pcb_status & PSOS_SUSPEND)
	strcat (pString, "+S");		/* task also explicitly suspended */

    return (OK);
    }
/*******************************************************************************
*
* taskIdDefault - default task id
*
* This routine maintains a global default task id.  This id is used by 
* libraries that want to allow a task id argument to take on a default
* value if the user did not explicitly supply one.
* If the task id supplied to this routine is not zero (i.e. the user
* did specify a taskId) then the default id is set to that value,
* and that value is returned.  If the task id supplied to this routine
* is zero (i.e. the user did not specify a task id) then the default id
* is not changed and its value is returned.  Thus the value returned
* is always the last task id the user specified.
*
* RETURNS: most recently set non-zero task id
*
* SEE ALSO: dbgLib (1), "Debugging"
*/

int taskIdDefault (tid)
    int tid;	/* user supplied task id, if 0 then just return default */

    {
    static int defaultTaskId;	/* current default task id */

    if (tid != 0)
	defaultTaskId = tid;		/* update default */

    return (defaultTaskId);
    }
/*******************************************************************************
*
* taskIdSelf - get task id of running task
*
* Get task id of calling task. The task id will be invalid if called
* at interrupt level.  This routine simply returns global variable
* `taskIdCurrent' which is only valid during a task context.
*
* RETURNS: task id of calling task
*/

int taskIdSelf ()

    {
    return (taskIdCurrent);
    }
/*******************************************************************************
*
* taskIdVerify - verify the existence of a task
*
* RETURNS: OK or ERROR if invalid task id
*/

STATUS taskIdVerify (tid)
    int tid;	/* task id */

    {
    /* check that tid is in psos work space, then that its key is ok */

    if ((tid < (int)usrCftbl.ramStart1) || (tid > (int)usrCftbl.ramEnd1) ||
	(((PSOS_PCB *)tid)->pcb_key !=
	 ((tid & 0xffff0000) | (~tid & 0x0000ffff))) ||
	(((PSOS_PCB *)tid)->pcb_status == PSOS_INACTIVE))
	{
	errnoSet (S_taskLib_TASK_ID_ERROR);
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* taskIdListGet - get list of active task ids
*
* This routine provides the calling task with a list of all of the active
* tasks.  The list consists of task ids, the last task id is a terminating 0.
* The list is unordered.
*
* WARNING:
* Kernel rescheduling is disabled with taskLock (2) while
* tasks are looked up.  There is no guarantee that all the
* tasks are valid and/or no new tasks have been created by
* the time this routine returns.
*
* RETURNS: number of tasks in list
*/

int taskIdListGet (idList, maxTasks)
    FAST int idList[];		/* array of task ids to be filled in */
    FAST int maxTasks;		/* max tasks idList can accommodate  */

    {
    FAST int *pId  = idList;
    PSOS_PCB *pTcb = ((PDAT *) (usrCftbl.ramStart1))->pActiveHead;
					/* ptr to the running pcb */ 

    taskLock ();

    while ((pTcb != NULL) && (--maxTasks >= 0))
	{
	*(pId++) = (int) pTcb;
        pTcb = pTcb->pcb_thread;
	}

    taskUnlock ();

    return (pId - idList);	/* return count of active tasks */
    }
/*******************************************************************************
*
* taskRegsGet - get a task's registers from TCB and stack
*
* This routine gathers information about a task which is normally kept
* scattered about.  It returns, in the locations whose pointers are
* passed as parameters, the task's address and data registers, and its
* SP, SR and PC.  The address and data registers are returned in
* separate arrays, each containing 8 registers.
*
* NOTE
* This routine only works well if the specified task's TCB is not that
* of the calling task.  Results are unpredictable if a task tries to
* discover its own registers.
*
* RETURNS: OK or ERROR if invalid task id
*
* SEE ALSO: taskRegsSet (2)
*/

STATUS taskRegsGet (tid, dregs, aregs, sp, sr, pc)
    int tid;		/* task id                            */
    int dregs[];	/* buffer for data registers (0-7)    */
    int aregs[];	/* buffer for address registers (0-6) */
    char **sp;		/* buffer for stack pointer           */
    USHORT *sr;		/* buffer for status register         */
    INSTR **pc;		/* buffer for program counter         */

    {
    FAST PSOS_PCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)		/* task non-existant */
	return (ERROR);

    dregs[0] = pTcb->pcb_d0;
    dregs[1] = pTcb->pcb_d1;
    dregs[2] = pTcb->pcb_d2;
    dregs[3] = pTcb->pcb_d3;
    dregs[4] = pTcb->pcb_d4;
    dregs[5] = pTcb->pcb_d5;
    dregs[6] = pTcb->pcb_d6;
    dregs[7] = pTcb->pcb_d7;
    
    aregs[0] = pTcb->pcb_a0;
    aregs[1] = pTcb->pcb_a1;
    aregs[2] = pTcb->pcb_a2;
    aregs[3] = pTcb->pcb_a3;
    aregs[4] = pTcb->pcb_a4;
    aregs[5] = pTcb->pcb_a5;
    aregs[6] = pTcb->pcb_a6;

    *sp = pTcb->pcb_ssp;
    *sr = pTcb->pcb_sr;
    *pc = pTcb->pcb_pc;

    return (OK);
    }
/*******************************************************************************
*
* taskRegsSet - set a task's registers 
*
* This routine loads the given values into the given tasks TCB.
* The registers are contained in two arrays, which contain d0-d7
* and a0-a7.
*
* RETURNS: OK or ERROR if invalid task id
*
* SEE ALSO: taskRegsGet (2)
*/

STATUS taskRegsSet (tid, pDregs, pAregs, sp, sr, pc)
    int tid;		/* task id                 */
    int pDregs[];	/* data registers (0-7)    */
    int pAregs[];	/* address registers (0-6) */
    char *sp;		/* stack pointer           */
    USHORT sr;		/* status register         */
    INSTR *pc;		/* program counter         */

    {
    FAST PSOS_PCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)		/* task non-existant */
	return (ERROR);

    pTcb->pcb_d0 = pDregs[0];
    pTcb->pcb_d1 = pDregs[1];
    pTcb->pcb_d2 = pDregs[2];
    pTcb->pcb_d3 = pDregs[3];
    pTcb->pcb_d4 = pDregs[4];
    pTcb->pcb_d5 = pDregs[5];
    pTcb->pcb_d6 = pDregs[6];
    pTcb->pcb_d7 = pDregs[7];

    pTcb->pcb_a0 = pAregs[0];
    pTcb->pcb_a1 = pAregs[1];
    pTcb->pcb_a2 = pAregs[2];
    pTcb->pcb_a3 = pAregs[3];
    pTcb->pcb_a4 = pAregs[4];
    pTcb->pcb_a5 = pAregs[5];
    pTcb->pcb_a6 = pAregs[6];
    pTcb->pcb_ssp = sp;

    pTcb->pcb_sr = sr;

    pTcb->pcb_pc = pc;

    return (OK);
    }
/*******************************************************************************
*
* taskRegsShow - print contents of a task's registers
*
* This routine prints to standard out the contents of a task's registers.
*
* EXAMPLE
* .CS
*  -> taskRegsShow (taskNameToId ("shell"))
*
*  D0 =       0  D4 =       0  A0 =       0  A4 =       0
*  D1 =       0  D5 =       0  A1 =       0  A5 =  3f4a08  SR =   3004
*  D2 =       0  D6 =       0  A2 =       0  A6 =  3d6458  PC =  39ae2
*  D3 =       0  D7 =       a  A3 =       0  A7 =  3d6458
*  value = 1 = 0x1
* .CE
*/

VOID taskRegsShow (tid)
    int tid;		/* task id */

    {
    int dregs[8];		/* task's data registers */
    int aregs[7];		/* task's address registers */
    char *sp;			/* task's stack pointer */
    USHORT sr;			/* task's status register */
    INSTR *pc;			/* task's pc */

    if (taskRegsGet (tid, dregs, aregs, &sp, &sr, &pc) == ERROR)
	{
	printf ("taskRegsShow: invalid task id %#x\n", tid);
	return;
	}

    printf ("\n");
    printf ("D0 = %8x   D4 = %8x   A0 = %8x   A4 = %8x\n",
            dregs[0], dregs[4], aregs[0], aregs[4]);
    printf ("D1 = %8x   D5 = %8x   A1 = %8x   A5 = %8x   SR = %8x\n",
            dregs[1], dregs[5], aregs[1], aregs[5], sr & 0xffff);
    printf ("D2 = %8x   D6 = %8x   A2 = %8x   A6 = %8x   PC = %8x\n",
            dregs[2], dregs[6], aregs[2], aregs[6], pc);
    printf ("D3 = %8x   D7 = %8x   A3 = %8x   A7 = %8x\n",
            dregs[3], dregs[7], aregs[3], sp);
    }
/*******************************************************************************
*
* taskSRSet - set task status register
*
* This routine sets the status register of a task not running (i.e. the TCB
* must NOT be that of the calling task).  It is used by the debugging
* facilities to set the trace bit in the SR of a task being single-stepped.
*
* RETURNS: OK or ERROR if invalid task id
*/

STATUS taskSRSet (tid, sr)
    int tid;	 	/* task id */
    USHORT sr;		/* new SR  */

    {
    FAST PSOS_PCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)		/* task non-existant */
	return (ERROR);

    pTcb->pcb_sr = sr;

    return (OK);
    }
