/* psosALib.s - VxWorks/pSOS kernel interface and support */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
	.data
	.globl	_copyright_wind_river
	.long	_copyright_wind_river

/*
modification history
--------------------
01h,17apr89,jcf  fixed psosRetBuf.
01g,24aug88,gae  documetation.
01f,13aug88,jcf  changed intEnt and intExit to match new intLib.c.
01e,12aug88,jcf  fixed psosVersion so it works.
	    rdc  fixed psosStartup so we switch to interrupt state before
		  starting psos up.
	    dnw  added NO_MANUAL to ioJumpTable.
01d,22jun88,dnw  changed erroneous "moveml d2,.." that sun 'as' didn't like.
01c,05jun88,dnw  changed vxCurrentTask to taskIdCurrent.
		 changed to set psos module number in error status values.
01b,30may88,dnw  changed to v4 names.
01a,28jan88,jcf  written
*/

/*
DESCRIPTION
This library contains support for pSOS facilities.

C callable interface routines are provided for most pSOS system calls.  
Each interface routine gets the parameters passed to it in a normal 
C fashion (on the stack) and puts them in the proper registers where 
pSOS expects to find them, then traps to pSOS.

RETURNING PSOS STATUS CODES
If a pSOS system call returns an error status (non-zero d7 on return from pSOS),
the interface routines set the process error status to the pSOS error code
returned, then return the value ERROR to the calling routine.  That routine
can find out the error code that pSOS returned by calling errnoGet (2).
If all went well with pSOS, the interface routines return OK.

TASK CREATION
No direct access is provided to the pSOS process creation system calls.
Instead all VxWorks processs should be created by the taskSpawn (2) or
taskCreate (2) calls in taskLib (1).

CHARACTER I/O
VxWorks does not use or support the character I/O facility
provided by the pSOS I/O system.  Instead, VxWorks supplies a complete
device independent I/O system and a set of device drivers.

NOTE
Many routines in this module "link" and "unlk" the "c" frame pointer
(a6) although they don't use it in any way.  This only for the benefit of
the stack trace facility to allow it to properly trace processs executing within
these routines.
*/

#define ASMLANGUAGE
#include "vxWorks.h"
#include "asm.h"

#define RETI		4	/* trap used to get to RETI */

	/* internals */

	.globl _psosStartup
	.globl _psosVersion
	.globl _psosSpawnP
	.globl _psosActivateP
	.globl _psosDeleteP
	.globl _psosSuspendP
	.globl _psosResumeP
	.globl _psosPriorityP
	.globl _psosIdentP
	.globl _psosIdentPRoot
	.globl _psosModeP
	.globl _psosSuperP
	.globl _psosAllocSeg
	.globl _psosFreeSeg
	.globl _psosCreatePt
	.globl _psosDeletePt
	.globl _psosCreatePt
	.globl _psosGetBuf
	.globl _psosRetBuf
	.globl _psosCreateX
	.globl _psosAttachX
	.globl _psosDeleteX
	.globl _psosSendX
	.globl _psosJamX
	.globl _psosLiberX
	.globl _psosReqX
	.globl _psosSignalV
	.globl _psosWaitV
	.globl _psosGetV
	.globl _psosPauseP
	.globl _psosAnnounceT
	.globl _psosGetT
	.globl _psosSetT
	.globl _psosTswitchHook
	.globl _vxTaskEntry
	.globl _intEnt
	.globl _intExit
	.globl _ioJumpTable


	/* externals */

	.globl _errnoSet
	.globl _sysKernelTrap		/* trap to the kernel */
	.globl _taskSwitchTable		/* table of process switch routines */
	.globl _taskIdCurrent		/* global variable of who is running */

	.text
	.even

/*******************************************************************************
*
* ioJumpTable - the jump table pointed to in the configuration table
*
* psos forces us to produce one device which simply returns for each of
* the six routines.
*
* NOMANUAL
*/

_ioJumpTable:
	moveb	#0,d7		/* clear error code */
	rts			/* go back to psos */
	moveb	#0,d7		/* clear error code */
	rts			/* go back to psos */
	moveb	#0,d7		/* clear error code */
	rts			/* go back to psos */
	moveb	#0,d7		/* clear error code */
	rts			/* go back to psos */
	moveb	#0,d7		/* clear error code */
	rts			/* go back to psos */
	moveb	#0,d7		/* clear error code */
	rts			/* go back to psos */

/*******************************************************************************
*
* psosStartup - start pSOS multi-processing
*
* This routine is a C interface to the pSOS initial jump.
* If the tables have all been set up right we should pop out into
* the routine usrRoot.
*
* RETURNS: Doesn't!
*
* SEE ALSO: pSOS User's Guide

* VOID psosStartup ()

*/

_psosStartup:
	/* pSOS requires that we be in interrupt state on entry
	   so that the interrupt stack get's set up properly */

	movel 	a7, d1		/* make a copy of sp */
	movew	sr, d0
	andw	#0xefff,d0	/* turn off the M bit */
	movew	d0,sr
	movel	d1,a7		/* restore the sp */
	jmp	_psosStart	/* to psos, we never come back. */

/*******************************************************************************
*
* psosVersion - return version number and revision
*
* This routine serves as a C interface to pSOS system call VERSION.
*
* RETURNS: OK
*	
* SEE ALSO: pSOS User's Guide

* STATUS psosVersion (pVersion)
*     int *pVersion;	 /* version number of pSOS *

*/

_psosVersion:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	moveq	#0,d7			/* VERSION */
	jsr	_sysKernelTrap		/* trap to kernel */

	andl	#0xffff,d7		/* mask off the noise */
	movel	a6@(ARG1),a1		/* get pVersion */
	movel	d7,a1@			/* stick version in ptr. */

	clrl	d7			/* make error status OK */
	bra	return

/*******************************************************************************
*
* psosSpawnP - spawn process
*
* This routine serves as a C interface to pSOS system call SPAWN_P.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with the following value:
*	1 = cannot allocate PCB, maximum active processes exceeded
*	2 = cannot allocate stack segment, no usable memory left
*	3 = stack to small to hold arglist
*	4 = priority = 0; reserved
*
* SEE ALSO: pSOS User's Guide

* int psosSpawnP (name, grId, priority, stkSize, argList, pid)
*     int name;		/* 4 byte name                    *
*     int grId;		/* groutp id number               *
*     int priority;	/* priority                       *
*     int stkSize;	/* stack size                     *
*     int *argList;	/* address of argList             *
*     int *pid;		/* where to put process id number *

*/

_psosSpawnP:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	moveml	d2-d3,a7@-		/* save reg d2-3 */
	movel	a6@(ARG1),d0		/* get name */
	movel	a6@(ARG2),d1		/* get grId */
	movel	a6@(ARG3),d2		/* get priority */
	movel	a6@(ARG4),d3		/* get stkSize */
	movel	a6@(ARG5),a0		/* get address of arglist */

	moveq	#1,d7			/* SPAWN_P */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	a6@(ARG6),a1		/* get pointer to pid */
	movel	a0,a1@			/* return pid */

        moveml	a7@+,d2-d3		/* restore reg d2-3 */

	movel	#1,d0			/* SPAWN_P */
	bra	return

/*******************************************************************************
*
* psosActivateP - activate a newly spawned process
*
* This routine serves as a C interface to pSOS system call SPAWN_P.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with the following value:
*	1 = process with this pid has been deleted
*	2 = pid failed validity check
*	3 = caller not of same group or group 0
*	5 = process already activated
*
* SEE ALSO: pSOS User's Guide

* int psosActivateP (pid, startAdr)
*     int pid;		/* process id    *
*     int *startAdr;	/* start address *

*/

_psosActivateP:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get pid */
	movel	a6@(ARG2),a1		/* get startAdr */

	moveq	#2,d7			/* ACTIVATE_P */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#2,d0			/* ACTIVATE_P */
	bra	return

/*******************************************************************************
*
* psosIdentP - name to process id, current status
*
* This routine serves as a C interface to pSOS system call IDENT_P.
* This routine gives the process id used for most calls.  Only SPAWN_P
* and IDENT_P use the process name for parameters.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with the following value:
*	1 = process not found (no process with specified id number)
*
* SEE ALSO: pSOS User's Guide

* STATUS psosIdentP (name, pid, status)
*     int id;		/* process id, or 0 to inquire about calling process *
*     int *pid;		/* where to return the process's id number           *
*     int *status;	/* where to return the process's status              *

*/

_psosIdentP:
	link	a6,#0
	movel	d7,a7@-			/* save reg d7 */

	movel	a6@(ARG1),d0		/* get name */

	moveq	#3,d7			/* IDENT_P */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	a6@(ARG2),a1		/* get pointer to id */
	movel	a0,a1@			/* return id */
	movel	a6@(ARG3),a1		/* get pointer to status */
	andl	#0xff,d0		/* mask off high order bits */
	movel	d0,a1@			/* return status */

	movel	#3,d0			/* IDENT_P */
	bra	return

/*******************************************************************************
*
* psosIdentPRoot - get root process id
*
* RETURNS: OK or ERROR
*	
*
* SEE ALSO: pSOS User's Guide

* int psosIdentPRoot ()

*/

_psosIdentPRoot:
	link	a6,#0
	movel	d7,a7@-			/* save reg d7 */

	clrl	d0			/* name = 0: get id of self */
	moveq	#3,d7			/* IDENT_P */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	a0,d0			/* return id */

        movel	a7@+,d7			/* restore reg d7 */

	unlk	a6
	rts

/*******************************************************************************
*
* psosDeleteP - process delete
*
* This routine serves as a C interface to pSOS system call DELETE_P.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with the following value:
*	1 = process already deleted (no process with specified id number)
*	2 = process id incorrect, (process failed validity check)
*	3 = process access violation (process not in correct group)
*	4 = process has files open (pHILE only)
*
* SEE ALSO: pSOS User's Guide

* int psosDeleteP (id)
*     int id;		/* process id number *

*/

_psosDeleteP:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get id */

	moveq	#4,d7			/* DELETE_P */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#4,d0			/* DELETE_P */
	bra	return

/*******************************************************************************
*
* psosSuspendP - process suspend
*
* This routine serves as a C interface to pSOS system call SUSPEND_P.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with the following value:
*	1 = process deleted (no process with specified id number)
*	2 = process id incorrect, (process failed validity check)
*	3 = process access violation (process not in correct group)
*	4 = process already suspended
*
* SEE ALSO: pSOS User's Guide

* int psosSuspendP (id)
*     int id;		/* process id number *

*/

_psosSuspendP:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get id */

	moveq	#5,d7			/* SUSPEND_P */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#5,d0			/* SUSPEND_P */
	bra	return

/*******************************************************************************
*
* psosResumeP - process resume
*
* This routine serves as a C interface to pSOS system call RESUME_P.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with the following value:
*	1 = process deleted (no process with specified id number)
*	2 = process id incorrect, (process failed validity check)
*	3 = process access violation (process not in correct group)
*	4 = process not suspended
*
* SEE ALSO: pSOS User's Guide

* int psosResumeP (id)
*     int id;		/* process id number *

*/

_psosResumeP:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get id */

	moveq	#6,d7			/* RESUME_P */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#6,d0			/* RESUME_P */
	bra	return

/*******************************************************************************
*
* psosPriorityP - process priority change
*
* This routine serves as a C interface to pSOS system call PRIORITY_P.
* It changes the priority of an existing process by adding an offset to the
* current priority.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with the following value:
*	1 = process deleted (no process with specified id number)
*	2 = process id incorrect, (process failed validity check)
*	3 = process access violation (process not in correct group)
*	0x81 = priority overflow, new priority = 255
*	0x82 = priority underflow, new priority = 1
*
* SEE ALSO: pSOS User's Guide

* int psosPriorityP (id, increment, pOldPriority)
*     int id;			/* process id number                    *
*     int increment;		/* new process priority                 *
*     int *pOldPriority;	/* where to return old process priority *

*/

_psosPriorityP:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get id */
	movel	a6@(ARG2),d0		/* priority offset */

	moveq	#7,d7			/* PRIORITY_P */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	a6@(ARG3),a1		/* get pointer to id */
	andl	#0xff,d0		/* mask high order bits */
	movel	d0,a1@			/* return id */

	movel	#7,d0			/* PRIORITY_P */
	bra	return

/*******************************************************************************
*
* psosModeP - configure pSOS scheduling
*
* This routine serves as a C interface to pSOS system call MODE_P.
* Bit 7 of mode is (0 = preemptive scheduling, 1 = preemptive scheduling lock)
* Bit 6 of mode is (0 = round-robin is off, 1 round-robin is on)
*
* RETURNS: OK
*	
* SEE ALSO: pSOS User's Guide

* int psosModeP (newMode, mask, oldMode)
*     int newMode;	/* mode desired                         *
*     int mask;		/* mask; 1 = ok to change this bit      *
*     int *oldMode;	/* where to return the old mode setting *
*/

_psosModeP:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),d0		/* get newMode */
	movel	a6@(ARG2),d1		/* get mask */

	moveq	#32,d7			/* MODE_P */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	a6@(ARG3),a1		/* get pointer to oldMode */
	andl	#0xff,d0		/* mask off the high order bits */
	movel	d0,a1@			/* return oldMode */

	movel	#32,d0			/* MODE_P */
	bra	return

/*******************************************************************************
*
* psosSuperP - put process is supervisor state
*
* This routine serves as a C interface to pSOS system call SUPER_P.
*
* RETURNS: OK
*	 
* SEE ALSO: pSOS User's Guide

* int psosSuperP (oldSR)
*     short *oldSR;	/* where to put old status register *
*/

_psosSuperP:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a7,d1			/* save our stack ptr */

	moveq	#33,d7			/* SUPER_P */
	trap	#0			/* XXX trap to kernel XXX */

	movel	d1,a7			/* restore our stack ptr in case
					 * we changed stacks from usp->ssp */

        movel	a6@(ARG1),a1		/* pointer to old status register */
	movew	d0,a1@			/* return status register */

	movel	#33,d0			/* SUPER_P */
	bra	return

/*******************************************************************************
*
* psosAllocSeg - get memory segment
*
* This routine serves as a C interface to pSOS system call ALLOC_SEG.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet will be called with one of the following values:
*	 2 = region parametric error
*	0x80 = no free segment (only if no-wait option selected)
*	0x81 = timed out (only if wait and timeout options selected)
*
* SEE ALSO: pSOS User's Guide

* int psosAllocSeg (size, region, wait, timeout, address)
*     int size;	      /* required segment size                     *
*     int region;     /* 0=don't care, 1=region_1, 2=region_2      *
*     int wait;	      /* 0=wait for segment, !=0 unconditional ret *
*     int timeout;    /* # of clock ticks (0 = forever)            *
*     char **address; /* receives the logical address of the block *

*/

_psosAllocSeg:
	link	a6,#0
	movel	d7,a7@-			/* save regs */
	moveml	d2-d3,a7@-		/* save regs */
	movew	a6@(ARG1),d1		/* get size */
	movew	a6@(ARG2),d2		/* get region */
	movew	a6@(ARG3),d3		/* get wait */
	movew	a6@(ARG4),d0		/* get timeout */

	moveq	#8,d7			/* ALLOC_SEG */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	a6@(ARG5),a1		/* get pointer to address */
	movel	a0,a1@			/* return address */
	moveml	a7@+,d2-d3		/* restore regs */

	movel	#8,d0			/* ALLOC_SEG */
	bra	return

/*******************************************************************************
*
* psosFreeSeg - free memory segment
*
* This routine serves as a C interface to pSOS system call FREE_SEG.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet will be called with one of the following values:
*	 1 = segment not owned by caller
*	 5 = bad segment address
*
* SEE ALSO: pSOS User's Guide

* int psosFreeSeg (address)
*     char *address;	/* logical address of the block *

*/

_psosFreeSeg:
	link	a6,#0
	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get address */

	moveq	#9,d7			/* FREE_SEG */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#9,d0			/* FREE_SEG */
	bra	return

/*******************************************************************************
*
* psosAssignSeg - assign ownership to a segment
*
* This routine serves as a C interface to pSOS system call ASSIGN_SEG.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet will be called with one of the following values:
*	 1 = segment not owned by caller
*	 2 = incorrect pid (failed validity check)
*	 3 = violation, caller must be in same group or group 0
*	 5 = bad segment address
*	
* SEE ALSO: pSOS User's Guide

* int psosAssignSeg (address, newPid)
*     char *address;	/* segment to affect  *
*     int newPid;	/* new owning process *
*/

_psosAssignSeg:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get newMode */
	movel	a6@(ARG2),a1		/* get pointer to id */

	moveq	#10,d7			/* ASSIGN_SEG */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#10,d0			/* ASSIGN_SEG */
	bra	return

/*******************************************************************************
*
* psosGrabSeg - take ownership of a segment
*
* This routine serves as a C interface to pSOS system call GRAP_SEG.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet will be called with one of the following values:
*	 1 = target process not owner of segment
*	 2 = incorrect pid (failed validity check)
*	 3 = violation, caller must be in same group or group 0
*	 5 = bad segment address
*	
* SEE ALSO: pSOS User's Guide

* int psosGrabSeg (address, fromPid)
*     char *address;	/* segment to affect      *
*     int fromPid;	/* current owning process *
*/

_psosGrabSeg:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get newMode */
	movel	a6@(ARG2),a1		/* get pointer to id */

	moveq	#16,d7			/* GRAB_SEG */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#16,d0			/* GRAB_SEG */
	bra	return

/*******************************************************************************
*
* psosCreatePt - create memory partition
*
* This routine serves as a C interface to pSOS system call CREATE_PT.
*
* RETURNS: OK
*	 
* SEE ALSO: pSOS User's Guide

* int psosCreatePt (paddress, bufSize, numBufs, ptId)
*     char *paddress;	/* partition starting address *
*     int bufSize;	/* buffer size                *
*     int numBufs;	/* number of buffers to make  *
*     int *ptId;	/* partition's id             *

*/

_psosCreatePt:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get partition address */
	movel	a6@(ARG2),d0		/* get id */
	movel	a6@(ARG3),d1		/* get partition size */

	moveq	#35,d7			/* CREATE_PT */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	a6@(ARG4),a1		/* pointer to partition id */
	movel	a0,a1@			/* return id */

	movel	#35,d0			/* CREATE_PT */
	bra	return

/*******************************************************************************
*
* psosDeletePt - delete memory partition
*
* This routine serves as a C interface to pSOS system call DELETE_PT.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet will be called with one of the following values:
*	 1 = buffer size does not match partition
*	 2 = cannot delete because some buffers are still in use
*	 
* SEE ALSO: pSOS User's Guide

* int psosDeletePt (ptId, bufSize)
*     int ptId;		/* partition's id *
*     int bufSize;	/* buffer size    *

*/

_psosDeletePt:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get partition id */
	movel	a6@(ARG2),d0		/* get buffer size */

	moveq	#36,d7			/* CREATE_PT */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#36,d0			/* CREATE_PT */
	bra	return

/*******************************************************************************
*
* psosGetBuf - get buffer from memory partition
*
* This routine serves as a C interface to pSOS system call GET_BUF.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet will be called with one of the following values:
*	 1 = buffer size does not match partition
*	 2 = no more partition buffers
*	 
* SEE ALSO: pSOS User's Guide

* int psosGetBuf (ptId, bufSize,bufAdr)
*     int ptId;		/* partition's id          *
*     int bufSize;	/* buffer size             *
*     int *bufAdr;	/* start address of buffer *

*/

_psosGetBuf:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get partition id */
	movel	a6@(ARG2),d0		/* get buffer size */

	moveq	#37,d7			/* GET_BUF */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	a6@(ARG3),a0		/* get buffer address */
	movel	a1,a0@			/* return buffer address */

	movel	#37,d0	 		/* GET_BUF */
	bra	return

/*******************************************************************************
*
* psosRetBuf - return buffer to a memory partition
*
* This routine serves as a C interface to pSOS system call RET_BUF.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet will be called with one of the following values:
*	 1 = buffer size does not match partition
*	 
* SEE ALSO: pSOS User's Guide

* int psosRetBuf (ptId, bufAdr)
*     int ptId;		/* partition's id          *
*     int bufAdr;	/* start address of buffer *

*/

_psosRetBuf:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get partition id */
	movel	a6@(ARG2),a1		/* get buffer address */

	moveq	#38,d7			/* RET_BUF */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#38,d0			/* RET_BUF */
	bra	return

/*******************************************************************************
*
* psosCreateX - create a message exchange
*
* This routine serves as a C interface to pSOS system call CREATE_X.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with one of the following values:
*	1 = cannot create exchange because no unused XCB left
*
* SEE ALSO: pSOS User's Guide

* int psosCreateX (name, property, msgQMax, xid)
*     int name;		/* 4 byte name code    *
*     int property;	/* property mask       *
*     int msgQMax;	/* msgQMax mask        *
*     int *xid;	 	/* where to return xid *

*/

_psosCreateX:
	link	a6,#0
	movel	d7,a7@-			/* save reg d7 */
	movel	d2,a7@-			/* save reg d2 */
	movel	a6@(ARG1),d0		/* get name */
	movel	a6@(ARG2),d1		/* get bit mask */
	movel	a6@(ARG3),d2		/* get msg queue depth */

	moveq	#20,d7			/* CREATE_X */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	a6@(ARG4),a1		/* get xid pointer */
	movel	a0,a1@			/* return xid */

        movel	a7@+,d2 		/* restore reg d2 */

	movel	#20,d0			/* CREATE_X */
	bra	return

/*******************************************************************************
*
* psosAttachX - attach to and use an active message exchange
*
* This routine serves as a C interface to pSOS system call ATTACH_X.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with one of the following values:
*	1 = named exchange cannot be found
*	3 = access violation, restricted to creators group or group 0
*
* SEE ALSO: pSOS User's Guide

* int psosAttachX (name, xid, procQ, msgQ)
*     int name;		/* 4 byte name code                                  *
*     int *xid;	 	/* where to return x  id                             *
*     int *procQ;	/* where to return # of messages waiting at exchange *
*     int *msgQ;	/* where to return # of messages presently posted    *

*/

_psosAttachX:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	d2,a7@-			/* save reg d2 */
	movel	a6@(ARG1),d0		/* get name */

	moveq	#21,d7			/* ATTACH_X */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	a6@(ARG2),a1		/* get xid pointer */
	movel	a0,a1@			/* return xid */
	movel	a6@(ARG3),a1		/* get procQ pointer */
	movel	d1,a1@			/* return procQ */
	movel	a6@(ARG4),a1		/* get msgQ pointer */
	movel	d2,a1@			/* return msgQ */

        movel	a7@+,d2			/* restore reg d2 */

	movel	#21,d0	 		/* ATTACH_X */
	bra	return

/*******************************************************************************
*
* psosDeleteX - delete an active message exchange
*
* This routine serves as a C interface to pSOS system call DELETE_X.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with one of the following values:
*	1 = exchange already deleted
*	2 = xid incorrect, failed validity check
*	3 = access violation, restricted to creators group or group 0
*	0x80 = deleted, waiting processes were unblocked
*	0x81 = deleted, waiting message buffers were returned
*
* SEE ALSO: pSOS User's Guide

* int psosDeleteX (xid)
*     int xid;	 	/* where to return xid *

*/

_psosDeleteX:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get xid */

	moveq	#22,d7			/* DELETE_X */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#22,d0			/* DELETE_X */
	bra	return

/*******************************************************************************
*
* psosSendX - send a message to an exchange
*
* This routine serves as a C interface to pSOS system call SEND_X.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with one of the following values:
*	1 = exchange already deleted
*	2 = xid incorrect, failed validity check
*	3 = access violation, restricted to creators group or group 0
*	4 = home exchange error, must be 0 or same as xid input
*	5 = cannot post, out of system message buffers.
*	6 = cannot post, exchange message queue at max length
*
* SEE ALSO: pSOS User's Guide

* int psosSendX (xid,msg)
*     int xid;	 	/* xid of target exchange *
*     int *msg;	 	/* address of message     *

*/

_psosSendX:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get xid */
	movel	a6@(ARG2),a1		/* get message */

	moveq	#23,d7			/* SEND_X */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#23,d0			/* SEND_X */
	bra	return

/*******************************************************************************
*
* psosJamX - jam a message at the head of an exchange's message queue
*
* This routine serves as a C interface to pSOS system call JAM_X.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with one of the following values:
*	1 = exchange already deleted
*	2 = xid incorrect, failed validity check
*	3 = access violation, restricted to creators group or group 0
*	4 = home exchange error, must be 0 or same as xid input
*	5 = cannot post, out of system message buffers.
*	6 = cannot post, exchange message queue at max length
*
* SEE ALSO: pSOS User's Guide

* int psosJamX (xid,msg)
*     int xid;	 	/* xid of target exchange *
*     int *msg;	 	/* address of message     *

*/

_psosJamX:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get xid */
	movel	a6@(ARG2),a1		/* get message */

	moveq	#19,d7			/* JAM_X */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#19,d0			/* JAM_X */
	bra	return

/*******************************************************************************
*
* psosLiberX - broadcast N identical messages to an exchange
*
* This routine serves as a C interface to pSOS system call LIBER_X.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with one of the following values:
*	1 = exchange already deleted
*	2 = xid incorrect, failed validity check
*	3 = access violation, restricted to creators group or group 0
*	5 = extra messages would run out of system message buffers
*	6 = extra messages would exceed max message queue
*	7 = cannot liberate with special messages
*
* SEE ALSO: pSOS User's Guide

* int psosLiberX (xid, msg, minCnt, nMsg, nLib)
*     int xid;	 	/* xid of target exchange                     *
*     int *msg;	 	/* address of message                         *
*     int minCnt; 	/* minimum # of msgs to post                  *
*     int *nMsg; 	/* where to put number of extra msgs posted   *
*     int *nLib; 	/* where to put number of processes liberated *

*/

_psosLiberX:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get xid */
	movel	a6@(ARG2),a1		/* get message */
	movel	a6@(ARG3),d0		/* get message */
	
	moveq	#18,d7			/* LIBER_X */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	a6@(ARG4),a1		/* get pointer to nMsg */
	movel	d0,a1@			/* return nMsg */
	movel	a6@(ARG5),a1		/* get pointer to nLib */
	movel	d1,a1@			/* return nLib */

	movel	#18,d0	 		/* LIBER_X */
	bra	return

/*******************************************************************************
*
* psosReqX - request a message from an exchange
*
* This routine serves as a C interface to pSOS system call REQ_X.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with one of the following values:
*	1 = exchange already deleted
*	2 = xid incorrect, failed validity check
*	3 = access violation, restricted to creators group or group 0
*	0x80 = no message (only if no wait specified)
*	0x81 = timed out (only if + timeout specified)
*
* SEE ALSO: pSOS User's Guide

* int psosReqX (xid, msg, wait, timeout)
*     int xid;	 	/* xid of target exchange                           *
*     int *msg;	 	/* address of message                               *
*     int wait; 	/* 0 = wait if necessary, else unconditional return *
*     int timeout; 	/* 0 = wait forever, n = wait n clock ticks         *

*/

_psosReqX:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get xid */
	movel	a6@(ARG2),a1		/* get message */
	movel	a6@(ARG3),d1		/* get wait option */
	movel	a6@(ARG4),d0		/* get timeout option */

	moveq	#24,d7			/* REQ_X */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#24,d0			/* REQ_X */
	bra	return

/*******************************************************************************
*
* psosSignalV - signal an event to a process
*
* This routine serves as a C interface to pSOS system call SIGNAL_V.
*
* RETURNS: OK or ERROR
*	
* If ERROR, errnoSet will be called with one of the following values:
*	1 = process not found
*	2 = pid failed validitiy check
*	3 = access violation, must be same group or group 0
*
* SEE ALSO: pSOS User's Guide

* int psosSignalV (id, events)
*     int id;		/* id code          *
*     int events;	/* events to signal *

*/

_psosSignalV:
	link	a6,#0
	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),a0		/* get process id */
	movel	a6@(ARG2),d1		/* get signal */

	moveq	#25,d7			/* SIGNAL_V */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#25,d0			/* SIGNAL_V */
	bra	return

/*******************************************************************************
*
* psosWaitV - wait on a signal event
*
* This routine serves as a C interface to pSOS system call WAIT_V.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet will be called with one of the following values:
*	0x81 = timed out before condition met
*
* SEE ALSO: pSOS User's Guide

* int psosWaitV (event, condition, timeout, caughtV)
*     int event;	/* event bits 0 - 12 user defined     *
*     int condition;	/* 0 = AND, <> 0 = OR                 *
*     int timeout;	/* number of clock ticks till give up *
*     int *caughtV;	/* actual events caught               *

*/

_psosWaitV:
	link	a6,#0

	movel	d7,a7@-			/* save reg d7 */
	movel	d2,a7@-			/* save reg d2 */
	movel	a6@(ARG1),d1		/* get event */
	movel	a6@(ARG2),d2		/* get condition */
	movel	a6@(ARG3),d0		/* get timeout */

	moveq	#26,d7			/* WAIT_V */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	a6@(ARG4),a1		/* pointer to caughtV */
	movel	d1,a1@			/* return caughtV */

        movel	a7@+,d2			/* restore reg d2 */

	movel	#26,d0			/* WAIT_V */
	bra	return

/*******************************************************************************
*
* psosGetV - like psosWaitV only it never waits
*
* This routine serves as a C interface to pSOS system call GET_V.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet will be called with the following value:
*	0x80 = event condition never met
*
* SEE ALSO: pSOS User's Guide

* int psosGetV (event, condition, caughtV)
*     int event;	/* event to wait for *
*     int condition;	/* condition         *
*     int *caughtV;	/* signals caught    *

*/

_psosGetV:
	link	a6,#0
	movel	d7,a7@-			/* save reg d7 */
	movel	a6@(ARG1),d1		/* get events */
	movel	a6@(ARG2),d2		/* get condition */

	moveq	#27,d7			/* GET_V */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	a6@(ARG3),a0		/* get pointer to caughtV */
	movel	d1,a0@			/* return caughtV*/

	movel	#27,d0			/* GET_V */
	bra	return

/*******************************************************************************
*
* psosPauseP - pause the calling process for a specified duration
*
* This routine serves as a C interface to pSOS system call PAUSE_P.
*
* RETURNS: OK
*	 
*
* SEE ALSO: pSOS User's Guide

* int psosPauseP (ticks)
*     int ticks;	/* ticks to wait *

*/

_psosPauseP:
	link	a6,#0
	movel	d7,a7@-			/* save reg d7 */

	movel	a6@(ARG1),d0		/* get number of ticks */

	moveq	#28,d7			/* PAUSE_P */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#28,d0	 		/* PAUSE_P */
	bra	return

/*******************************************************************************
*
* psosAnnounceT - announce system time tick to time manager
*
* This routine serves as a C interface to pSOS system call ANNOUNCE_T.
*
* RETURNS: OK
*	 
* SEE ALSO: pSOS User's Guide

* int psosAnnounceT ()

*/

_psosAnnounceT:
	link	a6,#0
	movel	d7,a7@-			/* save reg d7 */

	moveq	#29,d7			/* ANNOUNCE_T */
	jsr	_sysKernelTrap		/* trap to kernel */

	movel	#29,d0			/* ANNOUNCE_T */
	bra	return

/*******************************************************************************
*
* psosGetT - obtain systems conception of current date and time
*
* This routine serves as a C interface to pSOS system call GET_T.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet will be called with the following value:
*	1 = date and time have not been set since last restart
*
* SEE ALSO: pSOS User's Guide

* int psosGetT (date, time, ticks)
*     int *date;	/* where to put date  *
*     int *time;	/* where to put time  *
*     short *ticks;	/* where to put ticks *

*/

_psosGetT:
	link	a6,#0
	movel	d7,a7@-			/* save reg d7 */
	movel	d2,a7@-			/* save reg d2 */

	moveq	#30,d7			/* GET_T */
	jsr	_sysKernelTrap	        /* trap to kernel */

	movel	a6@(ARG1),a0		/* get pointer to date */
	movel	d0,a0@			/* return date */
	movel	a6@(ARG2),a0		/* get pointer to time */
	movel	d1,a0@			/* return time */
	movel	a6@(ARG3),a0		/* get pointer to ticks */
	movew	d2,a0@			/* return ticks */

        movel	a7@+,d2			/* restore reg d2 */
	movel	#30,d0			/* GET_T */
	bra	return

/*******************************************************************************
*
* psosSetT - set systems conception of current date and time
*
* This routine serves as a C interface to pSOS system call SET_T.
*
* RETURNS: OK or ERROR
*	 
* If ERROR, errnoSet will be called with the following value:
*	1 = date parameter error
*       2 = time parameter error
*
* SEE ALSO: pSOS User's Guide

* int psosSetT (date, time, ticks)
*     int date;		/* date  *
*     int time;		/* time  *
*     int ticks;	/* ticks *

*/

_psosSetT:
	link	a6,#0
	movel	d7,a7@-			/* save reg d7 */
	movel	d2,a7@-	 		/* save reg d2 */
	movel	a6@(ARG1),d0		/* get date */
	movel	a6@(ARG2),d1		/* get time */
	movel	a6@(ARG3),d2		/* get ticks */

	moveq	#31,d7			/* SET_T */
	jsr	_sysKernelTrap	        /* trap to kernel */

        movel	a7@+,d2			/* restore reg d2 */
	movel	#31,d0			/* SET_T */
	bra	return

/*******************************************************************************
*
* psosTswitchHook - task switch hook
*
* This routine should be specified in the PSOS configuration table,
* in usrConfig (1), as the task creation extension routine.
* It loops through a NULL terminated table of C task switch routines
* that have been specified by calls to taskSwitchHookAdd (2).
* Each routine in the table is called with pointers to
* the old and  new task's TCBs extension on each task
* switch. THIS IS NOT A C CALLABLE ROUTINE.
*
* SEE ALSO: taskSwitchHookAdd (2)

* psosTswitchHook ()

*/

_psosTswitchHook:
	moveml	d0-d2/a0-a2,a7@-	/* save regs */
	
	movel	a5,_taskIdCurrent	/* update the current task */

	movel	a5@(0x82),a7@-		/* push pointer to new tcbX */
	movel	a4@(0x82),a7@-		/* push pointer to old tcbX*/

	movel	#_taskSwitchTable,a2	/* get adrs of task switch rtn list */

tsloop:
	movel	a2@,d0			/* get next task switch rtn */
	beq	tsend			/* quit if end of list */

	movel	d0,a1
	jsr	a1@			/* call routine */

	addl	#4,a2			/* bump to next task switch routine */
	bra	tsloop			/* loop */

tsend:
	addl	#8,a7			/* clean up stack */

	moveml	a7@+,d0-d2/a0-a2	/* restore regs */
	rts

/*******************************************************************************
*
* _vxTaskEntry - task startup code following spawn
*
* This hunk of code is the initial entry point to every task created via
* the "spawn" routines.  Spawn has put the true entry point of the
* task into the tcb extension before creating the task,
* and then pushed exactly ten arguments (although the task may use
* fewer) onto the stack.  This code picks up the real entry point and calls it.
* Upon return, the 10 task args are popped, and the result of the main
* routine is passed to "exit" which terminates the task.
* This way of doing things has several purposes.  First a task is easily
* "restartable" (see "restart" routine in vxLib.c) since the real
* entry point is available in the tcb extension.  Second, the call to the main
* routine is a normal call including the usual stack clean-up afterwards,
* which means that debugging stack trace facilities will handle the call of
* the main routine properly.  
*
* NOMANUAL
*/

_vxTaskEntry:
	movel	#0,a6		/* make sure frame pointer is 0 */

	subql	#4,a7		/* make space for return value */
	movel	a7,d0
	movel	d0,a7@-		/* push address where to return value */
	jsr	_psosSuperP	/* supervise us */
	addql	#8,a7		/* pop return value and arg off stack */

	movel	_taskIdCurrent,a0	/* get current task id */
	movel	a0@(0x82),a0	/* get pointer to tcb extension */
	movel	a0@,a0		/* entry adr is 1st member of tcb extension */
	jsr	a0@		/* call main routine */
	addl	#40,a7		/* pop args to main routine */
	movel	d0,a7@-		/* pass result to exit */
	jsr	_exit		/* gone for good */

/*******************************************************************************
*
* return - universal exit
*
* All C callable routines in this library exit 
* via here. If the pSOS return code in d7 is not 0, the pSOS module number
* is or'd with d0 and errnoSet is called, and ERROR is returned in d0.
*
* NOMANUAL
*/

return:
	tstb	d7			/* test if OK */
	beq	retOK			/* exit OK */

	andl	#0xff,d7		/* mask off high order bits */
	andl	#0xff,d0		/* mask off high order bits */
	roll	#8,d0			/* shift call number up 8 bits */
	orl	d0,d7			/* or in the call number */
	orl	#0x20000,d7		/* or in M_psos */
	movel	d7,a7@-			/* push pSOS return code */
	jsr	_errnoSet		/* set process's status word */
	addql	#4,a7

	moveq	#-1,d0			/* return (ERROR) */
	bra	byebye

retOK:	clrl	d0			/* return (OK) */

byebye:
	movel	a7@+,d7			/* restore reg d7 */
	unlk	a6
	rts
/*******************************************************************************
*
* intEnt - enter interrupt level
*
* IntEnt for pSOS is a null routine and is here only for symmetry.
*
* IntEnt should NEVER be called from C.
*
* SEE ALSO: intConnect(2)

* VOID intEnt ()

*/

_intEnt:
	addql	#1,_intCnt	/* bump the counter */
	rts

/*******************************************************************************
*
* intExit - leave interrupt level
*
* This routine should be jumped to in place of the pSOS RETI call. This
* routine can NEVER be called from C.  It can only be jump'ed to from 
* assembly code.

* intExit ()

*/

_intExit:
	subql	#1,_intCnt	/* debump the counter */
	trap	#RETI		/* to pSOS (never to return here) */
