/* kernelLib.c - VxWorks/pSOS kernel library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01e,17apr89,jcf  workspace was being zeroed one to many bytes.
01d,24aug88,gae  documentation.
01c,29jun88,jcf  extended kernelInit to spawn a new routine kernelRoot
		   which does any kernel specific work before calling
		   usrRoot ().
01b,30may88,dnw  changed to v4 names.
01a,19jan88,jcf  written.
*/

/*
DESCRIPTION
This library provides a generic interface to the pSOS kernel
management routines.  Kernel management includes:

    - initialize the kernel,
    - start the kernel,
    - get a generic version number of the kernel,
    - turn on/off round robin (time slice) scheduling.

SEE ALSO: "Architecture", taskLib (1), tickLib (1)
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "taskLib.h"
#include "memLib.h"
#include "iv68k.h"
#include "psos.h"

IMPORT PSOS_CONFIG_TABLE usrCftbl;	/* pSOS config table */
IMPORT VOID psosReti ();		/* pSOS interrupt return trap entry */
IMPORT VOID psosSvc ();			/* pSOS trap entry */
IMPORT VOID ioJumpTable ();
IMPORT int  intLockLevel;		/* VxWorks interrupt lock out level */

#define IDLE_TASK_NAME	0x49444c45	/* "IDLE" */

/* globals */

int kernelType = KERNEL_UNINIT;		/* initially uninitalized */

/* locals */

LOCAL FUNCPTR rootRoutine;
LOCAL char *poolStart;
LOCAL char *poolEnd;

/* forward declarations */

LOCAL VOID kernelRoot ();


/*******************************************************************************
*
* kernelInit - initialize kernel
*
* This routine initializes and starts the kernel.  It should only be called
* once.  The routine `rootRtn' is the user's startup code that subsequently
* installs system facilities (eg. initialize the I/O system),
* spawns tasks, etc..
*
* INTERNAL
* The routine kernelRoot is called before the user's root routine so that
* memory management can be initialized.
*
* ARGSUSED
*/

VOID kernelInit (trapNum, rootRtn, rootStackSize, memPoolStart, memPoolEnd,
		 intStackSize, lockOutLevel)
    int trapNum;                /* kernel trap number             */
    FUNCPTR rootRtn;            /* user startup routine           */
    int rootStackSize;          /* stack size for startup routine */
    char *memPoolStart;         /* beginning of memory            */
    char *memPoolEnd;           /* end of memory                  */
    int intStackSize;           /* interrupt stack size           */
    int lockOutLevel;           /* interrupt lock out level (1-7) */

    {
    /* initialize vxWorks interrupt lock out level */
    intLockLevel = (int) lockOutLevel;

    rootRoutine = rootRtn;
    poolStart   = memPoolStart;
    poolEnd     = memPoolEnd;

    /* set kernel config table pointer and kernel trap */

    intVecSet (TRAPNUM_TO_IVEC (trapNum), psosSvc);
    intVecSet (TRAPNUM_TO_IVEC (4), psosReti);	/* initialize RETI vector */

    /* clear workspace */
    bzero (usrCftbl.ramStart1, (int) ((ULONG) usrCftbl.ramEnd1 + 1 - 
				         (ULONG) usrCftbl.ramStart1));

    usrCftbl.ioJTable  = (int)ioJumpTable;
    usrCftbl.rootStart = (int)kernelRoot;

    psosStartup ();	     /* start psos which automatically spawns root */
    }
/*******************************************************************************
*
* kernelVersion - get kernel version
*
* RETURNS: pointer to string of format `kernel version xxx.xxx'
*/

char *kernelVersion ()

    {
    static char name [30];
    int version;

    psosVersion (&version);
    sprintf (name, "pSOS version %1x.%1x", (version & 0xf0) >> 4, 
					    version & 0xf);
    return (name);
    }
/*******************************************************************************
*
* kernelTimeSlice - set time slice to specified ticks
*
* This routine enables round-robin selection among tasks of same priority.
*
* RETURNS: OK or ERROR
*
* ARGSUSED
*/

STATUS kernelTimeSlice (ticks)
    int ticks;		/* number of ticks for each task */

    {
    int bitBucket;	/* temp */

    return (psosModeP (0x40, 0x40, &bitBucket));
    }
/*******************************************************************************
*
* kernelRoot - temp root
*
* This routine is the first task to run under VxWorks.  In initializes the
* memory management, calls a user specified root routine (usually usrRoot(1)),
* and then cleans up after itself.
*/  

LOCAL VOID kernelRoot ()

    {
    static TCBX rootTcbX;		/* task extension for ROOT task */
    static TCBX idleTcbX;		/* task extension for IDLE task */
    TCBX *pTcbX;
    PSOS_PCB *pPcb;
    int bb;

    kernelType = KERNEL_PSOS;	/* variable as to what kernel we're using */

    /* ROOT should execute in supervisor state */

    psosSuperP (&bb);

    /* initialize tcb extension for ROOT */

    pTcbX = &rootTcbX;

    pPcb = (PSOS_PCB *) psosIdentPRoot ();
    pPcb->pcb_extend = pTcbX;

    pPcb->pcb_name = (int) "ROOT";

    pTcbX->errorStatus = OK;
    pTcbX->entry       = kernelRoot;
    pTcbX->botOfStack  = pPcb->pcb_ssp;
    pTcbX->topOfStack  = pPcb->pcb_ssp;
    pTcbX->initialSP   = NULL;
    pTcbX->options     = VX_SUPERVISOR_MODE;
    pTcbX->taskVar     = NULL;
    pTcbX->fpContext   = NULL;
    pTcbX->taskId      = (int) pPcb;
    pTcbX->name        = (char *) pPcb->pcb_name;

    /* initialize tcb extension IDLE */

    pTcbX = &idleTcbX;

    psosIdentP (IDLE_TASK_NAME, &pPcb, &bb);
    pPcb->pcb_extend = pTcbX;

    pPcb->pcb_name = (int) "IDLE";

    pTcbX->errorStatus = OK;
    pTcbX->entry       = (FUNCPTR) pPcb->pcb_pc;
    pTcbX->botOfStack  = pPcb->pcb_ssp;
    pTcbX->topOfStack  = pPcb->pcb_ssp;
    pTcbX->initialSP   = NULL;
    pTcbX->options     = VX_SUPERVISOR_MODE;
    pTcbX->taskVar     = NULL;
    pTcbX->fpContext   = NULL;
    pTcbX->taskId      = (int) pPcb;
    pTcbX->name        = (char *) pPcb->pcb_name;

    /* initialize memory pool */

    memInit ((char *) poolStart, (unsigned) (poolEnd - poolStart));

    (* rootRoutine) ();		/* call root routine */

    /* delete the psos IDLE and ROOT tasks */

    psosDeleteP (taskNameToId ("IDLE"));

    taskDelete (0);		/* commit suicide */
    }
