/* if_ee.c - LANCE ethernet network interface for the Eagle-5 */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
#define UNIWORKS
#ifndef UNIWORKS
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;
#endif UNIWORKS

/*
modification history
--------------------
	modified from if_ln.c for the Eagle-5 board

	ln- prefix ==>  LANCE Information
	ee- prefix ==>  Driver Information
*/

/*
DESCRIPTION
The only user callable routine is:

    eeattach (unit, memAdrs, memSize)

*/

#ifdef UNIWORKS
#include "UniWorks.h"
#else UNIWORKS
#include "vxWorks.h"
#endif UNIWORKS
#include "iv68k.h"
#include "memLib.h"
#include "ioctl.h"
#include "etherLib.h"

#include "param.h"
#include "mbuf.h"
#include "protosw.h"
#include "socket.h"
#include "errno.h"

#include "if.h"
#include "route.h"

#include "in.h"
#include "in_systm.h"
#include "in_var.h"
#include "ip.h"
#include "if_ether.h"

#include "if_ln.h"

IMPORT VOID ipintr ();


#define MAX_EE		1	/* max number of LANCE chips on MVME-147 */
#define LN_RMD_RLEN	5	/* ring size as a power of 2 */
#define LN_TMD_TLEN	5	/* same for transmit ring */


#define	ItoKval(p,type,base)	((type)((ULONG)(p) + (ULONG)(base)))
#define	KtoIval(p,type,base)	((type)((ULONG)(p) - (ULONG)(base)))
#define	ItoK(p,type,base)	((p) = ItoKval(p,type,base))
#define	KtoI(p,type,base)	((p) = KtoIval(p,type,base))

unsigned char eeEnetAddr [6];  /* ethernet address to load into lance */

/* globals */

int eeLogCount = 50;		/* log every eeLogCount number of errors */

/* locals */

LOCAL int lnTsize = LN_TMD_TLEN;	/* deflt xmit ring size as power of 2 */
LOCAL int lnRsize = LN_RMD_RLEN;	/* deflt recv ring size as power of 2 */

LOCAL struct ls_softc  *ls_softc [MAX_EE];

/* forward declarations */

LOCAL VOID		eeInt ();
LOCAL VOID		eeHandleCsr0Err ();
LOCAL VOID		eeHandleXmitInt ();
LOCAL VOID		eeHandleRecvInt ();


LOCAL VOID		eeInit ();
LOCAL int		eeOutput ();
LOCAL int		eeIoctl ();
LOCAL VOID		eeReset ();
LOCAL VOID		eeStartOutput ();

LOCAL struct mbuf      *eeGet ();
LOCAL ln_tmd	       *eeGetFreeTMD ();
LOCAL ln_rmd	       *eeGetFullRMD ();


/*******************************************************************************
*
* eeattach -
*
* Interface exists: make available by filling in network interface
* record.  System will initialize the interface when it is ready
* to accept packets.
* The memory pool will be malloced if NONE is specified.
*
*/

STATUS eeattach (unit, memAdrs, memSize)
    int	      unit;		/* unit number */
    char     *memAdrs;		/* address of memory pool (-1 == malloc it) */
    ULONG     memSize;		/* only used if memory pool is NOT malloced */

    {
    FAST struct ls_softc  *ls;
    FAST struct ifnet	  *ifp;
    FAST unsigned int	   sz;		/* temporary size holder */
    char *memPool;			/* start of the LANCE memory pool */

    if (unit) 	/* don't allow any other lances on this board */
	{
	return (ERROR);
	}

    if ((int) memAdrs != NONE)	/* specified memory pool */
	{
	/* With a specified memory pool we want to maximize
	 * lnRsize and lnTsize */

	sz = (memSize - (sizeof (ln_rmd) + sizeof (ln_tmd) + sizeof (ln_ib)))
	       / ((2 * LN_BUFSIZE) + sizeof (ln_rmd) + sizeof (ln_tmd));

	sz >>= 1;		/* adjust for roundoff */

	for (lnRsize = 0; sz != 0; lnRsize++, sz >>= 1)
	    ;

	lnTsize = lnRsize;	/* lnTsize = lnRsize for convenience */
	}

    /* limit ring sizes to reasonable values */

    if (lnRsize < 2)
        lnRsize = 2;		/* 4 buffers is reasonable min */

    if (lnRsize > 7)
        lnRsize = 7;		/* 128 buffers is max for chip */

    /* limit ring sizes to reasonable values */

    if (lnTsize < 2)
        lnTsize = 2;		/* 4 buffers is reasonable min */

    if (lnTsize > 7)
        lnTsize = 7;		/* 128 buffers is max for chip */


    /* calculate the total size of LANCE memory pool.  (basic softc structure
     * is just allocated from free memory pool since LANCE never references).
     */
    sz = ((sizeof (ln_ib))        + (((1 << lnRsize) + 1) * sizeof (ln_rmd)) +
	  (LN_BUFSIZE << lnRsize) + (((1 << lnTsize) + 1) * sizeof (ln_tmd)) +
	  (LN_BUFSIZE << lnTsize));

    if ((int) memAdrs == NONE)	/* if -1 then allocate memPool from free pool */
	{
	memPool = (char *) malloc (sz);	/* allocate memory */

	if ((int)memPool == NULL)
	    return (ERROR);		/* bail out if we don't get memory */
	}
    else
	{
	if (memSize < sz)
	    return (ERROR);	/* bail out if memSize specified is too small */

	memPool = memAdrs;	/* set the beginning of pool */
	}

    /*                        memPool partitioning
     *                        --------------------
     *
     *                                                 LOW MEMORY
     *
     *   		memPool,ls->ib	|-------------------------------------|
     *					|				      |
     *					|         (sizeof (ln_ib))	      |
     *					|				      |
     *   		  ls->ls_rring  |-------------------------------------|
     *					|				      |
     * 					| ((1 << lnRsize) + 1)*sizeof (ln_rmd)|
     *					|				      |
     *   	   ls->rmd_ring.r_bufs	|-------------------------------------|
     *					|				      |
     * 					|       (LN_BUFSIZE << lnRsize)       |
     *					|				      |
     *  		  ls->ls_tring	|-------------------------------------|
     *					|				      |
     *					| ((1 << lnTsize) + 1)*sizeof (ln_tmd)|
     *					|				      |
     *   	   ls->tmd_ring.t_bufs	|-------------------------------------|
     *					|				      |
     * 					|       (LN_BUFSIZE << lnTsize)       |
     *					|				      |
     *  			        |-------------------------------------|
     *
     *                                                HIGH MEMORY
     */

    /* allocate basic softc structure */

    ls = (struct ls_softc *)malloc (sizeof (struct ls_softc));
    if (ls == NULL)
	{
	if ((int) memAdrs == -1)	  /* if we malloced memPool */
	    free ((ULONG) memPool);		/* free the memPool */
	return (ERROR);
	}

    /* zero out entire softc structure */
    bzero ((char *)ls, sizeof (struct ls_softc));

    /* fill in high byte of memory base (A24:A31) and data port width */

    ls->memBase  = (char *)((ULONG)memPool & 0xff000000);
    ls->memWidth = -1;			/* don't care */


    /* allocate initialization block */

    ls->ib = (ln_ib *)memPool;
    sz = sizeof (ln_ib);		/* size of initialization block */
    bzero ((char *)ls->ib, sz);		/* zero out entire ib */

    /* allocate receive message descriptor (RMD) ring structure */

    ls->ls_rpo2 = lnRsize;		/* remember for eeConfig */
    ls->ls_rsize = (1 << lnRsize);	/* receive msg descriptor ring size */

    /* leave room to adjust low three bits */

    ls->ls_rring = (ln_rmd *) ((int)ls->ib + sz);   /* of pointer to be 000 */
    sz = ((1 << lnRsize) + 1) * sizeof (ln_rmd);
    bzero ((char *)ls->ls_rring, (int)sz);	/* zero out entire RMD ring */

    /* allocate receive buffer space */

    ls->rmd_ring.r_bufs = (char *)((int)ls->ls_rring + sz);
    sz = (LN_BUFSIZE << lnRsize);	 /* room for all the receive buffers */
    bzero (ls->rmd_ring.r_bufs, (int)sz);/* zero out entire rbuf area */

    /* allocate transmit message descriptor (TMD) ring structure */

    ls->ls_tpo2 = lnTsize;		/* remember for eeConfig */
    ls->ls_tsize = (1 << lnTsize);	/* transmit msg descriptor ring size */

    ls->ls_tring = (ln_tmd *) ((int)ls->rmd_ring.r_bufs + sz);
    sz = ((1 << lnTsize) + 1) * sizeof (ln_tmd);
    bzero ((char *)ls->ls_tring, (int)sz);	/* zero out entire TMD ring */

    /* allocate transmit buffer space */

    ls->tmd_ring.t_bufs = (char *)((int)ls->ls_tring + sz);
    sz = (LN_BUFSIZE << lnTsize);	/* room for all the transmit buffers */
    bzero (ls->tmd_ring.t_bufs, (int)sz);	/* zero out entire tbuf area */

    ls_softc [unit] = ls;		/* remember address for this unit */


    /* initialize device structure */

    ls->ivec    	= freevec();		/* interrupt vector */
    ls->ilevel		= 5;			/* IPIN interrupt level 5 */
    ls->devAdrs		= (LN_DEVICE *)0xfe401300;	/* LANCE i/o address */

    /* copy the enet address into the softc */

    eeGetAddr(ls);

    ifp = &ls->ls_if;

    ifp->if_unit = unit;
    ifp->if_name = "ee";
    ifp->if_mtu  = 1000;

    /* attach and enable the LANCE interrupt service routine to vector */

    lnChipReset (ls);				/* reset LANCE */

    sysLanIntConnect (ls->ivec, eeInt, ls);
    sysLanIntEnable ();				/* enable LANCE interrupts */

    ifp->if_init   = eeInit;
    ifp->if_output = eeOutput;
    ifp->if_ioctl  = eeIoctl;
    ifp->if_reset  = eeReset;
    ifp->if_flags  = IFF_BROADCAST;

    if_attach (ifp);

    return (OK);
    }
/*******************************************************************************
*
* eeReset - reset the interface
*
* Mark interface as inactive & reset the chip
*/

LOCAL VOID eeReset (unit)
    int unit;

    {
    FAST struct ls_softc  *ls = ls_softc [unit];

    ls->ls_if.if_flags &= ~IFF_RUNNING;
    lnChipReset (ls);				/* reset LANCE */
    }
/*******************************************************************************
*
* eeInit -
*
* Initialization of interface; clear recorded pending operations.
* Called at boot time (with interrupts disabled?),
* and at ifconfig time via eeIoctl, with interrupts disabled.
*/

LOCAL VOID eeInit (unit)
    int unit;

    {
    FAST struct ls_softc  *ls = ls_softc [unit];
    FAST struct ifnet	  *ifp = &ls->ls_if;

    ifp->if_flags &= ~(IFF_UP | IFF_RUNNING | IFF_PROMISC);

    lnChipReset (ls);				/* disable chip during init */
    eeConfig (ls);				/* reset all ring structures */

    ifp->if_flags |= (IFF_UP | IFF_RUNNING);
    if (ls->promiscuous)
	ifp->if_flags |= (IFF_PROMISC);

    lnChipInit (ls);			/* on return LANCE is running */

    eeStartOutput (ls);			/* tell chip about any pending output */
    }
/*******************************************************************************
*
* eeConfig - fill in initialization block with mode information.
*
* Fill in all fields in the Initialization Block with relevant values.
* In addition, fill in all fields in Transmit Message Descriptor ring
* and Receive Message Descriptor ring.
*/

LOCAL VOID eeConfig (ls)
    FAST struct ls_softc  *ls;

    {
    FAST ln_rmd	  *rmd;
    FAST ln_tmd	  *tmd;
    FAST char	  *buf;
    FAST ln_ib	  *ib;
    int		   i;

    rmd = ls->ls_rring;			/* receive message descriptor ring */
    rmd = (ln_rmd *)(((int)rmd + 7) & ~7);	/* low 3 bits must be 000 */
    ls->ls_rring = rmd;				/* fix softc as well */
    buf = ls->rmd_ring.r_bufs;
    for (i = 0; i < ls->ls_rsize; i++)
        {
	rmd->rbuf_ladr = (u_short)buf;	/* bits 15:00 of buffer address */
	rmd->rbuf_rmd1 = (u_short)((int)buf >> 16) & 0xff;
					/* bits 23:16 of buffer address */

	rmd->rbuf_bcnt = -LN_BUFSIZE;	/* neg of buffer byte count */
	rmd->rbuf_mcnt = 0;		/* no message byte count yet */
	rmd->rbuf_own = 1;		/* buffer now owned by LANCE */
	rmd++;				/* step to next message descriptor */
	buf += LN_BUFSIZE;		/* step to start of next buffer */
	}
    ls->ls_rindex = 0;			/* LANCE will use at start of ring */

    tmd = ls->ls_tring;			/* transmit message descriptor ring */
    tmd = (ln_tmd *)(((int)tmd + 7) & ~7);	/* low 3 bits must be 000 */
    ls->ls_tring = tmd;			/* fix softc as well */
    buf = ls->tmd_ring.t_bufs;
    for (i = 0; i < ls->ls_tsize; i++)
        {
	tmd->tbuf_ladr = (u_short)buf;	/* bits 15:00 of buffer address */
	tmd->tbuf_tmd1 = (u_short)((int)buf >> 16) & 0xff;
					/* bits 23:16 of buffer address */

	tmd->tbuf_bcnt = 0;		/* no message byte count yet */
	tmd->tbuf_tmd3 = 0;		/* no error status yet */
	tmd++;				/* step to next message descriptor */
	buf += LN_BUFSIZE;		/* step to start of next buffer */
	}
    ls->ls_tindex = 0;			/* LANCE will use at start of ring */
    ls->ls_dindex = 0;			/* buffer disposal will lag tindex */

    ib = ls->ib;
    if (ls->promiscuous)
        ib->lnIBMode = 0x8000;	/* chip will be in promiscuous receive mode */
    else
        ib->lnIBMode = 0;	/* chip will be in normal receive mode */

    swab ((char *)ls->ls_enaddr, ib->lnIBPadr, 6);

    ib->lnIBRdraLow = (u_short)ls->ls_rring;
    ib->lnIBRdraHigh = (u_short)(((int)ls->ls_rring >> 16) & 0xff)
      			 | (ls->ls_rpo2 << 13);

    ib->lnIBTdraLow = (u_short)ls->ls_tring;
    ib->lnIBTdraHigh = (u_short)(((int)ls->ls_tring >> 16) & 0xff)
      			 | (ls->ls_tpo2 << 13);
    }
/*******************************************************************************
*
* eeInt - handle controller interrupt
*
* This routine is called at interrupt level in response to an interrupt from
* the controller.
*/

LOCAL VOID eeInt (ls)
    FAST struct ls_softc  *ls;

    {
    FAST LN_DEVICE  *dv = ls->devAdrs;
    FAST u_short     stat;
    FAST ln_rmd	    *rmd;
    FAST ln_tmd	    *tmd;

    dv->RAP = 0;			/* select CSR0 */
    stat = dv->lncsr_0;			/* get whole register all at once */
    dv->lncsr_0 = stat & lncsr_INTMASK;	/* reset all interrupt bits */


    /* have task level handle csr0 errors;
     * CERR (collision errors) are not included in this because they
     * are not really errors!
     */
    if (stat & (lncsr_BABL | lncsr_MISS | lncsr_MERR))
#ifdef UNIWORKS
	netToDoAdd (eeHandleCsr0Err, ls, stat);
#else UNIWORKS
	netJobAdd (eeHandleCsr0Err, ls, stat);
#endif UNIWORKS


    /* have task level handle any input packets */

    if (((rmd = eeGetFullRMD (ls)) != NULL) && (!ls->recvHandlingAtTaskLevel))
	{
	ls->recvHandlingAtTaskLevel = TRUE;
#ifdef UNIWORKS
	netToDoAdd (eeHandleRecvInt, ls);
#else UNIWORKS
	netJobAdd (eeHandleRecvInt, ls);
#endif UNIWORKS
	}


    /* clean up completed transmit commands */

    if (ls->xmitHandlingAtTaskLevel)
	return;

    while (ls->ls_dindex != ls->ls_tindex)
        {
	/* disposal has not caught up */
	tmd = ls->ls_tring + ls->ls_dindex;

	if (tmd->tbuf_own != 0)
	    return;			/* chip owns this one, all done */

	/* buffer not owned by LANCE */

	/* have task level handle any output errors */

	if (tmd->tbuf_err != 0)
	    {
	    ls->xmitHandlingAtTaskLevel = TRUE;
#ifdef UNIWORKS
	    netToDoAdd (eeHandleXmitInt, ls);
#else UNIWORKS
	    netJobAdd (eeHandleXmitInt, ls);
#endif UNIWORKS
	    return;
	    }

	ls->ls_if.if_opackets++;	/* output packet complete */

	tmd->tbuf_tmd1 &= 0xff;		/* clear high byte */
	tmd->tbuf_tmd3 = 0;		/* clear all error & stat stuff */

	/* now bump the tmd disposal index pointer around the ring */
	ls->ls_dindex = (ls->ls_dindex + 1) & (ls->ls_tsize - 1);
        }
    }
/*******************************************************************************
*
* eeHandleXmitInt - task level interrupt service for xmit interrupts
*/

LOCAL VOID eeHandleXmitInt (ls)
    FAST struct ls_softc *ls;

    {
    FAST ln_tmd	    *tmd;
    int oldLevel;

    while (ls->ls_dindex != ls->ls_tindex)
        {
	/* disposal has not caught up */

	tmd = ls->ls_tring + ls->ls_dindex;

	oldLevel = intLock ();

	if (tmd->tbuf_own != 0)
	    {
	    /* chip owns this one so we're done -
	     * hand xmit handling back to int level */

	    ls->xmitHandlingAtTaskLevel = FALSE;

	    intUnlock (oldLevel);
	    return;
	    }

	intUnlock (oldLevel);

	/* buffer not owned by LANCE */

	if (!tmd->tbuf_err)
	    ls->ls_if.if_opackets++;	/* output packet complete */
	else
	    {
	    ls->ls_if.if_oerrors++;	/* output error */

	    /* check for no carrier */

	    if (tmd->tbuf_lcar)
		logMsg ("ee%d: no carrier\n", ls->ls_if.if_unit);

	    /* every eeLogCount errors show other interesting bits of tmd3 */

	    if ((tmd->tbuf_tmd3 & 0xf400) &&
		(ls->ls_if.if_oerrors % eeLogCount) == 0)
		logMsg ("ee%d: xmit error, status(tmd3)=0x%x, err count=%d\n",
			ls->ls_if.if_unit, tmd->tbuf_tmd3 & 0xfc00,
			ls->ls_if.if_oerrors);

	    /* restart chip on fatal errors */

	    if (tmd->tbuf_buff || tmd->tbuf_uflo)
		{
		eeInit (ls->ls_if.if_unit);
		return;
		}
	    }

	tmd->tbuf_tmd1 &= 0xff;		/* clear high byte */
	tmd->tbuf_tmd3 = 0;		/* clear all error & stat stuff */

	/* now bump the tmd disposal index pointer around the ring */
	ls->ls_dindex = (ls->ls_dindex + 1) & (ls->ls_tsize - 1);
	}

    ls->xmitHandlingAtTaskLevel = FALSE;
    }
/*******************************************************************************
*
* eeHandleCsr0Err - task level interrupt service for csr0 errors
*/

LOCAL VOID eeHandleCsr0Err (ls, status)
    FAST struct ls_softc *ls;
    int status;

    {
    ++ls->csr0Errs;

    /* every eeLogCount errors show interesting bits of csr0 */

    if ((ls->csr0Errs % eeLogCount) == 0)
	logMsg ("ee%d: csr0 error, csr0=0x%x, err count=%d\n",
		ls->ls_if.if_unit, status & 0xf800, ls->csr0Errs);

    /* restart chip on fatal error */

    if (status & lncsr_MERR)
	eeInit (ls->ls_if.if_unit);
    }
/*******************************************************************************
*
* eeHandleRecvInt - task level interrupt service for input packets
*
* This routine is called at task level indirectly by the interrupt
* service routine to do any message received processing.
*/

LOCAL VOID eeHandleRecvInt (ls)
    FAST struct ls_softc *ls;

    {
    FAST ln_rmd	*rmd;

    do
	{
	ls->recvHandlingAtTaskLevel = TRUE;

	while ((rmd = eeGetFullRMD (ls)) != NULL)
	    {				
	    /* RMD available */

	    eeRecv (ls, rmd);

	    ls->ls_rindex = (ls->ls_rindex + 1) & (ls->ls_rsize - 1);

	    rmd->rbuf_rmd1 &= 0xff;	/* clear high byte (errors) */
	    rmd->rbuf_mcnt = 0;

	    rmd->rbuf_own = 1;		/* give buffer back to LANCE */
	    }
	
	/* There is a RACE right here.  The ISR could add a receive packet
	 * and check the boolean below, and decide to exit.  Thus the
	 * packet could be dropped if we don't double check before we
	 * return.
	 */

	ls->recvHandlingAtTaskLevel = FALSE;
	}
    while (eeGetFullRMD (ls) != NULL);	/* this double check solves the RACE */
    }
/*******************************************************************************
*
* eeRecv -
*
* Process Ethernet receive completion:
*	If input error just drop packet.
*	Otherwise purge input buffered data path and examine 
*	packet to determine type.  If can't determine length
*	from type, then have to drop packet.  Otherwise decapsulate
*	packet based on type and pass to type-specific higher-level
*	input routine.
*/

LOCAL VOID eeRecv (ls, rmd)
    FAST struct ls_softc   *ls;
    FAST ln_rmd		   *rmd;

    {
    FAST struct ether_header  *eh;
    FAST struct mbuf	      *m;
    FAST struct ifqueue	      *inq;
    FAST int		       len, off, resid;
    FAST u_char		      *pData;
    FAST int		       s;

    ls->ls_if.if_ipackets++;

    /* check for read error */

    if (rmd->rbuf_err != 0)
	{
    	ls->ls_if.if_ierrors++;

	/* every eeLogCount errors show interesting bits of csr0 */

	if ((ls->ls_if.if_ierrors % eeLogCount) == 0)
	    logMsg ("ee%d: receive error, rmd1=0x%x\n", ls->ls_if.if_unit,
		    rmd->rbuf_rmd1 & 0xff00);
    	return;
	}

    len = rmd->rbuf_mcnt;
    eh = (struct ether_header *)(rmd->rbuf_ladr | (rmd->rbuf_hadr << 16));
    ItoK (eh, struct ether_header *, ls->memBase);

    /* call input hook if any */

    if ((etherInputHookRtn != NULL) &&
	(* etherInputHookRtn) (&ls->ls_if, (char *)eh, len))
	{
	return;	/* input hook has already processed this packet */
	}

    /* do software filter if controller is in promiscuous mode */

    if (ls->promiscuous)
	{
	if ((bcmp ((char *)eh->ether_dhost,	/* not our adrs? */
		   (char *)ls->ls_enaddr,
		   sizeof (eh->ether_dhost)) != 0) &&
	    (bcmp ((char *)eh->ether_dhost,	/* not broadcast? */
		   (char *)etherbroadcastaddr,
		   sizeof (eh->ether_dhost)) != 0))
	    return;				/* not for us */
	}
    /*
     * Deal with trailer protocol: if type is trailer
     * get true type from first 16-bit word past data.
     * Remember that type was trailer by setting off.
     */

    if (len >= sizeof (struct ether_header))
        len -= sizeof (struct ether_header);
    else
        len = 0;
    pData = ((u_char *)eh) + (sizeof (struct ether_header));

    eh->ether_type = ntohs ((u_short)eh->ether_type);
    if (eh->ether_type >= ETHERTYPE_TRAIL &&
        eh->ether_type < ETHERTYPE_TRAIL + ETHERTYPE_NTRAILER)
	{
    	off = (eh->ether_type - ETHERTYPE_TRAIL) * 512;

    	if (off >= ETHERMTU)
	    return;				/* sanity */

	eh->ether_type = ntohs (*(u_short *)(pData + off));
	resid          = ntohs (*(u_short *)(pData + off + 2));

    	if (off + resid > len)
	    return;				/* sanity */

    	len = off + resid;
	}
    else
    	off = 0;

    if (len == 0)
	return;					/* sanity */
    /*
     * Pull packet off interface.  Off is nonzero if packet
     * has trailing header; eeGet will then force this header
     * information to be at the front, but we still have to drop
     * the type and length which are at the front of any trailer data.
     */
    m = eeGet ((char *)pData, len, off, (struct ifnet *)&ls->ls_if);
    if (m == 0)
    	return;

    if (off)
	{
    	struct ifnet *ifp;

    	ifp = *(mtod (m, struct ifnet **));
    	m->m_off += 2 * sizeof (u_short);
    	m->m_len -= 2 * sizeof (u_short);
    	*(mtod (m, struct ifnet **)) = ifp;
	}

    switch (eh->ether_type)
	{
	case ETHERTYPE_IP:
	    inq = &ipintrq;
	    break;

	case ETHERTYPE_ARP:
	    arpinput (&ls->ls_ac, m);
	    return;

	default:
	    m_freem (m);
	    return;
	}

    s = splimp();

    if (IF_QFULL (inq))
	{
    	IF_DROP (inq);
    	m_freem (m);
    	splx (s);
    	return;
	}

    IF_ENQUEUE (inq, m);

    ipintr ();			/* process the ip packet */
    }

/*******************************************************************************
* 
* eeGet - Pull read data off an interface.
*
* len is length of data, with local net header stripped.
* off is non-zero if a trailer protocol was used, and
* gives the offset of the trailer information.
* We copy the trailer information and then all the normal
* data into mbufs.  
* Prepend a pointer to the interface structure,
* so that protocols can determine where incoming packets arrived.
*/

LOCAL struct mbuf *eeGet (buf0, totlen, off0, ifp)
    char	 *buf0;
    int		  totlen;
    int		  off0;
    struct ifnet *ifp;

    {
    struct mbuf	          *top;
    struct mbuf	         **mp;
    FAST struct mbuf      *m;
    FAST int		   off, len;
    FAST char	          *buf;
    FAST struct ls_softc  *ls = ls_softc [ifp->if_unit];

    buf = buf0;
    off = off0;

    top = 0;
    mp = &top;

    while (totlen > 0)
	{
	MGET (m, M_DONTWAIT, MT_DATA);
	if (m == 0)
	    {
	    m_freem (top);
	    return (0);
	    }

	if (off)
	    {
	    len = totlen - off;
	    buf += off;
	    }
	else
	    len = totlen;

	m->m_off = MMINOFF;
	if (ifp)
	    {
	    /* Leave room for ifp. */
	    m->m_len = MIN (MLEN - sizeof (ifp), len);
	    m->m_off += sizeof (ifp);
	    }
	else 
	    m->m_len = MIN (MLEN, len);

	bcopy (buf, mtod (m, caddr_t), (unsigned) m->m_len);

	buf += m->m_len;

	*mp = m;
	mp = &m->m_next;

	if (off)
	    {
	    /* sort of an ALGOL-W style for statement... */
	    off += m->m_len;
	    if (off == totlen)
		{
		/* reached end of data; setup to move trailer front */

		buf = buf0;
		off = 0;
		totlen = off0;
		}
	    }
	else
	    totlen -= m->m_len;

	if (ifp)
	    {
	    /* Prepend interface pointer to first mbuf. */
	    m->m_len += sizeof (ifp);
	    m->m_off -= sizeof (ifp);
	    *(mtod (m, struct ifnet **)) = ifp;
	    ifp = (struct ifnet *)0;
	    }
	}

    return (top);
    }
/*******************************************************************************
*
* eeOutput -
*
* Ethernet output routine.
* Encapsulate a packet of type family for the local net.
* Use trailer local net encapsulation if enough data in first
* packet leaves a multiple of 512 bytes of data in remainder.
*/

LOCAL int eeOutput (ifp, m0, dst)
    FAST struct ifnet	*ifp;
    FAST struct mbuf	*m0;
    struct sockaddr	*dst;

    {
    int			   type, s, error;
    u_char		   edst [6];
    struct in_addr	   idst;
    FAST struct ls_softc  *ls = ls_softc [ifp->if_unit];
    FAST struct mbuf	  *m = m0;
    FAST struct ether_header *eh;
    FAST int		   off;
    int			   usetrailers;

    if ((ifp->if_flags & (IFF_UP|IFF_RUNNING)) != (IFF_UP|IFF_RUNNING))
	{
    	error = ENETDOWN;
    	goto bad;
	}

    switch (dst->sa_family)
	{

#ifdef INET
	case AF_INET:
	    idst = ((struct sockaddr_in *)dst)->sin_addr;

	    if (!arpresolve (&ls->ls_ac, m, &idst, edst, &usetrailers))
    		return (0);			/* if not yet resolved */

	    off = ntohs ((u_short)mtod (m, struct ip *)->ip_len) - m->m_len;

	    if (usetrailers && off > 0 && (off & 0x1ff) == 0 &&
		m->m_off >= MMINOFF + 2 * sizeof (u_short))
		{
    		type = ETHERTYPE_TRAIL + (off>>9);
    		m->m_off -= 2 * sizeof (u_short);
    		m->m_len += 2 * sizeof (u_short);
    		*mtod (m, u_short *) = htons (ETHERTYPE_IP);
    		*(mtod (m, u_short *) + 1) = htons ((u_short)m->m_len);
    		goto gottrailertype;
		}

	    type = ETHERTYPE_IP;
	    off = 0;
	    goto gottype;
#endif
	case AF_UNSPEC:
	    eh = (struct ether_header *)dst->sa_data;
	    bcopy ((caddr_t)eh->ether_dhost, (caddr_t)edst, sizeof (edst));
	    type = eh->ether_type;
	    goto gottype;

	default:
	    logMsg ("ee%d: can't handle af%d\n", ifp->if_unit, dst->sa_family);
	    error = EAFNOSUPPORT;
	    goto bad;
	}

gottrailertype:
    /*
     * Packet to be sent as trailer: move first packet
     * (control information) to end of chain.
     */
    while (m->m_next)
    	m = m->m_next;
    m->m_next = m0;
    m = m0->m_next;
    m0->m_next = 0;
    m0 = m;

gottype:
    /*
     * Add local net header.  If no space in first mbuf,
     * allocate another.
     */
    if (m->m_off > MMAXOFF || MMINOFF + sizeof (struct ether_header) > m->m_off)
	{
    	m = m_get (M_DONTWAIT, MT_HEADER);
    	if (m == 0)
	    {
	    error = ENOBUFS;
	    goto bad;
	    }
    	m->m_next = m0;
    	m->m_off = MMINOFF;
    	m->m_len = sizeof (struct ether_header);
	}
    else
	{
    	m->m_off -= sizeof (struct ether_header);
    	m->m_len += sizeof (struct ether_header);
	}
    eh = mtod (m, struct ether_header *);
    eh->ether_type = htons ((u_short)type);
    bcopy ((caddr_t)edst, (caddr_t)eh->ether_dhost, sizeof (edst));
    bcopy ((caddr_t)ls->ls_enaddr, (caddr_t)eh->ether_shost, 6);


    /* output, enqueue, or drop packet */

    s = splimp ();
    if (IF_QFULL (&ifp->if_snd))
	{
	IF_DROP (&ifp->if_snd);
	splx (s);
	m_freem (m);
	return (ENOBUFS);
	}

    IF_ENQUEUE (&ifp->if_snd, m);

    eeStartOutput (ls);

    splx (s);
    return (0);

bad:
    m_freem (m0);
    return (error);
    }
/*******************************************************************************
*
* eeStartOutput - start pending output
*
* Start output to LANCE.
* Queue up all pending datagrams for which transmit buffers are available.
* Kick start the transmitter to avoid the polling interval latency.
* This routine is called by eeInit () and eeOutput ().
* It is very important that this routine be executed with splimp set.
* If this is not done, another task could allocate the same tmd!
*/

LOCAL VOID eeStartOutput (ls)
    FAST struct ls_softc  *ls;

    {
    FAST struct mbuf	  *m;
    struct mbuf		  *mh;
    FAST ln_tmd		  *tmd;
    FAST char		  *buf;
    char		  *buf0;
    struct ether_header	  *eh;
    int			   len;
    int			   s = splimp ();
    int			   oldLevel;

    /* Loop placing message buffers in output ring until no more or no room */

    while (ls->ls_if.if_snd.ifq_head != NULL)
        {
	/* there is something to send */

	if ((tmd = eeGetFreeTMD (ls)) == NULL)	/* get a transmit buffer */
	    break;			/* no transmit buffers available */

        buf = (char *)(tmd->tbuf_ladr | (tmd->tbuf_hadr << 16));
	ItoK (buf, char *, ls->memBase);
	buf0 = buf;			/* remember where it started */

	/* get message to send */

	IF_DEQUEUE (&ls->ls_if.if_snd, m); /* get head of next mbuf chain */
	mh = m;					/* remember head of chain */

	eh = mtod (m, struct ether_header *);
	bcopy ((char *)ls->ls_enaddr, (char *)eh->ether_shost, 6);

	/* copy packet to write buffer;
	 * note that freeing mbuf is delayed until after we give LANCE 
	 * the buffer so we can overlap operation */

	for (; m != NULL; m = m->m_next)
	    {
	    bcopy (mtod (m, caddr_t), buf, (unsigned)m->m_len);
	    buf += m->m_len;
	    }

	len = max (ETHERMIN + sizeof (struct ether_header), buf - buf0);

	/* call output hook if any */

	if ((etherOutputHookRtn != NULL) &&
	     (* etherOutputHookRtn) (&ls->ls_if, buf0, len))
	    {	
	    /* output hook has already processed this packet */
	    }
	else
	    {
	    /* place a transmit request */

	    oldLevel = intLock ();	/* disable ints during update */

	    tmd->tbuf_tmd3 = 0;		/* clear buffer error status */
	    tmd->tbuf_bcnt = -len;	/* negative message byte count */

	    tmd->tbuf_enp = 1;		/* this buffer is end of packet */
	    tmd->tbuf_stp = 1;		/* this buffer is start of packet */
	    tmd->tbuf_def = 0;		/* clear status bit */
	    tmd->tbuf_one = 0;		/* ditto */
	    tmd->tbuf_more = 0;
	    tmd->tbuf_err = 0;

	    ls->ls_tindex = (ls->ls_tindex + 1) & (ls->ls_tsize - 1);
	    tmd->tbuf_own = 1;		/* give buffer to LANCE */

	    intUnlock (oldLevel);	/* now eeInt won't get confused */

	    lnTransmitDemand (ls);	/* kick start the transmitter */
	    }

	m_freem (mh);		/* now can recycle these */
        }

    splx (s);
    }
/*******************************************************************************
*
* eeIoctl -
*
* Process an ioctl request.
*/

LOCAL int eeIoctl (ifp, cmd, data)
    FAST struct ifnet	*ifp;
    int			 cmd;
    caddr_t		 data;

    {
    int			   unit = ifp->if_unit;
    FAST struct ls_softc  *ls = ls_softc [unit];
    FAST struct ifaddr	  *ifa = (struct ifaddr *)data;
    int			   s = splimp ();
    int			   error = 0;

    switch (cmd)
	{
	case SIOCSIFADDR:
	    ifp->if_flags |= IFF_UP;
	    eeInit (unit);

	    switch (ifa->ifa_addr.sa_family)
		{
#ifdef	INET
		case AF_INET:
		    ((struct arpcom *)ifp)->ac_ipaddr = IA_SIN (ifa)->sin_addr;
		    arpwhohas ((struct arpcom *)ifp, &IA_SIN (ifa)->sin_addr);
		    break;
#endif	INET
		}
	    break;

	case SIOCGIFADDR:
	    bcopy((caddr_t) ls->ls_enaddr,
		  (caddr_t) ((struct ifreq *)data)->ifr_addr.sa_data, 6);
	    break;

	case SIOCGIFFLAGS:
	    *(short *)data = ifp->if_flags;
	    break;

	case SIOCSIFFLAGS:
	    {
	    short flags = *(short *)data;
	    int up      = flags & IFF_UP;
	    int promisc = flags & IFF_PROMISC;

	    if (promisc)
	        ls->promiscuous = TRUE;
	    else
	        ls->promiscuous = FALSE;

	    if ((up != (ifp->if_flags & IFF_UP))
	      | (promisc != (ifp->if_flags & IFF_PROMISC)))
		{
	        if (up)
		    eeInit (unit);		/* going up */
		else
		    lnChipReset (ls);		/* going down */
		}

	    if (ifp->if_flags & IFF_UP)
	        flags |= (IFF_UP|IFF_RUNNING);
	    else
	        flags &= ~(IFF_UP|IFF_RUNNING);

	    ifp->if_flags = flags;
	    }
	    break;

	default:
	    error = EINVAL;
	}

    splx (s);
    return (error);
    }
/*******************************************************************************
*
* lnChipReset - hardware reset of chip (stop it)
*/

LOCAL VOID lnChipReset (ls)
    struct ls_softc  *ls;

    {
    FAST LN_DEVICE	*dv = ls->devAdrs;

    dv->RAP = 0;				/* select CSR0 */
    dv->lncsr_0 = lncsr_STOP;			/* set the stop bit */
    }
/*******************************************************************************
*
* lnChipInit - hardware init of chip (init & start it)
*/

LOCAL VOID lnChipInit (ls)
    FAST struct ls_softc  *ls;

    {
    FAST LN_DEVICE	*dv = ls->devAdrs;

    dv->RAP = 0;		/* select CSR0 */
    dv->lncsr_0 = lncsr_STOP;	/* set the stop bit */
				/* can't write csr1..3 if not stopped */
    dv->RAP = 3;
    dv->lncsr_bswp = 1;		/* addr of word is addr of high byte */
    dv->lncsr_acon = 0;		/* address latch enable, rather than strobe */
    dv->lncsr_bcon = 0;		/* byte mask mode */

    dv->RAP = 2;
    dv->lncsr_iadr_h = (u_short)(((int)ls->ib >> 16) & 0xff);

    dv->RAP = 1;
    dv->lncsr_iadr_l = (u_short)ls->ib;

    dv->RAP = 0;
    dv->lncsr_0 = lncsr_INIT;	/* init chip (read IB) */

    while ((dv->lncsr_0 & (lncsr_IDON | lncsr_ERR)) == 0)
        ;			/* hang until Initialization DONe or ERRor */

    /* log chip initialization failure */

    if (dv->lncsr_0 & lncsr_ERR)
	logMsg ("ee%d: ERROR: Lance chip initialization failure, csr0=0x%x\n",
		ls->ls_if.if_unit, dv->lncsr_0);

    /* start chip */

    ls->ls_rindex = 0;		/* chip will start at beginning */
    ls->ls_tindex = 0;
    ls->ls_dindex = 0;
    ls->xmitHandlingAtTaskLevel = FALSE;

    dv->lncsr_0 = (lncsr_IDON | lncsr_INEA | lncsr_STRT);
					/* reset IDON state */
					/* enable interrupts from LANCE */
					/* start chip operation */
    }
/*******************************************************************************
*
* lnTransmitDemand - kick start the transmit hardware
*/

LOCAL VOID lnTransmitDemand (ls)
    struct ls_softc  *ls;

    {
    FAST LN_DEVICE	*dv = ls->devAdrs;

    dv->RAP = 0;				/* select CSR0 */
    dv->lncsr_0 = (lncsr_INEA | lncsr_TDMD);	/* kick start the transmitter */
    }
/*******************************************************************************
*
* eeGetFreeTMD - get next available TMD
*/

LOCAL ln_tmd *eeGetFreeTMD (ls)
    FAST struct ls_softc  *ls;

    {
    FAST ln_tmd	  *tmd;

    /* check if buffer is available (owned by host) */
    /* also check for tindex overrun onto dindex */

    tmd = ls->ls_tring + ls->ls_tindex;
    if ((tmd->tbuf_own != 0)
      | (((ls->ls_tindex + 1) & (ls->ls_tsize - 1)) == ls->ls_dindex))
        tmd = (ln_tmd *) NULL;

    return (tmd);
    }
/*******************************************************************************
*
* eeGetFullRMD - get next received message RMD
*/

LOCAL ln_rmd *eeGetFullRMD (ls)
    FAST struct ls_softc  *ls;

    {
    FAST ln_rmd	  *rmd;

    /* check if buffer is full (owned by host) */

    rmd = ls->ls_rring + ls->ls_rindex;

    if (rmd->rbuf_own == 0)
        return (rmd);
    else
        return ((ln_rmd *) NULL);
    }

/*******************************************************************************
*
* eeGetAddr - Read ethernet address from IPIN module NVRAM
*/
LOCAL eeGetAddr(ls)
    FAST struct ls_softc  *ls;
{
    register int i;
    u_short *addr = (u_short *)0xfe403080;

    for (i = 0; i < 6; ++i) {
	eeEnetAddr[i] = (*addr++ & 0xf) << 4;
	eeEnetAddr[i] |= (*addr++ & 0xf);
    }
    bcopy (eeEnetAddr, (char *)ls->ls_enaddr, sizeof (ls->ls_enaddr));
    return (0);
}
