/* if_bp.c - VxWorks & SunOS backplane driver */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
03f,28mar89,gae  turned off debugging; fixed bp BUFSIZ to be bigger
		   than MTU to allow for ether_header overhead -- bug fix.
03e,10feb89,gae  taught SunOS to do mailboxes and about bus interrupts;
	   +jcf  added testing in bpInit for sufficient memory;
           +dnw  added read mailboxes; fixed mbuf bug in copy_to_mbufs.
		 UNIX ver. now compiles under SunOS 4.0 and has SUN3_5 flag.
03c,10dec88,gae  SunOS version tweaks.
03b,20nov88,dnw  lint
03a,13sep88,gae  merged SunOS and VxWorks versions.  Documentation.
	   +dnw  debugged SunOS version.
	   +ak   wrote SunOS version.
02l,23jun88,rdc  added IFF_BROADCAST to if_flags.
02k,06jun88,dnw  changed taskSpawn/taskCreate args.
		 changed NCPU from 9 to 32.
02j,30may88,dnw  changed to v4 names.
02i,12apr88,gae  added include of ioLib.h for READ/WRITE/UPDATE.
02h,28jan88,jcf  made kernel independent.
02g,20feb88,dnw  lint
02f,05jan88,rdc  added include of systm.h 
02e,20nov87,ecs  lint
	    dnw  changed bpattach to take int vec num instead of vec adrs.
02d,13nov87,dnw  added heartbeat & anchor.
02c,03nov87,dnw  fixed bug in bpRead of throwing away packets that were
		   larger than 1500 bytes.
		 make bpPoll be global so that it is legible in i().
		 added intConnect() call for bus interrupt connects in
		   bpConnect().
		 changed bpReboot() into bpReset().
02b,26oct87,jcf	 added bpConnect that does a sysMailboxConnect in the case of
	     &    mailbox interrupts or a spawn of bpPoll for non-interrupt
	    dnw   driven backplanes
		 changed to not probe shared memory unless cpu 0.
02a,22apr87,dnw  rewritten.
01b,08apr87,rdc  moved call to bpConnect from bpAttach to bpInit.
01x,01apr87,jlf  delinted, removed ifdef CLEAN.
		 changed calls to tas () to vxTas ().
		 added copyright.
*/

/*
DESCRIPTION
This module implements the VxWorks backplane network driver.
The backplane driver allows several CPUs to communicate via shared memory.
Usually, the first CPU to boot in a card cage is considered the
backplane master.  This CPU has either dual-ported memory or has an
additional memory board which all other CPUs can access.
Each CPU must have some way to be interrupted by another CPU
and to be able interrupt all other CPUs.  There are three
interrupt types: polling, mailboxes, VME bus interrupts.
Polling is used when there are no hardware interrupts; each CPU spawns
a polling task to manage transfers.  Mailbox interrupts are
the preferred method because they do not require an interrupt level.
Using VME bus interrupts is much better than polling but
may require hardware jumpers.

There are three user callable routines: bpInit, bpattach, and bpShow.
The backplane master, usually processor 0, must initialize the backplane
memory and structures by first calling the routine:

.CS
    STATUS bpInit (pAnchor, pMem, memSize, tasOK)
	char *pAnchor;	/* backplane anchor address *
	char *pMem;	/* start of backplane shared memory, NONE=alloc *
	int memSize;	/* num bytes in bp shared memory, 0=0x100000 *
	BOOL tasOK;	/* TRUE = hardware can do test-and-set *
.CE
Typically, the `pAnchor' and `pMem' parameters both point to the same
block of shared memory.  If the first processor is dual-porting its
memory then, by convention, the anchor is at 0x600 (only 16 bytes are
required) and the start of memory `pMem' is dynamically allocated
by using the value NONE (-1).  Memory size ranges from a
minimum of 64K to a comfortable 512K, although this may be larger.
The `tasOK' parameter is provided for CPUs (viz. SUN-3's) who do not
support the test-and-set instruction.  If any test-and-set deficient
CPUs are in the system then all CPUs must use the software "test-and-set".

Once the backplane has been initialized, all processors may attach via:

.CS
    STATUS bpattach (unit, pAnchor, procNum, intType, intArg1, intArg2, intArg3)
	int unit;	/* backplane unit number *
	char *pAnchor;	/* bus pointer to bp anchor *
	int procNum;	/* processor number in backplane *
	int intType;	/* interrupt type: poll, bus, mailbox *
	int intArg1;	/* as per interrupt type *
	int intArg2;	/* as per interrupt type *
	int intArg3;	/* as per interrupt type *
.CE

Usually, bpInit and bpattach are called automatically in usrConfig (1)
when backplane parameters are specified in the boot line. 

As a diagnostic aid to show all the CPUs configured on a backplane use:

.CS
    VOID bpShow (bpName, zero)
	char *bpName;	/* backplane interface name (NULL == "bp0") *
	BOOL zero;	/* TRUE = zap totals *
.CE

MEMORY LAYOUT
All pointers in shared memory are kept relative to start of shared memory,
since dual-ported memory may appear in different places for different CPUs.

.CS
   ----------------------------	low address (anchor)
   |       ready value        |	    1234567
   |         heartbeat        |	    increments 1/sec
   |   pointer to bp header   |
   |         watchdog         |     for backplane master CPU
    -------------------------- 
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   the backplane header may be contiguous or
   allocated elsewhere on the master CPU
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~

   ---------------------------- backplane header
   |    backplane header      |	    1234567
   |       num CPUs           |     unused
   |    ethernet address      |     (6 bytes)
   |   pointer to free ring   |
   |--------------------------|

   |--------------------------| cpu descriptor
   |        active            |	    true/false
   |   processor number       |	    0-NCPU
   |         unit             |     0-NBP
   |   pointer to input ring  |
   |    interrupt type        |     POLL  | MAILBOX    |  BUS
   |    interrupt arg1        |     none  | addr space | level
   |    interrupt arg2        |     none  | address    | vector
   |    interrupt arg3        |     none  | value      | none
   |--------------------------|
              ...               repeated MAXCPU times
   |--------------------------|
   |       free ring          |	    contains pointers to buffer nodes
   |--------------------------|
   |       input ring 1       |	    contains pointers to buffer nodes
   |--------------------------|
	      ...
   |--------------------------|
   |       input ring n       |
   |--------------------------|

   |--------------------------| buffer node 1
   |     address, length      |
   |--------------------------|
   |       data buffer 1      |
   |--------------------------|
	      ...
   |--------------------------| buffer node m
   |     address, length      |
   |--------------------------|
   |       data buffer m      |
   ----------------------------	high address
.CE

INTERNAL
 SOFTWARE ARCHITECTURE
.CS
    bpIoctl   |    bpOutput   bpPoll   bpattach  bpReset  bpInit
      |    __/        |         |         / \               |
      |   /           |         |        /   \              |
      |  /            |         |       /     \             |
     bpIfinit    ether_output   |     bpAlive bpInit*    bpSizeMem
      / | \           |         |
 bpAlive|bpIntEnable bpPut      |
     bpConnect        |         | (interrupt level)
       |            bpXmit      /    /
       |               \       /    /
     bpPollTask         \     /    /
     bpPollUnix          \   /    /
                          \ /  __/
                         bpintr
                           |
                        bpReadAll
                           |
                         bpRead

 * UNIX only; called directly in VxWorks

.CE
*/

#ifdef	UNIX

#ifdef	SUN3_5
#include "h/types.h"
#include "h/systm.h"
#include "h/mbuf.h"
#include "h/buf.h"
#include "h/map.h"
#include "h/socket.h"
#include "h/param.h"
#include "h/vmmac.h"
#include "h/errno.h"
#else
#include "sys/types.h"
#include "sys/systm.h"
#include "sys/mbuf.h"
#include "sys/buf.h"
#include "sys/map.h"
#include "sys/socket.h"
#include "sys/param.h"
#include "sys/vmmac.h"
#include "sys/errno.h"
#endif	SUN3_5

#include "machine/param.h"
#include "machine/pte.h"

#ifdef	SUN3_5
#include "h/time.h"
#include "h/kernel.h"
#include "h/ioctl.h"
#else
#include "sys/time.h"
#include "sys/kernel.h"
#include "sys/ioctl.h"
#endif	SUN3_5
#include "net/if.h"
#include "netinet/in.h"
#include "netinet/if_ether.h"

#include "sundev/mbvar.h"
#ifdef	SUN3_5
#include "h/protosw.h"
#else
#include "sys/protosw.h"
#endif	SUN3_5
#include "net/route.h"
#include "netinet/in_systm.h"
#include "netinet/ip.h"
#include "netinet/ip_var.h"

#include "vxWorks.h"
#include "vme.h"	/* vxWorks vme AM's */

#define NBP     2	/* must be >= bp_n in #include "bp.h" */
#define BP_MASTER       (bp_procnum == 0)

IMPORT int bp_tas;
IMPORT int bp_procnum;
IMPORT int bp_n;
IMPORT int bp_mem_size;
IMPORT char *bp_mem_adrs;
IMPORT int bp_int_type;
IMPORT int bp_int_arg1;
IMPORT int bp_int_arg2;
IMPORT int bp_int_arg3;

IMPORT VOID wakeup ();

#else

#include "vxWorks.h"

#include "iv68k.h"
#include "taskLib.h"
#include "memLib.h"
#include "ioLib.h"
#include "ioctl.h"
#include "etherLib.h"
#include "wdLib.h"
#include "iosLib.h"

#include "mbuf.h"
#include "socket.h"
#include "protosw.h"
#include "errno.h"
#include "systm.h"
#include "param.h"
#include "route.h"
#include "in.h"
#include "in_systm.h"
#include "ip.h"
#include "ip_var.h"
#include "if.h"
#include "if_ether.h"
#include "in_var.h"

#endif	UNIX

#include "if_ring.h"
#include "if_bp.h"

IMPORT int ipintr ();
IMPORT struct ifnet *ifunit ();
IMPORT struct mbuf *copy_to_mbufs ();

 
#define	MAXCPU		32		/* don't change! */
#define	NCPU		32		/* number of input rings to init */
#define	MINPKTSIZE	60		/* minimum packet size to send */

#define BP_MTU		((2 * 1024) + 128)
int bp_mtu = BP_MTU;			/* patch before calling bpInit */ 
#define BP_BUFSIZ       (bp_mtu + sizeof (struct ether_header))

#ifdef UNIX
#define logMsg          printf
#define	WDOG_ID		int
#define vir2phys(v)	(unsigned)ptob((unsigned)hat_getkpfnum(v))
#define	splimp()	spl3()

#else
#define	hz	(sysClkRateGet ())
#define NBP		2		/* max. # bp interfaces */
#define vir2phys(v)	(v)
#endif	UNIX


typedef struct ring RING;

typedef struct			/* buffer node */
    {
    char	*b_addr;	/* buffer address (as offset) */
    short	b_len;		/* length */
    } BUF_NODE;

typedef struct			/* CPU descriptor in bp header */
    {
    BOOL	cd_active;	/* CPU is present */
    int		cd_cpuNum;	/* processor number */
    int		cd_unit;	/* unit on own CPU */
    RING	*cd_pInputRing;
    int		cd_intType;	/* interrupt: poll, bus, or mailbox */
    int		cd_intArg1;
    int		cd_intArg2;
    int		cd_intArg3;
    } CPU_DESC;

typedef struct			/* bp header */
    {
    int		hdr_ready;	/* used only as marker for debugging */
    short	hdr_ncpu;	/* unused */
    u_char	hdr_enetAdrs[6];/* "ethernet address" */
    RING	*hdr_pFreeRing;
    CPU_DESC	hdr_cpuDesc[MAXCPU];
    } BP_HEADER;

typedef struct			/* bp anchor */
    {
    int		an_ready;	/* special value 1234.... */
    UINT	an_heartbeat;	/* increments every second */
    BP_HEADER	*an_pHdr;	/* separate to minimize usage of low memory */
    WDOG_ID	an_wdog;	/* for VxWorks version only */
    } BP_ANCHOR;


#define BP_READY	  0x12345678	/* distinctive ready value */
#define BP_READY_SOFT_TAS 0x12344321	/* ready value with software tas */

#define	ItoKval(p,type,base)	((type)((int)(p) + (int)(base)))
#define	KtoIval(p,type,base)	((type)((int)(p) - (int)(base)))
#define	ItoK(p,type,base)	((p) = ItoKval(p,type,base))
#define	KtoI(p,type,base)	((p) = KtoIval(p,type,base))

/*
 * Net software status per interface.
 *
 * Each interface is referenced by a network interface structure,
 * bs_if, which the routing code uses to locate the interface.
 * This structure contains the output queue for the interface, its address, ...
 */

typedef struct
    {
    struct	arpcom bs_ac;	/* common ethernet structures */
#define bs_if           bs_ac.ac_if     /* network-visible interface */
#define bs_enaddr       bs_ac.ac_enaddr /* hardware ethernet address */
    int		bs_procNum;	/* processor number on this net */
    BP_ANCHOR	*bs_pAnchor;	/* pointer to bp anchor */
    BP_HEADER	*bs_pHdr;	/* pointer to bp header */
    RING	*bs_pFreeRing;	/* pointer to free ring */
    RING	*bs_pInputRing;	/* pointer to our input ring */
    BOOL	bs_readPending;	/* read interrupt is outstanding */
    int		bs_intType;	/* interrupt type: poll, bus, mailbox */
    int		bs_intArg1;
    int		bs_intArg2;
    int		bs_intArg3;
#ifdef	UNIX
    caddr_t bs_addr;
    u_char bs_ivec;
#endif	UNIX
    } BP_SOFTC;


#ifdef	UNIX
extern struct mb_driver bpdriver;
struct mb_device *bpinfo[NBP];

LOCAL BP_SOFTC bp_softc [NBP];
#define	BP_SOFTC_P	&bp_softc
#define	dbgprintf	printf
#define	sysProcNumGet()	(bp_procnum)
FUNCPTR bpProbeType;

LOCAL char *cpudesc_vaddr [MAXCPU];
#else

/* bpPollTask parameters */
int bpPollTaskId;
int bpPollTaskOptions	= VX_SUPERVISOR_MODE | VX_UNBREAKABLE;
int bpPollTaskStackSize	= 2000;
int bpPollTaskPriority	= 250;

LOCAL BP_SOFTC *bp_softc[NBP];
#define	BP_SOFTC_P	bp_softc
#define	dbgprintf	logMsg
int bpProbeType;
#endif	UNIX

#ifdef	UNIX
IMPORT int bpParity;
IMPORT int bpDebug;	/* flags: 1-usage, 2-dropped packets */
#else
int bpParity = FALSE;
int bpDebug  = 0x0;	/* flags: 1-usage, 2-dropped packets */
#endif	UNIX

#define	DEBUG_OPTION_1	(bpDebug & 0x1)
#define	DEBUG_OPTION_2	(bpDebug & 0x2)

#define	BPLOCAL	LOCAL	/* ultimately global routines in etherLib.c */

u_char bpEnetAdrs [6]	= { 60, 123, 0, 0, 0, 0 };

/* local variables */

LOCAL BOOL bpTasOK;		/* whether hardward can do test-and-set */
LOCAL int bpHeartbeatRate;
int bpBitBucket;		/* spare change */

/* forward declarations */

VOID bpPollTask ();
VOID bpintr ();
LOCAL VOID bpIfInit ();
LOCAL int bpIoctl ();
LOCAL VOID bpOutput ();
LOCAL VOID bpReset ();
LOCAL int bpReadAll ();
LOCAL int bpHeartbeat ();
#ifdef	UNIX
/*******************************************************************************
*
* bpattach - make interface available
*
* Make interface available by filling in network interface record.
* System will initialize the interface when it is ready to accept packets.
*/

VOID bpattach (md)
    struct mb_device *md;

    {
    int unit          = md->md_unit;
    FAST BP_SOFTC *bs = BP_SOFTC_P [unit];
    struct ifnet *ifp = &bs->bs_if;

    bs->bs_procNum = bp_procnum;

    bs->bs_intType = bp_int_type;
    bs->bs_intArg1 = bp_int_arg1;
    bs->bs_intArg2 = bp_int_arg2;
    bs->bs_intArg3 = bp_int_arg3;

    bs->bs_addr = md->md_addr;
    bs->bs_ivec = md->md_intr->v_vec;

    ether_attach (ifp, unit, "bp", bpIfInit, bpIoctl, bpOutput, bpReset);

    ifp->if_mtu = bp_mtu;

    bzero ((char*) cpudesc_vaddr, sizeof (cpudesc_vaddr));
    }
/*******************************************************************************
*
* bpPoke - map in physical memory and write value to address
*
* This routine is used to map in CPU boards' mailboxes.
* An array up to MAXCPU is kept so that a mailbox is
* only mapped in once per CPU.
*/

LOCAL VOID bpPoke (addr, type, value, cpuNum)
    char *addr;		/* physical address */
    int type;		/* mailbox size + read/write */
    int value;		/* value to write */
    int cpuNum;		/* which CPU */

    {
    char *vaddr;		/* virtual address */
    int kmx;			/* kernel map index */
    int numClicks;
    /* int numBytes; */
    int addrOffset;

    if (cpuNum < 0 && cpuNum > MAXCPU)
	{
	printf ("bp: bpPoke invalid cpu #%d\n", cpuNum);
	return;
	}

    if ((vaddr = cpudesc_vaddr [cpuNum]) == 0)
	{
	numClicks = btoc (sizeof (int));
	/* numBytes = ctob (numClicks); */
	addrOffset = (int)(addr - ptob (btop (addr)));

	/* get some virtual address space */

	if ((kmx = rmalloc (kernelmap, numClicks)) == 0)
	    {
	    printf ("bp: no kernelmap for bp mailbox\n");
	    panic ("bpPoke");
	    }

	vaddr = (char *)kmxtob (kmx);
     
	mapin (&Usrptmap [kmx], btop (vaddr), btop (addr) | PGT_VME_D16,
		numClicks, PG_V | PG_KW);

	cpudesc_vaddr [cpuNum] = vaddr += addrOffset;

	if (DEBUG_OPTION_1)
	    printf ("\nbp: cpu #%d mailbox addr = 0x%x, vaddr = 0x%x\n",
		    cpuNum, addr, vaddr);
	}

    switch (type)
	{
	case BP_INT_MAILBOX_1: *(char  *) vaddr = value; break;
	case BP_INT_MAILBOX_2: *(short *) vaddr = value; break;
	case BP_INT_MAILBOX_4: *(long  *) vaddr = value; break;
	case BP_INT_MAILBOX_R1: value = *(char  *) vaddr; break;
	case BP_INT_MAILBOX_R2: value = *(short *) vaddr; break;
	case BP_INT_MAILBOX_R4: value = *(long  *) vaddr; break;
	default: printf ("bp: bpPoke - bad mailbox type 0x%x\n", type); break;
	}

#ifdef	MAP_OUT
    /* example of mapping out memory */

    cpudesc_vaddr [cpuNum] = 0;
    mapout (&Usrptmap [kmx], numClicks, kmx);
    rmfree (kernelmap, numClicks, kmx);
#endif	MAP_OUT
    }
/*******************************************************************************
*
* bpProbe - probe for backplane memory
*
* RETURNS: memory available, or 0 if error
*/

int bpProbe (addr, unit)
    caddr_t addr;	/* address of backplane memory */
    int unit;		/* device number */

    {
    int memSize = bp_mem_size;
    FAST BP_SOFTC *bs = BP_SOFTC_P [unit];
    extern int peek (), pokec ();

    if (BP_MASTER)
	bpProbeType = bpParity ? pokec : peek;
    else
	bpProbeType = peek;

    if (DEBUG_OPTION_1)
	printf ("bp%d: debugging on\n", unit);

    /* anchor is at the start of shared memory */

    bs->bs_pAnchor = (BP_ANCHOR *) addr;

    if ((*bpProbeType) (addr, 0) == -1)
	{
	printf ("bp%d: shared memory not present at 0x%x (0x%x)\n",
		unit, bs->bs_pAnchor, vir2phys ((char*)bs->bs_pAnchor));
	return (0);
	}

    if (BP_MASTER &&
	bpInit ((char*)bs->bs_pAnchor, addr, memSize, bp_tas) == ERROR)
	{
	printf ("bp%d: bpProbe cannot initialize anchor!\n", unit);
	return (0);
	}

    if (DEBUG_OPTION_1)
	printf ("bp%d: shared memory size is %d bytes, at 0x%x (0x%x)\n",
	    unit, memSize, bs->bs_pAnchor, vir2phys ((char*)bs->bs_pAnchor));

    return (memSize);
    }

#else

/*******************************************************************************
*
* bpattach - make interface available
*
* Make interface available by filling in network interface record.
* System will initialize the interface when it is ready to accept packets. 
*
* RETURN: OK or ERROR
*/

STATUS bpattach (unit, pAnchor, procNum, intType, intArg1, intArg2, intArg3)
    int unit;		/* backplane unit number */
    char *pAnchor;	/* bus pointer to bp anchor */
    int procNum;	/* processor number in backplane */
    int intType;	/* interrupt type: poll, bus, mailbox */
    int intArg1;	/* as per interrupt type */
    int intArg2;	/* as per interrupt type */
    int intArg3;	/* as per interrupt type */

    {
    FAST BP_SOFTC *bs;
    FAST struct ifnet *ifp;
    FAST BP_ANCHOR *pBpAnchor = (BP_ANCHOR *) pAnchor;

    if (procNum == 0)
	bpProbeType = bpParity ? WRITE : READ;
    else
	bpProbeType = READ;

    /* allocate bp_softc structure */

    if ((bs = (BP_SOFTC *) malloc (sizeof (BP_SOFTC))) == NULL)
	return (ERROR);

    bp_softc [unit] = bs;
    ifp = &bs->bs_if;

    /* initialize the interface structure */

    bzero ((char *) bs, sizeof (BP_SOFTC));

    bs->bs_pAnchor = pBpAnchor;
    bs->bs_procNum = procNum;
    bs->bs_intType = intType;
    bs->bs_intArg1 = intArg1;
    bs->bs_intArg2 = intArg2;
    bs->bs_intArg3 = intArg3;

    /* wait for shared memory to be initialized; master CPU does sanity check */

    if (bpAlive (bs) == ERROR)
	{
	free ((char *) bs);
	return (ERROR);
	}

    if (DEBUG_OPTION_1)
	{
	printf ("bp%d: debugging on\n", unit);
	printf ("bpattach: anchor CPU can%s do real TAS.\n",
		bpTasOK ? "" : "not");
	}

    ether_attach (ifp, unit, "bp", bpIfInit, bpIoctl, bpOutput, bpReset);

    ifp->if_mtu = bp_mtu;

    return (OK);
    }
#endif	UNIX
/*******************************************************************************
*
* bpInit - initialize backplane anchor
*
* RETURNS:
*  OK or
*  ERROR if unable to setup data structures, or insufficient memory
*/

STATUS bpInit (pAnchor, pMem, memSize, tasOK)
    FAST char *pAnchor;	/* backplane anchor address */
    FAST char *pMem;	/* start of backplane shared memory, NONE=alloc */
    int memSize;	/* num bytes in bp shared memory, 0=0x100000 */
    BOOL tasOK;		/* TRUE = hardware can do test-and-set */

    {
    FAST BP_ANCHOR *pBpAnchor = (BP_ANCHOR *) pAnchor;
    FAST RING *pFreeRing;
    FAST int ix;
    FAST char *pMemEnd;	/* end of backplane shared memory */
    FAST CPU_DESC *pCpu;
    FAST BP_HEADER *pHdr;
    int npackets = 0;

    memSize = (memSize == 0) ? 0x100000 : memSize;
    bpTasOK = tasOK;

    /* if bp header specified, probe it, otherwise malloc it */

    if (pMem != (char *) NONE)
	{
	if ((memSize = bpSizeMem (pMem, memSize)) == 0)
	    {
	    printf ("bp: no shared memory at address 0x%x\n", pMem);
	    return (ERROR);
	    }

	/* if anchor is at start of shared region, skip over it */

	if (pMem == pAnchor)
	    {
	    pMem    += sizeof (BP_ANCHOR);
	    memSize -= sizeof (BP_ANCHOR);
	    }
	}
    else
	{
#ifndef	UNIX
	if ((pMem = malloc ((unsigned) memSize)) == NULL)
#endif	!UNIX
	    {
	    printf ("bp: can't allocate shared memory\n", pMem);
	    return (ERROR);
	    }
	}

    pMemEnd = pMem + memSize;
    pHdr = (BP_HEADER *) pMem;

    /* initialize bp anchor region */

    pBpAnchor->an_ready     = 0;		/* set bp not ready */
    pBpAnchor->an_heartbeat = 0;		/* clear heartbeat */
    pBpAnchor->an_pHdr      = KtoIval (pHdr, BP_HEADER *, pAnchor);

    if (pMem + sizeof (BP_HEADER) > pMemEnd)
	{
	printf ("bp: bpInit out of memory!\n");
	return (ERROR);
	}

    /* initialize bp header region */

    pHdr->hdr_ready = 0;                /* set CPU region not ready */
    pHdr->hdr_ncpu = NCPU;              /* set number of CPUs */

    bcopy ((char *) bpEnetAdrs, (char *) pHdr->hdr_enetAdrs, 6);

    /* initialize each CPU descriptor */

    for (ix = 0; ix < MAXCPU; ++ix)
	{
	pCpu = &pHdr->hdr_cpuDesc[ix];

	pCpu->cd_active      = FALSE;
	pCpu->cd_cpuNum      = ix;
	pCpu->cd_pInputRing  = NULL;
	}

    /* initialize free ring */

    pMem += sizeof (BP_HEADER);

    if ((pMem + sizeof (struct ring256) + NCPU * sizeof (struct ring32))
								> pMemEnd)
	{
	printf ("bp: bpInit out of memory!\n");
	return (ERROR);
	}

    ringinit ((pFreeRing = (RING *) pMem), 256);
    pHdr->hdr_pFreeRing = KtoIval (pFreeRing, RING *, pAnchor);
    pMem += sizeof (struct ring256);

    /* initialize each input ring */

    for (ix = 0; ix < NCPU; ++ix)
	{
	ringinit ((RING *) pMem, 32);
	pHdr->hdr_cpuDesc[ix].cd_pInputRing = KtoIval (pMem, RING *, pAnchor);
	pMem += sizeof (struct ring32);
	}

    /* initialize as many buffer blocks as will fit in remaining shared memory*/

    if ((pMem + sizeof (BUF_NODE) + BP_BUFSIZ) > pMemEnd)
	{
	printf ("bp: bpInit out of memory!\n");
	return (ERROR);
	}

    while ((pMem + sizeof (BUF_NODE) + BP_BUFSIZ) <= pMemEnd)
	{
	((BUF_NODE *)pMem)->b_addr = KtoIval (pMem + sizeof (BUF_NODE),
					      char*, pAnchor);
	((BUF_NODE *)pMem)->b_len  = BP_BUFSIZ;

	(void)ringput (pFreeRing, KtoIval (pMem, int, pAnchor));
	pMem += sizeof (BUF_NODE) + BP_BUFSIZ;
	npackets++;
	}

    if (DEBUG_OPTION_1)
	{
	printf ("bp: number of %d byte packets is %d\n", BP_BUFSIZ, npackets);
	printf ("bp: %d bytes unused from 0x%x to 0x%x\n",
		pMemEnd - pMem, vir2phys (pMem), vir2phys (pMemEnd));
	}

    /* mark bp as ready */

    if (bpTasOK)
	pHdr->hdr_ready = pBpAnchor->an_ready = BP_READY;
    else
	pHdr->hdr_ready = pBpAnchor->an_ready = BP_READY_SOFT_TAS;

    /* start heartbeat */

    bpHeartbeatRate = hz;
#ifdef	UNIX
    timeout (bpHeartbeat, (caddr_t) pAnchor, bpHeartbeatRate);
#else
    /* create watchdog timer for heartbeat */
    
    if ((pBpAnchor->an_wdog = wdCreate ()) == NULL)
	return (ERROR);
    wdStart (pBpAnchor->an_wdog, bpHeartbeatRate, bpHeartbeat, (int) pAnchor);
#endif	UNIX

    return (OK);
    }
/*******************************************************************************
*
* bpReset - mark a bp inactive
*
* called from reboot and panic
*/

LOCAL VOID bpReset (unit)
    FAST int unit;

    {
    FAST BP_SOFTC *bs;
    FAST BUF_NODE *pBufNode;
#ifdef	UNIX
    FAST struct mb_device *md;

    if (unit >= NBP || (md = bpinfo[unit]) == 0 || md->md_alive == 0)
	return;
#endif	UNIX

    bs = BP_SOFTC_P [unit];

    if (bs != NULL && bs->bs_pHdr != NULL)
	{
	/* turn off active flag in our CPU descriptor in the
	 * shared header, then throw away anything on our input ring */

	bs->bs_pHdr->hdr_cpuDesc[bs->bs_procNum].cd_active = FALSE;

	while (pBufNode = (BUF_NODE *) ringget (bs->bs_pInputRing))
	    (void)ringput (bs->bs_pFreeRing, (int) pBufNode);
	}
    }
/*******************************************************************************
*
* bpAlive - check for throbbing heartbeat
*
* This routine is called for CPU 0 (master) as well as the rest.
* The timeout is quick if the master can't see its own ticker.
*
* RETURNS: OK or ERROR if unable to find anchor after 10 minutes
*/

LOCAL STATUS bpAlive (bs)
    BP_SOFTC *bs;

    {
    int readyValue;
    int heartbeat;
    FAST unsigned int oldBeat = 0;
    FAST int countdown             = 10 * 60 * 60 * hz;	/* wait 10 minutes */
    FAST BP_ANCHOR *pAnchor        = bs->bs_pAnchor;
#ifdef	UNIX
    int level = spl3 ();	/* find out priority level */
    splx (level);
#endif	UNIX

    if (bs->bs_procNum == 0)
	countdown = 2 * hz;	/* don't wait around for self if master CPU */

    while (--countdown > 0)
	{
	/* probe test:
	 *  1) anchor must be readable (memory must be visable to us),
	 *  2) anchor ready value must be BP_READY or BP_READY_SOFT_TAS,
	 *  3) anchor heartbeat must be increasing.
	 */

#ifdef	UNIX
	if (peek ((short *) &pAnchor->an_ready)     != -1 &&
	    peek ((short *) &pAnchor->an_heartbeat) != -1)
	    {
	    readyValue = pAnchor->an_ready;
	    heartbeat  = pAnchor->an_heartbeat;
#else
	if (vxMemProbe ((char *) &pAnchor->an_ready,
			 READ, 4, (char *) &readyValue) == OK &&
	    vxMemProbe ((char *) &pAnchor->an_heartbeat,
			 READ, 4, (char *) &heartbeat) == OK)
	    {
#endif	UNIX

	    if ((countdown % 10) == 6)
		{
		printf ("\nbp: bpAlive ready = 0x%x, heartbeat = %d\n",
			    readyValue, heartbeat);
		} 

	    if (readyValue == BP_READY || readyValue == BP_READY_SOFT_TAS)
		{
		if (oldBeat == 0 || heartbeat <= oldBeat)
		    oldBeat = heartbeat;
		else
		    {
		    /* heartbeat increased, passed test */
		    bpTasOK = readyValue == BP_READY;
		    return (OK);
		    }
		}
	    }
	else if ((countdown % 10) == 8)
	    printf ("\nbp: bpAlive bus error\n");

#ifdef	UNIX
	timeout (wakeup, (caddr_t)bs, 1);
        sleep (bs, level);
#else
	taskDelay (hz);
#endif	UNIX
	}

#ifndef	UNIX
    errnoSet (S_iosLib_CONTROLLER_NOT_PRESENT);
#endif	UNIX

    /* give up - didn't get initialized */

    return (ERROR);
    }
/*******************************************************************************
*
* bpIfInit - initialize interface
*
* RETURNS: OK or ERROR
*/

LOCAL STATUS bpIfInit (unit)
    int unit;

    {
    FAST CPU_DESC *pCpu;
    FAST BP_SOFTC *bs = BP_SOFTC_P [unit];
    FAST BP_HEADER *pHdr;

    /* find the anchor and CPU descriptor */
    pHdr = ItoKval (bs->bs_pAnchor->an_pHdr, BP_HEADER *, bs->bs_pAnchor);
    bs->bs_pHdr = pHdr;

#ifdef	UNIX
    /* wait for shared memory to be initialized */

    if (!BP_MASTER && bpAlive (bs) == ERROR)
	return (ERROR); 
#endif	UNIX

    pCpu = &pHdr->hdr_cpuDesc [bs->bs_procNum];

    bs->bs_pFreeRing = ItoKval (pHdr->hdr_pFreeRing, RING *, bs->bs_pAnchor);

    bs->bs_pInputRing= ItoKval(pHdr->hdr_cpuDesc[bs->bs_procNum].cd_pInputRing,
				RING *, bs->bs_pAnchor);

    /* fill in ethernet address, which is the backplane ethernet address with
     * our processor number as the last byte. */

#ifdef	UNIX
    bcopy ((char *) pHdr->hdr_enetAdrs,
	    (char *) bs->bs_enaddr.ether_addr_octet, 6);
    bs->bs_enaddr.ether_addr_octet [5] = (char) bs->bs_procNum;
#else
    bcopy ((char *) pHdr->hdr_enetAdrs, (char *) bs->bs_enaddr, 6);
    bs->bs_enaddr [5] = (char) bs->bs_procNum;
#endif	UNIX

    /* check that our input buffer was established
     * (in case NCPU < our cpu < MAXCPU) */

    if (pCpu->cd_pInputRing == NULL)
	return (ERROR);

    /* fill in our info in the CPU descriptor in shared memory */

    pCpu->cd_unit = unit;	/* set our internal unit num */

    pCpu->cd_intType = bs->bs_intType;	/* tell others how to interrupt us */
    pCpu->cd_intArg1 = bs->bs_intArg1;
    pCpu->cd_intArg2 = bs->bs_intArg2;
    pCpu->cd_intArg3 = bs->bs_intArg3;

    /* connect and enable our interprocessor interrupt */

    if (DEBUG_OPTION_1)
	printf ("bp%d: connecting...\n", unit);

    if (bpConnect (bs, unit) != OK || bpIntEnable (bs) != OK)
	return (ERROR);

    /* tell others we're here now */

    pCpu->cd_active = TRUE;

    if ((bs->bs_if.if_flags & IFF_RUNNING) == 0)
	bs->bs_if.if_flags |= IFF_UP | IFF_RUNNING;

    return (OK);
    }
/*******************************************************************************
*
* bpintr - interrupt handler
*/

VOID bpintr (unit)
    int unit;

    {
    FAST BP_SOFTC *bs = BP_SOFTC_P [unit];

    if (bs != NULL && bs->bs_pInputRing != NULL)	/* input ring exists? */
	{
#ifndef	UNIX
	bpIntAck (bs);
#endif	UNIX

	if (!bs->bs_readPending)
	    {
	    bs->bs_readPending = TRUE;
#ifdef	UNIX
	    bpReadAll (unit);
#else
	    netJobAdd (bpReadAll, unit);	/* read from input ring */
#endif	UNIX
	    }
	}
    }
/*******************************************************************************
*
* bpReadAll -
*/

LOCAL VOID bpReadAll (unit)
    int unit;

    {
    FAST BP_SOFTC *bs = BP_SOFTC_P [unit];
    FAST BUF_NODE *pBufNode;

    bs->bs_readPending = FALSE;

    while ((pBufNode = (BUF_NODE *) ringget (bs->bs_pInputRing)) != 0)
	{
	bpRead (bs, ItoKval (pBufNode, BUF_NODE *, bs->bs_pAnchor));
	(void)ringput (bs->bs_pFreeRing, (int) pBufNode);
	}
    }
/*******************************************************************************
*
* bpRead -
*/

LOCAL VOID bpRead (bs, pBufNode)
    BP_SOFTC *bs;
    BUF_NODE *pBufNode;

    {
    FAST struct ether_header *eh;
    FAST struct mbuf *m;
    int len;
    int off;
    FAST unsigned char *pData;

    bs->bs_if.if_ipackets++;	/* count input packets */

    /* get length and pointer to packet */

    len = pBufNode->b_len;
    eh = ItoKval (pBufNode->b_addr, struct ether_header *, bs->bs_pAnchor);


#ifndef	UNIX
    /* call input hook if any */

    if ((etherInputHookRtn != NULL) &&
	(* etherInputHookRtn) (&bs->bs_if, (char *) eh, pBufNode->b_len))
	{
	return;		/* input hook has already processed packet */
	}

#endif	!UNIX

    /*
     * Deal with trailer protocol: if type is PUP trailer
     * get true type from first 16-bit word past data.
     * Remember that type was trailer by setting off.
     */

    len -= sizeof (struct ether_header);
    pData = ((unsigned char *) eh) + sizeof (struct ether_header);

#ifdef	UNIX
    check_trailer (eh, pData, &len, &off);
#else
    check_trailer (eh, pData, &len, &off, &bs->bs_if);
#endif	UNIX

    if (len == 0)
	return;
#ifdef	UNIX
    m = copy_to_mbufs (pData, len, off);
#else
    m = copy_to_mbufs (pData, len, off, &bs->bs_if);
#endif	UNIX

    if (m != 0)
	do_protocol (eh, m, &bs->bs_ac, len);
    }
#ifdef	UNIX
/*******************************************************************************
*
* bpStart -
*/

LOCAL VOID bpStart (unit)
    int unit;

    {
    FAST BP_SOFTC *bs = BP_SOFTC_P [unit];
    FAST struct mbuf *m;

    while (bs->bs_if.if_snd.ifq_head > 0)
	{
	IF_DEQUEUE(&bs->bs_if.if_snd, m);
	(void) bpPut (bs, m);
	}
    }
#endif	UNIX
/*******************************************************************************
*
* bpOutput - net output routine
*
* Encapsulate a packet of type family for the local net.
* Use trailer local net encapsulation if enough data in first
* packet leaves a multiple of 512 bytes of data in remainder.
*
* RETURNS: 0 if successful, errno otherwise (as per network requirements)
*/

LOCAL int bpOutput (ifp, m0, dst)
    struct ifnet *ifp;
    struct mbuf *m0;
    struct sockaddr *dst;

    {
#ifdef	UNIX
    return (ether_output (ifp, m0, dst, bpStart,
	    &(BP_SOFTC_P [ifp->if_unit])->bs_ac));
#else
    return (ether_output (ifp, m0, dst, (FUNCPTR) NULL,	/* XXX to be remedied */
	    &BP_SOFTC_P [ifp->if_unit]->bs_ac));
#endif	UNIX
    }
/*******************************************************************************
*
* bpPut - copy from mbuf chain to shared memory buffer
*
* RETURNS: OK or ERROR if no room left in free ring
*/

LOCAL STATUS bpPut (bs, mp)
    FAST BP_SOFTC *bs;
    FAST struct mbuf *mp;

    {
    FAST struct mbuf *m;
    FAST BUF_NODE *pBufNode;
    FAST u_char *buf;
    u_char *bufStart;

    if ((pBufNode = (BUF_NODE *) ringget (bs->bs_pFreeRing)) == NULL)
	{
	bs->bs_if.if_oerrors++;	/* could be counted elsewhere */
	return (ERROR);
	}

    ItoK (pBufNode, BUF_NODE *, bs->bs_pAnchor);
    bufStart = ItoKval (pBufNode->b_addr, u_char *, bs->bs_pAnchor);

    for (buf = bufStart, m = mp; m != NULL; buf += m->m_len, m = m->m_next)
	bcopy ((caddr_t) mtod (m, u_char *), (caddr_t) buf, m->m_len);

    pBufNode->b_len = max (MINPKTSIZE, buf - bufStart);

#ifndef	UNIX
    /* call output hook if any */

    if ((etherOutputHookRtn != NULL) &&
	(* etherOutputHookRtn) (&bs->bs_if, bufStart, pBufNode->b_len))
	{
	return (OK);	/* output hook has already processed packet */
	}

#endif	!UNIX

    /* send to specified processor:
     * proc number is in last byte of ethernet address */

    bpXmit (bs, pBufNode, (int) ((struct ether_header *)
#ifdef	UNIX
	       (bufStart))->ether_dhost.ether_addr_octet[5], bs->bs_procNum);
#else
	       (bufStart))->ether_dhost[5], bs->bs_procNum);
#endif	UNIX

    m_freem (mp);

    return (OK);
    }
/*******************************************************************************
*
* bpXmit - transmit buffer
*/

LOCAL VOID bpXmit (bs, pBufNode, destProcNum, srcProcNum)
    FAST BP_SOFTC *bs;
    BUF_NODE *pBufNode;
    int destProcNum;	/* destination CPU */
    int srcProcNum;	/* originator */

    {
    BP_ANCHOR *pAnchor = bs->bs_pAnchor;
    u_char *buf;
    FAST CPU_DESC *pCpu;
    FAST BUF_NODE *pBufNode0;

    if (destProcNum < 0 || destProcNum >= MAXCPU)
	{
	/* broadcast to all CPUs on network */

	buf = ItoKval (pBufNode->b_addr, u_char *, pAnchor);

	for (destProcNum = MAXCPU - 1; destProcNum >= 0; --destProcNum)
	    {
	    pCpu = &bs->bs_pHdr->hdr_cpuDesc[destProcNum];

	    if (!pCpu->cd_active)
		continue;

	    if ((pBufNode0 = (BUF_NODE *) ringget (bs->bs_pFreeRing)) == NULL)
		break;

	    ItoK (pBufNode0, BUF_NODE *, pAnchor);

	    pBufNode0->b_len = pBufNode->b_len;
	    bcopy ((caddr_t) buf,
		   (caddr_t) ItoKval (pBufNode0->b_addr, u_char *, pAnchor),
		   pBufNode->b_len);

	    switch (ringput (ItoKval (pCpu->cd_pInputRing, RING *, pAnchor),
			     KtoIval (pBufNode0, int, pAnchor)))
		{
		/* if input ring was full, then put bufNode back on free ring */
		case 0:
		    if (DEBUG_OPTION_2)
			dbgprintf ("bp: dropped packet\n");
		    (void)ringput (bs->bs_pFreeRing,
			     KtoIval (pBufNode0, int, pAnchor));
		    break;

		/* if input ring was empty, then interrupt destination CPU */
		case 1:
		    bpIntGen (pCpu, srcProcNum);
		    break;
		}
	    }

	(void)ringput (bs->bs_pFreeRing, KtoIval (pBufNode, int, pAnchor));
	}
    else
	{
	pCpu = &bs->bs_pHdr->hdr_cpuDesc [destProcNum];

	if (!pCpu->cd_active)
	    (void)ringput (bs->bs_pFreeRing, KtoIval (pBufNode, int, pAnchor));
	else
	    {
	    switch (ringput (ItoKval (pCpu->cd_pInputRing, RING *, pAnchor),
	    		     KtoIval (pBufNode,int, pAnchor)))
		{
		/* if input ring was full, then put bufNode back on free ring */
		case 0:
		    if (DEBUG_OPTION_2)
			dbgprintf ("bp: dropped packet\n");
		    (void)ringput (bs->bs_pFreeRing,
			     KtoIval (pBufNode, int, pAnchor));
		    break;

		/* if input ring was empty, then interrupt destination CPU */
		case 1:
		    bpIntGen (pCpu, srcProcNum);
		    break;
		}
	    }
	}
    }
#ifdef	UNIX
/*******************************************************************************
*
* bpPollUnix - UNIX backplane poll
*
* This routine is called by UNIX's timeout routine to poll the
* backplane input rings for this CPU.
*/

LOCAL VOID bpPollUnix ()

    {
    FAST int unit;

    for (unit = 0; unit < NBP; unit++)
	{
	/* if header structure non-zero then unit is initialized */

	if (bp_softc[unit].bs_pHdr)
	    bpintr (unit);
	}

    /* loop forever, with a delay of 1 */

    timeout (bpPollUnix, (caddr_t)0, 1);
    }
#else
/*******************************************************************************
*
* bpPollTask - backplane poll task
*
* This routine is spawned as task to poll the backplane input rings for
* this CPU.  It is only used if no backplane interrupt mechanism is
* used.
*/

VOID bpPollTask ()

    {
    FAST int unit;

    FOREVER
	{
	for (unit = 0; unit < NBP; unit++)
	    {
	    if (BP_SOFTC_P [unit] != 0)
		bpintr (unit);
	    }
	}
    }
/*******************************************************************************
*
* bpIntAck - acknowledge our backplane interrupt
*/

LOCAL VOID bpIntAck (bs)
    FAST BP_SOFTC *bs;

    {
    switch (bs->bs_intType)
	{
	case BP_INT_NONE:

	case BP_INT_MAILBOX_1:
	case BP_INT_MAILBOX_2:
	case BP_INT_MAILBOX_4:
	case BP_INT_MAILBOX_R1:
	case BP_INT_MAILBOX_R2:
	case BP_INT_MAILBOX_R4:
	    break;

	/* bus interrupt:
	 *   arg1 = level
	 *   arg2 = vector
	 */

	case BP_INT_BUS:
	    sysBusIntAck (bs->bs_intArg1);
	    break;
	}
    }
#endif	!UNIX
/*******************************************************************************
*
* bpConnect - connect to our backplane using the requested interrupt type
*
* RETURNS: OK or ERROR if interrupt type not supported
*/

LOCAL STATUS bpConnect (bs, unit)
    FAST BP_SOFTC *bs;
    FAST int unit;

    {
    STATUS status;

    switch (bs->bs_intType)
	{
	case BP_INT_NONE:
	/* no interprocessor interrupt - spawn background polling task */

#ifdef	UNIX
	    {
	    static BOOL bpIsPolling = FALSE;

	    if (!bpIsPolling)
		{
		bpIsPolling = TRUE;
		timeout (bpPollUnix, (caddr_t)0, 1);
		}
	    status = OK;
	    }
#else
	    bpPollTaskId = taskSpawn ("bpPollTask", bpPollTaskPriority,
				      bpPollTaskOptions, bpPollTaskStackSize,
				      bpPollTask, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);

	    status = bpPollTaskId == ERROR ? ERROR : OK;
#endif	UNIX

	    break;

	case BP_INT_MAILBOX_1:
	case BP_INT_MAILBOX_2:
	case BP_INT_MAILBOX_4:
	case BP_INT_MAILBOX_R1:
	case BP_INT_MAILBOX_R2:
	case BP_INT_MAILBOX_R4:
	/* interrupt by mailbox (write to bus address):
	 *   arg1 = bus address space
	 *   arg2 = address
	 *   arg3 = value
	 */

#ifdef	UNIX
	    printf ("bp%d: bpConnect interrupt type not implemented\n", unit);
	    status = ERROR;
#else
	    status = sysMailboxConnect (bpintr, unit);
#endif	UNIX
	    break;

	case BP_INT_BUS:
	/* bus interrupt:
	 *   arg1 = level
	 *   arg2 = vector
	 */
#ifdef	UNIX
	    status = OK;  /* nothing to do, done in kernel configuration */
#else
	    status = intConnect (INUM_TO_IVEC (bs->bs_intArg2), bpintr, unit);
#endif	UNIX
	    break;

	default:
	    printf ("bp%d: bpConnect bad interrupt type: %d\n",
		    unit, bs->bs_intType);
	    status = ERROR;
	    break;
	}

    return (status);
    }
/*******************************************************************************
*
* bpIntEnable - enable our backplane interrupt
*
* RETURNS: OK or ERROR
*/

LOCAL STATUS bpIntEnable (bs)
    FAST BP_SOFTC *bs;

    {
    STATUS status;
    int unit = bs->bs_if.if_unit;

    switch (bs->bs_intType)
	{
	case BP_INT_NONE:
	    status = OK;	/* no interrupt (polling) */
	    break;

	case BP_INT_MAILBOX_1:
	case BP_INT_MAILBOX_2:
	case BP_INT_MAILBOX_4:
	case BP_INT_MAILBOX_R1:
	case BP_INT_MAILBOX_R2:
	case BP_INT_MAILBOX_R4:
	/* interrupt by mailbox (write to bus address):
	 *   arg1 = bus address space
	 *   arg2 = address
	 *   arg3 = value
	 */

#ifdef	UNIX
	    status = ERROR;
	    printf ("bp%d: bpIntEnable interrupt type 0x%x not implemented\n",
		    unit, bs->bs_intType);
#else
	    status = sysMailboxEnable ((char *) bs->bs_intArg2);
	    if (status == ERROR)
		printf ("bp%d: error enabling mailbox interrupt addr 0x%x.\n",
			  unit, bs->bs_intArg2);
#endif	UNIX
	    break;

	case BP_INT_BUS:
	/* bus interrupt:
	 *   arg1 = level
	 *   arg2 = vector
	 */

#ifdef	UNIX
	    status = OK;  /* nothing to do */
#else
	    status = sysIntEnable (bs->bs_intArg1);
	    if (status == ERROR)
		printf ("bp%d: error enabling bus interrupt level %d.\n",
			  unit, bs->bs_intArg1);
#endif	UNIX
	    break;

	default:
	    printf ("bp%d: bpIntEnable bad interrupt type: %d\n",
		    unit, bs->bs_intType);
	    status = ERROR;
	    break;
	}

    return (status);
    }
/*******************************************************************************
*
* bpIntGen - interrupt another CPU
*/

LOCAL VOID bpIntGen (pCpu, srcProcNum)
    FAST CPU_DESC *pCpu;	/* ptr to descriptor of CPU to interrupt */
    int srcProcNum;		/* our procNum on this network */

    {
    char *pMailbox;

    /* if we're trying to interrupt ourself (a bad idea on some CPUs (is20))
     * simply invoke the interrupt handler */

    if (pCpu->cd_cpuNum == srcProcNum)
	{
	/* read input ring */

#ifdef	UNIX
	bpReadAll (pCpu->cd_unit);
#else
	netJobAdd (bpReadAll, pCpu->cd_unit);
#endif	UNIX
	return;
	}

    switch (pCpu->cd_intType)
	{
	case BP_INT_NONE:
	    /* no interrupt (CPU is polling) */
	    break;

	case BP_INT_MAILBOX_1:
	case BP_INT_MAILBOX_R1:
	/* interrupt by mailbox (read/write to bus address):
	 *   arg1 = bus address space
	 *   arg2 = address
	 *   arg3 = value
	 */

	    if (sysBusToLocalAdrs (pCpu->cd_intArg1,
				   (char *) pCpu->cd_intArg2,
				   &pMailbox) == OK)
		{
#ifdef	UNIX
		bpPoke (pMailbox, pCpu->cd_intType, pCpu->cd_intArg3,
			pCpu->cd_cpuNum);
#else
		if (pCpu->cd_intType == BP_INT_MAILBOX_R1)
		    bpBitBucket = * (char *)pMailbox;
		else
		    *pMailbox = pCpu->cd_intArg3;
#endif	UNIX
		}
	    else
		{
		logMsg (
    "bp%d: bpIntGen bad mailbox interrupt, addr space = 0x%x, addr = 0x%x\n",
		     pCpu->cd_unit, pCpu->cd_intArg1, pCpu->cd_intArg2);
		}
	    break;

	case BP_INT_MAILBOX_2:
	case BP_INT_MAILBOX_R2:
	    if (sysBusToLocalAdrs (pCpu->cd_intArg1,
				   (char *) pCpu->cd_intArg2,
				   &pMailbox) == OK)
		{
#ifdef	UNIX
		bpPoke (pMailbox, pCpu->cd_intType, pCpu->cd_intArg3,
			pCpu->cd_cpuNum);
#else
		if (pCpu->cd_intType == BP_INT_MAILBOX_R2)
		    bpBitBucket = * (short *)pMailbox;
		else
		    * (short *) pMailbox = pCpu->cd_intArg3;
#endif	UNIX
		}
	    else
		{
		logMsg (
    "bp%d: bpIntGen bad mailbox interrupt, addr space = 0x%x, addr = 0x%x\n",
		     pCpu->cd_unit, pCpu->cd_intArg1, pCpu->cd_intArg2);
		}
	    break;

	case BP_INT_MAILBOX_4:
	case BP_INT_MAILBOX_R4:
	    if (sysBusToLocalAdrs (pCpu->cd_intArg1,
				   (char *) pCpu->cd_intArg2,
				   &pMailbox) == OK)
		{
#ifdef	UNIX
		bpPoke (pMailbox, pCpu->cd_intType, pCpu->cd_intArg3,
			pCpu->cd_cpuNum);
#else
		if (pCpu->cd_intType == BP_INT_MAILBOX_R4)
		    bpBitBucket = * (long *)pMailbox;
		else
		    * (long *) pMailbox = pCpu->cd_intArg3;
#endif	UNIX
		}
	    else
		{
		logMsg (
    "bp%d: bpIntGen bad mailbox interrupt, addr space = 0x%x, addr = 0x%x\n",
		     pCpu->cd_unit, pCpu->cd_intArg1, pCpu->cd_intArg2);
		}
	    break;


	case BP_INT_BUS:
	/* bus interrupt:
	 *   arg1 = level
	 *   arg2 = vector
	 */

#ifndef	UNIX
	    if (sysBusIntGen (pCpu->cd_intArg1, pCpu->cd_intArg2) == ERROR)
		{
		logMsg (
		"bp%d: bpIntGen interrupt failed (level=%d vector=0x%x)\n",
			pCpu->cd_unit, pCpu->cd_intArg1, pCpu->cd_intArg2);
		}
	    break;
#endif	!UNIX
	/* UNIX drops through */

	default:
	    logMsg ("bp%d: bpIntGen bad interrupt type: proc = %d, type = %d\n",
		     pCpu->cd_unit, pCpu->cd_cpuNum, pCpu->cd_intType);
	    break;
	}
    }
#ifdef	UNIX
/********************************************************************************
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/
 
STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,     */
                        /* use address modifier codes as defined in vme.h, */
                        /* such as VME_AM_STD_SUP_DATA                     */
    char *busAdrs;      /* bus address to convert                          */
    char **pLocalAdrs;  /* where to return local address                   */
 
    {
    switch (adrsSpace)
        {
        case VME_AM_SUP_SHORT_IO:
        case VME_AM_USR_SHORT_IO:
            *pLocalAdrs = (char *) (0xffff0000 | (int) busAdrs);
            return (OK);
 
        case VME_AM_STD_SUP_ASCENDING:
        case VME_AM_STD_SUP_PGM:
        case VME_AM_STD_SUP_DATA:
        case VME_AM_STD_USR_ASCENDING:
        case VME_AM_STD_USR_PGM:
        case VME_AM_STD_USR_DATA:
            *pLocalAdrs = (char *) (0xff000000 | (int) busAdrs);
            return (OK);
 
        case VME_AM_EXT_SUP_ASCENDING:
        case VME_AM_EXT_SUP_PGM:
        case VME_AM_EXT_SUP_DATA:
        case VME_AM_EXT_USR_ASCENDING:
        case VME_AM_EXT_USR_PGM:
        case VME_AM_EXT_USR_DATA:
            *pLocalAdrs = busAdrs; /* XXX check if on board memory? */
            return (OK);
 
        default:
            return (ERROR);
        }
    }
#endif	UNIX
/*******************************************************************************
*
* bpHeartbeat -
*/

LOCAL VOID bpHeartbeat (pAnchor)
    FAST BP_ANCHOR *pAnchor;

    {
    ++pAnchor->an_heartbeat;
#ifdef	UNIX
    timeout (bpHeartbeat, (caddr_t) pAnchor, bpHeartbeatRate);
#else
    wdStart (pAnchor->an_wdog, bpHeartbeatRate, bpHeartbeat, (int) pAnchor);
#endif	UNIX
    }
/*******************************************************************************
*
* bpSizeMem - size shared memory pool
*
* RETURNS: amount of available memory
*/

LOCAL int bpSizeMem (memAdrs, memSize)
    caddr_t memAdrs;	/* address of start of pool to start probing */
    int memSize;	/* max size of memory pool */

    {
    FAST int size;

    for (size = 0; size < memSize; size += 4096)
	{
#ifdef UNIX
        if ((*bpProbeType) ((char *)(memAdrs + size), 0) == -1)
#else
	char bitBucket;

	if (vxMemProbe (memAdrs + size, bpProbeType, 1, &bitBucket) != OK)
#endif UNIX
	    break;
	}

    if (bpParity)
	bzero (memAdrs, memSize);

    /* return the amount of memory found */

    return (size);
    }
/*******************************************************************************
*
* bpIoctl - process an ioctl request
*
* RETURNS: 0 if successful, errno otherwise (as per network requirements)
*/

LOCAL int bpIoctl (ifp, cmd, data)
    FAST struct ifnet *ifp;
    int cmd;
    caddr_t data;

    {
    int error = 0;
    int unit  = ifp->if_unit;
    FAST BP_SOFTC *bs = BP_SOFTC_P [unit];
    int level = splimp ();

    switch (cmd)
	{
	case SIOCSIFADDR:
#ifdef	UNIX
	    error = set_if_addr (ifp, data, &bs->bs_enaddr);
#else
	    error = set_if_addr (ifp, data, bs->bs_enaddr);
#endif	UNIX
	    break;

	case SIOCGIFADDR:
#ifdef	UNIX
	    bcopy ((caddr_t) &bs->bs_enaddr,
		  (caddr_t) &((struct ifreq *)data)->ifr_addr.sa_data[0],
		  sizeof (struct ether_addr));
#else
	    bcopy ((caddr_t) bs->bs_enaddr,
		  (caddr_t) &((struct ifreq *)data)->ifr_addr.sa_data[0],
		  6);
#endif	UNIX
	    break;

	case SIOCGIFFLAGS:
	    *(short *)data = ifp->if_flags;
	    break;

	case SIOCSIFFLAGS:
	    {
	    int promisc;
	    struct sockaddr sa;
	    short flags = *(short *)data;
	    int up = flags & IFF_UP;

	    if (up != (ifp->if_flags & IFF_UP))
		{
		if (!up)
		    {
		    if (bpIfInit (unit) == ERROR)
			printf ("bp%d: bpIoctl error going up\n", unit);
		    }
		else
		    {
#ifdef	SUN3_5
		    sa = ifp->if_addr;
		    bzero ((caddr_t)&ifp->if_addr, sizeof (struct sockaddr));
		    if (bpIfInit (unit) == ERROR)
			printf ("bp%d: bpIoctl error going down\n", unit);
		    ifp->if_addr = sa;
#else
		    sa = ifp->if_addrlist->ifa_addr;
		    bzero ((caddr_t)&ifp->if_addrlist->ifa_addr,
			    sizeof (struct sockaddr));
		    if (bpIfInit (unit) == ERROR)
			printf ("bp%d: bpIoctl error going down\n", unit);
		    ifp->if_addrlist->ifa_addr = sa;
#endif	SUN3_5
		    }
		}

	    if (ifp->if_flags & IFF_UP)
		flags |= (IFF_UP|IFF_RUNNING);
	    else
		flags &= ~(IFF_UP|IFF_RUNNING);

	    promisc = flags & IFF_PROMISC;
	    if (promisc != (ifp->if_flags & IFF_PROMISC))
		{
		if ((ifp->if_flags & IFF_UP) == 0)
		    {
		    flags &= ~IFF_PROMISC;
		    error = ENETDOWN;
		    } 
		else
		    {
		    /* set promiscuous? */
		    }
		}

	    ifp->if_flags = flags;
	    }
	break;

	default:
	    printf ("bp%d: ioctl 0x%x not implemented\n", unit, cmd);
	    error = EINVAL;
	    break;
	}

    (void) splx (level);
    return (error);
    }



#define TAS(x)	(bpTasOK ? sysBusTas (x) : vxTasac (x))

#ifdef	UNIX
#define intLock		spl7
#define intUnlock	splx
#endif	UNIX


#define	DELAY_COUNT	5000	/* how high to count when waiting on TAS */

/*******************************************************************************
*
* ringinit -
*
* Initialize the ring pointed to by "rp" to contain "size" slots.
* The ring is set up to be empty and unlocked.
* Size had better be a power of 2!
*/

LOCAL VOID ringinit (rp, size)
    FAST struct ring *rp;
    short size;

    {
    rp->r_tas = rp->r_rdidx = rp->r_wrtidx = 0;
    rp->r_size = size;
    }
/*******************************************************************************
*
* ringput -
*
* Add a new entry "v" to the ring "rp".
* Returns 0 if ring was full, else the new number of entries in the ring.
* Uses test-and-set routine, allowing multi-processor interlocked access
* to the ring IF the ring is in VME memory.  
* Masks all interrupts, in order to prevent reentrancy and to
* minimize ring lockout.
*
* RETURNS: new number of entries, or 0 if ring was full
*/

LOCAL int ringput (rp, v)
    FAST struct ring *rp;
    int v;

    {
    FAST int x = DELAY_COUNT;	/* don't wait forever for lock */
    FAST int s;

    for (;;)
	{
	/* disable interrupts, then lock the ring */
	s = intLock ();
	if (TAS(&rp->r_tas))
	    break;
	intUnlock (s);
	if (--x == 0)
	    return (0);	/* ring is stuck! */
	}

    if ((x = (rp->r_wrtidx + 1) & (rp->r_size - 1)) != rp->r_rdidx)
	{
	rp->r_slot[rp->r_wrtidx] = v;
	rp->r_wrtidx = x;
	}			/* if x == rdidx then ring is full */

    if ((x -= rp->r_rdidx) < 0)
	x += rp->r_size;

    rp->r_tas = 0;			/* unlock ring */
    intUnlock (s);
    return (x);
    }
/*******************************************************************************
*
* ringget -
*
* Extract next entry from the ring "rp".
* Uses test-and-set routine, allowing multi-processor interlocked access
* to the ring IF the ring is in VME memory.  
* Masks all interrupts, in order to prevent reentrancy and to
* minimize ring lockout.
*
* RETURNS: 0 if ring is empty, else the extracted entry
*/

LOCAL int ringget (rp)
    FAST struct ring *rp;

    {
    FAST int x = DELAY_COUNT;	/* don't wait forever for lock */
    FAST int s;

    for (;;)
	{
	/* disable interrupts, then lock the ring */
	s = intLock ();
	if (TAS(&rp->r_tas))
	    break;
	intUnlock (s);

	if (--x == 0)
	    return (0);	/* ring is stuck! */
	}

    if (rp->r_rdidx != rp->r_wrtidx)
	{
	x = rp->r_slot[rp->r_rdidx];
	rp->r_rdidx = (rp->r_rdidx + 1) & (rp->r_size - 1);
	}
    else
	x = 0;

    rp->r_tas = 0;			/* unlock ring */

    intUnlock (s);
    return (x);
    }
/******************************************************************************
*
* ringcount - get number of entries in ring
*/

LOCAL int ringcount (pRing)
    FAST RING *pRing;

    {
    int n = pRing->r_wrtidx - pRing->r_rdidx;

    if (n < 0)
	n += pRing->r_size;

    return (n);
    }
/********************************************************************************
* vxTasac - test-and-set-and-check
*
* This routine provides quick and dirty test and set when hardware isn't up to
* the tas.
*
* RETURNS:
*   TRUE if value had been not set (but now is),
*   FALSE if value was already set.
*/

LOCAL BOOL vxTasac (address)
    FAST char *address;         /* address to be tested */
 
    {
    FAST int tasWait = 10;
    FAST int count;
    FAST int value = 0x80 + sysProcNumGet ();
    int oldLevel = intLock ();
 
    /* check for lock available */
 
    if (*address != 0)
        {
        intUnlock (oldLevel);
        return (FALSE);
        }
 
    /* lock available - take it */
 
    *address = value;

    /* check that we got it and nobody stepped on it, several times */

    for (count = 0; count < tasWait; count++)
        {
        if (* (unsigned char *) address != value)
            {
            intUnlock (oldLevel);
            return (FALSE);
            }
        }    

    intUnlock (oldLevel);
    return (TRUE);
    }
/*******************************************************************************
*
* bpShow - show info about backplane network
*
* This routine shows information about the different CPUs configured
* in the backplane network.  It also prints error statistics (and zeros).
*
* .CS
*  -> bpShow
*  Anchor at 0x800000
*  heartbeat = 705, header at 0x800010, free pkts = 237.
*
*  cpu int type    arg1       arg2       arg3    queued pkts rd index
*  --- -------- ---------- ---------- ---------- ----------- --------
*   0  poll            0x0        0x0        0x0       0        27
*   1  poll            0x0        0x0        0x0       0        11 
*   2  bus-int         0x3       0xc9        0x0       0        9
*   3  mbox-2         0x2d     0x8000        0x0       0        1
*
*  input packets = 192     output packets = 164
*  output errors = 0       collisions = 0
*  value = 1 = 0x1
* .CE
*/

VOID bpShow (bpName, zero)
    char *bpName;	/* backplane interface name (NULL == "bp0") */
    BOOL zero;		/* TRUE = zap totals */

    {
    FAST BP_SOFTC *bs;
    FAST BP_ANCHOR *pAnchor;
    FAST BP_HEADER *pHdr;
    FAST CPU_DESC *pCpu;
    RING *pRing;
    FAST int cpuNum;

    /* use "bp0" if no interface name specified */

    if (bpName == NULL)
	bpName = "bp0";

    /* get pointer to interface descriptor */

    if ((bs = (BP_SOFTC *) ifunit (bpName)) == NULL)
	{
	printf ("interface %s not attached.\n", bpName);
	return;
	}

    /* check validity of anchor */

    pAnchor = bs->bs_pAnchor;

    printf ("Anchor at 0x%x", pAnchor);
#ifdef	UNIX
    if (peek ((short *)pAnchor) == -1)
#else
    if (vxMemProbe ((char *) pAnchor, READ, 4, (char *) &bpBitBucket) != OK)
#endif	UNIX
	{
	printf (", UNREADABLE\n");
	return;
	}

    if (pAnchor->an_ready != BP_READY)
	{
	if (pAnchor->an_ready == BP_READY_SOFT_TAS)
	    printf (", SOFT TAS\n");
	else
	    {
	    printf (", UNINITIALIZED\n");
	    return;
	    }
	}

    if (bpAlive (bs) == ERROR)
	{
	printf (", NO HEARTBEAT\n");
	return;
	}

    
    /* anchor valid - print out anchor and header info */

    pHdr = ItoKval (((BP_ANCHOR *)pAnchor)->an_pHdr, BP_HEADER *, pAnchor);

    printf (" heartbeat = %d, header at 0x%x, free pkts = %d.\n\n",
	    pAnchor->an_heartbeat, pHdr, ringcount (bs->bs_pFreeRing));

    
    /* print out info for each CPU */

    printf (
	"cpu int type    arg1       arg2       arg3    queued pkts rd index\n");
    printf (
	"--- -------- ---------- ---------- ---------- ----------- --------\n");

    for (cpuNum = 0; cpuNum < MAXCPU; ++cpuNum)
	{
	pCpu = &pHdr->hdr_cpuDesc[cpuNum];

	if (pCpu->cd_active)
	    {
	    printf ("%2d  ", cpuNum);

	    switch (pCpu->cd_intType)
		{
		case BP_INT_NONE:	printf ("poll    "); break;
		case BP_INT_MAILBOX_1:	printf ("mbox-1  "); break;
		case BP_INT_MAILBOX_2:	printf ("mbox-2  "); break;
		case BP_INT_MAILBOX_4:	printf ("mbox-4  "); break;
		case BP_INT_MAILBOX_R1:	printf ("mbox-1r "); break;
		case BP_INT_MAILBOX_R2:	printf ("mbox-2r "); break;
		case BP_INT_MAILBOX_R4:	printf ("mbox-4r "); break;
		case BP_INT_BUS:	printf ("bus-int "); break;
		default:		printf ("%#8x",pCpu->cd_intType); break;
		}

	    printf (" %#10x %#10x %#10x",
		    pCpu->cd_intArg1, pCpu->cd_intArg2, pCpu->cd_intArg3);

	    pRing = ItoKval (pCpu->cd_pInputRing, RING *, pAnchor);
	    printf ("     %3d       %3d\n", ringcount (pRing), pRing->r_rdidx);
	    }
	}

    printf ("\ninput packets = %d\t", bs->bs_if.if_ipackets);
    if (zero)
	bs->bs_if.if_ipackets = 0;

    printf ("output packets = %d\n", bs->bs_if.if_opackets);
    if (zero)
	bs->bs_if.if_opackets = 0;

    printf ("output errors = %d\t", bs->bs_if.if_oerrors);
    if (zero)
	bs->bs_if.if_oerrors = 0;

    printf ("collisions = %d\n", bs->bs_if.if_collisions);
    if (zero)
	bs->bs_if.if_collisions = 0;
    }


/* XXX "generic" SunOS network calls that will eventually live in etherLib.c */

#ifndef	UNIX
/*******************************************************************************
*
* check_trailer -
*/

BPLOCAL VOID check_trailer (eh, pData, pLen, pOff, ifp)
    FAST struct ether_header *eh;
    FAST unsigned char *pData;
    int *pLen;
    int *pOff;
    struct ifnet *ifp;

    {
    eh->ether_type = ntohs((u_short)eh->ether_type);

    if ((eh->ether_type >= ETHERTYPE_TRAIL)  &&
	(eh->ether_type < ETHERTYPE_TRAIL+ETHERTYPE_NTRAILER))
	{
	int resid;

	*pOff = (eh->ether_type - ETHERTYPE_TRAIL) * 512;


	if (*pOff >= ifp->if_mtu)
	    return;		/* sanity */

	eh->ether_type = ntohs (*(u_short *) (pData + *pOff));
	resid          = ntohs (*(u_short *) (pData + *pOff + 2));

	if ((*pOff + resid) > *pLen)
	    return;		/* sanity */

	*pLen = *pOff + resid;
	}
    else
	*pOff = 0;
    }
/*******************************************************************************
*
* copy_to_mbufs - 
*
* RETURNS: first mbuf or 0 if error
*/

BPLOCAL struct mbuf *copy_to_mbufs (buf0, totlen, off0, ifp)
    FAST unsigned char *buf0;
    int totlen;
    int off0;
    struct ifnet *ifp;

    {
    FAST struct mbuf *m;
    FAST unsigned char *cp = buf0;
    FAST int off     = off0;
    struct mbuf *top = 0;
    struct mbuf **mp = &top;
    int len;

    while (totlen > 0)
	{
	MGET (m, M_DONTWAIT, MT_DATA);

	if (m == 0)
	    {
	    m_freem (top);
	    return ((struct mbuf *)0);
	    }

	if (off)
	    {
	    len = totlen - off;
	    cp = buf0 + off;
	    }
	else
	    len = totlen;

	m->m_off = MMINOFF;

	if (ifp)
	    {
	    /* Leave room for ifp. */
	    m->m_len = MIN (MLEN - sizeof (ifp), len);
	    m->m_off += sizeof (ifp);
	    }
	else 
	    m->m_len = MIN (MLEN, len);

	bcopy ((caddr_t) cp, (caddr_t) mtod (m, u_char *), m->m_len);
	cp += m->m_len;
	*mp = m;
	mp = &m->m_next;

	if (off)
	    {
	    /* sort of an ALGOL-W style for statement... */
	    off += m->m_len;
	    if (off == totlen)
		{
		cp = buf0;
		off = 0;
		totlen = off0;
		}
	    }
	else
	    totlen -= m->m_len;

	if (ifp)
	    {
	    /* Prepend interface pointer to first mbuf. */
	    m->m_len += sizeof (ifp);
	    m->m_off -= sizeof (ifp);
	    *(mtod (m, struct ifnet **)) = ifp;
	    ifp = (struct ifnet *)0;
	    }
	}

    /*
     * Off is nonzero if packet has trailing header then force this header
     * information to be at the front, but we still have to drop
     * the type and length which are at the front of any trailer data.
     */

    if (off0)
        {   
        struct ifnet *ifp;

        ifp = *(mtod (top, struct ifnet **));
        top->m_off += 2 * sizeof (u_short);
        top->m_len -= 2 * sizeof (u_short);
        *(mtod (top, struct ifnet **)) = ifp;
        }

    return (top);
    }
#if	FALSE
/*******************************************************************************
*
* copy_from_mbufs - 
*/

BPLOCAL struct mbuf *copy_from_mbufs (buf0, mp)
    FAST unsigned char *buf0;
    FAST struct mbuf *m;

    {
    logMsg ("copy_from_mbufs: not implemented\n");
    }
#endif	FALSE
/*******************************************************************************
*
* do_protocol -
*
* ARGSUSED
*/

BPLOCAL VOID do_protocol (eh, m, pArpcom, len)
    FAST struct ether_header *eh;
    FAST struct mbuf *m;
    struct arpcom *pArpcom;
    int len;

    {
    FAST struct ifqueue *inq;
    FAST int spl;

    switch (eh->ether_type)
	{
#ifdef	INET
	case ETHERTYPE_IP:
	    netJobAdd (ipintr, 0);
	    inq = &ipintrq;
	    break;

	case ETHERTYPE_ARP:
	    arpinput (pArpcom, m);
	    return;
#endif	INET
#ifdef	NS
        case ETHERTYPE_NS:
            schednetisr (NETISR_NS);
            inq = &nsintrq;
            break;

#endif	NS
#ifdef  RFS
          case ETHERTYPE_RFS:
            schednetisr (NETISR_RFS);
            inq = &rfsintrq;
            break;
#endif	RFS

	default:
	    m_freem (m);
	    return;
	}

    spl = splnet ();

    if (IF_QFULL (inq))
	{
	IF_DROP (inq);
	m_freem (m);
	splx (spl);
	return;
	}

    IF_ENQUEUE (inq, m);
    splx (spl);
    }
/*******************************************************************************
*
* ether_output -
*
* RETURNS: 0 if successful, errno otherwise (as per network requirements)
*
* ARGSUSED
*/

BPLOCAL int ether_output (ifp, m0, dst, startRtn, pArpcom)
    struct ifnet *ifp;
    struct mbuf *m0;
    struct sockaddr *dst;
    FUNCPTR startRtn;		/* XXX ignored? */ 
    struct arpcom *pArpcom;

    {
    FAST BP_SOFTC *bs = (BP_SOFTC *) ifp;
    FAST struct mbuf *m = m0;
    FAST struct ether_header *bp;
    FAST int off;
    int type, error;
    u_char edst[6];
    int usetrailers;
    struct in_addr idst;
    int spl;

    if ((ifp->if_flags & IFF_UP) == 0)	/* not ready yet */
	{
	m_freem (m0);
	return (0);
	}

    switch (dst->sa_family)
	{
#ifdef	INET
	case AF_INET:
	    idst = ((struct sockaddr_in *) dst)->sin_addr;
	    if (!arpresolve (pArpcom, m, &idst, edst, &usetrailers))
		return (0);     /* if not yet resolved */

	    off = ((u_short) mtod (m, struct ip *)->ip_len) - m->m_len;
	    if (usetrailers && off > 0 && (off & 0x1ff) == 0 &&
		m->m_off >= MMINOFF + 2 * sizeof (u_short))
		{
		type = ETHERTYPE_TRAIL + (off>>9);
		m->m_off -= 2 * sizeof (u_short);
		m->m_len += 2 * sizeof (u_short);
		*mtod (m, u_short *) = ETHERTYPE_IP;
		*(mtod (m, u_short *) + 1) = m->m_len;
		goto gottrailertype;
		}
	    type = ETHERTYPE_IP;
	    off = 0;
	    goto gottype;
#endif	INET
	case AF_IMPLINK:
	    /* should do some ARP here? */
	    bp = mtod (m, struct ether_header *);
	    goto gotheader;

	case AF_UNSPEC:
	    bp = (struct ether_header *) dst->sa_data;
	    bcopy ((char *) bp->ether_dhost, (char *) edst, 6);
	    type = bp->ether_type;
	    goto gottype;

	default:
	    error = EAFNOSUPPORT;
	    goto bad;
	}

gottrailertype:
    /*
     * Packet to be sent as trailer: move first packet
     * (control information) to end of chain.
     */
    while (m->m_next)
	m = m->m_next;
    m->m_next = m0;
    m = m0->m_next;
    m0->m_next = 0;
    m0 = m;

gottype:
    /* Add local net header.  If no space in first mbuf, allocate another.  */
    if (m->m_off > MMAXOFF || MMINOFF + sizeof (struct ether_header) > m->m_off)
	{
	m = m_get (M_DONTWAIT, MT_DATA);
	if (m == 0)
	    {
	    error = ENOBUFS;
	    goto bad;
	    }
	m->m_next = m0;
	m->m_off = MMINOFF;
	m->m_len = sizeof (struct ether_header);
	}
    else
	{
	m->m_off -= sizeof (struct ether_header);
	m->m_len += sizeof (struct ether_header);
	}

    bp = mtod (m, struct ether_header *);
    bcopy ((char *) edst, (char *) bp->ether_dhost, 6);
    bp->ether_type = type;

gotheader:

    bcopy ((char *) ((struct arpcom *)ifp)->ac_enaddr,
	   (char *) bp->ether_shost, 6);

    /* Queue message on interface if possible */

    spl = splnet ();

    /* XXX - start output after queuing? */
    if (bpPut (bs, m) != OK)
	{
	error = ENOBUFS;
	m0 = m;
	splx (spl);
	goto bad;
	}

    ifp->if_opackets++;

    splx (spl);
    return (0);

bad:
    m_freem (m0);
    return (error);
    }
/********************************************************************************
* set_if_addr -
*
* RETURNS: 0 or appropriate errno
*
* ARGSUSED2
*/
 
BPLOCAL int set_if_addr (ifp, data, enaddr)
    struct ifnet *ifp;  /* interface */
    char *data;         /* ? ifa */
    u_char *enaddr;       /* ethernet address */
 
    {
    int error = 0;
    FAST struct ifaddr *ifa = (struct ifaddr *) data;
 
    ifp->if_flags |= IFF_UP;
 
    if ((error = (*ifp->if_init) (ifp->if_unit)) != 0)
        return (error);

    switch (ifa->ifa_addr.sa_family)
        {
#ifdef  INET
        case AF_INET:
            ((struct arpcom *) ifp)->ac_ipaddr = IA_SIN (ifa)->sin_addr;
            arpwhohas ((struct arpcom *) ifp, &IA_SIN (ifa)->sin_addr);
            break;
#endif  INET
#ifdef  NS
        case AF_NS:
            {
            struct ns_addr *ina = &(IA_SNS (ifa)->sns_addr);
 
            if (ns_nullhost (*ina))
                ina->x_host = *(union ns_host *)(xs->xs_addr);
            else
                ex_setaddr (ina->x_host.c_host, ifp->if_unit);
            break;
            }
#endif  NS
	default:
	    dbgprintf ("bp: set_if_addr unknown family 0x%x\n",
			ifa->ifa_addr.sa_family);
	    break;
        }

    return (0);
    }
/********************************************************************************
* ether_attach -
*/
 
BPLOCAL VOID ether_attach (ifp, unit, name, initRtn, ioctlRtn, outputRtn, resetRtn)
    struct ifnet *ifp;  /* interface */
    int unit;
    char *name;
    FUNCPTR initRtn;
    FUNCPTR ioctlRtn;
    FUNCPTR outputRtn;
    FUNCPTR resetRtn;
 
    {
    char *uname = malloc ((unsigned)(strlen (name) + 2)); /* unit # + EOS */
 
    (void)sprintf (uname, "%s%d", name, unit);
 
    bzero ((char *) ifp, sizeof (*ifp));
 
    ifp->if_unit   = unit;
    ifp->if_name   = uname;
    ifp->if_mtu    = ETHERMTU;
    ifp->if_init   = initRtn;
    ifp->if_ioctl  = ioctlRtn;
    ifp->if_output = outputRtn;
    ifp->if_reset  = resetRtn;
    ifp->if_flags  = IFF_BROADCAST;
 
    if_attach (ifp);
    }

#endif	!UNIX
