/* ts.c - tape driver */

static char *copyright = "Copyright 1988, Integrated Solutions, Inc.";

/* 
modification history
--------------------
*/

/*
DESCRIPTION
*/

/*#define DEBUG	/**/

#include "vxWorks.h"
#include "types.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "wdLib.h"
#include "semLib.h"
#include "ts.h"
#include "tsreg.h"
#include "mtio.h"

/*
 * per-drive status structure.
 */
typedef struct	ts_softc {
	DEV_HDR	sc_dev_hdr;
	short	sc_unit;
	short	sc_flags;	/* see list below */
	int	sc_task;	/* who opened it last */
	int 	sc_num;		/* number of tasks with fd's here */
	SEMAPHORE sc_sem;
	SEMAPHORE sc_int_box;	/* semaphore for interrupts */
	struct	ts_cmd *sc_cmd;	/* the command packet */
	struct  ts_sts *sc_sts_wind;	/* vdma window to status packet */
	struct	ts_sts sc_sts;	/* status packet, for returned status */
	struct	ts_char sc_char; /* characteristics packet */
} TS_DEV;

TS_DEV ts_softc[NTS];

/*
 * stuff in sc_flags
 */
#define SC_FREAD	0x01	/* opened in READ mode */
#define SC_FWRITE	0x02	/* opened in WRITE mode */
#define	SC_FNOREWIND	0x04	/* do not rewind on close */
#define	SC_FSWAPBYTE	0x08	/* swap bytes at controller level */
#define SC_FEXISTS	0x10	/* device physically exists */
#define SC_FCREATED	0x20	/* device has been created */
#define SC_FEXCLUDE	0x40	/* only one open to file at a time */
/* for tsOpen and tsClose... */
#define SC_FOPEN	(SC_FREAD|SC_FWRITE)
#define SC_FCLOSED	(SC_FEXISTS|SC_FCREATED)

typedef struct	ts_iobuf {
	short	iob_unit;	/* unit number to get back to TS_DEV */
	char	*iob_addr;	/* data transfer address */
	u_short	iob_bcount;	/* transfer/operation count */
	short	iob_command;	/* command to execute */
	short	iob_errcnt;	/* error count for retries */
	short	iob_repcnt;	/* repetition count */
	short	iob_resid;	/* residual byte/operation count */
	short	iob_flags;
	short	iob_error;
	long	iob_wind;	/* vdma window address for transfer */
} TS_IOB;

TS_IOB	ts_iob[NTS];

/*
 * stuff in iob_flags
 */
#define	IOB_READ	0x01	/* read type command */
#define	IOB_WRITE	0x02	/* write type command */
#define	IOB_ERROR	0x80	/* some error occured */

/*
 * convenient macros for referencing longs as two shorts
 */
#define lobyte(X)	(((unsigned char *)&X)[1])
#define hibyte(X)	(((unsigned char *)&X)[0])
#define loword(X)	(((u_short *)&(X))[1])
#define hiword(X)	(((u_short *)&(X))[0])


/*
 * tc50 bus address and vectors
 */

extern char vme_window_std[];

extern char vme_std[];
u_short *TSstd[] = {
		(u_short *)&vme_std[0xfff550],
		(u_short *)&vme_std[0xfff520],
		(u_short *)&vme_std[0xfff540],
		 0 };
struct {
    struct tsdevice	*addr;
    int			vec;
} tsaddrlist[NTS] = {
    {(struct tsdevice *) 0, 4*0x94},  /* vector numbers must match the */
    {(struct tsdevice *) 0, 4*0x98},  /* boards in the same order as devaddr.c */
};

int       tsNum;		/* driver number assigned to this driver */

/* forward declarations */

int       tsOpen();
int       tsClose();
int       tsRead();
int       tsWrite();
STATUS    tsIoctl();
VOID      tsInterrupt();

#ifdef	DEBUG
char	*comstr();
int	tsdebug=1;
#endif	DEBUG

STATUS
tsDrv()
{
	register struct tsdevice *tsaddr;
	register TS_DEV *sc;
	register TS_IOB	*iob;
	int             result, status=ERROR;
	short		dummy, unit;
	char		found[NTS];
	char		name[6];
	extern int	sysProcNum;
	register int	count;

	/* initialize address list */
	for (count = 0; count < NTS && TSstd[count] != 0; count++) 
	    tsaddrlist[count].addr = (struct tsdevice *) TSstd[count];
	    
	if ( sysProcNum != 0 )
		return(NULL);
	for (unit=0; unit<NTS; ++unit) {
		/*
		 * initialize the ts_dev entry
		 */
		tsaddr = tsaddrlist[unit].addr;
		sc = &ts_softc[unit];
		iob = &ts_iob[unit];
		sc->sc_unit = iob->iob_unit = unit;
		sc->sc_flags = iob->iob_flags = 0;
		sc->sc_num = 0;
		semInit (&sc->sc_int_box);
		dummy = 0xff;

		/* test to see if the board is there */
		result = vxMemProbe ((char *)tsaddr + 2, READ, 2, &dummy);

		if( result != OK) {
			found[unit] = FALSE;
			continue;
		}
		if ( result == OK ) {
			found[unit] = TRUE;
			tsaddr->tssr = TSSR_RESET;
			if (tswait(tsaddr)) {
			    printf("tsDrv: unit %d failed RESET\n", unit);
			    continue;
			}

			/* allocate a vdma window for this unit's status
			 * buffer
			 */
	    sc->sc_sts_wind =
		(struct ts_sts *)sysSetVdma(&sc->sc_sts,sizeof(struct ts_sts));
	    if (sc->sc_sts_wind < 0) {
		printf ("ts: could not allocate vdma window for ts_sts\n");
		continue;
	    }

			if (tssetchar(tsaddr, unit, 0)) {
			    printf("tsDrv: unit %d failed setchar.\n", unit);
			    continue;
			}

			sc->sc_flags |= SC_FEXISTS;
			semInit(&sc->sc_sem);
			semGive(&sc->sc_sem);
			intConnect(tsaddrlist[unit].vec, tsInterrupt, unit);
			status = OK;	/* at least one drive is there */
			printf("ts%d at (0x%06x/0x%02x)\n",
			    unit,
			    tsaddrlist[unit].addr, tsaddrlist[unit].vec/4);
		}
	}
	if(status==ERROR)
		return;
	tsNum = iosDrvInstall(tsOpen, (FUNCPTR) NULL, tsOpen, tsClose,
			      tsRead, tsWrite, tsIoctl);
	for (unit=0; unit<NTS; ++unit) {
		if(found[unit] == FALSE)
			continue;
		sprintf(name,"/ts%d",unit);
		tsDevCreate(name, unit);
	}
	return (status);
}

STATUS
tsDevCreate(name, unit)
	char           *name;	/* Name to use for this device */
	int             unit;	/* Physical unit for this device (0 or 1) */
{
	register TS_DEV	*sc = &ts_softc[unit];	

	if ( sc->sc_flags & SC_FEXISTS ) {
		sc->sc_flags |= SC_FCREATED;
		return (iosDevAdd((DEV_HDR *) &ts_softc[unit], name, tsNum));
	}
	else
		return(ERROR);
}

/*
 * tsOpen() -
 */

int
tsOpen(sc, name, mode)
	TS_DEV         *sc;
	char           *name;
	int             mode;
{
	register int unit = sc->sc_unit & 0xffff;
	int	dummy;

	semTake(&sc->sc_sem);
	if (unit>=NTS || (sc->sc_flags & SC_FCREATED) == 0) {
		logMsg("ts%d open failed: does not exist\n", unit);
		goto open_err;
	}
	if (sc->sc_num) {
	    /* if SC_FEXCLUDE mode is set, return ERROR
	     * if the file is already open to another task
	     */
	    if (sc->sc_flags & SC_FEXCLUDE) {
		logMsg("ts%d open failed: last task to open was %d\n",
		    unit, sc->sc_task);
		goto open_err;
	    }
	    /* if EXCLUDE is not set, set some parms and
	     * return OK
	     */
	    sc->sc_num++;
	    if (mode != READ && (sc->sc_sts.s_xs0&TS_WLK)) {
		logMsg("ts%d: no write ring\n", unit);
		goto open_err;
	    }
	    sc->sc_task = taskIdSelf();
	    sc->sc_flags |= (mode == READ ? SC_FREAD : SC_FWRITE);
	    semGive(&sc->sc_sem);
	    return ((int) sc);
	}

	if (tsinit(unit)) {
		logMsg("ts%d open failed: tsinit()\n");
		goto open_err;
	}
	tscommand(sc, TS_SENSE, 1, 1);
	if ((sc->sc_sts.s_xs0&TS_ONL) == 0) {
		logMsg("ts%d: not online\n", unit);
		goto open_err;
	}
	if (mode != READ && (sc->sc_sts.s_xs0&TS_WLK)) {
		logMsg("ts%d: no write ring\n", unit);
		goto open_err;
	}
	sc->sc_num++;
	sc->sc_task = taskIdSelf();
	switch (mode) {
	case READ : sc->sc_flags |= SC_FREAD; break;
	case WRITE : sc->sc_flags |= SC_FWRITE; break;
	case UPDATE :
	    sc->sc_flags |= (SC_FWRITE | SC_FREAD); break;
	}
#ifdef DEBUG
	if (tsdebug) logMsg ("tsopen: flags = 0x%08x\n", sc->sc_flags);
#endif DEBUG
	semGive(&sc->sc_sem);
	return ((int) sc);

open_err:
	semGive(&sc->sc_sem);
	return (ERROR);
}

/*
 * tsClose() -
 *
 * If tape was open for writing or last operation was a write, then 
 * write two EOF's and backspace over the last one.  Unless no-rewind
 * ioctl has been issued, rewind the tape.
 */

STATUS
tsClose(sc)
	TS_DEV         *sc;
{
	semTake(&sc->sc_sem);
	if (!(sc->sc_flags & SC_FOPEN)) {
		logMsg("ERROR -  ts device unit %d not open\n", sc->sc_unit);
		semGive(&sc->sc_sem);
		return (ERROR);
	}
	/* if more than one task has the device open,
	 * don't bother writing EOT or rewinding
	 */
	if (sc->sc_num == 1) {
	    if (sc->sc_flags & SC_FWRITE) {
		    tscommand(sc, TS_WEOF, 1, 1);
		    tscommand(sc, TS_WEOF, 1, 1);
		    tscommand(sc, TS_SREV, 1, 1);
	    }

	    if ((sc->sc_flags & SC_FNOREWIND) == 0) {
		    taskDelay(2 * sysClkRateGet());	/* bug in CDC drive */
		tscommand(sc, TS_REW, 1, 1);
	    }
	}
	if (--sc->sc_num == 0)
	    sc->sc_flags = SC_FCLOSED;
	if (sc->sc_num < 0)
	    logMsg ("File descriptor count error\n");
	semGive(&sc->sc_sem);
	return(OK);
}

int
tsRead(sc, buffer, nbytes)
	TS_DEV         *sc;
	char           *buffer;
	int             nbytes;

{
	register TS_IOB *iob;
	register int    unit;
	int             resid;

#ifdef	DEBUG
	if(tsdebug)printf("tsRead nbytes=%d\n", nbytes);
	if (tsdebug) logMsg ("tsread: flags = 0x%08x\n", sc->sc_flags);
#endif	DEBUG
	unit = sc->sc_unit;
	iob = &ts_iob[unit];

	semTake(&sc->sc_sem);
	if ((sc->sc_flags & SC_FREAD) == 0) {
#ifdef	DEBUG
		if (tsdebug) logMsg("tsRead: not opened for reading\n");
#endif	DEBUG
		semGive(&sc->sc_sem);
		return (ERROR);
	}

	iob->iob_addr = buffer;
	iob->iob_bcount = nbytes;
	iob->iob_errcnt = 0;
	iob->iob_command = TS_RCOM;
	iob->iob_flags &= ~IOB_ERROR;
	iob->iob_flags |= IOB_READ;
	tsstart(iob, 1);
	iob->iob_flags &= ~IOB_READ;

	/* we need to flush the cache for the 
	 * status buffer and for the data
	 * buffer, so just flush the whole thing
	sysDataCacheFlushEntry (buffer, nbytes); 
	 */
	sysDataCacheFlush ();
	if (iob->iob_flags & IOB_ERROR) {
		logMsg("tsRead: read error\n");
		semGive(&sc->sc_sem);
		return (ERROR);
	} else {
		int nread = nbytes - iob->iob_resid;
		semGive(&sc->sc_sem);
		return (nread);
	}
}

int
tsWrite(sc, buffer, nbytes)
	TS_DEV         *sc;
	char           *buffer;
	int             nbytes;

{
	register TS_IOB *iob;
	register int    unit;
	int             resid;

#ifdef	DEBUG
	if(tsdebug)printf("tsWrite nbytes=%d\n", nbytes);
	if (tsdebug) logMsg ("tswrite: flags = 0x%08x\n", sc->sc_flags);
#endif	DEBUG
	unit = sc->sc_unit;
	iob = &ts_iob[unit];

	semTake(&sc->sc_sem);
	if ((sc->sc_flags & SC_FWRITE) == 0) {
		semGive(&sc->sc_sem);
#ifdef 	DEBUG
		if (tsdebug)logMsg ("No write permission to ts%d\n", unit);
#endif 	DEBUG
		return (ERROR);
	}

	iob->iob_addr = buffer;
	iob->iob_bcount = nbytes;
	iob->iob_errcnt = 0;
	iob->iob_command = TS_WCOM;
	iob->iob_flags &= ~IOB_ERROR;
	iob->iob_flags |= IOB_WRITE;
	tsstart(iob, 1);
	iob->iob_flags &= ~IOB_WRITE;

	if (iob->iob_flags & IOB_ERROR) {
		logMsg("tsWrite: write error\n");
		semGive(&sc->sc_sem);
		return (ERROR);
	} else {
		int nwrite = nbytes - iob->iob_resid;
		semGive(&sc->sc_sem);
		return (nwrite);
	}
}

STATUS
tsIoctl(sc, request, arg)
	TS_DEV         *sc;		/* device to control */
	int             request;	/* request code */
	u_long             arg;		/* some argument */
{
	register TS_IOB	*iob;
	register int	unit;
	int		callcount;
	int		fcount;
	struct mtop	*op;
	/*
	 * We depend of the values and order of the MT codes here 
	 */
	static          tsops[] = {
		TS_WEOF, TS_SFORWF, TS_SREVF, TS_SFORW, TS_SREV,
		TS_REW, TS_OFFL, TS_SENSE, TS_RET, NULL, TS_ERASE
	};

	unit = sc->sc_unit;
	iob = &ts_iob[unit];

	semTake(&sc->sc_sem);
	if (sc->sc_flags & SC_FOPEN == 0) {
		semGive(&sc->sc_sem);
		return (ERROR);
	}

	iob->iob_errcnt = 0;

	switch (request) {
	case MTIOCTOP:
	    op = (struct mtop *)arg;
	    switch (op->mt_op) {
	    case MTWEOF:
	    case MTERASE:
		    callcount = 1;
		    fcount = 1;
		    break;
	    case MTFSF:
	    case MTBSF:
	    case MTFSR:
	    case MTBSR:
		    callcount = 1;
		    fcount = op->mt_count;
		    break;
	    case MTREW:
		    /*
		     * BUG in CDC streaming 1/2 inch needs a delay here
		     */
		    taskDelay(2);
	    case MTOFFL:
	    case MTNOP:
	    case MTRET:
		    callcount = 1;
		    fcount = 1;
		    break;
	    case MTRST:
		    tsinit(unit);	/* reset characteristics */
		    semGive(&sc->sc_sem);
		    return (OK);
	    case MTEXCLUDE :
		    if (op->mt_count & 1)
			sc->sc_flags |= SC_FEXCLUDE;
		    else
			sc->sc_flags &= ~SC_FEXCLUDE;
		    semGive(&sc->sc_sem);
		    return (OK);
	    case MTSWAPBYTE :
		    if (op->mt_count & 1)
			sc->sc_flags &= ~SC_FSWAPBYTE;
		    else
			sc->sc_flags |= SC_FSWAPBYTE;
		    semGive(&sc->sc_sem);
		    return (OK);
	    case MTNOREWIND :
		    if (op->mt_count & 1)
			sc->sc_flags |= SC_FNOREWIND;
		    else
			sc->sc_flags &= ~SC_FNOREWIND;
		    semGive(&sc->sc_sem);
		    return (OK);
	    default:
		    semGive(&sc->sc_sem);
#ifdef DEBUG
		    if (tsdebug) logMsg ("Bad ioctl argument\n");
#endif DEBUG
		    return (ERROR);
	    }

	    if (callcount <= 0 || fcount <= 0) {
#ifdef DEBUG
		    if (tsdebug) logMsg ("Bad count number\n");
#endif DEBUG
		    semGive(&sc->sc_sem);
		    return (ERROR);
	    }
	    while (--callcount >= 0) {
		    tscommand(sc, tsops[op->mt_op], fcount, 1);
		    if (iob->iob_flags & IOB_ERROR) {
			    semGive(&sc->sc_sem);
			    return (ERROR);
		    }

		    /* if the command was FSR or BSR, then we want to return
		     * OK if iob_resid = 0 so the interface looks like Unix.
		     */
		    if ((op->mt_op == MTFSR || op->mt_op == MTBSR)) {
			    semGive(&sc->sc_sem);
			    return (iob->iob_resid ? ERROR : OK);
		    }
	    }

	    /* If the command was backspace file, then we just went
	     * back past the tape marker, and we have to seek forward
	     * past it (unless we hit BOT).
	     */
	    if (op->mt_op == MTBSF) {
		if (! (sc->sc_sts.s_xs0 & TS_BOT)) {
		    tscommand (sc, TS_SFORWF, 1, 1);
		    if (iob->iob_flags & IOB_ERROR) {
			    semGive(&sc->sc_sem);
			    return (ERROR);
		    }
		}
	    }

	    semGive(&sc->sc_sem);
	    return (OK);
	    break;
	
	case MTIOCGET:
	case MTIOCIEOT:
	case MTIOCEEOT:
	default:
#ifdef DEBUG
	    if (tsdebug) logMsg ("Bad ioctl request %d\n", request);
#endif DEBUG
	    semGive(&sc->sc_sem);
	    return (OK);
	    break;
	}
}

/*
 * tsInterrupt -
 */

VOID
tsInterrupt(unit)
	FAST int        unit;
{
	register TS_DEV		*sc;
	register TS_IOB		*iob;
	register struct tsdevice *tsaddr;

	sc = &ts_softc[unit];
	iob = &ts_iob[unit];
	tsaddr = tsaddrlist[unit].addr;

	/*
	 * Check for errors.
	 */
	if (tsaddr->tssr&TS_SC) {
#ifdef	DEBUG
if(tsdebug){
logMsg("ts%d: tssr=0x%04x\n", unit, tsaddr->tssr);
logMsg("rbpcr=%d\nxs0=0x%04x\nxs1=0x%04x\nxs2=0x%04x\nxs3=0x%04x\nxs4=0x%04x\n",
sc->sc_sts.s_rbpcr,
sc->sc_sts.s_xs0,
sc->sc_sts.s_xs1,
sc->sc_sts.s_xs2,
sc->sc_sts.s_xs3,
sc->sc_sts.s_xs4);
}
#endif	DEBUG
		switch (tsaddr->tssr & TS_TC) {
		case TS_SUCC:		/* 000 - success termination */
			goto opdone;

		case TS_UNREC:		/* 110 - unrecoverable */
		case TS_FATAL:		/* 111 - fatal error */
		case TS_ATTN:		/* 001 - attention(shouldn't happen) */
		case TS_RECNM:		/* 101 - recoverable, no motion */
			goto errdone;

		case TS_ALERT:		/* 010 -tape status alert */
			/* If searching forward or backward records, then
			 * this is not an error 
			 */
			if ((iob->iob_command == TS_SFORW) 
			    || (iob->iob_command == TS_SREV))
				    goto opdone;

			/*
			 * If we hit the end of the tape file on anything
			 * other than a read, error.  For read, it is just
			 * a zero byte count.
			 */
			if (sc->sc_sts.s_xs0 & (TS_TMK|TS_EOT)) {
				if (iob->iob_command != TS_RCOM) {
					goto errdone;
				}
			}
			if (sc->sc_sts.s_xs0 & TS_EOT) {
				logMsg("ts%d: unexpected physical EOT\n",unit);
				goto errdone;
			}
			/*
			 * If reading and the record was too long
			 * or too short, then we don't consider this an error.
			 */
			if (iob->iob_command == TS_RCOM) {
				if (sc->sc_sts.s_xs0&(TS_RLS|TS_RLL)) {
					goto opdone;
				}
			}

			/* Anything else generates an ERROR 
			 */
			goto errdone;

		case TS_RECOV:		/* 100 - recoverable, tape moved */
			/* if the command was TS_SREVF and rbpcr == 0, then
			 * no error has occurred but we have hit BOT.
			 */
			if ((iob->iob_command == TS_SREVF) 
			    && (sc->sc_sts.s_rbpcr == 0))
			goto opdone;
			/*
			 * If this was an i/o operation retry up to 8 times.
			 */
			goto errdone;

		case TS_REJECT:		/* 011 - function reject */
			if (iob->iob_command == TS_WCOM
			    && sc->sc_sts.s_xs0 & TS_WLE)
				logMsg("ts%d: write locked\n",unit);
			else if ((sc->sc_sts.s_xs0 & TS_ONL) == 0)
				logMsg("ts%d: offline\n", unit);
			else
				logMsg("ts%d: function reject\n", unit);
			goto errdone;
		default:
			logMsg("ts%d: illegal termination code\n", unit);
			goto errdone;
		}
	}
	else
		goto opdone;

errdone:
	iob->iob_flags |= IOB_ERROR;
opdone:
	iob->iob_errcnt = 0;
	iob->iob_resid = sc->sc_sts.s_rbpcr;
#ifdef DEBUG
	if (tsdebug) logMsg ("rbpcr = %d\n", sc->sc_sts.s_rbpcr);
#endif DEBUG
	semGive(&sc->sc_int_box);
}

/*
 * tswait - wait for SUBSYSTEM READY
 */
tswait(tsaddr)
register struct tsdevice *tsaddr;
{
	u_short sreg;
	register int i, result;

	for (i = 8; i; i--) {
	    TS_DELAY(60);
	    result = vxMemProbe (&(tsaddr->tssr), READ, 2, &sreg);
	    if ((result == OK) && (sreg & TS_SSR))
		break;
	}

	if (i == 0) {
	    logMsg ("ts: timeout\n");
	    return (ERROR);
	}

#ifdef DEBUG
	if (tsdebug) {
	    if (sreg & TS_NXM) logMsg ("tswait: non-existent memory error\n");
	    if (sreg & TS_RMR) logMsg ("tswait: register mod refused\n");
	    if (sreg & TS_SC) logMsg ("tswait: special condition\n");
	}
#endif DEBUG

	return (OK);
}

/*
 * tssleep() - give up processor while waiting for SUBSYSTEM READY.
 */
#define MAXTIME	300
tssleep(tsaddr)
register struct tsdevice *tsaddr;
{
	register int i;
	register u_short addr;
	unsigned	dummy_box, dummy_msg;

	for ( i = 0 ; i < MAXTIME ; i++) {
		if ((tsaddr->tssr & TS_SSR) != 0)
			return (0);
		taskDelay(15);	/* sleep 1/4 second */
	}
	return (1);
}

/*
 * tssetchar - set characteristics
 */
tssetchar(tsaddr, unit, ie)
struct tsdevice *tsaddr;
{
	register TS_DEV *sc = &ts_softc[unit];
	int		i;
	extern char	ts_cmd_buf[];	/* in sysALib.s */
	int		ts_cmd_base = ((int)ts_cmd_buf+3)&~3;
	long		charwind;

	/* ts_cmd_base is allocated in sysALib.s
	 * In usrConfig, a window is allocated for the
	 * ts_cmd buffer
	 */
	sc->sc_cmd = (struct ts_cmd *)
		(ts_cmd_base + (sizeof (struct ts_cmd) * unit)); /* KLUDGE */
#ifdef	DEBUG
if(tsdebug)
	printf("tssetchar: sc->sc_cmd=0x%08x ie=%d\n", sc->sc_cmd, ie);
#endif	DEBUG
	i = (int)sc->sc_cmd-(int)vme_window_std;
	if (hiword(i) & ~0x03)
		panic("ts: command packet not in 18 bit land\n"); 
	if (i & 0x03)
		panic("ts: command packet not long aligned\n"); 

	sc->sc_char.char_ladr = loword(sc->sc_sts_wind);
	sc->sc_char.char_hadr = hiword(sc->sc_sts_wind);
	sc->sc_char.char_size = sizeof(struct ts_sts);
	
	sc->sc_char.char_mode = TS_ESS;
	sc->sc_char.ext_char_mode = 0 | TS_WRB  /* | TS_HSP | TS_RDB*/;
	if (ie)
		sc->sc_cmd->c_cmd = TS_IE | TS_ACK | TS_CVC | TS_SETCHR;
	else
		sc->sc_cmd->c_cmd = TS_ACK | TS_CVC | TS_SETCHR;

	charwind = sysSetVdma (&sc->sc_char, sizeof (struct ts_char));
	if (charwind < 0) {
	    printf ("ts: could not allocate vdma window for ts_char\n");
	    return (ERROR);
	}
	sc->sc_cmd->c_loba = loword(charwind);
	sc->sc_cmd->c_hiba = hiword(charwind);
	sc->sc_cmd->c_size = sizeof(struct ts_char);
	sysDataCacheFlush ();

	if (tswait (tsaddr)) {
	    logMsg ("tssetchar: Could not access TSSR\n");
	    sysGiveVdma (charwind);
	    return (ERROR);
	}

	TS_DELAY(30);
	tsaddr->tssr = ((((int)sc->sc_cmd-(int)vme_window_std) >> 16) & 0x3FFF);
	TS_DELAY(30);
	tsaddr->tsdb = (int)sc->sc_cmd & 0xFFFF;

	if (tswait(tsaddr)) {
#ifdef DEBUG
	    if (tsdebug) logMsg ("tssetchar: Could not send command address\n");
#endif DEBUG
	    sysGiveVdma (charwind);
	    return (ERROR);
	}

	TS_DELAY(30);
	if (tsaddr->tssr & TS_NBA) {
#ifdef DEBUG
	    if (tsdebug) logMsg ("tssetchar: returned NBA\n");
#endif DEBUG
	    sysGiveVdma (charwind);
	    return(ERROR);
	}
	/* give up the vdma window for the setchar and
	 * status buffers
	 */
	sysGiveVdma (charwind);
	return(0);
}

/*
 * Initialize the TS11.  Set up device characteristics.
 */
tsinit(unit)
register int unit;
{
	register struct ts_softc *sc = &ts_softc[unit];
	register struct tsdevice *tsaddr = tsaddrlist[unit].addr;
	register u_short status;
	int i;

	/*
	 * Now initialize the TS11 controller. Set the characteristics.
	 */
	if ((status = tsaddr->tssr) & (TS_NBA|TS_OFL))
		tsaddr->tssr = TSSR_RESET;	/* subsystem initialize */
#ifdef DEBUG
	if (tsdebug) logMsg ("tsinit: tssr = 0x%04x\n", status);
#endif DEBUG
	if (tssleep(tsaddr))
		return(1);
	if (tssetchar(tsaddr, unit, 0))
		return(1);
	return(0);
}

tscommand(sc, com, count, wait)
TS_DEV	*sc;
int com, count;
{
	register int	unit = sc->sc_unit;
	register TS_IOB *iob = &ts_iob[unit];
	int		dummy;

#ifdef	DEBUG
	if (tsdebug)
	logMsg("tscommand unit=%d com=%s count=%d\n", unit, comstr(com), count);
#endif	DEBUG
	iob->iob_repcnt = count;
	iob->iob_command = com;
	iob->iob_flags &= ~IOB_ERROR;

	tsstart(iob, 0);
	if (wait)
		semTake(&sc->sc_int_box);
}


tsstart(iob, wait)
register TS_IOB *iob;
{
	int			unit, dummy;
	register TS_DEV	*sc;
	register struct tsdevice *tsaddr;
	register struct ts_cmd *tc;
	int cmd;
	daddr_t blkno;

	unit = iob->iob_unit;
	sc = &ts_softc[unit];
	tsaddr = tsaddrlist[unit].addr;
	tc = sc->sc_cmd;

#ifdef	DEBUG
	if (tsdebug)
	logMsg("tsstart   unit=%d com=%s\n", unit, comstr(iob->iob_command));
#endif	DEBUG
	switch (iob->iob_command) {
		case TS_SENSE :
		case TS_REW :
		case TS_WEOF :
		case TS_ERASE :
		case TS_SREV :
		case TS_SFORW :
		case TS_SREVF :
		case TS_SFORWF :
		case TS_OFFL :
		case TS_RET:
			tc->c_repcnt = iob->iob_repcnt;
			tc->c_cmd = TS_ACK | TS_CVC | TS_IE | iob->iob_command;
			break;
		case TS_RCOM :
		case TS_WCOM :
			cmd = iob->iob_command;
			wait = 1;
			tc->c_cmd = TS_ACK | TS_CVC | TS_IE | cmd;
			if (sc->sc_flags & SC_FSWAPBYTE)
				tc->c_cmd |= TS_SWB;
			tc->c_size = iob->iob_bcount + (iob->iob_bcount & 1);

			iob->iob_wind = sysSetVdma (iob->iob_addr, tc->c_size);
			if (iob->iob_wind < 0) {
			    printf ("ts: could not allocate vdma window\n");
			    return (ERROR);
			}
			tc->c_loba = loword(iob->iob_wind);
			tc->c_hiba = hiword(iob->iob_wind);
			break;
		default:
			logMsg("tsstart: cmd unknown=%d\n", iob->iob_command);
			return;
			break;
	}
	tsaddr->tssr = ((((int)sc->sc_cmd-(int)vme_window_std) >> 16) & 0x3FFF);
	tsaddr->tsdb = (int)sc->sc_cmd & 0xFFFF;
	if (wait)
		semTake(&sc->sc_int_box);
	if (iob->iob_command == TS_RCOM || iob->iob_command == TS_WCOM)
	    sysGiveVdma (iob->iob_wind);
}

/*
 * TS_DELAY(n) - wait for about n microseconds
 */
TS_DELAY(n)
{
	taskDelay(n);
}

#ifdef	DEBUG

tsio (fd, request, num)
int fd, request, num;
{
    struct mtop op;

    op.mt_op = request;
    op.mt_count = num;

    logMsg ("tsio: fd = %d, &op = 0x%08x\n", fd, &op);
    ioctl (fd, MTIOCTOP, &op);
}

char *
comstr(com)
{
	if (com == TS_RCOM)
		return ("TS_RCOM");
	if (com == TS_REREAD)
		return ("TS_REREAD");
	if (com == TS_SETCHR)
		return ("TS_SETCHR");
	if (com == TS_WCOM)
		return ("TS_WCOM");
	if (com == TS_REWRITE)
		return ("TS_REWRITE");
	if (com == TS_RETRY)
		return ("TS_RETRY");
	if (com == TS_SFORW)
		return ("TS_SFORW");
	if (com == TS_SREV)
		return ("TS_SREV");
	if (com == TS_SFORWF)
		return ("TS_SFORWF");
	if (com == TS_SREVF)
		return ("TS_SREVF");
	if (com == TS_REW)
		return ("TS_REW");
	if (com == TS_OFFL)
		return ("TS_OFFL");
	if (com == TS_WEOF)
		return ("TS_WEOF");
	if (com == TS_RET)
		return ("TS_RET");
	if (com == TS_SENSE)
		return ("TS_SENSE");
	return("TS_????");
}
#endif	DEBUG
