/* dr.c - Driver for IKON 10084 DR11-W EMULATOR  */

/*#define	DEBUG		/**/
/*#define	DEBUG2		/**/

#include "types.h"
#include "param.h"
#include "vxWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "wdLib.h"
#include "buf_uw.h"
#include "iv68k.h"

#include "dr.h"
#include "drreg.h"

#if NDR > 0 || Ndr > 0

extern char vme_std[];
extern char vme_window_std[];

u_short *DRstd[] = {
	(u_short *)&vme_std[0xfffe00], 
	(u_short *)&vme_std[0xffff00],
	0};

/*
 * this structure is used to hold pointers, flags, commands, and status for
 * each dr11 device.
 */
struct	dr_dev	{
	DEV_HDR	dev_hdr;		/* for I/O system */
	struct	drdevice *dr_pcurad;	/* physical address of active dr1 */
	struct	buf	 dr_actf;     	/* these pointers to the drdevice */
	char	dr_unit;		/* unit number */
	WDOG_ID	dr_wdid;		/* watch dog timer id */
	ushort	dr_flags;          	/* See drreg.h */
	ushort	dr_cmd;			/* placed by ioctl for drstrategy */
	ushort	dr_istat;		/* latest interrupt status */
	ushort	dr_time;     		/* # of tocks until time-out */
	ushort	dr_tock;     		/* # of tocks accumulated */
	ushort	dr_cseq;     		/* current sequence number */
	ushort	dr_lseq;     		/* same name for type and struct */
} dr_dev[NDR];

/* 
 * semaphores
 */

SEMAPHORE	dr_cntlsem[NDR];
SEMAPHORE	dr_dmasem[NDR];	
SEMAPHORE	dr_attnsem[NDR];	

/*
 * Misc.
 */

int	dr_num;	/* driver number assigned by UniWorks */
int	drOpen(), drRead(), drintr(), drWrite(), drClose(), drIoctl(),
	drWatch();

/*************************************************************************
*
* drDrv - look for hardware and initialize
*
*/

drDrv()

{
	register struct drdevice *draddr;
	register struct dr_dev 	*drstat;
	register char 		found;
	register int		unit;
	char test=0xff;

	/*
	 * Check for hardware. 
	 */
	found = FALSE;
	for (unit = 0; unit < NDR; unit++) {
		if (vxMemProbe(DRstd[unit], WRITE, 1, &test) == ERROR) 
			continue;
		found = TRUE;
		draddr = (struct drdevice *)DRstd[unit];    
		draddr->dr_intvect = freevec(); 
		intConnect(INUM_TO_IVEC(draddr->dr_intvect), drintr, unit);
		draddr->dr_addmod = DRADMD;

	}

	if(found) {
		dr_num = iosDrvInstall(drOpen, (FUNCPTR)NULL, drOpen, 
					drClose, drRead, drWrite, drIoctl);
		for (unit = 0; unit < NDR; unit++) {
			/* Init dr_dev stuff */
			draddr = (struct drdevice *)DRstd[unit];    
			drstat = &dr_dev[unit];
			drstat->dr_flags = DR_PRES;
			drstat->dr_pcurad = draddr;
			drstat->dr_istat = 0;
			drstat->dr_cmd = 0;
			drstat->dr_unit = unit;
			drstat->dr_wdid = wdCreate();
			semInit(&dr_cntlsem[unit]);
			semGive(&dr_cntlsem[unit]);
			semInit(&dr_dmasem[unit]);
			semInit(&dr_attnsem[unit]);
			drDevCreate(unit);

		}
		return(OK);
	}
	return(ERROR);
}


/***********************************************************************
*
* drDevCreate - create a device
*
*/

STATUS drDevCreate(unit)
	register int	unit;		/* unit number for this device */
{
	char name[20];

	sprintf(name,"/dr%d",unit);

	return (iosDevAdd((struct dr_dev *)&dr_dev[unit], name, dr_num));
}

/**********************************************************************
*
* drWatch - local watchdog timer routine
*
*/

drWatch( drstat )
        register struct dr_dev	 *drstat;
{
	register struct drdevice *draddr;
	register int		 unit;
	register struct buf	 *bp;
	register struct qb_device *qi;

	/*
	 * drWatch() is the local watchdog.  It is called by the system
	 * clock routine approximately every 10 seconds.
	 */
	unit = drstat->dr_unit;
	draddr = drstat->dr_pcurad;

	if( drstat->dr_cseq != drstat->dr_lseq ) {
		drstat->dr_tock = 0;
	} else {
		if( drstat->dr_flags & DR_ACTV ) {
			drstat->dr_tock++;
			if( drstat->dr_tock >= drstat->dr_time ) {
#ifdef	DEBUG
				  logMsg("drWatch(%d): end-of-range timeout\n",
				    unit );
#endif	DEBUG
				/*
				 * Disable interrupts and clear DMA.
				 */
				draddr->dr_pulse = RMSK;
				draddr->dr_pulse = MCLR;
				/*
				 * Some applications will not issue a master
				 * clear after DMA timeout since doing so
				 * sends an INIT H pulse to the external
				 * device which may produce udesirable
				 * side-effects.
				 */
				drstat->dr_flags &= ~DR_ACTV;
				drstat->dr_flags |= DR_TMDM;     
				bp = &drstat->dr_actf;
				bp->b_flags |= B_ERROR; /* flag error */
				semGive(&dr_dmasem[unit]);
			}
		}
		if( drstat->dr_flags & DR_ATWT ) {
			drstat->dr_tock++;     
			if( drstat->dr_tock >= drstat->dr_time ) {
#ifdef	DEBUG
			    logMsg("drWatch(%d): attention timeout\n",unit );
#endif	DEBUG
				draddr->dr_pulse = RMSK;
				drstat->dr_flags &= ~DR_ATWT;
				drstat->dr_flags |= DR_TMAT;    
#ifdef	DEBUG
				printf("drWatch: giving attn sem\n");
#endif	DEBUG
				semGive(&dr_attnsem[unit]);
			}
		}
	}
	if( drstat->dr_flags & DR_OPEN ) {
#ifdef	DEBUG1
		logMsg("drWatch(%d): calling timeout while OPEN\n", unit );
#endif	DEBUG1
		wdStart(drstat->dr_wdid, DR_TICK, drWatch, (char *) drstat);
	}
	drstat->dr_lseq = drstat->dr_cseq;
}

/***********************************************************************
*
* drOpen - open a device
*
*/

int drOpen(drstat, name, mode)
	register struct dr_dev	*drstat;
	register char	*name;
	register int	mode;
{
	int s;
	register struct drdevice *draddr;

	/* Check for error conditions */
	if (name[0] != 0) {
		printf("ERROR: dr%d name not allowed on open\n",
			drstat->dr_unit);
		return(ERROR);
	}
	if ((drstat->dr_flags & DR_PRES) == 0) {
		printf("ERROR: dr%d unit not present\n",drstat->dr_unit);
		return(ERROR);
	}
	if (drstat->dr_flags & DR_OPEN) {
		printf("ERROR: dr%d unit already open\n",drstat->dr_unit);
		return(ERROR);
	}
		
	s = intLock();
	drstat->dr_flags |= DR_OPEN;
	draddr = drstat->dr_pcurad;
	draddr->dr_cstat = DR_ZERO;	/* clr func & intr enable latch */
	draddr->dr_pulse = RDMA | RATN;	/* and e-o-r flag */
	draddr->dr_pulse = IENB;	/* allow interrupts */
	/*
	 * Attempt to establish a local watch dog time.
	 * This call will start it -- it will continue to
	 * call itself as long as the dr11 is open
	 */ 
	drstat->dr_time = DR_TOCK;
	drstat->dr_cseq = 1;
	drstat->dr_lseq = 0;
	wdStart(drstat->dr_wdid, DR_TICK, drWatch, (char *) drstat);
	intUnlock(s);
	return((int) drstat);
	return(OK);
}

/**********************************************************************
*
* drClose - close the device
*
*/

drClose(drstat)
	register struct dr_dev	*drstat;
{
	register int	s;
	register struct drdevice *draddr;

	if ((drstat->dr_flags & DR_OPEN) == 0)
		return(OK);
	s = intLock();
	drstat->dr_flags &= ~DR_OPEN;
	draddr = drstat->dr_pcurad;
	draddr->dr_cstat = DR_ZERO;
	intUnlock(s);
	return(OK);
}


/*************************************************************************
*
* drRead - read from the device
*
*/

drRead(drstat, buffer, nbytes) 

	register struct dr_dev	*drstat;
	register char	*buffer;
	register int	nbytes;

{
	register int stat;

	semTake(&dr_cntlsem[drstat->dr_unit]);
	stat = drstrategy(drstat,buffer,nbytes,READ);
	semGive(&dr_cntlsem[drstat->dr_unit]);
	return(stat);
}

/************************************************************************
*
* drWrite - write to the device
*
*/

drWrite(drstat, buffer, nbytes)
	register struct dr_dev	*drstat;
	register char	*buffer;
	register int	nbytes;

{
	int stat;

	semTake(&dr_cntlsem[drstat->dr_unit]);
	stat = drstrategy(drstat,buffer,nbytes,WRITE);
	semGive(&dr_cntlsem[drstat->dr_unit]);
	return(stat);
}

/**********************************************************************
*
* drstrategy - do the I/O
*
*/

drstrategy(drstat, buffer, nbytes, dir)
	register struct dr_dev	*drstat;
	register char	*buffer;
	register int	nbytes;
	register int	dir;

{
	register struct drdevice   	*draddr;
	register unsigned int		bufadr;
	register unsigned short		dmareg;
	register int			i, unit;

	if(nbytes > BLOCKSIZE)
		return(ERROR);

	drstat->dr_actf.b_flags = 0;

	/* get some of those vdma registers for the
	 * standard address space transfer
	 *
	bufadr = (unsigned int) buffer - (int)vme_window_std;
	 */
	bufadr = sysSetVdma (buffer, nbytes);
	if (!bufadr) {
	    printf ("dr: could not allocate vdma\n");
	    return (ERROR);
	}

	draddr = drstat->dr_pcurad;
	draddr->dr_walo = (u_short)(bufadr >> 1); /* right shoft 1 bit */
	draddr->dr_wahi = (char)(bufadr >> 17);   /* get high byte & shift */
	draddr->dr_range = (u_short)(((nbytes + 1) >> 1) - 1);
	unit = drstat->dr_unit;

#ifdef	DEBUG2
	printf("drstrategy(%d): dma %s\n",unit,
		(dir == READ) ? "read" : "write");
	printf( "drstrategy(%d): low adr          = 0x%x\n", unit,
	    draddr->dr_ralo );
	printf( "drstrategy(%d): high adr         = 0x%x\n", unit,
	    (ushort)draddr->dr_rahi );
	printf( "drstrategy(%d): range            = 0x%x\n", unit,
	    draddr->dr_range );
	printf( "drstrategy(%d): address modifier = 0x%x\n", unit,
	    draddr->dr_addmod );
	printf( "drstrategy(%d): interrupt vector = 0x%x\n", unit,
	    draddr->dr_intvect );
#endif	DEBUG2

	/* Clear dmaf and attf to assue a clean dma start */
	draddr->dr_pulse = RDMA | RATN;
	
	if( drstat->dr_cmd & DR_DFCN ) {
#ifdef	DEBUG
		printf("drstrategy(%d): deferred function bit write\n", unit );
#endif	DEBUG
		drstat->dr_cmd &= ~DR_DFCN;
		draddr->dr_cstat = drstat->dr_cmd & DR_FMSK;
	}
	/*
	 * Increment sequence counter for watchdog timer.
	 */
	drstat->dr_cseq++;
	drstat->dr_flags |= DR_ACTV;
	draddr->dr_pulse = IENB | GO;

	/*
	 * Check for software cycle request -- usually by
	 * transmitter in link mode.
	 */
	if( drstat->dr_cmd & DR_PCYL ) {     
#ifdef	DEBUG
		printf("drstrategy(%d): software cycle request\n", unit );
#endif	DEBUG
		drstat->dr_cmd &= ~DR_PCYL;     /* clear request */
		draddr->dr_pulse = CYCL;     /* use pulse reg again */
	}
	/*
	 * Now check for deferred aclo fnct2 pulse request -- usually to tell
	 * the transmitter (via its attention) that we have enabled dma
	 */
	if( drstat->dr_cmd & DR_DACL ) {     
#ifdef	DEBUG
		printf("drstrategy(%d): sending deferred aclo pulse\n", unit);
#endif	DEBUG
		drstat->dr_cmd &= ~DR_DACL;     /* clear request */
		draddr->dr_pulse = FCN2;     /* pulse aclo fcn2 sig */
	}
	/*
	 * Wait for interrupt semaphore. drintr gives this semaphore
	 * when it's done.
	 */
	semTake(&dr_dmasem[unit]);

	/* now we can give back those vdma registers
	 */
	sysGiveVdma (bufadr);
	sysDataCacheFlush ();

	if(drstat->dr_actf.b_flags & B_ERROR) {
		return(ERROR);
	}
	return(nbytes);
}

/***********************************************************************
*
* drintr - handle interrupts
*
*/

drintr(unit)
	register int	unit;
{
	register struct drdevice *draddr;
        register struct dr_dev	 *drstat;
	register struct buf 	 *bp;
	register ushort		  status;

	drstat = &dr_dev[unit];
	bp = &drstat->dr_actf;
	draddr = drstat->dr_pcurad;
	status = draddr->dr_cstat;

	drstat->dr_istat = status;	/* save interrupt status */
	/*
	 * Make sure that the dr11 is really interrupting -- at least one of
	 * attf or dmaf must be set.
	 * If neither is set this is a spurious interrupt -- a very serious
	 * hardware or software error
	 */ 

	if( !(status & (ATTF | DMAF)) ) {
		bp->b_flags |= B_ERROR;
  logMsg("drintr(%d):  spurious interrupt (dmaf or attf not set)\n",unit);
	}
	if( status & DMAF ) {
		drstat->dr_flags |= DR_DMAX;
#ifdef	DEBUG
		logMsg("drintr(%d): dma end of range interrupt\n",unit);
#endif	DEBUG

		if( !(drstat->dr_flags & DR_ACTV) ) {
#ifdef	DEBUG
			logMsg("drintr(%d): EOR interrupt when dma not active\n", unit );
#endif	DEBUG
			bp->b_flags |= B_ERROR;    /* report error */
		} else {
			drstat->dr_flags &= ~DR_ACTV;
#ifdef	DEBUG
			logMsg("drintr: giving dr_dmasem\n");
#endif	DEBUG
			semGive(&dr_dmasem[unit]);
		}
		draddr->dr_pulse = RDMA;
	}
	/* Now test for attention interrupt -- it may be set in
	 * addition to the dma end of range interrupt.  If we get
	 * one we will issue a wakeup to the drioctl() routine
	 * which is presumably waiting for one. 
	 */
	if( status & ATTF ) {
		drstat->dr_flags |= DR_ATRX;
		drstat->dr_flags &= ~DR_ATWT;
		draddr->dr_pulse = RATN;
#ifdef	DEBUG
		logMsg("drintr(%d): ATTN interrupt, giving attn sem\n",unit);
#endif	DEBUG
		semGive(&dr_attnsem[unit]);
	}
	/*
	 * The IKON 10089 manual claims IENB is reset by the hardware
	 * when an interrupt acknowledge cycle occurs (the interrupt
	 * itself makes sense, but not the mask bit??).  Therefore,
	 * re-enable it.
	 */
	draddr->dr_pulse = IENB;


}

/***********************************************************************
*
* drIoctl
*
*/

drIoctl(drstat, cmd, drop)
	register struct dr_dev	*drstat;
	register int		cmd;
	register struct	drop	*drop;
{

	register int		  unit;
	register struct drdevice  *draddr;
	int			  intlevel;
	unsigned short		  junk;

	unit = drstat->dr_unit;
	draddr = drstat->dr_pcurad;

	if(cmd != DR_IOC) {
		printf("ERROR: dr%d invalid IORW designator\n", unit );
		return(ERROR);	/* invalid argument */
	}
	drstat->dr_cmd = drop->dr_cmd;

	if( drstat->dr_cmd & DR_STIM ) {    
#ifdef	DEBUG
		printf("drioctl(%d): setting time-out = 0x%x\n",
		    unit, drop->dr_timeout );
#endif	DEBUG
		drstat->dr_time = drop->dr_timeout;
	}

	if( drstat->dr_cmd & DR_PIOW ) {   
#ifdef	DEBUG
		printf("drioctl(%d): p-i/o write = 0x%x\n",
		    unit, drop->dr_outdata );
#endif	DEBUG
		draddr->dr_data = drop->dr_outdata;	/* write to p-i/o reg */
	}

	if( drstat->dr_cmd & DR_RATN ) { 
#ifdef	DEBUG
		printf("drioctl(%d): clearing attn flag\n", unit );
#endif	DEBUG
		draddr->dr_pulse = RATN;  /* use pulse register */
	}

	if( drstat->dr_cmd & DR_RDMA ) { 
#ifdef	DEBUG
		printf("drioctl(%d): clearing dmaf flag\n", unit );
#endif	DEBUG
		draddr->dr_pulse = RDMA;  /* use pulse register */
	}

	if( drstat->dr_cmd & DR_RPER ) {  
#ifdef	DEBUG
		printf("drioctl(%d): clearing parity error flag\n", unit );
#endif	DEBUG
		draddr->dr_pulse = RPER;  /* use pulse register */
	}

	if(drstat->dr_cmd & DR_RSET) {  
#ifdef	DEBUG
		printf("drioctl(%d): issuing mclr\n", unit );
#endif	DEBUG
		draddr->dr_pulse = MCLR;
		draddr->dr_pulse = IENB;	/* MCLR resets IENB */
	}

	/*
	 * DR_DFCN indicates delayed function bit set.  In this event,
	 * the FCN bits are set in drstrategy().
	 */
	if( (drstat->dr_cmd & DR_SFCN) && !(drstat->dr_cmd & DR_DFCN) ) {   
#ifdef	DEBUG
		printf("drioctl(%d): fcn bit set = 0x%x\n", unit,
		    drstat->dr_cmd & DR_FMSK );
#endif	DEBUG
		draddr->dr_cstat = drstat->dr_cmd & DR_FMSK;
	}

	if( drstat->dr_cmd & DR_STGO ) {   
#ifdef	DEBUG
		printf("drioctl(%d): GO bit set\n", unit );
#endif	DEBUG
		draddr->dr_cstat = GO;
	}

	if(drstat->dr_cmd & DR_PACL) {  
#ifdef	DEBUG
		printf("drioctl(%d): aclo fnct2 pulse\n", unit );
#endif	DEBUG
		draddr->dr_pulse = FCN2;  /* use pulse register */
	}

	if( drstat->dr_cmd & DR_WAIT ) {  
#ifdef	DEBUG
		printf("drioctl(%d): waiting for ATTN interrupt\n", unit );
#endif	DEBUG
		/* The setting of the dr11 interrupt enable latch and
		 * the call to sleep will be protected from dr11
		 * interrupts by raising the processor priority above
		 * that of the dr11.
		 */
		intlevel = intLock();
#ifdef	DEBUG
		printf("drIoctl: dr_flags=0x%x\n",drstat->dr_flags);
#endif	DEBUG
		if( !(drstat->dr_flags & DR_ATRX) ) {  
			drstat->dr_cseq++;
			drstat->dr_flags |= DR_ATWT;	/* set waiting flag  */
			draddr->dr_pulse = IENB;	/* enable interrupts */
			semTake(&dr_attnsem[unit]);
#ifdef	DEBUG
		printf("drIoctl: got attnsem, dr_flags=0x%x\n",drstat->dr_flags);
#endif	DEBUG
		} else	/* take sem 'cause it was given, so we stay in sync */
			semTake(&dr_attnsem[unit]);

		drstat->dr_flags &= ~DR_ATRX; /* clear flag set by drintr() */
		intUnlock(intlevel);
	}

	junk = drstat->dr_flags;
	drop->dr_indata = draddr->dr_data;
	drop->dr_status = draddr->dr_cstat;
	drop->dr_flags = junk;

	if( drstat->dr_flags & DR_TMAT ) {
		drstat->dr_flags &= ~DR_TMAT;
		printf("ERROR: drIoctl: watchdog timeout\n");
		return(-2);
	}
	return(OK);
}

