/* cpDrv.c - The Integrated Solutions 68020 ICP-board tty handler */

/*
 * modification history -------------------- 
 */
/* 
 * BUGS: 
 *	 need to implement silo dma
 *	 no modem control
 * 	 no parallel port support
 */

/*
 * DESCRIPTION This is the driver for the Integrated Solutions 68020
 * ICP-board serial ports. 
 *
 * USER CALLABLE ROUTINES Most of the routines in this driver are accessible
 * only through the I/O system.  Two routines, however, must be called
 * directly, cpDrv () to initialize the driver, and cpDevCreate to create
 * devices. 
 *
 * ICPCRV Before using the driver, it must be initialized by calling the
 * routine: 
 *
 * .CS cpDrv () .CE 
 *
 * This routine should be called exactly once, before any reads, writes, or
 * cpDevCreates.  Normally, it is called from usrRoot. 
 *
 * CREATING TERMINAL DEVICES Before a terminal can be used, it must be created. 
 * This is done with the cpDevCreate call. Each port to be used should have
 * exactly one device associated with it, by calling this routine. 
 *
 * .CS 
 *
 * int cpDevCreate (name, channel, rdBufSize, wrtBufSize) 
 *	char *name;		/* Name to use for this device * 
 *	int channel;		/* Physical channel for this * device (0 or 1) * 
 *	int rdBufSize;		/* Read buffer size, in bytes * 
 *	int wrtBufSize;		/* Write buffer size, in bytes * 
 *
 * .CE 
 *
 * For instance, to create the device "/cp/0", with buffer sizes of 512 bytes,
 * the proper call would be: 
 *
 * .CS cpDevCreate ("/cp/0", 0, 512, 512); 
 *
 * .CE 
 *
 * IOCTL This driver responds to all the same ioctl codes as a normal ty driver.
 * The baud rates available are 50, 110, 134, 200, 300, 600, 1200, 1050,
 * 2400, 4800, 7200, 9600, 38400. 
 *
 */

/* START Driver
 */

#define IDLE 1
#define BUSY 2
#define XOFFED 4

#include "types.h"
#include "vxWorks.h"
#include "cp.h"
#include "cpreg.h"
#include "ioLib.h"
#include "iosLib.h"
#include "wdLib.h"
#include "tyLib.h"
#include "rngLib.h"
#include "iv68k.h"

#define loword(X)   ( (unsigned long)(X) & 0xffff )
#define hiword(X)   (((unsigned long)(X)>>16) & 0xffff)

typedef struct {		/* BAUD */
    int      rate;	/* a baud rate */
    u_short  csrVal;	/* code to write to the csr to get that baud rate */
}   BAUD;

typedef struct {
    int              alive;
    struct cpdevice *cpaddr;	 	/* pointer to controller */
    BAUD            *baud;	 	/* baud rate conversion table */
    int              nlines;		/* number of lines on controller */
    u_char	     vec;		/* int vector used by this controller */
}   CP_CONT;

static CP_CONT   cp_cont[NICP];

typedef struct {		/* ICP_DEV */
    TY_DEV          tyDev;
    int             created;	/* true if this device has really been
				 * created */
    int             open;	/* true if this device has been opened */
    int             unit;	/* unit number for this structure */
    u_short	    line;	/* line for this unit, 00100000=6, 00000010=2 */
    int             state;	/* channel state */
    int             putpt;	/* put pointer for ring buffer after transfer */
    int             options;	/* tty channel options */
    char 	   *write_wind; /* vme window for write buffer dma */
    CP_CONT        *cont;	/* ptr to controller */
}   ICP_DEV;

/* declare a device structure for each channel */
static ICP_DEV   cp_dev[N_ICP_CHANNELS];

/*
 * baudTable is a table of the available baud rates, and the values to write
 * to the csr reg to get those rates 
 */
static BAUD      bconv0[] = {
 {75, 0x00}, {110, 0x11}, {134, 0x22}, {200, 0x33},
 {300, 0x44}, {600, 0x55}, {1200, 0x66}, {2000, 0x77},
 {2400, 0x88}, {4800, 0x99}, {1800, 0xaa}, {9600, 0xbb},
 {19200, 0xcc}, {0, 0}
};
static BAUD      bconv1[] = {
 {50, 0x00}, {110, 0x11}, {134, 0x22}, {200, 0x33},
 {300, 0x44}, {600, 0x55}, {1200, 0x66}, {1050, 0x77},
 {2400, 0x88}, {4800, 0x99}, {7200, 0xaa}, {9600, 0xbb},
 {0, 0}
};

static int cpNum = 0;		/* driver number assigned to this driver */

/* forward declarations */

static int      cpOpen();
static int	cpClose();
static int      cpRead();
static int      cpWrite();
static int	cpIoctl();

static int      cpStartup();
static int	cpSint();
static int	cptxInt();
static int	cpInt();

/* Bus address of ICPs */

extern char vme_std[];
u_short *CPstd[] = {
                (u_short *)&vme_std[0xfff520],
                (u_short *)&vme_std[0xfff5e0],
                (u_short *)&vme_std[0xfff6c0],
                (u_short *)&vme_std[0xfff540],
                (u_short *)&vme_std[0xfff600],
                (u_short *)&vme_std[0xfff6e0],
                (u_short *)&vme_std[0xfff560],
                (u_short *)&vme_std[0xfff620],
                (u_short *)&vme_std[0xfff580],
                (u_short *)&vme_std[0xfff640],
                (u_short *)&vme_std[0xfff5a0],
                (u_short *)&vme_std[0xfff660],
                (u_short *)&vme_std[0xfff5c0],
                (u_short *)&vme_std[0xfff680],
                 0 };

static int device_present;

/***********************************************************************
*
* cpDrv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/
cpDrv()
{
    register struct cpdevice *cpaddr;
    register int i, j, conf;
    register int cpno = 0;
    u_short test;
    u_long oldvec;

    for (i = 0; i < NICP; i++) {
	/* look for a board in the list of possible 
	 * locations
	 */
	while ((cpaddr = (struct cpdevice *)CPstd[cpno++]) != 0) {
	    test = SEL_MC;

	    /* issue a master clear and probe for board 
	     */
	    if (vxMemProbe(&(cpaddr->cp_sel), WRITE, 2, &test) != OK)  {
		continue;
	    }
	    else
		break;
	}
	if (cpaddr == 0)
	    break;

	cp_cont[i].cpaddr = cpaddr;
	while (cpaddr->cp_sel & SEL_MC)
	    taskDelay(1);

	/*
	 * cause interrupt by setting small silo age time and writing
	 * something in the silo 
	 */
	/* connect controller to interrupt routine */
	cp_cont[i].vec = freevec ();
	oldvec = *((u_long *)(INUM_TO_IVEC(cp_cont[i].vec)));
	intConnect(cp_cont[i].vec << 2, cpInt, i);

	cpaddr->cp_isr = (ISR_IE << 8) | cp_cont[i].vec;
	cpaddr->cp_sel = SEL_SILO;
	cpaddr->cp_pr = 1;	/* silo age time */

	device_present = FALSE;

	/* simulate reception of character */
	cpaddr->cp_swr = SWR_VDP | 0xAB;
	taskDelay(1);

	if (device_present) {
	    cp_cont[i].alive = TRUE;

	    /* get configuration from sel register on controller */
	    /* issue a master clear */
	    cpaddr->cp_sel = SEL_MC;

	    while (cpaddr->cp_sel & SEL_MC)
		taskDelay(1);

	    conf = cpaddr->cp_sel;
	    cp_cont[i].nlines = conf & SEL_CONF_NLINES;
	    cp_cont[i].baud = (conf & SEL_CONF_BR) ? bconv1 : bconv0;

	    /* point to silo */
	    cpaddr->cp_sel = SEL_SILO;

	    /* silo fill level | silo age time */
	    cpaddr->cp_pr = (40 << 8) | 11;

	    /* enable all interrupts on ICP using single vector */
	    cpaddr->cp_isr = (ISR_IE << 8) | cp_cont[i].vec;

	    /* set default line parameters to
	     * 8 bits, no parity, 1 stop bit 9600 baud 
	     * and initialize data structures for all
	     * controller devices
	     */
	    for (j = 0; j < cp_cont[i].nlines; j++) {
		cpaddr->cp_sel = j << SEL_SHIFT;
		cpaddr->cp_pr = PR_BITS8 | 0xbb;
		cp_dev[16*i + j].created = FALSE;
		cp_dev[16*i + j].open = FALSE;
		cp_dev[16*i + j].unit = 16*i + j;
		cp_dev[16*i + j].state = IDLE;
		cp_dev[16*i + j].cont = &cp_cont[i];
		cp_dev[16*i + j].line = 1 << j;
	    }
	    printf ("   CP%d\tat 0x%x\t\t\tvector 0x%x\t %d terminal lines\n",
		i, cpaddr, cp_cont[i].vec, cp_cont[i].nlines);

	} else {
	    logMsg("cp%d: could not generate interrupt\n", i);
	    /* restore old interrupt vector 
	     */
	    *((u_long *)(INUM_TO_IVEC(cp_cont[i].vec))) = oldvec;
	    cp_cont[i].alive = FALSE;
	}
    }

    /* install driver and create the devices on this controller
     */
    for (i = 0; i < NICP ; i++) {
	if (cp_cont[i].alive) {
	    char b[10];

	    if (!cpNum)		/* only install driver once */
		cpNum = iosDrvInstall(cpOpen, (FUNCPTR) NULL, cpOpen, cpClose,
			  cpRead, cpWrite, cpIoctl);
	    for (j = 0; j < cp_cont[i].nlines; j++) {
		sprintf (b, "/cp%d%x", i, j);
		if (cpDevCreate (b, (i<<4) + j, 2048, 2048) != OK)
		    logMsg ("cpDrv: could not create device %s\n", b);
	    }
	}
    }
    return (cpNum ? OK : ERROR);
}

/******************************************************************
*
* cpDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, 
* by calling this routine.
*/
cpDevCreate(name, channel, rdBufSize, wrtBufSize)
char *name;		/* Name to use for this device */
int   channel;		/* Physical channel for this device (0 or 1) */
int   rdBufSize;	/* Read buffer size, in bytes */
int   wrtBufSize;	/* Write buffer size, in bytes */
{
    register char *dmabuf;
    register ICP_DEV *pcp_dev = &cp_dev[channel];

    /* check to see if a controller exists for this device */
    if (!(pcp_dev->cont->alive)) {
	logMsg("ERROR - cpDevCreate - controller not alive\n");
	return (ERROR);
    }
    /* if there is a device already on this channel, don't do it */
    if (pcp_dev->created) {
	logMsg("cpDevCreate: device already on this channel.\n");
	return (ERROR);
    }

    /*
     * initialize the ty descriptor, and turn on the bit for this receiver in
     * the interrupt mask 
     */

    if (tyDevInit((TY_DEV_ID) pcp_dev, rdBufSize,
	    wrtBufSize, cpStartup) != OK) {
	logMsg("cpDevCreate: tyDevInit failed.\n");
	return (ERROR);
    }

    /* create a vdma window to the write buffer */
    dmabuf = (char *)sysSetVdma (pcp_dev->tyDev.wrtBuf->buf, wrtBufSize);
    if (dmabuf < 0) {
	logMsg("cpDevCreate: could not allocate vdma buffer.\n");
	return (ERROR);
    }
    else 
	pcp_dev->write_wind = dmabuf;

    /* mark device as created 
     */
    pcp_dev->created = TRUE;

    return (iosDevAdd((DEV_HDR *) pcp_dev, name, cpNum));
}

/******************************************************************
*
* cpInt - this interrupt handler routes the IRQ to the appropriate
*	  interrupt function
*/
cpInt (cont)
register int cont;
{
    register u_short  isr  = cp_cont[cont].cpaddr->cp_isr;

    switch (isr & 0xFF) {
    case ISR_SI:
	cpSint (cont);	
	break;
    case ISR_TI:
	cptxInt (cont);	
	break;
    case ISR_NI:
	logMsg ("cp: NXM Error\n");
    default:
	break;
    }
    return;
}


/**********************************************************************
*
* cpOpen - open file to isi ICP line
*
*/
static int 
cpOpen(pcp_dev, name, mode)
ICP_DEV  *pcp_dev;
char     *name;
int       mode;
{
    register u_short line = pcp_dev->line;

    if (pcp_dev->open) {
	logMsg("cpOpen: device unit %d already open\n", pcp_dev->unit);
	return (ERROR);
    }

    /* enable serial line and mark device as open */
    pcp_dev->cont->cpaddr->cp_ler |= line;
    pcp_dev->cont->cpaddr->cp_acr |= line;
    pcp_dev->open = TRUE;
    pcp_dev->state = IDLE;
    return ((int) pcp_dev);
}

/**********************************************************************
*
* cpClose - close file to isi ICP line
*
*/
static int 
cpClose(pcp_dev)
ICP_DEV        *pcp_dev;
{
    register u_short line = pcp_dev->line;
    register int cnt = 10;

    if (!pcp_dev->open) {
	logMsg("ERROR -  cpClose - cp device unit %d not open\n", pcp_dev->unit);
	return (ERROR);
    }
    /* disable serial line and mark device as closed */
    while ((pcp_dev->state != IDLE) && --cnt)
	taskDelay(60);		/* keep open long enough to finish a trans. */

    pcp_dev->cont->cpaddr->cp_ler &= ~line;
    pcp_dev->cont->cpaddr->cp_acr &= ~line;
    pcp_dev->open = FALSE;

    if (!cnt) {
	logMsg ("cpClose: cp %d would not close.\n", pcp_dev->unit);
	return (ERROR);
    }
    return (0);
}

/**********************************************************************
*
* cpRead - Task level read routine for isi ICP line.
*
* This routine fields all read calls to the isi usart.  It
* calls tyRead with a pointer to the appropriate element of ty_dev.
*/
static int 
cpRead(pcp_dev, buffer, maxbytes)
ICP_DEV        *pcp_dev;
char           *buffer;
int             maxbytes;
{
    return (tyRead((TY_DEV_ID) pcp_dev, buffer, maxbytes));
}

/**********************************************************************
*
* cpWrite - Task level write routine for isi ICP line.
*
* This routine fields all write calls to the isi ICP line.  It
* calls tyWrite with a pointer to the appropriate element of ty_dev.
*/
static int 
cpWrite(pcp_dev, buffer, nbytes)
ICP_DEV        *pcp_dev;
char           *buffer;
int             nbytes;
{
    return (tyWrite((TY_DEV_ID) pcp_dev, buffer, nbytes));
}

/***********************************************************************
*
* cpIoctl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to cpIoctl.
*/
static int 
cpIoctl(pcp_dev, request, arg)
ICP_DEV        *pcp_dev;	/* device to control */
int             request;	/* request code */
int             arg;		/* some argument */
{
    int             i;
    u_short          lpr;
    int          status;

    /* get line parameter register */
    pcp_dev->cont->cpaddr->cp_sel = (pcp_dev->unit & 0xf) << SEL_SHIFT;
    lpr = pcp_dev->cont->cpaddr->cp_pr;

    switch (request) {
    case FIOBAUDRATE:
	status = ERROR;
	for (i = 0; pcp_dev->cont->baud[i].rate != 0; ++i) {
	    if (pcp_dev->cont->baud[i].rate == arg) {
		lpr = (lpr & 0xFF00) | pcp_dev->cont->baud[i].csrVal;
		status = OK;
		break;
	    }
	}
	if (status != OK)
	    logMsg("cpIoctl: Invalid baud rate.\n");
	break;
    case FIOOPTIONS:
	if (arg & OPT_TANDEM)
	    lpr |= PR_XOFF;
	else
	    lpr &= ~PR_XOFF;
	status = tyIoctl((TY_DEV_ID) pcp_dev, request, arg & ~OPT_TANDEM);
	pcp_dev->options = arg;
	break;
    case FIOGETOPTIONS:
	status = tyIoctl((TY_DEV_ID) pcp_dev, request, arg);
	if (status == OK)
	    if (lpr & PR_XOFF)
		*((int *)arg) |= OPT_TANDEM;
	    else
		*((int *)arg) &= ~OPT_TANDEM;
	return (status);
    default:
	return (tyIoctl((TY_DEV_ID) pcp_dev, request, arg));
    }

    /* set line parameter register */
    pcp_dev->cont->cpaddr->cp_sel = (pcp_dev->unit & 0xf) << SEL_SHIFT;
    pcp_dev->cont->cpaddr->cp_pr = lpr;

    return (status);
}

/**********************************************************************
*
* cpSint - handle a receiver interrupt
*
* This routine gets called to handle a receiver interrupt.
* It doesn't do dma, but reads from the silo window register
*/
static int 
cpSint(cont)
register int cont;
{
    register struct cpdevice *cpaddr;
    register int    overrun = 0;
    register ICP_DEV *pcp_dev;
    register int    c;

    cpaddr = cp_cont[cont].cpaddr;
    /* Loop fetching characters from the silo for this cp.  */
    while ((c = cpaddr->cp_swr) & SWR_VDP) {
	pcp_dev = &cp_dev[(cont * 16) + ((c & SWR_LN_MASK) >> SWR_LN_SHIFT)];
	if (!pcp_dev->open) {
	    device_present = TRUE;
	    continue;
	}
	if ((c & SWR_DO) && overrun++ == 0)
	    if (c & SWR_PE)
		logMsg("ERROR - cp %d silo overflow\n", cont);
	    else
		logMsg("ERROR - cp %d line %d overflow\n", cont,
		       (c & SWR_LN_MASK) >> SWR_LN_SHIFT);
	if (c & SWR_PE)
	    logMsg("ERROR - cp %d line %d parity error\n", cont,
		   (c & SWR_LN_MASK) >> SWR_LN_SHIFT);
	if (c & SWR_FE)
	    logMsg("ERROR - cp %d line %d framing error\n", cont,
		   (c & SWR_LN_MASK) >> SWR_LN_SHIFT);

	c &= SWR_CH_MASK;

	/* if XON recieved and in tandem mode restart device */
	if (c == '\021' && (OPT_TANDEM & pcp_dev->options)) {
	    cpaddr->cp_sel = (pcp_dev->unit & 0xf) << SEL_SHIFT;
	    cpaddr->cp_tcr = pcp_dev->line;
	}
	tyIRd((TY_DEV_ID) pcp_dev, c);
    }
}

/**********************************************************************
*
* cptxInt - handle a transmitter interrupt
*
* This routine gets handles a xmitter interrupt.
* If there is more characters to be transmitted, cpStart is called. 
*/
static int 
cptxInt(cont)
register int cont;
{
    register u_short tcr;
    register struct cpdevice *cpaddr;
    register int    line, bytes;
    register ICP_DEV *pcp_dev;

    cpaddr = cp_cont[cont].cpaddr;
    tcr = cpaddr->cp_tcr;

    /* check all lines for transmit termination */

    for (line = 0; tcr; line++) {
	if (tcr & (1 << line)) {
	    cpaddr->cp_sel = 0;

	    tcr &= ~(1 << line);
	    pcp_dev = &cp_dev[(16 * cont) + line];
	    if ((bytes = cpaddr->cp_bc) > 0) {
		pcp_dev->state = XOFFED;
	    } else {

		/* update the put pointer now that the transfer is done */
		pcp_dev->tyDev.wrtBuf->pFromBuf = pcp_dev->putpt;

		/* some ring buffer space is availible so give semaphore 
		*/
		if (pcp_dev->tyDev.wrtState.semNeeded == 1) {
		    pcp_dev->tyDev.wrtState.semNeeded = 0;
		    semGive(pcp_dev->tyDev.wrtSemId);
		}
		/* mark device as idle and try to restart */
		pcp_dev->state = IDLE;
		pcp_dev->tyDev.wrtState.busy = 0;
		cpStartup(pcp_dev);
	    }
	}
    }
}

/**********************************************************************
*
* cpStartup - Transmitter startup routine
*
* Call interrupt level character output routine for isi ICP line.
*/
cpStartup(pcp_dev)
ICP_DEV        *pcp_dev;	/* ty device to start up */
{
    register struct cpdevice *cpaddr;
    register RING_ID bufpt;
    char           *wrtbuf;
    register int x, bytes;

    /* do dma out of ring buffer */
    bufpt = pcp_dev->tyDev.wrtBuf;

    /* if the device is idle check for data in the transmit buffer */
    if (pcp_dev->state == IDLE) {
	/* check if anything is in the ring buffer */
	if (bufpt->pFromBuf != bufpt->pToBuf) {
	    wrtbuf = &bufpt->buf[bufpt->pFromBuf];
	    /* do dma to put buffer point or end of buffer whichever is first */
	    if (bufpt->pToBuf > bufpt->pFromBuf) {
		bytes = bufpt->pToBuf - bufpt->pFromBuf;
		pcp_dev->putpt = bufpt->pToBuf;
	    } else {
		bytes = bufpt->bufSize - bufpt->pFromBuf;
		pcp_dev->putpt = 0;
	    }

	    /* setup transmit registers */
	    cpaddr = pcp_dev->cont->cpaddr;
	    cpaddr->cp_sel = (pcp_dev->unit & 0xf) << SEL_SHIFT;

	    x = (int)(pcp_dev->write_wind) + bufpt->pFromBuf;
	    cpaddr->cp_bah =  hiword(x);
	    cpaddr->cp_bal =  loword(x);
	    sysDataCacheFlushEntry ((int)bufpt->buf + bufpt->pFromBuf, bytes);

	    cpaddr->cp_bc = bytes;

	    /* mark channel as busy and start I/O */
	    pcp_dev->state = BUSY;
	    pcp_dev->tyDev.wrtState.busy = 1;

	    cpaddr->cp_tcr = pcp_dev->line;
	}
    } 
    else if (pcp_dev->state == XOFFED) {
	logMsg ("device XOFFED\n");
    }
    else {
	logMsg("channel busy\n");
    }
    return;
}
