
# line 2 "imToBsd.yacc"
/* imToBsd - Intermetrics to UNIX BSD 4.3 a.out object module converter */
 
static char *copyright = "Copyright 1986-1988, Wind River Systems, Inc.";
  
/*
modification history
--------------------
02d,22sep88,gae  documentation.
02c,30may88,dnw  changed to v4 names.
02b,21apr88,dnw  corrected -s flag in "usage".
02a,15dec87,dnw  massive changes.  added zillions of options.
01b,22jun87,dnw  changed mode of output file from "w" to "w+" to fix vax
		   version bug causing zeroed data segments.
*/

/*
SYNOPSIS
imToBsd [-x n] [-v] [-u] [-t textAdrs] [-d dataAdrs] [-b bssAdrs] [-e entryAdrs]
	[-o outfile] [-s symfile] <file>

DESCRIPTION
imToBsd converts Intermetrics object modules to UNIX BSD 4.2/4.3 a.out
format object modules.  The resulting object module retains all the relocation
and symbol table information of the original.  The converted object module
is written to a file with the same base name as the original but with a
suffix of ".ob", unless the -o flag is used to specify another name.

RELOCATION
imToBsd can process both relocatable object modules produced by the
Intermetrics assembler or linker, or absolute object modules produced by
the Intermetrics locator.
imToBsd is also capable of performing the relocations itself with the
-t, -d, and -b flags.  In most cases, this eliminates the need for the
Intermetrics locator in generating absolute modules.  This saves a pass
over the object module and simplifies the syntax in generating absolute
modules (the Intermetrics locator having a flexible but cumbersome syntax).
Thus typical sequences of generating a.out format object modules from
Intermetrics tools are:

 compiler -> assembler -> imToBsd => RELOCATABLE OBJECT MODULE
 compiler -> assembler -> linker -> imToBsd => RELOCATABLE OBJECT MODULE
 compiler -> assembler -> linker -> locator -> imToBsd => ABSOLUTE OBJECT MODULE
 compiler -> assembler -> linker -> imToBsd -t... => ABSOLUTE OBJECT MODULE

UNIX SYMBOLS
The UNIX linker/loader will automatically define the symbols `etext', `edata',
and `end' as the end of the text segment, data segment,
and bss segments respectively.
imToBsd will define these symbols if the -u flag is specified.
This is only meaningful if an absolute object module is being generated.

SYMBOL TABLE ONLY MODULE
The VxWorks booting process requires a module that contains just the symbol
table of the system image being booted.  This module is read when the
system is started in order to initialize the system symbol table.
VxWorks includes another tool, `xsym', that extracts the symbol table
from an absolute object module.  However, imToBsd can generate the symbol
table module directly by specifying the -s flag with the desired name of the
symbol table module.

OPTIONS
  -x  n      debug bits:
               0x01 = print each lex item scanned
               0x02 = print yacc state transitions
  -v         verbose - print message for each conversion step
  -u         UNIX symbols - define symbols `etext', `edata', and `end'
               the way UNIX compilers do, as end of text segment,
               end of data segment, and end of bss respectively.
               These  symbols are meaningful only if -t, -d,
               and/or -b are specified.
  -t adrs    locate text segment at specified adrs
  -d adrs    locate data segment at specified adrs
               (default = immediately following text segment).
  -b adrs    locate bss segment at specified adrs
               (default = immediately following data segment).
  -e adrs    set entry address to specified adrs
               (default = start of text segment).
  -o file    output BSD format object module to specified file
               (default = input filename with .ob suffix).
  -s file    generate symbol-table-only module for use in
               downloading vxWorks system image to specified file
               (i.e. vxWorks.sym).

EXAMPLES
The following converts an application module that has been compiled and
assembled into an a.out format module suitable for downloading to VxWorks
with the VxWorks loader (ld).

    imToBsd test.o

The following converts a completely linked VxWorks system image that has
been compiled, assembled, and linked, into an a.out format module suitable
for downloading to the VxWorks boot roms.  It performs the relocations necessary
to locate the code starting at 0x1000.  It also generates the required
vxWorks.sym symbol table module.

    imToBsd -v -u -t 1000 -o vxWorks -s vxWorks.sym vxWorks.o

DEFICIENCIES
The full expression syntax in Intermetrics object modules includes possibly
complex expressions with several arithmetric operators.
However, with the Green Hills compiler/Intermetrics assembler we use,
the only expressions we've encountered are constant values,
constant offsets from an id (i.e. <id> + offset),
and, in a few of the library routines,
a difference of two offsets from the same segment
(i.e. (<seg1> + offset1) - (<seg1> + offset2)) which obviously reduces to
a constant.
imToBsd is currently capable of handling only these forms of expressions.
Anything more elaborate will be reported as an error and imToBsd will fail.

SEE ALSO
UNIX BSD 4.3 a.out documentation,
Intermetrics Object Module Definition
*/

#include <stdio.h>
#include <ctype.h>
#include "vxWorks.h"
#include "a_out.h"
#ifdef VAX
#include <perror>
#endif

/* traditional byte ordering macros */

#ifndef VAX
#define htonl(x)	(x)
#define htons(x)	(x)
#else
#define htonl(x)	((((int)(x) >> 24) & 0x000000ff) | \
			 (((int)(x) >>  8) & 0x0000ff00) | \
			 (((int)(x) <<  8) & 0x00ff0000) | \
			 (((int)(x) << 24) & 0xff000000))

#define htons(x)	((((int)(x) >> 8) & 0x00ff) | \
			 (((int)(x) << 8) & 0xff00))
#endif

#define ntohl(x)	htonl(x)
#define ntohs(x)	htons(x)


/* VMS compatibility */

#ifdef VAX
#define rindex	strrchr
#else
IMPORT char *rindex ();
#endif


/* constants */

#define OMAGIC		0407		/* value of a_magic field in header */
#define MAX_IDS		10000		/* maximum number of ids */
#define DEFAULT_TAG	'L'		/* default type (Long) for expressions*/
#define YYSTYPE 	VALUE		/* type of parse stack */

/* debug options */

#define DBG_LEX		0x01		/* print each lex item */
#define DBG_YACC	0x02		/* print yacc state transitions */


/* SEGMENTS -
 * Each segment (text, data, bss) has a segment descriptor.
 * The actual image of the text and data segments are malloc'ed as soon
 * as we know how big they are (by a .len statement). */

typedef struct			/* SEG - segment descriptor */
    {
    BOOL defined;		/* TRUE = seg has been defined by "seg" stmt */
    BOOL isAbs;			/* TRUE = seg is absolute, not relocatable */
    char *abs;			/* absolute address if isAbs */
    char *image;		/* ptr to image of segment */
    int length;			/* length of seg in bytes (NONE if unknown) */
    int nRelNodes;		/* num of relocations on this segment */
    struct relInfoNode *firstRelNode;	/* ptr to 1st reloc in list */
    struct relInfoNode *lastRelNode;	/* ptr to last reloc in list */
    } SEG;


/* RELOCATION COMMANDS -
 * Relocation commands define the contents of memory particular locations 
 * in the text or data segments to be relative to particular symbols (pId).
 * That symbol can be an external symbol, a segment, an internal absolute,
 * or internal relative symbol (see below).  Non-external relocations
 * are generally specified relative to a segment.  That is,
 * the compiler/assembler/linker generates all references to internal
 * symbols as direct offsets from segments rather than as an offset
 * from the internal symbols themselves.
 * Currently, the only known use of internal relative symbols in a relocation
 * command is for the specially generated symbols etext, edata, and end
 * if the -u flag is specified.
 *
 * Each relocation command is entered in a REL_INFO_NODE and strung on a
 * linked list in the SEGMENT structure for the segment that contains the
 * memory location being relocated.
 */

typedef struct relInfoNode	/* REL_INFO_NODE - relocation entry */
    {
    struct relInfoNode *next;	/* ptr to next relocation in seg list */
    char *address;		/* address to be relocated */
    int length;			/* length of relocation (1,2,4) */
    struct id *pId;		/* symbol to which to relocate */
    } REL_INFO_NODE;


/* ID -
 * An ID is a named object that has been assigned an Intermetrics id number.
 * This includes segments as well as text and data symbols.
 * Thus an ID can be:
 *
 *  - an external symbol	.external == FALSE
 *  - a segment			.external == TRUE, pIsSeg != NULL
 *  - an internal absolute sym	.external == TRUE, pIsSeg == NULL, pSeg = NULL
 *				  (.offset is absolute address)
 *  - an internal relative sym	.external == TRUE, pIsSeg == NULL, pSeg != NULL
 *				  (.offset is offset from pSeg)
 *
 * EXTERNAL SYMBOLS are symbols that are not defined in this module.
 *     They have no segment or offset information.
 * SEGMENTS are the symbols that refer to the text, data, or bss segments.
 *     They have a pointer, 'pIsSeg', to the SEGMENT structure that defines
 *     them.
 * INTERNAL ABSOLUTE SYMBOLS are symbols whose address is fixed.  Such symbols
 *     are generated by the Intermetrics locator.  The absolute address
 *     of the symbol is in 'offset'.
 * INTERNAL RELATIVE SYMBOLS are symbols whose address is relative to the
 *     start of some segment.  The segment and offset are in 'pSeg' and
 *     'offset' respectively.
 * 
 * Relocation commands can refer to each of these types of symbols (see above).
 */

typedef struct id	/* ID - symbol id for symbol table */
    {
    struct id *next;	/* chain of id's to be put in bsd symbol table */
    int idNum;		/* Intermetrics id number of this identifier */
    char *name;		/* ptr to actual identifier string in memory */
    int bsdNum;		/* ordinal number of symbol in bsd symbol table */
    BOOL global;	/* TRUE = symbol is global; FALSE = local (static) */
    BOOL external;	/* TRUE = symbol is unresolved external */
    int type;		/* type of symbol (N_TEXT, N_DATA, N_BSS) */
    SEG *pIsSeg;	/* ptr to seg struct if symbol is seg name */
    SEG *pSeg;		/* ptr to seg to which sym is relative (NULL=absolute)*/
    int offset;		/* offset of symbol address (see table above) */
    int stringIndex;	/* character index to name in strings tbl */
    } ID;


/* EXPRESSIONS -
 * The full Intermetrics syntax includes possibly complex expressions with
 * several arithmetric operators (see syntax below).  However, with the
 * Green Hills compiler/Intermetrics assembler we use, the only expressions
 * we've encountered are constant values, constant offsets from an id
 * (i.e. <id> + offset), and, in a few of the library routines,
 * a difference of two offsets from the same segment
 * (i.e. (<seg1> + offset1) - (<seg1> + offset2)) which obviously reduces to
 * a constant.  The following structure is sufficient for those two types
 * of expressions.  Anything more elaborate will be reported as an error
 * and imToBsd will fail. */

typedef struct		/* EXPR - expression */
    {
    ID *pId;		/* ptr to id; NULL = constant */
    int offset;		/* constant value, offset from id if pId != NULL */
    } EXPR;


/* TOKEN VALUES -
 * Parse tokens on the yacc stack can be any of the types in the union. */

typedef union		/* VALUE */
    {
    char ch;		/* char  : value is single character */
    char *string;	/* string: value is pointer to string */
    int num;		/* number: value is a number */
    ID *pId;		/* id    : value is ptr to id structure */
    EXPR expr;		/* expr  : value is ptr to expression structure */
    } VALUE;


/* The following macros are the primatives to set and get values in a segment 
 * image. */

#define putSeg(pSeg, offset, value, type)	\
	* (type *) ((pSeg)->image + (offset)) = (value)

#define getSeg(pSeg, offset, type)	\
	(* (type *) ((pSeg)->image + (offset)))



char usage [] = "\
Usage: imToBsd [-z] [-v] [-u] [-t textAdrs] [-d dataAdrs] [-b bssAdrs] \n\
               [-e entryAdrs] [-o outfile] [-s symfile] <file>\
";
char myname[] = "imToBsd";


int lexlineno;		/* input line number of last lex item */
int debugOptions;	/* debug options requested */
ID *idTbl [MAX_IDS];	/* table of id's by idNum */


/* segment descriptors */

SEG textSeg;		/* descriptor for text segment */
SEG dataSeg;		/* descriptor for data segment */
SEG bssSeg;		/* descriptor for bss segment */


/* current location - where image is being emitted currently - set by org */

SEG *pCurSeg;		/* ptr to cur seg descriptor */
int curOffset;		/* cur offset within cur seg */


/* BSD symbol table list */

ID *bsdSymFirst;	/* ptr to first symbol to go into bsd sym tbl */
ID *bsdSymLast;		/* ptr to last symbol to go into bsd sym tbl */
int bsdNSyms;		/* number of symbols in bsd sym tbl */
int bsdStrLen = 4;	/* number of chars in bsd string table;
			 *   1st 4 bytes are length of string table */


/* forward declarations */

SEG *segFind ();
char *segAddress ();
SEG *segOfType ();
char *strsave ();
ID *oldId ();
ID *newId ();
# define L_IDENT 257
# define L_STRING 258
# define L_EXPR_TAG 259
# define L_NUMBER 260
# define L_HEX_STRING 261
# define L_ID_NUM 262
# define L_EOLN 263
# define L_T_ABS 264
# define L_T_AORG 265
# define L_T_BGN 266
# define L_T_DCL 267
# define L_T_GLOBAL 268
# define L_T_LOCAL 269
# define L_T_END 270
# define L_T_EXT 271
# define L_T_FILE_STMT 272
# define L_T_GROUP 273
# define L_T_ID 274
# define L_T_LENGTH 275
# define L_T_LINE 276
# define L_T_MARK 277
# define L_T_ORIGIN 278
# define L_T_SEGMENT 279
# define L_T_START 280
# define L_T_BADCMD 281
#define yyclearin yychar = -1
#define yyerrok yyerrflag = 0
extern int yychar;
extern short yyerrflag;
#ifndef YYMAXDEPTH
#define YYMAXDEPTH 150
#endif
#ifndef YYSTYPE
#define YYSTYPE int
#endif
YYSTYPE yylval, yyval;
# define YYERRCODE 256

# line 572 "imToBsd.yacc"


#ifdef YYDEBUG
extern int yydebug;
#endif

#include "imToBsd.lex_c"

main (argc, argv)
    int argc;
    char *argv[];

    {
    char outname [100];		/* output filename buffer */
    FILE *out;			/* file descriptor for output file */
    FILE *xsym;			/* file descriptor for sym-tbl-only file */
    char *pInname   = NULL;	/* pointer to input filename */
    char *pOutname  = NULL;	/* pointer to output filename */
    char *pXsymname = NULL;	/* pointer to symbol-table-only filename */
    int entryAdrs   = NONE;
    int textAdrs    = NONE;
    int dataAdrs    = NONE;
    int bssAdrs     = NONE;
    BOOL verbose    = FALSE;
    BOOL unixSyms   = FALSE;
    int argIndex    = 1;


    if (argc < 2)
	error (usage, NULL);

    /* Crack the arguments */

    while (argIndex < (argc - 1))
	{
	if (strcmp (argv [argIndex], "-x") == 0)	/* debug */
	    sscanf (argv [++argIndex], "%x", &debugOptions);
	else if (strcmp (argv [argIndex], "-v") == 0)	/* verbose */
	    verbose = TRUE;
	else if (strcmp (argv [argIndex], "-u") == 0)	/* etext,etdata */
	    unixSyms = TRUE;
	else if (strcmp (argv [argIndex], "-t") == 0)	/* text adrs */
	    sscanf (argv [++argIndex], "%x", &textAdrs);
	else if (strcmp (argv [argIndex], "-d") == 0)	/* data adrs */
	    sscanf (argv [++argIndex], "%x", &dataAdrs);
	else if (strcmp (argv [argIndex], "-b") == 0)	/* bss adrs */
	    sscanf (argv [++argIndex], "%x", &bssAdrs);
	else if (strcmp (argv [argIndex], "-e") == 0)	/* entry point */
	    sscanf (argv [++argIndex], "%x", &entryAdrs);
	else if (strcmp (argv [argIndex], "-o") == 0)	/* output file name */
	    pOutname = argv [++argIndex];
	else if (strcmp (argv [argIndex], "-s") == 0)	/* xsym output name */
	    pXsymname = argv [++argIndex];
	else
	    break;

	++argIndex;
	}

    if ((argc - argIndex) != 1)
	{
	/* not exactly one arg left */
	error (usage, NULL);
	}

    pInname = argv [argIndex];		/* last arg is input filename */


    /* construct output filename unless output name was specified */

    if (pOutname == NULL)
	{
	char *pDot;

	strcpy (outname, pInname);
	pDot = rindex (outname, '.');
	if (pDot != NULL)
	    *pDot = EOS;
	strcat (outname, ".ob");
	pOutname = outname;
	}

    /* open output and xsym streams */

#ifdef VAX
    out = fopen (pOutname, "w+", "rfm=var", "mrs=512");
#else
    out = fopen (pOutname, "w+");
#endif
    if (out == NULL)
	fileError (pOutname);

    if (pXsymname != NULL)
	{
#ifdef VAX
	xsym = fopen (pXsymname, "w+", "rfm=var", "mrs=512");
#else
	xsym = fopen (pXsymname, "w+");
#endif
	if (xsym == NULL)
	    fileError (pXsymname);
	}


    /* initialize parse */

    segInit (&textSeg);
    segInit (&dataSeg);
    segInit (&bssSeg);

#ifdef YYDEBUG
    yydebug = ((debugOptions & DBG_YACC) != 0);
#else
    if (debugOptions & DBG_YACC)
	fprintf (stderr, "%s: yacc debugging not compiled in\n", myname);
#endif


    /* do it -
     *   assemble images of text and data segments,
     *   lists of relocation commands for both segments,
     *   and a list of symbols in the module.
     */

    if (freopen (pInname, "r", stdin) == NULL)
	fileError (pInname);

    if (verbose)
	printf ("imToBsd: parsing...\n");

    yyparse ();

    fclose (stdin);

    /* do UNIX style relocation and add UNIX symbols if specified */

    segRelocate (textAdrs, dataAdrs, bssAdrs);
    if (unixSyms)
	addUnixSyms ();


    /* go back and fill in relocation values */

    putSegRelVals (&textSeg);
    putSegRelVals (&dataSeg);


    /* write out bsd object module */

    if (verbose)
	printf ("imToBsd: writing object module...\n");

    startOutput (out);

    writeHdr (out, entryAdrs, FALSE);			/* header */
    writeBuf (out, textSeg.image, textSeg.length);	/* text */
    writeBuf (out, dataSeg.image, dataSeg.length);	/* data */
    if (!textSeg.isAbs)
	writeRelInfo (out, textSeg.firstRelNode);	/* text reloc cmds */
    if (!dataSeg.isAbs)
	writeRelInfo (out, dataSeg.firstRelNode);	/* data reloc cmds */
    writeSymTbl (out);					/* symbol table */
    writeStrings (out);					/* strings */

    endOutput (out);
    fclose (out);


    /* if specified, write out xsym module */

    if (pXsymname != NULL)
	{
	if (verbose)
	    printf ("imToBsd: writing sym module...\n");

	startOutput (xsym);

	writeHdr (xsym, 0, TRUE);			/* header */
	writeSymTbl (xsym);				/* symbol table */
	writeStrings (xsym);				/* strings */

	endOutput (xsym);
	fclose (xsym);
	}
    }
/*******************************************************************************
*
* yyerror - handle syntax error
*/

yyerror (string)
    char * string;

    {
    fprintf (stderr, "%s: ERROR line %d:\n", myname, lexlineno);
    fprintf (stderr, "%s\n", string);
    }
/*******************************************************************************
*
* yywrap -
*/

BOOL yywrap ()
    {
    return (TRUE);
    }
/*******************************************************************************
*
* addIdToSymTbl - enter id into symbol table
*/

VOID addIdToSymTbl (pId, pExpr, global)
    ID *pId;
    EXPR *pExpr;
    BOOL global;

    {
    SEG *pSeg;

    /* check if already entered in sym tbl */

    if (pId->bsdNum != NONE)
	return;

    /* set global flag */

    pId->global = global;

    /* determine if external, internal absolue, or internal relative */

    if (pExpr == NULL)
	pId->external = TRUE;		/* external */

    else
	{
	pId->external = FALSE;
	pId->offset   = pExpr->offset;

	if (pExpr->pId == NULL)
	    pId->pSeg = NULL;		/* internal absolute */

	else
	    {				/* internal relative */
	    pId->pSeg = pExpr->pId->pIsSeg;
	    if (pSeg == NULL)
		error ("symbol relative to non-segment id: %s", pId->name);
	    }
	}

    /* get index into bsd string table and bump string table length */

    pId->stringIndex = bsdStrLen;
    bsdStrLen += strlen (pId->name) + 1;

    /* get symbol ordinal and add symbol to end of chain */

    pId->bsdNum = bsdNSyms++;

    if (bsdSymFirst == NULL)
	bsdSymFirst = pId;
    else
	bsdSymLast->next = pId;

    bsdSymLast = pId;
    }
/*******************************************************************************
*
* addReloc - add relocation entry to relocation list
*/

VOID addReloc (pSeg, address, length, pId, tag)
    FAST SEG *pSeg;
    char *address;
    int length;
    ID *pId;
    int tag;

    {
    FAST REL_INFO_NODE *pNew;

    pNew = (REL_INFO_NODE *) malloc (sizeof (REL_INFO_NODE));

    pNew->next          = NULL;
    pNew->address	= address;
    pNew->pId		= pId;
    pNew->length	= length;

    if (pSeg->firstRelNode == NULL)
	pSeg->firstRelNode = pNew;
    else
	pSeg->lastRelNode->next = pNew;

    pSeg->lastRelNode = pNew;

    ++pSeg->nRelNodes;
    }
/*******************************************************************************
*
* exprAsConst - get a constant expression
*/

int exprAsConst (pExpr)
    EXPR *pExpr;

    {
    if (pExpr->pId != NULL)
	error ("expression must be constant: non-constant part is %s",
	       pExpr->pId->name);
    
    return (pExpr->offset);
    }
/*******************************************************************************
*
* strsave - make a string permanent in allocated memory
*/

char *strsave (string)
    char *string;

    {
    char *new = (char *) malloc (strlen (string) + 1);
    strcpy (new, string);
    return (new);
    }
/*******************************************************************************
*
* oldId - get an already existing id descriptor by id number
*/

ID *oldId (idNum)
    int idNum;

    {
    if (idTbl [idNum] == NULL)
	error ("id number %d not defined", idNum);

    return (idTbl [idNum]);
    }
/*******************************************************************************
*
* newId - make a new id descriptor with specified name
*/

ID *newId (idNum, name)
    int idNum;
    char *name;

    {
    ID *pId = (ID *) malloc (sizeof (ID));

    idTbl [idNum] = pId;

    pId->idNum  = idNum;
    pId->name   = name;
    pId->bsdNum = NONE;
    pId->next   = NULL;
    pId->global = FALSE;
    pId->external = FALSE;
    pId->pSeg   = NULL;
    pId->pIsSeg = segFind (name);
    pId->offset = 0;
    pId->stringIndex = 0;

    return (pId);
    }
/*******************************************************************************
*
* lengthOfTag - get byte length of specified tag
*/

VOID lengthOfTag (tag)
    char tag;

    {
    switch (tag)
	{
	case 'b':
	case 'c':
	case 's':
	    return (1);

	case 'w':
	case 'W':
	case 'i':
	case 'I':
	case 'u':
	case 'U':
	    return (2);

	case 'l':
	case 'L':
	    return (4);

	default:
	    error ("invalid tag: %c", tag);
	}
    }
/*******************************************************************************
*
* relCodeOfLength - get bsd relocation code for relocation of specified length
*/

int relCodeOfLength (length)
    int length;

    {
    switch (length)
	{
	case 1: return (0);
	case 2: return (1);
	case 4: return (2);
	}
    }
/*******************************************************************************
*
* getHexByte - get a hex byte value from next two chars of hex string
*/

int getHexByte (string)
    char *string;

    {
    FAST char ch;
    FAST int val = 0;
    int i;

    for (i = 0; i < 2; ++i)
	{
	val <<= 4;

	ch = *string++;

	if (isdigit (ch))
	    val += ch - '0';
	else if (isupper (ch))
	    val += ch - 'A' + 10;
	else
	    val += ch - 'a' + 10;
	}

    return (val);
    }
/*******************************************************************************
*
* error - print error message and die
*
* This routine prints an error message on the standard error output and
* aborts the program with the error status ERROR.  The error message is
* output with the specified format with the single specified argument.
*
* VARARGS1
*/

error (format, arg)
    char *format;	/* format of error message */
    char *arg;		/* argument supplied to format (arbitrary type!) */

    {
    if (lexlineno != 0)
	fprintf (stderr, "%s: ERROR line %d:\n", myname, lexlineno);
    fprintf (stderr, format, arg);
    fprintf (stderr, "\n");
    exit (ERROR);
    }
/*******************************************************************************
*
* fileError - print file system error message and die
*
* This routine prints the file system error message on the standard error
* output and aborts the program with the error status ERROR.
* The error message is output with the program name and specified file name.
*
* VARARGS1
*/

VOID fileError (filename)
    char *filename;	/* filename */

    {
    char string [200];

    sprintf (string, "%s: %s", myname, filename);
    perror (string);
    exit (ERROR);
    }
/*******************************************************************************
*
* segRelocate - make segments be absolute at specified addresses
*/

VOID segRelocate (textAdrs, dataAdrs, bssAdrs)
    char *textAdrs;
    char *dataAdrs;
    char *bssAdrs;

    {
    if ((textAdrs == (char *) NONE) &&
	(dataAdrs == (char *) NONE) &&
	(bssAdrs == (char *) NONE))
	return;

    if (textSeg.isAbs || dataSeg.isAbs || bssSeg.isAbs)
	error ("relocation address specified for module already absolute\n");

    textSeg.isAbs = TRUE;
    textSeg.abs   = (textAdrs != (char *) NONE) ? textAdrs : 0;

    dataSeg.isAbs = TRUE;
    dataSeg.abs   = (dataAdrs != (char *) NONE) ?
			dataAdrs : (textSeg.abs + segLen (&textSeg));

    bssSeg.isAbs = TRUE;
    bssSeg.abs   = (bssAdrs != (char *) NONE) ?
			bssAdrs : (dataSeg.abs + segLen (&dataSeg));
    }
/*******************************************************************************
*
* addUnixSyms - add UNIX symbols
*/

VOID addUnixSyms ()
    {
    FAST ID *pId;
    FAST SEG *pSeg;

    for (pId = bsdSymFirst; pId != NULL; pId = pId->next)
	{
	if (pId->external)
	    {
	    if (strcmp (pId->name, "_etext") == 0)
		pSeg = &textSeg;
	    else if (strcmp (pId->name, "_edata") == 0)
		pSeg = &dataSeg;
	    else if (strcmp (pId->name, "_end") == 0)
		pSeg = &bssSeg;
	    else
		continue;

	    pId->external = FALSE;
	    pId->pSeg   = pSeg;
	    pId->offset = segLen (pSeg);
	    }
	}
    }

/*******************************************************************************
*
* segInit - initialize segment descriptor
*/

VOID segInit (pSeg)
    SEG *pSeg;

    {
    pSeg->defined = FALSE;
    pSeg->length  = NONE;
    pSeg->isAbs   = FALSE;
    pSeg->image   = NULL;
    }
/*************************************************************************************
*
* segFind - find which segment is specified by id
*/

SEG *segFind (name)
    char *name;

    {
    if (strcmp (name, "$$seg9") == 0)
	return (&textSeg);

    if (strcmp (name, "$$seg13") == 0)
	return (&dataSeg);

    if (strcmp (name, "$$seg14") == 0)
	return (&bssSeg);

    return (NULL);
    }
/*******************************************************************************
*
* segLen - get length of segment
*/

int segLen (pSeg)
    SEG *pSeg;

    {
    if (pSeg == NULL)
	error ("length requested of invalid segment");
    
    if (!pSeg->defined)
	return (0);

    if (pSeg->length == NONE)
	error ("segment length not specified");

    return (pSeg->length);
    }
/*******************************************************************************
*
* segLenSet - set length of segment
*/

VOID segLenSet (pSeg, length)
    SEG *pSeg;
    int length;

    {
    pSeg->length = length;

    if (pSeg == &bssSeg)
	return;

    pSeg->image = (char *) calloc (1, length);	/* calloc zeros memory */
    if (pSeg->image == NULL)
	error ("calloc of %d bytes failed.\n", length);
    }
/*******************************************************************************
*
* segAddress - get segment address
*/

char *segAddress (pSeg)
    FAST SEG *pSeg;

    {
    if (pSeg == NULL)
	error ("address requested of NULL segment");

    if (pSeg->isAbs)
	return (pSeg->abs);

    return ((char *) segOffset (pSeg));
    }
/*******************************************************************************
*
* segOffset - get segment offset from start of image
*/

int segOffset (pSeg)
    FAST SEG *pSeg;

    {
    if (pSeg == &textSeg)
	return (0);

    if (pSeg == &dataSeg)
	return (segLen (&textSeg));

    if (pSeg == &bssSeg)
	return (segLen (&textSeg) + segLen (&dataSeg));

    error ("offset requested of invalid seg");
    }
/*******************************************************************************
*
* segOffsetKnown - determine if segment offset is known
*/

BOOL segOffsetKnown (pSeg)
    FAST SEG *pSeg;

    {
    if (pSeg == &textSeg)
	return (TRUE);

    if (pSeg == &dataSeg)
	return (!textSeg.defined || (textSeg.length != NONE));

    if (pSeg == &bssSeg)
	return ((!textSeg.defined || (textSeg.length != NONE)) &&
		(!dataSeg.defined || (dataSeg.length != NONE)));

    error ("offset requested of invalid seg");
    }
/*******************************************************************************
*
* segOfAddress - determine which segment contains address
*/

SEG *segOfAddress (adrs)
    char *adrs;

    {
    if (textSeg.isAbs &&
	(adrs >= textSeg.abs) &&
	(adrs < (textSeg.abs + textSeg.length)))
	return (&textSeg);

    if (dataSeg.isAbs &&
	(adrs >= dataSeg.abs) &&
	(adrs < (dataSeg.abs + dataSeg.length)))
	return (&dataSeg);

    if (textSeg.isAbs &&
	(adrs >= bssSeg.abs) &&
	(adrs < (bssSeg.abs + bssSeg.length)))
	return (&bssSeg);

    error ("address %x not in any segment", adrs);
    }
/*******************************************************************************
*
* segType - get bsd type of segment
*/

int segType (pSeg)
    FAST SEG *pSeg;

    {
    if (pSeg == &textSeg)
	return (N_TEXT);

    if (pSeg == &dataSeg)
	return (N_DATA);

    if (pSeg == &bssSeg)
	return (N_BSS);

    error ("type requested of invalid segment");
    }
/*******************************************************************************
*
* segOfType - find segment of specified type
*/

SEG *segOfType (type)
    int type;	/* N_TEXT | N_DATA | N_BSS */

    {
    if (type == N_TEXT)
	return (&textSeg);

    if (type == N_DATA)
	return (&dataSeg);

    if (type == N_BSS)
	return (&bssSeg);

    error ("invalid segment type specified");
    }

/*******************************************************************************
*
* putSegExpr - put expression in segment
*
* RETURNS: number of bytes put in segment
*/

int putSegExpr (pSeg, offset, pExpr, tag, count)
    SEG *pSeg;		/* segment in which to put expression */
    int offset;		/* byte offset in segment where to put expression */
    EXPR *pExpr;	/* expression to write */
    int tag;		/* type */
    int count;		/* number of times to write expression */

    {
    BOOL reloc;
    int length = lengthOfTag (tag);
    int ix;

    /* don't emit anything in bss */

    if (pSeg == &bssSeg)
	{
	error ("can't emit expression in bss");
	return;
	}


    /* check for constant */

    if (pExpr->pId == NULL)
	reloc = FALSE;			/* expression is a constant */

    else 
	{				/* expression is relocatable */
	reloc = TRUE;

	/* add relocation symbol to symbol table if not already there,
	 * and symbol is not a segment name */

	if ((pExpr->pId->bsdNum == NONE) && (pExpr->pId->pIsSeg == NULL))
	    addIdToSymTbl (pExpr->pId, NULL, TRUE);
	}


    /* put expression "count" times */

    for (ix = 0; ix < count; ++ix)
	{
	/* output value and if relocatable datum, add relocation command */

	putSegVal (pSeg, offset, pExpr->offset, length);

	if (reloc)
	    addReloc (pSeg, offset, length, pExpr->pId);

	offset += length;
	}

    return (length * count);
    }
/*******************************************************************************
*
* putSegHexString - put hex string in segment
*
* RETURNS: number of bytes put in segment
*/

int putSegHexString (pSeg, offset, hexString)
    SEG *pSeg;		/* segment in which to put expression */
    int offset;		/* byte offset in segment where to put expression */
    char *hexString;	/* string of hex values to be written */

    {
    int origOffset = offset;

    /* don't put anything in bss */

    if (pSeg == &bssSeg)
	return;


    /* put bytes in segment */

    ++hexString;	/* skip first "'" */

    while (*hexString != '\'')
	{
	putSeg (pSeg, offset, getHexByte (hexString), char);
	hexString += 2;
	++offset;
	}

    return (offset - origOffset);
    }
/*******************************************************************************
*
* putSegRelVals - update relative values in segment
*/

VOID putSegRelVals (pSeg)
    SEG *pSeg;

    {
    FAST REL_INFO_NODE *pRelNode = pSeg->firstRelNode;
    FAST int value;
    FAST ID *pId;

    /* do each relocation in list */

    while (pRelNode != NULL)
	{
	pId = pRelNode->pId;

	/* relocation can be relative to
	 *  1) external symbol
	 *  2) segment
	 *  3) internal absolute symbol
	 *  4) internal relative symbol
	 */

	if (!pId->external)
	    {
            if (pId->pIsSeg != NULL)
	        value = (int) segAddress (pId->pIsSeg);	/* segment */

	    else if (pId->pSeg == NULL)
	        value = pId->offset;			/* internal abs sym */

	    else
		{
	        value = (int) segAddress (pId->pSeg) + pId->offset;
							/* internal rel sym */
		}

	    if (value != 0)
		{
		value += ntohl (getSegVal (pSeg, pRelNode->address, 
					   pRelNode->length));
	        putSegVal (pSeg, pRelNode->address, value, pRelNode->length);
		}
	    }

	pRelNode = pRelNode->next;
	}
    }
/*******************************************************************************
*
* putSegVal - put a value of the specified length in a segment
*/

VOID putSegVal (pSeg, offset, value, length)
    SEG *pSeg;
    int offset;
    int value;
    int length;

    {
    switch (length)
	{
	case 1:
	    putSeg (pSeg, offset, value, char);
	    break;

	case 2:
	    putSeg (pSeg, offset, htons (value), short);
	    break;

	case 4:
	    putSeg (pSeg, offset, htonl (value), int);
	    break;
	}
    }
/*******************************************************************************
*
* getSegVal - put a value of the specified length in a segment
*/

VOID getSegVal (pSeg, offset, length)
    SEG *pSeg;
    int offset;
    int length;

    {
    switch (length)
	{
	case 1:	return (getSeg (pSeg, offset, char));
	case 2: return (getSeg (pSeg, offset, short));
	case 4: return (getSeg (pSeg, offset, int));
	}
    }

#ifdef VAX
#define VAXIO_BUFSIZE	512
static int vaxioOffset;
static char vaxioBuf [VAXIO_BUFSIZE];
#endif
/*******************************************************************************
*
* startOutput -
*/

VOID startOutput (out)
    FILE *out;		/* file descriptor for output file */

    {
#ifdef VAX
    vaxioOffset = 0;
#endif
    }
/*******************************************************************************
*
* endOutput -
*/

VOID endOutput (out)
    FILE *out;		/* file descriptor for output file */

    {
#ifdef VAX
    if (vaxioOffset != 0)
	{
	if (fwrite (vaxioBuf, vaxioOffset, 1, out) != 1)
	    {
	    perror (myname);
	    error ("fwrite error");
	    }
	}
#endif
    }
/*******************************************************************************
*
* writeBuf - write to output file
*/

VOID writeBuf (out, buf, len)
    FILE *out;		/* file descriptor for output file */
    char *buf;
    int len;

    {
#ifdef VAX
    int bytesLeftInBuffer;
    char *pBuf;

    while (len > 0)
	{
	/* Three cases:
	 *  1) the length to be output is less than the number of bytes
	 *     left in the internal buffer, then just copy the callers
	 *     buffer into the internal buffer and return (no write);
	 *  2) there is nothing in the internal buffer and the length to
	 *     be output is longer than bufsize, then just write one block
	 *     directly from callers buffer (no copy);
	 *  3) there is something in the internal buffer already and the
	 *     length to be written is greater than or equal to bytes
	 *     left in the internal buffer, then copy enough of the callers
	 *     buffer to fill the internal buffer and write the internal
	 *     buffer.
	 */

	bytesLeftInBuffer = VAXIO_BUFSIZE - vaxioOffset;
	if (len < bytesLeftInBuffer)
	    {
	    /* callers buffer won't fill up the internal buffer;
	     * just copy callers buffer to internal buffer and return */

	    bcopy (buf, &vaxioBuf [vaxioOffset], len);
	    vaxioOffset += len;
	    return;
	    }

	if (vaxioOffset == 0)
	    {
	    /* nothing in internal buffer and callers buffer is longer than
	     * a block; just write one complete block from callers buffer */

	    pBuf = buf;
	    len -= VAXIO_BUFSIZE;
	    buf += VAXIO_BUFSIZE;
	    }
	else
	    {
	    /* something already in internal buffer;
	     * fill internal buffer and write it out */

	    bcopy (buf, &vaxioBuf [vaxioOffset], bytesLeftInBuffer);
	    pBuf = vaxioBuf;
	    len -= bytesLeftInBuffer;
	    buf += bytesLeftInBuffer;
	    }


	/* output block */

	if (fwrite (pBuf, VAXIO_BUFSIZE, 1, out) != 1)
	    {
	    perror (myname);
	    error ("fwrite error");
	    }

	vaxioOffset = 0;
	}
#else		
    if (fwrite (buf, len, 1, out) != 1)
	{
	perror (myname);
	error ("fwrite error");
	}
#endif
    }
/*******************************************************************************
*
* writeHdr - write bsd object module header to output file
*/

VOID writeHdr (out, entry, xsymFlag)
    FILE *out;		/* file descriptor for output file */
    int entry;
    BOOL xsymFlag;	/* TRUE = write xsym header */

    {
    struct exec hdr;

    if (entry == NONE)
	entry = (int) segAddress (&textSeg);

    hdr.a_magic  = htonl (OMAGIC);
    hdr.a_syms   = htonl (bsdNSyms * sizeof (struct nlist));

    if (xsymFlag)
	{
	hdr.a_text   = 0;
	hdr.a_data   = 0;
	hdr.a_bss    = 0;
	hdr.a_entry  = 0;
	hdr.a_trsize = 0;
	hdr.a_drsize = 0;
	}
    else
	{
	hdr.a_text   = htonl (segLen (&textSeg));
	hdr.a_data   = htonl (segLen (&dataSeg));
	hdr.a_bss    = htonl (segLen (&bssSeg));
	hdr.a_entry  = htonl (entry);
	hdr.a_trsize = textSeg.isAbs ? 0 :
			htonl (textSeg.nRelNodes *
			       sizeof (struct relocation_info));
	hdr.a_drsize = dataSeg.isAbs ? 0 :
			htonl (dataSeg.nRelNodes *
			       sizeof (struct relocation_info));
	}

    writeBuf (out, &hdr, sizeof (hdr));
    }
/*******************************************************************************
*
* writeRelInfo - write relocation command to output file
*/

VOID writeRelInfo (out, pRelNode)
    FILE *out;		/* file descriptor for output file */
    FAST REL_INFO_NODE *pRelNode;

    {
    FAST ID *pId;
    FAST char *pByte;
    struct relocation_info rel;
    FAST int symbolnum;

    for (; pRelNode != NULL; pRelNode = pRelNode->next)
	{
	pId = pRelNode->pId;

	/* relocation can be relative to
	 *  1) segment
	 *  2) external symbol
	 *  3) internal absolute symbol
	 *  4) internal relative symbol
	 */

        if (pId->pIsSeg != NULL)
	    symbolnum = segType (pId->pIsSeg);	/* segment */

	else if (pId->external)
	    symbolnum = pId->bsdNum;		/* external symbol */

	else if (pId->pSeg == NULL)
	    symbolnum = pId->bsdNum;		/* internal absolute symbol */

	else
	    symbolnum = segType (pId->pSeg);	/* internal relative symbol */


	/* the following gets the relocation structure written out in the
	 * in the desired form, independent of the host compilers
	 * interpretation of the relocation structure.
	 */

	rel.r_address = (int) htonl (pRelNode->address);

	pByte = ((char *) &rel) + 4;
	*pByte++ = (symbolnum >> 16) & 0xff;
	*pByte++ = (symbolnum >> 8) & 0xff;
	*pByte++ = symbolnum & 0xff;

	*pByte = relCodeOfLength (pRelNode->length) << 5;
	*pByte |= pId->external << 4;

	writeBuf (out, &rel, sizeof (rel));
	}
    }
/*******************************************************************************
*
* writeStrings - write string table to output file
*/

VOID writeStrings (out)
    FILE *out;		/* file descriptor for output file */

    {
    FAST ID *pId = bsdSymFirst;
    int length = htonl (bsdStrLen);

    writeBuf (out, &length, 4);

    for (; pId != NULL; pId = pId->next)
	writeBuf (out, pId->name, strlen (pId->name) + 1);
    }
/*******************************************************************************
*
* writeSymTbl - write symbol table to output file
*/

VOID writeSymTbl (out)
    FILE *out;		/* file descriptor for output file */

    {
    FAST ID *pId = bsdSymFirst;
    struct nlist sym;
    FAST SEG *pRelSeg;

    for (; pId != NULL; pId = pId->next)
	{
	/* construct nlist structure for this symbol */

	/* fill in type and value of id */

	if (pId->external)
	    {
	    /* undefined external */

	    sym.n_type  = N_UNDF | N_EXT;
	    sym.n_value = 0;
	    }
	else
	    {
	    /* internal symbol */

	    sym.n_value = pId->offset;

	    if ((pRelSeg = pId->pSeg) == NULL)
		{
		/* symbol is absolute:
	 	 *   determine which segment symbol is in by address */

		pRelSeg = segOfAddress ((char *) pId->offset);
		}
	    else
		{
	 	/* symbol is relative to a segment:
	 	 *   offset value by segment address */

		sym.n_value += (int) segAddress (pId->pSeg);
		}

	    sym.n_type = segType (pRelSeg);

	    if (pId->global)
		sym.n_type |= N_EXT;
	    }

	sym.n_un.n_strx = htonl (pId->stringIndex);
	sym.n_value     = htonl (sym.n_value);
	sym.n_other     = 0;
	sym.n_desc      = 0;

	writeBuf (out, &sym, sizeof (sym));
	}
    }
short yyexca[] ={
-1, 1,
	0, -1,
	263, 3,
	-2, 0,
	};
# define YYNPROD 66
# define YYLAST 245
short yyact[]={

  34,  87,  38,  88,  39,  70,  35,  36,  94,  53,
  33,  74,  73,  61,  42,  99,  85,  43,  89,  22,
  90,  58,  72,  65,  34,  37,  93,  30,  60,  65,
  34,  96,  26,  92,  33,  65,  34,  63,   9,   8,
  33,  67,  34,  63,   7,   6,  33,  67,  34,  63,
   5,  41,  33,  67,  97,  64,   2,   1,  33,   0,
   0,  64,   0,  76,  75,   0,   0,  64,   0,  62,
   0,   0,   0,  66,  31,  62,  32,   0,   0,  66,
  40,  62,   0,   0,  59,  66,  29,  44,  45,  46,
  47,  48,  49,  50,  51,  54,  55,   0,   0,  57,
   0,   0,   0,  52,   0,   0,   0,   0,  56,   0,
  68,  69,   0,   0,   0,  71,   0,   0,   0,   0,
   0,  79,   0,  81,  82,   0,   0,   0,  29,  77,
  78,   0,  80,  86,   0,  83,   0,   0,   0,   0,
  84,   0,   0,   0,   0,   0,  91,   0,   0,   0,
   0,   0,   0,   0,  95,   0,   0,   0,   0,   0,
   0,   0,   0,   0,   0,   0,   0,   0,  98,  35,
   0,   0,   0,   0, 100,   0,   0,   0,   0,   0,
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
   0,  38,  28,   0,  35,  27,  36,   0,  13,  15,
  23,  24,  10,  11,  25,  17,  19,   3,   4,  14,
  20,  21,  16,  12,  18,  38,   0,   0,  35,   0,
  36,  38,   0,  61,  35,   0,  36,  38,   0,   0,
  35,   0,  36,  38,  28,   0,  35,  27,  36,  38,
   0,   0,  35,   0,  36 };
short yypact[]={

-1000, -66,-259,-255,-244,-1000,-1000,-1000, -27,-1000,
-255,-255,-255,-255,-255,-255,-255,-255, -18,-249,
-254,-254,-1000, -18,-254,-1000,-1000,-1000,-1000, -36,
-1000,-1000,-1000,-255,-255,-1000,-1000,-257,-1000,-1000,
-254,-1000,-246, -24, -18, -18,-254, -18,-254,-254,
 -18,-1000, -30,-1000,-1000, -18, -30,-241,-254, -42,
-1000,-1000,-1000,-1000,-1000,-1000,-1000,-1000,-1000,-1000,
-1000,-254,-1000,-1000,-1000,-1000,-1000, -30, -30, -91,
 -30,-1000,-1000, -30, -30,-1000,-1000,-1000,-1000,-1000,
-1000,-255,-242,-1000,-1000,-1000,-1000,-255,-1000,-1000,
-1000 };
short yypgo[]={

   0,  57,  56,  76,  74,  54,  51,  50,  45,  44,
  39,  38,  84,  33,  19,  32,  31,  21,  28,  27,
  25,  22 };
short yyr1[]={

   0,   1,   1,   2,   2,   2,   2,   2,   2,   2,
   2,   7,   7,   7,   7,   7,   7,   7,   8,   8,
  10,  10,  14,  14,  14,   9,   9,   9,  11,  11,
  11,  11,  16,  13,  13,  13,  15,  15,  15,  17,
  12,  12,  12,  12,  12,  12,  18,  18,  18,  18,
  18,  18,   5,   5,   3,   3,  20,   6,  21,  21,
  21,   4,  19,  19,  19,  19 };
short yyr2[]={

   0,   0,   3,   0,   5,   2,   1,   1,   1,   1,
   1,   3,   3,   5,   3,   3,   3,   3,   2,   2,
   1,   3,   1,   1,   1,   2,   2,   3,   1,   2,
   4,   1,   0,   1,   1,   1,   1,   2,   3,   1,
   3,   3,   3,   3,   2,   1,   1,   1,   1,   1,
   1,   1,   2,   1,   1,   2,   1,   2,   1,   1,
   1,   1,   1,   1,   2,   2 };
short yychk[]={

-1000,  -1,  -2, 273, 274,  -7,  -8,  -9, -10, -11,
 268, 269, 279, 264, 275, 265, 278, 271, 280, 272,
 276, 277, -14, 266, 267, 270, -15, 261, 258, -12,
 -19,  -4,  -3,  76,  66, 260, 262, -20, 257, 263,
  -3,  -6, 258,  44,  -3,  -3,  -3,  -3,  -3,  -3,
  -3,  -3, -12, 258,  -4,  -4, -12,  -4, -17, -12,
 -18, 259, 111,  79,  97,  65, 115,  83,  -3,  -3,
 262,  -4, -21, 258, 257, -17, -14, -12, -12,  -4,
 -12,  -4,  -4, -12, -12, 257,  -4,  43,  45,  60,
  62,  -4, -13, 117,  99,  -4, -16,  -5,  -3, 257,
  -3 };
short yydef[]={

   1,  -2,   0,   0,   0,   6,   7,   8,   9,  10,
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
   0,   0,  20,  28,   0,  31,  22,  23,  24,  36,
  45,  62,  63,   0,   0,  61,  54,   0,  56,   2,
   0,   5,   0,   0,   0,   0,   0,   0,   0,   0,
   0,  18,  19,  25,  26,   0,  29,   0,  37,   0,
  44,  39,  46,  47,  48,  49,  50,  51,  64,  65,
  55,   0,  57,  58,  59,  60,  21,  11,  12,   0,
  14,  15,  16,  17,  27,  32,  38,  40,  41,  42,
  43,   0,   0,  33,  34,  35,  30,   4,  53,  13,
  52 };
#ifndef lint
static char yaccpar_sccsid[] = "@(#)yaccpar	4.1	(Berkeley)	2/11/83";
#endif not lint

#
# define YYFLAG -1000
# define YYERROR goto yyerrlab
# define YYACCEPT return(0)
# define YYABORT return(1)

/*	parser for yacc output	*/

#ifdef YYDEBUG
int yydebug = 0; /* 1 for debugging */
#endif
YYSTYPE yyv[YYMAXDEPTH]; /* where the values are stored */
int yychar = -1; /* current input token number */
int yynerrs = 0;  /* number of errors */
short yyerrflag = 0;  /* error recovery flag */

yyparse() {

	short yys[YYMAXDEPTH];
	short yyj, yym;
	register YYSTYPE *yypvt;
	register short yystate, *yyps, yyn;
	register YYSTYPE *yypv;
	register short *yyxi;

	yystate = 0;
	yychar = -1;
	yynerrs = 0;
	yyerrflag = 0;
	yyps= &yys[-1];
	yypv= &yyv[-1];

 yystack:    /* put a state and value onto the stack */

#ifdef YYDEBUG
	if( yydebug  ) printf( "state %d, char 0%o\n", yystate, yychar );
#endif
		if( ++yyps> &yys[YYMAXDEPTH] ) { yyerror( "yacc stack overflow" ); return(1); }
		*yyps = yystate;
		++yypv;
		*yypv = yyval;

 yynewstate:

	yyn = yypact[yystate];

	if( yyn<= YYFLAG ) goto yydefault; /* simple state */

	if( yychar<0 ) if( (yychar=yylex())<0 ) yychar=0;
	if( (yyn += yychar)<0 || yyn >= YYLAST ) goto yydefault;

	if( yychk[ yyn=yyact[ yyn ] ] == yychar ){ /* valid shift */
		yychar = -1;
		yyval = yylval;
		yystate = yyn;
		if( yyerrflag > 0 ) --yyerrflag;
		goto yystack;
		}

 yydefault:
	/* default state action */

	if( (yyn=yydef[yystate]) == -2 ) {
		if( yychar<0 ) if( (yychar=yylex())<0 ) yychar = 0;
		/* look through exception table */

		for( yyxi=yyexca; (*yyxi!= (-1)) || (yyxi[1]!=yystate) ; yyxi += 2 ) ; /* VOID */

		while( *(yyxi+=2) >= 0 ){
			if( *yyxi == yychar ) break;
			}
		if( (yyn = yyxi[1]) < 0 ) return(0);   /* accept */
		}

	if( yyn == 0 ){ /* error */
		/* error ... attempt to resume parsing */

		switch( yyerrflag ){

		case 0:   /* brand new error */

			yyerror( "syntax error" );
		yyerrlab:
			++yynerrs;

		case 1:
		case 2: /* incompletely recovered error ... try again */

			yyerrflag = 3;

			/* find a state where "error" is a legal shift action */

			while ( yyps >= yys ) {
			   yyn = yypact[*yyps] + YYERRCODE;
			   if( yyn>= 0 && yyn < YYLAST && yychk[yyact[yyn]] == YYERRCODE ){
			      yystate = yyact[yyn];  /* simulate a shift of "error" */
			      goto yystack;
			      }
			   yyn = yypact[*yyps];

			   /* the current yyps has no shift onn "error", pop stack */

#ifdef YYDEBUG
			   if( yydebug ) printf( "error recovery pops state %d, uncovers %d\n", *yyps, yyps[-1] );
#endif
			   --yyps;
			   --yypv;
			   }

			/* there is no state on the stack with an error shift ... abort */

	yyabort:
			return(1);


		case 3:  /* no shift yet; clobber input char */

#ifdef YYDEBUG
			if( yydebug ) printf( "error recovery discards char %d\n", yychar );
#endif

			if( yychar == 0 ) goto yyabort; /* don't discard EOF, quit */
			yychar = -1;
			goto yynewstate;   /* try again in the same state */

			}

		}

	/* reduction by production yyn */

#ifdef YYDEBUG
		if( yydebug ) printf("reduce %d\n",yyn);
#endif
		yyps -= yyr2[yyn];
		yypvt = yypv;
		yypv -= yyr2[yyn];
		yyval = yypv[1];
		yym=yyn;
			/* consult goto table to find next state */
		yyn = yyr1[yyn];
		yyj = yypgo[yyn] + *yyps + 1;
		if( yyj>=YYLAST || yychk[ yystate = yyact[yyj] ] != -yyn ) yystate = yyact[yypgo[yyn]];
		switch(yym){
			
case 11:
# line 368 "imToBsd.yacc"
{ addIdToSymTbl (yypvt[-1].pId, &yypvt[-0].expr, TRUE); } break;
case 12:
# line 370 "imToBsd.yacc"
{ addIdToSymTbl (yypvt[-1].pId, &yypvt[-0].expr, FALSE); } break;
case 13:
# line 372 "imToBsd.yacc"
{
			/* initial reference to seg; note segment defined */
			if (yypvt[-3].pId->pIsSeg != NULL)
			    yypvt[-3].pId->pIsSeg->defined = TRUE;
			} break;
case 14:
# line 378 "imToBsd.yacc"
{
			/* segment is located absolutely */

			SEG *pSeg = yypvt[-1].pId->pIsSeg;
			if (pSeg != NULL)
			    {
			    pSeg->isAbs = TRUE;
			    pSeg->abs   = (char *) exprAsConst (&yypvt[-0].expr);
			    }
			} break;
case 15:
# line 389 "imToBsd.yacc"
{
			/* segment length is specified */

			SEG *pSeg = yypvt[-1].pId->pIsSeg;
			if (pSeg != NULL)
			    segLenSet (pSeg, yypvt[-0].num);
			} break;
case 16:
# line 397 "imToBsd.yacc"
{
			/* ORGing to absolute address */

			pCurSeg = yypvt[-1].pId->pIsSeg;
			if ((pCurSeg != NULL) && pCurSeg->isAbs)
			    curOffset = yypvt[-0].num - (int) pCurSeg->abs;
			else
			    {
			    if (strcmp (yypvt[-1].pId->name, "PSCT") != 0)
				error ("AORG to invalid id: %s", yypvt[-1].pId->name);
			    }
			} break;
case 17:
# line 410 "imToBsd.yacc"
{
			/* ORGing to relative address */

			pCurSeg = yypvt[-1].pId->pIsSeg;
			if ((pCurSeg != NULL) && !pCurSeg->isAbs)
			    curOffset = exprAsConst (&yypvt[-0].expr);
			else
			    {
			    if (strcmp (yypvt[-1].pId->name, "PSCT") != 0)
				error ("ORG to invalid id: %s", yypvt[-1].pId->name);
			    }
			} break;
case 23:
# line 436 "imToBsd.yacc"
{
			curOffset += putSegHexString (pCurSeg, curOffset, yypvt[-0]);
			} break;
case 24:
# line 440 "imToBsd.yacc"
{ error ("can't handle strings in input"); } break;
case 36:
# line 465 "imToBsd.yacc"
{
			curOffset += putSegExpr (pCurSeg, curOffset, &yypvt[-0].expr,
						 DEFAULT_TAG, 1);
			} break;
case 37:
# line 470 "imToBsd.yacc"
{
			curOffset += putSegExpr (pCurSeg, curOffset, &yypvt[-1].expr,
						 yypvt[-0].ch, 1);
			} break;
case 38:
# line 475 "imToBsd.yacc"
{
			curOffset += putSegExpr (pCurSeg, curOffset, &yypvt[-2].expr,
						 yypvt[-1].ch, yypvt[-0].num);
			} break;
case 40:
# line 485 "imToBsd.yacc"
{
			if ((yypvt[-2].expr.pId != NULL) && (yypvt[-1].expr.pId != NULL))
			    error ("can't handle expression with two symbols");

			if (yypvt[-2].expr.pId != NULL)
			    yyval.expr.pId = yypvt[-2].expr.pId;
			else
			    yyval.expr.pId = yypvt[-1].expr.pId;

			yyval.expr.offset = yypvt[-2].expr.offset + yypvt[-1].expr.offset;
			} break;
case 41:
# line 497 "imToBsd.yacc"
{
			if ((yypvt[-2].expr.pId != NULL) && (yypvt[-1].expr.pId != NULL) &&
			    (yypvt[-2].expr.pId != yypvt[-1].expr.pId))
			    error ("can't handle expression with two symbols");

			if (yypvt[-2].expr.pId == yypvt[-1].expr.pId)
			    yyval.expr.pId = NULL;
			else if (yypvt[-2].expr.pId != NULL)
			    yyval.expr.pId = yypvt[-2].expr.pId;
			else
			    yyval.expr.pId = yypvt[-1].expr.pId;

			yyval.expr.offset = yypvt[-2].expr.offset - yypvt[-1].expr.offset;
			} break;
case 42:
# line 511 "imToBsd.yacc"
{ error ("can't handle '<' expressions"); } break;
case 43:
# line 512 "imToBsd.yacc"
{ error ("can't handle '>' expressions"); } break;
case 44:
# line 513 "imToBsd.yacc"
{ error ("can't handle unop expressions"); } break;
case 54:
# line 529 "imToBsd.yacc"
{ yyval.pId = oldId (yypvt[-0].num); } break;
case 55:
# line 530 "imToBsd.yacc"
{ yyval.pId = newId (yypvt[-0].num, yypvt[-1].string); } break;
case 56:
# line 534 "imToBsd.yacc"
{
			yytext [yyleng-1] = EOS;	/* remove '}' */
			yyval.string = strsave (&yytext[1]);
			} break;
case 62:
# line 551 "imToBsd.yacc"
{ yyval.expr.pId = NULL;   yyval.expr.offset = yypvt[-0].num; } break;
case 63:
# line 552 "imToBsd.yacc"
{ yyval.expr.pId = yypvt[-0].pId; yyval.expr.offset = 0; } break;
case 64:
# line 554 "imToBsd.yacc"
{
			SEG *pSeg = yypvt[-0].pId->pIsSeg;
			if (pSeg == NULL)
			    error ("id must be segment name: %s", yypvt[-0].pId->name);

			yyval.expr.pId = NULL;
			yyval.expr.offset = segLen (pSeg);
			} break;
case 65:
# line 563 "imToBsd.yacc"
{
			SEG *pSeg = yypvt[-0].pId->pIsSeg;
			if (pSeg == NULL)
			    error ("id must be segment name: %s", yypvt[-0].pId->name);

			yyval.expr.pId = NULL;
			yyval.expr.offset = (int) segAddress (pSeg);
			} break;
		}
		goto yystack;  /* stack new state and value */

	}
