/* coffToBsd.c - convert System V coff file to BSD a.out format */

/*
modification history
--------------------
01g,12sep88,gae  documentation; some lint.
01f,29feb88,ecs  fixed data relocation bug.
01e,17feb88,ecs  delinted.
01d,03feb88,ecs  fixed to always clear string table.
01c,02feb88,ecs  fixed undefined external data bug in xlateReloc.
01b,27jan88,ecs  fixed bug in xlateReloc screwing up ld'ing of mods
		    with multiple .o's.
01a,09jan88,ecs  modified from hpToBsd.c version 01a.
*/

/*
SYNOPSIS
coffToBsd <coffFile >bsdFile

DESCRIPTION
This program converts System V Release 2 Common Object File Format (COFF)
file into a BSD 4.3 a.out file.
It is used to make VxWorks loadable/executable files on a
Unisoft System V Release 2 host.

CAVEAT
Certain unwarranted assumptions are made about the coff file.
Tested only on Unisoft coff files.

FILES
 a_out.h      BSD 4.3 a.out.h header
 coffAout.h   System V Common Object File Format (COFF) header

SEE ALSO
 UNIX BSD 4.3 a.out documentation
 System V Release 2 COFF documentation
*/

#ifdef	LINT
#include "vxWorks.h"
#include "stdioLib.h"
#else
#include <stdio.h>
#include "vxWorks.h"
#endif	LINT

#include "a_out.h"
#include "coffAout.h"
#include "strLib.h"

IMPORT char *malloc ();

#define TEXT		0
#define DATA		1
#define BSS		2
#define NUM_SCNS	3

#define NLIST	struct nlist		/* BSD symbol table entry */
#define EXEC	struct exec		/* BSD a.out header */
#define RINFO	struct relocation_info	/* BSD relocation entry */


FILHDR  inHdr;		/* coff file header */
EXEC    outHdr;		/* BSD file header */
AOUTHDR aoHdr;		/* coff optional header */
SCNHDR scnHdr[NUM_SCNS];/* coff section headers */

char   *textSect;	/* coff text section */
char   *data;		/* coff data section */

RELOC  *relText;	/* coff text relocation entries */
RELOC  *relData;	/* coff data relocation entries */
RINFO  *oRelText;	/* BSD text relocation entries */
RINFO  *oRelData;	/* BSD data relocation entries */
int	nTextRel;	/* number of text relocation entries */
int	nDataRel;	/* number of data relocation entries */

SYMENT *symTbl;		/* coff symbol table */
int     symix;		/* index into symTbl */
NLIST  *outSymTbl;	/* BSD symbol table */
int     osymix;		/* index into outSymTbl */
int    *stTranslator;	/* pointer to list of outSymTbl entry corresponding
			 * to each symTbl entry */

char   *strTbl;		/* coff string table */
long    strTblSize;	/* size of coff string table */
char   *outStrTbl;	/* BSD string table */
int     stix;		/* index into outStrTbl */
char   *sp;		/* points to string in outStrTbl */
long    outStrTblSize;	/* size of BSD string table */

/*******************************************************************************
*
* main - read specified file and modify 
*/

VOID main (argc)
    int argc;

    {
    if (argc != 1)
	error ("usage: coffToBsd <coffFile >bsdFile");

    /* read object module header */

    if (fread ((char *)&inHdr, sizeof (inHdr), 1, stdin) != 1)
	error ("error reading file header.");

    if (inHdr.f_magic != MC68MAGIC)
    	error ("error wrong magic number 0%o.", inHdr.f_magic);

    if (inHdr.f_opthdr)
	if (fread ((char *)&aoHdr, sizeof (aoHdr), 1, stdin) != 1)
	    error ("error reading optional file header.");


    if (inHdr.f_nscns != NUM_SCNS)
	error ("%d sections.", inHdr.f_nscns);

    if (fread ((char *)scnHdr, sizeof (SCNHDR), NUM_SCNS, stdin)
	!= NUM_SCNS)
	error ("error reading section headers.");


    if ((textSect = malloc ((UINT)scnHdr[TEXT].s_size)) == NULL)
    	error ("can't malloc text.");

    if (fread (textSect, (int) scnHdr[TEXT].s_size, 1, stdin) != 1)
    	error ("error reading text.");


    if ((data = malloc ((UINT)scnHdr[DATA].s_size)) == NULL)
    	error ("can't malloc data.");

    if (fread (data, (int) scnHdr[DATA].s_size, 1, stdin) != 1)
    	error ("error reading data.");


    nTextRel = scnHdr[TEXT].s_nreloc;
    nDataRel = scnHdr[DATA].s_nreloc;

    if (nTextRel > 0)
	{
	if ((relText = (RELOC *)malloc ((UINT)(nTextRel * RELSZ))) == NULL)
	    error ("can't malloc text relocation entries.");

	if ((oRelText = (RINFO *)malloc ((UINT)(nTextRel * sizeof (RINFO))))
	    == NULL)
	    error ("can't malloc output text relocation entries.");

	if (fread ((char *)relText, nTextRel * RELSZ, 1, stdin) != 1)
	    error ("error reading text relocation entries.");
	}


    if (nDataRel > 0)
	{
	if ((relData = (RELOC *)malloc ((UINT)(nDataRel * RELSZ))) == NULL)
	    error ("can't malloc data relocation entries.");

	if ((oRelData = (RINFO *)malloc ((UINT)(nDataRel * sizeof (RINFO))))
	    == NULL)
	    error ("can't malloc output data relocation entries.");

	if (fread ((char *)relData, nDataRel * RELSZ, 1, stdin) != 1)
	    error ("error reading data relocation entries.");
	}


    /* read symbol table and string table */

    if ((symTbl = (SYMENT *)malloc ((UINT)(inHdr.f_nsyms * SYMESZ))) == NULL)
    	error ("can't malloc symTbl.");

    /* make outSymTbl have same # of entries as symTbl, shrink later */

    if ((outSymTbl = (NLIST *)malloc ((UINT)(inHdr.f_nsyms * sizeof (NLIST))))
	== NULL)
    	error ("can't malloc outSymTbl.");

    if (fread ((char *)symTbl, (int) (inHdr.f_nsyms * SYMESZ), 1, stdin) != 1)
	error ("error reading symbol table.");

    if (fread ((char *) &strTblSize, sizeof (strTblSize), 1, stdin) != 1)
	strTblSize = 0;

    if (strTblSize > sizeof (strTblSize))	/* is there a string table? */
	{
	if ((strTbl = malloc ((UINT)strTblSize)) == NULL)
	    error ("can't malloc strTbl.");

	if (fread (&strTbl[4], (int) strTblSize - 4, 1, stdin) != 1)
	    error ("error reading string table.");
	}

    /* make outStrTbl a little oversized, shrink later as needed */

    if ((outStrTbl =
	malloc ((UINT)(strTblSize + (inHdr.f_nsyms * (SYMNMLEN + 1)))))
	 == NULL)
	error ("can't malloc outStrTbl.");

    /* zero out outStrTbl */

    for (stix = (strTblSize + (inHdr.f_nsyms * (SYMNMLEN + 1))) - 1;
	 stix >= 0;
	 --stix)
	outStrTbl[stix] = '\0';

    if ((stTranslator = (int *)malloc ((UINT)(inHdr.f_nsyms * sizeof (int))))
	== NULL)
	error ("unable to malloc string table translator.");

    xlateSymTbl ();	/* translate symbol table & string table */

    /* translate text and data relocation sections */

    xlateReloc (relText, oRelText, nTextRel, 0);
    xlateReloc (relData, oRelData, nDataRel, (int) scnHdr[TEXT].s_size);


    /* header */

    outHdr.a_magic = 0x107;
    outHdr.a_text = scnHdr[TEXT].s_size;
    outHdr.a_data = scnHdr[DATA].s_size;
    outHdr.a_bss = scnHdr[BSS].s_size;
    outHdr.a_syms = osymix * sizeof (NLIST);
    outHdr.a_entry = (inHdr.f_opthdr == 0) ? scnHdr[TEXT].s_paddr : aoHdr.entry;
    outHdr.a_trsize = nTextRel * sizeof (RINFO);
    outHdr.a_drsize = nDataRel * sizeof (RINFO);


    if (fwrite ((char *)&outHdr, sizeof (outHdr), 1, stdout) != 1)
        error ("error writing file header.");

    if (fwrite (textSect, (int) outHdr.a_text, 1, stdout) != 1)
        error ("error writing text.");

    if (fwrite (data, (int) outHdr.a_data, 1, stdout) != 1)
        error ("error writing data.");

    if (outHdr.a_trsize != 0) 
	{
        if (fwrite ((char *)oRelText, (int) outHdr.a_trsize, 1, stdout) != 1)
    	    error ("error writing text relocation.");
        }

    if (outHdr.a_drsize != 0) 
	{
        if (fwrite ((char *)oRelData, (int) outHdr.a_drsize, 1, stdout) != 1)
    	    error ("error writing data relocation.");
        }

    if (outHdr.a_syms != 0) 
	{
    	if (fwrite ((char *)outSymTbl, (int) outHdr.a_syms, 1, stdout) != 1)
    	    error ("error writing symbol table.");
    	if (fwrite (outStrTbl, stix, 1, stdout) != 1)
    	    error ("error writing string table.");
        }

    exit (OK);
    }
/*******************************************************************************
*
* xlateSymTbl - translate coff symbol table into BSD symbol table
*/

VOID xlateSymTbl ()

    {
    for (symix = osymix = 0, stix = 4; symix < inHdr.f_nsyms; ++symix)
	{
	if (symTbl[symix].n_numaux != 0)
	    {
	    /* skip entries with auxiliary entries */
	    stTranslator[symix] = NONE;
	    stTranslator[++symix] = NONE;
	    continue;
	    }

	stTranslator[symix] = osymix;		/* note the correspondence */

	outSymTbl[osymix].n_un.n_strx = stix;	/* point to string */

	outStrTbl[stix++] = '_';	/* prepend entry with underscore */

	/* get symbol name from symbol or string table, put in string table */

	if (symTbl[symix].n_zeroes == 0)
	    {
	    if (strTblSize == 0)
		error ("nonexistent source string table.");
	    sp = strcpy (&outStrTbl[stix], &strTbl[symTbl[symix].n_offset]);
	    }
	else
	    sp = strncpy (&outStrTbl[stix], symTbl[symix].n_name, SYMNMLEN);

	stix += strlen (sp) + 2;		/* '_' + chars copied + EOS */

	switch (symTbl[symix].n_scnum)
	    {
	    case N_ABSLT:
		outSymTbl[osymix].n_type = N_ABS;
		break;
	    case 1:
		outSymTbl[osymix].n_type = N_TEXT;
		break;
	    case 2:
		outSymTbl[osymix].n_type = N_DATA;
		break;
	    case 3:
		outSymTbl[osymix].n_type = N_BSS;
		break;
	    default:
		outSymTbl[osymix].n_type = N_UNDF;
		break;
	    }

	if (symTbl[symix].n_sclass == C_EXT)	/* external symbol? */
	    outSymTbl[osymix].n_type |= N_EXT;	/* set the external bit */

	outSymTbl[osymix].n_other = '\0';	/* don't care */
	outSymTbl[osymix].n_desc = 0;		/* don't care */

	/* copy the value of the symbol */

	outSymTbl[osymix].n_value = symTbl[symix].n_value;

#ifdef DEBUG
    	symPrint (&outSymTbl[osymix], outStrTbl);
#endif
	++osymix;
	}

    *((long *)(&outStrTbl[0])) = stix;	/* string table gets its length */
    }
/*******************************************************************************
*
* xlateReloc - translate relocation section
*/

VOID xlateReloc (inRel, outRel, numRel, offset)
    FAST RELOC *inRel;	/* pointer to array of coff relocation entries */
    FAST RINFO *outRel;	/* where to put BSD relocation entries */
    FAST int numRel;	/* number of relocation entries to translate */
    FAST int offset;	/* 0 for text rel, size of text for data rel */

    {
    FAST int ix;

    for (ix = 0; ix < numRel; ++ix)
	{
	outRel[ix].r_address = inRel[ix].r_vaddr - offset;

	switch (inRel[ix].r_type)
	    {
	    case R_RELBYTE:
		outRel[ix].r_pcrel = FALSE;
		outRel[ix].r_length = 0;
		break;
	    case R_RELWORD:
		outRel[ix].r_pcrel = FALSE;
		outRel[ix].r_length = 1;
		break;
	    case R_RELLONG:
		outRel[ix].r_pcrel = FALSE;
		outRel[ix].r_length = 2;
		break;
	    case R_PCRBYTE:
		outRel[ix].r_pcrel = TRUE;
		outRel[ix].r_length = 0;
		break;
	    case R_PCRWORD:
		outRel[ix].r_pcrel = TRUE;
		outRel[ix].r_length = 1;
		break;
	    case R_PCRLONG:
		outRel[ix].r_pcrel = TRUE;
		outRel[ix].r_length = 2;
		break;
	    }

	/* if not extern, relocation is relative to text segment origin */

	if ((int)(outRel[ix].r_symbolnum = stTranslator[inRel[ix].r_symndx])
	    == NONE)
	    {
	    if (symTbl[inRel[ix].r_symndx].n_scnum == 1)
		outRel[ix].r_symbolnum = N_TEXT;	/* rel to text seg */
	    else
		outRel[ix].r_symbolnum = N_DATA;	/* rel to data seg */
	    outRel[ix].r_extern = FALSE;
	    }
	else if (symTbl[inRel[ix].r_symndx].n_scnum == N_UNDEF)
	    {
	    outRel[ix].r_extern = TRUE;		/* find in existing sym table */

	    /* if the symbol table entry of an undefined external has a non-zero
	     * value, that value is the size of the object, and that size must
	     * be subtracted from the relocation target */

	    if (symTbl[inRel[ix].r_symndx].n_value != 0)
		{
		*((long *) (&textSect[inRel[ix].r_vaddr]))
		    -= symTbl[inRel[ix].r_symndx].n_value;
		}
	    }
	else
	    outRel[ix].r_extern = FALSE;

#ifdef DEBUG
	relPrint (&outRel[ix], outSymTbl, outStrTbl);
#endif
	}
    }
/*******************************************************************************
*
* error - print error message and die
*
* This routine prints an error message on the standard error output and
* aborts the program with the error status ERROR.  The error message is
* output with the specified format with the single specified argument.
*
* VARARGS1
*/

VOID error (format, arg)
    char *format;	/* format of error message */
    char *arg;		/* argument supplied to format (arbitrary type!) */

    {
    fprintf (stderr, "coffToBsd: ");
    fprintf (stderr, format, arg);
    fprintf (stderr, "\n");
    exit (ERROR);
    }

#ifdef DEBUG
/*******************************************************************************
*
* symPrint - print the BSD version of a symbol table entry
*/

VOID symPrint (sym, strtbl)
    NLIST *sym;
    char *strtbl;

    {
    static BOOL firstTime = TRUE;
    static char *headerString =
	"Symbol Name            StrIndex Type        Value\n";

    if (firstTime)
	{
	fprintf (stderr, headerString);
	firstTime = FALSE;
	}
    fprintf (stderr, "%-22s ", strtbl+sym->n_un.n_strx);
    fprintf (stderr, "%8x ", sym->n_un.n_strx);
    fprintf (stderr, "%2x ", sym->n_type);
    if (sym->n_type & N_STAB)
    	fprintf (stderr, "STAB ");
    else if ((sym->n_type & N_FN) == N_FN)
    	fprintf (stderr, "FN   ");
    else
	{
    	switch (sym->n_type & 0x1e) 
	    {
    	    case 0x0:
    		fprintf (stderr, "UNDF "); break;
    	    case 0x2:
    		fprintf (stderr, "ABS  "); break;
    	    case 0x4:
    		fprintf (stderr, "TEXT "); break;
    	    case 0x6:
    		fprintf (stderr, "DATA "); break;
    	    case 0x8:
    		fprintf (stderr, "BSS  "); break;
    	    case 0x12:
    		fprintf (stderr, "COMM "); break;
    	    }
    	if (sym->n_type & 0x1)
    	    fprintf (stderr, "EXT ");
	else
    	    fprintf (stderr, "    ");
    	}

    fprintf (stderr, "%8x\n", sym->n_value);
    }
/*******************************************************************************
*
* relPrint - print the BSD version of a relocation table
*/
    
VOID relPrint (rel, symtbl, strtbl)
    RINFO *rel;		/* relocation table entry */
    NLIST *symtbl;	/* symbol table to use */
    char *strtbl;	/* string table to use */

    {
    static BOOL firstTime = TRUE;
    static char *lenString[] = {"Byte", "Word", "Long"};
    static char *headerString =
	"\nSymbol Name            Address  SymNum PcRel Length Extern\n";

    if (firstTime)
	{
	fprintf (stderr, headerString);
	firstTime = FALSE;
	}

    if (rel->r_extern)
    	fprintf (stderr, "%-22s ", strtbl+symtbl[rel->r_symbolnum].n_un.n_strx);
    else
    	fprintf (stderr, "                     ");
    fprintf (stderr, "%8x ", rel->r_address);
    fprintf (stderr, "%6x ", rel->r_symbolnum);
    if (rel->r_pcrel)
	fprintf (stderr, "TRUE  ");
    else
	fprintf (stderr, "FALSE ");
    fprintf (stderr, " %s   ", lenString[rel->r_length]);
    if (rel->r_extern)
	fprintf (stderr, "TRUE  \n");
    else
	fprintf (stderr, "FALSE \n");
    }
#endif DEBUG
