/* wdLib.c - watchdog timer subroutine library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01j,01sep88,gae  documentation.
01i,04nov87,ecs  documentation.
01h,25mar87,jlf  documentation
01g,21dec86,dnw  changed to not get include files from default directories.
01f,05sep86,jlf  minor documentation.
01e,14apr86,rdc  changed memAllocates to mallocs. 
01d,07sep84,jlf  added copyright notice and comments.
01c,02aug84,dnw  changed calls to vxInt... to int...
01b,11jul84,ecs  changed calls to vxIntLock/vxIntUnlock to restore old level.
01a,22may84,dnw  written
*/

/*
This module provides a general watchdog timer facility.  Any task may
create a watchdog timer, then use it to provide events that can happen
after a specified delay, outside the context of the task itself.

Once a timer has been created with wdCreate, that timer can be started
with wdStart.  On the wdStart call, a timeout routine, an arbitrary
parameter, and a delay count are specified.  After the specified delay
time has elapsed, and if the timer has not been canceled with wdCancel,
the timeout routine will be invoked with the parameter which was on the
wdStart.  The timeout routine will be invoked
even if the task which started the watchdog is running, suspended or deleted.

Note that the timeout routine is invoked at interrupt level, rather than
in the context of the task.  Therefore, it must be careful about what it
can and can't do.

The routine wdLibInit must be called before any watchdogs are created
and the routine wdTick must be called on every clock tick.

EXAMPLE
.CS
    WDOG_ID wid = wdCreate ();
    wdStart (wid, 60, logMsg, "Help, I've timed out!");
    maybeSlowRoutine ();
    wdCancel (wid);
.CE
In the above fragment, if maybeSlowRoutine takes more than 60 ticks,
logMsg (2) will be called with the string as a parameter, which
will cause the message to be printed on the console.
Normally, of course, more significant corrective action would be taken.

INCLUDE FILE: wdLib.h

SEE ALSO: "Architecture", "Cross-Development", logLib (1), usrConfig (1)
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "lstLib.h"
#include "memLib.h"
#include "wdLib.h"


LOCAL LIST wdList;


/*******************************************************************************
*
* wdLibInit - initialize watchdog timer module
*
* Initialize the watchdog timer data structures.  This must be called
* before any other watchdog functions (including wdTick (2)).
* Normally, this will be done from usrConfig (1).
*/

VOID wdLibInit ()

    {
    lstInit (&wdList);
    }
/*******************************************************************************
*
* wdCreate - create a watchdog timer
*
* This routine creates a watchdog timer.  It allocates a WDOG structure for
* the timer, and puts that structure on the watchdog list.
*
* RETURNS:
*  ID for the watchdog created, or
*  NULL if out of memory
*/

WDOG_ID wdCreate ()
    
    {
    int oldlevel;		/* current interrupt level mask */
    WDOG_ID wd_id = (WDOG_ID) malloc (sizeof (WDOG));

    if (wd_id != NULL)
	{
	wd_id->wdCount = 0;

	oldlevel = intLock ();
	lstAdd (&wdList, &wd_id->wdNode);
	intUnlock (oldlevel);
	}

    return (wd_id);
    }
/*******************************************************************************
*
* wdStart - start up a watchdog timer
*
* This routine starts a watchdog timer by writing the specified count, routine
* pointer, and parameter into the specified watchdog.  The watchdog is actually 
* maintained at interrupt level.  Use wdCancel to stop the watchdog.
*
* SEE ALSO: wdCancel (2)
*/

VOID wdStart (wd_id, delay, pRoutine, parameter)
    FAST WDOG_ID wd_id;		/* watchdog id                          */
    int delay;			/* delay count, in ticks                */
    FUNCPTR pRoutine;		/* routine to call on time-out          */
    int parameter;		/* parameter with which to call routine */

    {
    wd_id->wdRoutine	= pRoutine;
    wd_id->wdParameter	= parameter;
    wd_id->wdCount	= delay;
    }
/*******************************************************************************
*
* wdCancel - cancel a currently counting watchdog
*
* This routine cancels a watchdog timer that is currently running, by
* zeroing it's delay count.
*
* SEE ALSO: wdStart (2)
*/

VOID wdCancel (wd_id)
    WDOG_ID wd_id;	/* id of watchdog to cancel */

    {
    wd_id->wdCount = 0;
    }
/*******************************************************************************
*
* wdTick - announce tick to watchdogs
*
* This routine is called from the clock interrupt to announce a tick
* to the watchdog timers.  All the watchdog delays are decremented.
* If a watchdog is decremented to zero, it's timeout routine is invoked
* with the specified parameter.
* 
* This routine is normally called in usrClock (2) in usrConfig (1).
*/

VOID wdTick ()

    {
    FAST WDOG_ID wd_id;

    for (wd_id = (WDOG *) lstFirst (&wdList); wd_id != NULL; 
				wd_id = (WDOG *) lstNext (&wd_id->wdNode))
	{
	if (wd_id->wdCount > 0)
	    {
	    if (--wd_id->wdCount == 0)
		(* wd_id->wdRoutine) (wd_id->wdParameter);
	    }
	}
    }
