/* tyLib.c - tty handler support library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
03k,14oct88,gae  fixed bug in FIOCANCEL; couldn't cancel twice in a row.
03j,07sep88,gae  documentation.
03i,01sep88,gae  documentation; changed RNG_MACRO parameters to int's.
03h,30may88,dnw  changed to v4 names.
03g,28may88,dnw  removed NOT_GENERIC stuff.
03f,04may88,jcf  changed semInits to semClear.
03e,29mar88,gae  added FIOISATTY to tyIoctl repertoire.
03d,26feb88,jcf  changed reboot to excToDoAdd (reboot ..) so reboot routines
		  can print stuff out.
03c,22feb88,dnw  fix coding conventions to keep f2cgen happy.
03b,23nov87,ecs  added VARARGS2 to tyIoctl to keep lint happy.
03a,23oct87,ecs  fixed mangen problems in tlRdRaw/tyRead & tyIRdRaw/tyIRd.
		 documentation.
	    jcf  changed call sysToMonitor into reboot
02z,03oct87,gae  added FIO[GS}ETOPTIONS.
02y,24jul87,gae  made ty{Backspace,DeleteLine,Eof}Char's global for shellLib.c.
02x,25jun87,ecs  changed tyIRdRaw to return ERROR on full ring buffer.
02w,05jun87,ecs  added FIOCANCEL to tyIoctl.
02v,13may87,dnw  fixed checksum bug caused by new "correct" compiler.
	   +gae 
02u,23mar87,jlf  documentation.
02t,23feb87,jlf  improved documentation of tyAbortFunc.
...deleted pre 87 history - see RCS
*/

/*
This library is meant to be used by ty drivers.  It handles all the
device independent functions of a normal serial device, such as XON-XOFF
processing, CR-LF conversion, and line editing functions.  Routines
contained in this library are only for use by drivers - there are no 
functions here that would normally be needed by a user.

For each device that uses this library, there must be a device, in the I/O
system's device list.  That device must contain, as part of it, a TY_DEV,
a pointer to which is a parameter to almost all the routines in this
library.  TyDevInit should be called exactly once to
initialize this TY_DEV.

Thereafter, various routines in this library may be called to service requests.

INCLUDE FILE: tyLib.h

SEE ALSO: "I/O System", ioLib (1), iosLib (1), tyCoDrv (3)
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "ioLib.h"
#include "memLib.h"
#include "rngLib.h"
#include "wdLib.h"
#include "tyLib.h"
#include "sysLib.h"


IMPORT VOID reboot ();

/* special characters */

#define XON		0x11	/* ctrl-Q XON handshake */
#define XOFF		0x13	/* ctrl-S XOFF handshake */

LOCAL int tyXoffChars = 0;	/** TEMP **/
LOCAL int tyXoffMax   = 0;	/** TEMP **/

/* these are global only for the shell */

char tyBackspaceChar	= 0x08;	/* default is control-H */
char tyDeleteLineChar	= 0x15;	/* default is control-U */
char tyEofChar		= 0x04;	/* default is control-D */

LOCAL char tyAbortChar	= 0x03;	/* default is control-C */
LOCAL char tyMonTrapChar= 0x18;	/* default is control-X */

LOCAL int tyXoffThreshold = 80;	/* max bytes free in input buffer before
				 * XOFF will be sent in OPT_TANDEM mode */
LOCAL int tyXonThreshold  = 100;/* min bytes free in input buffer before
				 * XON will be sent in OPT_TANDEM mode */
LOCAL int tyWrtThreshold  = 20;	/* min bytes free in output buffer before
				 * the next writer will be enabled */

LOCAL FUNCPTR tyAbortFunc = NULL;	/* function to call when abort char received */

/*******************************************************************************
*
* tyDevInit - initialize ty device descriptor
*
* This routine initializes a ty device descriptor according to the
* specified parameters.  The initialization includes allocating read and
* write buffers of the specified sizes from the memory pool, 
* and initializing their respective buffer descriptors.
* The semaphores are initialized and the write semaphore 'given'
* to enable writers.  Also the transmitter startup routine pointer is set
* to the specified routine.  All other fields in the descriptor are zeroed.
*
* This routine should only be called by serial drivers.
*
* RETURNS
*  OK, or
*  ERROR if not enough memory to allocate data structures
*/

STATUS tyDevInit (pTyDev, rdBufSize, wrtBufSize, txStartup)
    FAST TY_DEV_ID pTyDev; /* pointer to ty device descriptor     */
                           /* to be initialized                   */
    int rdBufSize;         /* required read buffer size in bytes  */
    int wrtBufSize;        /* required write buffer size in bytes */
    FUNCPTR txStartup;     /* device transmit startup routine     */

    {
    /* clear out device */

    bzero ((char *) pTyDev, sizeof (TY_DEV));

    /* allocate read and write ring buffers */

    if ((pTyDev->wrtBuf = rngCreate (wrtBufSize)) == NULL)
	return (ERROR);

    if ((pTyDev->rdBuf = rngCreate (rdBufSize)) == NULL)
	return (ERROR);


    /* initialize rest of device descriptor */

    pTyDev->txStartup = txStartup;

    pTyDev->rdSemId = semCreate ();
    pTyDev->wrtSemId = semCreate ();

    tyFlush (pTyDev);

    return (OK);
    }
/*******************************************************************************
*
* tyFlush - clear out a ty device descriptors buffers
*
* This routine resets a ty device's buffers to be empty and in the
* "initial" state.  Thus any characters input but not read by a task,
* and those written by a task but not yet output, are lost.
*/

LOCAL VOID tyFlush (pTyDev)
    FAST TY_DEV_ID pTyDev;	/* ptr to ty device descriptor to be cleared */

    {
    tyFlushRd (pTyDev);
    tyFlushWrt (pTyDev);
    }
/*******************************************************************************
*
* tyFlushRd - clear out a ty device descriptors read buffer
*/

LOCAL VOID tyFlushRd (pTyDev)
    FAST TY_DEV_ID pTyDev;	/* ptr to ty device descriptor to be cleared */

    {
    int level = intLock ();		/** disable interrupts **/

    rngFlush (pTyDev->rdBuf);
    semClear (pTyDev->rdSemId);
    pTyDev->rdState.semNeeded = TRUE;

    /* output an XON if necessary */

    if (pTyDev->rdState.xoff)
	{
	pTyDev->rdState.xoff = FALSE;
	pTyDev->rdState.pending = !pTyDev->rdState.pending;
	if (pTyDev->rdState.pending && !pTyDev->wrtState.busy)
	    (*pTyDev->txStartup) (pTyDev);
	}

    pTyDev->lnNBytes     = 0;
    pTyDev->lnBytesLeft  = 0;

    intUnlock (level);			/** re-enable interrupts **/
    }
/*******************************************************************************
*
* tyFlushWrt - clear out a ty device descriptors write buffer
*/

LOCAL VOID tyFlushWrt (pTyDev)
    FAST TY_DEV_ID pTyDev;	/* ptr to ty device descriptor to be cleared */

    {
    int level = intLock ();		/** disable interrupts **/

    rngFlush (pTyDev->wrtBuf);
    semClear (pTyDev->wrtSemId);
    semGive (pTyDev->wrtSemId);
    pTyDev->wrtState.semNeeded = FALSE;

    intUnlock (level);			/** re-enable interrupts **/
    }
/*******************************************************************************
*
* tyAbortFuncSet - set abort function
*
* This routine sets the function that will be called when the abort
* character is received on a tty.  There is only one global abort function,
* used for any ty on which OPT_ABORT is enabled.
* When the abort character is received from a terminal with OPT_ABORT
* set, the function specified here will be called, with no parameters, from
* interrupt level.
*
* Setting an abort function of NULL will disable the abort function.
*
* SEE ALSO: tyAbortSet (2)
*/

VOID tyAbortFuncSet (func)
    FUNCPTR func;	/* function to call when abort char is received */

    {
    tyAbortFunc = func;
    }
/*******************************************************************************
*
* tyAbortSet - change abort character
*
* This routine sets the abort character to the specified character.
* Typing the abort character to any device whose OPT_ABORT option is set
* will cause the shell task to be killed and restarted.
* Note that the character set by this routine applies to all devices
* whose handlers use the standard "tty" package "tyLib".
*
* The default abort character is ^C.
*
* SEE ALSO: tyAbortFuncSet (2)
*/

VOID tyAbortSet (ch)
    char ch;		/* character to be made the abort character */

    {
    tyAbortChar = ch;
    }
/*******************************************************************************
*
* tyBackspaceSet - change backspace character
*
* This routine sets the backspace character to the specified character.
* Typing the backspace character to any device operating in "line protocol"
* mode (OPT_LINE option set) will cause the previous character typed to be
* deleted, up to the beginning of the current line.
* Note that the character set by this routine applies to all devices
* whose handlers use the standard "tty" package "tyLib".
*
* Default backspace character is ^H.
*/

VOID tyBackspaceSet (ch)
    char ch;		/* character to be made the backspace character */

    {
    tyBackspaceChar = ch;
    }
/*******************************************************************************
*
* tyDeleteLineSet - change line-delete character
*
* This routine sets the delete character to the specified character.
* Typing the delete character to any device operating in "line protocol"
* mode (OPT_LINE option set) will cause all characters in the current
* line to be deleted.
* Note that the character set by this routine applies to all devices
* whose handlers use the standard "tty" package "tyLib".
*
* Default delete-line character is ^U.
*/

VOID tyDeleteLineSet (ch)
    char ch;		/* character to be made the line-delete character */

    {
    tyDeleteLineChar = ch;
    }
/*******************************************************************************
*
* tyEOFSet - change end-of-file character
*
* This routine sets the EOF character to the specified character.
* Typing the EOF character to any device operating in "line protocol"
* mode (OPT_LINE option set) will cause no character to be entered in
* the current line, but will cause the current line to be terminated
* (thus without a newline character).  The line is made available to
* reading tasks.  Thus if the EOF character is the first character input
* on a line, a line length of zero characters is returned to the reader.
* This is the standard end-of-file indication on a read call.
* Note that the character set by this routine applies to all devices
* whose handlers use the standard "tty" package "tyLib".
*
* Default EOF char is ^D.
*/

VOID tyEOFSet (ch)
    char ch;		/* character to be made the eof character */

    {
    tyEofChar = ch;
    }
/*******************************************************************************
*
* tyMonitorTrapSet - change monitor trap character
*
* This routine sets the trap-to-monitor character to the specified character.
* Typing the trap-to-monitor character to any device whose OPT_MON_TRAP
* option is set will cause the resident rom-monitor to be entered,
* if one exists in the system.  Once the rom-monitor is entered, 
* the normal vxWorks multitasking system is halted.  
*
* The default monitor trap character is ^X.
*
* Note that the character set by this routine applies to all devices
* whose handlers use the standard "tty" package "tyLib".  Also note that
* not all systems have a monitor trap available.
*/

VOID tyMonitorTrapSet (ch)
    char ch;		/* character to be made the monitor trap character */

    {
    tyMonTrapChar = ch;
    }
/*******************************************************************************
*
* tyIoctl - special device control
*
* This routine handles device control requests for "tty" devices.
* It handles the following requests:
*
*   ioctl (fd, FIONREAD, &nBytes)
*	return in nBytes the number of characters 
*	available in the input buffer
*
*   ioctl (fd, FIONWRITE, &nBytes)
*	return in nBytes the number of characters 
*	available in the output buffer
*
*   ioctl (fd, FIOFLUSH)
*       discard all characters currently in input and 
*	output buffers
*
*   ioctl (fd, FIOSETOPTIONS, options)
*   ioctl (fd, FIOOPTIONS, options)
*       set device options. If the line-protocol (OPT_LINE) 
*	is changed, the input buffer is flushed.
*	The various options are described in tyLib.h
*
*   ioctl (fd, FIOGETOPTIONS)
*	returns the current device options
*
*   ioctl (fd, FIOCANCEL)
*       cancel read or write
*
*   ioctl (fd, FIOISATTY)
*       returns TRUE
*
* Any other request will return ERROR, and set status to
* S_ioLib_UNKNOWN_REQUEST.
*
* BUGS:
* In line-protocol mode (OPT_LINE option set) FIONREAD actually returns
* the number of characters available PLUS the number of lines in the buffer.
* Thus if five lines consisting of just newlines were in the input buffer,
* FIONREAD would return the value ten (five characters + five lines).
*
* VARARGS2 - not all requests include an arg.
*/

STATUS tyIoctl (pTyDev, request, arg)
    FAST TY_DEV_ID pTyDev;	/* pointer to device to control */
    int request;		/* request code                 */
    int arg;			/* some argument                */

    {
    FAST int status = OK;
    int oldOptions;
    int oldlevel;		/* current interrupt level mask */

    switch (request)
	{
	case FIONREAD:
	    *((int *) arg) = rngNBytes (pTyDev->rdBuf);
	    break;

	case FIONWRITE:
	    *((int *) arg) = rngNBytes (pTyDev->wrtBuf);
	    break;

	case FIOFLUSH:
	    tyFlush (pTyDev);
	    break;

	case FIOGETOPTIONS:
	    return (pTyDev->options);

	case FIOSETOPTIONS:
	/*
	case FIOOPTIONS:
	*/
	    oldOptions = pTyDev->options;
	    pTyDev->options = arg;

	    if ((oldOptions & OPT_LINE) != (pTyDev->options & OPT_LINE))
		tyFlushRd (pTyDev);

	    if ((oldOptions & OPT_TANDEM) && !(pTyDev->options & OPT_TANDEM))
		{
		/* TANDEM option turned off: reset read and write states */

		oldlevel = intLock ();		/** disable interrupts **/

		/* output an XON if necessary */

		if (pTyDev->rdState.xoff)
		    {
		    pTyDev->rdState.xoff = FALSE;
		    pTyDev->rdState.pending = !pTyDev->rdState.pending;
		    if (pTyDev->rdState.pending && !pTyDev->wrtState.busy)
			    (*pTyDev->txStartup) (pTyDev);
		    }

		/* restart an XOFF'd transmitter */

		if (pTyDev->wrtState.xoff)
		    {
		    pTyDev->wrtState.xoff = FALSE;
		    if (!pTyDev->wrtState.busy)
			(*pTyDev->txStartup) (pTyDev);
		    }

		intUnlock (oldlevel);		/** re-enable interrupts **/
		}
	    break;

	case FIOCANCEL:
	    if (pTyDev->rdState.semNeeded)
		{
		pTyDev->rdState.canceled = TRUE;
		semGive (pTyDev->rdSemId);
		}

	    if (pTyDev->wrtState.semNeeded)
		{
		pTyDev->wrtState.canceled = TRUE;
		semGive (pTyDev->wrtSemId);
		}
	    break;

	case FIOISATTY:
	    status = TRUE;
	    break;

	default:
	    errnoSet (S_ioLib_UNKNOWN_REQUEST);
	    status = ERROR;
	}

    return (status);
    }

/* raw mode I/O routines */

/*******************************************************************************
*
* tyWrite - task-level write routine for ttys
*
* This routine handles the task-level portion of the tty handler
* write function.
*
* RETURNS: number of bytes actually written to device
*/

int tyWrite (pTyDev, buffer, nbytes)
    FAST TY_DEV_ID pTyDev;	/* pointer to device structure */
    char *buffer;		/* buffer of data to write     */
    FAST int nbytes;		/* number of bytes in buffer   */

    {
    FAST int bytesput;
    int oldlevel;		/* current interrupt level mask */
    int nbStart = nbytes;

    pTyDev->wrtState.canceled = FALSE;

    while (nbytes > 0)
	{
	semTake (pTyDev->wrtSemId); 

	if (pTyDev->wrtState.canceled)
	    {
	    errnoSet (S_ioLib_CANCELED);
	    return (nbStart - nbytes);
	    }

	bytesput = rngBufPut (pTyDev->wrtBuf, buffer, nbytes);

	oldlevel = intLock ();			/** disable interrupts **/

	/* if xmitter not busy, start it */

	if (!pTyDev->wrtState.busy)
	    (*pTyDev->txStartup) (pTyDev);

	/* If more room in ringId, enable next writer. */

	if (rngFreeBytes (pTyDev->wrtBuf) > tyWrtThreshold)
	    semGive (pTyDev->wrtSemId);
	else
	    pTyDev->wrtState.semNeeded = TRUE;

	intUnlock (oldlevel);			/** re-enable interrupts **/

	nbytes -= bytesput;
	buffer += bytesput;
	}

    return (nbStart);
    }
/*******************************************************************************
*
* tyRead - task level read routine for ttys
*
* Handles the task level portion of the tty handler read function.
* Reads as many bytes as are available, but not more than maxbytes, into the
* buffer passed as an argument.
*
* This should only be called from serial device drivers.
*
* RETURNS: number of bytes actually read into buffer
*/

int tyRead (pTyDev, buffer, maxbytes)
    FAST TY_DEV_ID pTyDev;	/* device to read         */
    char *buffer;		/* buffer to read into    */
    int maxbytes;		/* maximum length of read */

    {
    FAST int nbytes;
    int oldlevel;		/* current interrupt level mask */
    FAST RING_ID ringId;
    FAST int n;
    int freeBytes;

    pTyDev->rdState.canceled = FALSE;

    semTake (pTyDev->rdSemId);

    if (pTyDev->rdState.canceled)
	{
	errnoSet (S_ioLib_CANCELED);
	return (0);
	}

    ringId = pTyDev->rdBuf;

    if (pTyDev->options & OPT_LINE)
	{
	if (pTyDev->lnBytesLeft == 0)
	    RNG_ELEM_GET (ringId, &pTyDev->lnBytesLeft, n);

	nbytes = rngBufGet (ringId, buffer,
			    min ((int)pTyDev->lnBytesLeft, maxbytes));

	pTyDev->lnBytesLeft -= nbytes;
	}

    else
	nbytes = rngBufGet (ringId, buffer, maxbytes);


    oldlevel = intLock ();		/** disable interrupts **/

    /* check if XON needs to be output */

    if ((pTyDev->options & OPT_TANDEM) && pTyDev->rdState.xoff)
	{
	freeBytes = rngFreeBytes (ringId);
	if (pTyDev->options & OPT_LINE)
	    freeBytes -= pTyDev->lnNBytes + 1;

	if (freeBytes > tyXonThreshold)
	    {
	    pTyDev->rdState.xoff = FALSE;
	    pTyDev->rdState.pending = !pTyDev->rdState.pending;
	    if (pTyDev->rdState.pending && !pTyDev->wrtState.busy)
		(*pTyDev->txStartup) (pTyDev);
	    }
	}

    /* If more characters in ringId, enable next reader. */

    if (rngIsEmpty (ringId))
	pTyDev->rdState.semNeeded = TRUE;
    else
	semGive (pTyDev->rdSemId);

    intUnlock (oldlevel);		/** re-enable interrupts **/

    return (nbytes);
    }
/*******************************************************************************
*
* tyITx - interrupt level output
*
* Get single character to be output to a device.  This routine looks
* at the ring buffer for the specified device and gives the caller the next
* available character, if there is one.
*
* RETURNS:
*  OK if more chars to send, or
*  ERROR if no more chars
*  (location pointed at by pChar gets the character to be output)
*/

STATUS tyITx (pTyDev, pChar)
    FAST TY_DEV_ID pTyDev;   /* pointer to tty device descriptor        */
    char *pChar;            /* ptr where to put character to be output */

    {
    FAST RING_ID ringId = pTyDev->wrtBuf;
    FAST int n;

    /* check if we need to output XON/XOFF for the read side */

    if (pTyDev->rdState.pending)
	{
	if (pTyDev->rdState.xoff)		/** TEMP **/
	    {					/** TEMP **/
	    if (tyXoffChars > tyXoffMax)	/** TEMP **/
		tyXoffMax = tyXoffChars;	/** TEMP **/
	    tyXoffChars = 0;			/** TEMP **/
	    }					/** TEMP **/
	*pChar = pTyDev->rdState.xoff ? XOFF : XON;
	pTyDev->rdState.pending = FALSE;
	pTyDev->wrtState.busy = TRUE;
	}

    /* check if output is in XOFF state before we return any char */

    else if (pTyDev->wrtState.xoff)
	pTyDev->wrtState.busy = FALSE;

    /* check for carriage return needed after linefeed */

    else if (pTyDev->wrtState.cr)
	{
	*pChar = '\r';
	pTyDev->wrtState.cr   = FALSE;
	pTyDev->wrtState.busy = TRUE;
	}

    /* check for no more characters to output */

    else if (RNG_ELEM_GET (ringId, pChar, n) == 0)	/* no more chars */
	pTyDev->wrtState.busy = FALSE;

    else
	{
	/* got a character to be output */

	pTyDev->wrtState.busy = TRUE;

	/* check for carriage return needs to be added after linefeed */

	if ((pTyDev->options & OPT_CRMOD) && (*pChar == '\n'))
	    pTyDev->wrtState.cr = TRUE;

	/* if task level needs semaphore, give it */

	if (pTyDev->wrtState.semNeeded && rngFreeBytes(ringId) > tyWrtThreshold)
	    {
	    pTyDev->wrtState.semNeeded = FALSE;
	    semGive (pTyDev->wrtSemId);
	    }
	}

    return (pTyDev->wrtState.busy ? OK : ERROR);
    }
/*******************************************************************************
*
* tyIRd - interrupt level input
*
* Interrupt level character input processing routine for ttys.
* A device driver calls this routine when it has received a character.  This
* routine adds the character to the ring buffer for the specified device and, if
* task level is waiting for a semaphore, gives the semaphore.
*
* This routine also handles all the special characters, as specified in
* the option word for the device, such as X-ON, X-OFF, nextline, backspace,
* etc.
*
* RETURNS: OK, or ERROR if ring buffer full
*/

STATUS tyIRd (pTyDev, inchar)
    FAST TY_DEV_ID pTyDev;	/* pointer to tty device descriptor */
    FAST char inchar;		/* character read                   */

    {
    FAST RING_ID ringId;
    FAST int n;
    int freeBytes;
    BOOL releaseTaskLevel;
    FAST int options = pTyDev->options;
    BOOL charEchoed = FALSE;
    STATUS status = OK;

    /* strip off parity bit if '7 bit' option set */

    if (options & OPT_7_BIT)
	inchar &= 0x7f;

    /* check for abort */

    if ((options & OPT_ABORT) && inchar == tyAbortChar && tyAbortFunc != NULL)
	(*tyAbortFunc) ();

    /* check for trap to rom monitor */

    else if ((options & OPT_MON_TRAP) && (inchar == tyMonTrapChar))
	excJobAdd (reboot, BOOT_WARM_AUTOBOOT);

    /* check for XON/XOFF received */

    else if ((options & OPT_TANDEM) && (inchar == XOFF))
	pTyDev->wrtState.xoff = TRUE;

    else if ((options & OPT_TANDEM) && (inchar == XON))
	{
	pTyDev->wrtState.xoff = FALSE;
	if (!pTyDev->wrtState.busy)
	    (*pTyDev->txStartup) (pTyDev);
	}
    else
	{
	if (pTyDev->rdState.xoff)	/** TEMP **/
	    tyXoffChars++;		/** TEMP **/

	/* check for carriage return needs to be turned into linefeed */

	if ((options & OPT_CRMOD) && (inchar == '\r'))
	    inchar = '\n';

	/* check for output echo required */

	if (options & OPT_ECHO)
	    {
	    ringId = pTyDev->wrtBuf;

	    /* echo the char.  some special chars are echoed differently */

	    if (options & OPT_LINE)
		{
		if (inchar == tyDeleteLineChar)
		    {
		    /* echo a newline */

		    RNG_ELEM_PUT (ringId, '\n', n);
		    charEchoed = TRUE;
		    }

		else if (inchar == tyBackspaceChar)
		    {
		    if (pTyDev->lnNBytes != 0)
			{
			/* echo BS-space-BS */

			rngBufPut (ringId, " ", 3);
			charEchoed = TRUE;
			}
		    }

		else if ((inchar < 0x20) && (inchar != '\n'))
		    {
		    /* echo ^-char */

		    RNG_ELEM_PUT (ringId, '^', n);
		    RNG_ELEM_PUT (ringId, inchar + '@', n);
		    charEchoed = TRUE;
		    }

		else
		    {
		    RNG_ELEM_PUT (ringId, inchar, n);
		    charEchoed = TRUE;
		    }
		}
	    else
		/* just echo the char */
		{
		RNG_ELEM_PUT (ringId, inchar, n);
		charEchoed = TRUE;
		}

	    if ((!pTyDev->wrtState.busy) && charEchoed)
		(*pTyDev->txStartup) (pTyDev);
	    }


	/* put the char in the read buffer. */

	ringId = pTyDev->rdBuf;
	releaseTaskLevel = FALSE;

	if (!(options & OPT_LINE))
	    {
	    /* not line-mode; 
	     * just enter character and make immediately available to tasks */

	    if (RNG_ELEM_PUT (ringId, inchar, n) == 0)
		status = ERROR;			/* buffer full, not entered */

	    releaseTaskLevel = TRUE;
	    }
	else
	    {
	    /* line-mode;
	     * process backspace, line-delete, EOF, and newline chars special */

	    freeBytes = rngFreeBytes (ringId);

	    if (inchar == tyBackspaceChar)
		{
		if (pTyDev->lnNBytes != 0)
		    pTyDev->lnNBytes--;
		}
	    else if (inchar == tyDeleteLineChar)
		pTyDev->lnNBytes = 0;
	    else if (inchar == tyEofChar)
		{
		if (freeBytes > 0)
		    releaseTaskLevel = TRUE;
		}
	    else
		{
		if (freeBytes >= 2)
		    {
		    if (freeBytes >= (pTyDev->lnNBytes + 2))
			pTyDev->lnNBytes++;
		    else
			status = ERROR;	/* no room, overwriting last char */

		    rngPutAhead (ringId, inchar, (int)pTyDev->lnNBytes);

		    if ((inchar == '\n') || (pTyDev->lnNBytes == 255))
			releaseTaskLevel = TRUE;
		    }
		else
		    status = ERROR;	/* no room, not even for overwriting */
		}

	    /* if line termination indicated, put line byte count
	     * in 0th character of line and advance ring buffer pointers */

	    if (releaseTaskLevel)
		{
		rngPutAhead (ringId, (char) pTyDev->lnNBytes, 0);
		rngMoveAhead (ringId, (int) pTyDev->lnNBytes + 1);
		pTyDev->lnNBytes = 0;
		}
	    }


	/* check if XON/XOFF needs to be output */

	if (options & OPT_TANDEM)
	    {
	    freeBytes = rngFreeBytes (ringId);
	    if (pTyDev->options & OPT_LINE)
		freeBytes -= pTyDev->lnNBytes + 1;

	    if (!pTyDev->rdState.xoff)
		{
		/* if input buffer is close to full, send XOFF */

		if (freeBytes < tyXoffThreshold)
		    {
		    pTyDev->rdState.xoff = TRUE;
		    pTyDev->rdState.pending = !pTyDev->rdState.pending;
		    if (pTyDev->rdState.pending & !pTyDev->wrtState.busy)
			(*pTyDev->txStartup) (pTyDev);
		    }
		}
	    else
		{
		/* if input buffer has enough room now, send XON */

		if (freeBytes > tyXonThreshold)
		    {
		    pTyDev->rdState.xoff = FALSE;
		    pTyDev->rdState.pending = !pTyDev->rdState.pending;
		    if (pTyDev->rdState.pending & !pTyDev->wrtState.busy)
			(*pTyDev->txStartup) (pTyDev);
		    }
		}
	    }


	/* if task level needs semaphore, give it */

	if (releaseTaskLevel && pTyDev->rdState.semNeeded)
	    {
	    pTyDev->rdState.semNeeded = FALSE;
	    semGive (pTyDev->rdSemId);
	    }
	}

    return (status);
    }
