/* timexLib.c - function execution timer utility */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
03f,20aug88,gae  documentation.
03e,07jul88,jcf  lint.
03d,05jun88,dnw  changed name to timexLib.
03c,30may88,dnw  changed to v4 names.
03b,01may88,gae  added null routine timLib().
03a,27jan88,jcf  made kernel independent.
02s,18nov87,ecs  documentation.
02r,04nov87,ecs  documentation.
02q,08jun87,llk  fixed initialization of tiPreCalls, tiTimeCalls, tiPostCalls.
02p,23mar87,jlf  documentation.
02o,12jan87,llk  corrected information in tihelp.
02n,21dec86,dnw  changed to not get include files from default directories.
02m,02dec86,llk  changed tiHelp to tihelp.
02l,04sep86,jlf  documentation.
02k,27jul86,llk  prints error messages to standard error (uses printErr)
02j,05jun86,dnw  changed sstLib calls to symLib.
02i,24mar86,dnw  de-linted.
02h,19mar86,dnw  fixed calibration to include pushing of args and calling
		   routine.
		 corrected documentation.
		 changed timeN to report how many reps it made.
02g,11oct85,dnw  de-linted.
02f,37aug85,rdc  changed MAX_SYM_LEN to MAX_SYS_SYM_LEN.
02e,21jul85,jlf  documentation.
02d,01jun85,rdc  updated documentation.
02c,14aug84,jlf  changed calls to clkGetRate to sysClkGetRate. 
02b,16jul84,ecs  changed to use unix-style format codes.
02a,24jun84,dnw  rewritten
*/


/*
TimexLib provides a set of routines which may be used to 
time the execution of programs, individual routines and arbitrary groups 
of routines.  The VxWorks system clock is used as a time base to measure 
execution times.  Routines which have a short execution time relative to 
this time base may be called repetitively to establish an average execution 
time with an acceptable error percentage. 

Up to 4 routines may be specified as the group of functions to be
timed.  Additionally, sets of up to 4 routines may be specified as pre 
or post functions which are executed before and after the timed routines 
respectively.
timexPre and timexPost are used to specify pre and post functions,
while timexFunc is used to specify the routines to be timed.  

The routine timex (2) is used to time one iteration of a function.
If no argument list is given, timex uses the functions specified in the
timexPre, timexPost and timexFunc calls.
If arguments are supplied, they are used in lieu of the
functions previously specified.  The routine timexN (2) may be used to iterate
function calls; like timex, arguments supplied in the timexN call are used
instead of previously defined functions.

EXAMPLES

timex may be used to obtain the execution time of a routine:
.CS
    timex (myFunc, myArg1, myArg2);
.CE
timexN calls a function repeatedly until a set error tolerence is obtained:
.CS
    timexN (myFunc, myArg1, myArg2);
.CE
timexPre, timexPost, and timexFunc may be used to specify groups of functions:
.CS
    timexPre (0, myFirstPreFunc, preArg1, preArg2);
    timexPre (1, mySecondPreFunc, preArg3);

    timexFunc (0, myFunc1, myArg1, myArg2);
    timexFunc (1, myFunc2, myArg3);
    timexFunc (2, myFunc3, myarg4, myArg5, myArg6);

    timexPost (0, myOnlyPostFunc, postArg);

    timex ();
	 or
    timexN ();
.CE
When timex or timexN are called, myFirstPreFunc and mySecondPreFunc are 
called with their respective arguments.  MyFunc1, myFunc2, and myFunc3 
are then called in sequence, and timed.  If timexN was used, the sequence 
is called repeatedly until a 2% or better error tolerence is achieved.
Finally, myOnlyPostFunc is called with its argument.  The timing
results are reported after the post-functions are called.

NOTE
The timings measure the time of execution of the routine body, without
the usual subroutine entry and exit code (usually just LINK, UNLINK, and
RTS instructions).  Also the time required to set up the arguments 
and call the routines is not included in the reported times.  
This is because these timing routines automatically calibrate themselves 
by timing the invocation of a null routine, 
and thereafter subtracting that constant overhead.

SEE ALSO: spyLib (1)
*/

#include "vxWorks.h"
#include "symLib.h"
#include "sysSymTbl.h"

#define MAX_CALLS	4		/* max functions in each category */
#define MAX_ARGS	8		/* max args to each function */

#define MAX_PERCENT	2		/* max percent error for auto-timing */
#define MAX_REPS	20000		/* max reps for auto-timing */

IMPORT ULONG tickGet ();		/* to get time in ticks */

typedef struct			/* CALL */
    {
    FUNCPTR func;			/* function to call */
    int arg [MAX_ARGS];			/* args to function */
    } CALL;

typedef CALL CALL_ARRAY [MAX_CALLS];

LOCAL char *timexScaleText [] =
    {
    "secs",			/* scale = 0 */
    "millisecs",		/* scale = 1 */
    "microsecs",		/* scale = 2 */
    };


LOCAL VOID timexNull ();	/* forward declaration */

LOCAL CALL_ARRAY timexPreCalls =	/* calls to be made just before timing*/
    {
    { timexNull },
    { timexNull },
    { timexNull },
    { timexNull },
    };

LOCAL CALL_ARRAY timexTimeCalls =	/* calls to be timed */
    {
    { timexNull },
    { timexNull },
    { timexNull },
    { timexNull },
    };

LOCAL CALL_ARRAY timexPostCalls =	/* calls to be made just after timing */
    {
    { timexNull },
    { timexNull },
    { timexNull },
    { timexNull },
    };

LOCAL CALL_ARRAY timexNullCalls =
    {
    { timexNull },
    { timexNull },
    { timexNull },
    { timexNull },
    };

LOCAL int overhead;		/* usecs of overhead per rep in timing test */

/*******************************************************************************
*
* timexInit - include timex library
*
* This null routine is provided so that timexLib can be linked in.
*/

VOID timexInit ()
    {
    }
/*******************************************************************************
*
* timexClear - clear timex function calls
*
* This routine clears all the lists of functions involved in the timing.
*/

VOID timexClear ()

    {
    timexClrArrays ();
    timexShow ();
    }
/*******************************************************************************
*
* timexFunc - specify a function to be timed
*
* This routine allows alteration of the list of functions to be timed.
* Any function in the list of 4 functions can be added or deleted.
*/

VOID timexFunc (i, func, arg1)
    int i;        /* function number in list (0..3)               */
    FUNCPTR func; /* function to be added (NULL if to be deleted) */
    int arg1;     /* first of up to 8 params called with function */

    {
    timexAddCall (timexTimeCalls, i, func, &arg1);
    timexShow ();
    }
/*******************************************************************************
*
* timexHelp - print help screen for timex
*
* This routine prints a help screen which shows all the available
* execution timer functions.
*
* .CS
*  -> timexHelp
*               
*  timexHelp                      Print this list
*  timex       [func,[args...]]   Time single execution
*  timexN      [func,[args...]]   Time repeated execution
*  timexClear                     Clear all functions
*  timexFunc   i,func,[args...]   Add timed function 'i'
*  timexPre    i,func,[args...]   Add pre function 'i' (before timing)
*  timexPost   i,func,[args...]   Add post function 'i' (after timing)
*  timexShow                      Show all functions to be called
*
*  Notes:
*    1) timexN will repeat calls enough times to get
*       timing accuracy to approximately 2%.
*    2) one function can be specified on timex and timexN;
*       alternatively functions can be pre-set with timexFunc.
*    3) up to 4 functions can be pre-set with timexFunc,
*       timexPre and timexPost, i.e. 'i' in range 0..3.
*    4) timexPre and timexPost allow locking/unlocking, or
*       raising/lowering priority before/after timing.
*  value = 1 = 0x1
*  -> 
* .CE
*/

VOID timexHelp ()

    {
    static char *helpMsg [] =
    {
    "timexHelp                      Print this list",
    "timex       [func,[args...]]   Time single execution",
    "timexN      [func,[args...]]   Time repeated execution",
    "timexClear                     Clear all functions",
    "timexFunc   i,func,[args...]   Add timed function 'i'",
    "timexPre    i,func,[args...]   Add pre function 'i' (before timing)",
    "timexPost   i,func,[args...]   Add post function 'i' (after timing)",
    "timexShow                      Show all functions to be called\n",
    "Notes:",
    "  1) timexN will repeat calls enough times to get",
    "     timing accuracy to approximately 2%.",
    "  2) one function can be specified on timex and timexN;",
    "     alternatively functions can be pre-set with timexFunc.",
    "  3) up to 4 functions can be pre-set with timexFunc,",
    "     timexPre and timexPost, i.e. 'i' in range 0..3.",
    "  4) timexPre and timexPost allow locking/unlocking, or",
    "     raising/lowering priority before/after timing.",
    NULL,	/* list terminator */
    };

    FAST char **ppMsg;

    printf ("\n");
    for (ppMsg = helpMsg; *ppMsg != NULL; ++ppMsg)
	printf ("%s\n", *ppMsg);
    }
/*******************************************************************************
*
* timex - time single execution
*
* This routine performs the timing of a single execution of the list of
* functions to be timed.  If a function (and optional arguments) is specified
* in the call then the entire list is replace by the single specified
* function.  The time of the execution is printed. If the execution
* was so fast in comparison to the clock rate that the time is
* meaningless (error > 50%), then a warning message is printed instead. In that
* case, use timexN (2).
*/

VOID timex (func, arg1)
    FUNCPTR func;	/* function to time (optional)                 */
    int arg1;		/* args with which to call function (optional) */

    {
    int scale;
    int time;
    int error;
    int percent;

    /* if function specified, clear any existing functions and add this one */

    if (func != NULL)
	{
	timexClrArrays ();
	timexAddCall (timexTimeCalls, 0, func, &arg1);
	}


    /* calibrate if necessary */

    if (overhead == 0)
	timexCal ();


    /* time calls */

    timexTime (1, timexPreCalls, timexTimeCalls, timexPostCalls, 
	    &scale, &time, &error, &percent);

    if (percent > 50)
	{
	printErr (
	    "timex: execution time too short to be measured meaningfully\n");
	printErr ("       in a single execution.\n");
	printErr ("       Type \"timexN\" to time repeated execution.\n");
	printErr ("       Type \"timexHelp\" for more information.\n");
	}
    else
	printErr ("timex: time of execution = %d +/- %d (%d%%) %s\n",
		  time, error, percent, timexScaleText [scale]);
    }
/*******************************************************************************
*
* timexN - time repeated execution
*
* This routine times the execution of the list of functions to be timed.
* If a function (and optional arguments) is specified in the call,
* then the entire list is replaced by the single specified function.
* The list is called a variable number of times until sufficient resolution
* is achieved to establish the time with an error less than 2% (since each 
* iteration of the list may be measured to a resolution of +/- 1 clock 
* tick, repetitive timings decrease this error to 1/N ticks, where N is 
* the number of repetitions.)
*/

VOID timexN (func, arg1)
    FUNCPTR func;  /* function to time (optional)                   */
    int arg1;      /* first of up to 8 params to call function with */
		   /* (optional) */
    {
    int scale;
    int time;
    int error;
    int nreps;
    int percent;

    /* if function specified, clear any existing functions and add this one */

    if (func != NULL)
	{
	timexClrArrays ();
	timexAddCall (timexTimeCalls, 0, func, &arg1);
	}


    /* calibrate if necessary then time calls */

    if (overhead == 0)
	timexCal ();

    timexAutoTime (timexPreCalls, timexTimeCalls, timexPostCalls, 
		   &nreps, &scale, &time, &error, &percent);

    printErr ("timex: %d reps, time per rep = %d +/- %d (%d%%) %s\n",
	      nreps, time, error, percent, timexScaleText [scale]);
    }
/*******************************************************************************
*
* timexPost - specify a function to be called immediately after timing
*
* This routine allows alteration of the list of functions which are to be 
* called immediately following the timing.
* Any function in the list of 4 functions can be added or deleted.
* Up to 8 arguments may be passed to each function.
*/

VOID timexPost (i, func, arg1)
    int i;        /* function number in list (0..3)                */
    FUNCPTR func; /* function to be added (NULL if to be deleted)  */
    int arg1;     /* first of up to 8 params to call function with */

    {
    timexAddCall (timexPostCalls, i, func, &arg1);
    timexShow ();
    }
/*******************************************************************************
*
* timexPre - specify a function to be called immediately prior to timing
*
* This routine allows alteration of  the list of functions which are to be 
* called immediately prior to the timing.
* Any function in the list of 4 functions can be added or deleted.
*/

VOID timexPre (i, func, arg1)
    int i;        /* function number in list (0..3)                */
    FUNCPTR func; /* function to be added (NULL if to be deleted)  */
    int arg1;     /* first of up to 8 params to call function with */

    {
    timexAddCall (timexPreCalls, i, func, &arg1);
    timexShow ();
    }
/*******************************************************************************
*
* timexShow - display all established function calls
*
* This routine displays each function currently established in each of
* the lists of function calls.  These lists are established by calls
* to timexPre (2), timexFunc (2), and timexPost (2).
*/

VOID timexShow ()

    {
    printf ("\ntimex:");
    printf ("\n    pre-calls:\n");
    timexShowCalls (timexPreCalls);
    printf ("\n    timed calls:\n");
    timexShowCalls (timexTimeCalls);
    printf ("\n    post-calls:\n");
    timexShowCalls (timexPostCalls);
    }

/*******************************************************************************
*
* timexAddCall - enter a call in a call array
*/

LOCAL VOID timexAddCall (callArray, i, func, argptr)
    CALL_ARRAY callArray;	/* array to be altered                      */
    int i;			/* number of function in list to be altered */
    FUNCPTR func;		/* function to add (NULL to delete)         */
    int *argptr;		/* pointer to optional args to function     */

    {
    int j;

    if (func == NULL)
	func = timexNull;

    if ((0 <= i) && (i < MAX_CALLS))
	{
	callArray[i].func = func;

	for (j = 0; j < MAX_ARGS; j++)
	    callArray[i].arg[j] = argptr[j];
	}
    else
	printf ("timex: call number must be in range 0..%d\n", MAX_CALLS - 1);
    }
/*******************************************************************************
*
* timexAutoTime - time specified function calls with automatic scaling
*
* This routine performs the specified timing, dynamically increasing
* the number of reps until the desired accuracy is achieved.
*/

LOCAL VOID timexAutoTime (preCalls, timeCalls, postCalls, 
		       pNreps, pScale, pTime, pError, pPercent)
    CALL_ARRAY preCalls;	/* list of functions to call before timing */
    FAST CALL_ARRAY timeCalls;	/* list of functions to time */
    CALL_ARRAY postCalls;	/* list of functions to call after timing */
    int *pNreps;	/* ptr where to return number of times called */
    int *pScale;	/* ptr where to return scale:
			 *   0 = secs, 1 = millisecs, 2 = microsecs */
    int *pTime;		/* ptr where to return time per rep in above units */
    int *pError;	/* ptr where to return error margin in above units */
    int *pPercent;	/* ptr where to return percent error (0..100) */

    {
    FAST int reps;

    /* start with one rep then increase reps until it takes a long
     * enough interval to provide sufficient resolution */

    reps = 1;

    timexTime (reps, preCalls, timeCalls, postCalls,
    	       pScale, pTime, pError, pPercent);

    while ((*pPercent > MAX_PERCENT) && ((reps < MAX_REPS) || (*pTime > 0)))
	{
	reps = reps * (*pPercent) / MAX_PERCENT;

	timexTime (reps, preCalls, timeCalls, postCalls,
		   pScale, pTime, pError, pPercent);
	}

    *pNreps = reps;
    }
/*******************************************************************************
*
* timexCal - calibrate timex by timing null functions
*
* This routine establishes the constant per rep overhead in the timing
* function.
*/

LOCAL VOID timexCal ()

    {
    int scale;
    int time;
    int error;
    int percent;
    int nreps;

    overhead = 0;

    timexAutoTime (timexNullCalls, timexNullCalls, timexNullCalls, 
		   &nreps, &scale, &time, &error, &percent);

    overhead = time;
    }
/*******************************************************************************
*
* timexClrArrays - clear out function arrays
*/

LOCAL timexClrArrays ()
    {
    bcopy ((char *) timexNullCalls, (char *) timexPreCalls,
	   sizeof (timexNullCalls));
    bcopy ((char *) timexNullCalls, (char *) timexTimeCalls,
	   sizeof (timexNullCalls));
    bcopy ((char *) timexNullCalls, (char *) timexPostCalls,
	   sizeof (timexNullCalls));
    }
/*******************************************************************************
*
* timexMakeCalls - make function calls in specified call array
*/

LOCAL VOID timexMakeCalls (calls)
    CALL_ARRAY calls;	/* list of functions to call */

    {
    FAST int ix;
    FAST CALL *pCall = &calls[0];

    for (ix = 0; ix < MAX_CALLS; ix++, pCall++)
	{
	(* pCall->func) (pCall->arg[0], pCall->arg[1], pCall->arg[2],
			 pCall->arg[3], pCall->arg[4], pCall->arg[5],
			 pCall->arg[6], pCall->arg[7]);
	}
    }
/*******************************************************************************
*
* timexNull - null routine
*
* This routine is used as a place holder for null routines in the
* timing function arrays.  It is used to guarantee a constant calling
* overhead in each iteration.
*/

LOCAL VOID timexNull ()
    {
    }
/*******************************************************************************
*
* timexScale - scale raw timing data
*/

LOCAL VOID timexScale (ticks, reps, pScale, pTime, pError, pPercent)
    int ticks;		/* total ticks required for all reps */
    int reps;		/* number of reps performed */
    FAST int *pScale;	/* ptr where to return scale:
			 *   0 = secs, 1 = millisecs, 2 = microsecs */
    FAST int *pTime;	/* ptr where to return time per rep in above units */
    int *pError;	/* ptr where to return error margin in above units */
    int *pPercent;	/* ptr where to return percent error (0..100) */

    {
    FAST int scalePerSec;

    /* calculate time per rep in best scale */

    *pScale = 0;		/* start with "seconds" scale */
    scalePerSec = 1;

    *pTime = ticks * scalePerSec / sysClkRateGet () / reps;

    while ((*pScale < 2) && (*pTime < 100))
	{
	(*pScale)++;
	scalePerSec = scalePerSec * 1000;
	*pTime = ticks * scalePerSec / sysClkRateGet () / reps;
	}


    /* adjust for overhead if in microseconds scale */

    if (*pScale == 2)
	{
	*pTime -= overhead;
	if (*pTime < 0)
	    *pTime = 0;
	}


    /* calculate error */

    *pError = scalePerSec / sysClkRateGet () / reps;

    if (*pTime == 0)
	*pPercent = 100;
    else
	*pPercent = 100 * *pError / *pTime;
    }
/*******************************************************************************
*
* timexShowCalls - print specified call array
*/

LOCAL VOID timexShowCalls (calls)
    CALL_ARRAY calls;		/* list of functions to be displayed */

    {
    char name [MAX_SYS_SYM_LEN + 1];
    int value;
    UTINY type;
    int offset;
    int i;
    int j;
    int arg;
    int ncalls = 0;

    for (i = 0; i < MAX_CALLS; i++)
	{
	if (calls[i].func != timexNull)
	    {
	    ncalls++;

	    symFindByValue (sysSymTbl, (int)calls[i].func, name, &value, &type);
	    name [MAX_SYS_SYM_LEN] = EOS;
	    offset = (int) calls[i].func - value;

	    if (offset == 0)
		printf ("        %d: %s (", i, name);
	    else
		printf ("        %d: %s+%x (", i, name, offset);

	    for (j = 0; j < MAX_ARGS; j++)
		{
		if (j != 0)
		    printf (", ");

		arg = calls[i].arg[j];
		if ((-9 <= arg) && (arg <= 9))
		    printf ("%d", arg);
		else
		    printf ("0x%x", arg);
		}

	    printf (")\n");
	    }
	}

    if (ncalls == 0)
	printf ("        (none)\n");
    }
/*******************************************************************************
*
* timexTime - time a specified number of reps
*/

LOCAL VOID timexTime (reps, preCalls, timeCalls, postCalls,
		      pScale, pTime, pError, pPercent)
    FAST int reps;		/* number of reps to perform */
    CALL_ARRAY preCalls;	/* list of functions to call before timing */
    FAST CALL_ARRAY timeCalls;	/* list of functions to be timed */
    CALL_ARRAY postCalls;	/* list of functions to call after timing */
    int *pScale;	/* ptr where to return scale:
			 *   0 = secs, 1 = millisecs, 2 = microsecs */
    int *pTime;		/* ptr where to return time per rep in above units */
    int *pError;	/* ptr where to return error margin in above units */
    int *pPercent;	/* ptr where to return percent error (0..100) */

    {
    int start;
    int end;
    FAST int i;

    timexMakeCalls (preCalls);

    start = (int) tickGet ();

    for (i = 0; i < reps; i++)
	timexMakeCalls (timeCalls);

    end = (int) tickGet ();

    timexMakeCalls (postCalls);

    timexScale (end - start, reps, pScale, pTime, pError, pPercent);
    }
