/* taskVarLib.c - task variable support library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01g,08apr89,dnw  added taskVarInit().
01f,17aug88,gae  documentation.
01d,22jun88,dnw  name tweaks.
01c,30may88,dnw  changed to v4 names.
01b,08apr88,gae  added taskId parm. to taskVar{Add,Del}();
		 made taskVar{Get,Set}() work with active task.
		 Fixed fatal bug in taskVarDel() of not replacing bg value.
		 Added taskVarDeleteHook() to cleanup after tasks.
		 Lint. Documentation.
01a,25jan88,jcf  written by extracting from vxLib.c.
*/

/*
DESCRIPTION
VxWorks provides a facility called "task variables" which allows 4 byte 
variables to be added to a task's context, such that the value of 
such a variable is switched every time a task switch occurs to, or from, 
its owner task.  Typically, several tasks declare the same variable 
(4 byte memory location) as a task variable.  Each of those tasks can 
then treat that single memory location as its own private variable.
This can be used, for example, when the same routine must be spawned
more than once as several simultaneous tasks.

The routine taskVarAdd adds a task variable to the task context.
The routine taskVarDelete deletes a previously added task variable.
The routines taskVarGet and taskVarSet allow getting and setting
task variables of tasks other than the calling task.

NOTE
If you are using task variables in a "task delete hook" (see taskHookLib (1)),
then please see the manual entry for taskVarInit (2) for warnings
on proper usage.
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "memLib.h"
#include "taskLib.h"

/* forward declarations */

LOCAL VOID taskVarSwitchHook ();
LOCAL VOID taskVarDeleteHook ();

/*******************************************************************************
*
* taskVarInit - initialize task variables facility
*
* This routine initializes the task variables facility.
* It installs task switch and delete hooks that are used in implementing
* task variables.  If it is not called explicitly, it will automatically
* be called by taskVarAdd (2) when the first task variable is added.
*
* After the first invocation of this routine, subsequent invocations
* have no affect.
*
* WARNING
* There are often order dependencies in task delete hooks involving
* task variables.  That is, if a facility uses task variables and has a
* task delete hook that expects to use those task variables, then that
* facility's delete hook must run before the task variables delete hook.
* Otherwise the task variables would have been deleted by the time the
* other facility's delete hook ran.  VxWorks is careful to run the
* delete hooks in the REVERSE of the order in which they were installed.
* So any facility that has a delete hook that will use task variables
* can guarantee proper ordering by calling taskVarInit before adding
* its own delete hook.
*
* It should be understood that this is not an issue in normal use of
* task variables.  The issue only arises when adding another task
* delete hook that uses task variables.
*
* RETURNS: OK or ERROR if couldn't install task switch/delete hooks
*/

STATUS taskVarInit ()

    {
    static BOOL taskVarInstalled = FALSE;	/* TRUE = facility installed */

    /* if task variables facility is not already installed, then install it
     * by adding the switch and delete hooks
     */

    if (!taskVarInstalled)
	{
	if ((taskSwitchHookAdd (taskVarSwitchHook) != OK) ||
	    (taskDeleteHookAdd (taskVarDeleteHook) != OK))
	    {
	    return (ERROR);
	    }

	taskVarInstalled = TRUE;
	}

    return (OK);
    }

/*******************************************************************************
*
* taskVarDeleteHook - delete task variables of exiting tasks
*
* This routine is the task delete routine that deletes all task
* variables of an exiting task.
*/

LOCAL VOID taskVarDeleteHook (pTcbX)
    TCBX *pTcbX;

    {
    FAST TASK_VAR *pTaskVar;
    FAST TASK_VAR *pTaskVarNext;

    for (pTaskVar = pTcbX->taskVar;
	 pTaskVar != NULL;
	 pTaskVar = pTaskVarNext)
	{
	pTaskVarNext = pTaskVar->next;
	free ((char *)pTaskVar);	/* free storage of deleted cell */
	}
    }
/*******************************************************************************
*
* taskVarSwitchHook - switch task variables of switching tasks
*
* This routine is the task switch routine that implements the task variable
* facility.  It swaps the current and saved values of all the task variables
* of the out-going and in-coming tasks.
*/

LOCAL VOID taskVarSwitchHook (pOldTcbX, pNewTcbX)
    TCBX *pOldTcbX;
    TCBX *pNewTcbX;

    {
    FAST TASK_VAR *pTaskVar;
    FAST int temp;

    /* swap task variables of old task */

    for (pTaskVar = pOldTcbX->taskVar;
	 pTaskVar != NULL;
	 pTaskVar = pTaskVar->next)
	{
	/* swap current and save value of task variable */

	temp = pTaskVar->value;
	pTaskVar->value = *(pTaskVar->address);
	*(pTaskVar->address) = temp;
	}

    /* swap task variables of new task */

    for (pTaskVar = pNewTcbX->taskVar;
	 pTaskVar != NULL;
	 pTaskVar = pTaskVar->next)
	{
	/* swap current and save value of task variable */

	temp = pTaskVar->value;
	pTaskVar->value = *(pTaskVar->address);
	*(pTaskVar->address) = temp;
	}
    }
/*******************************************************************************
*
* taskVarAdd - add a task variable to a task
*
* This routine adds the specified variable (4 byte memory location)
* to the task's context.  After calling this routine, the specified
* variable will be "private" to the task.  That task can access
* and modify the variable in any way, but its modifications to the
* variable will not appear to other tasks, and other tasks' modifications
* to that variable will not affect the value seen by the that task.
* This is accomplished by saving and restoring the value of that variable
* each time a task switch is made to or from the calling task.
*
* This facility can be used when a single routine is to be spawned
* repeatedly as several independent tasks.  Although each task
* will have its own stack, and thus separate stack variables, they will
* all share the same "static" and "global" variables.  To make such a
* variable NOT shared, the routine can call taskVarAdd (2), thus
* making a separate copy of that variable for each task, but all
* at the same physical address.
*
* Note that task variables increase the task switch time to and from
* the tasks that own them, so it is desirable to limit the number of
* task variables that a task uses.  An efficient use of task variables is
* to have a single task variable that is a pointer to a dynamically 
* allocated structure containing the task's private data.
*
* EXAMPLE:
* Suppose three identical tasks were spawn'd with the routine "operator".
* They use a structure "OP_GLOBAL" for all variables that are specific
* to the particular incarnation of the task.  The following code fragment
* shows how this is set up.
* .CS
* OP_GLOBAL *opGlobal;	/* pointer to operator task's global variables *
*
* operator (opNum)
*     int opNum;		/* number of this operator task *
*     {
*     if (taskVarAdd (0, (int *)&opGlobal) != OK)
*         {
*         printErr ("operator%d: can't taskVarAdd opGlobal\n", opNum);
*         taskSuspend (0);
*         }
*
*     if ((opGlobal = (OP_GLOBAL *) malloc (sizeof (OP_GLOBAL))) == NULL)
*         {
*         printErr ("operator%d: can't malloc opGlobal\n", opNum);
*         taskSuspend (0);
*         }
*     ...
*     }
* .CE
*
* RETURNS:
*   OK, or
*   ERROR if not enough memory for task variable descriptor.
*
* INTERNAL
* The first time this routine is called the task switch and delete
* routines, taskVarSwitchHook and taskVarDeleteHook, are installed.
*
* SEE ALSO: taskVarDelete(2), taskVarGet(2), taskVarSet(2)
*/

STATUS taskVarAdd (tid, pVar)
    int tid;	/* id of task to have new variable */
    int *pVar;	/* pointer to variable to be switched for task */

    {
    FAST TCBX *pTcbX = taskTcbX (tid);
    FAST TASK_VAR *pTaskVar;

    /* make sure task variable facility is installed */

    if (taskVarInit () != OK)
	return (ERROR);


    /* allocate descriptor for new task variable */

    pTaskVar = (TASK_VAR *) malloc (sizeof (TASK_VAR));

    if (pTaskVar == NULL)
	return (ERROR);

    pTaskVar->address = pVar;
    pTaskVar->value   = *pVar;


    /* link new task variable into list in stack header */

    pTaskVar->next = pTcbX->taskVar;
    pTcbX->taskVar = pTaskVar;

    return (OK);
    }
/*******************************************************************************
*
* taskVarDelete - remove a task variable from a task
*
* This routine removes the specified task variable from the calling
* task's context.  The "private" value of that variable is lost.
*
* RETURNS
*  OK, or
*  ERROR if calling task does not own the specified task variable.
*
* SEE ALSO: taskVarAdd(2), taskVarGet(2), taskVarSet(2)
*/

STATUS taskVarDelete (tid, pVar)
    int tid;	/* task id whose task variable is to be retrieved */
    int *pVar;	/* pointer to task variable to be removed from task */

    {
    TCBX *pTcbX = taskTcbX (tid);
    FAST TASK_VAR **ppTaskVar;	/* ptr to ptr to next node */
    FAST TASK_VAR *pTaskVar;

    /* find descriptor for specified task variable */

    for (ppTaskVar = &pTcbX->taskVar;
	 *ppTaskVar != NULL;
	 ppTaskVar = &((*ppTaskVar)->next))
	{
	pTaskVar = *ppTaskVar;

	if (pTaskVar->address == pVar)
	    {
	    /* if active task, replace background value */

	    if (taskIdSelf () == pTcbX->taskId)
		*pVar = pTaskVar->value;

	    *ppTaskVar = pTaskVar->next;/* delete variable from list */

	    free ((char *)pTaskVar);	/* free storage of deleted cell */

	    return (OK);
	    }
	}


    /* specified address is not a task variable for specified task */

    errnoSet (S_taskLib_TASK_VAR_NOT_FOUND);
    return (ERROR);
    }
/*******************************************************************************
*
* taskVarGet - get value of a task variable
*
* This routine gets the "private" value of the task variable 
* for a specific task.
* The specified task is usually not the calling task,
* which can get its private value by direct access to the variable.
* This routine is provided primarily for debugging purposes.
*
* RETURNS:
*  the private value of that variable, or
*  ERROR if specified task not found, or
*  specified task does not own task variable
*
* SEE ALSO: taskVarAdd(2), taskVarDelete(2), taskVarSet(2)
*/

int taskVarGet (tid, pVar)
    int tid;		/* task id whose task variable is to be retrieved */
    int *pVar;		/* pointer to task variable */
			 
    {
    FAST TASK_VAR *pTaskVar;	/* ptr to next node */
    TCBX *pTcbX = taskTcbX (tid);

    if (pTcbX == NULL)
	{
	errnoSet (S_taskLib_TASK_ID_ERROR);
	return (ERROR);
	}

    /* find descriptor for specified task variable */

    for (pTaskVar = pTcbX->taskVar;
	 pTaskVar != NULL;
	 pTaskVar = pTaskVar->next)
	{
	if (pTaskVar->address == pVar)
	    return (taskIdSelf () == pTcbX->taskId ? *pVar : pTaskVar->value);
	}


    /* specified address is not a task variable for specified task */

    errnoSet (S_taskLib_TASK_VAR_NOT_FOUND);
    return (ERROR);
    }
/*******************************************************************************
*
* taskVarSet - set value of a task variable
*
* This routine sets the "private" value of the task variable 
* for a specific task.
* The specified task is usually not the calling task,
* which can set its private value by direct modification to the variable.
* This routine is provided primarily for debugging purposes.
*
* RETURNS:
*  OK, or
*  ERROR if specified task not found, or
*  specified task does not own task variable
*
* SEE ALSO: taskVarAdd(2), taskVarDelete(2), taskVarGet(2)
*/

STATUS taskVarSet (tid, pVar, value)
    int tid;	/* task id whose task variable is to be set */
    int *pVar;	/* pointer to task variable to be set for this task */
    int value;	/* new value of task variable */

    {
    FAST TASK_VAR *pTaskVar;	/* ptr to next node */
    TCBX *pTcbX = taskTcbX (tid);
    
    if (pTcbX == NULL)
	{
	errnoSet (S_taskLib_TASK_ID_ERROR);
	return (ERROR);
	}

    /* find descriptor for specified task variable */

    for (pTaskVar = pTcbX->taskVar;
	 pTaskVar != NULL;
	 pTaskVar = pTaskVar->next)
	{
	if (pTaskVar->address == pVar)
	    {
	    if (taskIdSelf () == pTcbX->taskId)
		*pVar = value;
	    else
		pTaskVar->value = value;
	    return (OK);
	    }
	}

    /* specified address is not a task variable for specified task */

    errnoSet (S_taskLib_TASK_VAR_NOT_FOUND);
    return (ERROR);
    }
