/* stdioLib.c - standard I/O library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01m,08apr89,dnw  changed stdioInit() to call taskVarInit().
01l,23mar89,dab  changed numerical constants to appropriate defines.
		 fixed fseek() to return only OK or ERROR.
01k,15nov88,dnw  documentation touchup.
01j,23sep88,gae  documentation touchup.
01i,13sep88,gae  removed ifdef'd sprintf which got into documentation.
01h,06sep88,gae  adjusted some argument declarations to please f2cgen.
01g,20aug88,gae  documentation.
01f,07jul88,jcf  changed malloc to match new declaration.
01e,29jun88,gae  documentation.  Added error messages in stioInit().
01d,22jun88,dnw  name tweaks.
01c,30may88,dnw  changed to v4 names.
01b,28may88,dnw  removed routines that had been excluded with "#if FALSE".
		 made stdioFlushBuf LOCAL.
		 cleaned up stdio{Init,Exit}Task; improved error msgs.
01a,28mar88,gae  created.
*/

/*
DESCRIPTION
The library stdioLib provides a complete UNIX compatible "standard I/O"
buffering scheme.  It is beyond the scope of this manual entry to
describe all the functions and vagaries of the buffering --
see the I/O chapter of the VxWorks Programmers Guide and
the Kernighan & Ritchie C manual.
This manual entry primarily highlights the differences
between the UNIX and VxWorks stdio.

VX_STDIO TASK OPTION
Traditionally stdin, stdout, and stderr are macros but,
in VxWorks, they are in fact variables.
They will only be defined for the task context of tasks spawned with
the task option bit VX_STDIO.
They are unique to each such task and correspond to the file descriptors
0, 1, and 2 in the basic I/O system.
Their values are undefined when used at any other time,
eg. interrupt level code or another task without the VX_STDIO option.

  IF A TASK WILL USE stdin, stdout, OR stderr,
  THE TASK MUST BE SPAWNED WITH THE ``VX_STDIO'' TASK OPTION BIT.

The only exception is the use of the routine printf(1), which, as
explained below, does not actually use stdio buffering.

HEADER FILE
To use VxWorks standard I/O one must include the header file stdioLib.h.
This file defines all the usual macros.

All the macros defined in stdioLib.h are also implemented as real functions
so that they are available from the VxWorks shell.

FILE POINTERS
The routine fopen (2) creates a ``FILE'' .  Use of the FILE pointer
is just as in any conventional UNIX usage.  Remember though, that
in a shared address space, and perhaps more critically with the system
symbol table of VxWorks, tasks may not use each others' file pointers,
at least not without some interlocking mechanism.  If it is necessary
to use the same name for a file pointer but have incarnations for
each task then use task variables; see taskVarLib (1).

FIOLIB
There are a few routines not implemented in stdioLib (1).
The routines printf, sscanf, and sprintf are implemented in fioLib (1).
They do not use the stdio buffering scheme, but instead are self-contained,
formatted but unbuffered I/O functions.
This allows minimal formatted I/O to be achieved without the overhead
of the entire stdioLib package.

TASK TERMINATION
When a task exits, unlike UNIX, it is the responsibility of the
task to fclose (2) its file pointers, except stdin, stdout, and stderr.
If a task is to be terminated asynchronously, use kill (2)
and arrange for a signal handler to cleanup.

DEFICIENCIES
Beware of the routine fseek (2) and the fopen (2) modes of "r+",
"w+", "a" and "a+".  These do not work in many cases due to
the present limitations of lseek (2) in ioLib (1).

SEE ALSO
fioLib (1), ioLib (1), taskVarLib (1), sigLib (1),
VxWorks Programmers Guide: I/O Chapter, and Kernighan & Ritchie C manual

INCLUDE FILES
stdioLib.h, taskLib.h
*/


#include "vxWorks.h"
#include "stdioLib.h"
#include "types.h"
#include "ctype.h"
#include "ioLib.h"
#include "memLib.h"
#include "taskLib.h"

IMPORT TCBX *taskTcbX ();

FILE *stdin;
FILE *stdout;
FILE *stderr;

int stdioMode = 0666;	/* open/creat mode for fopen/freopen */

#define active(fp)	((fp)->flag & (_STDIO_READ|_STDIO_WRT|_STDIO_RW))

/* forward declarations */

LOCAL FILE *makeFile ();

LOCAL VOID stdioCreateHook ();
LOCAL VOID stdioDeleteHook ();

/*******************************************************************************
*
* stdioInit - initialize stdioLib support
*
* This routine must be called before using the stdioLib (1) buffering.
* It is usually called by the root task, usrRoot(2), in usrConfig(1).
*
* RETURNS: OK or ERROR if error installing stdio facilities
*/

STATUS stdioInit ()

    {
    static BOOL stdioInitialized = FALSE;

    if (!stdioInitialized)
	{
	if ((taskVarInit () == ERROR) ||
	    (taskCreateHookAdd (stdioCreateHook) == ERROR) ||
	    (taskDeleteHookAdd (stdioDeleteHook) == ERROR))
	    return (ERROR);

	stdioInitialized = TRUE;
	}

    return (OK);
    }
/*******************************************************************************
*
* stdioCreateHook - initialize stdioLib support for task
*
* This routine creates the 3 task variables stdin, stdout, and stderr
* for tasks with the VX_STDIO option bit set.
*
* RETURNS: OK | ERROR if unable to allocate storage, or add task switch
*/

LOCAL STATUS stdioCreateHook (pTcbX)
    FAST TCBX *pTcbX;

    {
    FAST int tid = pTcbX->taskId;

    /* check for option bit */

    if (pTcbX->options & VX_STDIO &&
	(stdioInitStd (tid, &stdin,  STD_IN,  _STDIO_READ) == ERROR ||
	 stdioInitStd (tid, &stdout, STD_OUT, _STDIO_WRT)  == ERROR ||
	 stdioInitStd (tid, &stderr, STD_ERR, _STDIO_WRT|_STDIO_NBF) == ERROR))
	{
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* stdioInitStd - initialize use of a standard file
*/

LOCAL STATUS stdioInitStd (tid, ppStdFile, fd, flag)
    int tid;		/* task id */
    FILE **ppStdFile;	/* pointer to standard file pointer */
    int fd;		/* file descriptor to assign */
    int flag;		/* initial file options */

    {
    FILE *pFile;

    if ((taskVarAdd (tid, (int *) ppStdFile) != OK)	||
	((pFile = makeFile ()) == NULL)			||
	taskVarSet (tid, (int *) ppStdFile, (int) pFile) == ERROR)
	{
	logMsg ("stdioLib: error initializing std file - status = %#x\n",
		errnoGet ());
	return (ERROR);
	}

    pFile->fd     = fd;
    pFile->flag   = flag;
    pFile->taskId = tid;

    return (OK);
    }
/*******************************************************************************
*
* stdioDeleteHook - quit stdioLib support for task
*
* This routine free's the memory used for the stdioLib buffer area.
*/

LOCAL VOID stdioDeleteHook (pTcbX)
    FAST TCBX *pTcbX;	/* task for which to end support */

    {
    FAST int tid = pTcbX->taskId;

    /* check for option bit */

    if (pTcbX->options & VX_STDIO)
	{
	stdioExitStd (tid, &stdin);
	stdioExitStd (tid, &stdout);
	stdioExitStd (tid, &stderr);
	}
    }
/*******************************************************************************
*
* stdioExitStd - do exit processing on standard file
*/

LOCAL VOID stdioExitStd (tid, ppStdFile)
    int tid;		/* task id */
    FILE **ppStdFile;	/* pointer to standard file pointer */

    {
    FILE *pFile = (FILE *) taskVarGet (tid, (int *) ppStdFile);

    if (pFile == (FILE *) ERROR)
	{
	logMsg ("stdioLib: can't access task variable - status = %#x\n",
		errnoGet());
	}
    else
	{
	if (pFile == NULL)
	    logMsg ("stdioLib: task variable is NULL!\n");
	else
	    fclose (pFile);

	taskVarDelete (tid, (int *) ppStdFile);
	}
    }
/*******************************************************************************
*
* isatty - return whether underlying driver is a tty device
*
* This routine simply invokes the ioctl FIOISATTY on the `fd'.
*
* RETURNS: TRUE if driver says so, otherwise FALSE
*/

BOOL isatty (fd)
    int fd;	/* file descriptor to check */

    {
    return (ioctl (fd, FIOISATTY) == TRUE);
    }
/*******************************************************************************
*
* fclose - empty stream buffers and close file
*
* The buffer associated with the FILE pointer `fp' is emptied
* and the file closed.
*
* RETURNS:
*    OK if buffers flushed, or
*    EOF if unable to write out buffers, or trouble closing file.
*
* SEE ALSO: fflush (2)
*/

STATUS fclose (fp)
    FAST FILE *fp;	/* stream */

    {
    FAST int result = EOF;
    int fd;

    if (active (fp) && (fp->flag & _STDIO_STRG) == 0)
	{
	result = fflush (fp);
	fd = fileno (fp);

	if (fd >= 0 && fd < 3)
	    ;	/* XXX don't close! */
	else if (close (fd) < 0)
	    result = EOF;

	if (fp->flag & _STDIO_MYBUF)
	    free (fp->base);
	}

    fp->count   = 0;
    fp->base    = NULL;
    fp->ptr     = NULL;
    fp->bufsize = 0;
    fp->flag    = 0;
    fp->fd      = 0;
    fp->taskId  = 0;

    /* put on free list */

    free ((char *)fp);

    return (result);
    }
/*******************************************************************************
*
* fdopen - associate stream with file descriptor
*
* The routine fdopen associates a stream with the file descriptor
* obtained by open(2) or creat(2).
*
* The mode type has to be repeated because you can't query its status,
* it must agree with the mode of the already open fd.
*
* RETURNS:
*    stream pointer, or
*    NULL if invalid fd
*
* SEE ALSO: fopen(2), freopen(2)
*/

FILE *fdopen (fd, type)
    int fd;		/* already open file descriptor                */
    FAST char *type;	/* mode to open file (must agree with open fd) */

    {
    FAST FILE *fp = makeFile ();

    if (fp == NULL)
	return (NULL);

    fp->count   = 0;
    fp->bufsize = 0;
    fp->ptr     = NULL;
    fp->base    = NULL;
    fp->fd      = fd;

    switch (*type)
	{
	case 'r':
	    fp->flag = _STDIO_READ;
	    break;

	case 'a':
	    (void)lseek (fd, 0L, L_XTND);
	    /* drop thru */

	case 'w':
	    fp->flag = _STDIO_WRT;
	    break;

	default:
	    return (NULL);
    }

    if (type[1] == '+')
	fp->flag = _STDIO_RW;

    return (fp);
    }
/*******************************************************************************
*
* fgetc - return next character in input stream
*
* This routine returns the next character from the named input stream
* as an integer.
* It also moves the file pointer ahead one character in the stream.
* Same as getc(2), but is a genuine routine, not a macro.
*
* RETURNS: next character, or EOF on end-of-file or error
*/

int fgetc (fp)
    FAST FILE *fp;	/* stream */

    {
    return (getc (fp));
    }
/*******************************************************************************
*
* fgets - read a string from input stream
*
* At most n-1 characters, or up to the newline character, are read
* from the specified stream into the buffer `s'.
* The buffer `s' is NULL terminated.
*
* RETURNS: first argument, or NULL on end-of-file or error
*
* SEE ALSO: fopen(2), fread(2)
*/

char *fgets (s, n, fp)
    char *s;		/* buffer to hold characters read */
    int n;		/* number of bytes to read -1 for newline */
    FAST FILE *fp;	/* stream */

    {
    FAST int c;
    FAST char *cs = s;

    while (--n > 0 && (c = getc (fp)) != EOF)
	{
	*cs++ = c;
	if (c == '\n')
	    break;
	}

    if (c == EOF && cs == s)
	return (NULL);

    *cs++ = EOS;

    return (s);
    }
/*******************************************************************************
*
* stdioFillBuf - fill buffer
*
* RETURNS: last character written, or EOF
*
* NOMANUAL
*/

int stdioFillBuf (fp)
    FAST FILE *fp;

    {
    int size;
    char ch;

    if (fp->flag & _STDIO_RW)
	fp->flag |= _STDIO_READ;

    if ((fp->flag & _STDIO_READ) == 0 || (fp->flag & (_STDIO_STRG|_STDIO_EOF)))
	return (EOF);

retry:
    if (fp->base == NULL)
	{
	if (fp->flag & _STDIO_NBF)
	    {
	    fp->base = &ch;
	    goto retry;
	    }

	size = BUFSIZE;

	if ((fp->base = malloc ((unsigned) size)) == NULL)
	    {
	    fp->flag |= _STDIO_NBF;
	    goto retry;
	    }

	fp->flag |= _STDIO_MYBUF;
	fp->bufsize = size;
	}

    if (fp == stdin)
	{
	if (stdout->flag & _STDIO_LBF)
	    fflush (stdout);
	if (stderr->flag & _STDIO_LBF)
	    fflush (stderr);
	}

    fp->count = read (fileno (fp), fp->base,
		      fp->flag & _STDIO_NBF ? 1 : fp->bufsize);
    fp->ptr = fp->base;

    if ((fp->flag & _STDIO_NBF) && fp->base == &ch)
	fp->base = NULL;

    if (--fp->count < 0)
	{
	if (fp->count == -1)
	    {
	    fp->flag |= _STDIO_EOF;
	    if (fp->flag & _STDIO_RW)
		fp->flag &= ~_STDIO_READ;
	    }
	else
	    fp->flag |= _STDIO_ERR;

	fp->count = 0;

	return (EOF);
	}

    return (*fp->ptr++ & 0377);		/* clear top bit */
    }
/*******************************************************************************
*
* stdioFlushBuf - flush buffer
*
* RETURNS: last character written, or EOF
*
* NOMANUAL
*/

int stdioFlushBuf (c, fp)
    unsigned char c;
    FAST FILE *fp;

    {
    FAST char *base;
    FAST int n;
    FAST int rn;
    char c1;
    int size;

    if (fp->flag & _STDIO_RW)
	{
	fp->flag |= _STDIO_WRT;
	fp->flag &= ~(_STDIO_EOF|_STDIO_READ);
	}

    if ((fp->flag & _STDIO_WRT)==0)
	return (EOF);
retry:
    if (fp->flag & _STDIO_LBF)
	{
	base = fp->base;
	*fp->ptr++ = c;
	if (fp->ptr >= (base + fp->bufsize) || c == '\n')
	    {
	    n = write (fileno (fp), base, rn = fp->ptr - base);
	    fp->ptr = base;
	    fp->count = 0;
	    }
	else
	    rn = n = 0;
	}
    else if (fp->flag & _STDIO_NBF)
	{
	c1 = c;
	rn = 1;
	n = write (fileno (fp), &c1, rn);
	fp->count = 0;
	}
    else
	{
	if ((base = fp->base) == NULL)
	    {
	    size = BUFSIZE;

	    if ((fp->base = base = malloc ((unsigned) size)) == NULL)
		{
		fp->flag |= _STDIO_NBF;
		goto retry;
		}

	    fp->flag |= _STDIO_MYBUF;
	    fp->bufsize = size;

	    if (fp == stdout && isatty (fileno (stdout)))
		{
		fp->flag |= _STDIO_LBF;
		fp->ptr = base;
		goto retry;
		}
	    rn = n = 0;
	    }
	else if ((rn = n = fp->ptr - base) > 0)
	    {
	    fp->ptr = base;
	    n = write (fileno (fp), base, n);
	    }
	fp->count = fp->bufsize - 1;
	*base++   = c;
	fp->ptr   = base;
	}

    if (rn != n)
	{
	fp->flag |= _STDIO_ERR;
	return (EOF);
	}

    return (c);
    }
/*******************************************************************************
*
* fflush - cause any buffers on output stream to be written.
*
* The buffer associated with the FILE pointer `fp' is written out.
* The stream remains open.
*
* RETURNS:
*    OK, or
*    EOF if unable to write out buffers
*/

int fflush (fp)
    FAST FILE *fp;	/* stream */

    {
    FAST char *base;
    FAST n;

    if ((fp->flag & (_STDIO_NBF|_STDIO_WRT)) == _STDIO_WRT	&&
	(base = fp->base) != NULL		&&
	(n = fp->ptr - base) > 0)
	{
	fp->ptr = base;
	fp->count = (fp->flag & (_STDIO_LBF|_STDIO_NBF)) ? 0 : fp->bufsize;

	if (write (fileno (fp), base, n) != n)
	    {
	    fp->flag |= _STDIO_ERR;
	    return (EOF);
	    }
	}

    return (OK);
    }
/*******************************************************************************
*
* fopen - open a stream on a file
*
* The routine fopen opens the file named `filename' and associates
* it with a stream.  The returned FILE pointer is used to identify
* the stream with subsequent operations.
*
* The type is a character string consisting of:
*   "r" - read
*   "w" - write
*   "a" - update, open for write at end, or create if doesn't exist
*
* In addition, a following "+" indicates that the file is opened
* for both reading and writing.
*   "r+" - position at the beginning
*   "w+" - create or truncate
*   "a+" - position at the end
*
* Both reads and writes may be intermixed as long as an intervening
* fseek (2) or rewind (2) occurrs.
*
* RETURNS:
*    file pointer, or
*    NULL if unable to open file
*
* SEE ALSO: freopen(2), fdopen(2)
*/

FILE *fopen (filename, type)
    char *filename;	/* name of file */
    FAST char *type;	/* mode to open file */

    {
    FAST int fd;
    FAST int rw;
    FAST int oflags;
    FAST FILE *fp = makeFile ();

    if (fp == NULL)
	return (NULL);

    rw = (type[1] == '+');

    switch (*type)
	{
	case 'a':
	    oflags = O_CREAT | (rw ? O_RDWR : O_WRONLY);
	    break;
	case 'r':
	    oflags = rw ? O_RDWR : O_RDONLY;
	    break;
	case 'w':
	    oflags = O_TRUNC | O_CREAT | (rw ? O_RDWR : O_WRONLY);
	    break;

	default:
	    return (NULL);
	}

    if (*type == 'w')
	{
	/* XXX should be `stdioMode' fix after 4.0.2 */
	if ((fd = creat (filename, oflags)) < 0)
	    return (NULL);
	}
    else
	{
	if ((fd = open (filename, oflags, stdioMode)) < 0)
	    return (NULL);
	}

    /* for append, position file pointer at the end of the file */
    if (*type == 'a')
	(void)lseek (fd, 0L, L_XTND);

    fp->count   = 0;
    fp->fd      = fd;
    fp->bufsize = 0;

    if (rw)
	fp->flag = _STDIO_RW;
    else if (*type == 'r')
	fp->flag = _STDIO_READ;
    else
	fp->flag = _STDIO_WRT;

    fp->base = fp->ptr = NULL;

    return (fp);
    }
/*******************************************************************************
*
* fprintf - print formatted string to stream
*
* Like printf (2) except outputs through file pointer `fp'.
*
* NOTE:
* The standard output file pointer `stdout' is buffered so
* fflush (2) is must be used to force output.
* 
* RETURNS: OK or EOF on error
*
* SEE ALSO: fopen (2)
*
* VARARGS2
*/

int fprintf (fp, fmt, args)
    FAST FILE *fp;	/* output stream                           */
    char *fmt;		/* format specification                    */
    int args;		/* first of a variable number of arguments */

    {
    char localbuf[BUFSIZE];

    if (fp->flag & _STDIO_NBF)
	{
	fp->flag   &= ~_STDIO_NBF;
	fp->ptr     = fp->base = localbuf;
	fp->bufsize = BUFSIZE;

	doprnt (fmt, &args, fp);
	fflush (fp);

	fp->flag   |= _STDIO_NBF;
	fp->base    = NULL;
	fp->bufsize = NULL;
	fp->count   = 0;
	}
    else
	doprnt (fmt, &args, fp);

    /* XXX should return number of chars transmitted */

    return (ferror (fp) ? EOF: OK);
    }
/*******************************************************************************
*
* fputc - append character to output stream
*
* Same as putc(2), but is a genuine routine, not a macro.
*
* RETURNS: character written, or EOF on error
*
* SEE ALSO: fopen(2), fputs(2)
*/

int fputc (c, fp)
    int c;	/* character to append */
    FILE *fp;	/* stream */

    {
    return (putc (c, fp));
    }
/*******************************************************************************
*
* fputs - copy NULL terminated string to output stream
*
* This routine copies the NULL terminated string `s' to the stream `fp'.
*
* RETURNS: last character written, or 0 if empty string
*/

int fputs (s, fp)
    FAST char *s;	/* string to copy to output stream */
    FAST FILE *fp;	/* output stream                   */

    {
    FAST int result = 0;
    FAST int c;
    char localbuf[BUFSIZE];
    int unbuffered = fp->flag & _STDIO_NBF;

    if (unbuffered)
	{
	fp->flag &= ~_STDIO_NBF;
	fp->ptr = fp->base = localbuf;
	fp->bufsize = BUFSIZE;
	}

    while (c = *s++)
	result = putc (c, fp);

    if (unbuffered)
	{
	fflush (fp);
	fp->flag   |= _STDIO_NBF;
	fp->base    = NULL;
	fp->bufsize = NULL;
	fp->count   = 0;
	}

    return (result);
    }
/*******************************************************************************
*
* fread - buffered binary input
*
* The routine fread reads into a block beginning at `ptr',
* `count' items of data of the size `size' from stream `fp'.
*
* RETURNS:
*    number of items read, or
*    0 on end-of-file or error
*/

int fread (ptr, size, count, fp)
    FAST char *ptr;	/* where to copy into */
    unsigned size;	/* size of item       */
    unsigned count;	/* number of items    */
    FAST FILE *fp;	/* input stream       */

    {
    int c;
    FAST int s = size * count;

    while (s > 0)
	{
	if (fp->count < s)
	    {
	    if (fp->count > 0)
		{
		bcopy(fp->ptr, ptr, fp->count);
		ptr += fp->count;
		s -= fp->count;
		}
	    /*
	     * stdioFillBuf clobbers count & ptr,
	     * so don't waste time setting them.
	     */
	    if ((c = stdioFillBuf (fp)) == EOF)
		break;

	    *ptr++ = c;
	    s--;
	    }
	if (fp->count >= s)
	    {
	    bcopy(fp->ptr, ptr, s);
	    fp->ptr += s;
	    fp->count -= s;
	    return (count);
	    }
	}

    return (size != 0 ? count - ((s + size - 1) / size) : 0);
    }
/*******************************************************************************
*
* freopen - substitute named file in place of the open stream
*
* The original stream is closed.  The new file is opened in its place.
*
* Typically used to substitute a file in place of the standard
* preopened streams stdin, stdout, and stderr.
*
* A file may be changed from unbuffered or line buffered
* to block buffered by using freopen.
* A file can be changed from block buffered or line buffered
* to unbuffered by using freopen followed by setbuf with a
* NULL buffer argument.
*
* RETURNS:
*    original value of stream, or
*    NULL if unable to open file
*
* SEE ALSO: fopen(2), fdopen(2)
*/

FILE *freopen (filename, type, fp)
    char *filename;	/* new file to open  */
    FAST char *type;	/* mode to open file */
    FAST FILE *fp;	/* old stream        */

    {
    FAST int fd;
    FAST int oflags;
    FAST int rw = (type[1] == '+');

    fclose (fp);

    switch (*type)
	{
	case 'a':
	    oflags = O_CREAT | (rw ? O_RDWR : O_WRONLY);
	    break;

	case 'r':
	    oflags = rw ? O_RDWR : O_RDONLY;
	    break;

	case 'w':
	    oflags = O_TRUNC | O_CREAT | (rw ? O_RDWR : O_WRONLY);
	    break;

	default:
	    return (NULL);
	}

    if (*type == 'w')
	{
	/* XXX should be `stdioMode' - fix after 4.0.2 */
	if ((fd = creat (filename, oflags)) < 0)
	    return (NULL);
	}
    else
	{
	if ((fd = open (filename, oflags, stdioMode)) < 0)
	    return (NULL);
	}

    if (fd < 0)
	return (NULL);

    /* for append, position file pointer at the end of the file */
    if (*type == 'a')
	(void)lseek (fd, 0L, L_XTND);

    fp->count   = 0;
    fp->fd      = fd;
    fp->bufsize = 0;
    fp->base    = fp->ptr = NULL;

    if (rw)
	fp->flag = _STDIO_RW;
    else if (*type == 'r')
	fp->flag = _STDIO_READ;
    else
	fp->flag = _STDIO_WRT;

    return (fp);
    }
/*******************************************************************************
*
* fseek - reposition a stream
*
* This routine sets the position of the next I/O operation on the
* stream.  The new position is at the signed distance `offset' bytes
* from the beginning, the current position, or the end of the file,
* according to `ptrname' is 0, 1, or 2.
*
* INTERNAL
* Coordinates with buffering.
*
* RETURNS: OK, or ERROR for improper seeks
*
* SEE ALSO: lseek(2), fopen(2)
*/

STATUS fseek (fp, offset, ptrname)
    FAST FILE *fp;	/* stream */
    long offset;	/* byte to seek to */
    int ptrname;	/* type of seek (L_SET,L_INCR,L_XTND) */

    {
    FAST int resync;
    FAST int c;
    long p;
    long curpos = ERROR;

    fp->flag &= ~_STDIO_EOF;
    if (fp->flag & _STDIO_READ)
	{
	if (ptrname < L_XTND && fp->base && !(fp->flag & _STDIO_NBF))
	    {
	    c = fp->count;
	    p = offset;
	    if (ptrname == L_SET)
		{
		curpos = lseek (fileno (fp), 0L, L_INCR);
		if (curpos == ERROR)
		    return (ERROR); 
		p += c - curpos;
		}
	    else
		offset -= c;

	    if (!(fp->flag & _STDIO_RW) && c > 0 &&
		p <= c && p >= fp->base - fp->ptr)
		{
		fp->ptr += (int)p;
		fp->count -= (int)p;
		if (curpos == ERROR)
		    curpos = lseek (fileno (fp), 0L, L_INCR);

		return ((curpos == ERROR) ? ERROR : OK);
		}

	    resync = offset & 01;
	    }
	else 
	    resync = 0;

	if (fp->flag & _STDIO_RW)
	    {
	    fp->ptr = fp->base;
	    fp->flag &= ~_STDIO_READ;
	    resync = 0;
	    }

	p = lseek (fileno (fp), offset - resync, ptrname);
	fp->count = 0;

	if (resync && getc (fp) != EOF && p != ERROR)
	    p++;
	}
    else if (fp->flag & (_STDIO_WRT|_STDIO_RW))
	{
	fflush (fp);
	if (fp->flag & _STDIO_RW)
	    {
	    fp->count = 0;
	    fp->flag &= ~_STDIO_WRT;
	    fp->ptr = fp->base;
	    }

	p = lseek (fileno (fp), offset, ptrname);
	}

    return ((p == ERROR) ? ERROR : OK);
    }
/*******************************************************************************
*
* ftell - return the current offset in the stream
*
* The routine ftell returns the current value, in bytes,
* of the offset relative to the beginning of the file
* associated with the named stream.
*
* INTERNAL
* Coordinates with buffering.
*
* RETURNS:
*    offset, or
*    ERROR for invalid stream
*
* SEE ALSO: fseek(2), fopen(2)
*/

long ftell (fp)
    FAST FILE *fp;	/* stream on which to report */

    {
    FAST long pos;
    FAST int adjust;

    if (fp->count < 0)
	fp->count = 0;

    if (fp->flag & _STDIO_READ)
	adjust = - fp->count;
    else if (fp->flag & (_STDIO_WRT | _STDIO_RW))
	{
	adjust = 0;
	if ((fp->flag & _STDIO_WRT) && fp->base && (fp->flag & _STDIO_NBF) == 0)
	    adjust = fp->ptr - fp->base;
	}
    else
	return (ERROR);

    pos = lseek (fileno (fp), 0L, L_INCR);

    if (pos < 0)
	return (pos);

    return (pos + adjust);
    }
/*******************************************************************************
*
* fwrite - buffered binary output
*
* The routine fwrite appends at most `count' items of data of the size `size',
* from the block beginning at `ptr' to the stream `fp'.
*
* RETURNS:
*    number of items written, or
*    0 on error
*
* SEE ALSO: write(2), putc(2), puts(2)
*/

int fwrite (ptr, size, count, fp)
    FAST char *ptr;	/* where to copy from */
    unsigned size;	/* size of item       */
    unsigned count;	/* number of items    */
    FAST FILE *fp;	/* output stream      */

    {
    FAST int s = size * count;

    if (fp->flag & _STDIO_LBF)
	while (s > 0)
	    {
	    if (--fp->count > -fp->bufsize && *ptr != '\n')
		*fp->ptr++ = *ptr++;
	    else if (stdioFlushBuf(*(unsigned char *)ptr++, fp) == EOF)
		break;
	    s--;
	    }
    else while (s > 0)
	{
	if (fp->count < s)
	    {
	    if (fp->count > 0)
		{
		bcopy(ptr, fp->ptr, fp->count);
		ptr += fp->count;
		fp->ptr += fp->count;
		s -= fp->count;
		}

	    if (stdioFlushBuf (*(unsigned char *)ptr++, fp) == EOF)
		break;

	    s--;
	    }

	if (fp->count >= s)
	    {
	    bcopy(ptr, fp->ptr, s);
	    fp->ptr += s;
	    fp->count -= s;
	    return (count);
	    }
	}

    return (size != 0 ? count - ((s + size - 1) / size) : 0);
    }
#undef getchar
/*******************************************************************************
*
* getchar - return next character in standard input stream
*
* The same as the invocation: getc(stdin).
*
* RETURNS: next character in stdin
*/

int getchar ()

    {
    return (getc (stdin));
    }
/*******************************************************************************
*
* gets - read a string from standard input stream
*
* A string is read from stdin into the buffer `s'.
* The string is terminated by a newline character which is replaced
* in `s' by NULL.
*
* RETURNS: argument `s', or NULL on end-of-file or error
*
* SEE ALSO: fopen(2), fread(2)
*/

char *gets (s)
    char *s;		/* buffer to hold characters read */

    {
    FAST int c;
    FAST char *cs = s;

    while ((c = getchar()) != '\n' && c != EOF)
	*cs++ = c;

    if (c == EOF && cs == s)
	return (NULL);

    *cs++ = EOS;

    return (s);
    }
/*******************************************************************************
*
* getw - read next word (32-bit integer) from stream
*
* The routine getw reads the next 32-bit quantity from stream `fp'.
* The routine returns EOF on end-of-file or error, however this
* is also a valid integer so feof and ferror must be used to check
* for a real end-of-file.
*
* RETURNS:
*    32-bit number from stream, or
*    EOF on end-of-file or error
*    but check feof(2) and ferror(2)
*
* SEE ALSO: fopen(2), getc(2)
*/

int getw (fp)
    FAST FILE *fp;	/* input stream */

    {
    int w;
    FAST char *p = (char *)&w;
    FAST int i = sizeof (int);

    while (--i >= 0)
	*p++ = getc (fp);

    if (feof (fp))
	return (EOF);

    return (w);
    }
#undef putchar
/*******************************************************************************
*
* putchar - put character on standard output stream
*
* The same as the invocation: putc(c, stdout).
* This function replicates the macro by the same name.
* This is useful for invocations from the shell.
*
* RETURNS: character written, or EOF on error
*/

int putchar (c)
    int c;	/* character to output */

    {
    return (putc (c, stdout));
    }
/*******************************************************************************
*
* puts - copy NULL terminated string to output stream
*
* This routine is the same fputs(s,stdout) followed by output of a newline.
*
* RETURNS: last character written (viz. newline), or EOF
*
* SEE ALSO: fopen(2), gets(2)
*/

int puts (s)
    FAST char *s;	/* string to copy to output */

    {
    FAST int c;

    while (c = *s++)
	putchar (c);

    return (putchar ('\n'));
    }
/*******************************************************************************
*
* putw - append word (32-bit integer) to output stream
*
* The routine putw writes the 32-bit quantity `w' to the stream `fp'.
*
* RETURNS: the value written
*/

int putw (w, fp)
    int w;		/* word (32-bit integer) */
    FAST FILE *fp;	/* stream */

    {
    FAST i = sizeof (int);
    FAST char *p = (char *)&w;

    while (--i >= 0)
	putc (*p++, fp);

    /* XXX should return EOF on ferror??? */
    return (w);
    }
/*******************************************************************************
*
* rewind - position stream at the beginning
*
* Same as the call: fseek (fp, 0L, 0).
*
* SEE ALSO: fseek(2), ftell(2), fopen(2)
*/

VOID rewind (fp)
    FAST FILE *fp;

    {
    fflush (fp);
    (void) lseek (fileno (fp), 0L, L_SET);

    fp->count = 0;
    fp->ptr   = fp->base;
    fp->flag &= ~(_STDIO_ERR|_STDIO_EOF);

    if (fp->flag & _STDIO_RW)
	fp->flag &= ~(_STDIO_READ|_STDIO_WRT);
    }
/*******************************************************************************
*
* scanf - read and convert from standard input stream
*
* Reads characters from standard input stream,
* interprets them according to format in string `fmt',
* and assigns values to arguments starting with the one
* pointed to by `args'.
*
* See sscanf (2) for full description of format specification.
*
* RETURNS: number of items scanned, or EOF on end-of-file
*
* SEE ALSO: fopen (2), fscanf (2), sscanf (2), K & R C manual
*
* VARARGS1
*/

int scanf (fmt, args)
    char *fmt;		/* format specification */
    int *args;		/* where to store scanned values */

    {
    return (doscan (stdin, fmt, &args));
    }
/*******************************************************************************
*
* fscanf - read and convert from input stream
*
* Reads characters from stream, interprets them according to format
* in string `fmt', and assigns values to arguments starting with the
* one pointed to by `args'.
*
* See sscanf (2) for full description of format specification.
*
* RETURNS: number of items scanned, or EOF on end-of-file
*
* SEE ALSO: fopen (2), sscanf (2), K & R C manual
*
* VARARGS2
*/

int fscanf (fp, fmt, args)
    FILE *fp;		/* stream to read from           */
    char *fmt;		/* format specification          */
    int *args;		/* where to store scanned values */

    {
    return (doscan (fp, fmt, &args));
    }
/*******************************************************************************
*
* setbuf - specify buffer to be used on stream
*
* The character array buf is used in place of the automatically allocated
* buffer.  If buf is NULL then the stream is unbuffered.
* Must be called before reading or writing on stream.
*
* The array must be of size BUFSIZE.  The routine setbuffer (2) allows
* a user specified size for the buffer.
*
* SEE ALSO: freopen (2), setbuffer (2)
*/

VOID setbuf (fp, buf)
    FAST FILE *fp;	/* stream */
    char *buf;		/* buffer to be used for stream; NULL = unbuffered */

    {
    if (fp->base != NULL && fp->flag & _STDIO_MYBUF)
	free (fp->base);

    fp->flag &= ~(_STDIO_MYBUF|_STDIO_NBF|_STDIO_LBF);

    if ((fp->base = buf) == NULL)
	{
	fp->flag |= _STDIO_NBF;
	fp->bufsize = NULL;
	}
    else
	{
	fp->ptr = fp->base;
	fp->bufsize = BUFSIZE;
	}
    fp->count = 0;
    }
/*******************************************************************************
*
* setbuffer - set buffer to be used on stream
*
* The character array buf is used in place of the automatically allocated
* buffer.  If buf is NULL then the stream is unbuffered.
* Must be called before reading or writing on stream.
* The routine setbuf (2) may be used if the buffer size is to be of
* size BUFSIZ.
*
* SEE ALSO: freopen (2), setbuf (2)
*/

VOID setbuffer (fp, buf, size)
    FAST FILE *fp;	/* stream */
    char *buf;		/* buffer to be used for stream; NULL = unbuffered */
    int size;		/* size of buffer */

    {
    if (fp->base != NULL && fp->flag & _STDIO_MYBUF)
	free (fp->base);

    fp->flag &= ~(_STDIO_MYBUF|_STDIO_NBF|_STDIO_LBF);

    if ((fp->base = buf) == NULL)
	{
	fp->flag |= _STDIO_NBF;
	fp->bufsize = NULL;
	}
    else
	{
	fp->ptr = fp->base;
	fp->bufsize = size;
	}

    fp->count = 0;
    }
/*******************************************************************************
*
* setlinebuf - set line buffering for either stdout or stderr
*
* At any time, the streams stdout or stderr can be changed from
* block buffered, or unbuffered, to line buffered.
*
* SEE ALSO: freopen(2)
*/

VOID setlinebuf (fp)
    FAST FILE *fp;	/* stream */

    {
    char *buf;

    fflush (fp);

    setbuffer (fp, (char *)NULL, 0);
    
    if ((buf = malloc (BUFSIZE)) != NULL)
	{
	setbuffer (fp, buf, BUFSIZE);
	fp->flag |= _STDIO_LBF | _STDIO_MYBUF;
	}
    }
/*******************************************************************************
*
* ungetc - push character back into input stream
*
* This routine pushes `c' back into an input stream.  The next call
* to getc (2) will return `c'.  Only a one character push is guaranteed.
*
* RETURNS: character `c', or EOF if unable to push back
*
* SEE ALSO: fopen(2), getc(2)
*/

int ungetc (c, fp)
    int c;		/* character to push */
    FAST FILE *fp;	/* stream */

    {
    if (c == EOF || (fp->flag & (_STDIO_READ | _STDIO_RW)) == 0 ||
	fp->ptr == NULL || fp->base == NULL)
	{
	return (EOF);
	}

    if (fp->ptr == fp->base)
	fp->ptr++;

    fp->count++;
    *--fp->ptr = c;

    return (c);
    }

/* function versions of macros */
#undef	clearerr
/*******************************************************************************
*
* clearerr - reset the error and end-of-file indicators
*
* The error and end-of-file indicators on the specified stream are cleared.
*/

VOID clearerr (fp)
    FAST FILE *fp;	/* stream */

    {
    fp->flag &= ~(_STDIO_ERR|_STDIO_EOF);
    }
#undef	feof
/*******************************************************************************
*
* feof - determine if end-of-file has been read
*
* If an end-of-file has been read from the given input
* stream then TRUE is returned.
* The end-of-file indication remains until
* the stream is closed, or reset by clearerr(2).
*
* RETURNS: TRUE if end-of-file read, otherwise FALSE
*
* SEE ALSO: clearerr(2), ferror(2)
*/

int feof (fp)
    FAST FILE *fp;	/* stream */

    {
    return ((fp->flag & _STDIO_EOF) != 0);
    }
#undef	ferror
/*******************************************************************************
*
* ferror - determine if an error has occurred while reading or writing
*
* If an error has occurred while reading from or writing to the
* stream then TRUE is returned.  The error remains until
* the stream is closed, or reset by clearerr(2).
*
* RETURNS: TRUE if error has occurred, otherwise FALSE
*
* SEE ALSO: clearerr(2), feof(2)
*/

BOOL ferror (fp)
    FAST FILE *fp;	/* stream */

    {
    return ((fp->flag & _STDIO_ERR) != 0);
    }
#undef	fileno
/*******************************************************************************
*
* fileno - get file descriptor associated with stream
*
* This routine returns the file descriptor associated with the given `fp'.
*
* RETURNS: file descriptor
*
* SEE ALSO: open (2)
*/

int fileno (fp)
    FAST FILE *fp;	/* stream */

    {
    return (fp->fd);
    }
#undef	getc
/*******************************************************************************
*
* getc - return next character in input stream
*
* This function replicates the macro by the same name.
* This is useful for invocations from the shell.
*
* RETURNS: next character, or EOF on end-of-file or error
*/

int getc (fp)
    FAST FILE *fp;	/* stream */

    {
    return (--(fp)->count >= 0 ? (int)(*(unsigned char *)(fp)->ptr++)
			      : stdioFillBuf (fp));
    }
#undef	putc
/*******************************************************************************
*
* putc - append character to output stream
*
* This function replicates the macro by the same name.
* This is useful for invocations from the shell.
*
* RETURNS: character written, or EOF on error
*/

int putc (c, fp)
    int c;		/* character */
    FAST FILE *fp;	/* stream */

    {
    return ((--(fp)->count >= 0 ?
	(int)(*(unsigned char *)(fp)->ptr++ = (c)) :
	(((fp)->flag & _STDIO_LBF) && -(fp)->count < (fp)->bufsize ?
		((*(fp)->ptr = (c)) != '\n' ?
			(int)(*(unsigned char *)(fp)->ptr++) :
			stdioFlushBuf(*(unsigned char *)(fp)->ptr, fp)) :
		stdioFlushBuf((unsigned char)(c), fp))));
    }

/* support routines */

/*******************************************************************************
*
* makeFile - make a new file buffer
*
* RETURNS: pointer to newly malloc'd file, or NULL if out of memory
*/

LOCAL FILE *makeFile ()

    {
    FAST FILE *fp = (FILE *)malloc (sizeof (FILE));

    if (fp != NULL)
	{
	fp->count   = 0;
	fp->ptr     = NULL;
	fp->base    = NULL;
	fp->bufsize = 0;
	fp->flag    = 0;
	fp->fd      = 0;
	fp->taskId  = taskTcbX (0)->taskId;
	}

    return (fp);
    }
/*******************************************************************************
*
* putbuf - put buffer on output stream
*/

LOCAL VOID putbuf (buffer, nbytes, fp)
    char *buffer;	/* source */
    int nbytes;		/* number of bytes in source */
    FILE *fp;		/* stream */

    {
    FAST int ix;

    for (ix = 0; ix < nbytes; ix++)
	putc (buffer[ix], fp);
    }
/*******************************************************************************
*
* doprnt - put characters into stream
*/

LOCAL VOID doprnt (fmt, argp, fp)
    FAST char *fmt;	/* format specification */
    FAST int *argp;	/* bunch of arguments */
    FILE *fp;		/* stream */

    {
    fioFormat (fmt, argp, putbuf, fp);
    }
/*******************************************************************************
*
* doscan - low-level scan routine
*
* RETURNS: number of items scanned or ERROR
*/

LOCAL int doscan (fp, fmt, argp)
    FILE *fp;	/* stream */
    char *fmt;	/* format specification */
    int **argp;	/* where to stick resulting conversions */

    {
    BOOL suppress;		/* assignment was suppressed */
    int unget;
    int args_assigned = 0;
    int ch;

    FOREVER
	{
	unget = -1;

	switch (*fmt++)
	    {
	    case EOS:			/* format string exhausted */
		return (args_assigned);

	    case '%':			/* conversion specification */

		if (convert (&fmt, argp, &suppress, fgetc, (int)fp, &unget))
		    {
		    if (!suppress)
			{
			argp++;
			args_assigned++;		/* successful */
			}
		    }
		else
		    return (args_assigned);	/* unsuccessful */
		break;

	    case ' ':
	    case '\n':
	    case '\t':

		while (isspace (ch = getc (fp)))
		    ;

		if (ch != EOF)
		    ungetc (ch, fp);
		break;

	    default:			/* character to match */

		ch = getc (fp);
		if (ch != *fmt)
		    {
		    if (ch == EOF)
			return (ERROR);
		    ungetc (ch, fp);
		    return (args_assigned);	/* match failed */
		    }

		break;
	    }

	if (feof (fp))
	    return (args_assigned == 0 ? ERROR : args_assigned);

	if (unget != -1)
	    ungetc (unget, fp);
	}
    }
/*******************************************************************************
*
* stdioShow - display file pointer internals
*
* NOMANUAL
*/

VOID stdioShow (fp)
    FAST FILE *fp;

    {
    static char *fmt1 =
    "%9.9s %5.5s %9.9s %9.9s %9.9s %6.6s %2s %9.9s\n";
    static char *fmt2 =
    "%#9x %5d %#9x %#9x %#9d %#6x %2d %#9x\n";

    if (fp != NULL)
	{
	printf (fmt1, "fp", "count", "base", "ptr", "bufsize", "flag", "fd",
		"task");

	printf (fmt2, fp,
		fp->count, fp->base, fp->ptr, fp->bufsize, fp->flag, fp->fd,
		fp->taskId);
	}
    }
