/* sigLib.c - software signal facility library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01n,25may89,jcf  changed sigHandler to loop instead of suspending (pSOS fix).
01m,16nov88,rdc  sigRaise now checks to see if task was already suspended
		 before resuming it.
01l,06sep88,jcf  fixed semTake and semGive calls in sigHandlerCleanUp.
01k,18aug88,gae  documentation.
01j,08jul88,jcf  lint.
01i,02jul88,rdc  documentation.
01h,30jun88,rdc  post-review cleanup.
01g,22jun88,dnw  name tweaks.
01f,13jun88,rdc  cleanup (races and assorted bugs).  sigDeliverSignal
		 no longer malloc's context structure.
01e,30may88,dnw  changed to v4 names.
01d,29may88,dnw  fixed screw-up in 01c.
		 removed unnecessary routine sigTaskVecGet().
		 cleaned-up interaction with excLib.
01c,28may88,dnw  cleanup, lint.
01b,24may88,gae  cleanup, lint.
01a,12apr88,rdc  written.
*/

/*
DESCRIPTION
This library provides a UNIX BSD 4.3 compatible software signal facility.
Signals are used to communicate asynchronous events
within and between task contexts.
In many ways, signals are analogous to hardware interrupts.
The signal facility provides a set if 31 distinct signals.
A "signal handler" is bound to a particular signal with sigvec (2)
in much the same way that an interrupt service routine is
connected to an interrupt vector with intConnect (2).
A signal may be asserted by calling kill - this is analogous
to the occurrence of an interrupt or hardware exception.
Signals are blocked for the duration of the signal handler,
as interrupts are locked out for the duration of the interrupt service routine.
Tasks may block the occurrence of certain signals with sigblock (2)
and sigsetmask (2), as the interrupt level may be raised or lowered to
block out levels of interrupts.  If a signal is blocked when it is raised,
it's handler routine will be called when the signal becomes unblocked.

Signal handlers are passed several parameters and should be defined as:

.CS
VOID sigHandler (sig, code, sigContext)
    int sig;			/* signal number *
    int code;			/* additional code *
    SIGCONTEXT *sigContext;	/* context of task before signal *
    {
    .
    .
    .
    }
.CE

The code parameter is used to distinguish variants of a signal.
For instance, numeric overflow and zero divide both raise SIGFPE
(floating point exception) but have different a code parameter.


EXCEPTION PROCESSING:
Certain signals (defined below) are asserted automatically when
exceptions are raised.
This mechanism allows the user to install her own exception handlers.
This is useful for recovering from catastrophic events
such as bus errors, or arithmetic errors.
Typically, setjmp (2) is called to 
define the point in the program where control shall be restored,
and longjmp (2) is called in the signal handler to restore that context.  
Note that longjmp (2) restores the state of the task's signal mask and its 
onstack flag.  If no user defined handler is installed for the given signal, 
the default action is to log a message to the console and suspend the task.

UNIX BSD 4.3 defines a set of 31 signals.
Of these, only a handful are meaningful in the VxWorks environment.
The following may be asserted by VxWorks:

.CS
    signal:	code:			exception:
    -------     -----			----------
    SIGSEGV	NULL			bus error
    SIGBUS	BUS_ADDERR		address error
    SIGILL	ILL_ILLINSTR_FAULT	illegal instruction
    SIGFPE	FPE_INTDIV_TRAP		zero divide
    SIGFPE	FPE_CHKINST_TRAP	chk trap
    SIGFPE	FPE_TRAPV_TRAP		trapv trap
    SIGILL	ILL_PRIVVIO_FAULT	privilege violation
    SIGTRAP	NULL			trace exception
    SIGEMT 	EMT_EMU1010		line 1010 emulator
    SIGEMT	EMT_EMU1111		line 1111 emulator
    SIGILL	ILL_ILLINSTR_FAULT	coprocessor protocol violation
    SIGFMT	NULL			format error

    floating point coprocessor stuff 

    SIGFPE    	FPE_FLTBSUN_TRAP	compare unordered
    SIGFPE	FPE_FLTINEX_TRAP	inexact result
    SIGFPE	FPE_FLTDIV_TRAP		divide by zero
    SIGFPE	FPE_FLTUND_TRAP		underflow
    SIGFPE	FPE_FLTOPERR_TRAP	operand error
    SIGFPE	FPE_FLTOVF_TRAP		overflow
    SIGFPE	FPE_FLTNAN_TRAP		signaling "Not A Number"
.CE

Signals other than these may be used by an application,
with the warning that additional signals may be used by VxWorks in the future
(SIGUSR1 and SIGUSR2 will never be used by VxWorks).

IMPLEMENTATION NOTES AND CAVEATS:

If a signal is directed at a task that is pended, the signal handler will 
not be invoked until the task becomes runnable.
A future release of the VxWorks kernel will be able to start pended tasks,
and this problem will go away.  

SEE ALSO: UNIX BSD 4.3 documentation

INCLUDE FILE: sigLib.h
*/

#include "vxWorks.h"
#include "sigLib.h"
#include "taskLib.h"
#include "memLib.h"
#include "errno.h"

/* forward declarations */

LOCAL VOID sigCreateHook ();
LOCAL VOID sigDeleteHook ();
LOCAL VOID sigHandler ();
LOCAL VOID sigHandlerCleanUp ();
STATUS sigRaise ();


/*******************************************************************************
* 
* sigInit - initialize the signal facility library
*
* This routine initializes the signal facilities.
*
* WHEN TO CALL
* This routine is usually called from the system start-up routine,
* usrInit (2) in usrConfig (1), before interrupts are enabled.
*
* RETURNS: OK or ERROR if create/delete hooks could not be installed
*/

STATUS sigInit ()

    {
    static BOOL sigInstalled = FALSE;

    if (!sigInstalled)
	{
	if (taskCreateHookAdd (sigCreateHook) == ERROR) 
	    {
	    logMsg ("sigInit: couldn't add create hook.\n");
	    return (ERROR);
	    }
	if (taskDeleteHookAdd (sigDeleteHook) == ERROR)
	    {
	    logMsg ("sigInit: couldn't add delete hook.\n");
	    return (ERROR);
	    }

	sigInstalled = TRUE;
	}

    return (OK);
    }
/*******************************************************************************
* 
* sigCreateHook - task create hook for signal facility
*
* Allocate data structures for the signal facility
* at task creation time; install them in the task's tcb extension.
*/

LOCAL VOID sigCreateHook (pNewTcbX)
    TCBX *pNewTcbX;		/* pointer to new task's TCB */

    {
    FAST TASK_SIGNAL_INFO *pSignalInfo;

    /* allocate signal handler table and install in tcbx */

    pSignalInfo = (TASK_SIGNAL_INFO *) malloc (sizeof (TASK_SIGNAL_INFO));

    if (pSignalInfo == NULL)
	{
	logMsg
	    ("sigCreateHook: unable to install signal info - malloc failed!\n");
	return;
	}

    pNewTcbX->signalInfo = pSignalInfo;

    /* initialize all signal handlers to NULL */

    bzero ((char *) pSignalInfo, sizeof (TASK_SIGNAL_INFO));

    semInit (&pSignalInfo->sigPauseSem);
    pSignalInfo->taskPaused = FALSE;

    semInit (&pSignalInfo->sigMaskSem);
    semGive (&pSignalInfo->sigMaskSem);
    }
/*******************************************************************************
* 
* sigDeleteHook - task delete hook for signal facility
*
* Deallocate signal data structures when a task exits.
*/

LOCAL VOID sigDeleteHook (pTcbX)
    TCBX *pTcbX;		/* pointer to task's TCB */

    {
    if (pTcbX->signalInfo != NULL)
	free ((char *)pTcbX->signalInfo);
    }
/*******************************************************************************
* 
* sigvec - install a signal handler 
*
* The routine sigvec is used to bind a signal handler to a particular signal.
* It may be used to see if a handler has been bound to a particular signal.
*
* RETURNS: OK or ERROR if invalid signal number
*/

STATUS sigvec (sig, pVec, pOvec)
    int sig;		/* signal to which the handler will be attached */
    SIGVEC *pVec;	/* handler information if != NULL */
    SIGVEC *pOvec;	/* previous handler info is copyed here if != NULL */

    {
    FAST TASK_SIGNAL_INFO *pSignalInfo = taskTcbX(0)->signalInfo;

    if (sig < 1 || sig >= NUM_SIGNALS)
	return (ERROR);

    /* install signal handler in calling task's signal handler table */
    /* copy previous handler info into *pOvec if pOvec != NULL */

    if (pOvec != NULL)
	*pOvec = pSignalInfo->sigHandlers [sig];

    if (pVec != NULL)
	pSignalInfo->sigHandlers [sig] = *pVec;

    return (OK);
    }
/*******************************************************************************
* 
* sigstack - install a separate signal stack
*
* The routine sigstack may be used to specify an alternate stack which
* should be switched in for the duration of signal processing.  When a
* signal handler is installed with sigvec, the sv_flags element of the
* SIGVEC structure must have the SV_ONSTACK bit turned on for the signal
* stack to be switched in.
*
* RETURNS: OK (always)
*/

STATUS sigstack (pSs, pOss)
    SIGSTACK *pSs;	/* new signal stack info if != NULL */
    SIGSTACK *pOss;	/* copy old signal stack info here if != NULL */

    {
    FAST TASK_SIGNAL_INFO *pSignalInfo = taskTcbX(0)->signalInfo;

    if (pOss)
	*pOss = pSignalInfo->sigStack;

    if (pSs)
	pSignalInfo->sigStack.ss_sp = pSs->ss_sp;

    return (OK);
    }
/*******************************************************************************
* 
* sigsetmask - set the signal mask 
*
* The signal mask for the calling task is set to the given value.
* A one in the bit mask indicates that the given signal is blocked
* from delivery.
* Use the macro SIGMASK to construct the mask for a given signal number.
*
* RETURNS: previous value of the signal mask
*
* SEE ALSO: sigblock (2)
*/

int sigsetmask (mask)
    int mask;		/* new signal mask */

    {
    FAST TASK_SIGNAL_INFO *pSignalInfo = taskTcbX(0)->signalInfo;
    FAST int maskBuf;

    semTake (&pSignalInfo->sigMaskSem);
    maskBuf = pSignalInfo->sigMask;
    pSignalInfo->sigMask = mask;
    semGive (&pSignalInfo->sigMaskSem);
    
    /* test for newly runnable signals */

    if (~mask & pSignalInfo->sigPendingMask)
	sigPendedRaise (taskIdSelf (), pSignalInfo);

    return (maskBuf);
    }
/*******************************************************************************
* 
* sigblock - add to set of blocked signals 
*
* The signals in mask are added to the task's set of blocked signals.
* A one in the bit mask indicates that the given signal is blocked
* from delivery.
* Use the macro SIGMASK to construct the mask for a given signal number.
*
* RETURNS: previous value of the signal mask
*
* SEE ALSO: sigsetmask (2)
*/

int sigblock (mask)
    int mask;	/* mask of additional signals to be blocked */

    {
    FAST TASK_SIGNAL_INFO *pSignalInfo = taskTcbX(0)->signalInfo;
    FAST int maskBuf;

    semTake (&pSignalInfo->sigMaskSem);
    maskBuf = pSignalInfo->sigMask;
    pSignalInfo->sigMask |= mask;
    semGive (&pSignalInfo->sigMaskSem);

    return (maskBuf);
    }
/*******************************************************************************
* 
* sigPendedRaise - deliver pended signal
*
* Find and deliver a previously pended signal.
*/

LOCAL VOID sigPendedRaise (task, pSignalInfo)
    FAST int task;		/* task id */
    FAST TASK_SIGNAL_INFO *pSignalInfo;

    {
    FAST int signal;
    FAST int readyPendedSigs = 
	 ~pSignalInfo->sigMask & pSignalInfo->sigPendingMask;

    /* extract the signal from the pended signals mask */

    for (signal = 1; signal < NUM_SIGNALS; signal++)
	{
	if (readyPendedSigs & 1)
	    break;		/* found one */
	readyPendedSigs >>= 1;
	}

    if (signal == NUM_SIGNALS)
	return;		/* huh? */

    /* un-pend the signal */

    pSignalInfo->sigPendingMask &= ~(SIGMASK (signal));

    /* we deliver the signal by invoking sigRaise via excTask */

    excJobAdd (sigRaise, task, signal, NULL);
    }
/*******************************************************************************
* 
* pause - sleep until signal 
*
* pause may be used to block task execution until the occurrence of 
* a signal.  After the signal handler has executed, pause will return.
*
* RETURNS: ERROR (always)
*/

STATUS pause ()

    {
    FAST TASK_SIGNAL_INFO *pSignalInfo = taskTcbX(0)->signalInfo;

    if (pSignalInfo == NULL)
	{
	logMsg ("pause: signals not installed for task!\n");
	errnoSet (S_sigLib_NO_HANDLER_FOR_SIGNAL);
	return (ERROR);
	}

    pSignalInfo->taskPaused = TRUE;
    semTake (&pSignalInfo->sigPauseSem);
    errnoSet (EINTR);
    return (ERROR);
    }
/*******************************************************************************
* 
* kill - send a signal to a task
*
* The function kill sends a signal to a given task.
* The task must have previously installed a signal handler with sigvec (2).
*
* CAVEAT
* If the task is pended, the signal will not be delivered until
* the task is made runnable.
*
* RETURNS:
*    OK or
*    ERROR if task not found, or
*    has no signal handler for signal, or
*    signal cannot be delivered
*
* SEE ALSO: sigvec (2), sigRaise (2)
*/

STATUS kill (tid, signal)
    int tid;    /* task to which the signal is directed */
    int signal;	/* the signal to send the task */

    {
    return (sigRaise (tid, signal, NULL));
    }
/*******************************************************************************
* 
* sigRaise - send a signal to a task
*
* The function sigRaise sends a signal to a given task.
* It provides the same mechanism as kill (2), but allows
* an additional code to passed on to the signal handler.
*
* RETURNS:
*    OK or
*    ERROR if task not found, or
*    has no signal handler for signal, or
*    signal cannot be delivered
*
* SEE ALSO: sigvec (2), kill (2)
*/

STATUS sigRaise (tid, signal, code)
    int tid;    /* task to which the signal is directed */
    int signal;	/* the signal to send the task */
    int code;	/* additional code */

    {
    TCBX *pTcbx;
    FAST TASK_SIGNAL_INFO *pSignalInfo;
    FAST SIGVEC *pSigvec;
    FAST SIGCONTEXT *pContext; 
    int *sp;		/* MUST be "int *" because of auto-decrement below! */
    USHORT sr;
    int sigMaskBuf;
    int dregs [8];
    int aregs [7];
    INSTR *pc;
    BOOL resumeTask;

    /* check that task exists and has a handler for this signal */

    if ((pTcbx = taskTcbX (tid)) == NULL)
	return (ERROR);

    if (((pSignalInfo = pTcbx->signalInfo) == NULL) ||
        ((pSigvec = &pSignalInfo->sigHandlers[signal])->sv_handler == NULL))
	{
	errnoSet (S_sigLib_NO_HANDLER_FOR_SIGNAL);
	return (ERROR);
	}

    /* if we are at interrupt level or signaling to self,
     * we have the exception task deliver the signal */

    if (intContext () || (taskIdSelf () == tid))
	{
	excJobAdd (sigRaise, tid, signal, code);
	return (OK);
	}

    /* if signal is blocked, set task's pending signal mask and return */

    /* CRITICAL SECTION: take sigMask semaphore to ensure mutual exclusion */

    semTake (&pSignalInfo->sigMaskSem);

    /* this operation checks the signal mask and sets the signal pending
     * mask - IT MUST BE INDIVISABLE - all routines that set the signal
     * mask must first take the sigMaskSem.
     */

    if (pSignalInfo->sigMask & SIGMASK (signal))
	{
	pSignalInfo->sigPendingMask |= SIGMASK (signal);
	semGive (&pSignalInfo->sigMaskSem);
	return (OK);
	}

    /* make sure the task is suspended, so we don't have a moving target */

    if (taskIsSuspended (tid))
	resumeTask = FALSE;
    else
	{
	taskSuspend (tid);
	resumeTask = TRUE;
	}

    /* block the signal for the task, and or in the the handler's mask */

    sigMaskBuf = pSignalInfo->sigMask;
    pSignalInfo->sigMask |= (SIGMASK (signal) | pSigvec->sv_mask);

    if (taskRegsGet (tid, dregs, aregs, (char **) &sp, &sr, &pc) == ERROR)
	{
	semGive (&pSignalInfo->sigMaskSem);
	return (ERROR);
	}

    /* make room for the context structure on the task's stack */

    pContext = (SIGCONTEXT *) ((int)sp - sizeof (SIGCONTEXT) - sizeof (int));
    pContext->sc_sp = (int) sp;
    sp = (int *)((int) sp - sizeof (SIGCONTEXT) - sizeof (int));

    /* fill in the rest of the context structure */

    pContext->sc_ps      = (int) sr;
    pContext->sc_onstack = pSignalInfo->sigStack.ss_onstack;
    pContext->sc_mask    = sigMaskBuf;
    bcopy ((char *) aregs, (char *) pContext->sc_aregs, sizeof (aregs));
    bcopy ((char *) dregs, (char *) pContext->sc_dregs, sizeof (dregs));
    pContext->sc_pc = (int) pc;

    /* see if we should be switching stacks */

    if ((pSigvec->sv_flags & SV_ONSTACK) && !pSignalInfo->sigStack.ss_onstack)
	{
	pSignalInfo->sigStack.ss_onstack = TRUE;
	sp = (int *) pSignalInfo->sigStack.ss_sp;
	}

    /* push sigHandler parameters onto stack (in reverse order) */

    *(--sp) = code;		
    *(--sp) = signal;
    *(--sp) = (int) pContext; 
    *(--sp) = (int) pSigvec->sv_handler;
    *(--sp) = 0xbcbcbcbc; 	/* dummy return address */

    /* set up the new context - put sigHandler in task's pc in tcb,
     * set the new sp. */

    if (taskRegsSet (tid, pContext->sc_dregs, pContext->sc_aregs, (char *) sp, 
		    (USHORT)pContext->sc_ps, (INSTR *)(int)sigHandler) == ERROR)
	{
	semGive (&pSignalInfo->sigMaskSem);
	return (ERROR);
	}

    /* check to see if the task is "paused" */

    if (pSignalInfo->taskPaused)
	{
	/* crank it up */
	pSignalInfo->taskPaused = FALSE;
	semGive (&pSignalInfo->sigPauseSem);
	}

    semGive (&pSignalInfo->sigMaskSem);

    /* END CRITICAL SECTION */

    /* resume the task if it wasn't already suspended */

    if (resumeTask)
	taskResume (tid);

    return (OK);
    }
/*******************************************************************************
* 
* sigHandler - preamble and cleanup for user signal handler
*/

LOCAL VOID sigHandler (handler, context, signal, code)
    FUNCPTR handler;
    SIGCONTEXT *context;
    int signal;
    int code;

    {
    /* call the user signal handler */
     
    (*handler) (signal, code, context);

    /* excTask cleans up */

    excJobAdd (sigHandlerCleanUp, context, taskIdSelf ());

    /* never come back, but just in case ... */

    FOREVER
	taskDelay (1);		/* we delay because suspend tricks PSOS */
    }
/*******************************************************************************
* 
* sigHandlerCleanUp - restore task's previous context
*
* sigHandlerCleanUp is called via excTask by sigHandler ()
* to restore the task's context previous to the current signal.
*/

LOCAL VOID sigHandlerCleanUp (context, task)
    SIGCONTEXT *context;	/* context to restore */
    int task;			/* task id */

    {
    FAST TASK_SIGNAL_INFO *pSignalInfo;
    FAST TCBX *pTcbx = taskTcbX (task);

    if (pTcbx == NULL)
	{
	logMsg ("sigHandlerCleanUp: nonexistent task %#x\n", task);
	return;
	}
    
    pSignalInfo = pTcbx->signalInfo;

    /* restore the task's previous context */

    if (taskRegsSet (task, context->sc_dregs, context->sc_aregs, 
		     (char *) context->sc_sp, (USHORT) context->sc_ps,
		     (INSTR *) context->sc_pc) == ERROR)
	{
	logMsg ("sigHandlerCleanUp: task regs error with task %#x\n", task);
	return;
	}

    /* restore the previous onstack flag and signal mask */

    pSignalInfo->sigStack.ss_onstack = context->sc_onstack;

    semTake (&pSignalInfo->sigMaskSem);
    pSignalInfo->sigMask = context->sc_mask;
    semGive (&pSignalInfo->sigMaskSem);
   
    /* test for newly runnable signals */

    if (~pSignalInfo->sigMask & pSignalInfo->sigPendingMask)
	sigPendedRaise (task, pSignalInfo);
    }
/*******************************************************************************
* 
* sigOnstackFlagSet - set the onstack flag 
*
* This routine is provided for longjmp (2).
*
* RETURNS: OK | ERROR if task not found
*
* NOMANUAL
*/

STATUS sigOnstackFlagSet (onstack)
    int onstack;	/* onstack flag to set */

    {
    FAST TASK_SIGNAL_INFO *pSignalInfo;
    FAST TCBX *pTcbx = taskTcbX (0);

    if (pTcbx == NULL || (pSignalInfo = pTcbx->signalInfo) == NULL)
	return (ERROR);

    pSignalInfo->sigStack.ss_onstack = onstack;

    return (OK);
    }
/*******************************************************************************
* 
* sigOnstackFlagGet - return the calling task's onstack flag
*
* This routine is provided for setjmp (2).
*
* RETURNS: onstack flag
*
* NOMANUAL
*/

int sigOnstackFlagGet ()

    {
    FAST TASK_SIGNAL_INFO *pSignalInfo;
    FAST TCBX *pTcbx = taskTcbX (0);

    pSignalInfo = pTcbx->signalInfo;

    /* restore the previous onstack flag and signal mask */

    return (pSignalInfo->sigStack.ss_onstack);
    }
