/* rpcLib.c - rpc support library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01q,23apr89,gae  rpcDeleteHook now invokes each rpc facility's cleanup routine.
		 Removed documentation on callrpc's failings -- fixed.
		 changed rpcTaskInit to not bother passing the module list.
01p,08apr89,dnw  changed rpcInit() to call taskVarInit().
		 changed rpcInit() to remember if init has already been done.
01o,26mar89,llk  added a call to nfsClientCacheCleanUp in rpcTaskDeleteHook.
01n,30jun88,llk  moved rpcClntErrnoSet() here from nfsLib.
01m,22jun88,dnw  name tweaks.
01l,06jun88,dnw  changed taskSpawn/taskCreate args.
01k,30may88,dnw  changed to v4 names.
01j,13apr88,gae  lint, documentation, changed parm. to taskVar{Add,Del}().
		 fixed portmapd to run at priority 100.  Made rpcTaskExit()
		 local and made it become automatic as a task delete hook.
		 checked for valid nfsClientCache in rpcTaskExit().
01j,19apr88,llk  added nfsClientCache field to taskModuleList.
01i,30mar88,rdc  beefed up documentation.
01h,22feb88,jcf  made kernel independent.
01g,29dec87,rdc  rpcTaskInit now checks to see if it has already been called.
01f,13dec87,rdc  added rpcTaskExit.
01e,04dec87,rdc  removed perror (now in unixLib.c).
01d,05nov87,dnw  moved definition of taskModuleList here from rpcGbl.h.
01c,23oct87,dnw  changed rpcInit to rpcTaskInit.
		 added new rpcInit that initializes overall rpc facilities.
01b,14oct87,rdc	 added perror and abort.
01a,04oct87,rdc	 written.
*/

/* 
VxWorks supports the SUN Microsystems' Remote Procedure Call library.  
RPC provides facilities for implementing client/server based architectures.
The underlying interprocess communication mechanism can be completely hidden,
permitting applications to be written without any reference to network sockets.
The package is structured such that a sophisticated user may access
lower level routines allowing greater control of the communication protocols.
The user is directed to SUN's "Remote Procedure Call Programming Guide" for
a complete discussion and tutorial.  An example of the use of RPC is provided
in the VxWorks demonstration directory; it is called "sprites".

The RPC facility is enabled by defining the constant "INCLUDE_RPC" in
the target CPU config.h file, e.g.:
.CS
    #define INCLUDE_RPC
.CE
VxWorks supports NFS (Network File System) and SUN's dbxtool,
both of which are built on top of RPC.
If either NFS or DBX are configured into the system, then RPC is as well.

IMPLEMENTATION
Every task which is going to make calls to the RPC library must first call
rpcTaskInit (2).  This routine installs a VxWorks task variable which contains
pointers to static data structures in the RPC package that must be switched
on each task context switch.

RPC related objects (such as SVCXPRT's and CLIENT's) may not be 
shared among tasks;  objects created by one task may not be passed to 
another for use.  This is because the RPC package
contains static data structures that refer to these objects, and the
VxWorks implementation of RPC uses VxWorks task variables to switch
in a different copy of the static data structures for each task.

INCLUDE FILE: rpc.h

SEE ALSO: SUN RPC programmers guide, nfsLib (1), nfsDrv (3), dbxLib (1)
*/

#include "rpctypes.h"
#include "vxWorks.h"
#include "rpcGbl.h"
#include "taskLib.h"
#include "memLib.h"


IMPORT VOID portmapd ();
IMPORT struct moduleStatics *auth_noneInit ();
IMPORT struct moduleStatics *clnt_rawInit ();
IMPORT struct moduleStatics *clnt_simpleInit ();
IMPORT struct moduleStatics *svcInit ();
IMPORT struct moduleStatics *svc_rawInit ();
IMPORT struct moduleStatics *svc_simpleInit ();


MODULE_LIST *taskModuleList;	/* task variable cell that points to rpc
				 * task structure */

int portmapdId;
int portmapdPriority  = 100;
int portmapdStackSize = 5000;
int portmapdOptions   = VX_SUPERVISOR_MODE | VX_UNBREAKABLE;

/* forward declarations */

LOCAL VOID rpcTaskDeleteHook ();


/*******************************************************************************
*
* rpcInit - initialize RPC package
*
* This routine must be called before using RPC; it spawns the portmap daemon.
* It is usually called by the root task, usrRoot(2), in usrConfig(1).
*
* RETURNS: OK or ERROR if unable to spawn portmapd
*/

STATUS rpcInit ()

    {
    static BOOL rpcInitialized = FALSE;		/* TRUE = rpc inited */

    if (!rpcInitialized)
	{
	/* make sure task variables facility is installed */

	if (taskVarInit () == ERROR)
	    return (ERROR);

	/* spawn the portmap daemon */

	portmapdId = taskSpawn ("portmapd", portmapdPriority,
				portmapdOptions, portmapdStackSize,
				portmapd, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);

	if (portmapdId == ERROR)
	    return (ERROR);

	rpcInitialized = TRUE;
	}

    return (OK);
    }
/*******************************************************************************
*
* rpcTaskInit - initialize task's access to RPC package
*
* This routine must be called by a task before it makes any calls to 
* other routines in the RPC package.
* 
* RETURNS: OK or ERROR if out of memory, or unable to add task switch hooks
*/

STATUS rpcTaskInit ()

    {
    static BOOL rpcInstalled = FALSE;
    FAST MODULE_LIST *pModuleList;

    /* if rpcTaskInit has already been called for this task, 
       the task variable will no longer be null */

    if (taskModuleList != NULL)
	return (OK);

    /* allocate the block of pointers which becomes the task switch variable */

    pModuleList = (MODULE_LIST *) malloc (sizeof (MODULE_LIST));

    if (pModuleList == NULL)
	return (ERROR);

    bzero ((char *) pModuleList, sizeof (MODULE_LIST));

    /* each module contains an initialization routine which is responsible
       for allocating its own static variable space and filling in the 
       pointer to that space in the supplied MODULE_LIST */

    if ((pModuleList->auth_none = auth_noneInit ())	== NULL ||
	(pModuleList->clnt_raw = clnt_rawInit ())	== NULL ||
	(pModuleList->clnt_simple = clnt_simpleInit ())	== NULL ||
	(pModuleList->svc = svcInit ())			== NULL ||
	(pModuleList->svc_raw = svc_rawInit ())		== NULL ||
	(pModuleList->svc_simple = svc_simpleInit ())	== NULL)
        {
        return (ERROR);
        }

    /* NOTE: nfsClientCache may be set by nfs */

    /* now install taskModuleList as a task variable */

    if (taskVarAdd (0, (int *) &taskModuleList) == ERROR)
	return (ERROR);

    taskModuleList = pModuleList;

    if (!rpcInstalled)
	{
	if (taskDeleteHookAdd (rpcTaskDeleteHook) == ERROR)
	    return (ERROR);
	rpcInstalled = TRUE;
	}

    return (OK);
    }
/*******************************************************************************
*
* rpcTaskDeleteHook - deallocate RPC resources of exiting task
*
* This routine is the task delete hook for tasks using the RPC package.
* It is installed by rpcTaskInit.
*/

LOCAL VOID rpcTaskDeleteHook (pTcbX)
    FAST TCBX *pTcbX;	/* TCB extension of exiting task */

    {
    FAST MODULE_LIST *pModuleList;
    
    pModuleList = (MODULE_LIST *) taskVarGet (pTcbX->taskId,
					      (int *) &taskModuleList);

    if ((int) pModuleList == ERROR)
	return;	/* task didn't use RPC */

    /* free all the module list data structures */

    auth_noneExit (pModuleList->auth_none);
    clnt_rawExit (pModuleList->clnt_raw);
    clnt_simpleExit (pModuleList->clnt_simple);
    svcExit (pModuleList->svc);
    svc_rawExit (pModuleList->svc_raw);
    svc_simpleExit (pModuleList->svc_simple);

    if ((char *)pModuleList->nfsClientCache != NULL)
	{
	nfsClientCacheCleanUp (pModuleList->nfsClientCache);
	free ((char *)pModuleList->nfsClientCache);
	}

    /* free the pModuleList itself */

    free ((char *)pModuleList);
    }
/*******************************************************************************
*
* abort - force a task to exit
*
* abort is provided for the handling of catastrophic RPC failures.
* It is referenced internally by the RPC library, and SHOULD NOT BE
* CALLED BY THE USER.
*
* NOMANUAL
*/

VOID abort ()

    {
    printErr ("task %d - aborted.\n", taskIdSelf ());
    exit (-1);
    }
/******************************************************************************
*
* rpcClntErrnoSet - set RPC client status
*
* rpcClntErrnoSet calls errnoSet with the given "rpc stat" or'd with the
* rpc status prefix.  
*
* NOMANUAL
*/

VOID rpcClntErrnoSet (status)
    enum clnt_stat status;

    {
    errnoSet (M_rpcClntStat | (int) status);
    }
