/* rngLib.c - ring buffer subroutine library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01n,22mar89,dab  changed shorts to ints in rngGetBuf, rngPutBuf. 
01m,03feb89,dab  added rngDelete to delete a ring buffer.
01l,20aug88,gae  documentation.
01k,07jul88,jcf  fixed malloc to match new declaration.
01j,30may88,dnw  changed to v4 names.
01i,03nov87,ecs  documentation.
01h,20oct87,gae  documentation.
01g,23mar87,jlf  documentation.
01f,21dec86,dnw  changed to not get include files from default directories.
01e,14apr86,rdc  changed memAllocates to mallocs. 
01d,20jul85,jlf  documentation.
01c,09sep84,jlf  added comments and copyright.
01b,15aug84,dnw  changed rngBufCreate to allocate ring buffer one byte larger
		   than requested, because ring buffer algorithm always leaves
		   at least one empty byte in buffer.
		 added rngEmpty, rngFull, rngFreeBytes, rngNBytes, rngFlush,
		   rngPutAhead, rngMoveAhead, some of which used to be
		   macros in rngLib.h.
		 changed rngBufGet,rngBufPut to rngGetBuf,rngPutBuf.
		 changed rngBufCreate to rngCreate.
01a,06jun84,dnw  culled from old drvLib.c
*/

/*
This library provides facilities for creating and using ring buffers,
which are first-in-first-out circular buffers.

The routines here have several important virtues.  They are fast, which
is often important in the sort of circumstances where they will be used.
They are also race-free.  Data may be added to and removed from the
same ring asynchronously without locking out or using a semaphore.
This makes it easy when, for example, bytes are added at interrupt level
and removed by a task, or added by one task and removed by another.
There can be problems, though, if two or more asynchronous processes are
adding to (or removing from) the same ring.  In that case, a semaphore is
necessary.

This library also supplies two macros, RNG_ELEM_PUT and RNG_ELEM_GET,
for putting and getting single bytes from a ring buffer.  These are somewhat
faster than the rngBufPut and rngBufGet subroutines, which can put and get 
multi-byte buffers.  These macros are defined in rngLib.h.
.CS
    int RNG_ELEM_GET (ringId, pch, fromP)
    int RNG_ELEM_PUT (ringId, ch, toP)
.CE
Both macros require a temporary variable (fromP or toP), which should
be declared as "register int" for maximum efficiency.
RNG_ELEM_GET returns 1 if there was a character in the buffer to be gotten,
0 otherwise.  RNG_ELEM_PUT returns 1 if there was room in the buffer, 0
otherwise.

INCLUDE FILE: rngLib.h
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "memLib.h"
#include "rngLib.h"

/*******************************************************************************
*
* rngCreate - create an empty ring buffer
*
* This routine creates a ring buffer of the specified size, and initializes
* it.  The buffer itself is allocated from the system memory pool.
*
* RETURNS
*  ID of the ring buffer, or
*  NULL if memory couldn't be allocated
*/

RING_ID rngCreate (nbytes)
    int nbytes;		/* number of bytes in ring buffer */

    {
    char *buffer;
    RING_ID ringId = (RING_ID) malloc (sizeof (RING));

    if (ringId == NULL)
	return (NULL);

    /* bump number of bytes requested because ring buffer algorithm
     * always leaves at least one empty byte in buffer */

    buffer = malloc ((unsigned) ++nbytes);

    if (buffer == NULL)
	{
	free ((char *)ringId);
	return (NULL);
	}

    ringId->bufSize = nbytes;
    ringId->buf	    = buffer;

    rngFlush (ringId);

    return (ringId);
    }
/*******************************************************************************
*
* rngDelete - delete a ring buffer
*
* This routine deletes the specified ring buffer.
* Any data currently in the buffer will be lost.
*/

VOID rngDelete (ringId)
    FAST RING_ID ringId;	/* ring buffer to delete */

    {
    free (ringId->buf);
    free ((char *)ringId);
    }
/*******************************************************************************
*
* rngFlush - make a ring buffer empty
*
* This routine initializes the specified ring buffer to be empty.
* Any data currently in the buffer will be lost.
*/

VOID rngFlush (ringId)
    FAST RING_ID ringId;	/* ring buffer to initialize */

    {
    ringId->pToBuf   = 0;
    ringId->pFromBuf = 0;
    }
/*******************************************************************************
*
* rngBufGet - get characters from ring buffer
*
* Copies bytes from the ring buffer into your local
* buffer.  It will copy as many bytes as are in the ring, up to maxbytes.
* The bytes copied will be removed from the ring.
* 
* RETURNS:
* number of bytes actually gotten from ring buffer;
* may be zero if ring buffer empty at time of call
*/

int rngBufGet (rngId, buffer, maxbytes)
    FAST RING_ID rngId;		/* ring buffer to get data from      */
    char *buffer;		/* pointer to buffer to receive data */
    int maxbytes;		/* maximum number of bytes to get    */

    {
    FAST int bytesgot = 0;
    int pToBuf = rngId->pToBuf;
    int bytes2;

    if (pToBuf >= rngId->pFromBuf)
	{
	/* pToBuf has not wrapped around */

	bytesgot = min (maxbytes, pToBuf - rngId->pFromBuf);
	bcopy (&rngId->buf [rngId->pFromBuf], buffer, bytesgot);
	rngId->pFromBuf += bytesgot;
	}
    else
	{
	/* pToBuf has wrapped around.  Grab chars up to the end of the
	 * buffer, then wrap around if we need to. */

	bytesgot = min (maxbytes, rngId->bufSize - rngId->pFromBuf);
	bcopy (&rngId->buf [rngId->pFromBuf], buffer, bytesgot);
	rngId->pFromBuf += bytesgot;

	/* If pFromBuf is equal to bufSize, we've read the entire buffer,
	 * and need to wrap now.  If bytesgot < maxbytes, copy some more chars
	 * in now. */

	if (rngId->pFromBuf == rngId->bufSize)
	    {
	    bytes2 = min (maxbytes - bytesgot, pToBuf);
	    bcopy (rngId->buf, buffer + bytesgot, bytes2);
	    rngId->pFromBuf = bytes2;
	    bytesgot += bytes2;
	    }
	}
    return (bytesgot);
    }
/*******************************************************************************
*
* rngBufPut - put bytes to ring buffer
*
* This routine puts bytes from the specified buffer into the specified
* ring.  The specified number of bytes will be put into the ring, up to
* the number of bytes available in the ring.
*
* INTERNAL
* Always leaves at least one byte empty between pToBuf and pFromBuf, to
* eliminate ambiguities which could otherwise occur when the two pointers
* are equal.
*
* RETURNS:
* number of bytes actually put into ring buffer;
* may be less than number requested, even zero,
* if insufficient room in ring buffer at time of call
*/

int rngBufPut (rngId, buffer, nbytes)
    FAST RING_ID rngId;		/* ring buffer to put data into  */
    char *buffer;		/* buffer to get data from       */
    int nbytes;			/* number of bytes to try to put */

    {
    FAST int bytesput = 0;
    int pFromBuf = rngId->pFromBuf;
    int bytes2;

    if (pFromBuf > rngId->pToBuf)
	{
	/* pFromBuf is ahead of pToBuf.  We can fill up to two bytes
	 * before it */

	bytesput = min (nbytes, pFromBuf - rngId->pToBuf - 1);
	bcopy (buffer, &rngId->buf [rngId->pToBuf], bytesput);
	rngId->pToBuf += bytesput;
	}
    else if (pFromBuf == 0)
	{
	/* pFromBuf is at the beginning of the buffer.  We can fill till
	 * the next-to-last element */

	bytesput = min (nbytes, rngId->bufSize - rngId->pToBuf - 1);
	bcopy (buffer, &rngId->buf [rngId->pToBuf], bytesput);
	rngId->pToBuf += bytesput;
	}
    else
	{
	/* pFromBuf has wrapped around, and its not 0, so we can fill
	 * at least to the end of the ring buffer.  Do so, then see if
	 * we need to wrap and put more at the beginning of the buffer. */

	bytesput = min (nbytes, rngId->bufSize - rngId->pToBuf);
	bcopy (buffer, &rngId->buf [rngId->pToBuf], bytesput);
	rngId->pToBuf += bytesput;

	if (rngId->pToBuf == rngId->bufSize)
	    {
	    /* We need to wrap, and perhaps put some more chars */

	    bytes2 = min (nbytes - bytesput, pFromBuf - 1);
	    bcopy (buffer + bytesput, rngId->buf, bytes2);
	    rngId->pToBuf = bytes2;
	    bytesput += bytes2;
	    }
	}
    return (bytesput);
    }
/*******************************************************************************
*
* rngIsEmpty - test for ring buffer empty
*
* This routine determines if the specified ring buffer is empty.
*
* RETURNS:
*  TRUE if empty, FALSE if not
*/

BOOL rngIsEmpty (ringId)
    RING_ID ringId;	/* ring buffer to test */

    {
    return (ringId->pToBuf == ringId->pFromBuf);
    }
/*******************************************************************************
*
* rngIsFull - test for ring buffer full (no more room)
*
* This routine determines if the specified ring buffer is completely full.
*
* RETURNS:
*  TRUE if full, FALSE if not
*/

BOOL rngIsFull (ringId)
    FAST RING_ID ringId;	/* ring buffer to test */

    {
    int n = ringId->pToBuf - ringId->pFromBuf + 1;

    return ((n == 0) || (n == ringId->bufSize));
    }
/*******************************************************************************
*
* rngFreeBytes - determine number of free bytes in ring buffer
*
* This routine determines the number of bytes currently unused in the 
* specified ring buffer.
*
* RETURNS: number of bytes free in ring buffer
*/

int rngFreeBytes (ringId)
    FAST RING_ID ringId;	/* ring buffer to examine */

    {
    FAST int n = ringId->pFromBuf - ringId->pToBuf - 1;

    if (n < 0)
	n += ringId->bufSize;

    return (n);
    }
/*******************************************************************************
*
* rngNBytes - determine number of bytes in ring buffer
*
* This routine determines the number of bytes currently in the specified 
* ring buffer.
*
* RETURNS: number of bytes filled in ring buffer
*/

int rngNBytes (ringId)
    FAST RING_ID ringId;	/* ring buffer to be enumerated */

    {
    FAST int n = ringId->pToBuf - ringId->pFromBuf;

    if (n < 0)
	n += ringId->bufSize;

    return (n);
    }
/*******************************************************************************
*
* rngPutAhead - put a byte ahead in ring buffer without moving ring pointers
*
* This routine writes a byte into the ring, but doesn't move the ring buffer
* pointers.  Thus the byte will not yet be available in the ring by
* rngBufGet (2) calls.  The byte is written `offset' bytes ahead of the next
* input location in the ring.  Thus an offset of 0 puts the byte in the same
* place that RNG_ELEM_PUT would put a byte, except that the input pointer is not
* updated.
*
* Bytes written ahead in the ring buffer with this routine can be made available
* all at once by subsequently moving the ring buffer pointers with the routine
* rngMoveAhead (2).
*
* It is up the caller to verify that at least `offset' + 1 bytes are available
* in the ring buffer, before calling this routine.
*/

VOID rngPutAhead (ringId, byte, offset)
    FAST RING_ID ringId;	/* ring buffer to put byte in    */
    char byte;			/* byte to be put in ring        */
    int offset;			/* offset beyond next input byte */
				/* where to put byte             */
    {
    FAST int n = ringId->pToBuf + offset;

    if (n >= ringId->bufSize)
	n -= ringId->bufSize;

    *(ringId->buf + n) = byte;
    }
/*******************************************************************************
*
* rngMoveAhead - advance ring "to" pointer `n' bytes
*
* This routine advances the ring buffer input pointer by the specified
* number of bytes.  This is used to make that number of bytes available
* in the ring buffer, after having been written ahead in the ring buffer
* with rngPutAhead (2).
*/

VOID rngMoveAhead (ringId, n)
    FAST RING_ID ringId;  /* ring buffer to be advanced                  */
    FAST int n;		  /* number of bytes ahead to move input pointer */

    {
    n += ringId->pToBuf;

    if (n >= ringId->bufSize)
	n -= ringId->bufSize;

    ringId->pToBuf = n;
    }
