/* rlogLib.c - remote login library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
02e,06jun88,dnw  changed taskSpawn/taskCreate args.
02d,30may88,dnw  changed to v4 names.
02c,28may88,dnw  changed to use shellOrigStdSet (...) instead of shellSetOrig...
		 changed not to use shellTaskId global variable.
02b,31mar88,gae  made it work with I/O system revision.
02a,26jan88,jcf  made kernel independent.
01x,19nov87,dnw  made rlogin set terminal 7-bit option so that "~." and
		   XON-XOFF will be reliably detected.
		 cleaned-up rlogin loop.
		 changed rlogind to wait for shell to exist before accepting
		   remote connections.
	   +gae  fixed problem of rlogin() exiting before a possible
	           rlogind session gets to finish.
01w,18nov87,ecs  lint.
		 added include of inetLib.h.
01v,15nov87,dnw  changed rlogInit() to return status.
01u,03nov87,ecs  documentation.
	     &   fixed bug in use of setsockopt().
	    dnw  changed rlogin() to accept inet address as well as host name.
		 changed to call shellLogoutInstall() instead of logoutInstall.
01t,24oct87,gae  changed setOrig{In,Out,Err}Fd() to shellSetOrig{In,Out,Err}().
		 made rlogindOut() exit on EOF from master pty fd.
		 made rlogin return VxWorks prompt when client disconnects.
		 made rlogInit() not sweat being called more than once.
		 added shellLock() to rlogind & rlogin to get exclusive use
		   of shell.
01s,20oct87,gae  added logging device for rlogin shell; made rlogInit()
		   create pty device.
01r,03oct87,gae  moved logout to usrLib.c and made rlogindExit() from rlogindIn.
		 removed gratuitious standard I/O ioctl()'s.
		 made "disconnection" cleaner by having shell do the restart.
01q,04sep87,llk  added logout().  Made rlogindSock global so that logout() could
		   access it.
		 changed rlogind so that it doesn't exit when an accept fails.
01p,22apr87,dnw  changed rlogin to turn on xon-xoff to terminal.
		 fixed handling of termination sequence "~.".
		 changed default priority of rlogin tasks from 100 to 2.
		 made priority and task ids be global variables so they
		   can be accessed from outside this module.
01o,02apr87,ecs	 changed references to "struct sockaddr" to "SOCKADDR".
01n,27mar87,dnw  documentation.
01m,23mar87,jlf  documentation.
01l,27feb87,dnw  changed to spawn rlog tasks as UNBREAKABLE.
01k,20dec86,dnw  changed to use new call to remGetCurId().
		 changed old socket calls to normal i/o calls.
		 changed to not get include files from default directories.
		 added rlogInit ().
01j,27oct86,rdc  everything is now spawned in system mode with bigger stacks.
		 delinted.
01i,10oct86,gae  'remUser' made available through remGetCurId()...
		   included remLib.h.  Housekeeping.
01h,04sep86,jlf  documentation.
01g,31jul86,llk  uses new spawn
01f,27jul86,llk  added standard error fd which prints to the rlogin terminal.
		 Error messages go to standard error.
01e,07jul86,rdc  made rlogindInChild restart the shell after disconnect.
		 Documentation.
01d,19jun86,rdc  make rlogin close the socket after disconnect.
01c,18jun86,rdc  delinted.
01b,29apr86,rdc  rlogin now tries multiple local port numbers in its attempt
		 to connect to a remote host.
		 rlogind now uses KEEPALIVE option on sockets connected to 
		 clients.
		 made rlogind unbreakable.
01a,02apr86,rdc  written.
*/

/*
DESCRIPTION
The library rlogLib provides a remote login facility for VxWorks that 
uses the UNIX rlogin protocol as implemented in UNIX BSD 4.2 to
1) allow users at a VxWorks terminal to login to remote systems via the network,
and 2) allow users at remote systems to login to VxWorks via the network.

A VxWorks user may login to any other remote VxWorks or UNIX system via
the network, by calling the routine rlogin from the shell.

The rlogin daemon (rlogind) allows remote users to log into VxWorks.
The rlogin daemon is started by calling rlogInit,
typically by the root task, usrRoot(2), in usrConfig.c.
It accepts remote login requests from another VxWorks or UNIX system(s),
and causes the shell's input and output to be redirected to the remote user.

Internally, the rlogin daemon provides a tty-like interface to the remote
user through the use of the VxWorks pseudo terminal driver (ptyDrv (3)).

SEE ALSO:
ptyDrv (3), telnetLib (3),
and the UNIX documentation for rlogin, rlogind, and pty
*/

#include "vxWorks.h"
#include "socket.h"
#include "in.h"
#include "inetLib.h"
#include "ioLib.h"
#include "remLib.h"
#include "taskLib.h"

#define LOGIN_SERVICE		513

#define STDIN_BUF_SIZE		200
#define STDOUT_BUF_SIZE		200

#define MAX_CONNECT_TRIES	5

char *rlogShellName	= "shell";	/* task name we connect to */
char *rlogTermType	= "dumb/9600";	/* default terminal type */

/* rlogin task parameters */

int rlogTaskPriority	= 2;		/* task priority of rlogin tasks */
int rlogTaskOptions	= VX_SUPERVISOR_MODE | VX_UNBREAKABLE;
int rlogTaskStackSize	= 2000;		/* stack size of rlogin tasks */

int rlogindId;				/* rlogind task id */
int rlogInTaskId;			/* rlogInTask task id */
int rlogOutTaskId;			/* rlogOutTask task id */
int rlogChildTaskId;			/* rlogChildTask task id */

/* local variables */

LOCAL char *ptyRlogName  = "/pty/rlogin.";
LOCAL char *ptyRlogNameM = "/pty/rlogin.M";
LOCAL char *ptyRlogNameS = "/pty/rlogin.S";

LOCAL BOOL activeFlag = FALSE;	/* TRUE if there is an active connection */
LOCAL int rlogindSocket;	/* rlogind socket */
LOCAL int rlogindM;		/* rlogind master pty */
LOCAL int rlogindS;		/* rlogind slave pty */
LOCAL int rloginSocket;		/* rlogin socket */

LOCAL int shellInFd;		/* original console input */
LOCAL int shellOutFd;		/* original console output */
LOCAL int shellErrFd;		/* original console error output */


/* forward declarations */

VOID rlogind ();
VOID rlogInTask ();
VOID rlogOutTask ();
VOID rlogChildTask ();
LOCAL VOID rlogExit ();

/*******************************************************************************
*
* rlogInit - initialize the rlogin facility
*
* This routine initializes the rlogin facility, which supports remote login
* to the VxWorks shell via the UNIX rlogin protocol.
* It creates a pty device, and spawns the rlogin daemon.
* It is called normally from usrRoot(2),
* before any other system tries to rlogin to this VxWorks system via rlogin.
*
* RETURNS: OK if rlogin facility successfully initialized, otherwise ERROR.
*/

STATUS rlogInit ()

    {
    static BOOL done;	/* FALSE = not done */

    if (done)
	{
	printErr ("rlogInit: already initialized.\n");
	return (ERROR);
	}
    else
	done = TRUE;

    if (ptyDrv () == ERROR || ptyDevCreate (ptyRlogName, 1024, 1024) == ERROR)
	{
	printErr ("rlogInit: unable to create pty device.\n");
	return (ERROR);
	}

    rlogindId = taskSpawn ("rlogind", rlogTaskPriority,
			   rlogTaskOptions, rlogTaskStackSize,
			   rlogind, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);

    if (rlogindId == ERROR)
	{
	printErr ("rlogInit: unable to spawn rlogind.\n");
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
* 
* rlogind - the VxWorks rlogin daemon
*
* rlogind provides a facility for remote users to log into VxWorks over the
* network.  It should be spawned by rlogInit (2), which should be called by
* usrRoot (2) at boot time.
*
* Remote login requests will cause stdin, stdout, and stderr to be stolen
* away from the console.  When the remote user disconnects, stdin, stdout,
* and stderr are restored, and the shell is restarted.  rlogind uses the
* remote user verification protocol specified by the UNIX remote shell daemon
* documentation, but ignores all the information except the user name, which
* is used to set the VxWorks remote identity (iam (2)).
* 
* rlogind requires the existence of a pseudo terminal device
* which is created by rlogInit before rlogind is spawned.
* rlogind creates two child processes, rlogInTask and rlogOutTask,
* whenever a remote user is logged in.  These processes exit when
* the remote connection is terminated.
*/

VOID rlogind ()

    {
    char remUser [MAX_IDENTITY_LEN];
    char buf [STDIN_BUF_SIZE];
    int optval;
    struct sockaddr_in myAddress;
    struct sockaddr_in clientAddress;
    int clientAddrLen;
    int masterFd;
    int slaveFd;
    int client;
    int sd;

    /* open a socket and wait for a client */

    sd = socket (AF_INET, SOCK_STREAM, 0);
   
    bzero ((char *) &myAddress, sizeof (myAddress));
    myAddress.sin_family = AF_INET;
    myAddress.sin_port   = LOGIN_SERVICE;

    if (bind (sd, (SOCKADDR *) &myAddress, sizeof (myAddress)) == ERROR)
	{
	printErr ("rlogind: bind failed.\n");
	return;
	}

    listen (sd, 1);

    FOREVER
	{
	/* wait for shell to exist */

	while (taskNameToId (rlogShellName) == ERROR)
	    taskDelay (sysClkRateGet ());

	errnoSet (OK);		/* clear errno for pretty i() display */

	/* now accept connection */

	clientAddrLen = sizeof (clientAddress);
	client = accept (sd, (SOCKADDR *)&clientAddress, &clientAddrLen);

	if (client == ERROR)
	    {
	    printErr ("rlogind: accept failed - status = 0x%x", errnoGet());
	    continue;
	    }

	/* turn on KEEPALIVE so if the client crashes, we'll know about it */

	optval = 1;
	setsockopt (client, SOL_SOCKET, SO_KEEPALIVE,
		    (char *)&optval, sizeof (optval));


	/* read in initial strings from remote rlogin */

	if ((recvStr (client, buf) == ERROR) ||	    /* ignore stderr */
	    (recvStr (client, remUser) == ERROR) || /* get local user name */
	    (recvStr (client, buf) == ERROR) ||	    /* ignore remote user name*/
	    (recvStr (client, buf) == ERROR))	    /* ignore terminal stuff */
	    {
	    close (client);
	    continue;
	    }

	/* acknowlege connection */

	write (client, "", 1);

	/* check to see if there's already an active connection */

	if (activeFlag)
	    {
	    char *msg = "\r\nSorry, this system is engaged.\r\n";

	    write (client, msg, strlen (msg));
	    close (client);
	    continue;
	    }

	/* create the pseudo terminal:
	 * the master side is connected to the socket to the
	 * remote machine - two processes rlogInTask & rlogOutTask
	 * handle input and output.
	 */

	if ((masterFd = open (ptyRlogNameM, UPDATE)) == ERROR)
	    {
	    char *msg = "\r\nSorry, trouble with pty.\r\n";

	    printErr ("rlogind: error opening %s\n", ptyRlogNameM);
	    write (client, msg, strlen (msg));
	    close (client);
	    continue;
	    }

	if ((slaveFd = open (ptyRlogNameS, UPDATE)) == ERROR)
	    {
	    char *msg = "\r\nSorry, trouble with pty.\r\n";

	    printErr ("rlogind: error opening %s\n", ptyRlogNameS);
	    write (client, msg, strlen (msg));
	    close (client);
	    close (masterFd);
	    continue;
	    }

	if (!shellLock (TRUE))
	    {
	    char *msg = "\r\nSorry, shell is locked.\r\n";

	    printErr ("rlogind: someone tried to login.\n");
	    write (client, msg, strlen (msg));
	    close (client);
	    close (masterFd);
	    close (slaveFd);
	    continue;
	    }

	/* setup the slave device to act like a terminal */

	ioctl (slaveFd, FIOOPTIONS, OPT_TERMINAL);

	printf ("\nrlogind: This system *IN USE* via rlogin.\n");

	shellLogoutInstall (rlogExit, TRUE);
	activeFlag    = TRUE;
	rlogindSocket = client;
	rlogindM      = masterFd;
	rlogindS      = slaveFd;

	/* flush out pty device */

	ioctl (slaveFd, FIOFLUSH);

	/* save the shell's standard I/O fd's so we can restore them later */

	shellInFd  = ioGlobalStdGet (STD_IN);
	shellOutFd = ioGlobalStdGet (STD_OUT);
	shellErrFd = ioGlobalStdGet (STD_ERR);

	/* set shell's standard I/O to pty device; add extra logging device */

	shellOrigStdSet (STD_IN,  slaveFd);
	shellOrigStdSet (STD_OUT, slaveFd);
	shellOrigStdSet (STD_ERR, slaveFd);

	logFdAdd (slaveFd);

	/* the shell is currently stuck in a read from the console,
	 * so restart it */

	taskRestart (taskNameToId (rlogShellName));

	/* spawn the process which transfers data from the master pty 
	 * to the socket;
	 * spawn the process which transfers data from the client socket
	 * to the master pty.
	 */

	if (((rlogOutTaskId = taskSpawn("rlogOutTask", rlogTaskPriority,
					rlogTaskOptions, rlogTaskStackSize,
					rlogOutTask, client, masterFd,
					0, 0, 0, 0, 0, 0, 0, 0)) == ERROR) ||
	    ((rlogInTaskId = taskSpawn ("rlogInTask", rlogTaskPriority,
					rlogTaskOptions, rlogTaskStackSize,
					rlogInTask, client, masterFd,
					0, 0, 0, 0, 0, 0, 0, 0)) == ERROR))
	    {
	    printErr ("rlogind: error spawning %s child - status = 0x%x\n",
		      (rlogOutTaskId != ERROR) ? "output" : "input",
		      errnoGet ());

	    if (rlogOutTaskId != ERROR)
		taskDelete (rlogOutTaskId);

	    rlogExit (FALSE);		/* try to do tidy clean-up */
	    }
	}
    }
/*******************************************************************************
* 
* rlogOutTask - stdout to socket process
*
* This routine gets spawned by the rlogin daemon to move
* data between the client socket and the pseudo terminal.
* The task exits when the pty has been closed.
* 
* NOMANUAL - but not LOCAL for i(2)
*/

VOID rlogOutTask (sock, ptyMfd)
    FAST int sock;	/* socket to copy output to */
    int ptyMfd;		/* pty Master fd */

    {
    int n;
    char buf [STDOUT_BUF_SIZE];

    while ((n = read (ptyMfd, buf, sizeof (buf))) > 0)
	write (sock, buf, n);
    }
/*******************************************************************************
* 
* rlogInTask - socket to stdin process 
*
* This routine gets spawned by the rlogin daemon to move
* data between the pseudo terminal and the client socket.
* The task exits, calling rlogExit(), when the client disconnects.
*
* NOMANUAL - but not LOCAL for i(2)
*/

VOID rlogInTask (sock, ptyMfd)
    FAST int sock;	/* socket to copy input from */
    int ptyMfd;		/* pty Master fd */

    {
    FAST int n;
    char buf [STDIN_BUF_SIZE];

    while ((n = read (sock, buf, sizeof (buf))) > 0)
	write (ptyMfd, buf, n);

    /* exit and cleanup */

    rlogExit (FALSE);
    }
/*******************************************************************************
*
* rlogExit - exit and cleanup for rlogind
*
* This is the support routine for logout(2).
* The client socket is closed, shell standard I/O is redirected
* back to the console, and the shell is restarted by sending an EOF.
*/

LOCAL VOID rlogExit (usedLogout)
    BOOL usedLogout;	/* called by logout(2) */

    {
    activeFlag = FALSE;
    shellLogoutInstall ((FUNCPTR) NULL, 0);

    /* cancel extra logging device; restore shell's standard I/O & unlock */

    logFdDelete (rlogindS);

    shellOrigStdSet (STD_IN,  shellInFd);
    shellOrigStdSet (STD_OUT, shellOutFd);
    shellOrigStdSet (STD_ERR, shellErrFd);

    shellLock (FALSE);

    write (rlogindSocket, "\r\n", 2);
    close (rlogindSocket);

    /* send EOF to shell to cause restart */

    if (!usedLogout)
	write (rlogindM, "\04", 1);	/* control-D to shell */

    close (rlogindM);
    close (rlogindS);
    }
/*******************************************************************************
* 
* recvStr - receive a null terminated string from a socket
*
* Similar to fioRdString, but for sockets.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS recvStr (sd, buf)
    FAST int sd;	/* socket */
    FAST char *buf;	/* where to put the string */

    {
    do 
	{
	if (read (sd, buf, 1) != 1)
	    return (ERROR);
	}
    while (*(buf++) != EOS);

    return (OK);
    }
/*******************************************************************************
* 
* rlogin - login to a remote host
*
* The function rlogin may be called from the VxWorks shell as follows:
*
* -> rlogin "remoteSystem"
*
* where remoteSystem is either a host name which has been previously added
* to the remote host table by a call to hostAdd (2),
* or an inet address in "dot" notation (i.e. "90.0.0.2").
* The remote system will be logged into with the current user name as set
* by a call to iam(2).
* 
* The user may disconnect from the remote system by typing "<CR>~.<CR>"
* or by simply logging out from the remote system.
*
* RETURNS:
*     OK, or
*     ERROR if unknown host, no available privileged ports, unable to connect
*		to host, or trouble spawning child process.
*/

STATUS rlogin (host)
    char *host;		/* name of host to connect to */

    {
    char remUser [MAX_IDENTITY_LEN];
    char remPasswd [MAX_IDENTITY_LEN];
    struct sockaddr_in hostAddr;
    int port;
    char ch;
    int quitFlag;
    int status;
    int nTries;

    /* connect to host */

    if (((hostAddr.sin_addr.s_addr = hostGetByName (host)) == ERROR) &&
        ((hostAddr.sin_addr.s_addr = inet_addr (host)) == ERROR))
	{
	printErr ("rlogin: unknown host.\n");
	return (ERROR);
	}

    hostAddr.sin_family = AF_INET;
    hostAddr.sin_port   = LOGIN_SERVICE;

    remCurIdGet (remUser, remPasswd);

    port   = 1000;
    status = ERROR;

    for (nTries = 0; status == ERROR && nTries < MAX_CONNECT_TRIES; nTries++)
	{
	if ((rloginSocket = rresvport (&port)) == ERROR)
	    {
	    printErr ("rlogin: no available privileged ports.\n");
	    return (ERROR);
	    }

	status = connect(rloginSocket, (SOCKADDR *)&hostAddr, sizeof(hostAddr));
	port--;
	}

    if (status == ERROR)
	{
	printErr ("rlogin: could not connect to host.\n");
	close (rloginSocket);
	return (ERROR);
	}
    
    /* send a null (no seperate STDERR) */

    write (rloginSocket, "", 1);

    /* send the local and remote user names */

    write (rloginSocket, remUser, strlen (remUser) + 1);
    write (rloginSocket, remUser, strlen (remUser) + 1);

    /* send the terminal type */

    write (rloginSocket, rlogTermType, strlen (rlogTermType) + 1);

    /* spawn a process to handle stdin */

    rlogChildTaskId = taskSpawn ("rlogChildTask", rlogTaskPriority,
				 rlogTaskOptions, rlogTaskStackSize,
				 rlogChildTask, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);

    if (rlogChildTaskId == ERROR)
	{
	printErr ("rlogin: trouble spawning child - status = 0x%x\n",
		  errnoGet ());
	close (rloginSocket);
	return (ERROR);
	}

    /* force shell to be locked, shellLock is FALSE if already locked */

    shellLock (TRUE);

    /* set console to RAW mode except with XON-XOFF and 7 bit */

    ioctl (STD_IN, FIOOPTIONS, OPT_TANDEM | OPT_7_BIT);

    quitFlag = 0;

    while ((rloginSocket != ERROR) && (read (STD_IN, &ch, 1) == 1))
	{
	/* track input of "<CR>~.<CR>" to terminate connection */

	if ((quitFlag == 0) && (ch == '~'))
	    quitFlag = 1;
	else if ((quitFlag == 1) && (ch == '.'))
	    quitFlag = 2;
	else if ((quitFlag == 2) && (ch == '\r'))
	    break;		/* got "<CR>~.<CR>" */
	else
	    quitFlag = (ch == '\r') ? 0 : -1;

	write (rloginSocket, &ch, 1);
	}

    /* wait for other tasks to finish up, i.e. rlogind */

    taskDelay (sysClkRateGet () / 2);

    if (rloginSocket != ERROR)
	{
	taskDelete (rlogChildTaskId);
	close (rloginSocket);
	}

    /* reset console */

    ioctl (STD_IN, FIOOPTIONS, OPT_TERMINAL);
    printf ("\nClosed connection.\n");

    shellLock (FALSE);

    return (OK);
    }
/*******************************************************************************
* 
* rlogChildTask - rlogin child
*
* This routine gets spawned by rlogin to read data from
* the rloginSocket and write it to stdout.
* The task exits when the client disconnects;
* rlogin is informed by setting rloginSocket to ERROR.
*
* NOMANUAL	- but not LOCAL for i(2)
*/ 

VOID rlogChildTask ()

    {
    char buf [STDOUT_BUF_SIZE];
    int n;

    while ((n = read (rloginSocket, buf, sizeof (buf))) > 0)
	write (STD_OUT, buf, n);

    close (rloginSocket);

    /* indicate that client side caused termination,
     * and stop rlogin from reading stdin */

    rloginSocket = ERROR;

    ioctl (STD_IN, FIOCANCEL);
    }
