/* remLib.c - remote command library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/* 
modification history
--------------------
02c,03may89,dnw  moved iam() and whoami() here remLib.c from usrLib.c
		   to prevent usrLib from dragging in the network.
02b,30may88,dnw  changed to v4 names.
02a,28may88,dnw  removed host table routines to hostLib.
		 removed iam, whoami to usrLib.
		 changed call to fioStdErr to STD_ERR.
01w,04mar88,jcf  changed semaphores, and sem calls for new semLib.
01v,09jan88,gae  made hostTable a list -- no longer a fixed size.
01u,15dec87,gae  used inet_ntoa_b() instead of inet_ntoa(); misc. cleanups.
01t,14dec87,gae  documentation & checked malloc's for NULL.
01s,11nov87,jlf  documentation
01r,06nov87,dnw  changed remRcmd to take either host name or inet address.
01q,11jun87,llk  added remShowHosts() which shows all hosts in the host table
		   reachable from this VxWorks system.
		 changed remInetAddr() to call inet_addr.  remInetAddr() is
		   now considered obsolete.  Use inet_addr() instead.
		 implemented host name aliases in host table.
		 moved nethelp() from here to netLib.c.
		 added remFillHostName().
		 changed closeSocket() calls to close().
01p,02apr87,ecs  added include of strLib.h.
		 changed references to "struct sockaddr" to "SOCKADDR".
01o,24mar87,jlf  documentation.
01n,20dec86,dnw  changed iam() and remGetCurId to take password as 2nd param.
		 updated nethelp().
		 changed to not get include files from default directories.
01m,08nov86,dnw  change remRresvport to choose different port numbers
		   each time.  This is a TEMP (!?) fix for new net timing bug.
		 fixed remRcmd() not cleaning up correctly on some errors.
01l,29oct86,rdc	 deleted references to exLib.h and exos.h.
01k,28oct86,rdc	 lint.
01j,06oct86,gae	 got rid of rcd() and rpwd()...netDrv handles it.
		 Moved HOST and HOSTNAME here from remLib.h for better hiding.
01i,04sep86,jlf  documentation.
01h,30jul86,rdc  fixed remRcmd to comply with (undocumented) rshd protocol.
01g,17jul86,rdc  added seperate stderr stream to remRcmd.
		 remRcmd now checks for recv errors correctly.
01f,11jul86,rdc  wrote nethelp.
01e,21may86,llk	 moved remCat to netDrv.c, called it netGet.
		 changed memAllocate to malloc.
	    rdc  delinted.		
01d,27mar86,rdc  made remCat a little less sensitive to crashed sockets.
		 made remCat test for write errors properly.
01c,28jan86,jlf  documentation
01b,08oct85,rdc  de-linted
01a,03oct85,rdc  written
*/

/*
The library remLib supplies several routines found under BSD 4.2 (rcmd and 
rreservport), and several other routines useful for accessing files
via ethernet.

INCLUDE FILE
remLib.h

SEE ALSO
inetLib(1), "VxWorks Network"
*/

#include "vxWorks.h"
#include "socket.h"
#include "in.h"
#include "remLib.h"
#include "inetLib.h"
#include "strLib.h"

#define MAX_RETRY_CONNECT 5

LOCAL char remUser [MAX_IDENTITY_LEN];
LOCAL char remPasswd [MAX_IDENTITY_LEN];

int remLastResvPort = IPPORT_RESERVED - 1;

/*******************************************************************************
*
* rcmd - execute a shell command on a remote machine
*
* This routine executes a command on a remote machine, using the remote
* shell daemon (rshd) on the remote system.  It is analogous to the Unix
* rcmd (3) call.
* 
* RETURNS
*  socket descriptor if remote shell daemon accepts, or
*  ERROR if remote command failed.
*
* SEE ALSO: BSD 4.2 manual entry for rcmd
*/

int rcmd (host, remotePort, localUser, remoteUser, cmd, fd2p)
    char *host; 	/* host name or inet address */
    int remotePort;	/* remote port to connect to (rshd) */
    char *localUser;	/* local user name */
    char *remoteUser;	/* remote user name */
    char *cmd;		/* command */
    int *fd2p;		/* if this pointer is non-zero, stderr socket
			 * is opened and socket descriptor is filled in */

    {
    int sd;
    struct sockaddr_in sin;
    struct sockaddr_in mySin;
    int mySinLen;
    char c;
    int lport = IPPORT_RESERVED - 1;
    int nTries = 0;
    int stdErrPort;
    int stdErrSocket;
    char stringBuf [20];

    sin.sin_family = AF_INET;
    sin.sin_port = remotePort;
    if (((sin.sin_addr.s_addr = hostGetByName (host)) == ERROR) &&
        ((sin.sin_addr.s_addr = inet_addr (host)) == ERROR))
	{
	return (ERROR);
	}

    do
	{
	sd = rresvport (&lport);
	if (sd == ERROR)
	    return (ERROR);

	if (connect (sd, (SOCKADDR *) &sin, sizeof (sin)) == ERROR) 
	    {
	    close (sd);
	    lport--;
	    }
	else
	    break;
	}
    while (++nTries <= MAX_RETRY_CONNECT);

    if (nTries > MAX_RETRY_CONNECT)
	return (ERROR);

    if (fd2p == 0) 
        {
        send (sd, "", 1, 0);
        lport = 0;
        } 
    else 
        {
	stdErrPort = --lport;

	stdErrSocket = rresvport (&stdErrPort);
	if (stdErrSocket == ERROR)
	    {
	    close (sd);
	    return (ERROR);
	    }
	listen (stdErrSocket, 1);

	sprintf (stringBuf, "%d", stdErrPort);
	send (sd, stringBuf, strlen (stringBuf) + 1, 0);

	/* wait for rshd to connect */
	mySinLen = sizeof (mySin);
	*fd2p = accept (stdErrSocket, (SOCKADDR *)&mySin, &mySinLen);  
	if (*fd2p == ERROR)
	    {
	    close (sd);
	    close (stdErrSocket);
	    return (ERROR);
	    }
	close (stdErrSocket);
	}

    send (sd, localUser, strlen(localUser) + 1, 0);
    send (sd, remoteUser, strlen(remoteUser) + 1, 0);

    if (send (sd, cmd, strlen(cmd) + 1, 0) <= 0)
	{
	close (sd);
	if (fd2p != NULL)
	    close (*fd2p);
	return (ERROR);
	}

    /* bsd documentation for rshd is incorrect - null byte is actually 
       received on stdin socket */

    if (recv (sd, &c, 1, 0) <= 0)
	{
	close (sd);
	if (fd2p != NULL)
	    close (*fd2p);
	return (ERROR);
	}

    if (c != 0)
	{
	/* error will come in on stdin socket */
	while (recv (sd, &c, 1, 0) == 1) 
	    {
	    write (STD_ERR, &c, 1);
	    if (c == '\n')
		break;
	    }

	errnoSet (S_remLib_RSH_ERROR);
	close (sd);

	if (fd2p != NULL)
	    close (*fd2p);

	return (ERROR);
	}
    
    return (sd);
    }
/*******************************************************************************
*
* rresvport - open a socket with a privileged port bound to it
*
* Analogous to the Unix rresvport (3) command.
*
* RETURNS
*  Socket descriptor, or
*  ERROR if couldn't open socket, or
*  all ports in use.
*
* SEE ALSO: BSD 4.2 manual entry for rresvport
*/

int rresvport (alport)
    FAST int *alport;	/* port number to initially try */

    {
    struct sockaddr_in sin;
    int sd;

    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = 0;

    if ((sd = socket (AF_INET, SOCK_STREAM, 0)) == ERROR)
	return (ERROR);

    *alport = remLastResvPort;

    for (;;) 
        {
        (*alport)--;

        if (*alport == IPPORT_RESERVED - 400)	/* approx lower bound */
	    *alport = IPPORT_RESERVED - 1;

	if (*alport == remLastResvPort)
            {
	    errnoSet (S_remLib_ALL_PORTS_IN_USE);
            return (ERROR);
            }

        sin.sin_port = htons ((u_short)*alport);

        if (bind (sd, (SOCKADDR *) &sin, sizeof (sin)) != ERROR)
	    {
	    remLastResvPort = *alport;
            return (sd);
	    }
        }
    }
/*******************************************************************************
*
* remCurIdGet - get current user name and password
*
* This routine passes back the user name and password currently being used
* for remote host access priveleges.
* Either pointer may be NULL and the corresponding item will not be returned.
*
* SEE ALSO: iam(2), whoami(2)
*/

VOID remCurIdGet (user, passwd)
    char *user;		/* where to return current user name */
    char *passwd;	/* where to return current password */

    {
    if (user != NULL)
	strcpy (user, remUser);

    if (passwd != NULL)
	strcpy (passwd, remPasswd);
    }
/*******************************************************************************
*
* remCurIdSet - set remote user name and password
*
* The user name given here determines the access privileges available
* on the remote machine.  It should, of course, exist in /etc/passwd on
* that machine.
*
* Either pointer may be NULL in which case the corresponding item is set
* to the NULL string.
*
* The maximum length of the user name and the password is MAX_IDENTITY_LEN
* (defined in remLib.h).
*
* RETURNS
*  OK,   or
*  ERROR if name or password too long.
*
* SEE ALSO: iam(2), whoami(2)
*/

STATUS remCurIdSet (newUser, newPasswd)
    char *newUser;	/* user name to use on remote */
    char *newPasswd;	/* password to use on remote (NULL = none) */

    {
    if (((newUser != NULL) && (strlen (newUser) > MAX_IDENTITY_LEN-1)) ||
        ((newPasswd != NULL) && (strlen (newPasswd) > MAX_IDENTITY_LEN-1)))
	{
	errnoSet (S_remLib_IDENTITY_TOO_BIG);
	return (ERROR);
	}

    if (newUser == NULL)
	remUser[0] = EOS;
    else
	strcpy (remUser, newUser);

    if (newPasswd == NULL)
	remPasswd[0] = EOS;
    else
	strcpy (remPasswd, newPasswd);

    return (OK);
    }
/*******************************************************************************
*
* iam - set remote user name and password
*
* The user name given here determines the access privileges available
* on the remote machine.  It should, of course, exist in /etc/passwd on
* that machine.
*
* SEE ALSO: whoami(2), remCurIdGet(2), remCurIdSet(2)
*/

STATUS iam (newUser, newPasswd)
    char *newUser;	/* user name to use on remote */
    char *newPasswd;	/* password to use on remote (NULL = none) */

    {
    if (remCurIdSet (newUser, newPasswd) != OK)
	{
	printErr ("User name or password too long\n");
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* whoami - print current remote identity
*
* Prints the user name being used on a remote machine, which was set with
* iam (2), on standard output.
*
* SEE ALSO: iam(2), remCurIdGet(2), remCurIdSet(2)
*/

VOID whoami ()
    {
    char user [MAX_IDENTITY_LEN];

    remCurIdGet (user, (char *) NULL);
    printf ("%s\n", user);
    }
