/* pipeDrv.c - pipe I/O driver */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01f,21nov88,jcf  lint.
01e,01oct88,gae  restored FIONMSGS and FIOFLUSH lost in new version 01a.
01d,26aug88,gae  documentation.
01c,30may88,dnw  changed to v4 names.
01b,04may88,jcf  changed semaphores for new semLib.
01a,17mar88,dnw  re-written to use semaphores instead of VRTX blocks.
	  + jcf
... old versions deleted - see RCS
*/

/*
This is the pipe driver.  It allows I/O to be performed between tasks
similarly to the way it is performed to devices, via constructs
known as pipes.  Pipes are created and given names (with pipeDevCreate) and
can then be read and written with normal read and write calls.

The main difference between pipes and device I/O is that pipe I/O is done in
discrete units, known as messages.  All the data written as a single write
call is kept together, and will all be read by a single read call.  In
the same way, a read call will read only the number of bytes which were
written to the message being read, even though there may be more messages
in the pipe waiting to be read.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, ptyDrv () to
initialize the driver, and ptyDevCreate () to create devices.
 
PIPEDRV
Before using the driver, it must be initialized by calling the routine:
.CS
    pipeDrv ()
.CE
This routine must be called before any reads, writes, or
pipeDevCreates.  Normally, it is called from usrRoot (2) in usrConfig (1).
 
CREATING PIPES
Before a pipe can be used, it must be created.  This is done
with the pipeDevCreate call:
.CS
STATUS pipeDevCreate (name, nMessages, nBytes)
    char *name;      /* name of pipe to be created      *
    int nMessages;   /* max. number of messages in pipe *
    int nBytes;      /* size of each message            *
.CE
For instance, to create the device pipe "/pipe/demo",
with up to 10 messages of size 100 bytes, the proper call would be:
.CS
   pipeDevCreate ("/pipe/demo", 10, 100);
.CE
This device can now be opened, read, and written to;
creating and closing are not applicable.
 
IOCTL
The pipe driver ioctl call responds to the following ioctl codes.
.CS
    FIONREAD    - return the total number of bytes in the
                  pipe, in the variable pointed to by
                  argument.

    FIONMSGS    - return the number of discrete messages
                  available to be read in the pipe, in
                  the variable pointed to by argument.

    FIOFLUSH    - discard all messages in the pipe.
.CE

SEE ALSO: "I/O System"
*/

#include "vxWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "rngLib.h"
#include "semLib.h"

#define LOCK(lvl)	lvl = intLock ();
#define UNLOCK(lvl)	intUnlock (lvl);

#define MSG_OVERHEAD	(sizeof (short))


typedef struct			/* PIPE_DEV */
    {
    DEV_HDR devHdr;		/* pipe device header */
    RING_ID ringId;		/* ring buffer for this pipe */
    int maxBytes;		/* max bytes in pipe */
    int nBytes;			/* number of bytes in pipe */
    short moreBytes;		/* bytes remaining in partial message */
    int nMessages;		/* number of messages in pipe */
    SEM_ID readSemId;		/* read semaphore */
    BOOL readSemNeeded;		/* if TRUE then given on a write */
    } PIPE_DEV;


int pipeLostMessages;		/* to inspect # of dropped messages */

LOCAL int pipeDrvNum;		/* driver number of pipe driver */

/* forward declarations */

LOCAL int pipeOpen ();
LOCAL int pipeRead ();
LOCAL int pipeWrite ();
LOCAL STATUS pipeIoctl ();

/*******************************************************************************
*
* pipeDrv - pipe driver main routine
*
* This routine should be called before any pipes are created.
* It initializes and installs the driver.
*
* RETURNS: OK or ERROR if unable to install driver
*/

STATUS pipeDrv ()

    {
    /* check if driver already installed */

    if (pipeDrvNum == 0)
	{
	pipeDrvNum = iosDrvInstall ((FUNCPTR) NULL, (FUNCPTR) NULL, pipeOpen,
			     (FUNCPTR) NULL, pipeRead, pipeWrite, pipeIoctl);
	}

    return (pipeDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* pipeDevCreate - create a pipe device
*
* This routine is called once for each pipe to be created.
* It allocates the various structures necessary, and initializes the device.
* The pipe will have a maximum of `nMessages' messages in the pipe at once.
* When the the pipe is full, a task attempting to write to the pipe
* will be suspended until a message has been read.  If the pipe is written
* to at interrupt level, and is full, then that message is lost.
*
* RETURNS: OK or ERROR
*/

STATUS pipeDevCreate (name, nMessages, nBytes)
    char *name;		/* name of pipe to be created      */
    int nMessages;	/* max. number of messages in pipe */
    int nBytes;		/* size of each message            */

    {
    FAST PIPE_DEV *pPipeDev;
    int bufSize = nMessages * (nBytes + MSG_OVERHEAD);

    if (pipeDrvNum < 1)
        {
        errnoSet (S_ioLib_NO_DRIVER);
        return (ERROR);
        }

    if ((pPipeDev = (PIPE_DEV *) malloc (sizeof (PIPE_DEV))) == NULL)
	return (ERROR);

    if ((pPipeDev->ringId = rngCreate (bufSize)) == NULL)
	{
	free ((char *)pPipeDev);
	return (ERROR);
	}


    /* initialize pipe descriptor and add I/O device to system */

    pPipeDev->nBytes        = 0;
    pPipeDev->moreBytes     = 0;
    pPipeDev->maxBytes      = bufSize - MSG_OVERHEAD;
    pPipeDev->nMessages     = 0;
    pPipeDev->readSemId     = semCreate ();
    pPipeDev->readSemNeeded = TRUE;

    return (iosDevAdd ((DEV_HDR *) pPipeDev, name, pipeDrvNum));
    }

/* routines supplied to I/O system */

/*******************************************************************************
*
* pipeOpen - open a pipe file
*
* This routine is called to open a pipe.  It returns a pointer to the
* device.  This routine is normally reached only via the I/O system.
*
* ARGSUSED
*/

LOCAL int pipeOpen (pPipeDev, name, mode)
    PIPE_DEV *pPipeDev;		/* pipe descriptor */
    char *name;
    int mode;

    {
    return ((int) pPipeDev);
    }
/*******************************************************************************
*
* pipeRead - read bytes from a pipe
*
* This routine reads up to maxbytes bytes of the next message in the pipe.
* If the message is too long, the additional bytes are just discarded.
* 
* RETURNS:
*  number of bytes actually read;
*  will be between 1 and maxbytes, or zero if end of file, or ERROR
*/

LOCAL int pipeRead (pPipeDev, buffer, maxbytes)
    FAST PIPE_DEV *pPipeDev;	/* pointer to pipe descriptor */
    char *buffer;		/* buffer to receive bytes */
    unsigned int maxbytes;	/* max number of bytes to copy into buffer */

    {
    RING_ID ringId = pPipeDev->ringId;
    int nbytes;
    short messageSize;

    if (maxbytes == 0)
	return (0);

    /* wait for something to be in pipe */

    semTake (pPipeDev->readSemId);

    /* no more bytes left in last message? */

    if (pPipeDev->moreBytes == 0)
	{
	/* get length of next message (first two bytes) */

	rngBufGet (ringId, (char *) &messageSize, sizeof (short));
	}

    nbytes = min (messageSize, maxbytes);

    rngBufGet (ringId, buffer, nbytes);

    pPipeDev->moreBytes = messageSize - nbytes;	/* calculate remainder */
    pPipeDev->nBytes   -= nbytes;   /* decrement number of bytes in pipe */
    pPipeDev->nMessages--;	    /* decrement number of messages in pipe */


    /* if more stuff in buffer, enable next reader */

    if ((pPipeDev->moreBytes == 0) && rngIsEmpty (ringId))
	pPipeDev->readSemNeeded = TRUE;
    else
	semGive (pPipeDev->readSemId);

    return (nbytes);
    }
/*******************************************************************************
*
* pipeWrite - write bytes to a pipe
*
* This routine writes a message of `nbytes' to the pipe.
*
* RETURNS: number of bytes written or ERROR
*/

LOCAL int pipeWrite (pPipeDev, buffer, nbytes)
    FAST PIPE_DEV *pPipeDev;	/* pointer to pipe descriptor */
    char *buffer;		/* buffer from which to copy bytes */
    int nbytes;			/* number of bytes to copy from buffer */

    {
    int oldLevel;
    RING_ID ringId = pPipeDev->ringId;
    int bytesToWrite    = nbytes + MSG_OVERHEAD;
    short shortNbytes   = nbytes;

    /* check for valid number of bytes to write */

    if (nbytes == 0)
	return (0);

    if (bytesToWrite > pPipeDev->maxBytes)
	{
	logMsg ("pipeDrv: \"%s\" write too long; %d (len) > %d (max).\n",
	        pPipeDev->devHdr.name, nbytes,
		pPipeDev->maxBytes - MSG_OVERHEAD);
	errnoSet (S_ioLib_DEVICE_ERROR);
	return (ERROR);
	}


    /* wait for there to be room to put entire message; no partial messages */

    LOCK (oldLevel);

    while (bytesToWrite > rngFreeBytes (ringId))
	{
	if (intContext ())
	    {
	    ++pipeLostMessages;
	    UNLOCK (oldLevel);
	    return (ERROR);
	    }

	UNLOCK (oldLevel);
	taskDelay (1);
	LOCK (oldLevel);
	}


    /* put message length followed by message */

    rngBufPut (ringId, (char *) &shortNbytes, sizeof (short));
    rngBufPut (ringId, buffer, nbytes);

    pPipeDev->nBytes += nbytes;	/* increment number of bytes in pipe */
    pPipeDev->nMessages++;	/* increment number of messages in pipe */

    /* if pipe was empty, enable next reader */

    if (pPipeDev->readSemNeeded)
	{
	pPipeDev->readSemNeeded = FALSE;
	semGive (pPipeDev->readSemId);
	}

    UNLOCK (oldLevel);

    return (nbytes);
    }
/*******************************************************************************
*
* pipeIoctl - do device specific control function
*
* The ioctl requests recognized are FIONREAD, FIONMSGS, and FIOFLUSH.
*
* RETURNS:
*  OK and `arptr' gets number of bytes in pipe, or
*  ERROR if request is not FIONREAD, FIONMSGS, or FIOFLUSH.
*/

LOCAL STATUS pipeIoctl (pPipeDev, request, argptr)
    FAST PIPE_DEV *pPipeDev;	/* pointer to pipe descriptor */
    int request;		/* ioctl code */
    int	*argptr;		/* where to send answer */

    {
    STATUS status = OK;
    RING_ID ringId = pPipeDev->ringId;
    int oldLevel;

    switch (request)
	{
	case FIONREAD:
	    /* total number of bytes in pipe */

	    *argptr = pPipeDev->nBytes;
	    break;

	case FIONMSGS:
	    /* number of messages in pipe */

	    *argptr = pPipeDev->nMessages;
	    break;

	case FIOFLUSH:
	    /* discard all outstanding messages */

	    LOCK (oldLevel);

	    rngFlush (ringId);

	    pPipeDev->moreBytes     = 0;
	    pPipeDev->readSemNeeded = TRUE;
	    pPipeDev->nBytes        = 0;
	    pPipeDev->nMessages     = 0;
	    semClear (pPipeDev->readSemId);

	    UNLOCK (oldLevel);

	    break;

	default:
	    status = ERROR;
	    errnoSet (S_ioLib_UNKNOWN_REQUEST);
	    break;
	}

    return (status);
    }
