/* nfsLib.c - Network File System library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01m,23may89,dnw  lint.
01l,26mar89,llk  added nfsClientCacheCleanUp and nfsNameArrayFree to get rid
		   of memory eaters.
01k,09nov88,llk  rearranged error checking of nfs structures returned by server.
01j,28sep88,llk  added nfsSockBufSize, nfsTimeoutSec, nfsTimeoutUSec.
		 added nfsIdSet().
01i,25aug88,gae  documentation.
01h,07jul88,jcf  lint.
01g,30jun88,llk  changed to handle links.
		 added nfsHelp().
		 more name changes.
01f,16jun88,llk  changed to more v4 names.
		 added LOCAL and NOMANUAL where appropriate.
		 moved pathSplit() to pathLib.c.
01e,04jun88,llk  rewrote nfsLs and nfsDirRead so they clean up after themselves.
		 incorporated pathLib.
		 added nfsAuthUnix{Set Get Prompt Show} for unix authentication.
01d,30may88,dnw  changed to v4 names.
01c,28may88,dnw  copied promptParam...() stuff here from bootLib so that
		   this lib doesn't depend on bootLib.
01b,26apr88,llk  bug fix.  Internet address returned from remGetHostByName
		   was compared to <= NULL.  Some large addresses can be
		   mistaken for negative numbers.
01a,19apr88,llk  written.
*/

/*
DESCRIPTION
This library provides the client side of services for NFS (Network
File System) file oriented devices.  Most routines in this library
are NOT to be called directly by users, but rather by device drivers.
The driver is responsible for keeping track of file pointers,
mounted disks, cached buffers, and so on.  This library makes the
actual NFS calls using RPC (Remote Procedure Call).

NFS USER IDENTIFICATION
When making an NFS request to a host system, the NFS server expects
more in-depth information about the user than just the user's name.
NFS is built on top of Remote Procedure Call (RPC) and uses a type of RPC
authentication known as AUTH_UNIX.
VxWorks has an AUTH_UNIX structure which it passes on to the NFS server
with every NFS request.  This structure contains necessary information
for NFS, including the user's identification number, which is
uniquely mapped to the user's name, as well as a list of groups that the
user belongs to.  On UNIX systems, a user's identification number can be
found in the "/etc/passwd" file.
A list of groups that a user belongs to can be found in the "/etc/group" file.
The routine nfsAuthUnixPrompt allows the user to change the default parameters.
The routine nfsIdSet allows the user to just change the AUTH_UNIX id.
Usually, only the user id needs to be changed to indicate a new nfs user.

WARNING
Because of the recursive nature of the XDR (eXternal Data Representation)
routines in RPC, NFS tends to use a large stack.  Make sure that the tasks
running NFS have enough stack space (> 7000 bytes).

INCLUDE FILE: nfsLib.h

SEE ALSO
"Network", rpcLib (1), ioLib (1), nfsDrv (3)

INTERNAL
Every routine starts with a call to nfsInit.  nfsInit initializes RPC
and sets up the appropriate RPC task variables used for NFS.  nfsInit
may be called more than once with the only ill effect being wasted time.
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "ctype.h"
#include "strLib.h"
#include "ioLib.h"
#include "rpc.h"
#include "memLib.h"
#include "socket.h"
#include "in.h"
#include "auth.h"
#include "rpcGbl.h"
#include "xdr_nfs.h"
#include "nfsLib.h"

#define AUTH_UNIX_FIELD_LEN	50	/* UNIX authentication info */
#define MAX_GRPS		20	/* max. # of groups that user is in */


/* GLOBALS */

unsigned nfsMaxMsgLen = NFS_MAXDATA;	/* largest read or write buffer that
					 * can be transfered to/from the nfs
					 * server
					 */
u_int nfsTimeoutSec   = NFS_TIMEOUT_SEC;
u_int nfsTimeoutUSec  = NFS_TIMEOUT_USEC;
int   nfsSockBufSize  = NFS_SOCKOPTVAL;

/* LOCALS */

/*
Static RPC information.
This structure is kept around to be reused during subsequent NFS calls.
*/

struct moduleStatics
    {
    CLIENT *client;	/* pointer to client structure */
    int socket;		/* socket associated with this client */
    int oldprognum;	/* last program number that was used with this
			     client */
    int oldversnum; 	/* last version number that was used with this
			     client */
    int valid;		/* if TRUE, then this client information is valid */
    char *oldhost;	/* last host name that was used with this client */
    };

typedef struct		/* RPC_AUTH_UNIX_TYPE - UNIX authentication structure */
    {
    char machname [AUTH_UNIX_FIELD_LEN];  /* host name where client is */
    int uid;			/* client's UNIX effective uid */
    int gid;			/* client's current group id */
    int len;			/* element length of aup_gids */
    int aup_gids [MAX_GRPS];	/* array of groups user is in */
    } RPC_AUTH_UNIX_TYPE;

RPC_AUTH_UNIX_TYPE rpcAuthUnix;

/*******************************************************************************
*
* nfsDirMount - mount an NFS directory
*
* RETURNS:  OK | ERROR
*
* NOMANUAL
*/ 

STATUS nfsDirMount (hostName, dirname, pFileHandle)
    char *hostName;
    dirpath dirname;
    nfs_fh *pFileHandle;	/* the directory's file handle */

    {
    fhstatus fileHandleStatus;

    bzero ((char *) &fileHandleStatus, sizeof (fileHandleStatus));

    if (nfsClientCall (hostName, MOUNTPROG, MOUNTVERS, MNTPROC_MNT,
				xdr_dirpath, (char *) &dirname,
				xdr_fhstatus, (char *) &fileHandleStatus)
	== ERROR)
	{
	return (ERROR);
	}

    if (fileHandleStatus.status != 0)
	{
	netErrnoSet ((int) fileHandleStatus.status);	/* UNIX error */
	return (ERROR);
	}

    bcopy ((char *) &fileHandleStatus.fhstatus_u.directory,
	   (char *) pFileHandle, sizeof (nfs_fh));
    clntudp_freeres (taskModuleList->nfsClientCache->client, xdr_fhstatus,
		     (caddr_t) &fileHandleStatus);
    return (OK);
    }
/*******************************************************************************
*
* nfsDirUnmount - unmount an NFS directory
*
* RETURNS:  OK | ERROR
*
* NOMANUAL
*/ 

STATUS nfsDirUnmount (hostName, dirname)
    char *hostName;
    dirpath dirname;

    {
    return (nfsClientCall (hostName, MOUNTPROG, MOUNTVERS, MNTPROC_UMNT,
				xdr_dirpath, (char *) &dirname,
				xdr_void, (char *) NULL));
    }
/*******************************************************************************
*
* nfsAllUnmount - unmounts all directories on a particular host
*
* This routine umounts all filesystems/directories mounted from the
* host `hostName'.
*
* RETURNS: OK or ERROR
*/ 

STATUS nfsAllUnmount (hostName)
    char *hostName;		/* host machine to unmount from */

    {
    return (nfsClientCall (hostName, MOUNTPROG, MOUNTVERS, MNTPROC_UMNTALL,
				xdr_void, (char *) NULL,
				xdr_void, (char *) NULL));
    }
/*******************************************************************************
*
* nfsMntDump - prints a list of all NFS file systems mounted on the remote host
*
* This routine prints all the NFS file systems mounted on the remote host
*
* RETURNS: OK or ERROR
*/ 

STATUS nfsMntDump (hostName)
    char *hostName;		/* host machine */

    {
    mountentries mEntries;

    bzero ((char *) &mEntries, sizeof (mEntries));

    if (nfsClientCall (hostName, MOUNTPROG, MOUNTVERS, MNTPROC_DUMP,
				xdr_void, (char *) NULL,
				xdr_mountentries, (char *) &mEntries) == ERROR)
	{
	return (ERROR);
	}

    if (mEntries.more_entries)
	nfsMountListPrint (hostName, &mEntries.mountentries_u.mntlist);

    clntudp_freeres (taskModuleList->nfsClientCache->client, xdr_mountentries,
		     (caddr_t) &mEntries);
    return (OK);
    }
/*******************************************************************************
*
* nfsExportShow - prints a list of a remote host's exported file systems
*
* A list of file systems and the groups that are allowed to mount them
* is printed.
*
* EXAMPLE
* .CS
*  -> nfsExportShow "wrs"
*  export list for wrs:
*  /d0               staff
*  /d1               staff eng
*  /d2               eng
*  /d3               everyone
*  value = 0 = 0x0
*  -> 
* .CE
*
* RETURNS: OK or ERROR
*/ 

STATUS nfsExportShow (hostName)
    char *hostName;	/* host machine for which to show exports */

    {
    exportbody nfsExports;

    bzero ((char *) &nfsExports, sizeof (nfsExports));

    if (nfsClientCall (hostName, MOUNTPROG, MOUNTVERS, MNTPROC_EXPORT,
			xdr_void, (char *) NULL,
			xdr_exportbody, (char *) &nfsExports) == ERROR)
	{
	return (ERROR);
	}

    if (nfsExports.more_entries)
	nfsExpListPrint (hostName, &nfsExports.exportbody_u.export);

    clntudp_freeres (taskModuleList->nfsClientCache->client, xdr_exportbody,
		     (caddr_t) &nfsExports);
    return (OK);
    }
/*******************************************************************************
*
* nfsLookUpByName - looks up a remote file
*
* This routine is useful for gathering information on a file,
* particularly the file's NFS file handle and directory information.
*
* If the file name (directory path or filename itself) contains a symbolic
* link, this routine changes the file name to incorporate the name of the link.
*
* RETURNS:  OK | ERROR | FOLLOW_LINK,
*	    if FOLLOW_LINK, then fileName contains the name of the link
*
* NOMANUAL
*/

int nfsLookUpByName (hostName, fileName, pMountHandle, pDirOpRes, pDirHandle)
    char	*hostName;
    char	*fileName;	/* name is changed if symbolic link */
    nfs_fh	*pMountHandle;
				/* these args are returned to calling routine */
    diropres	*pDirOpRes;	/*    pointer to directory operation results */
    nfs_fh	*pDirHandle;	/*    pointer to file's directory file handle */

    {
    diropargs file;
    char *nameArray [NFS_MAXDIRNAMES];
    char linkName [NFS_MAXPATHLEN];	/* name of link file */
    char currentDir [NFS_MAXPATHLEN];	/* current directory for the look up's
					 *    that have been done */
    char newName [NFS_MAXPATHLEN];	/* new file name if file is symb link */
    int nameCount;

    if (nfsInit () == ERROR)
	return (ERROR);

    bzero ((char *) &file, sizeof (file));

    /* parse file name, enter each directory name in the path into the array */

    pathParse (fileName, nameArray);

    bcopy ((char *) pMountHandle, (char *) &file.dir, sizeof (nfs_fh));
    bcopy ((char *) pMountHandle, (char *) pDirHandle, sizeof (nfs_fh));

    /* start traversing file name */

    nameCount = 0;
    while ((nameCount < NFS_MAXDIRNAMES) && (nameArray [nameCount] != NULL))
	{
	file.name = nameArray [nameCount];

	if (nfsClientCall (hostName, NFSPROG, NFSVERS, NFSPROC_LOOKUP,
				    xdr_diropargs, (char *) &file,
				    xdr_diropres, (char *) pDirOpRes) == ERROR)

	    {
	    return (ERROR);
	    }

	if (pDirOpRes->status != NFS_OK)
	    {
	    nfsErrnoSet (pDirOpRes->status);
	    return (ERROR);
	    }

	if (pDirOpRes->diropres_u.finfo.attributes.type == NFDIR)
	    {
	    /* save directory information */

	    bcopy ((char *) &pDirOpRes->diropres_u.finfo.file,
		   (char *) pDirHandle, sizeof (nfs_fh));
	    bcopy ((char *) &pDirOpRes->diropres_u.finfo.file,
		   (char *) &file.dir, sizeof (nfs_fh));
	    }
	else if (pDirOpRes->diropres_u.finfo.attributes.type == NFLNK)
	    {
	    /* symbolic link, get real path name */

	    if (nfsLinkGet (hostName, &pDirOpRes->diropres_u.finfo.file,
			    linkName) == ERROR)
		{
		clntudp_freeres (taskModuleList->nfsClientCache->client,
				 xdr_diropres, (caddr_t) pDirOpRes);
		return (ERROR);
		}

	    /* Change fileName to include name of link.
	     * Concatenate the directory name, link name,
	     * and rest of the path name following the link.
	     */

	    pathBuild (nameArray, nameCount - 1, currentDir);
	    pathCat (currentDir, linkName, newName);
	    pathAppend (newName, nameArray, ++nameCount);
	    strcpy (fileName, newName);

	    (void) nfsNameArrayFree (nameArray);
	    clntudp_freeres (taskModuleList->nfsClientCache->client,
			     xdr_diropres, (caddr_t) pDirOpRes);
	    return (FOLLOW_LINK);
	    }
	nameCount++;	/* look up next directory */
	}
    nfsNameArrayFree (nameArray);
    clntudp_freeres (taskModuleList->nfsClientCache->client, xdr_diropres,
		     (caddr_t) pDirOpRes);
    return (OK);
    }
/*******************************************************************************
*
* nfsFileRemove - remove a file
*
* NOMANUAL
*/ 

STATUS nfsFileRemove (hostName, pMountHandle, fullFileName)
    char *hostName;
    nfs_fh *pMountHandle;	/* file handle of mount device */
    char *fullFileName;

    {
    u_int nfsProc;
    int lookUpStat;
    diropres dirResults;
    diropargs where;
    nfsstat status;
    char fileName [NFS_MAXNAMLEN];
    char dirName [NFS_MAXPATHLEN];
    nfs_fh dirHandle;

    if (nfsInit () == ERROR)
	return (ERROR);

    bzero ((char *) &dirResults, sizeof (dirResults));
    bzero ((char *) &status, sizeof (status));

    pathSplit (fullFileName, dirName, fileName);

    lookUpStat = nfsLookUpByName (hostName, fullFileName, pMountHandle,
				      &dirResults, &dirHandle);
    if (lookUpStat != OK)
	return (ERROR);

    bcopy ((char *) &dirHandle, (char *) &where.dir, sizeof (nfs_fh));

    where.name = fileName;

    switch (dirResults.diropres_u.finfo.attributes.type)
	{
	case NFREG:			/* remove regular file */
	    nfsProc = NFSPROC_REMOVE;
	    break;

	case NFDIR:			/* remove directory */
	    nfsProc = NFSPROC_RMDIR;
	    break;

	default:
	    errnoSet (S_nfsLib_NFS_INAPPLICABLE_FILE_TYPE);
	    return (ERROR);
	}

    if (nfsClientCall (hostName, NFSPROG, NFSVERS, nfsProc,
			    xdr_diropargs, (char *) &where,
			    xdr_nfsstat, (char *) &status) == ERROR)
	{
	return (ERROR);
	}

    if (status != NFS_OK)
	{
	nfsErrnoSet (status);
	return (ERROR);
	}

    clntudp_freeres (taskModuleList->nfsClientCache->client, xdr_nfsstat,
		     (caddr_t) &status);
    return (OK);
    }
/*******************************************************************************
*
* nfsRename - rename a file
*
* NOMANUAL
*/ 

STATUS nfsRename (hostName, pMountHandle, oldName, pOldDirHandle, newName)
    char *hostName;
    nfs_fh *pMountHandle;
    char *oldName;
    nfs_fh *pOldDirHandle;
    char *newName;

    {
    int lookUpStat;
    diropres dirResults;
    renameargs renameArgs;
    nfsstat status;
    char newFileName [NFS_MAXNAMLEN];
    char oldFileName [NFS_MAXNAMLEN];
    char newDirName [NFS_MAXPATHLEN];
    char oldDirName [NFS_MAXPATHLEN];
    nfs_fh dirHandle;

    if (nfsInit () == ERROR)
	return (ERROR);

    bzero ((char *) &dirResults, sizeof (dirResults));
    bzero ((char *) &status, sizeof (status));

    pathSplit (oldName, oldDirName, oldFileName);
    pathSplit (newName, newDirName, newFileName);

    bcopy ((char *) pOldDirHandle, (char *) &renameArgs.from.dir,
	   sizeof (nfs_fh));
    renameArgs.from.name = oldFileName;

    if (newDirName [0] == EOS)
	/* NULL directory name */

	bcopy ((char *) pMountHandle, (char *) &renameArgs.to.dir,
	       sizeof (nfs_fh));
    else
	{
	lookUpStat = nfsLookUpByName (hostName, newDirName, pMountHandle,
				      &dirResults, &dirHandle);
	if (lookUpStat != OK)
	    {
	    printErr ("nfsRename: rename of \"%s\" failed\n", newDirName);
	    return (ERROR);
	    }
	bcopy ((char *) &dirResults.diropres_u.finfo.file,
	       (char *) &renameArgs.to.dir, sizeof (nfs_fh));
	}

    renameArgs.to.name = newFileName;

    if (nfsClientCall (hostName, NFSPROG, NFSVERS, NFSPROC_RENAME,
			    xdr_renameargs, (char *) &renameArgs,
			    xdr_nfsstat, (char *) &status) == ERROR)
	{
	return (ERROR);
	}

    if (status != NFS_OK)
	{
	nfsErrnoSet (status);
	return (ERROR);
	}

    clntudp_freeres (taskModuleList->nfsClientCache->client, xdr_nfsstat,
		     (caddr_t) &status);
    return (OK);
    }
/*******************************************************************************
*
* nfsThingCreate - creates file or directory
*
* RETURNS:  OK | ERROR | FOLLOW_LINK,
*	    if FOLLOW_LINK, then fullFileName contains the name of the link
*
* NOMANUAL
*/ 

int nfsThingCreate (hostName, fullFileName, pMountHandle, pDirOpRes,
		       pDirHandle, mode)
    char	*hostName;
    char	*fullFileName;	/* name of thing being created */
    nfs_fh	*pMountHandle;	/* mount device's file handle */
    diropres	*pDirOpRes;	/* ptr to returned direc. operation results */
    nfs_fh	*pDirHandle;	/* ptr to returned file's directory file hndl */
    u_int	mode;		/* mode that file or dir is created with
				     UNIX chmod style */

    {
    int status;
    char newName [NFS_MAXPATHLEN];	/* new name if name contained a link */
    char dirName [NFS_MAXPATHLEN];	/* dir where file will be created */
    char fileName [NFS_MAXNAMLEN];	/* name of file (without path) */
    FAST sattr *pSattr;			/* ptr to attributes */
    diropres lookUpResult;
    createargs createArgs;
    nfs_fh dummyDirHandle;
    u_int nfsProc;			/* which nfs procedure to call,
					   depending on whether a file or
					   directory is made */

    if (nfsInit () == ERROR)
	return (ERROR);

    bzero ((char *) &createArgs, sizeof (createArgs));
    bzero ((char *) &lookUpResult, sizeof (lookUpResult));
    bzero ((char *) pDirOpRes, sizeof (diropres));

    pathSplit (fullFileName, dirName, fileName);

    status = nfsLookUpByName (hostName, dirName, pMountHandle, &lookUpResult,
			 &dummyDirHandle);
    
    if (status == FOLLOW_LINK)
	{
	pathCat (dirName, fileName, newName);
	strcpy (fullFileName, newName);
	}

    if (status != OK)
	return (status);

    bcopy ((char *) &lookUpResult.diropres_u.finfo.file,
	   (char *) &createArgs.dirargs.dir, sizeof (nfs_fh));

    bcopy ((char *) &lookUpResult.diropres_u.finfo.file,
	   (char *) pDirHandle, sizeof (nfs_fh));

    createArgs.dirargs.name = fileName;

    pSattr = &createArgs.attributes;

    if (mode == 0)
    	/* create a file with default permissions */
	pSattr->mode = NFS_FSTAT_REG | DEFAULT_FILE_PERM;
    else
	pSattr->mode = mode;

    /* don't set these attributes */

    pSattr->uid = pSattr->gid = pSattr->size = -1;	/* user id */
    pSattr->atime.tv_sec = pSattr->atime.tv_usec = -1;
    pSattr->mtime.tv_sec = pSattr->mtime.tv_usec = -1;

    if (mode & NFS_FSTAT_DIR)
	nfsProc = NFSPROC_MKDIR;
    else
        nfsProc = NFSPROC_CREATE;

    if (nfsClientCall (hostName, NFSPROG, NFSVERS, nfsProc,
			    xdr_createargs, (char *) &createArgs,
			    xdr_diropres, (char *) pDirOpRes) == ERROR)
	{
	return (ERROR);
	}

    if (pDirOpRes->status != NFS_OK)
	{
	nfsErrnoSet (pDirOpRes->status);
	return (ERROR);
	}

    clntudp_freeres (taskModuleList->nfsClientCache->client, xdr_diropres,
		     (caddr_t) pDirOpRes);
    return (OK);
    }
/*******************************************************************************
*
* nfsFileWrite - write to a file
*
* Either all characters will be written, or there is an ERROR.
*
* RETURNS:  number of characters written | ERROR
*
* NOMANUAL
*/ 

int nfsFileWrite (hostName, pHandle, offset, count, data, pNewAttr)
    char	*hostName;
    nfs_fh	*pHandle;	/* file handle of file being written to */
    unsigned	offset;		/* current byte offset in file */
    unsigned	count;		/* number of bytes to write */
    char	*data;		/* data to be written (up to NFS_MAXDATA) */
    fattr	*pNewAttr;

    {
    FAST unsigned nWrite = 0;	      /* bytes written in one nfs write */
    FAST unsigned nTotalWrite = 0;    /* bytes written in all writes together */
    writeargs writeArgs;	      /* arguments to pass to nfs server */
    attrstat writeResult;	      /* info returned from nfs server */

    if (nfsInit () == ERROR)
	return (ERROR);

    bzero ((char *) &writeArgs, sizeof (writeArgs));
    bzero ((char *) &writeResult, sizeof (attrstat));

    bcopy ((char *) pHandle, (char *) &writeArgs.file, sizeof (nfs_fh));

    /* set offset into file where write starts */

    writeArgs.offset = offset;

    while (nTotalWrite < count)
	{
	/* can only write up to NFS_MAXDATA bytes in one nfs write */

	if (count - nTotalWrite > nfsMaxMsgLen)
	    nWrite = nfsMaxMsgLen;
	else
	    nWrite = count - nTotalWrite;

	writeArgs.totalcount = nWrite;
	writeArgs.data.data_len = nWrite;
	writeArgs.data.data_val = data + nTotalWrite;
	writeArgs.offset = offset + nTotalWrite;

	if (nfsClientCall (hostName, NFSPROG, NFSVERS, NFSPROC_WRITE,
				xdr_writeargs, (char *) &writeArgs,
				xdr_attrstat, (char *) &writeResult) == ERROR)
	    {
	    return (ERROR);

	    /* XXX if some writes have been done and there was an error,
	     * nTotalWrite should be returned */
	    }

	if (writeResult.status != NFS_OK)
	    {
	    nfsErrnoSet (writeResult.status);
	    return (ERROR);
	    }
     
	nTotalWrite += nWrite;
	}

    bcopy ((char *) &writeResult.attrstat_u.attributes, (char *) pNewAttr,
	   sizeof (fattr));

    /* if nfs write returned ok, then all bytes were written */

    clntudp_freeres (taskModuleList->nfsClientCache->client, xdr_attrstat,
		     (caddr_t) &writeResult);
    return (count);
    }
/*******************************************************************************
*
* nfsFileRead - read from a file
*
* Reads up to the number of bytes in the largest nfs message (NFS_MAXDATA).
*
* RETURNS: number of characters read or ERROR
*
* NOMANUAL
*/ 

int nfsFileRead (hostName, pHandle, offset, count, buf, pNewAttr)
    char	*hostName;
    nfs_fh	*pHandle;	/* file handle of file being read */
    unsigned	offset;		/* start at offset bytes from beg. of file */
    unsigned	count;		/* number bytes to read up to */
    char	*buf;		/* buffer that data is read into */
    fattr	*pNewAttr;	/* arguments returned from server */

    {
    FAST unsigned nRead = 0;		/* number of bytes read in one read */
    readargs readArgs;			/* arguments to pass to nfs server */
    readreply readReply;

    if (nfsInit () == ERROR)
	return (ERROR);

    bzero ((char *) &readArgs, sizeof (readArgs));
    bzero ((char *) &readReply, sizeof (readReply));

    bcopy ((char *) pHandle, (char *) &readArgs.file, sizeof (nfs_fh));

    /* can only read up to NFS_MAXDATA bytes in one nfs read */

    if (count >= nfsMaxMsgLen)
	readArgs.count = nfsMaxMsgLen;
    else
	readArgs.count = count;

    readArgs.offset = offset;

    if (nfsClientCall (hostName, NFSPROG, NFSVERS, NFSPROC_READ,
			    xdr_readargs, (char *) &readArgs,
			    xdr_readreply, (char *) &readReply) == ERROR)
	{
	return (ERROR);
	}

    if (readReply.status != NFS_OK)
	{
	nfsErrnoSet (readReply.status);
	return (ERROR);
	}

    /* copy the bytes read and file's attributes into user's buffer */

    nRead = readReply.readreply_u.rdresult.data.data_len;
    bcopy (readReply.readreply_u.rdresult.data.data_val, buf, (int) nRead);
    bcopy ((char *) &(readReply.readreply_u.rdresult.attributes),
	   (char *) pNewAttr, sizeof (fattr));

    clntudp_freeres (taskModuleList->nfsClientCache->client, xdr_readreply,
		     (caddr_t) &readReply);
    return (nRead);
    }
/*******************************************************************************
*
* nfsLinkGet - gets name of real file from a symbolic link
*
* RETURNS:  OK | ERROR
*/ 

LOCAL STATUS nfsLinkGet (hostName, pHandle, realPath)
    char	*hostName;
    nfs_fh	*pHandle;	/* file handle of file being read */
    path	realPath;	/* the actual pathname gets returned here */

    {
    pathstat pathStatus;

    if (nfsInit () == ERROR)
	return (ERROR);

    bzero ((char *) &pathStatus, sizeof (pathstat));

    if (nfsClientCall (hostName, NFSPROG, NFSVERS, NFSPROC_READLINK,
			    xdr_nfs_fh, (char *) pHandle,
			    xdr_pathstat, (char *) &pathStatus) == ERROR)
	{
	return (ERROR);
	}

    if (pathStatus.status != NFS_OK)
	{
	nfsErrnoSet (pathStatus.status);
	return (ERROR);
	}

    strcpy (realPath, pathStatus.pathstat_u.name);

    clntudp_freeres (taskModuleList->nfsClientCache->client, xdr_pathstat,
		     (caddr_t) &pathStatus);
    return (OK);
    }
/*******************************************************************************
*
* nfsLs - list a directory
*
* RETURNS:  OK | ERROR
*
* NOMANUAL
*/ 

STATUS nfsLs (hostName, pFileHandle)
    char *hostName;		/* name of host machine */
    nfs_fh *pFileHandle;	/* ptr to file handle of directory being ls'd */

    {
    readdirres readDirRes;	/* results from nfsDirRead */
    FAST dirlist *pDirList;
    nfscookie cookie;
    entry *pEntry;

    if (nfsInit () == ERROR)
	return (ERROR);

    bzero (cookie, sizeof (cookie));

    /* Keep reading entries from the server's directory.
     * A "cookie" is used to keep track of the last entry that was retrieved
     * from the directory.
     */

    FOREVER
	{
	/* read directory entries starting with the current cookie.
	 * read as many entries from the directory as will fit in
	 * the biggest nfs message
	 */ 

	if (nfsDirRead (hostName, pFileHandle, cookie, nfsMaxMsgLen,
			&readDirRes)
	    == ERROR)
	    return (ERROR);

	/* print the directory entries from the last retrieval */

	pDirList = &readDirRes.readdirres_u.reply;
	nfsDirListPrint (pDirList);

	/* if the whole directory was read, we're done */

	if (pDirList->eof == TRUE)
	    {
	    if (nfsDirResFree (&readDirRes) == FALSE)
		return (ERROR);
	    break;
	    }

	/* search directory list for last cookie handed back from server
	 * (the cookie server)
	 */

	if ((pEntry = pDirList->entries) == NULL)
	    return (ERROR);

	while (pEntry->nextentry != NULL)
	    pEntry = pEntry->nextentry;

	/* copy the cookie from the last directory entry */

	bcopy (pEntry->cookie, cookie, sizeof (cookie));

	/* free up the readdirres structure used in nfsDirRead */

	if (nfsDirResFree (&readDirRes) == FALSE)
	    return (ERROR);
	}
    return (OK);
    }
/*******************************************************************************
*
* nfsDirRead - read entries from a directory
*
* RETURNS:  OK | ERROR
*/ 

LOCAL STATUS nfsDirRead (hostName, pDirHandle, cookie, count, pReadDirRes)
    char *hostName;
    nfs_fh *pDirHandle;
    nfscookie cookie;		/* hand this cookie to the server.  The cookie 
			           marks a place in the directory where entries
				   are to be read from */
    unsigned count;		/* number of bytes that nfs may return */
    readdirres *pReadDirRes;	/* return the results of directory read here */

    {
    readdirargs readDirArgs;

    if (nfsInit () == ERROR)
	return (ERROR);

    bzero ((char *) &readDirArgs, sizeof (readdirargs));
    bzero ((char *) pReadDirRes, sizeof (readdirres));

    bcopy ((char *) pDirHandle, (char *) &readDirArgs.dir, sizeof (nfs_fh));
    bcopy (cookie, readDirArgs.cookie, sizeof (nfscookie));
    readDirArgs.count = count;

    if (nfsClientCall (hostName, NFSPROG, NFSVERS, NFSPROC_READDIR,
			    xdr_readdirargs, (char *) &readDirArgs,
			    xdr_readdirres, (char *) pReadDirRes) == ERROR)
	{
	return (ERROR);
	}

    if (pReadDirRes->status != NFS_OK)
	{
	nfsErrnoSet (pReadDirRes->status);
	clntudp_freeres (taskModuleList->nfsClientCache->client,
			 xdr_readdirres, (caddr_t) pReadDirRes);
	return (ERROR);
	}

    /* normally, the readdirres structure would be freed up here, but
       the calling routine needs its information, so the calling routine
       must call clntudp_freeres after calling nfsDirRead */

    return (OK);
    }
/*******************************************************************************
*
* nfsDirResFree - free the rcp readdirres result structure
*
* RETURNS: OK | ERROR
*/ 

LOCAL STATUS nfsDirResFree (pReadDirRes)
    readdirres *pReadDirRes;

    {
    return (clntudp_freeres (taskModuleList->nfsClientCache->client,
			     xdr_readdirres, (caddr_t) pReadDirRes));
    }
/*******************************************************************************
*
* nfsNameArrayFree - free a name array structure
*
* RETURNS: OK | ERROR
*/ 

LOCAL STATUS nfsNameArrayFree (nameArray)
    char **nameArray;

    {
    int nameCount = 0;

    while ((nameCount < NFS_MAXDIRNAMES) && (nameArray [nameCount] != NULL))
	{
	if (free (nameArray [nameCount]) == ERROR)
	    return (ERROR);
	nameCount++;
	}
    return (OK);
    }
/*******************************************************************************
*
* nfsInit - initialize NFS
*
* nfsInit should be called by a task before it uses NFS.
* It is okay to call this routine more than once.
* nfsInit calls nfsClientCacheInit ().
*
* RETURNS: OK | ERROR
*/ 

LOCAL STATUS nfsInit ()

    {
    STATUS status = OK;

    if (rpcTaskInit () == ERROR)
	return (ERROR);
    if (taskModuleList->nfsClientCache == NULL)
	status = nfsClientCacheInit (taskModuleList);
    return (status);
    }
/*******************************************************************************
*
* nfsClientCacheInit - initialize RPC task variables that are used in NFS
*/ 

LOCAL STATUS nfsClientCacheInit (myModuleList)
    FAST MODULE_LIST *myModuleList;

    {
    /* allocate the module statics */

    myModuleList->nfsClientCache = 
	 (struct moduleStatics *) malloc (sizeof (struct moduleStatics));

    if (myModuleList->nfsClientCache == NULL)
	{
	printErr ("nfsClientCacheInit: out of Memory! \n");
	return (ERROR);
	}

    bzero ((char *) myModuleList->nfsClientCache,sizeof (struct moduleStatics)); 
    return (OK);
    }
/*******************************************************************************
*
* nfsClientCacheCleanUp - clean up the nfs client cache
*
* This routine is used to free up structures that are part of a valid
* nfs client cache.  The client cache is marked as invalid.
*
* NO_MANUAL
*/ 

VOID nfsClientCacheCleanUp (pClientCache)
    FAST struct moduleStatics *pClientCache;	/* pointer to client cache */

    {
    if (pClientCache->valid)
	{
	pClientCache->valid = FALSE;

	close (pClientCache->socket);

	if (pClientCache->oldhost != NULL)
	    {
	    free (pClientCache->oldhost);
	    pClientCache->oldhost = NULL;
	    }

	if (pClientCache->client != NULL)
	    {
	    if (pClientCache->client->cl_auth != NULL)
		{
		auth_destroy (pClientCache->client->cl_auth);
		pClientCache->client->cl_auth = NULL;
		}
	    clnt_destroy (pClientCache->client);
	    pClientCache->client = NULL;
	    }
	}
    }
/*******************************************************************************
*
* nfsClientCall - make an NFS request to the server using RPC
*
* This is the routine which does the actual RPC call to the server.
* All NFS routines which communicate with the file server use this routine.
*
* RETURNS:  OK | ERROR
*/ 

LOCAL STATUS nfsClientCall (host, prognum, versnum, procnum, inproc, in,
		      outproc, out)
    char *host;		/* server's host name */
    u_int prognum;	/* RPC program number */
    u_int versnum;	/* RPC version number */
    u_int procnum;	/* RPC procedure number */
    xdrproc_t inproc;	/* xdr routine for args */
    char *in;
    xdrproc_t outproc;	/* xdr routine for results */
    char *out;

    {
    int hp;				/* host's inet address */
    struct sockaddr_in server_addr;
    struct timeval timeout;
    struct timeval tottimeout;
    enum clnt_stat clientStat;
    struct moduleStatics *ms;
    int optval;

    if (nfsInit () == ERROR)
	return (ERROR);
    
    ms = taskModuleList->nfsClientCache;

    if (ms->valid && ms->oldprognum == prognum && ms->oldversnum == versnum
	&& strcmp(ms->oldhost, host) == 0)
	{
    	/* reuse old client */
    	}
    else
	{
	/* get rid of old client */

	if (ms->valid)
	    nfsClientCacheCleanUp (ms);

	/* create new client */

	ms->socket = RPC_ANYSOCK;

	if ((hp = hostGetByName (host)) == NULL)
	    return (ERROR);
 
	timeout.tv_usec = 0;
	timeout.tv_sec = 5;

	server_addr.sin_addr.s_addr = hp;
	server_addr.sin_family = AF_INET;
	server_addr.sin_port =  0;

	if ((ms->client = clntudp_create(&server_addr, prognum, versnum,
					 timeout, &ms->socket)) == NULL)
	    {
	    server_addr.sin_port = NFS_PORT;
	    ms->client = clntudp_create (&server_addr, prognum, versnum,
					 timeout, &ms->socket);
	    if (ms->client == NULL)
		return (ERROR);
	    }

	optval = nfsSockBufSize;
	if ((setsockopt (ms->socket, SOL_SOCKET, SO_SNDBUF, (char *)&optval,
			 sizeof (optval)) == ERROR)
	    || (setsockopt (ms->socket, SOL_SOCKET, SO_RCVBUF, (char *)&optval,
			    sizeof (optval)) == ERROR))
	    {
	    close (ms->socket);
	    return (ERROR);
	    }

	ms->client->cl_auth = authunix_create (rpcAuthUnix.machname,
					       rpcAuthUnix.uid, rpcAuthUnix.gid,
                                               rpcAuthUnix.len,
					       rpcAuthUnix.aup_gids);
	if (ms->client->cl_auth == NULL)
	    {
	    nfsClientCacheCleanUp (ms);
	    errnoSet (S_nfsLib_NFS_AUTH_UNIX_FAILED);
	    return (ERROR);
	    }

	ms->oldprognum = prognum;
	ms->oldversnum = versnum;

	if (ms->oldhost == NULL)
	    {
	    /* XXX define a constant here for oldhost string size */

	    ms->oldhost = malloc (256);
	    ms->oldhost[0] = 0;
	    }

	strcpy(ms->oldhost, host);

	ms->valid = 1;
	}

    /* set time to allow results to come back */

    tottimeout.tv_sec  = nfsTimeoutSec;
    tottimeout.tv_usec = nfsTimeoutUSec;

    clientStat = clnt_call (ms->client, procnum, inproc, in, outproc, out,
			    tottimeout);

    if (clientStat != RPC_SUCCESS)
	{
	/* XXX this should be more graceful */

	nfsClientCacheCleanUp (ms);
	rpcClntErrnoSet (clientStat);
	return (ERROR);
	}
    return (OK);
    }
/*******************************************************************************
*
* nfsClientClose - close the NFS client socket
*
* NOMANUAL
*/ 

VOID nfsClientClose ()

    {
    nfsClientCacheCleanUp (taskModuleList->nfsClientCache);
    }
/*******************************************************************************
*
* nfsMountListPrint - prints a list of mount entries
*/ 

LOCAL VOID nfsMountListPrint (hostName, pMountList)
    char *hostName;
    mountlist *pMountList;

    {
    printf ("filesystems mounted on %s:\n", hostName);
    while (pMountList != NULL)
	{
	printf ("%s:%s\n", pMountList->hostname, pMountList->directory);
	pMountList = pMountList->nextentry;
	}
    }
/*******************************************************************************
*
* nfsGroupsPrint - print a list of groups
*/ 

LOCAL VOID nfsGroupsPrint (pGroup)
    groups *pGroup;

    {
    if (pGroup == NULL)
	printf ("everyone");

    while (pGroup != NULL)
	{
	printf ("%s ", pGroup->grname);
	pGroup = pGroup->nextgroup;
	}
    }
/*******************************************************************************
*
* nfsExpListPrint - prints a list of exported file systems on a host
*/ 

LOCAL VOID nfsExpListPrint (hostName, pExport)
    char *hostName;
    exports *pExport;

    {
    printf ("export list for %s:\n", hostName);
    while (pExport != NULL)
	{
	printf ("%-19s ", pExport->filesys);
	nfsGroupsPrint (pExport->grouplist);
	printf ("\n");
	pExport = pExport->nextexport;
	}
    }
/*******************************************************************************
*
* nfsDirListPrint - print a list of directories
*/ 

LOCAL VOID nfsDirListPrint (pDirList)
    dirlist *pDirList;

    {
    entry *pEntry = pDirList->entries;

    while (pEntry != NULL)
	{
	printf ("%s\n", pEntry->name);
	pEntry = pEntry->nextentry;
	}
    }
/*******************************************************************************
*
* nfsHelp - help command for NFS
*
* Print a brief synopsis of NFS facilities that are typically called from
* the shell.
* .CS
*  nfsHelp                       Print this list
*  netHelp                       Print general network help list
*  nfsMount "host","filesystem"[,"devname"]  Create device with
*                                  filesystem/directory from host
*  nfsUnmount "devname"          Remove an NFS device           
*  nfsAuthUnixShow               Print current UNIX authentication
*  nfsAuthUnixPrompt             Prompt for UNIX authentication
*  nfsDevShow                    Print list of NFS devices
*  nfsExportShow "host"          Print a list of NFS file systems which
*                                  are exported on the specified host
*  mkdir "dirname"               Create directory
*  rm "file"                     Remove file
*  
*  EXAMPLE:  -> hostAdd "wrs", "90.0.0.2"
*            -> nfsMount "wrs","/disk0/path/mydir","/mydir/"
*            -> cd "/mydir/"                                 
*            -> nfsAuthUnixPrompt     /* fill in user id, etc. *
*            -> ls                    /* list /disk0/path/mydir *
*            -> copy < foo            /* copy foo to standard out *
*            -> ld < foo.o            /* load object module foo.o *
*            -> nfsUnmount "/mydir/"  /* remove NFS device /mydir/ *
* .CE
*/

VOID nfsHelp ()

    {
    static char *help_msg [] =
{
"nfsHelp                       Print this list",
"netHelp                       Print general network help list",
"nfsMount \"host\",\"filesystem\"[,\"devname\"]  Create device with",
"                                filesystem/directory from host",
"nfsUnmount \"devname\"          Remove an NFS device",
"nfsAuthUnixShow               Print current UNIX authentication",
"nfsAuthUnixPrompt             Prompt for UNIX authentication",
"nfsDevShow                    Print list of NFS devices",
"nfsExportShow \"host\"          Print a list of NFS file systems which",
"                                are exported on the specified host",
"mkdir \"dirname\"               Create directory",
"rm \"file\"                     Remove file",
"",
"EXAMPLE:  -> hostAdd \"wrs\", \"90.0.0.2\"",
"          -> nfsMount \"wrs\",\"/disk0/path/mydir\",\"/mydir/\"",
"          -> cd \"/mydir/\"",
"          -> nfsAuthUnixPrompt     /* fill in user id, etc. *",
"          -> ls                    /* list /disk0/path/mydir *",
"          -> copy < foo            /* copy foo to standard out *",
"          -> ld < foo.o            /* load object module foo.o *",
"          -> nfsUnmount \"/mydir/\"  /* remove NFS device /mydir/ *",
NULL
};
    FAST int i;

    for (i = 0; i < NELEMENTS(help_msg); i++)
	printf ("%s\n", help_msg [i]);
    printf ("\n");
    }
/*******************************************************************************
*
* nfsErrnoSet - set NFS status
*
* nfsErrnoSet calls errnoSet with the given "nfs stat" or'd with the
* NFS status prefix.  
*/

LOCAL VOID nfsErrnoSet (status)
    enum nfsstat status;

    {
    errnoSet (M_nfsStat | (int) status);
    }
/*******************************************************************************
*
* nfsAuthUnixPrompt - modify NFS UNIX authentication parameters
*
* This routine is provided so that the user can easily
* change UNIX authentication parameters from the shell.
*
* EXAMPLE
* .CS
*  -> nfsAuthUnixPrompt
*  machine name:   yuba 
*  user id:        2001 128
*  group id:       100 
*  num of groups:  1 3
*  group #1:        100 100
*  group #2:        0 120
*  group #3:        0 200
*  value = 3 = 0x3        
*  -> 
* .CE
*
* SEE ALSO: nfsAuthUnix{SG}et (2), nfsAuthUnixShow (2), nfsIdSet (2)
*/ 

VOID nfsAuthUnixPrompt ()

    {
    char machname [AUTH_UNIX_FIELD_LEN];  /* host name where client is */
    int uid;				/* client's UNIX effective uid */
    int gid;				/* client's current group id */
    int len;				/* element length of aup_gids */
    int aup_gids [MAX_GRPS];		/* array of groups user is in */
    int i;

    nfsAuthUnixGet (machname, &uid, &gid, &len, aup_gids);

    promptParamString ("machine name:  ", machname, sizeof (machname));
    promptParamNum ("user id:       ", &uid, 8, "%d ");
    promptParamNum ("group id:      ", &gid, 8, "%d ");
    promptParamNum ("num of groups: ", &len, 8, "%d ");
    for (i = 0; i < len; i++)
	{
	printf ("group #%d:       ", i + 1);
	promptParamNum ("", &aup_gids [i], 8, "%d ");
	}
    nfsAuthUnixSet (machname, uid, gid, len, aup_gids);
    }
/*******************************************************************************
*
* nfsAuthUnixShow - print the NFS UNIX authentication parameters
*
* This routine prints out the parameters set by nfsAuthUnixSet (2), or
* nfsAuthUnixPrompt (2).
*
* EXAMPLE:
* .CS
*  -> nfsAuthUnixShow
*  machine name = yuba
*  user id      = 2001
*  group id     = 100
*  group [0]    = 100
*  value = 1 = 0x1
*  ->
* .CE
*
* SEE ALSO: nfsAuthUnix{SG}et (2), nfsAuthUnixPrompt (2), nfsIdSet (2)
*/ 

VOID nfsAuthUnixShow ()

    {
    char machname [AUTH_UNIX_FIELD_LEN]; /* host name where client is */
    int uid;				/* client's UNIX effective uid */
    int gid;				/* client's current group id */
    int len;				/* element length of aup_gids */
    int aup_gids [MAX_GRPS];		/* array of groups user is in */
    int i;

    nfsAuthUnixGet (machname, &uid, &gid, &len, aup_gids);

    printf ("machine name = %s\n", machname);
    printf ("user id      = %d\n", uid);
    printf ("group id     = %d\n", gid);
    for (i = 0; i < len; i++)
	printf ("group [%d]    = %d\n", i, aup_gids [i]);
    }
/*******************************************************************************
*
* nfsAuthUnixSet - set the NFS UNIX authentication parameters
*
* This routine sets UNIX authentication.
* It is initially called in usrNetInit (2), in usrConfig (1).
*
* SEE ALSO: nfsAuthUnixPrompt (2), nfsAuthUnixGet (2), nfsAuthUnixShow (2),
*           nfsIdSet (2)
*/ 

VOID nfsAuthUnixSet (machname, uid, gid, ngids, aup_gids)
    char *machname;	/* host machine        */
    int	uid;		/* user id             */
    int	gid;		/* group id            */
    int	ngids;		/* number of group ids */
    int	*aup_gids;	/* array of group ids  */

    {
    int i;

    strcpy (rpcAuthUnix.machname, machname);
    rpcAuthUnix.uid = uid;
    rpcAuthUnix.gid = gid;
    rpcAuthUnix.len = (ngids < MAX_GRPS ? ngids : MAX_GRPS);
    for (i = 0; i < ngids; i++)
	rpcAuthUnix.aup_gids [i] = aup_gids [i];
    }
/*******************************************************************************
*
* nfsAuthUnixGet - get the NFS UNIX authentication parameters
*
* This routine returns the previously set UNIX authentication.
*
* SEE ALSO: nfsAuthUnixPrompt (2), nfsAuthUnixSet (2), nfsAuthUnixShow (2),
*           nfsIdSet (2)
*/ 
    
VOID nfsAuthUnixGet (machname, pUid, pGid, pNgids, gids)
    char *machname;	/* where to store host machine        */
    int	*pUid;		/* where to store user id             */
    int	*pGid;		/* where to store group id            */
    int	*pNgids;	/* where to store number of group ids */
    int	*gids;		/* where to store array of group ids  */

    {
    int i;

    strcpy (machname, rpcAuthUnix.machname);
    *pUid   = rpcAuthUnix.uid;
    *pGid   = rpcAuthUnix.gid;
    *pNgids = rpcAuthUnix.len;

    for (i = 0; i < rpcAuthUnix.len; i++)
	gids [i] = rpcAuthUnix.aup_gids [i];
    }
/*******************************************************************************
*
* nfsIdSet - set the id number of the NFS UNIX authentication parameters
*
* This routine sets UNIX authentication identification number.
* For most NFS permission needs, only the id number needs to be changed.
* Set the number to the user id on the NFS server.
*
* SEE ALSO: nfsAuthUnixPrompt (2), nfsAuthUnix{SG}et (2), nfsAuthUnixShow (2)
*/ 

VOID nfsIdSet (uid)
    int	uid;		/* user id on host machine */

    {
    rpcAuthUnix.uid = uid;
    }
/*******************************************************************************
*
* printClear - print string with '?' for unprintable characters
*/

LOCAL VOID printClear (param)
    FAST char *param;

    {
    FAST char ch;

    while ((ch = *(param++)) != EOS)
	printf ("%c", (isascii (ch) && isprint (ch)) ? ch : '?');
    }
/*******************************************************************************
*
* promptParamString - prompt the user for a string parameter
*
* - carriage return leaves the parameter unmodified;
* - "." clears the parameter (null string).
*/

LOCAL VOID promptParamString (msg, param, fieldWidth)
    char *msg;
    char *param;
    int fieldWidth;

    {
    int i;
    char buf [100];

    FOREVER 
	{
	printf ("%s ", msg);
	printClear (param);
	printf (" ");

	i = fioRdString (STD_IN, buf, sizeof (buf));
	if (i < fieldWidth)
	    break;
	printf ("too big - maximum field width = %d.\n", fieldWidth);
	}

    if (i == 1)
	return;			/* just CR; leave field unchanged */
    
    if (buf[0] == '.')
	{
	param [0] = EOS;	/* just '.'; make empty field */
	return;
	}

    strcpy (param, buf);	/* update parameter */
    }
/*******************************************************************************
*
* promptParamNum - prompt the user for a parameter
*
* - carriage return leaves the parameter unmodified;
* - "." clears the parameter (0).
*/

LOCAL VOID promptParamNum (msg, pParam, fieldWidth, format)
    char *msg;
    int *pParam;
    int fieldWidth;
    char *format;
    {

    int i;
    char buf [100];

    FOREVER 
	{
	strcpy (buf, "%s ");
	strcat (buf, format);

	printf (buf, msg, *pParam);

	i = fioRdString (STD_IN, buf, sizeof (buf));
	if (i < fieldWidth)
	    break;
	printf ("too big - maximum field width = %d.\n", fieldWidth);
	}

    if (i == 1)
	return;			/* just CR; leave field unchanged */
    
    if (buf[0] == '.')
	{
	pParam = 0;		/* just '.'; make empty field */
	return;
	}

    sscanf (buf, format, pParam);	/* scan field */
    }
