/* netDrv.c - network remote file I/O driver */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
03q,04apr89,gae  allowed open() to succeed for NFS directories so that ls()
	   +dab  would work correctly.  Added netLsStr variable.
03p,07jul88,jcf  lint.
03o,22jun88,dnw  name tweaks.
		 coalesced 3 malloc's into 1 in netCreate().
03n,16jun88,llk  added netLs.
03m,04jun88,llk  changed how netCreate retrieves directory and file name.
		 moved pathCat to pathLib.
03l,30may88,dnw  changed to v4 names.
03k,04may88,jcf  changed semaphores to be compatible with new semLib.
03j,21apr88,gae  fixed documentation about fseek().
03i,17mar88,gae  fixed netPut() bug.
		 fixed potential ssemGive() & releaseNetFd() ordering bug.
		 fixed bug in netCreate() - file not created if nothing written.
03h,28jan88,jcf  made kernel independent.
03g,18nov87,ecs  documentation.
03f,06nov87,ecs  documentation.
03e,16oct87,gae  fixed bug in netClose() - return value indeterminate.
		 changed "cd/cat" to use pathCat to construct UNIX filename.
		 Added pathCat().  Semaphored fd usage.
		 Added "super semaphores" to allow successive semTake's on
		 the same semaphore by the same task (needed for recursion).
03d,25mar87,jlf  documentation
	    dnw  added missing parameter on ftpGetReply calls.
		 prepended FTP_ to ftp reply codes.
03c,12feb86,dnw  fixed bug in netPut.  This time for sure.
03b,14jan87,dnw  fixed bug in netPut.
03a,01dec86,dnw  added ftp protocol option.
		 changed to not get include files from default directories.
02h,08nov86,llk  documentation.
02g,07oct86,gae	 made netCreate() use ioGetDefDevTail() instead of global
		 'currentDirectory' to conclude the remote directory name.
		 Use remGetCurId() to get 'currentIdentity'...included remLib.h
02f,05sep86,jlf  made netIoctrl return nbytes for FIONREAD in arg.
		 minor documentation.
02e,29jul86,llk  cleaned up netGet.
02d,27jul86,llk  receives standard error messages from Unix when giving
		 a remote command (remRcmd).  Forwards it on to user.
		 Added getNetStatus.
02c,02jul86,jlf  documentation
02b,21jun86,rdc  fixed bug in netCreate - wasn't allocating sufficient space
		 for strings.
02a,21may86,llk	 got rid of ram disk.  Implemented network file desciptors.
		 Wrote all routines except netDrv and netDevCreate.
01c,03mar86,jlf  changed ioctrl calls to ioctl.
		 changed netIoctrl to netIoctl.
01b,28jan86,jlf  documentation, made netIoctrl be LOCAL.
01a,14sep85,rdc  written.
*/

/*
NetDrv provides a device for accessing files transparently over the network.
By creating a network device with netDevCreate, files on a remote UNIX machine
may be accessed exactly like local files.

When a remote file is opened, the entire file is copied over the network to a 
local network file descriptor.  If a remote file is created, an empty local
network file descriptor is opened.  Reads, writes, and ioctl functions are 
done on the local copy of the file.  When the file is closed, if it was opened 
for WRITE or UPDATE and the file has changed, the local copy is sent back over 
the net to the UNIX machine.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, netDrv () to
initialize the driver, and netDevCreate () to create devices.

NETDRV
Before using the driver, it must be initialized by calling the routine:
.CS
    netDrv ()
.CE
This routine should be called exactly once, before any read's, write's, or
netDevCreate's.  Normally, it is called from usrRoot (2) in usrConfig (1).

CREATING NET DEVICES
In order to access a particular computer, a "network device" must
be created for it.
For instance, if there is a UNIX host on the network whose name is "wrs",
files may be used on it by making a device called "wrs:".  The routine
that does this is netDevCreate.  It is defined as follows:

.CS
    STATUS netDevCreate (devName, host, protocol)
        char *devName;	/* name of device to create *
        char *host;	/* host this device will talk to *
	int protocol;   /* remote file access protocol;
			 * 0 = rsh, 1 = ftp *
.CE

For instance, to create the "wrs:" device just discussed, the proper call
would be:
.CS
   netDevCreate ("wrs:", "wrs", 0);
.CE
After creating this device, a file on the UNIX system "wrs" named /usr/dog
may be accessed as "wrs:/usr/dog".

The host ("wrs" in this case) must have already been created
with a hostAdd (2) call.

IOCTL
The net driver ioctl (2) call responds to the following ioctl codes.
.CS
 FIOSEEK  - seek to a byte in the open file, same as lseek (2).
 FIOWHERE - return the current byte pointer into the open file.
 FIONREAD - return (in *arg) the number of unread bytes in the file.
.CE

SEE ALSO: remLib (1), netLib (1), sockLib (1), hostAdd (2)
*/

#include "vxWorks.h"
#include "socket.h"
#include "iosLib.h"
#include "ioLib.h"
#include "memLib.h"
#include "semLib.h"
#include "strLib.h"
#include "remLib.h"
#include "ftpLib.h"
#include "lstLib.h"
#include "netDrv.h"

#define FD_MODE		3	/* mode mask for opened file */
				/* READ, WRITE or UPDATE */
#define	FD_DIRTY	4	/* mask for the dirty bit */
				/* in the options field of NET_FD, bit [2] */
#define DATASIZE	512	/* bytes per block of data */

/* remote file access protocols */

#define PROTO_RSH	0	/* UNIX rsh protocol */
#define PROTO_FTP	1	/* ARPA ftp protocol */

#define	RSHD	514		/* rshd service */

/* XXX super semaphore - until semLib is more sophisticated */

typedef struct			/* SSEMAPHORE */
    {
    SEM_ID semId;
    int tid;
    } SSEMAPHORE;


typedef struct			/* NET_DEV */
    {
    DEV_HDR devHdr;
    char host[MAX_FILENAME_LENGTH];	/* host for this device	*/
    int protocol;			/* remote file access protocol */
    } NET_DEV;

/*
A network file descriptor consists of a NET_FD header with a linked list
of DATABLOCKs maintained by lstLib.  The file pointer (filePtrByte) and
end of file pointer (endOfFile) contain absolute byte numbers starting 
at 0.  For instance, the first byte in the first block is #0.  The first
byte in the second block is #512 for 512 byte blocks.  The first byte in 
the third block is #1024, etc.  Therefore, a byte number modulo 512 gives
the index of the byte in databuf.
*/

typedef struct			/* DATA_BLOCK */
    {
    NODE link;			/* link to adjacent DATABLOCKs in file */
    int used;			/* number of bytes used in databuf (offset) */
    char databuf [DATASIZE];	/* store data here */
    } DATABLOCK;

/* network file descriptor */

typedef struct			/* NET_FD */
    {
    LIST dataLst;		/* list of data blocks */
    NET_DEV *pNetDev;		/* pointer to this file's network device */
    DATABLOCK *filePtrBlock;	/* pointer to current datablock */
    int filePtrByte;		/* number of current byte */
    int endOfFile;		/* number of byte after last file byte */
    int options;		/* contains mode and dirty bit */
    char *remDirName;   	/* directory name on remote machine */
    char *remFileName;   	/* file name on remote machine */
    SSEMAPHORE netFdSem;	/* accessing semaphore */
    } NET_FD;


char *netLsStr = "/bin/ls -a %s";

LOCAL int netDrvNum;		/* this particular driver number */

/* forward declarations */

LOCAL int netCreate ();
LOCAL int netOpen ();
LOCAL STATUS netClose ();
LOCAL int netRead ();
LOCAL int netWrite ();
LOCAL int netIoctl ();
LOCAL int netSeek ();
LOCAL int netWhere ();

/*******************************************************************************
*
* netDrv - network remote file driver installation
*
* Initializes and installs the network driver.
* Must be called before other network remote file functions are used.
*
* RETURNS: OK or ERROR
*/

STATUS netDrv ()

    {
    if (netDrvNum > 0)
	return (OK);	/* driver already installed */

    netDrvNum = iosDrvInstall (netCreate, (FUNCPTR) NULL, 
			       netOpen, netClose, 
			       netRead, netWrite, netIoctl);

    return (netDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* netDevCreate - create a remote file device
*
* This routine creates a remote device.  Normally, a network device is
* created for each remote machine whose files are to be accessed.  For
* instance, if there is a UNIX host on the network whose name is "wrs",
* files may be used on it by making a device as follows:
*
*    netDevCreate ("wrs:", "wrs", protocol);
*
* After creating this device, a file on the UNIX system named /usr/dog
* may be accessed as wrs:/usr/dog.
*
* The host must have already been created with a hostAdd call.
*
* SEE ALSO: hostAdd (2)
*
* RETURNS: OK or ERROR
*/

STATUS netDevCreate (devName, host, protocol)
    char *devName;	/* name of device to create */
    char *host;		/* host this device will talk to */
    int protocol;	/* remote file access protocol */
			/* 0 = RSH, 1 = FTP */

    {
    FAST NET_DEV *pNetDev;
    
    if (netDrvNum < 1)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}

    pNetDev = (NET_DEV *) malloc (sizeof (NET_DEV));

    if (pNetDev == NULL)
	return (ERROR);

    strcpy (pNetDev->host, host);
    pNetDev->protocol = protocol;

    return (iosDevAdd ((DEV_HDR *) pNetDev, devName, netDrvNum));
    }

/* routines supplied to I/O system */

/*******************************************************************************
*
* netCreate - create a remote file
*
* Returns an open network file descriptor.
* Files are created with WRITE or UPDATE modes.
* A READ mode defaults to UPDATE.
*
* Called only by the I/O system.
*
* RETURNS: pointer to open network file descriptor or ERROR
*/

LOCAL int netCreate (pNetDev, name, mode)
    NET_DEV *pNetDev;
    char *name;		/* remote file name */
    int mode;		/* ignored, always set to WRITE */

    {
    FAST NET_FD *pNetFd;
    DATABLOCK *pData;
    char *pFileName;
    FAST int fileNameLen;
    FAST int dirNameLen;
    char *ptr;

    /* don't allow null filenames */

    if (name[0] == EOS)
	{
	errnoSet (S_ioLib_NO_FILENAME);
	return (ERROR);
	}

    /* get directory and filename pointers and lengths */

    pathLastName (name, &pFileName);
    fileNameLen = strlen (pFileName);
    dirNameLen  = pFileName - name;

    /* allocate and initialize net file descriptor;
     * leave room for directory and file name and both EOSs */

    if ((ptr = malloc ((unsigned) (sizeof (NET_FD) + fileNameLen +
				   dirNameLen + 2))) == NULL)
	return (ERROR);

    pNetFd = (NET_FD *) ptr;

    pNetFd->pNetDev      = pNetDev;
    pNetFd->filePtrByte  = 0;
    pNetFd->endOfFile    = 0;	/* XXX should be determined when UPDATE! */
    pNetFd->options      = FD_MODE & ((mode == WRITE) ? WRITE : UPDATE);
    pNetFd->options     |= FD_DIRTY;	/* set dirty */

    /* set remote directory name and file name */

    pNetFd->remFileName = ptr + sizeof (NET_FD);
    strcpy (pNetFd->remFileName, pFileName);

    pNetFd->remDirName = pNetFd->remFileName + fileNameLen + 1;
    strncpy (pNetFd->remDirName, name, dirNameLen);
    pNetFd->remDirName [dirNameLen] = EOS;

    /* attach first empty datablock */

    pData = (DATABLOCK *) malloc (sizeof (DATABLOCK));
    pData->used = 0;

    pNetFd->filePtrBlock = pData;

    lstInit (&pNetFd->dataLst);
    lstAdd (&pNetFd->dataLst, (NODE *) pData);


    ssemInit (&pNetFd->netFdSem);
    ssemGive (&pNetFd->netFdSem);
    
    return ((int) pNetFd);
    }
/*******************************************************************************
*
* netOpen - open a remote file
*
* netOpen loads the remote file from the host into a network file descriptor.
* If the file does not exist, or it is not readable on the UNIX machine, then
* the UNIX error message is printed on the standard error fd and ERROR is 
* returned.
*
* Called only by the I/O system.
*
* RETURNS: pointer to open network file descriptor or ERROR
*/

LOCAL int netOpen (pNetDev, name, mode)
    FAST NET_DEV *pNetDev;
    char *name;		/* remote file name to open	*/
    int mode;		/* READ, WRITE or UPDATE	*/

    {
    STATUS status;
    FAST NET_FD *pNetFd = (NET_FD *)netCreate (pNetDev, name, mode);

    if (pNetFd == (NET_FD *)ERROR)
	return (ERROR);

    ssemTake (&pNetFd->netFdSem);

    /* netCreate sets a READ mode to UPDATE and sets dirty bit,
     * set it back to original mode */

    pNetFd->options = mode & FD_MODE;

    if (netGet (pNetFd) != OK)
	{
	/* not able to get file from host */

	ssemGive (&pNetFd->netFdSem);
	releaseNetFd (pNetFd);
	return (ERROR);
	}

    /* netGet might have moved the file pointer, reset it to the beginning */

    status = netSeek (pNetFd, 0);
    
    ssemGive (&pNetFd->netFdSem);

    return (status == ERROR ? ERROR : (int) pNetFd);
    }
/*******************************************************************************
*
* netClose - close a remote file.  Copy it to the remote machine.
*
* If the file is WRITE or UPDATE mode and it has been written to (it's
* dirty bit is set), then the file is copied over to UNIX.
*
* Called only by the I/O system.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS netClose (pNetFd)
    FAST NET_FD *pNetFd;

    {
    STATUS status = OK;

    ssemTake (&pNetFd->netFdSem);

    /* if file has dirty bit set, it's been changed. Copy it to UNIX machine */

    if (pNetFd->options & FD_DIRTY)
	status = netPut (pNetFd);

    ssemGive (&pNetFd->netFdSem);
    releaseNetFd (pNetFd);

    return (status);
    }
/*******************************************************************************
*
* netLs - list a remote directory
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS netLs (pNetFd)
    FAST NET_FD *pNetFd;

    {
    int dataSock;
    int ctrlSock;
    char buf [DATASIZE];	/* make buf the same size as NET_FD's databuf */
    FAST int nBytes;
    char command[100];
    char buffer [100];
    STATUS status = OK;
    char usr [MAX_IDENTITY_LEN];
    char dummyPasswd [MAX_IDENTITY_LEN];

    remCurIdGet (usr, dummyPasswd);

    pathCat (pNetFd->remDirName, pNetFd->remFileName, buffer);
    sprintf (command, netLsStr, buffer);

    dataSock = rcmd (pNetFd->pNetDev->host, RSHD, usr,
			usr, command, &ctrlSock);

    if (dataSock == ERROR)
	return (ERROR);

    /* read bytes from socket and write them 
     * to standard out one block at a time 
     */

    while ((status == OK) &&
	   ((nBytes = read (dataSock, buf, sizeof (buf))) > 0))
    	{
	if (write (STD_OUT, buf, nBytes) != nBytes)
	    status = ERROR;
    	}

    if (nBytes < 0)		/* recv error */
	status = ERROR;

    /* check control socket for error */

    if ((nBytes = fioRead (ctrlSock, buf, sizeof (buf) - 1)) > 0)
	{
	/* print error message on standard error fd */

	buf [nBytes] = EOS;	/* insure string termination */
	printErr ("%s:%s", pNetFd->pNetDev->host, buf);

	/* set the task's status according to the UNIX error */
	getNetStatus (buf);

	status = ERROR;
	}

    close (dataSock);
    close (ctrlSock);

    return (status);
    }
/*******************************************************************************
*
* netGet - downLoad a file from a remote machine via the network.
*
* The remote shell daemon on the machine 'host' is used to download
* the given file to the specified previously opened network file descriptor.
* The remote userId should have been set previously by a call to iam (2).
* If the file does not exist, the error message from the UNIX 'host' 
* is printed to the VxWorks standard error fd and ERROR is returned.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS netGet (pNetFd)
    FAST NET_FD *pNetFd;

    {
    int dataSock;
    int ctrlSock;
    char buf [DATASIZE];	/* make buf the same size as NET_FD's databuf */
    FAST int nBytes;
    char command[100];
    char buffer [100];
    int saveOptions;
    STATUS status = OK;
    char usr [MAX_IDENTITY_LEN];
    char passwd [MAX_IDENTITY_LEN];
    char *errMsg = "cat: read error: Is a directory";
    int errMsgLen = strlen (errMsg);

    remCurIdGet (usr, passwd);

    if (pNetFd->pNetDev->protocol == PROTO_FTP)
	{
	if (ftpXfer (pNetFd->pNetDev->host, usr, passwd, "", 
		     "RETR %s", pNetFd->remDirName, pNetFd->remFileName,
		     &ctrlSock, &dataSock) == ERROR)
	    {
	    return (ERROR);
	    }
	}
    else
	{
	pathCat (pNetFd->remDirName, pNetFd->remFileName, buffer);
	sprintf (command, "/bin/cat < %s", buffer);

	dataSock = rcmd (pNetFd->pNetDev->host, RSHD, usr,
			    usr, command, &ctrlSock);

	if (dataSock == ERROR)
	    return (ERROR);
	}

    /* Set file pointer to beginning of file */

    if (netSeek (pNetFd, 0) == ERROR)
	{
	close (dataSock);
	close (ctrlSock);
	return (ERROR);
	}

    /* set mode to write so that file can be written to,
    *  save original options so they can be restored later
    */

    saveOptions = pNetFd->options;
    pNetFd->options = WRITE & FD_MODE;

    /* read bytes from socket and write them 
     * out to file descriptor one block at a time 
     */

    while ((status == OK) &&
	   ((nBytes = read (dataSock, buf, sizeof (buf))) > 0))
    	{
    	if (netWrite (pNetFd, buf, nBytes) != nBytes)
	    status = ERROR;
    	}

    if (nBytes < 0)		/* recv error */
	status = ERROR;

    if (pNetFd->pNetDev->protocol == PROTO_FTP)
	{
	if (ftpReplyGet (ctrlSock, FALSE) != FTP_COMPLETE)
	    status = ERROR;
	}
    else
	{
	/* check control socket for error */

	if ((nBytes = fioRead (ctrlSock, buf, sizeof (buf) - 1)) > 0)
	    {

	    /* print error message on standard error fd */

	    buf [nBytes] = EOS;	/* insure string termination */

	    /* check error message indicating cat of NFS mounted directory */

	    if (strncmp (buf, errMsg, errMsgLen) != 0)
		{
		printErr ("%s:%s", pNetFd->pNetDev->host, buf);

		/* set the task's status according to the UNIX error */
		getNetStatus (buf);

		status = ERROR;
		}
	    }
	}

    close (dataSock);
    close (ctrlSock);

    pNetFd->options = saveOptions;	/* restore original options */

    return (status);
    }
/*******************************************************************************
*
* netPut - upload a file to a remote machine via the network.
*
* The remote shell daemon on the machine 'host' is used to upload
* from the open network file descriptor to a remote file.
* The remote userId should have been set previously be a call
* to iam (2).  If an error occurs, the UNIX error is output to the
* VxWorks standard error fd.
* 
* RETURNS: OK | ERROR
*/

LOCAL STATUS netPut (pNetFd)
    FAST NET_FD *pNetFd;
    
    {
    int dataSock;
    int ctrlSock;
    char buf [DATASIZE];	/* make buf the same size as NET_FD's databuf */
    FAST int nBytes;
    char command[100];
    char buffer[100];
    int saveOptions;
    STATUS status = OK;
    char usr [MAX_IDENTITY_LEN];
    char passwd [MAX_IDENTITY_LEN];

    remCurIdGet (usr, passwd);

    if (pNetFd->pNetDev->protocol == PROTO_FTP)
	{
	if (ftpXfer (pNetFd->pNetDev->host, usr, passwd, "", 
		     "STOR %s", pNetFd->remDirName, pNetFd->remFileName,
		     &ctrlSock, &dataSock) == ERROR)
	    {
	    return (ERROR);
	    }
	}
    else
	{
	pathCat (pNetFd->remDirName, pNetFd->remFileName, buffer);
	sprintf (command, "/bin/cat > %s", buffer);

	dataSock = rcmd (pNetFd->pNetDev->host, RSHD, usr, 
			    usr, command, &ctrlSock);

	if (dataSock == ERROR)
	    return (ERROR);
	}

    /* Set file pointer to beginning of file */

    if (netSeek (pNetFd, 0) == ERROR)
	{
	close (dataSock);
	close (ctrlSock);
	return (ERROR);
	}

    /* set mode to write so that file can be written to,
    *  save original options so they can be restored later
    */

    saveOptions = pNetFd->options;
    pNetFd->options = READ & FD_MODE;

    /* Read the data from one DATABLOCK into buffer.
    *  Continue until file pointer reaches the end of file.
    */

    while (status == OK && (nBytes = netRead (pNetFd, buf, sizeof (buf))) > 0)
	{
	if (write (dataSock, buf, nBytes) != nBytes)
	    status = ERROR;
	}

    if (nBytes < 0)		/* netRead error */
	status = ERROR;

    if (close (dataSock) == ERROR)
	status = ERROR;

    if (pNetFd->pNetDev->protocol == PROTO_FTP)
	{
	if (ftpReplyGet (ctrlSock, FALSE) != FTP_COMPLETE)
	    status = ERROR;
	}
    else
	{
	/* check control socket for error */

	if ((nBytes = fioRead (ctrlSock, buf, sizeof (buf) - 1)) > 0)
	    {
	    /* print error message on standard error fd */

	    buf [nBytes] = EOS;	/* insure string termination */
	    printErr ("%s:%s", pNetFd->pNetDev->host, buf);

	    /* set the task's status according to the UNIX error */
	    getNetStatus (buf);

	    status = ERROR;
	    }
	}

    close (ctrlSock);

    pNetFd->options = saveOptions;	/* restore original options */

    return (status);
    }
/****************************************************************************
*
* getNetStatus - set task status according to network error
*
* Compares string in buf with some known UNIX errors that can occur when
* copying files over the network.  Sets task status accordingly.
*/

LOCAL VOID getNetStatus (buf)
    char *buf;		/* buffer containing string with UNIX error */
    
    {
    FAST char *pErr = (char *) index (buf, ':');

    if (pErr == NULL)
        errnoSet (S_netDrv_UNIX_FILE_ERROR);

    else if (strcmp (pErr, ": Permission denied\n") == 0)
        errnoSet (S_netDrv_PERMISSION_DENIED);

    else if (strcmp (pErr, ": No such file or directory\n") == 0)
	errnoSet (S_netDrv_NO_SUCH_FILE_OR_DIR);

    else if (strcmp (pErr, ": Is a directory\n") == 0)
	errnoSet (S_netDrv_IS_A_DIRECTORY);

    else
        errnoSet (S_netDrv_UNIX_FILE_ERROR);
    }
/*******************************************************************************
*
* netRead - read bytes from remote file
*
* netRead reads up to the specified number of bytes from the open network
* file descriptor and puts them into a buffer.  Bytes are read starting
* from the point marked by the file pointer.  The file pointer is then 
* updated to point immediately after the last character that was read.
* 
* Called only by the I/O system.
*
* SIDE EFFECTS: moves file pointer
*
* RETURNS: number of bytes read or ERROR.
*/

LOCAL int netRead (pNetFd, buf, maxBytes)
    FAST NET_FD *pNetFd;	/* pointer to open network file descriptor */
    char *buf;			/* pointer to buffer to receive bytes	*/
    FAST int maxBytes;		/* max number of bytes to read into buffer */

    {
    STATUS status;
    FAST int byteCount = 0;	/* number of bytes read so far */
    FAST DATABLOCK *dptr;	/* points to current datablock */
    FAST int cindex;		/* character index datablock's databuf */
    FAST char *cptr;		/* points to character being read in the file */
    FAST char *bptr;		/* points to current position in buffer */

    /* check for valid maxbytes */

    if (maxBytes <= 0)
	{
	errnoSet (S_netDrv_INVALID_NUMBER_OF_BYTES);
	return (ERROR);
	}
 
    /* if file opened for WRITE, don't read */
    if ((pNetFd->options & FD_MODE) == WRITE)
	{
	errnoSet (S_netDrv_WRITE_ONLY_CANNOT_READ);
	return (ERROR);
	}

    ssemTake (&pNetFd->netFdSem);

    /* if file pointer is at end of file, can't read any characters */

    if (pNetFd->filePtrByte == pNetFd->endOfFile)
	{
	ssemGive (&pNetFd->netFdSem);
	return (0);
	}

    cindex = pNetFd->filePtrByte % DATASIZE;	/* char index in datablock */
    dptr = pNetFd->filePtrBlock;
    cptr = &dptr->databuf [cindex];	/* point to current char in datablock */
    bptr = buf;				/* point to beginning of read buffer */

    /* read until maxBytes characters have been read */

    while (byteCount < maxBytes)
        {
        if ((maxBytes - byteCount) < (dptr->used - cindex))
	    {
	    /* stop reading when maxBytes characters have been read.
	    *  This is the last block to read from in order to finish
	    *  filling up the buffer.
	    */
	    bcopy (cptr, bptr, maxBytes - byteCount);
	    byteCount = maxBytes;
	    }
        else
	    {	
	    /* copy the rest of the datablock into buffer */

	    bcopy (cptr, bptr, dptr->used - cindex);
	    byteCount = byteCount + dptr->used - cindex;

	    if (dptr == (DATABLOCK *) lstLast (&pNetFd->dataLst))	
	        {
		/* this is the last block in the file. Seek to the end. */

	        status = netSeek (pNetFd, pNetFd->endOfFile);
		ssemGive (&pNetFd->netFdSem);
	        return (status == ERROR ? ERROR : byteCount);
	        }
	    else	/* get next block */
	        {
	        dptr = (DATABLOCK *) lstNext ((NODE *) dptr);
	        cindex = 0;
	        cptr = dptr->databuf;	/* point to first character in block */
	        bptr = buf + byteCount;	/* move buffer pointer */
	        }
	    }
        }  /* end of while */

    status = netSeek (pNetFd, pNetFd->filePtrByte + byteCount);
    ssemGive (&pNetFd->netFdSem);
    return (status == ERROR ? ERROR : byteCount);
    }
/*******************************************************************************
*
* netWrite - write bytes to remote file
*
* netWrite copies up to the specified number of bytes from the buffer
* to the open network file descriptor.  Bytes are written starting
* at the spot pointed to by the file's block and byte pointers.
* The file pointer is updated to point immediately after the last
* character that was written.
*
* Called only by the I/O system.
*
* SIDE EFFECTS: moves file pointer
*
* RETURNS:
*    Number of bytes written (error if != nbytes)
*    ERROR if nBytes < 0, or no more space can be allocated
*/

LOCAL int netWrite (pNetFd, buffer, nBytes)
    FAST NET_FD *pNetFd;	/* open file descriptor */
    char *buffer;		/* buffer being written from */
    FAST int nBytes;		/* number of bytes to write to file */

    {
    STATUS status;
    FAST int byteCount = 0;	/* number of written read so far */
    FAST DATABLOCK *dptr;	/* points to current datablock */
    FAST int cindex;		/* character index datablock's databuf */
    FAST char *cptr;		/* points to char being overwritten in file */
    FAST char *bptr;		/* points to current position in buffer */

    /* check for valid number of bytes */

    if (nBytes < 0)
	{
	errnoSet (S_netDrv_INVALID_NUMBER_OF_BYTES);
	return (ERROR);
	}
 
    /* if file opened for READ, don't write */
    if ((pNetFd->options & FD_MODE) == READ)
	{
	errnoSet (S_netDrv_READ_ONLY_CANNOT_WRITE);
	return (ERROR);
	}

    ssemTake (&pNetFd->netFdSem);

    cindex = pNetFd->filePtrByte % DATASIZE;
    dptr   = pNetFd->filePtrBlock;
    cptr   = &dptr->databuf [cindex];
    bptr   = buffer;

    while (byteCount < nBytes)
	{
	if ((nBytes - byteCount) < (DATASIZE - cindex))
	    {
	    /* almost done writing nBytes. This is the last block to write to */

	    bcopy (bptr, cptr, nBytes - byteCount);
	    byteCount = nBytes;

	    /* if we wrote past end of file, adjust end of file pointer */

	    if ((pNetFd->filePtrByte + byteCount > pNetFd->endOfFile) &&
		moveEndOfFile (pNetFd, pNetFd->filePtrByte + byteCount)
								    == ERROR)
		{
		ssemGive (&pNetFd->netFdSem);
		return (ERROR);
		}
	    }
	else		/* not last block to write to */
	    {
	    bcopy (bptr, cptr, DATASIZE - cindex);
	    byteCount = byteCount + DATASIZE - cindex;

	    /* if we wrote past end of file, adjust end of file pointer.
	    *  If necessary, moveEndOfFile will attach a new datablock 
	    *  to the end of the data chain.
	    */

	    if ((pNetFd->filePtrByte + byteCount > pNetFd->endOfFile) &&
		moveEndOfFile (pNetFd, pNetFd->filePtrByte + byteCount)
								    == ERROR)
		{
		ssemGive (&pNetFd->netFdSem);
		return (ERROR);
		}

	    /* point to first character in next datablock */

	    dptr   = (DATABLOCK *) lstNext ((NODE *) dptr);
	    cindex = 0;
	    cptr   = dptr->databuf;

	    /* adjust buffer pointer */

	    bptr = buffer + byteCount;
	    }
	} /* end of while loop */
    
    pNetFd->options |= FD_DIRTY;

    status = netSeek (pNetFd, pNetFd->filePtrByte + byteCount);

    ssemGive (&pNetFd->netFdSem);

    return (status == ERROR ? ERROR : byteCount);
    }
/*******************************************************************************
*
* netIoctl - do device specific control function
*
* Called only by the I/O system.
*
* RETURNS: whatever the called function returns
*/

LOCAL int netIoctl (pNetFd, function, arg)
    FAST NET_FD *pNetFd;	/* open network file descriptor */
    FAST int function;		/* function code */
    FAST int arg;		/* argument for function */

    {
    switch (function)
	{
	case FIOSEEK:
	    return (netSeek (pNetFd, arg));

	case FIOWHERE:
    	    return (pNetFd->filePtrByte);

	case FIONREAD:
    	    *((int *) arg) = pNetFd->endOfFile - pNetFd->filePtrByte;
	    return (OK);

	case FIODIRENTRY:
	    /* this is a kludge for 'ls'.  Do the listing, then return
	       ERROR so that 'ls' doesn't try listing an rt-11 device */

	    netLs (pNetFd);
	    return (ERROR);

	default:
	    errnoSet (S_netDrv_UNKNOWN_REQUEST);
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* netSeek - change file's current character position
*
* This routine moves the file pointer by the offset to a new 
* position.  If the new position is past the end of file, fill the 
* unused space with 0's.  The end of file pointer gets moved to the
* position immediately following the last '0' written.
* If the resulting file pointer would be negative, ERROR is returned.
* 
* Called only by the I/O system.
*
* INTERNAL
* There is deliberate co-recursion between netSeek and netWrite.
*
* RETURNS: OK | ERROR
*/

LOCAL int netSeek (pNetFd, newPos)
    FAST NET_FD *pNetFd;
    FAST int newPos;

    {
    FAST DATABLOCK *saveFilePtrBlock;
    FAST int saveFilePtrByte;
    int endOfFile;
    int newBlock;
    int curBlock;
    char *buf;
    DATABLOCK *dptr;
    int nbytes;

    if (newPos < 0)
	{
	errnoSet (S_netDrv_BAD_SEEK);
	return (ERROR);
	}

    saveFilePtrBlock = pNetFd->filePtrBlock;
    saveFilePtrByte  = pNetFd->filePtrByte;
    endOfFile        = pNetFd->endOfFile;
    newBlock         = newPos / DATASIZE;
    curBlock         = pNetFd->filePtrByte / DATASIZE;
    
    /* if new position is past end of file */

    if (newPos > endOfFile)
	{
	/* put 0's at the end of the file */

	if ((buf = malloc ((unsigned) (newPos - endOfFile))) == NULL)
	    {
	    return (ERROR);
	    }

	bzero (buf, newPos - endOfFile);

	/* move file pointer to end of file before writing 0's */
	pNetFd->filePtrBlock = (DATABLOCK *) lstLast (&pNetFd->dataLst);
	pNetFd->filePtrByte = endOfFile;

	/* netWrite will update the file pointer and end of file pointer */

	nbytes = netWrite (pNetFd, buf, (newPos - endOfFile));
	if (nbytes != (newPos - endOfFile))
	    {
	    if (nbytes != ERROR)
	    	errnoSet (S_netDrv_SEEK_PAST_EOF_ERROR);

	    pNetFd->filePtrBlock = saveFilePtrBlock;
	    pNetFd->filePtrByte = saveFilePtrByte;
	    free (buf);
	    return (ERROR);
	    }

	free (buf);
	}
    else	/* else, new position is within current file size */
	{
    	/* point to new block */
	/* system error if we go out of range of the datablocks */

	dptr = (DATABLOCK *) lstNStep ((NODE *) pNetFd->filePtrBlock, 
				       newBlock - curBlock);
	if (dptr == NULL)
	    {
	    errnoSet (S_netDrv_SEEK_FATAL_ERROR);
	    return (ERROR);
	    }
	
	pNetFd->filePtrBlock = dptr;
	
	/* point to new byte */

    	pNetFd->filePtrByte = newPos;
	}

    return (OK);
    }
/*******************************************************************************
*
* moveEndOfFile - moves end of file pointer to new position
*
* Adds a new datablock to the end of the datablock list if necessary.
* Assumes that end of file moves at most to the first position of 
* the next datablock.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS moveEndOfFile (pNetFd, newPos)
    FAST NET_FD *pNetFd;
    FAST int newPos;

    {
    FAST DATABLOCK *dptr;	/* pointer to new datablock */

    /*
    *  As soon as system error handling is implemented, a message should 
    *  be put here.
    *
    *  If the new position is before the current end of file, 
    *  OR more than one datablock ahead,
    *  OR further away than the first byte of the next datablock, 
    *  this is a system error ....looks awful, I know
    */

    if ((newPos <= pNetFd->endOfFile)				||
	((newPos - pNetFd->endOfFile) > DATASIZE)		||
	((newPos % DATASIZE < pNetFd->endOfFile % DATASIZE)	&&
	(newPos % DATASIZE != 0)))
	{
	errnoSet (S_netDrv_BAD_EOF_POSITION);
	return (ERROR);
	}

    /* if new position is in datablock after end of file,
    *  add another datablock to the file
    */

    if ((newPos / DATASIZE) > (pNetFd->endOfFile / DATASIZE))
	{	
	/* current datablock is full.
	*  New position is in the block after the current
	*  end of file.
	*/

	((DATABLOCK *) lstLast (&pNetFd->dataLst))->used = DATASIZE;

	if ((dptr = (DATABLOCK *) malloc (sizeof (DATABLOCK))) == NULL)
	    return (ERROR);

    	dptr->used = 0;
	lstAdd (&pNetFd->dataLst, (NODE *) dptr);
	}
    else
	((DATABLOCK *) lstLast(&pNetFd->dataLst))->used = newPos % DATASIZE;

    pNetFd->endOfFile = newPos;
    return (OK);
    }
/*******************************************************************************
*
* releaseNetFd - free up NetFd
*/

LOCAL VOID releaseNetFd (pNetFd)
    FAST NET_FD *pNetFd;

    {
    /* free up data list */

    lstFree (&pNetFd->dataLst);

    semDelete (pNetFd->netFdSem.semId);		/* XXX delete super sem XXX */

    free ((char *) pNetFd);
    }

/* XXX super semaphore routines - until semLib is more sophisticated */

/*******************************************************************************
*
* ssemInit - super semInit
*
* This is just like the standard semCreate except that the semaphore
* includes a tid field.
*
* NOMANUAL
*/

LOCAL VOID ssemInit (pSSemaphore)
    SSEMAPHORE *pSSemaphore;

    {
    pSSemaphore->tid = NONE;
    pSSemaphore->semId = semCreate ();
    }
/*******************************************************************************
*
* ssemGive - super semGive
*
* This is just like the standard semGive except that the semaphore
* includes a tid field.
*
* NOMANUAL
*/

LOCAL VOID ssemGive (pSSemaphore)
    SSEMAPHORE *pSSemaphore;

    {
    pSSemaphore->tid = NONE;
    semGive (pSSemaphore->semId);
    }
/*******************************************************************************
*
* ssemTake - super semTake
*
* This is just like the standard semTake except if the semaphore is already
* taken by the requesting task, it won't deadlock trying to take it again.
*
* NOMANUAL
*/

LOCAL VOID ssemTake (pSSemaphore)
    SSEMAPHORE *pSSemaphore;

    {
    int tid = taskIdSelf ();

    if (pSSemaphore->tid != tid)
	{
	semTake (pSSemaphore->semId);
	pSSemaphore->tid = tid;
	}
    }
