/* lstLib.c - doubly linked list subroutine library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01o,05nov87,jlf  documentation
01n,02apr87,ecs  hushed lint in lstFree.
01m,25mar87,jlf  documentation
01l,21dec86,dnw  changed to not get include files from default directories.
01k,01jul86,jlf  documentation.
01j,21may86,llk	 added lstFree and lstNStep.
01i,09apr86,rdc  added lstFind.
01h,20jul85,jlf  documentation.
01g,19sep84,jlf  fixed spacing in comments by adding .ne's.
01f,08sep84,jlf  added comments and copyright notice.
01e,29jun84,dnw  added lstConcat and lstExtract.
01d,03jun84,dnw  added lstFirst, lstLast.
		 changed list.{head,tail} to list.node.{next,previous}.
		 cleaned up comments, etc.
01c,07may84,ecs  added lstNext, lstPrevious, and lstCount.
01b,09jun83,ecs  modified the documentation
01a,06aug82,dnw  created from old singly-linked-list lib which is now "slllb".
*/


/*
DESCRIPTION
This subroutine library supports the creation and maintenance of a
doubly linked list.  The user supplies a list descriptor (type LIST)
that will contain pointers to the first and last nodes in the list,
and a count of the number of nodes in the list.  The nodes in the
list can be any user-defined structure, but they must reserve space
for two pointers as their first elements.  Both the forward and
backward chains are terminated with a NULL pointer.

.ne 16
NON-EMPTY LIST:
.CS

   ---------             --------          --------
   | head--------------->| next----------->| next---------
   |       |             |      |          |      |      |
   |       |       ------- prev |<---------- prev |      |
   |       |       |     |      |          |      |      |
   | tail------    |     | ...  |    ----->| ...  |      |
   |       |  |    v                 |                   v
   |count=2|  |  -----               |                 -----
   ---------  |   ---                |                  ---
              |    -                 |                   -
              |                      |
              ------------------------

.CE
.ne 12
EMPTY LIST:
.CS

	-----------
        |  head------------------
        |         |             |
        |  tail----------       |
        |         |     |       v
        | count=0 |   -----   -----
        -----------    ---     ---
                        -	-

.CE

INCLUDE FILE: lstLib.h
*/


/* LINTLIBRARY */

#include "vxWorks.h"
#include "lstLib.h"

#define HEAD	node.next		/* first node in list */
#define TAIL	node.previous		/* last node in list */

/*********************************************************************
*
* lstInit - initialize list descriptor
*
* This routine initializes the specified list to an empty list.
*/

VOID lstInit (pList)
    FAST LIST *pList;    /* pointer to list descriptor to be initialized */

    {
    pList->HEAD	 = NULL;
    pList->TAIL  = NULL;
    pList->count = 0;
    }
/*************************************************************************
*
* lstAdd - add node to end of list
*
* This routine adds the specified node to the end of the specified list.
*/

VOID lstAdd (pList, pNode)
    LIST *pList;	/* pointer to list descriptor */
    NODE *pNode;	/* pointer to node to be added */

    {
    lstInsert (pList, pList->TAIL, pNode);
    }
/**************************************************************************
*
* lstConcat - concatenate two lists
*
* This routine concatenates the second list to the end of the first list.
* The second list is left empty.  It is OK for either or both lists to be
* empty.
*/

VOID lstConcat (pDstList, pAddList)
    FAST LIST *pDstList;	/* Destination list */
    FAST LIST *pAddList;	/* List to be added to dstList */

    {
    if (pAddList->count == 0)		/* nothing to do if AddList is empty */
	return;

    if (pDstList->count == 0)
	*pDstList = *pAddList;
    else
	{
	/* both lists non-empty; update DstList pointers */

	pDstList->TAIL->next     = pAddList->HEAD;
	pAddList->HEAD->previous = pDstList->TAIL;
	pDstList->TAIL           = pAddList->TAIL;

	pDstList->count += pAddList->count;
	}

    /* make AddList empty */

    lstInit (pAddList);
    }
/**************************************************************************
*
* lstCount - report number of nodes in list
*
* This routine returns the number of nodes in the given list.
*/

int lstCount (pList)
    LIST *pList;	/* pointer to list descriptor */

    {
    return (pList->count);
    }
/**************************************************************************
*
* lstDelete - delete specified node from list
*
* This routine deletes the specified node from the specified list.
*/

VOID lstDelete (pList, pNode)
    FAST LIST *pList;	/* pointer to list descriptor */
    FAST NODE *pNode;	/* pointer to node to be deleted */

    {
    if (pNode->previous == NULL)
	pList->HEAD = pNode->next;
    else
	pNode->previous->next = pNode->next;

    if (pNode->next == NULL)
	pList->TAIL = pNode->previous;
    else
	pNode->next->previous = pNode->previous;

    /* update node count */

    pList->count--;
    }
/************************************************************************
*
* lstExtract - extract a sublist from a list
*
* This routine extracts the sublist that starts with pStartNode and ends
* with pEndNode from the source list and makes the destination list
* contain those nodes.  
*/

VOID lstExtract (pSrcList, pStartNode, pEndNode, pDstList)
    FAST LIST *pSrcList;	/* pointer to source list */
    FAST NODE *pStartNode;	/* first node in sublist to be
				 * extracted */
    FAST NODE *pEndNode;	/* last node in sublist to be
				 * extracted */
    FAST LIST *pDstList;	/* pointer to list where to put
				 * extracted list */

    {
    FAST int i;
    FAST NODE *pNode;

    /* fix pointers in original list */

    if (pStartNode->previous == NULL)
	pSrcList->HEAD = pEndNode->next;
    else
	pStartNode->previous->next = pEndNode->next;

    if (pEndNode->next == NULL)
	pSrcList->TAIL = pStartNode->previous;
    else
	pEndNode->next->previous = pStartNode->previous;


    /* fix pointers in extracted list */

    pDstList->HEAD = pStartNode;
    pDstList->TAIL = pEndNode;

    pStartNode->previous = NULL;
    pEndNode->next       = NULL;


    /* count number of nodes in extracted list and update counts in lists */

    i = 0;

    for (pNode = pStartNode; pNode != NULL; pNode = pNode->next)
	i++;

    pSrcList->count -= i;
    pDstList->count = i;
    }
/************************************************************************
*
* lstFirst - find first node in list
*
* DESCRIPTION
* Finds the first node in a linked list.
*
* RETURNS
*	Pointer to the first node in a list, or
*	NULL if the list is empty.
*/

NODE *lstFirst (pList)
    LIST *pList;	/* pointer to list descriptor */

    {
    return (pList->HEAD);
    }
/************************************************************************
*
* lstGet - get (delete and return) first node from list
*
* This routine gets the first node from the specified list, deletes the node
* from the list, and returns a pointer to the node gotten.
*
* RETURNS
*	Pointer to the node gotten, or
*	NULL if the list is empty.
*/

NODE *lstGet (pList)
    FAST LIST *pList;	/* pointer to list from which to get node */

    {
    FAST NODE *pNode = pList->HEAD;

    if (pNode != NULL)
	lstDelete (pList, pNode);

    return (pNode);
    }
/************************************************************************
*
* lstInsert - insert node in list after specified node
*
* This routine inserts the specified node in the specified list.
* The new node is placed following the specified 'previous' node in the list.
* If the specified previous node is NULL, the edit is inserted at the head
* of the list.
*/

VOID lstInsert (pList, pPrev, pNode)
    FAST LIST *pList;	/* pointer to list descriptor */
    FAST NODE *pPrev;	/* pointer to node after which to insert */
    FAST NODE *pNode;	/* pointer to node to be inserted */

    {
    FAST NODE *pNext;

    if (pPrev == NULL)
	{				/* new node is to be first in list */
	pNext = pList->HEAD;
	pList->HEAD = pNode;
	}
    else
	{				/* make prev node point fwd to new */
	pNext = pPrev->next;
	pPrev->next = pNode;
	}

    if (pNext == NULL)
	pList->TAIL = pNode;		/* new node is to be last in list */
    else
	pNext->previous = pNode;	/* make next node point back to new */


    /* set pointers in new node, and update node count */

    pNode->next		= pNext;
    pNode->previous	= pPrev;

    pList->count++;
    }
/************************************************************************
*
* lstLast - find last node in list
*
* This routine finds the last node in a list.
*
* RETURNS
*  pointer to the last node in list, or
*  NULL if the list is empty.
*/

NODE *lstLast (pList)
    LIST *pList;	/* pointer to list descriptor */

    {
    return (pList->TAIL);
    }
/************************************************************************
*
* lstNext - find next node in list
*
* Locates the node immediately after the node pointed to by the pNode.
*
* RETURNS:
* 	Pointer to the next node in the list, or
*	NULL if there is no
* next node.
*/

NODE *lstNext (pNode)
    NODE *pNode;	/* pointer to node whose successor
			 * is to be found */

    {
    return (pNode->next);
    }
/************************************************************************
*
* lstNth - find n-th node in list
*
* This routine returns a pointer to the node specified by the passed
* node number where the first node in the list is nodenum = 1.
* Note that the search is optimized by searching forward from the beginning
* if the node is closer to the head, and searching back from the end
* if it's closer to the tail.
*
* RETURNS:
*	Pointer to the Nth node, or
*	NULL if there is no Nth node.
*/

NODE *lstNth (pList, nodenum)
    FAST LIST *pList;		/* pointer to list descriptor */
    FAST int nodenum;		/* number of node to be found */

    {
    FAST NODE *pNode;

    /* verify node number is in list */

    if ((nodenum < 1) || (nodenum > pList->count))
	return (NULL);


    /* if nodenum is less than half way, look forward from beginning;
	otherwise look back from end */

    if (nodenum < (pList->count >> 1))
	{
	pNode = pList->HEAD;

	while (--nodenum > 0)
	    pNode = pNode->next;
	}

    else
	{
	nodenum -= pList->count;
	pNode = pList->TAIL;

	while (nodenum++ < 0)
	    pNode = pNode->previous;
	}

    return (pNode);
    }
/************************************************************************
*
* lstPrevious - find previous node in list
*
* Locates the node immediately before the node pointed to by the pNode.
*
* RETURNS:
*	Pointer to the previous node in the list, or
*	NULL if there is no previous node.
*/

NODE *lstPrevious (pNode)
    NODE *pNode;	/* pointer to node whose predecessor is
			 * to be found */

    {
    return (pNode->previous);
    }
/************************************************************************
*
* lstNStep - find a node in the list 'nStep' nodes away from the given node 
*
* Locates the node nStep steps away in either direction from the given node.  
* If nStep is positive move toward the tail, if nStep is negative move toward 
* the head.  If the number of steps goes out of range, a NULL is returned.
*
* RETURNS:
*	Pointer to the node nStep steps away, or
*	NULL if the node is out of range.
*/

NODE *lstNStep (pNode, nStep)
    FAST NODE *pNode;		/* the known node */
    int nStep;			/* number of steps away to find */

    {
    int i;

    for (i = 0; i < abs (nStep); i++)
	{
	if (nStep < 0)
	    pNode = pNode->previous;
	else if (nStep > 0)
	    pNode = pNode->next;
	if (pNode == NULL)
	    break;
	}
    return (pNode);
    }

/************************************************************************
*
* lstFind - find a node in a list
*
* Returns the node number of the given node (the first node is 1).
*
* RETURNS:
*  node number, or
*  ERROR if not found.
*/

int lstFind (pList, pNode)
    LIST *pList;		/* list in which to search */
    FAST NODE *pNode;		/* pointer to node to search for */

    {

    FAST NODE *pNextNode;
    FAST int index = 1;

    pNextNode = lstFirst (pList);

    while ((pNextNode != NULL) && (pNextNode != pNode))
	{
	index++;
	pNextNode = lstNext (pNextNode);
	}

    if (pNextNode == NULL)
	return (ERROR);
    else
	return (index);
    }

/************************************************************************
*
* lstFree - free up the list
*
* Turns any list into an empty list.  Frees up memory used for nodes.
*
* SEE ALSO: free(2)
*/

VOID lstFree (pList)
    LIST *pList;	/* list for which to free all nodes */

    {
    NODE *p1, *p2;

    if (pList->count > 0)
	{
	p1 = pList->HEAD;
	while (p1 != NULL)
	    {
	    p2 = p1->next;
	    free ((char *)p1);
	    p1 = p2;
	    }
	pList->count = 0;
	pList->HEAD = pList->TAIL = NULL;
	}
    }
