/* isiBootLib.c - ISI-specific boot daemon */

static char *copyright = "Copyright 1987-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
02e,22jun88,dnw  name tweaks.
02d,06jun88,dnw  changed taskSpawn/taskCreate args.
		 changed to spawn isiBootd unbreakable.
02c,30may88,dnw  changed to v4 names.
02b,21mar88,gae  added include of ioLib.h for READ/WRITE/UPDATE.
02a,25jan88,jcf  made kernel independent.
01f,18nov87,ecs  more lint.
01e,17nov87,ecs  lint.
01d,13nov87,dnw  changed calling sequence to input hook.
01c,05nov87,jlf  documentation
01b,20oct87,ecs  delinted.
		 added include of memLib.h & strLib.h.
01a,20aug87,jlf  first wrs version.  Created by modifying isi version ???
*/

/*
DESCRIPTION
This boot daemon runs on a VxWorks system which is CPU 0.  It allows autobooting
of other VxWorks CPUs in the same backplane which are running ISI boot PROMs.
The boot daemon is started by calling isiBootInit (2).  At that time, the
configuration file will be read, and the daemon spawned.

Packets are received by inserting a hook into the vb backplane network driver,
using etherInputHookAdd (2).  This causes the routine isiBootPacketInput (2)
to be called every time a packet is received.  If that packet was received from
the vb bus, and if its packet type is ISIBOOT, it is handled here.  Otherwise,
isiBootPacketInput (2) returns FALSE, and the packet will be processed
normally.

If the packet is ours, it is written to a pipe, which will later be read by
isiBootd (2) and acted upon.

PROTOCOL
The protocol used to request services of isiBootd is specific 
to the vb bus.  All operations are initiated by the remote machine by 
transmiting packets. All packets are the same length.  The host machine 
will transmit packets back (possibly with added information) to the remote
machine as an acknowledgement of its request.

The remote machine initiates a boot connection by broadcasting a
CONNECT packet, with the hostname of the boot server in the data portion of the
packet.  The daemon keeps track of this host name, because that name will
later be prepended to the requested file name.

Once the remote CPU has connected, it will send requests to get the boot
file.  The first request will be an OPEN, followed by some number of
READ's, and finally a CLOSE.

A state diagram of the protocol is shown below:

.CS
  +-------+             +----------+           +-------+--read|write-|
  | CLOSED|             | CONNECTED|           | OPENED|             |
  +-------+---connect-->+----------+---open--->+-------+<------------|
      ^                                          |
      |--------------close-----------------------|
.CE

AUTOBOOT
In addition to the requests described above, isiBootd also handles
AUTOBOOT requests.  When this request is received, isiBootd simply 
sends back a packet whose data is the boot line for that remote.  It
knows this from the configuration file (described below).

FAKING IT OUT
To some extent, this daemon is fooling the remote boot.  The remote boot
thinks it is making requests from a Unix host somewhere.  In fact, it
is is actually making the requests of a VxWorks system in the same chassis.
When the remote first does its CONNECT, the bootd keeps track of the
host name of the Unix host to which the CONNECT is really being done.  Later,
when an OPEN is performed, the bootd will open the requested file on that
Unix host.

MULTIPLE REQUESTS
In order for the bootd to support requests from more than one 
machine at once a "connection block" list is used.  When a packet comes
in we check through all active connections blocks to see if there is one with
a network address field the same as the packets source address.  If so, we 
use that connection state to service the packet.  If no connection was found
we get a new connection block from the idle pool.  When the connection is 
closed the connection block is returned to the idle pool.

BOOT CONFIGURATION FILE
At initialization, an initialization file is read, which contains the
boot line necessary to autoboot the remotes.
This file has information relating ethernet address to a machine's
host name its boot strings. Lines in the file consist of the following ASCII
representation: ethernet-address hostname bootstring. Such as:

3c:7b:00:00:00:01  dino		wrs:/usr/vw/config/is20/VxWorks b=91.0.0.5 h=90.0.0.2 g=91.0.0.8 u=jerry

where the first part is a 48 bit ethernet address, the second is the 
corresponding remote's name (which is currently ignored under VxWorks),
and remainder of the line is the boot string.
The ethernet address is represented as "x:x:x:x:x:x" where x is a hex 
number between 00 and ff;  the bytes are always in network order.
For remotes being booted over the backplane, the ethernet address is
always 3c:7b:00:00:00:<nn>, where <nn> is the CPU number of that remote.
Lines may be commented with a "#" as their first character.

The configuration file
is in /etc/bootd.conf.<in>, where <in> is the internet address of the
CPU on which the boot daemon is running (CPU 0) on the backplane net.
For instance, if the VxWorks CPU acting as CPU 0 was booted with the
following line:

 ex(0,0)wrs:/usr/vw/config/is20/VxWorks e=90.0.0.3 h=90.0.0.2 b=91.0.0.8 u=jerry

then the configuration file would be read from ``/etc/bootd.conf.91.0.0.8''.
*/

#define SERVER
#define MAX_CONNECTIONS 10
#define BOOT_CONFIG_FILE "/etc/bootd.conf."

#include "vxWorks.h"
#include "taskLib.h"
#include "types.h"
#include "socket.h"
#include "in.h"
#include "if.h"
#include "if_ether.h"
#include "lstLib.h"
#include "memLib.h"
#include "strLib.h"
#include "sysLib.h"


/* boot commands.  also used for states */

#define	BOOT_CONNECT		0x0
#define	BOOT_OPEN		0x1
#define	BOOT_READ		0x2
#define	BOOT_WRITE		0x3
#define	BOOT_CLOSE		0x4
#define	BOOT_AUTOBOOT		0x5

#define	BOOT_DATACNT		(1024+256+128+32)
#define	PRIORITY_ISIBOOTD	100
#define	ETHERTYPE_ISIBOOT	0x80Df	/* ISI: bootd enet protocol */

int isiPktCount;


int priorityIsiBootd = PRIORITY_ISIBOOTD;

IMPORT char *sysBootLine;

IMPORT char *bootStringToParams ();

typedef struct
    {
    int		pkid;			/* packet id */
    char	cmd;			/* command */
    int		offset;			/* offset */
    int		len;			/* length */
    char	data [BOOT_DATACNT];	/* data */
    } BOOT_PACKET;

typedef struct
    {
    struct ether_header	eh;		/* standard ethernet header */
    BOOT_PACKET packet;
    } BOOT_PIPE_MSG;

typedef struct
    {
    NODE node;		/* for maintenance of list */
    u_char address [6];	/* ethernet address */
    char name [14];	/* hostname */
    char bootLine [160]; /* bootstring to be filled in */
    } BOOT_CONFIG_ENTRY;

typedef struct
    {
    NODE	node;
    char	c_host [14];
    int 	c_fd;
    int		c_state;
    u_char	c_addr[6];
    } CONNECTION;

LOCAL char *bootstate[] =
    {
    "WAIT_OPEN",
    "OPENED",
    "ACTIVE",
    "ACTIVE",
    "WAIT_CONNECT",
    "WAIT_CONNECT_A",
    };

LOCAL char *command [] =
    {
    "CONNECT",
    "OPEN",
    "READ",
    "WRITE",
    "CLOSE",
    "AUTOBOOT",
    };

LOCAL int bootPipeFd = -1;	/* fd of pipe to isiBootd */

LOCAL char hostName [BOOT_FIELD_LEN];	/* name of host */
LOCAL char bad [BOOT_FIELD_LEN];	/* backplane internet addr */

LOCAL CONNECTION cnct [MAX_CONNECTIONS];
LOCAL LIST activep;
LOCAL LIST idlep;
LOCAL LIST configEntries;

LOCAL u_char myaddr[6] = { 0x70, 0x01, 0x00, 0x7f, 0xff, 0x00 };/* XXXX */
LOCAL struct ifnet *ourIf;

LOCAL BOOL trace = FALSE;

struct ifnet *ifunit ();

/* forward dec */

VOID isiBootd ();
BOOL isiBootPacketInput ();
LOCAL BOOT_CONFIG_ENTRY *getConfigEntry ();

/***********************************************************************
*
* isiBootInit - set up ISI-compatible boot daemon
*
* Initialize the ISI boot daemon.  This consists of initializing some data
* structures, spawning the boot daemon isiBootd (2), and inserting
* isiBootPacketInput (2) as the packet input hook.
*
* RETURNS:
*   OK, or
*   ERROR if can't spawn, or can't create a pipe, or no vb device.
*/

STATUS isiBootInit ()

    {
    char dum [BOOT_FIELD_LEN];		/* for things we don't care about */
    int i;

    /* get some necessary info, from the boot line */

    bootStringToParams (sysBootLine, dum, hostName, dum, dum, bad,
			dum, dum, dum, dum, &i, (int *) dum);

    if (sysProcNumGet() == 0)
	{
	lstInit (&configEntries);
	if ((readConfigFile () == ERROR) ||
	    (pipeDevCreate ("/pipe/isiBoot", 10, sizeof (BOOT_PIPE_MSG))!=OK) ||
	    ((bootPipeFd = open ("/pipe/isiBoot", UPDATE)) < 0)		    ||
	    ((ourIf = ifunit ("vb0")) == NULL)				    ||
	    (taskSpawn ("isiBootd", priorityIsiBootd, VX_UNBREAKABLE, 6000,
			isiBootd, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0) == ERROR))

	    return (ERROR);

	/* initialize the connection lists */

	lstInit (&activep);
	lstInit (&idlep);
	for (i = 0 ; i < (MAX_CONNECTIONS - 1) ; i++)
	    lstAdd (&idlep, (NODE *) &cnct [i]);

	/* add our hook, so we see all the packets */

	etherInputHookAdd (isiBootPacketInput);
	}
    else
	{
	/* boot demons only run on CPU 0 */

	if (trace)
	    logMsg ("isiBootd: only runs on CPU 0.  Exiting\n");
	}

    return (OK);
    }
/***********************************************************************
*
* isiBootPacketInput - receive a packet on a net
*
* This routine is inserted as a hook to be called directly by a network
* driver.  It checks every input packet.  If that packet came across the
* vb network, and if the packet type is ISIBOOT, then we handle it and
* return TRUE.  Otherwise, we return FALSE and the network will handle
* the packet normally.
*
* If it is one of our packets, we write it, with the ethernet header, to a
* pipe being read by isiBootd (), which will do all the real work.
*/

BOOL isiBootPacketInput (pIf, pEh, len)
    struct ifnet *pIf;		/* interface the packet was received on */
    struct ether_header *pEh;   /* packet received */
    int len;			/* length of data */

    {
    isiPktCount++;
    if ((pIf == ourIf) && (pEh->ether_type == ETHERTYPE_ISIBOOT)) {
	if (len > sizeof (BOOT_PIPE_MSG)) {
	    logMsg ("isiBootd: packet too long. (%d > %d)\n",
	      len, sizeof (BOOT_PIPE_MSG));
	    return (FALSE);
	}
	if (trace)
	    logMsg ("isiBootd: packet being written to pipe.\n");
	write (bootPipeFd, (char *) pEh, sizeof (BOOT_PIPE_MSG));
	return(TRUE);
    }
    else
	return (FALSE);
    }

/***********************************************************************
*
* isiBootd - ISI-compatible boot daemon task
*
* This routine is spawned as a task by isiBootInit (2).  It reads
* boot packets from a pipe (put there by isiBootPacketInput (2)) and
* handles them according to a state machine.
*
* Up to ten remotes can be booted at once.  Separate state info is kept
* for each of them.
*/

VOID isiBootd ()

    {
    BOOT_CONFIG_ENTRY *pConfig;
    BOOT_PIPE_MSG bp;
    BOOT_PACKET *packet;
    char filename [100];
    CONNECTION *ap;
    int n;
    BOOL sendPacket;

    /* act like a server */

    FOREVER
        {

	sendPacket = FALSE;

        /* get a packet from our pipe */

        if (fioRead (bootPipeFd, (char *) &bp, sizeof (BOOT_PIPE_MSG)) <= 0)
	    logMsg ("isiBootd: error reading boot msg from pipe.\n");

	packet = &bp.packet;

        if (trace) 
            {
	    printf ("isiBootd: pkid=%03d cmd=%-8s ", packet->pkid,
	      (packet->cmd > 5 ? "UNKNOWN" : command[packet->cmd]));
	    printf ("offset=%06d len=%03d\n",
		    packet->offset, packet->len);
            printf ("SRC %x:%x:%x:%x:%x:%x",
		    bp.eh.ether_shost[0], bp.eh.ether_shost[1],
		    bp.eh.ether_shost[2], bp.eh.ether_shost[3],
		    bp.eh.ether_shost[4], bp.eh.ether_shost[5]);
            printf (" DST %x:%x:%x:%x:%x:%x\n",
		    bp.eh.ether_dhost[0], bp.eh.ether_dhost[1],
		    bp.eh.ether_dhost[2], bp.eh.ether_dhost[3],
		    bp.eh.ether_dhost[4], bp.eh.ether_dhost[5]);
            }

        /* try to find an active connection at that source address */

        for (ap = (CONNECTION *) lstFirst (&activep); ap != NULL;
	     ap = (CONNECTION *) lstNext ((NODE *) ap))
	    
            if (bcmp ((char *)ap->c_addr, (char *)bp.eh.ether_shost, 6) == 0)
		{
		if (trace)
		    logMsg ("using connection 0x%x.\n", ap);
                break;
		}

        /* if no active connection then create a new connection */

        if (ap == NULL) 
            {
            if ((ap = (CONNECTION *) lstGet (&idlep)) == NULL) 
                {
                logMsg ("isiBootd: out of connections\n");
                continue;
                }
	    lstAdd (&activep, (NODE *) ap);
            ap->c_state = BOOT_CLOSE;
            ap->c_fd    = NONE;
            bcopy ((char *) bp.eh.ether_shost, (char *) ap->c_addr, 6);
            if (trace)
                logMsg ("isiBootd: ACTIVATE connection %x\n", ap);
            }

        if (trace)
            logMsg ("isiBootd: %x: STATE: %s; COMMAND: %s\n", ap, 
		    bootstate [ap->c_state], command [packet->cmd]);

	switch (packet->cmd)
	    {
	    case BOOT_AUTOBOOT:

		if ((pConfig = getConfigEntry ((char *)bp.eh.ether_shost))
			== NULL)
		    {
		    if (trace)
			logMsg ("isiBootd: AUTOBOOT NOT FOUND\n");
		    continue;
		    }

		if (trace)
		    logMsg ("isiBootd: AUTOBOOT %s\n",pConfig->bootLine);

		strcpy(packet->data, pConfig->bootLine);

		/* If this was a broadcast packet, make the destination be us */

		fixBroadcastAddress (&bp.eh);
		sendPacket = TRUE;
		break;

	    case BOOT_CONNECT:

		/* If we've already got a file open for it, this must be a
		   reconnect.  Close the old file first. */

		if (ap->c_fd != NONE)
		    close (ap->c_fd);

		/* We connect no matter who the final host is, but we keep
		   track. */

		strcpy (ap->c_host, packet->data);

		if (trace)
		    logMsg ( "isiBootd: HOST %s\n", packet->data);

		/* If this was a broadcast packet, make the destination be us */

		fixBroadcastAddress (&bp.eh);
		ap->c_state = BOOT_CONNECT;
		sendPacket = TRUE;
		break;

	    case BOOT_OPEN:
		if (ap->c_state == BOOT_CONNECT)
		    {
		    strcpy (filename, ap->c_host);
		    strcat (filename, ":");
		    strcat (filename, packet->data);
		    if (trace)
			logMsg ("isiBootd: OPEN %s\n", filename);
		    if ((ap->c_fd = open (filename, READ)) == ERROR) 
			{
			if (trace)
			    logMsg ("isiBootd: couldn't open file!!\n");
			ap->c_fd=NONE;
			}
		    else
			{
			ap->c_state = BOOT_OPEN;
			sendPacket = TRUE;
			}
		    if (trace)
			logMsg ("isiBootd: file opened\n");
		    }
		else
		    {
		    if (trace)
			logMsg ("isiBootd: invalid OPEN %s\n", packet->data);
		    }
		break;

	    case BOOT_WRITE:
		logMsg ("isiBootd: recvd BOOT_WRITE msg.  Not handled yet.\n");
		break;

	    case BOOT_READ:
		if (ap->c_state == BOOT_OPEN)
		    {
		    if (((n = lseek (ap->c_fd, (long) packet->offset, 0))
			    == ERROR) ||
			((n = fioRead (ap->c_fd, packet->data, packet->len))
			    == ERROR))
			{
			if (trace)
			    logMsg ("isiBootd: %x: read or seek error %x\n",
				    ap, packet->offset);
			}
		    if (n <= 0)
			packet->len = 0;
		    else
			packet->len = n;
		    sendPacket = TRUE;
		    }
		else
		    logMsg ("isiBootd: Invalid BOOT_READ.\n");

		break;

	    case BOOT_CLOSE:
		if (ap->c_state == BOOT_OPEN)

		    {
		    if (ap->c_fd != NULL) 
			{
			close (ap->c_fd);
			ap->c_fd = NONE;
			}

		    /* move the connection from the active list to idle */

		    lstDelete (&activep, (NODE *) ap);
		    lstAdd (&idlep, (NODE *) ap);
		    sendPacket = TRUE;
		    if (trace)
			logMsg ("isiBootd: CLOSE connection %x\n",ap);
		    }

		else
		    logMsg ("isiBootd: Invalid BOOT_CLOSE.\n");
		break;

	    default:
		logMsg ("isiBootd: bad packet\n");
		break;
            }

        /* swap source and dest addresses, and send back reply packet */

        bswap ((char *)bp.eh.ether_dhost, (char *)bp.eh.ether_shost,
		sizeof (bp.eh.ether_shost));

        if (trace) 
            {
            logMsg ("REP%d: ",
		    packet->pkid);
            logMsg ("DST %x:%x:%x:%x:%x:%x ",
		    bp.eh.ether_dhost[0], bp.eh.ether_dhost[1],
		    bp.eh.ether_dhost[2], bp.eh.ether_dhost[3],
		    bp.eh.ether_dhost[4], bp.eh.ether_dhost[5]);
            logMsg ("SRC %x:%x:%x:%x:%x:%x\n",
		    bp.eh.ether_shost[0], bp.eh.ether_shost[1],
		    bp.eh.ether_shost[2], bp.eh.ether_shost[3],
		    bp.eh.ether_shost[4], bp.eh.ether_shost[5]);
            }
	if (sendPacket)
	    etherOutput (ourIf, &bp.eh, (char *) packet, sizeof (BOOT_PACKET));
        }
    }
/***********************************************************************
*
* isiBootShow - show infor about isiBootd
*
* This routine prints some pertinent information about the current
* state of the boot daemon.
*/

VOID isiBootShow ()

    {
    CONNECTION *ap;

    logMsg ("isiBootd: connection status: trace %s\n", trace ? "on" : "off");
    for (ap = (CONNECTION *) lstFirst (&activep); ap != NULL;
	 ap = (CONNECTION *) lstNext ((NODE *) ap))

        logMsg ("%s (%x:%x:%x:%x:%x):%x\n", bootstate[ap->c_state],
		ap->c_addr[0], ap->c_addr[1], ap->c_addr[2], 
		ap->c_addr[3], ap->c_addr[4], ap->c_addr[5]);
    }
/***********************************************************************
*
* isiBootTrace - turn ISI boot daemon debug trace on/off
*
* This routine turns debug trace on or off for isiBootd.  This causes
* LOTS of info to printed, whenever packets are received or sent.
*
* Normally, this should only be used for debugging.
*/

VOID isiBootTrace (doTrace)
    BOOL doTrace;		/* TRUE starts trace */

    {
    trace = doTrace;

    if (trace) 
        logMsg ("isiBootd: trace on\n");
    else 
        logMsg ("isiBootd: trace off\n");

    }
/***********************************************************************
*
* getConfigEntry - get an entry from the config list
*
* Given a 48 bit ethernet address, this routine returns the config entry.
* Returns zero if not found;
*/

LOCAL BOOT_CONFIG_ENTRY *getConfigEntry (address)
    char *address;		/* ethernet address */

    {
    BOOT_CONFIG_ENTRY *pEntry;

    for (pEntry = (BOOT_CONFIG_ENTRY *) lstFirst (&configEntries);
	 pEntry != NULL;
	 pEntry = (BOOT_CONFIG_ENTRY *) lstNext ((NODE *) pEntry))
	{
        if (bcmp (pEntry->address, address, sizeof (pEntry->address)) == 0)
	    return (pEntry);
        }
    return (NULL);
    }
/**************************************************************************
*
* readConfigFile - get and digest the configuration file from the host
*
* The file we read is 
*
*	<host>:/etc/bootd.config.<bad>
*
* where
*
*	<host> is the host we booted from
*	<bad>  is our internet address on the backplane.
*/

LOCAL STATUS readConfigFile ()

    {
    int configFd;
    char configLine [256];
    char configFileName [80];
    BOOT_CONFIG_ENTRY entry;
    BOOT_CONFIG_ENTRY *pEntry;
    int intAddr [6];
    int i;

    /* make the config file name */

    strcpy (configFileName, hostName);
    strcat (configFileName, ":");
    strcat (configFileName, BOOT_CONFIG_FILE);
    strcat (configFileName, bad);

    printf ("isiBootd: opening config file '%s'.\n", configFileName);

    if ((configFd = open (configFileName, READ)) == ERROR)
	{
	logMsg ("isiBootd: error reading configuration file.\n");
	return (ERROR);
	}

    while ((i = fioRdString (configFd, configLine, sizeof (configLine))) != EOF)
	{
	if (configLine [0] == '#')
	    continue;

	if (sscanf(configLine, " %2x:%2x:%2x:%2x:%2x:%2x %s %[^\n] ", 
	    &intAddr [0], &intAddr [1], &intAddr [2],
	    &intAddr [3], &intAddr [4], &intAddr [5],
	    entry.name, entry.bootLine) != 8)
	    {
	    logMsg ("isiBootd: illegal line in config file.\n");
	    logMsg ("=>%s\n", configLine);
	    continue;
	    }
	/* Add the config entry to our list */

	for (i=0; i<6; i++)
	    entry.address [i] = intAddr [i];

	if ((pEntry = (BOOT_CONFIG_ENTRY *) malloc
				    (sizeof (BOOT_CONFIG_ENTRY))) == NULL)

	    {
	    logMsg ("isiBootd: couldn't add config entry - no mem.\n");
	    return (ERROR);
	    }

	bcopy ((char *) &entry, (char *) pEntry, sizeof (BOOT_CONFIG_ENTRY));
	strcpy (pEntry->name, entry.name);

	lstAdd (&configEntries, (NODE *) pEntry);

	if (trace)
	    {
	    logMsg ("isiBootd: entry added.  adr=%x:%x:%x:%x:%x:%x.\n",
		    entry.address [0], entry.address [1], entry.address [2],
		    entry.address [3], entry.address [4], entry.address [5]);

	    logMsg ("name = %s. bootLine = %s.\n", entry.name, entry.bootLine);
	    }
	}
    return (OK);
    }
/*************************************************************************
*
* fixBroadcastAddress - change a bradcast address to be our address
*
*/

LOCAL fixBroadcastAddress (pHeader)
    struct ether_header *pHeader;
    
    {
    /* If this was a broadcast packet, make the destination be us */

    if ((pHeader->ether_dhost [0] == 0xFF) &&
	(pHeader->ether_dhost [1] == 0xFF) &&
	(pHeader->ether_dhost [2] == 0xFF) &&
	(pHeader->ether_dhost [3] == 0xFF) &&
	(pHeader->ether_dhost [4] == 0xFF) &&
	(pHeader->ether_dhost [5] == 0xFF)) 
	{
	if (trace)
	    logMsg ("isiBootd: MYADDR->BROADCAST\n");

	bcopy ((char *) myaddr, (char *) pHeader->ether_dhost,
	       sizeof (pHeader->ether_dhost));
	}
    }
