/* iosLib.c - I/O system */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
03l,14nov88,dnw  changed ioTaskStdGet to take taskId arg.
03i,08oct88,gae  fixed checking for task specific std. input/output/error fd's.
03h,15aug88,jcf  lint.
03g,15jul88,llk  changed iosFdSet.  fd names are now allocated before iosFdSet
		   is called.
03f,30jun88,llk  added iosNextDevGet().
		 changed iosDevAdd() so that it will not add devices with
		   duplicate names.
03e,04jun88,llk  replaced ioDefDev with ioDefPath.
03d,30may88,dnw  changed to v4 names.
03c,28may88,dnw  made ios{Create,Delete,Open,Close,Read,Write,Ioctl} LOCAL.
03b,23apr88,jcf  fixed semaphore calls for new semLib.
03a,30mar88,gae  made drvTable & fdTable definition local.
		 made fd's 0, 1, & 2 be standard in/out/err a la UNIX.
		 added ios{Creat,Delete,Open,Read,Write,Ioctl,Close}() for use
		 by ioLib.c.  Got rid of iosFdGetFree() and added iosFdNew()
		 to simplify setting up of new file descriptor.
		 iosFdCheck() turned into a macro for speed/jsr overhead.
		 iosDevList() shows driver number.
		 iosFdList() shows driver number and indicates std. in/out/err.
02b,20feb88,dnw  fixed missing return value in iosDrvRemove().
02a,16dec87,jlf  added iosDevDelete (), iosDrvRemove (), and iosDrvList ().
		 changed iosDrvInstall to look for an unused entry, instead
		    of just adding new drivers at the end.
01p,05nov87,rdc  fixed documentation for iosDevFind.
01o,30sep87,gae  made fdTable LOCAL; added iosFdSet() to set values in
		   fdTable; now keep file name in table; added iosFdList().
01n,28apr87,gae  made iosLock() and iosUnlock() LOCAL.
01m,24mar87,jlf  documentation.
01l,20dec86,dnw  changed iosDevMatch() to find longest (instead of just first)
		   device name that is initial substring of file name.
		 changed to not get include files from default directories.
01k,01dec86,dnw  changed iosDevAdd() to put device name in allocated memory,
		   instead of in fixed length array.
01j,14apr86,rdc  changed memAllocates to mallocs. 
01i,11oct85,dnw  de-linted.
01h,20jul85,jlf  documentation.
01g,19sep84,jlf  worked on the comments a little.
01f,05sep84,jlf  fixed iosDevMatch to work properly.  Added copyright.  Added
		 comments.
01e,04sep84,dnw  added iosDevList.
01d,10aug84,dnw  changed ioDevFind to include "default device" search
		   if device name not explicitly specified.
01c,07aug84,ecs  added calls to setStatus to iosDevFind, iosDrvInstall,
		   iosFdCheck, and iosFdGetFree
01b,29jun84,ecs  changed iosDevFind to use new version of cmpbuf
01a,11jun84,dnw  culled from old drvLib and ioLib
*/

/*
This is the I/O system for VxWorks.  It's primary job is to route user I/O
requests to the proper drivers, and with the proper parameters.  In order
to do this, it keeps various tables of the available drivers and their names,
open files, etc.  

The I/O system should be initialized by calling iosInit before any other
routines here are called.  Each driver then installs itself by calling
iosDrvInstall.  Finally, each device serviced by each driver is added to
the I/O system by calling iosDevAdd.

The I/O system is described much more fully in the chapter "I/O System".

SEE ALSO: "I/O System", intLib (1), ioLib (1)
*/

#include "vxWorks.h"
#include "lstLib.h"
#include "memLib.h"
#include "semLib.h"
#include "ioLib.h"
#include "iosLib.h"
#include "strLib.h"


char ioDefPath [MAX_FILENAME_LENGTH];	/* current default I/O prefix */

/* I/O system data structures */

typedef struct		/* FD_ENTRY - entries in file table */
    {
    DEV_HDR *pDevHdr;		/* device header for this file */
    int value;			/* driver's id for this file */
    char *name;			/* actual file name */
    BOOL inuse;			/* active entry */
    } FD_ENTRY;

typedef struct		/* DRV_ENTRY - entries in driver jump table */
    {
    FUNCPTR de_create;
    FUNCPTR de_delete;
    FUNCPTR de_open;
    FUNCPTR de_close;
    FUNCPTR de_read;
    FUNCPTR de_write;
    FUNCPTR de_ioctl;
    BOOL de_inuse;
    } DRV_ENTRY;

#define	STD_FIX(fd)	((fd)+3)
#define	STD_UNFIX(fd)	((fd)-3)
#define	STD_MAP(fd)	(STD_UNFIX(((fd) >= 0 && (fd) < 3) ? \
						ioTaskStdGet (0, fd) : (fd)))

#define	FD_CHECK(fd)	(((fd) >= 0 && (fd) < maxFiles && fdTable[(fd)].inuse)?\
			&fdTable[(fd)] : (FD_ENTRY *) \
			(errnoSet (S_iosLib_INVALID_FILE_DESCRIPTOR), NULL))

/* function declarations */

LOCAL DRV_ENTRY *drvTable;	/* table of driver entry points */
LOCAL FD_ENTRY *fdTable;	/* table of fd entries */
LOCAL LIST iosDvList;		/* list of I/O device headers */
LOCAL SEM_ID iosSemaphoreId;	/* semaphore to interlock access to io tables */
LOCAL int maxDrivers;		/* maximum # of drivers that can be installed */
LOCAL int maxFiles;		/* maximum # of files that can be open */
LOCAL DEV_HDR nullDevHdr;	/* device header for null device */


/* forward declarations */

LOCAL DEV_HDR *iosDevMatch ();


/*******************************************************************************
*
* iosInit - initialize I/O system
*
* This routine initializes the I/O system.
* It must be called exactly once, before any other I/O system routine.
*/

STATUS iosInit (max_drivers, max_files, nullDevName)
    int max_drivers;		/* Maximum number of drivers allowed */
    int max_files;		/* Max number of files allowed open at once */
    char *nullDevName;		/* Name of the null device (bit bucket) */

    {
    int i;
    int size;

    maxDrivers	= max_drivers;
    maxFiles	= max_files;

    ioDefPath [0] = EOS;

    /* allocate file table and make all entries free */

    size = maxFiles * sizeof (FD_ENTRY);
    fdTable = (FD_ENTRY *) malloc ((unsigned) size);

    if (fdTable == NULL)
	return (ERROR);

    bzero ((char *)fdTable, size);

    for (i = 0; i < maxFiles; i++)
	iosFdFree (STD_FIX(i));

    /* allocate driver table and make all entries null */

    size = maxDrivers * sizeof (DRV_ENTRY);
    drvTable = (DRV_ENTRY *) malloc ((unsigned) size);

    if (drvTable == NULL)
	return (ERROR);

    bzero ((char *) drvTable, size);
    for (i = 0; i < maxDrivers; i++)
	drvTable [i].de_inuse = FALSE;

    /* initialize the device list and data structure semaphore;
     * add the null device to the system */

    iosSemaphoreId = semCreate ();
    semGive (iosSemaphoreId);

    lstInit (&iosDvList);

    iosDevAdd (&nullDevHdr, nullDevName, 0);

    return (OK);
    }
/*******************************************************************************
*
* iosDrvInstall - install I/O driver
*
* This routine should be called exactly once by each I/O driver.
* It hooks up the various I/O service calls to the driver's own
* service routines, assigns the driver a number, and adds the driver
* to the driver table.
*
* RETURNS:
*    Driver number of new driver, or
*    ERROR if no room for new drivers.
*/

int iosDrvInstall (pCreate, pDelete, pOpen, pClose, pRead, pWrite, pIoctl)
    FUNCPTR pCreate;	/* pointer to driver create function	*/
    FUNCPTR pDelete;	/* pointer to driver delete function	*/
    FUNCPTR pOpen;	/* pointer to driver open function	*/
    FUNCPTR pClose;	/* pointer to driver close function	*/
    FUNCPTR pRead;	/* pointer to driver read function	*/
    FUNCPTR pWrite;	/* pointer to driver write function	*/
    FUNCPTR pIoctl;	/* pointer to driver ioctl function	*/

    {
    FAST DRV_ENTRY *pDrvEntry = NULL;
    FAST int drvnum;

    iosLock ();

    /* Find a free driver table entry.  Never assign driver number 0. */

    for (drvnum = 1; drvnum < maxDrivers; drvnum++)
	if (! drvTable [drvnum].de_inuse)
	    {
	    /* We've got a free entry */

	    pDrvEntry = &drvTable [drvnum];
	    break;
	    }

    if (pDrvEntry == NULL)
	{
	/* we couldn't find a free driver table entry */

	errnoSet (S_iosLib_DRIVER_GLUT);
	iosUnlock ();
	return (ERROR);
	}


    pDrvEntry->de_inuse	= TRUE;
    pDrvEntry->de_create= pCreate;
    pDrvEntry->de_delete= pDelete;
    pDrvEntry->de_open	= pOpen;
    pDrvEntry->de_close	= pClose;
    pDrvEntry->de_read	= pRead;
    pDrvEntry->de_write	= pWrite;
    pDrvEntry->de_ioctl	= pIoctl;

    iosUnlock ();
    return (drvnum);
    }
/*******************************************************************************
*
* iosDrvRemove - remove I/O driver
*
* This routine should be called to remove an I/O driver that has been
* added by an iosDrvInstall.
* It removes the driver from the driver table.
*
* RETURNS: OK, or ERROR if driver has files open
*/

STATUS iosDrvRemove (drvnum, forceClose)
    int drvnum;			/* Number of the driver to remove.
				 * Returned by iosDrvInstall */
    BOOL forceClose;		/* if TRUE, force closure of all open files */

    {
    DEV_HDR *pDevHdr;
    FAST int fd;
    FAST FD_ENTRY *pFdEntry;
    FAST DRV_ENTRY *pDrvEntry = &drvTable [drvnum];
    FUNCPTR drvClose = pDrvEntry->de_close;

    iosLock ();

    /* See if there are any open fd's for this driver */

    for (fd = 0; fd < maxFiles; fd++)
	{
	pFdEntry = &fdTable [fd];
	if (pFdEntry->inuse && pFdEntry->pDevHdr->drvNum == drvnum)
	    {
	    if (! forceClose)
		{
		printf ("Can't remove driver with open files.\n");
		iosUnlock ();
		return (ERROR);
		}
	    else
		{
		if (drvClose != NULL)
		    (* drvClose) (pFdEntry->value);

		iosFdFree (STD_FIX(fd));
		}
	    }
	}

    /* remove any devices for this driver */

    for (pDevHdr = (DEV_HDR *) lstFirst (&iosDvList); pDevHdr != NULL;
				pDevHdr = (DEV_HDR *) lstNext (&pDevHdr->node))
	{
	if (pDevHdr->drvNum == drvnum)
	    {
	    free (pDevHdr->name);
	    lstDelete (&iosDvList, &pDevHdr->node);
	    }
	}


    pDrvEntry->de_inuse	= FALSE;
    pDrvEntry->de_create= NULL;
    pDrvEntry->de_delete= NULL;
    pDrvEntry->de_open	= NULL;
    pDrvEntry->de_close	= NULL;
    pDrvEntry->de_read	= NULL;
    pDrvEntry->de_write	= NULL;
    pDrvEntry->de_ioctl	= NULL;

    iosUnlock ();

    return (OK);
    }
/*******************************************************************************
*
* iosDevAdd - add device to I/O system
*
* This routine adds a device to the I/O system.  It adds the device's node
* to the device list, and gives the device the specified name.
* Thereafter, 'open's of that device name will be routed to the driver
* specified.
*
* The first portion of the device structure whose pointer is passed to
* this routine must be a DEV_HDR, defined in iosLib.h
*
* RETURNS: OK, or ERROR
*/

STATUS iosDevAdd (pDevHdr, name, drvnum)
    DEV_HDR *pDevHdr;		/* Pointer to the device's structure */
    char *name;			/* Name of the device */
    int drvnum;			/* Number of the servicing driver.
				 * Returned by iosDrvInstall */

    {
    DEV_HDR *pDevMatch = iosDevMatch (name);

    /* don't add a device with a name that already exists in the device list.
     * iosDevMatch will return NULL if a device name is a substring of the
     * named argument, so check that the two names really are identical.
     */

    if ((pDevMatch != NULL) && (strcmp (pDevMatch->name, name) == 0))
	{
	errnoSet (S_iosLib_DUPLICATE_DEVICE_NAME);
	return (ERROR);
	}

    pDevHdr->name   = malloc ((unsigned) (strlen (name) + 1));
    pDevHdr->drvNum = drvnum;

    if (pDevHdr->name == NULL)
	return (ERROR);

    strcpy (pDevHdr->name, name);

    iosLock ();

    lstAdd (&iosDvList, &pDevHdr->node);

    iosUnlock ();

    return (OK);
    }
/*******************************************************************************
*
* iosDevDelete - delete device from I/O system
*
* This routine deletes a device from the I/O system.  It removes the device's
* node from the device list.
*
* If the device has never been added to the device list, weird results will
* occur.
*/

VOID iosDevDelete (pDevHdr)
    DEV_HDR *pDevHdr;		/* Pointer to the device's structure */

    {
    iosLock ();
    lstDelete (&iosDvList, &pDevHdr->node);
    iosUnlock ();
    }
/*******************************************************************************
*
* iosDevFind - find an I/O device in the device list
*
* This routine searches the device list for a device whose name matches
* the first portion of "name".  If it finds one,
* it sets the character pointer pointed to by pNameTail to point to the
* first character in name following the portion which matched the device
* name, and returns a pointer to the device.  If it fails, it returns
* a pointer to the default device, and sets the pNameTail to point to 
* the beginning of name.  If there is no default device, NULL is returned.
*
* RETURNS:
*    Pointer to device header, or
*    NULL if device not found.
*/

DEV_HDR *iosDevFind (name, pNameTail)
    char *name;			/* Name of the device */
    char **pNameTail;		/* Where to return ptr to tail of the name */

    {
    FAST DEV_HDR *pDevHdr = iosDevMatch (name);

    if (pDevHdr != NULL)
	*pNameTail = name + strlen (pDevHdr->name);
    else
	{
	pDevHdr = iosDevMatch (ioDefPath);
	*pNameTail = name;
	}

    if (pDevHdr == NULL)
	errnoSet (S_iosLib_DEVICE_NOT_FOUND);

    return (pDevHdr);
    }
/**********************************************************************
*
* iosDevMatch - find device whose name matches specified string
*
* RETURNS: Pointer to device header, or NULL if device not found.
*/

LOCAL DEV_HDR *iosDevMatch (name)
    char *name;

    {
    FAST DEV_HDR *pDevHdr;
    FAST int len;
    DEV_HDR *pBestDevHdr = NULL;
    int maxLen = 0;

    iosLock ();

    for (pDevHdr = (DEV_HDR *) lstFirst (&iosDvList); pDevHdr != NULL;
				pDevHdr = (DEV_HDR *) lstNext (&pDevHdr->node))
	{
	len = strlen (pDevHdr->name);

	if (strncmp (pDevHdr->name, name, len) == 0)
	    {
	    /* this device name is initial substring of name;
	     *   if it is longer than any other such device name so far,
	     *   remember it.
	     */

	    if (len > maxLen)
		{
		pBestDevHdr = pDevHdr;
		maxLen = len;
		}
	    }
	}

    iosUnlock ();

    return (pBestDevHdr);
    }
/*******************************************************************************
*
* iosNextDevGet - given a pointer to a device, get the next device in the list
*
* If the passed pointer is NULL, start at the top of the list.
*
* RETURNS:  pointer to a device, NULL if pDev is the last device in the list
*/

DEV_HDR *iosNextDevGet (pDev)
    DEV_HDR *pDev;

    {
    if (pDev == NULL)
	return ((DEV_HDR *) lstFirst (&iosDvList));
    else
	return ((DEV_HDR *) lstNext (&pDev->node));
    }
/*******************************************************************************
*
* iosDevShow - print a list of devices in the system
*
* This routine prints a list of all devices to standard output.
* 
* SEE ALSO: devs (2)
*/

VOID iosDevShow ()

    {
    FAST DEV_HDR *pDevHdr;

    printf ("%3s %-20.20s\n", "drv", "name");

    for (pDevHdr = (DEV_HDR *) lstFirst (&iosDvList); pDevHdr != NULL;
				pDevHdr = (DEV_HDR *) lstNext (&pDevHdr->node))
	printf ("%3d %-20.20s\n", pDevHdr->drvNum, pDevHdr->name);
    }
/*******************************************************************************
*
* iosDrvShow - print a list of drivers in the system
*
* This routine prints all the drivers in the driver list to standard output.
*/

VOID iosDrvShow ()

    {
    FAST int i;

    printf ("%3s %9s  %9s  %9s  %9s  %9s  %9s  %9s\n",
	"drv", "create", "delete", "open", "close", "read", "write", "ioctl");

    for (i = 1; i < maxDrivers; i++)
	{
	if (drvTable[i].de_inuse)
	    {
	    printf ("%3d %9x  %9x  %9x  %9x  %9x  %9x  %9x\n", i,
		drvTable[i].de_create, drvTable[i].de_delete,
		drvTable[i].de_open, drvTable[i].de_close,
		drvTable[i].de_read, drvTable[i].de_write,
		drvTable[i].de_ioctl);
	    }
	}
    }
/*******************************************************************************
*
* iosFdShow - print a list of file descriptor names in the system
*
* This routine prints a list of all fd's to standard output.
* 
* SEE ALSO: ioctl (2)
*/

VOID iosFdShow ()

    {
    char *stin;
    char *stout;
    char *sterr;
    FD_ENTRY *pFdEntry;
    int fd;
    int xfd;

    printf ("%3s %-20.20s %3s\n", "fd", "name", "drv");

    for (fd = 0; fd < maxFiles; fd++)
	{
	pFdEntry = &fdTable [fd];
	if (pFdEntry->inuse)
	    {
	    xfd = STD_FIX(fd);

	    stin  = (xfd == ioGlobalStdGet (STD_IN))  ? "in"  : "";
	    stout = (xfd == ioGlobalStdGet (STD_OUT)) ? "out" : "";
	    sterr = (xfd == ioGlobalStdGet (STD_ERR)) ? "err" : "";

	    printf ("%3d %-20.20s %3d %s %s %s\n",
		    xfd,
		    (pFdEntry->name == NULL) ? "(unknown)" : pFdEntry->name,
		    pFdEntry->pDevHdr->drvNum, stin, stout, sterr);
	    }
	}
    }
/*******************************************************************************
*
* iosFdValue - verify valid open fd and return driver specific value
*
* This routine checks to see if a file descriptor is valid and if
* so return driver specific value.
*
* RETURNS: driver specific value, or ERROR if invalid fd
*/

int iosFdValue (fd)
    FAST int fd;	/* fd to check */

    {
    int xfd = STD_MAP (fd);

    if (xfd >= 0 && xfd < maxFiles && fdTable[xfd].inuse)
	return (fdTable[xfd].value);
    else
	{
	errnoSet (S_iosLib_INVALID_FILE_DESCRIPTOR);
	return (ERROR);
	}
    }
/*******************************************************************************
*
* iosFdFree - free an fd entry in the fd table
*
* This routine frees a file descriptor table entry.
*/

VOID iosFdFree (fd)
    int fd;		/* fd to free up */

    {
    FAST FD_ENTRY *pFdEntry;
    FAST int xfd = STD_MAP(fd);

    if ((pFdEntry = FD_CHECK (xfd)) != NULL)
	{
	pFdEntry->inuse = FALSE;

	if (pFdEntry->name != NULL)
	    {
	    free (pFdEntry->name);
	    pFdEntry->name = NULL;
	    }
	}
    }
/*******************************************************************************
*
* iosFdSet - fill in file descriptor specifics
*
* This routine records the passed information about an fd into the fdTable.
* The fd name should have already been allocated in memory.
*/

VOID iosFdSet (fd, pDevHdr, name, value)
    int fd;		/* file descriptor */
    DEV_HDR *pDevHdr;	/* associated driver header */
    char *name;		/* name of file */
    int value;		/* arbitrary driver info */

    {
    FAST FD_ENTRY *pFdEntry = &fdTable [STD_UNFIX(fd)];

    if (pFdEntry->name != NULL)
	free (pFdEntry->name);

    pFdEntry->name    = name;
    pFdEntry->pDevHdr = pDevHdr;
    pFdEntry->value   = value;
    }
/*******************************************************************************
*
* iosFdNew - allocate and initialize a new fd
*
* This routine gets the index of a free entry in the fd table.
* The entry is marked as reserved, and will not be available again until
* it is explicitly free'd, with iosFdFree (2).
* The fd name should have already been allocated in memory.
*
* RETURNS: fd or ERROR
*/

int iosFdNew (pDevHdr, name, value)
    DEV_HDR *pDevHdr;	/* associated driver header */
    char *name;		/* name of file */
    int value;		/* arbitrary driver info */

    {
    FAST int fd;
    FAST FD_ENTRY *pFdEntry;

    iosLock ();

    for (fd = 0; fd < maxFiles; fd++)
	{
	pFdEntry = &fdTable[fd];
	if (!pFdEntry->inuse)
	    {
	    pFdEntry->inuse  = TRUE; /* reserve this entry */
	    pFdEntry->name   = NULL;
	    break;
	    }
	}

    iosUnlock ();
    
    if (fd >= maxFiles)
	{
	errnoSet (S_iosLib_TOO_MANY_OPEN_FILES);
	return (ERROR);
	}

    fd = STD_FIX(fd);
    
    iosFdSet (fd, pDevHdr, name, value);

    return (fd);
    }
/*******************************************************************************
*
* iosCreat - invoke driver to open file
*
* RETURNS: OK if no create routine or, driver specific value
*
* NOMANUAL
*/

int iosCreate (pDevHdr, fileName, mode)
    DEV_HDR *pDevHdr;
    char *fileName;
    int mode;

    {
    FUNCPTR drvCreate = drvTable [pDevHdr->drvNum].de_create;

    if (drvCreate != NULL)
	return ((*drvCreate)(pDevHdr, fileName, mode));
    else
	return (OK);
    }
/*******************************************************************************
*
* iosDelete - invoke driver to delete file
*
* RETURNS: OK if no delete routine or, driver specific value
*
* NOMANUAL
*/

int iosDelete (pDevHdr, fileName)
    DEV_HDR *pDevHdr;
    char *fileName;

    {
    FUNCPTR drvDelete = drvTable [pDevHdr->drvNum].de_delete;

    if (drvDelete != NULL)
	return ((*drvDelete)(pDevHdr, fileName));
    else
	return (OK);
    }
/*******************************************************************************
*
* iosOpen - invoke driver to open file
*
* RETURNS:
*    OK if driver has no open routine, or
*    whatever driver open routine returns
*
* NOMANUAL
*/

int iosOpen (pDevHdr, fileName, flags, mode)
    DEV_HDR *pDevHdr;
    char *fileName;
    int flags;
    int mode;

    {
    FUNCPTR drvOpen = drvTable [pDevHdr->drvNum].de_open;

    if (drvOpen != NULL)
	return ((*drvOpen)(pDevHdr, fileName, flags, mode));
    else
	return (OK);
    }
/*******************************************************************************
*
* iosClose - invoke driver to close file
*
* RETURNS:
*    OK if driver has no close routine, or
*    whatever driver close routine returns, or
*    ERROR if invalid file descriptor
*
* NOMANUAL
*/

STATUS iosClose (fd)
    int fd;

    {
    STATUS status;
    FUNCPTR drvClose;
    FD_ENTRY *pFdEntry;
    FAST int xfd = STD_MAP(fd);

    if ((pFdEntry = FD_CHECK (xfd)) == NULL)
	status = ERROR;		/* status set when bad fd */
    else
	{
	drvClose = drvTable [pFdEntry->pDevHdr->drvNum].de_close;

	if (drvClose != NULL)
	    status = (*drvClose)(pFdEntry->value);
	else
	    status = OK;

	iosFdFree (STD_FIX(xfd));
	}

    return (status);
    }
/*******************************************************************************
*
* iosRead - invoke driver read routine
*
* RETURNS:
*    0 if driver has no read routine, or
*    whatever driver read routine returns (number of bytes read), or
*    ERROR if invalid file descriptor
*
* NOMANUAL
*/

int iosRead (fd, buffer, maxbytes)
    int fd;
    char *buffer;
    int maxbytes;

    {
    FUNCPTR drvRead;
    FAST FD_ENTRY *pFdEntry;
    FAST int xfd = STD_MAP(fd);

    if ((pFdEntry = FD_CHECK (xfd)) == NULL)
	return (ERROR);

    drvRead = drvTable [pFdEntry->pDevHdr->drvNum].de_read;

    if (drvRead == NULL)
	return (0);
    else
	return ((* drvRead) (pFdEntry->value, buffer, maxbytes));
    }
/*******************************************************************************
*
* iosWrite - invoke driver write routine
*
* RETURNS:
*    0 if driver has no write routine, or
*    whatever driver write routine returns (number of bytes written), or
*    ERROR if invalid file descriptor
*
* NOMANUAL
*/

int iosWrite (fd, buffer, nbytes)
    int fd;
    char *buffer;
    int nbytes;

    {
    FUNCPTR drvWrite;
    FAST FD_ENTRY *pFdEntry;
    FAST int xfd = STD_MAP(fd);

    if ((pFdEntry = FD_CHECK (xfd)) == NULL)
	return (ERROR);

    drvWrite = drvTable [pFdEntry->pDevHdr->drvNum].de_write;

    if (drvWrite == NULL)
	return (nbytes);
    else
	return ((* drvWrite) (pFdEntry->value, buffer, nbytes));
    }
/*******************************************************************************
*
* iosIoctl - invoke driver ioctl routine
*
* RETURNS:
*    if function is FIOGETNAME then OK (and arg set to name), or
*    if driver has no ioctl routine and
*    function is FIONREAD then 0 otherwise ERROR, or
*    whatever driver ioctl routine returns, or
*    ERROR if invalid file descriptor
*
* NOMANUAL
*/

int iosIoctl (fd, function, arg)
    int fd;
    int function;
    int arg;

    {
    FUNCPTR drvIoctl;
    FAST FD_ENTRY *pFdEntry;
    FAST int xfd = STD_MAP(fd);
    
    if ((pFdEntry = FD_CHECK (xfd)) == NULL)
	return (ERROR);

    if (function == FIOGETNAME)
	{
	strcpy ((char *) arg, pFdEntry->name);
	return (OK);
	}

    drvIoctl = drvTable [pFdEntry->pDevHdr->drvNum].de_ioctl;

    if (drvIoctl == NULL)
	{
	if (function == FIONREAD)
	    {
	    * (char *) arg = 0;
	    return (OK);
	    }
	else
	    {
	    errnoSet (S_ioLib_UNKNOWN_REQUEST);
	    return (ERROR);
	    }
	}
    else
	{
	return ((* drvIoctl) (pFdEntry->value, function, arg));
	}
    }
/*******************************************************************************
*
* iosLock - get exclusive use of I/O data structures
*
* This routine takes exclusive use of the I/O system's data structures,
* by taking the semaphore used for that purpose.  If the semaphore
* is already taken, the calling task will suspend until the
* semaphore is available.
*/

LOCAL VOID iosLock ()

    {
    semTake (iosSemaphoreId);
    }
/*******************************************************************************
*
* iosUnlock - release exclusive use of I/O data structures
*
* This routine releases the semaphore which had been checked out by iosLock.
* This routine should only be called by a task which currently has the
* semaphore.
*/

LOCAL VOID iosUnlock ()

    {
    semGive (iosSemaphoreId);
    }

#ifdef	is68k
/************************************************************************
*
* iosFdGetName - return name associated with file descriptor
*
* return the (possibly NULL) name pointer for this fd
*/

char *iosFdGetName (fd)
    int fd;
    {
    return fdTable[STD_MAP(fd)].name;
    }
#endif	is68k
