/* ifLib.c - network interface subroutine library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01c,18aug88,gae  documentation.
01b,30may88,dnw  changed to v4 names.
01a,28may88,dnw  extracted from netLib.
*/

/*
DESCRIPTION
The ifLib library contains routines to configure the network interface
parameters.  Each routine corresponds to one of the functions of UNIX's
ifconfig (2).

SEE ALSO: hostLib (1), "Network"
*/

#include "vxWorks.h"
#include "in.h"
#include "if.h"
#include "ioctl.h"
#include "inetLib.h"
#include "strLib.h"


/*******************************************************************************
*
* ifAddrSet - configure a network interface
*
* ifAddrSet is used to assign an internet address to a particular interface.
* The named interface (supplied in the first parameter, eg "ex0" or "vb0")
* is assigned the internet address supplied in the second parameter, which
* may be a host name (which has been previously added to the host table
* with hostAdd (2)) or an internet address in standard internet address
* format (eg 90.0.0.4). 
*
* RETURNS:
*  OK, or
*  ERROR if address is already in table, or interface can't be configured
*/

STATUS ifAddrSet (interfaceName, interfaceAddress)
    char *interfaceName;	/* Name of interface to configure */
    char *interfaceAddress;	/* Internet adrs to assign to interface */

    {
    struct ifreq ifr;
    int so;
    STATUS status;
    int inetAddr;

    /* see if the address is in the host table */

    if (((inetAddr = hostGetByName (interfaceAddress)) == ERROR) &&
	((inetAddr = (int) inet_addr (interfaceAddress)) == ERROR))
	return (ERROR);

    bzero ((caddr_t) &ifr.ifr_addr, sizeof (ifr.ifr_addr));
    ifr.ifr_addr.sa_family = AF_INET;
    ((struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr = inetAddr;

    strncpy (ifr.ifr_name, interfaceName, sizeof (ifr.ifr_name));

    /* we need a dummy socket to do the ioctl */    

    if ((so = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
	return (ERROR);

    status = ioctl (so, (int) SIOCSIFADDR, (int)&ifr);
    close (so);

    return (status);
    }
/*******************************************************************************
*
* ifBroadcastSet - set broadcast address for an interface
*
* ifBroadcastSet is used to change the broadcast address for a particular
* interface.
* The named interface (supplied in the first parameter, eg "ex0" or "bp0")
* is assigned the broadcast address supplied in the second parameter, which
* must be a string in standard internet address format (eg "90.0.0.0"). 
*
* The default broadcast address for an interface is the interface's inet
* address with a host part of all 1's (eg. "90.255.255.255").
* This conforms to current ARPA specifications.
* However, some older implementations of the Internet
* protocols use a host part of all 0's for the broadcast address.
* The VxWorks implementation will automatically receive a host part of 
* all 0's as a broadcast address, in addition to the default or specified
* broadcast address.  But if VxWorks is to broadcast to older systems
* using a host part of all 0's as the broadcast address, then this
* routine should be used to change the broadcast address of the interface.
*
* RETURNS: OK | ERROR
*/

STATUS ifBroadcastSet (interfaceName, broadcastAddress)
    char *interfaceName;	/* Name of interface to assign */
    char *broadcastAddress;	/* Broadcast adrs to assign to interface */

    {
    struct ifreq ifr;
    int so;
    STATUS status;
    int inetAddr;

    if ((inetAddr = (int) inet_addr (broadcastAddress)) == ERROR)
	return (ERROR);

    bzero ((caddr_t) &ifr.ifr_addr, sizeof (ifr.ifr_addr));
    ifr.ifr_addr.sa_family = AF_INET;
    ((struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr = inetAddr;

    strncpy (ifr.ifr_name, interfaceName, sizeof (ifr.ifr_name));

    /* we need a dummy socket to do the ioctl */    

    if ((so = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
	return (ERROR);

    status = ioctl (so, (int) SIOCSIFBRDADDR, (int)&ifr);
    close (so);

    return (status);
    }
/*******************************************************************************
*
* ifMaskSet - define a subnet for an interface
*
* ifMaskSet may be used to allocate additional bits in an internet
* address to the network portion of the address.  The network portion  
* is specified with a mask that must contain ones in all positions
* that are to be interpreted as the network portion. (This includes all
* the bits that are normally interpreted as the network portion for the
* given class of address, plus the additional bits that you wish to add.
* Note also that all bits must be contiguous.)
*
* NOTE:
* A subnet mask should be set for an interface prior to setting
* the interface's internet address with the routine ifAddrSet (2),
* so that the address will be interpreted correctly.
*
* SEE ALSO: ifAddrSet (2)
*
* RETURNS: OK | ERROR
*/

STATUS ifMaskSet (interfaceName, netMask)
    char *interfaceName;	/* Name of interface for which to set mask */
    int netMask;		/* subnet mask */

    {
    struct ifreq ifr;
    int so;
    STATUS status;

    strncpy (ifr.ifr_name, interfaceName, sizeof (ifr.ifr_name));

    bzero ((caddr_t) &ifr.ifr_addr, sizeof (ifr.ifr_addr));
    ifr.ifr_addr.sa_family = AF_INET;
    ((struct sockaddr_in *) &ifr.ifr_addr)->sin_addr.s_addr = netMask;

    if ((so = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
	return (ERROR);

    status = ioctl (so, (int) SIOCSIFNETMASK, (caddr_t) &ifr); 
    close (so);

    return (status);
    }
